/*
 * Copyright (C) 2022 Temporal Technologies, Inc. All Rights Reserved.
 *
 * Copyright (C) 2012-2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Modifications copyright (C) 2017 Uber Technologies, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this material except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.temporal.common.converter;

import com.google.protobuf.ByteString;
import io.temporal.api.common.v1.Payload;
import java.lang.reflect.Type;
import java.util.Optional;

public final class ByteArrayPayloadConverter implements PayloadConverter {
  @Override
  public String getEncodingType() {
    return EncodingKeys.METADATA_ENCODING_RAW_NAME;
  }

  @Override
  public Optional<Payload> toData(Object value) throws DataConverterException {
    if (!(value instanceof byte[])) {
      return Optional.empty();
    }
    return Optional.of(
        Payload.newBuilder()
            .putMetadata(EncodingKeys.METADATA_ENCODING_KEY, EncodingKeys.METADATA_ENCODING_RAW)
            .setData(ByteString.copyFrom((byte[]) value))
            .build());
  }

  @Override
  public <T> T fromData(Payload content, Class<T> valueClass, Type valueType)
      throws DataConverterException {
    ByteString data = content.getData();
    if (valueClass != byte[].class) {
      throw new IllegalArgumentException(
          "Raw encoding can be deserialized only to a byte array. valueClass="
              + valueClass.getName());
    }
    return (T) data.toByteArray();
  }
}
