/*
 * Decompiled with CFR 0.152.
 */
package io.temporal.internal.worker;

import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.uber.m3.tally.Scope;
import io.temporal.api.common.v1.WorkflowExecution;
import io.temporal.api.workflowservice.v1.PollWorkflowTaskQueueResponseOrBuilder;
import io.temporal.internal.common.WorkflowExecutionUtils;
import io.temporal.internal.replay.WorkflowRunTaskHandler;
import io.temporal.internal.worker.WorkflowRunLockManager;
import java.util.Objects;
import java.util.concurrent.Callable;
import javax.annotation.concurrent.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public final class WorkflowExecutorCache {
    private final Logger log = LoggerFactory.getLogger(WorkflowExecutorCache.class);
    private final WorkflowRunLockManager runLockManager;
    private final Cache<String, WorkflowRunTaskHandler> cache;
    private final Scope metricsScope;

    public WorkflowExecutorCache(int workflowCacheSize, WorkflowRunLockManager runLockManager, Scope scope) {
        Preconditions.checkArgument((workflowCacheSize > 0 ? 1 : 0) != 0, (Object)"Max cache size must be greater than 0");
        this.runLockManager = runLockManager;
        this.cache = CacheBuilder.newBuilder().maximumSize((long)workflowCacheSize).concurrencyLevel(128).removalListener(e -> {
            WorkflowRunTaskHandler entry = (WorkflowRunTaskHandler)e.getValue();
            if (entry != null) {
                try {
                    this.log.trace("Closing workflow execution for runId {}, cause {}", e.getKey(), (Object)e.getCause());
                    entry.close();
                    this.log.trace("Workflow execution for runId {} closed", (Object)e);
                }
                catch (Throwable t) {
                    this.log.error("Workflow execution closure failed with an exception", t);
                    throw t;
                }
            }
        }).build();
        this.metricsScope = Objects.requireNonNull(scope);
        this.metricsScope.gauge("temporal_sticky_cache_size").update((double)this.size());
    }

    public WorkflowRunTaskHandler getOrCreate(PollWorkflowTaskQueueResponseOrBuilder workflowTask, Scope workflowTypeScope, Callable<WorkflowRunTaskHandler> workflowExecutorFn) throws Exception {
        WorkflowExecution execution = workflowTask.getWorkflowExecution();
        String runId = execution.getRunId();
        if (WorkflowExecutionUtils.isFullHistory(workflowTask)) {
            this.invalidate(execution, this.metricsScope, "full history", null);
            this.log.trace("New Workflow Executor {}-{} has been created for a full history run", (Object)execution.getWorkflowId(), (Object)runId);
            return workflowExecutorFn.call();
        }
        WorkflowRunTaskHandler workflowRunTaskHandler = (WorkflowRunTaskHandler)this.cache.getIfPresent((Object)runId);
        if (workflowRunTaskHandler != null) {
            workflowTypeScope.counter("temporal_sticky_cache_hit").inc(1L);
            return workflowRunTaskHandler;
        }
        this.log.trace("Workflow Executor {}-{} wasn't found in cache and a new executor has been created", (Object)execution.getWorkflowId(), (Object)runId);
        workflowTypeScope.counter("temporal_sticky_cache_miss").inc(1L);
        return workflowExecutorFn.call();
    }

    public void addToCache(WorkflowExecution workflowExecution, WorkflowRunTaskHandler workflowRunTaskHandler) {
        this.cache.put((Object)workflowExecution.getRunId(), (Object)workflowRunTaskHandler);
        this.log.trace("Workflow Execution {}-{} has been added to cache", (Object)workflowExecution.getWorkflowId(), (Object)workflowExecution.getRunId());
        this.metricsScope.gauge("temporal_sticky_cache_size").update((double)this.size());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean evictAnyNotInProcessing(WorkflowExecution inFavorOfExecution, Scope workflowTypeScope) {
        try {
            String inFavorOfRunId = inFavorOfExecution.getRunId();
            for (String key : this.cache.asMap().keySet()) {
                boolean locked;
                if (key.equals(inFavorOfRunId) || !(locked = this.runLockManager.tryLock(key))) continue;
                try {
                    this.log.trace("Workflow Execution {}-{} caused eviction of Workflow Execution with runId {}", new Object[]{inFavorOfExecution.getWorkflowId(), inFavorOfRunId, key});
                    this.cache.invalidate((Object)key);
                    workflowTypeScope.counter("temporal_sticky_cache_thread_forced_eviction").inc(1L);
                    workflowTypeScope.counter("temporal_sticky_cache_total_forced_eviction").inc(1L);
                    boolean bl = true;
                    return bl;
                }
                finally {
                    this.runLockManager.unlock(key);
                }
            }
            this.log.trace("Failed to evict from Workflow Execution cache, cache size is {}", (Object)this.cache.size());
            boolean bl = false;
            return bl;
        }
        finally {
            this.metricsScope.gauge("temporal_sticky_cache_size").update((double)this.size());
        }
    }

    public void invalidate(WorkflowExecution execution, Scope workflowTypeScope, String reason, Throwable cause) {
        String runId = execution.getRunId();
        WorkflowRunTaskHandler present = (WorkflowRunTaskHandler)this.cache.getIfPresent((Object)runId);
        if (this.log.isTraceEnabled()) {
            this.log.trace("Invalidating {}-{} because of '{}', value is present in the cache: {}", new Object[]{execution.getWorkflowId(), runId, reason, present, cause});
        }
        this.cache.invalidate((Object)runId);
        if (present != null) {
            workflowTypeScope.counter("temporal_sticky_cache_total_forced_eviction").inc(1L);
            this.metricsScope.gauge("temporal_sticky_cache_size").update((double)this.size());
        }
    }

    public long size() {
        return this.cache.size();
    }

    public void invalidateAll() {
        this.cache.invalidateAll();
        this.metricsScope.gauge("temporal_sticky_cache_size").update((double)this.size());
    }
}

