/*
 * Decompiled with CFR 0.152.
 */
package io.temporal.internal.common;

import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import io.temporal.internal.common.AsyncBackoffThrottler;
import io.temporal.internal.common.BackoffThrottler;
import io.temporal.internal.common.CheckedExceptionWrapper;
import io.temporal.internal.common.StatusUtils;
import io.temporal.serviceclient.RpcRetryOptions;
import java.time.Duration;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class GrpcRetryer {
    private static final Logger log = LoggerFactory.getLogger(GrpcRetryer.class);

    public static <T extends Throwable> void retry(RpcRetryOptions options, RetryableProc<T> r) throws T {
        GrpcRetryer.retryWithResult(options, () -> {
            r.apply();
            return null;
        });
    }

    public static <R, T extends Throwable> R retryWithResult(RpcRetryOptions options, RetryableFunc<R, T> r) throws T {
        int attempt = 0;
        long startTime = System.currentTimeMillis();
        BackoffThrottler throttler = new BackoffThrottler(options.getInitialInterval(), options.getMaximumInterval(), options.getBackoffCoefficient());
        while (true) {
            try {
                ++attempt;
                throttler.throttle();
                R result = r.apply();
                throttler.success();
                return result;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new CancellationException();
            }
            catch (StatusRuntimeException e) {
                if (e.getStatus().getCode() == Status.Code.CANCELLED) {
                    throw new CancellationException();
                }
                throttler.failure();
                for (RpcRetryOptions.DoNotRetryPair pair : options.getDoNotRetry()) {
                    if (pair.getCode() != e.getStatus().getCode() || pair.getDetailsClass() != null && !StatusUtils.hasFailure(e, pair.getDetailsClass())) continue;
                    GrpcRetryer.rethrow((Exception)((Object)e));
                }
                long elapsed = System.currentTimeMillis() - startTime;
                int maxAttempts = options.getMaximumAttempts();
                Duration expiration = options.getExpiration();
                if (maxAttempts > 0 && attempt >= maxAttempts || expiration != null && elapsed >= expiration.toMillis()) {
                    GrpcRetryer.rethrow((Exception)((Object)e));
                }
                log.warn("Retrying after failure", (Throwable)e);
                continue;
            }
            break;
        }
    }

    public static <R> CompletableFuture<R> retryWithResultAsync(RpcRetryOptions options, Supplier<CompletableFuture<R>> function) {
        int attempt = 0;
        long startTime = System.currentTimeMillis();
        AsyncBackoffThrottler throttler = new AsyncBackoffThrottler(options.getInitialInterval(), options.getMaximumInterval(), options.getBackoffCoefficient());
        CompletableFuture unwrappedExceptionResult = new CompletableFuture();
        CompletableFuture<R> result = GrpcRetryer.retryWithResultAsync(options, function, attempt + 1, startTime, throttler);
        CompletionStage ignored = result.handle((r, e) -> {
            if (e == null) {
                unwrappedExceptionResult.complete(r);
            } else {
                unwrappedExceptionResult.completeExceptionally(CheckedExceptionWrapper.unwrap(e));
            }
            return null;
        });
        return unwrappedExceptionResult;
    }

    private static <R> CompletableFuture<R> retryWithResultAsync(RpcRetryOptions options, Supplier<CompletableFuture<R>> function, int attempt, long startTime, AsyncBackoffThrottler throttler) {
        options.validate();
        return ((CompletableFuture)((CompletableFuture)throttler.throttle().thenCompose(ignore -> {
            try {
                CompletableFuture result = (CompletableFuture)function.get();
                if (result == null) {
                    return CompletableFuture.completedFuture(null);
                }
                return result.handle((r, e) -> {
                    if (e == null) {
                        throttler.success();
                        return r;
                    }
                    throttler.failure();
                    throw CheckedExceptionWrapper.wrap(e);
                });
            }
            catch (Throwable e2) {
                throttler.failure();
                throw CheckedExceptionWrapper.wrap(e2);
            }
        })).handle((r, e) -> GrpcRetryer.failOrRetry(options, function, attempt, startTime, throttler, r, e))).thenCompose(pair -> {
            if (pair.getException() != null) {
                throw CheckedExceptionWrapper.wrap(pair.getException());
            }
            return pair.getValue();
        });
    }

    private static <R> ValueExceptionPair<R> failOrRetry(RpcRetryOptions options, Supplier<CompletableFuture<R>> function, int attempt, long startTime, AsyncBackoffThrottler throttler, R r, Throwable e) {
        if (e == null) {
            return new ValueExceptionPair<R>(CompletableFuture.completedFuture(r), null);
        }
        if (!(e instanceof StatusRuntimeException)) {
            return new ValueExceptionPair(null, e);
        }
        StatusRuntimeException exception = (StatusRuntimeException)e;
        long elapsed = System.currentTimeMillis() - startTime;
        for (RpcRetryOptions.DoNotRetryPair pair : options.getDoNotRetry()) {
            if (pair.getCode() != exception.getStatus().getCode() || pair.getDetailsClass() != null && !StatusUtils.hasFailure(exception, pair.getDetailsClass())) continue;
            return new ValueExceptionPair(null, e);
        }
        int maxAttempts = options.getMaximumAttempts();
        if (maxAttempts > 0 && attempt >= maxAttempts || options.getExpiration() != null && elapsed >= options.getExpiration().toMillis()) {
            return new ValueExceptionPair(null, e);
        }
        log.debug("Retrying after failure", e);
        CompletableFuture<R> next = GrpcRetryer.retryWithResultAsync(options, function, attempt + 1, startTime, throttler);
        return new ValueExceptionPair<R>(next, null);
    }

    private static <T extends Throwable> void rethrow(Exception e) throws T {
        if (e instanceof RuntimeException) {
            throw (RuntimeException)e;
        }
        Exception toRethrow = e;
        throw toRethrow;
    }

    private GrpcRetryer() {
    }

    private static class ValueExceptionPair<V> {
        private final CompletableFuture<V> value;
        private final Throwable exception;

        public ValueExceptionPair(CompletableFuture<V> value, Throwable exception) {
            this.value = value;
            this.exception = exception;
        }

        public CompletableFuture<V> getValue() {
            return this.value;
        }

        public Throwable getException() {
            return this.exception;
        }
    }

    public static interface RetryableFunc<R, E extends Throwable> {
        public R apply() throws E;
    }

    public static interface RetryableProc<E extends Throwable> {
        public void apply() throws E;
    }
}

