/*
 * Decompiled with CFR 0.152.
 */
package io.milvus.pool;

import io.milvus.v2.exception.ErrorCode;
import io.milvus.v2.exception.MilvusClientException;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.pool2.impl.GenericKeyedObjectPool;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClientCache<T> {
    public static final int THRESHOLD_INCREASE = 100;
    public static final int THRESHOLD_DECREASE = 50;
    private static final Logger logger = LoggerFactory.getLogger(ClientCache.class);
    private final String key;
    private final GenericKeyedObjectPool<String, T> clientPool;
    private final CopyOnWriteArrayList<ClientWrapper<T>> activeClientList = new CopyOnWriteArrayList();
    private final CopyOnWriteArrayList<ClientWrapper<T>> retireClientList = new CopyOnWriteArrayList();
    private final ScheduledExecutorService scheduler;
    private final AtomicLong totalCallNumber = new AtomicLong(0L);
    private final Lock clientListLock;
    private long lastCheckMs = 0L;
    private float fetchClientPerSecond = 0.0f;

    protected ClientCache(String key, GenericKeyedObjectPool<String, T> pool) {
        this.key = key;
        this.clientPool = pool;
        this.clientListLock = new ReentrantLock(true);
        ThreadFactory threadFactory = new ThreadFactory(){

            @Override
            public Thread newThread(@NotNull Runnable r) {
                Thread t = new Thread(r);
                t.setPriority(10);
                return t;
            }
        };
        this.scheduler = Executors.newScheduledThreadPool(1, threadFactory);
        this.startTimer(1000L);
    }

    public void preparePool() {
        try {
            this.clientPool.preparePool((Object)this.key);
            int minIdlePerKey = this.clientPool.getMinIdlePerKey();
            for (int i = 0; i < minIdlePerKey; ++i) {
                this.activeClientList.add(new ClientWrapper<Object>(this.clientPool.borrowObject((Object)this.key)));
            }
            if (logger.isDebugEnabled()) {
                logger.debug("ClientCache key: {} cache clients: {} ", (Object)this.key, (Object)this.activeClientList.size());
                logger.debug("Pool initialize idle: {} active: {} ", (Object)this.clientPool.getNumIdle((Object)this.key), (Object)this.clientPool.getNumActive((Object)this.key));
            }
        }
        catch (Exception e) {
            logger.error("Failed to prepare pool {}, exception: ", (Object)this.key, (Object)e);
            throw new MilvusClientException(ErrorCode.CLIENT_ERROR, (Throwable)e);
        }
    }

    private void checkQPS() {
        if (this.activeClientList.isEmpty()) {
            this.lastCheckMs = System.currentTimeMillis();
            return;
        }
        long totalCallNum = this.totalCallNumber.get();
        float perClientCall = (float)totalCallNum / (float)this.activeClientList.size();
        long timeGapMs = System.currentTimeMillis() - this.lastCheckMs;
        if (timeGapMs == 0L) {
            timeGapMs = 1L;
        }
        float perClientPerSecond = perClientCall * 1000.0f / (float)timeGapMs;
        this.fetchClientPerSecond = (float)(totalCallNum * 1000L) / (float)timeGapMs;
        if (logger.isDebugEnabled()) {
            logger.debug("ClientCache key: {} fetchClientPerSecond: {} perClientPerSecond: {}, cached clients: {}", new Object[]{this.key, Float.valueOf(this.fetchClientPerSecond), Float.valueOf(perClientPerSecond), this.activeClientList.size()});
            logger.debug("Pool idle: {} active: {} ", (Object)this.clientPool.getNumIdle((Object)this.key), (Object)this.clientPool.getNumActive((Object)this.key));
        }
        this.totalCallNumber.set(0L);
        this.lastCheckMs = System.currentTimeMillis();
        if (perClientPerSecond >= 100.0f) {
            T client;
            int expectedNum = (int)Math.ceil((double)totalCallNum / 100.0);
            int moreNum = expectedNum - this.activeClientList.size();
            if (moreNum > 3) {
                moreNum = 3;
            }
            for (int k = 0; k < moreNum && (client = this.fetchFromPool()) != null; ++k) {
                ClientWrapper<T> wrapper = new ClientWrapper<T>(client);
                this.activeClientList.add(wrapper);
                if (!logger.isDebugEnabled()) continue;
                logger.debug("ClientCache key: {} borrows a client", (Object)this.key);
            }
        }
        if (this.activeClientList.size() > 1 && perClientPerSecond <= 50.0f) {
            int maxLoad = -1000;
            int maxIndex = -1;
            for (int i = 0; i < this.activeClientList.size(); ++i) {
                ClientWrapper<T> wrapper = this.activeClientList.get(i);
                int refCount = wrapper.getRefCount();
                if (refCount <= maxLoad) continue;
                maxLoad = refCount;
                maxIndex = i;
            }
            if (maxIndex >= 0) {
                ClientWrapper<T> wrapper = this.activeClientList.get(maxIndex);
                this.activeClientList.remove(maxIndex);
                this.retireClientList.add(wrapper);
            }
        }
        this.returnRetiredClients();
    }

    private void returnRetiredClients() {
        this.retireClientList.removeIf(wrapper -> {
            if (wrapper.getRefCount() <= 0) {
                this.returnToPool(wrapper.getClient());
                if (logger.isDebugEnabled()) {
                    logger.debug("ClientCache key: {} returns a client", (Object)this.key);
                }
                return true;
            }
            return false;
        });
    }

    private void startTimer(long interval) {
        if (interval < 1000L) {
            interval = 1000L;
        }
        this.lastCheckMs = System.currentTimeMillis();
        this.scheduler.scheduleAtFixedRate(new Runnable(){

            @Override
            public void run() {
                ClientCache.this.checkQPS();
            }
        }, interval, interval, TimeUnit.MILLISECONDS);
    }

    public void stopTimer() {
        this.scheduler.shutdown();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public T getClient() {
        if (this.activeClientList.isEmpty()) {
            this.clientListLock.lock();
            try {
                if (this.activeClientList.isEmpty()) {
                    T client = this.fetchFromPool();
                    if (client == null) {
                        T t = null;
                        return t;
                    }
                    ClientWrapper<T> wrapper = new ClientWrapper<T>(client);
                    this.activeClientList.add(wrapper);
                    this.totalCallNumber.incrementAndGet();
                    T t = wrapper.getClient();
                    return t;
                }
            }
            finally {
                this.clientListLock.unlock();
            }
        }
        int minLoad = Integer.MAX_VALUE;
        ClientWrapper<T> wrapper = null;
        for (ClientWrapper<T> tempWrapper : this.activeClientList) {
            if (tempWrapper.getRefCount() >= minLoad) continue;
            minLoad = tempWrapper.getRefCount();
            wrapper = tempWrapper;
        }
        if (wrapper == null) {
            wrapper = this.activeClientList.get(0);
        }
        this.totalCallNumber.incrementAndGet();
        return wrapper.getClient();
    }

    public void returnClient(T grpcClient) {
        for (ClientWrapper<T> wrapper : this.activeClientList) {
            if (!wrapper.equals(grpcClient)) continue;
            wrapper.returnClient();
            return;
        }
        for (ClientWrapper<T> wrapper : this.retireClientList) {
            if (!wrapper.equals(grpcClient)) continue;
            wrapper.returnClient();
            return;
        }
    }

    private T fetchFromPool() {
        try {
            if (this.activeClientList.size() + this.retireClientList.size() >= this.clientPool.getMaxTotalPerKey()) {
                return null;
            }
            return (T)this.clientPool.borrowObject((Object)this.key);
        }
        catch (Exception e) {
            logger.error("Failed to get client, exception: ", (Throwable)e);
            return null;
        }
    }

    private void returnToPool(T grpcClient) {
        try {
            this.clientPool.returnObject((Object)this.key, grpcClient);
        }
        catch (Exception e) {
            logger.error("Failed to return client, exception: ", (Throwable)e);
        }
    }

    public float fetchClientPerSecond() {
        return this.fetchClientPerSecond;
    }

    private static class ClientWrapper<T> {
        private final T client;
        private final AtomicInteger refCount = new AtomicInteger(0);

        public ClientWrapper(T client) {
            this.client = client;
        }

        public int hashCode() {
            return this.client.hashCode();
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() == obj.getClass()) {
                return Objects.equals(this.client, ((ClientWrapper)obj).client);
            }
            if (this.client != null && this.client.getClass() == obj.getClass()) {
                return Objects.equals(this.client, obj);
            }
            return false;
        }

        public T getClient() {
            this.refCount.incrementAndGet();
            return this.client;
        }

        public void returnClient() {
            this.refCount.decrementAndGet();
        }

        public int getRefCount() {
            return this.refCount.get();
        }
    }
}

