/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.faulttolerance;

import io.helidon.faulttolerance.Bulkhead;
import io.helidon.faulttolerance.BulkheadConfig;
import io.helidon.faulttolerance.BulkheadException;
import io.helidon.faulttolerance.MetricsUtils;
import io.helidon.faulttolerance.SupplierHelper;
import io.helidon.metrics.api.Counter;
import io.helidon.metrics.api.Tag;
import io.helidon.metrics.api.Timer;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;

class BulkheadImpl
implements Bulkhead {
    private static final System.Logger LOGGER = System.getLogger(BulkheadImpl.class.getName());
    private final Lock inProgressLock;
    private final Semaphore inProgress;
    private final String name;
    private final BarrierQueue queue;
    private final AtomicLong concurrentExecutions = new AtomicLong(0L);
    private final AtomicLong callsAccepted = new AtomicLong(0L);
    private final AtomicLong callsRejected = new AtomicLong(0L);
    private final AtomicLong callsWaiting = new AtomicLong(0L);
    private final List<Bulkhead.QueueListener> listeners;
    private final Set<Supplier<?>> cancelledSuppliers = new CopyOnWriteArraySet();
    private final BulkheadConfig config;
    private final boolean metricsEnabled;
    private Counter callsCounterMetric;
    private Timer waitingDurationMetric;

    BulkheadImpl(BulkheadConfig config) {
        this.inProgress = new Semaphore(config.limit(), true);
        this.name = config.name().orElseGet(() -> "bulkhead-" + System.identityHashCode(config));
        this.listeners = config.queueListeners();
        this.queue = config.queueLength() > 0 ? new BlockingQueue(config.queueLength()) : new ZeroCapacityQueue();
        this.inProgressLock = new ReentrantLock(true);
        this.config = config;
        boolean bl = this.metricsEnabled = config.enableMetrics() || MetricsUtils.defaultEnabled();
        if (this.metricsEnabled) {
            Tag nameTag = Tag.create((String)"name", (String)this.name);
            this.callsCounterMetric = MetricsUtils.counterBuilder("ft.bulkhead.calls.total", nameTag);
            this.waitingDurationMetric = MetricsUtils.timerBuilder("ft.bulkhead.waitingDuration", nameTag);
            MetricsUtils.gaugeBuilder("ft.bulkhead.executionsRunning", this.concurrentExecutions::get, nameTag);
            MetricsUtils.gaugeBuilder("ft.bulkhead.executionsWaiting", this.callsWaiting::get, nameTag);
            MetricsUtils.gaugeBuilder("ft.bulkhead.executionsRejected", this.callsRejected::get, nameTag);
        }
    }

    public BulkheadConfig prototype() {
        return this.config;
    }

    @Override
    public String name() {
        return this.name;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public <T> T invoke(Supplier<? extends T> supplier) {
        boolean full;
        boolean acquired;
        this.inProgressLock.lock();
        try {
            if (this.metricsEnabled) {
                this.callsCounterMetric.increment();
            }
            acquired = this.inProgress.tryAcquire();
        }
        catch (Throwable t) {
            this.inProgressLock.unlock();
            throw t;
        }
        if (acquired) {
            this.inProgressLock.unlock();
            if (LOGGER.isLoggable(System.Logger.Level.DEBUG)) {
                LOGGER.log(System.Logger.Level.DEBUG, this.name + " invoke immediate " + String.valueOf(supplier));
            }
            return this.execute(supplier);
        }
        try {
            full = this.queue.isFull();
        }
        catch (Throwable t) {
            this.inProgressLock.unlock();
            throw t;
        }
        if (full) {
            this.inProgressLock.unlock();
            this.callsRejected.incrementAndGet();
            throw new BulkheadException("Bulkhead queue \"" + this.name + "\" is full");
        }
        try {
            Barrier barrier;
            long start = 0L;
            try {
                this.listeners.forEach(l -> l.enqueueing(supplier));
                if (this.metricsEnabled) {
                    start = System.nanoTime();
                    this.callsWaiting.incrementAndGet();
                }
                barrier = this.queue.enqueue(supplier);
            }
            finally {
                try {
                    if (this.metricsEnabled) {
                        this.waitingDurationMetric.record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
                        this.callsWaiting.decrementAndGet();
                    }
                }
                finally {
                    this.inProgressLock.unlock();
                }
            }
            if (barrier == null) {
                throw new BulkheadException("Bulkhead queue \"" + this.name + "\" is full");
            }
            barrier.waitOn();
            this.listeners.forEach(l -> l.dequeued(supplier));
            if (this.cancelledSuppliers.remove(supplier)) {
                return null;
            }
            if (LOGGER.isLoggable(System.Logger.Level.DEBUG)) {
                LOGGER.log(System.Logger.Level.DEBUG, this.name + " invoking " + String.valueOf(supplier));
            }
            return this.execute(supplier);
        }
        catch (InterruptedException e) {
            this.callsRejected.incrementAndGet();
            throw new BulkheadException("Bulkhead \"" + this.name + "\" interrupted while acquiring");
        }
        catch (ExecutionException e) {
            throw new BulkheadException(e.getMessage());
        }
    }

    @Override
    public Bulkhead.Stats stats() {
        return new Bulkhead.Stats(){

            @Override
            public long concurrentExecutions() {
                return BulkheadImpl.this.concurrentExecutions.get();
            }

            @Override
            public long callsAccepted() {
                return BulkheadImpl.this.callsAccepted.get();
            }

            @Override
            public long callsRejected() {
                return BulkheadImpl.this.callsRejected.get();
            }

            @Override
            public long waitingQueueSize() {
                return BulkheadImpl.this.queue.size();
            }
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T> T execute(Supplier<? extends T> supplier) {
        this.callsAccepted.incrementAndGet();
        this.concurrentExecutions.incrementAndGet();
        try {
            T result = supplier.get();
            if (LOGGER.isLoggable(System.Logger.Level.DEBUG)) {
                LOGGER.log(System.Logger.Level.DEBUG, this.name + " finished execution: " + String.valueOf(supplier) + " (success)");
            }
            T t = result;
            return t;
        }
        catch (Throwable t) {
            Throwable throwable = SupplierHelper.unwrapThrowable(t);
            LOGGER.log(System.Logger.Level.DEBUG, this.name + " finished execution: " + String.valueOf(supplier) + " (failure)", throwable);
            throw SupplierHelper.toRuntimeException(throwable);
        }
        finally {
            this.concurrentExecutions.decrementAndGet();
            this.inProgressLock.lock();
            try {
                boolean dequeued = this.queue.dequeueAndRetract();
                if (!dequeued) {
                    this.inProgress.release();
                }
            }
            finally {
                this.inProgressLock.unlock();
            }
        }
    }

    @Override
    public boolean cancelSupplier(Supplier<?> supplier) {
        boolean cancelled = this.queue.remove(supplier);
        if (cancelled) {
            this.cancelledSuppliers.add(supplier);
        }
        return cancelled;
    }

    private static class BlockingQueue
    implements BarrierQueue {
        private final int capacity;
        private final ReentrantLock lock;
        private final Queue<Supplier<?>> queue;
        private final Map<Supplier<?>, Barrier> map;

        BlockingQueue(int capacity) {
            if (capacity <= 0) {
                throw new IllegalArgumentException("Queue capacity must be greater than 0");
            }
            this.capacity = capacity;
            this.queue = new LinkedBlockingQueue(capacity);
            this.map = new IdentityHashMap();
            this.lock = new ReentrantLock();
        }

        @Override
        public int size() {
            return this.queue.size();
        }

        @Override
        public boolean isFull() {
            this.lock.lock();
            try {
                boolean bl = this.queue.size() == this.capacity;
                return bl;
            }
            finally {
                this.lock.unlock();
            }
        }

        @Override
        public Barrier enqueue(Supplier<?> supplier) {
            this.lock.lock();
            try {
                Barrier barrier = this.doEnqueue(supplier);
                return barrier;
            }
            finally {
                this.lock.unlock();
            }
        }

        @Override
        public boolean dequeueAndRetract() {
            this.lock.lock();
            try {
                Barrier barrier = this.dequeue();
                if (barrier != null) {
                    barrier.retract();
                    boolean bl = true;
                    return bl;
                }
                boolean bl = false;
                return bl;
            }
            finally {
                this.lock.unlock();
            }
        }

        @Override
        public boolean remove(Supplier<?> supplier) {
            this.lock.lock();
            try {
                boolean bl = this.queue.remove(supplier);
                return bl;
            }
            finally {
                this.lock.unlock();
            }
        }

        private Barrier dequeue() {
            Supplier<?> supplier = this.queue.poll();
            return supplier == null ? null : this.map.remove(supplier);
        }

        private Barrier doEnqueue(Supplier<?> supplier) {
            boolean added = this.queue.offer(supplier);
            return added ? this.map.computeIfAbsent(supplier, s -> new Barrier()) : null;
        }
    }

    private static class ZeroCapacityQueue
    implements BarrierQueue {
        private ZeroCapacityQueue() {
        }

        @Override
        public int size() {
            return 0;
        }

        @Override
        public boolean isFull() {
            return true;
        }

        @Override
        public Barrier enqueue(Supplier<?> supplier) {
            return null;
        }

        @Override
        public boolean dequeueAndRetract() {
            return false;
        }

        @Override
        public boolean remove(Supplier<?> supplier) {
            return false;
        }
    }

    private static interface BarrierQueue {
        public int size();

        public boolean isFull();

        public Barrier enqueue(Supplier<?> var1);

        public boolean dequeueAndRetract();

        public boolean remove(Supplier<?> var1);
    }

    private static class Barrier {
        private final CompletableFuture<Void> future = new CompletableFuture();

        private Barrier() {
        }

        void waitOn() throws ExecutionException, InterruptedException {
            this.future.get();
        }

        void retract() {
            this.future.complete(null);
        }
    }
}

