package io.gatling.recorder.internal.bouncycastle.jcajce.provider.asymmetric.ecgost;

import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.spec.AlgorithmParameterSpec;

import io.gatling.recorder.internal.bouncycastle.asn1.x9.X9IntegerConverter;
import io.gatling.recorder.internal.bouncycastle.crypto.CipherParameters;
import io.gatling.recorder.internal.bouncycastle.crypto.DerivationFunction;
import io.gatling.recorder.internal.bouncycastle.crypto.agreement.ECVKOAgreement;
import io.gatling.recorder.internal.bouncycastle.crypto.digests.GOST3411Digest;
import io.gatling.recorder.internal.bouncycastle.crypto.params.AsymmetricKeyParameter;
import io.gatling.recorder.internal.bouncycastle.crypto.params.ECDomainParameters;
import io.gatling.recorder.internal.bouncycastle.crypto.params.ECPrivateKeyParameters;
import io.gatling.recorder.internal.bouncycastle.crypto.params.ParametersWithUKM;
import io.gatling.recorder.internal.bouncycastle.jcajce.provider.asymmetric.ec.BCECPublicKey;
import io.gatling.recorder.internal.bouncycastle.jcajce.provider.asymmetric.util.BaseAgreementSpi;
import io.gatling.recorder.internal.bouncycastle.jcajce.provider.asymmetric.util.ECUtil;
import io.gatling.recorder.internal.bouncycastle.jcajce.spec.UserKeyingMaterialSpec;
import io.gatling.recorder.internal.bouncycastle.jce.interfaces.ECPrivateKey;
import io.gatling.recorder.internal.bouncycastle.jce.interfaces.ECPublicKey;

public class KeyAgreementSpi
    extends BaseAgreementSpi
{
    private static final X9IntegerConverter converter = new X9IntegerConverter();

    private String                 kaAlgorithm;

    private ECDomainParameters     parameters;
    private ECVKOAgreement agreement;

    private byte[]             result;

    protected KeyAgreementSpi(
        String kaAlgorithm,
        ECVKOAgreement agreement,
        DerivationFunction kdf)
    {
        super(kaAlgorithm, kdf);

        this.kaAlgorithm = kaAlgorithm;
        this.agreement = agreement;
    }

    protected Key engineDoPhase(
        Key     key,
        boolean lastPhase) 
        throws InvalidKeyException, IllegalStateException
    {
        if (parameters == null)
        {
            throw new IllegalStateException(kaAlgorithm + " not initialised.");
        }

        if (!lastPhase)
        {
            throw new IllegalStateException(kaAlgorithm + " can only be between two parties.");
        }

        CipherParameters pubKey;
        {
            if (!(key instanceof PublicKey))
            {
                throw new InvalidKeyException(kaAlgorithm + " key agreement requires "
                    + getSimpleName(ECPublicKey.class) + " for doPhase");
            }

            pubKey = generatePublicKeyParameter((PublicKey)key);
        }

        try
        {
            result = agreement.calculateAgreement(pubKey);
        }
        catch (final Exception e)
        {
            throw new InvalidKeyException("calculation failed: " + e.getMessage())
            {
                public Throwable getCause()
                            {
                                return e;
                            }
            };
        }

        return null;
    }

    protected void doInitFromKey(Key key, AlgorithmParameterSpec parameterSpec, SecureRandom random)
        throws InvalidKeyException, InvalidAlgorithmParameterException
    {
        if (!(key instanceof PrivateKey))
        {
            throw new InvalidKeyException(kaAlgorithm + " key agreement requires "
                + getSimpleName(ECPrivateKey.class) + " for initialisation");
        }

        if (parameterSpec != null && !(parameterSpec instanceof UserKeyingMaterialSpec))
        {
            throw new InvalidAlgorithmParameterException("No algorithm parameters supported");
        }

        ECPrivateKeyParameters privKey = (ECPrivateKeyParameters)ECUtil.generatePrivateKeyParameter((PrivateKey)key);
        this.parameters = privKey.getParameters();
        ukmParameters = (parameterSpec instanceof UserKeyingMaterialSpec) ? ((UserKeyingMaterialSpec)parameterSpec).getUserKeyingMaterial() : null;
        agreement.init(new ParametersWithUKM(privKey, ukmParameters));
    }

    private static String getSimpleName(Class clazz)
    {
        String fullName = clazz.getName();

        return fullName.substring(fullName.lastIndexOf('.') + 1);
    }

    protected byte[] doCalcSecret()
    {
        return result;
    }

    static AsymmetricKeyParameter generatePublicKeyParameter(
            PublicKey key)
        throws InvalidKeyException
    {
        return (key instanceof BCECPublicKey) ? ((BCECGOST3410PublicKey)key).engineGetKeyParameters() : ECUtil.generatePublicKeyParameter(key);
    }

    public static class ECVKO
        extends KeyAgreementSpi
    {
        public ECVKO()
        {
            super("ECGOST3410", new ECVKOAgreement(new GOST3411Digest()), null);
        }
    }
}
