package io.gatling.recorder.internal.bouncycastle.crypto.generators;

import java.math.BigInteger;

import io.gatling.recorder.internal.bouncycastle.crypto.AsymmetricCipherKeyPair;
import io.gatling.recorder.internal.bouncycastle.crypto.AsymmetricCipherKeyPairGenerator;
import io.gatling.recorder.internal.bouncycastle.crypto.CryptoServicePurpose;
import io.gatling.recorder.internal.bouncycastle.crypto.CryptoServicesRegistrar;
import io.gatling.recorder.internal.bouncycastle.crypto.KeyGenerationParameters;
import io.gatling.recorder.internal.bouncycastle.crypto.constraints.ConstraintUtils;
import io.gatling.recorder.internal.bouncycastle.crypto.constraints.DefaultServiceProperties;
import io.gatling.recorder.internal.bouncycastle.crypto.params.DHKeyGenerationParameters;
import io.gatling.recorder.internal.bouncycastle.crypto.params.DHParameters;
import io.gatling.recorder.internal.bouncycastle.crypto.params.DHPrivateKeyParameters;
import io.gatling.recorder.internal.bouncycastle.crypto.params.DHPublicKeyParameters;

/**
 * a Diffie-Hellman key pair generator.
 *
 * This generates keys consistent for use in the MTI/A0 key agreement protocol
 * as described in "Handbook of Applied Cryptography", Pages 516-519.
 */
public class DHKeyPairGenerator
    implements AsymmetricCipherKeyPairGenerator
{
    private DHKeyGenerationParameters param;

    public void init(
        KeyGenerationParameters param)
    {
        this.param = (DHKeyGenerationParameters)param;

        CryptoServicesRegistrar.checkConstraints(new DefaultServiceProperties("DHKeyGen", ConstraintUtils.bitsOfSecurityFor(this.param.getParameters().getP()), this.param.getParameters(), CryptoServicePurpose.KEYGEN));
    }

    public AsymmetricCipherKeyPair generateKeyPair()
    {
        DHKeyGeneratorHelper helper = DHKeyGeneratorHelper.INSTANCE;
        DHParameters dhp = param.getParameters();

        BigInteger x = helper.calculatePrivate(dhp, param.getRandom()); 
        BigInteger y = helper.calculatePublic(dhp, x);

        return new AsymmetricCipherKeyPair(
            new DHPublicKeyParameters(y, dhp),
            new DHPrivateKeyParameters(x, dhp));
    }
}
