/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.kubernetes.client.utils;

import com.fasterxml.jackson.core.JsonProcessingException;
import io.fabric8.kubernetes.api.model.AuthInfo;
import io.fabric8.kubernetes.api.model.NamedAuthInfo;
import io.fabric8.kubernetes.api.model.NamedContext;
import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.http.HttpClient;
import io.fabric8.kubernetes.client.http.HttpRequest;
import io.fabric8.kubernetes.client.http.HttpResponse;
import io.fabric8.kubernetes.client.internal.KubeConfigUtils;
import io.fabric8.kubernetes.client.internal.SSLUtils;
import io.fabric8.kubernetes.client.utils.Serialization;
import io.fabric8.kubernetes.client.utils.URLUtils;
import io.fabric8.kubernetes.client.utils.Utils;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.spec.InvalidKeySpecException;
import java.util.Base64;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.function.Consumer;
import javax.net.ssl.KeyManager;
import javax.net.ssl.TrustManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OpenIDConnectionUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(OpenIDConnectionUtils.class);
    public static final String EMPTY = "";
    public static final String ID_TOKEN_KUBECONFIG = "id-token";
    public static final String ISSUER_KUBECONFIG = "idp-issuer-url";
    public static final String REFRESH_TOKEN_KUBECONFIG = "refresh-token";
    public static final String REFRESH_TOKEN_PARAM = "refresh_token";
    public static final String GRANT_TYPE_PARAM = "grant_type";
    public static final String CLIENT_ID_PARAM = "client_id";
    public static final String CLIENT_SECRET_PARAM = "client_secret";
    public static final String ID_TOKEN_PARAM = "id_token";
    public static final String ACCESS_TOKEN_PARAM = "access_token";
    public static final String CLIENT_ID_KUBECONFIG = "client-id";
    public static final String CLIENT_SECRET_KUBECONFIG = "client-secret";
    public static final String IDP_CERT_DATA = "idp-certificate-authority-data";
    public static final String TOKEN_ENDPOINT_PARAM = "token_endpoint";
    public static final String WELL_KNOWN_OPENID_CONFIGURATION = ".well-known/openid-configuration";
    public static final String GRANT_TYPE_REFRESH_TOKEN = "refresh_token";

    private OpenIDConnectionUtils() {
    }

    public static CompletableFuture<String> resolveOIDCTokenFromAuthConfig(Config currentConfig, Map<String, String> currentAuthProviderConfig, HttpClient.Builder clientBuilder) {
        String accessToken = currentAuthProviderConfig.get(ID_TOKEN_KUBECONFIG);
        String issuer = currentAuthProviderConfig.get(ISSUER_KUBECONFIG);
        String clientId = currentAuthProviderConfig.get(CLIENT_ID_KUBECONFIG);
        String refreshToken = currentAuthProviderConfig.get(REFRESH_TOKEN_KUBECONFIG);
        String clientSecret = currentAuthProviderConfig.getOrDefault(CLIENT_SECRET_KUBECONFIG, EMPTY);
        String idpCert = currentAuthProviderConfig.getOrDefault(IDP_CERT_DATA, currentConfig.getCaCertData());
        if (OpenIDConnectionUtils.isTokenRefreshSupported(currentAuthProviderConfig)) {
            return OpenIDConnectionUtils.getOIDCProviderTokenEndpointAndRefreshToken(issuer, clientId, refreshToken, clientSecret, idpCert, clientBuilder).thenApply(map -> {
                Object token = map.get(ID_TOKEN_PARAM);
                if (token == null) {
                    LOGGER.warn("token response did not contain an id_token, either the scope \\\"openid\\\" wasn't requested upon login, or the provider doesn't support id_tokens as part of the refresh response.");
                    return accessToken;
                }
                try {
                    OpenIDConnectionUtils.persistKubeConfigWithUpdatedToken(currentConfig, map);
                }
                catch (IOException e) {
                    LOGGER.warn("oidc: failure while persisting new tokens into KUBECONFIG", (Throwable)e);
                }
                return String.valueOf(token);
            });
        }
        return CompletableFuture.completedFuture(accessToken);
    }

    static boolean isTokenRefreshSupported(Map<String, String> currentAuthProviderConfig) {
        return Utils.isNotNull(currentAuthProviderConfig.get(REFRESH_TOKEN_KUBECONFIG));
    }

    static CompletableFuture<Map<String, Object>> refreshOidcToken(HttpClient client, String clientId, String refreshToken, String clientSecret, String tokenURL) {
        HttpRequest request = OpenIDConnectionUtils.getTokenRefreshHttpRequest(client, tokenURL, clientId, refreshToken, clientSecret);
        CompletableFuture<HttpResponse<String>> response = client.sendAsync(request, String.class);
        return response.thenApply(r -> {
            String body = (String)r.body();
            if (body != null) {
                if (r.isSuccessful()) {
                    try {
                        return OpenIDConnectionUtils.convertJsonStringToMap(body);
                    }
                    catch (JsonProcessingException e) {
                        LOGGER.warn("Failure in fetching refresh token: ", (Throwable)e);
                    }
                } else {
                    LOGGER.warn("Response: {}", (Object)body);
                }
            }
            return Collections.emptyMap();
        });
    }

    static CompletableFuture<Map<String, Object>> getOIDCDiscoveryDocumentAsMap(HttpClient client, String issuer) {
        HttpRequest request = client.newHttpRequestBuilder().uri(OpenIDConnectionUtils.getWellKnownUrlForOpenIDIssuer(issuer)).build();
        return client.sendAsync(request, String.class).thenApply(response -> {
            try {
                if (response.isSuccessful() && response.body() != null) {
                    return OpenIDConnectionUtils.convertJsonStringToMap((String)response.body());
                }
                String responseBody = (String)response.body();
                LOGGER.warn("oidc: failed to query metadata endpoint: {} {}", (Object)response.code(), (Object)responseBody);
            }
            catch (IOException e) {
                LOGGER.warn("Could not refresh OIDC token, failure in getting refresh URL", (Throwable)e);
            }
            return Collections.emptyMap();
        });
    }

    static String getWellKnownUrlForOpenIDIssuer(String issuer) {
        return URLUtils.join(issuer, "/", WELL_KNOWN_OPENID_CONFIGURATION);
    }

    static String getParametersFromDiscoveryResponse(Map<String, Object> responseAsJson, String key) {
        if (responseAsJson.containsKey(key)) {
            return String.valueOf(responseAsJson.get(key));
        }
        LOGGER.warn("oidc: oidc: discovery object doesn't contain a {}", (Object)key);
        return EMPTY;
    }

    static boolean persistKubeConfigWithUpdatedToken(Config currentConfig, Map<String, Object> updatedAuthProviderConfig) throws IOException {
        return OpenIDConnectionUtils.persistKubeConfigWithUpdatedAuthInfo(currentConfig, a -> {
            Map authProviderConfig = a.getAuthProvider().getConfig();
            authProviderConfig.put(ID_TOKEN_KUBECONFIG, String.valueOf(updatedAuthProviderConfig.get(ID_TOKEN_PARAM)));
            authProviderConfig.put(REFRESH_TOKEN_KUBECONFIG, String.valueOf(updatedAuthProviderConfig.get("refresh_token")));
        });
    }

    public static boolean persistKubeConfigWithUpdatedAuthInfo(Config currentConfig, Consumer<AuthInfo> updateAction) throws IOException {
        if (currentConfig.getFile() == null) {
            return false;
        }
        io.fabric8.kubernetes.api.model.Config config = KubeConfigUtils.parseConfig(currentConfig.getFile());
        NamedContext currentNamedContext = currentConfig.getCurrentContext();
        if (currentNamedContext == null) {
            return false;
        }
        String userName = currentNamedContext.getContext().getUser();
        NamedAuthInfo namedAuthInfo = config.getUsers().stream().filter(n -> n.getName().equals(userName)).findFirst().orElseGet(() -> {
            NamedAuthInfo result = new NamedAuthInfo(userName, new AuthInfo());
            config.getUsers().add(result);
            return result;
        });
        if (namedAuthInfo.getUser() == null) {
            namedAuthInfo.setUser(new AuthInfo());
        }
        updateAction.accept(namedAuthInfo.getUser());
        KubeConfigUtils.persistKubeConfigIntoFile(config, currentConfig.getFile().getAbsolutePath());
        return true;
    }

    private static Map<String, Object> convertJsonStringToMap(String jsonString) throws JsonProcessingException {
        return (Map)Serialization.jsonMapper().readValue(jsonString, Map.class);
    }

    private static HttpClient getDefaultHttpClientWithPemCert(String idpCert, HttpClient.Builder clientBuilder) {
        TrustManager[] trustManagers = null;
        KeyManager[] keyManagers = null;
        String pemCert = new String(Base64.getDecoder().decode(idpCert));
        try {
            trustManagers = SSLUtils.trustManagers(pemCert, null, false, null, null);
            keyManagers = SSLUtils.keyManagers(pemCert, null, null, null, null, null, null, null);
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException | CertificateException | InvalidKeySpecException e) {
            throw new RuntimeException("Could not import idp certificate", e);
        }
        clientBuilder.sslContext(keyManagers, trustManagers);
        return clientBuilder.build();
    }

    private static HttpRequest getTokenRefreshHttpRequest(HttpClient client, String tokenEndpointUrl, String clientId, String refreshToken, String clientSecret) {
        HttpRequest.Builder httpRequestBuilder = client.newHttpRequestBuilder().uri(tokenEndpointUrl);
        Map<String, String> requestBody = OpenIDConnectionUtils.getRequestBodyContentForRefresh(clientId, refreshToken, clientSecret);
        String credentials = Base64.getEncoder().encodeToString((clientId + ':' + clientSecret).getBytes(StandardCharsets.UTF_8));
        httpRequestBuilder.header("Authorization", "Basic " + credentials);
        httpRequestBuilder.post(requestBody);
        return httpRequestBuilder.build();
    }

    private static Map<String, String> getRequestBodyContentForRefresh(String clientId, String refreshToken, String clientSecret) {
        LinkedHashMap<String, String> result = new LinkedHashMap<String, String>();
        result.put("refresh_token", refreshToken);
        result.put(GRANT_TYPE_PARAM, "refresh_token");
        result.put(CLIENT_ID_PARAM, clientId);
        result.put(CLIENT_SECRET_PARAM, clientSecret);
        return result;
    }

    private static CompletableFuture<Map<String, Object>> getOIDCProviderTokenEndpointAndRefreshToken(String issuer, String clientId, String refreshToken, String clientSecret, String idpCert, HttpClient.Builder clientBuilder) {
        HttpClient newClient = OpenIDConnectionUtils.getDefaultHttpClientWithPemCert(idpCert, clientBuilder);
        CompletionStage result = OpenIDConnectionUtils.getOIDCDiscoveryDocumentAsMap(newClient, issuer).thenCompose(wellKnownOpenIdConfiguration -> {
            String oidcTokenEndpoint = OpenIDConnectionUtils.getParametersFromDiscoveryResponse(wellKnownOpenIdConfiguration, TOKEN_ENDPOINT_PARAM);
            return OpenIDConnectionUtils.refreshOidcToken(newClient, clientId, refreshToken, clientSecret, oidcTokenEndpoint);
        });
        ((CompletableFuture)result).whenComplete((s, t) -> newClient.close());
        return result;
    }
}

