/*
 * Decompiled with CFR 0.152.
 */
package io.activej.http;

import io.activej.bytebuf.ByteBuf;
import io.activej.bytebuf.ByteBufPool;
import io.activej.common.Checks;
import io.activej.http.MalformedHttpException;
import java.util.function.Supplier;
import java.util.zip.CRC32;
import java.util.zip.DataFormatException;
import java.util.zip.Deflater;
import java.util.zip.Inflater;

public final class GzipProcessorUtils {
    private static final boolean CHECKS = Checks.isEnabled(GzipProcessorUtils.class);
    private static final byte[] GZIP_HEADER = new byte[]{31, -117, 8, 0, 0, 0, 0, 0, 0, -1};
    private static final int GZIP_HEADER_SIZE = GZIP_HEADER.length;
    private static final int GZIP_FOOTER_SIZE = 8;
    private static final int FHCRC = 2;
    private static final int FEXTRA = 4;
    private static final int FNAME = 8;
    private static final int FCOMMENT = 16;
    private static final int SPARE_BYTES_COUNT = 8;
    private static final double DEFLATE_MAX_BYTES_OVERHEAD_PER_16K_BLOCK = 5.0;

    public static ByteBuf fromGzip(ByteBuf src, int maxMessageSize) throws MalformedHttpException {
        int expectedSize;
        if (CHECKS) {
            Checks.checkArgument((src.readRemaining() > 0 ? 1 : 0) != 0);
        }
        GzipProcessorUtils.check((expectedSize = GzipProcessorUtils.readExpectedInputSize(src)) >= 0, src, () -> new MalformedHttpException("Incorrect uncompressed input size"));
        GzipProcessorUtils.check(expectedSize <= maxMessageSize, src, () -> new MalformedHttpException("Decompressed data size exceeds max expected size"));
        GzipProcessorUtils.processHeader(src);
        ByteBuf dst = ByteBufPool.allocate((int)expectedSize);
        Inflater decompressor = GzipProcessorUtils.ensureDecompressor();
        decompressor.setInput(src.array(), src.head(), src.readRemaining());
        try {
            GzipProcessorUtils.readDecompressedData(decompressor, src, dst, maxMessageSize);
        }
        catch (DataFormatException ignored) {
            src.recycle();
            dst.recycle();
            throw new MalformedHttpException("Data format exception");
        }
        GzipProcessorUtils.check(expectedSize == dst.readRemaining(), src, dst, () -> new MalformedHttpException("Decompressed data size is not equal to input size from GZIP trailer"));
        GzipProcessorUtils.check(src.readRemaining() == 8, src, dst, () -> new MalformedHttpException("Compressed data was not read fully"));
        src.recycle();
        return dst;
    }

    public static ByteBuf toGzip(ByteBuf src) {
        if (CHECKS) {
            Checks.checkArgument((src.readRemaining() >= 0 ? 1 : 0) != 0);
        }
        Deflater compressor = GzipProcessorUtils.ensureCompressor();
        compressor.setInput(src.array(), src.head(), src.readRemaining());
        compressor.finish();
        int dataSize = src.readRemaining();
        int crc = GzipProcessorUtils.getCrc(src, dataSize);
        int maxDataSize = GzipProcessorUtils.estimateMaxCompressedSize(dataSize);
        ByteBuf dst = ByteBufPool.allocate((int)(GZIP_HEADER_SIZE + maxDataSize + 8 + 8));
        dst.put(GZIP_HEADER);
        dst = GzipProcessorUtils.writeCompressedData(compressor, src, dst);
        dst.writeInt(Integer.reverseBytes(crc));
        dst.writeInt(Integer.reverseBytes(dataSize));
        src.recycle();
        return dst;
    }

    private static int readExpectedInputSize(ByteBuf buf) throws MalformedHttpException {
        GzipProcessorUtils.check(buf.readRemaining() >= 8, buf, () -> new MalformedHttpException("Corrupted GZIP header"));
        int w = buf.tail();
        int r = buf.head();
        buf.head(w - 4);
        int bigEndianPosition = buf.readInt();
        buf.head(r);
        return Integer.reverseBytes(bigEndianPosition);
    }

    private static void processHeader(ByteBuf buf) throws MalformedHttpException {
        GzipProcessorUtils.check(buf.readRemaining() >= GZIP_HEADER_SIZE, buf, () -> new MalformedHttpException("Corrupted GZIP header"));
        GzipProcessorUtils.check(buf.readByte() == GZIP_HEADER[0], buf, () -> new MalformedHttpException("Incorrect identification bytes. Not in GZIP format"));
        GzipProcessorUtils.check(buf.readByte() == GZIP_HEADER[1], buf, () -> new MalformedHttpException("Incorrect identification bytes. Not in GZIP format"));
        GzipProcessorUtils.check(buf.readByte() == GZIP_HEADER[2], buf, () -> new MalformedHttpException("Unsupported compression method. Deflate compression required"));
        byte flag = buf.readByte();
        buf.moveHead(6);
        if ((flag & 4) > 0) {
            GzipProcessorUtils.skipExtra(buf);
        }
        if ((flag & 8) > 0) {
            GzipProcessorUtils.skipToTerminatorByte(buf);
        }
        if ((flag & 0x10) > 0) {
            GzipProcessorUtils.skipToTerminatorByte(buf);
        }
        if ((flag & 2) > 0) {
            buf.moveHead(2);
        }
    }

    private static void readDecompressedData(Inflater decompressor, ByteBuf src, ByteBuf dst, int maxSize) throws DataFormatException, MalformedHttpException {
        int totalUncompressedBytesCount = 0;
        int count = decompressor.inflate(dst.array(), dst.tail(), dst.writeRemaining());
        dst.moveTail(count);
        GzipProcessorUtils.check((totalUncompressedBytesCount += count) < maxSize, dst, src, () -> new MalformedHttpException("Decompressed data size exceeds max expected size"));
        GzipProcessorUtils.check(decompressor.finished(), dst, src, () -> new MalformedHttpException("Decompressed data size is not equal to input size from GZIP trailer"));
        int totalRead = decompressor.getTotalIn();
        src.moveHead(totalRead);
    }

    private static int estimateMaxCompressedSize(int dataSize) {
        return (int)((double)dataSize + (double)(dataSize / 16383 + 1) * 5.0);
    }

    private static ByteBuf writeCompressedData(Deflater compressor, ByteBuf src, ByteBuf dst) {
        int unprocessedDataSize = src.readRemaining();
        while (!compressor.finished()) {
            int count = compressor.deflate(dst.array(), dst.tail(), dst.writeRemaining());
            dst.moveTail(count);
            if (compressor.finished()) break;
            int processedDataSize = compressor.getTotalIn();
            int newTailRemaining = GzipProcessorUtils.estimateMaxCompressedSize(unprocessedDataSize - processedDataSize);
            dst = ByteBufPool.ensureWriteRemaining((ByteBuf)dst, (int)newTailRemaining);
        }
        src.moveHead(compressor.getTotalIn());
        return dst;
    }

    private static int getCrc(ByteBuf buf, int dataSize) {
        CRC32 crc32 = new CRC32();
        crc32.update(buf.array(), buf.head(), dataSize);
        return (int)crc32.getValue();
    }

    private static void skipExtra(ByteBuf buf) throws MalformedHttpException {
        GzipProcessorUtils.check(buf.readRemaining() >= 2, buf, () -> new MalformedHttpException("Corrupted GZIP header"));
        short subFieldDataSize = buf.readShort();
        short reversedSubFieldDataSize = Short.reverseBytes(subFieldDataSize);
        GzipProcessorUtils.check(buf.readRemaining() >= reversedSubFieldDataSize, buf, () -> new MalformedHttpException("Corrupted GZIP header"));
        buf.moveHead((int)reversedSubFieldDataSize);
    }

    private static void skipToTerminatorByte(ByteBuf buf) throws MalformedHttpException {
        while (buf.readRemaining() > 0) {
            if (buf.get() != 0) continue;
            return;
        }
        throw new MalformedHttpException("Corrupted GZIP header");
    }

    private static Inflater ensureDecompressor() {
        return new Inflater(true);
    }

    private static Deflater ensureCompressor() {
        return new Deflater(-1, true);
    }

    private static void check(boolean condition, ByteBuf buf1, ByteBuf buf2, Supplier<MalformedHttpException> exceptionSupplier) throws MalformedHttpException {
        if (!condition) {
            buf1.recycle();
            buf2.recycle();
            throw exceptionSupplier.get();
        }
    }

    private static void check(boolean condition, ByteBuf buf, Supplier<MalformedHttpException> exceptionSupplier) throws MalformedHttpException {
        if (!condition) {
            buf.recycle();
            throw exceptionSupplier.get();
        }
    }
}

