/*
 * Decompiled with CFR 0.152.
 */
package io.activej.dns;

import io.activej.async.exception.AsyncCloseException;
import io.activej.async.exception.AsyncTimeoutException;
import io.activej.bytebuf.ByteBuf;
import io.activej.common.Checks;
import io.activej.common.builder.AbstractBuilder;
import io.activej.common.exception.MalformedDataException;
import io.activej.common.inspector.AbstractInspector;
import io.activej.common.inspector.BaseInspector;
import io.activej.dns.IDnsClient;
import io.activej.dns.protocol.DnsProtocol;
import io.activej.dns.protocol.DnsQuery;
import io.activej.dns.protocol.DnsQueryException;
import io.activej.dns.protocol.DnsResponse;
import io.activej.dns.protocol.DnsTransaction;
import io.activej.jmx.api.attribute.JmxAttribute;
import io.activej.jmx.stats.EventStats;
import io.activej.net.socket.udp.IUdpSocket;
import io.activej.net.socket.udp.UdpPacket;
import io.activej.net.socket.udp.UdpSocket;
import io.activej.promise.Promise;
import io.activej.promise.Promises;
import io.activej.promise.SettablePromise;
import io.activej.reactor.AbstractNioReactive;
import io.activej.reactor.Reactive;
import io.activej.reactor.jmx.ReactiveJmxBeanWithStats;
import io.activej.reactor.net.DatagramSocketSettings;
import io.activej.reactor.nio.NioReactor;
import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.nio.channels.DatagramChannel;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DnsClient
extends AbstractNioReactive
implements IDnsClient,
ReactiveJmxBeanWithStats {
    private static final Logger logger = LoggerFactory.getLogger(DnsClient.class);
    private static final boolean CHECKS = Checks.isEnabled(DnsClient.class);
    public static final Duration DEFAULT_TIMEOUT = Duration.ofSeconds(3L);
    private static final int DNS_SERVER_PORT = 53;
    private final Map<DnsTransaction, SettablePromise<DnsResponse>> transactions = new HashMap<DnsTransaction, SettablePromise<DnsResponse>>();
    private final InetSocketAddress dnsServerAddress;
    private DatagramSocketSettings datagramSocketSettings = DatagramSocketSettings.create();
    private Duration timeout = DEFAULT_TIMEOUT;
    @Nullable
    private IUdpSocket socket;
    @Nullable
    private UdpSocket.Inspector socketInspector;
    @Nullable
    private Inspector inspector;

    private DnsClient(NioReactor reactor, InetSocketAddress dnsServerAddress) {
        super(reactor);
        this.dnsServerAddress = dnsServerAddress;
    }

    public static DnsClient create(NioReactor reactor, InetSocketAddress dnsServerAddress) {
        return (DnsClient)DnsClient.builder(reactor, dnsServerAddress).build();
    }

    public static DnsClient create(NioReactor reactor, InetAddress dnsServerAddress) {
        return (DnsClient)DnsClient.builder(reactor, dnsServerAddress).build();
    }

    public static Builder builder(NioReactor reactor, InetSocketAddress dnsServerAddress) {
        return new DnsClient(reactor, dnsServerAddress).new Builder();
    }

    public static Builder builder(NioReactor reactor, InetAddress dnsServerAddress) {
        InetSocketAddress address = new InetSocketAddress(dnsServerAddress, 53);
        return new DnsClient(reactor, address).new Builder();
    }

    @Override
    public void close() {
        Reactive.checkInReactorThread((Reactive)this);
        if (this.socket == null) {
            return;
        }
        this.socket.close();
        this.socket = null;
        AsyncCloseException closeException = new AsyncCloseException();
        this.transactions.values().forEach(s -> s.setException((Exception)closeException));
    }

    private Promise<IUdpSocket> getSocket() {
        IUdpSocket socket = this.socket;
        if (socket != null) {
            return Promise.of((Object)socket);
        }
        try {
            logger.trace("Incoming query, opening UDP socket");
            DatagramChannel channel = NioReactor.createDatagramChannel((DatagramSocketSettings)this.datagramSocketSettings, null, (InetSocketAddress)this.dnsServerAddress);
            return UdpSocket.connect((NioReactor)this.reactor, (DatagramChannel)channel).map(s -> {
                if (this.socketInspector != null) {
                    this.socketInspector.onCreate(s);
                    s.setInspector(this.socketInspector);
                }
                this.socket = s;
                return this.socket;
            });
        }
        catch (IOException e) {
            logger.error("UDP socket creation failed.", (Throwable)e);
            return Promise.ofException((Exception)e);
        }
    }

    @Override
    public Promise<DnsResponse> resolve(DnsQuery query) {
        DnsResponse fromQuery;
        if (CHECKS) {
            Reactive.checkInReactorThread((Reactive)this);
        }
        if ((fromQuery = IDnsClient.resolveFromQuery(query)) != null) {
            logger.trace("{} already contained an IP address within itself", (Object)query);
            return Promise.of((Object)fromQuery);
        }
        int labelSize = 0;
        String domainName = query.getDomainName();
        for (int i = 0; i < domainName.length(); ++i) {
            if (domainName.charAt(i) == '.') {
                labelSize = 0;
                continue;
            }
            if (++labelSize <= 63) continue;
            return Promise.ofException((Exception)new IllegalArgumentException("Label size cannot exceed 63 octets"));
        }
        return this.getSocket().then(socket -> {
            logger.trace("Resolving {} with DNS server {}", (Object)query, (Object)this.dnsServerAddress);
            DnsTransaction transaction = DnsTransaction.of(DnsProtocol.generateTransactionId(), query);
            SettablePromise promise = new SettablePromise();
            this.transactions.put(transaction, (SettablePromise<DnsResponse>)promise);
            ByteBuf payload = DnsProtocol.createDnsQueryPayload(transaction);
            if (this.inspector != null) {
                this.inspector.onDnsQuery(query, payload);
            }
            socket.send(UdpPacket.of((ByteBuf)payload, (InetSocketAddress)this.dnsServerAddress));
            socket.receive().whenResult(packet -> {
                try {
                    DnsResponse queryResult = DnsProtocol.readDnsResponse(packet.getBuf());
                    SettablePromise<DnsResponse> cb = this.transactions.remove(queryResult.getTransaction());
                    if (cb == null) {
                        logger.warn("Received a DNS response that had no listener (most likely because it timed out) : {}", (Object)queryResult);
                        return;
                    }
                    if (queryResult.isSuccessful()) {
                        cb.set((Object)queryResult);
                    } else {
                        cb.setException((Exception)new DnsQueryException(queryResult));
                    }
                    this.closeIfDone();
                }
                catch (MalformedDataException e) {
                    logger.warn("Received a UDP packet than cannot be decoded as a DNS server response.", (Throwable)e);
                }
                finally {
                    packet.recycle();
                }
            });
            return Promises.timeout((Duration)this.timeout, (Promise)promise).thenCallback((queryResult, cb) -> {
                if (this.inspector != null) {
                    this.inspector.onDnsQueryResult(query, (DnsResponse)queryResult);
                }
                logger.trace("DNS query {} resolved as {}", (Object)query, (Object)queryResult.getRecord());
                cb.set(queryResult);
            }, (e, cb) -> {
                if (e instanceof AsyncTimeoutException) {
                    if (this.inspector != null) {
                        this.inspector.onDnsQueryExpiration(query);
                    }
                    logger.trace("{} timed out", (Object)query);
                    e = new DnsQueryException(DnsResponse.ofFailure(transaction, DnsProtocol.ResponseErrorCode.TIMED_OUT));
                    this.transactions.remove(transaction);
                    this.closeIfDone();
                } else if (this.inspector != null) {
                    this.inspector.onDnsQueryError(query, (Exception)e);
                }
                cb.setException(e);
            });
        });
    }

    private void closeIfDone() {
        if (!this.transactions.isEmpty()) {
            return;
        }
        logger.trace("All queries are completed, closing UDP socket");
        this.close();
    }

    @JmxAttribute
    @Nullable
    public UdpSocket.JmxInspector getSocketStats() {
        return (UdpSocket.JmxInspector)BaseInspector.lookup((BaseInspector)this.socketInspector, UdpSocket.JmxInspector.class);
    }

    @JmxAttribute(name="")
    @Nullable
    public JmxInspector getStats() {
        return (JmxInspector)BaseInspector.lookup((BaseInspector)this.inspector, JmxInspector.class);
    }

    public final class Builder
    extends AbstractBuilder<Builder, DnsClient> {
        private Builder() {
        }

        public Builder withDatagramSocketSetting(DatagramSocketSettings setting) {
            Builder.checkNotBuilt((AbstractBuilder)this);
            DnsClient.this.datagramSocketSettings = setting;
            return this;
        }

        public Builder withTimeout(Duration timeout) {
            Builder.checkNotBuilt((AbstractBuilder)this);
            DnsClient.this.timeout = timeout;
            return this;
        }

        public Builder withInspector(Inspector inspector) {
            Builder.checkNotBuilt((AbstractBuilder)this);
            DnsClient.this.inspector = inspector;
            return this;
        }

        public Builder withSocketInspector(UdpSocket.Inspector socketInspector) {
            Builder.checkNotBuilt((AbstractBuilder)this);
            DnsClient.this.socketInspector = socketInspector;
            return this;
        }

        protected DnsClient doBuild() {
            return DnsClient.this;
        }
    }

    public static interface Inspector
    extends BaseInspector<Inspector> {
        public void onDnsQuery(DnsQuery var1, ByteBuf var2);

        public void onDnsQueryResult(DnsQuery var1, DnsResponse var2);

        public void onDnsQueryError(DnsQuery var1, Exception var2);

        public void onDnsQueryExpiration(DnsQuery var1);
    }

    public static class JmxInspector
    extends AbstractInspector<Inspector>
    implements Inspector {
        private static final Duration SMOOTHING_WINDOW = Duration.ofMinutes(1L);
        private final EventStats queries = EventStats.create((Duration)SMOOTHING_WINDOW);
        private final EventStats failedQueries = EventStats.create((Duration)SMOOTHING_WINDOW);
        private final EventStats expirations = EventStats.create((Duration)SMOOTHING_WINDOW);

        @Override
        public void onDnsQuery(DnsQuery query, ByteBuf payload) {
            this.queries.recordEvent();
        }

        @Override
        public void onDnsQueryResult(DnsQuery query, DnsResponse result) {
            if (!result.isSuccessful()) {
                this.failedQueries.recordEvent();
            }
        }

        @Override
        public void onDnsQueryError(DnsQuery query, Exception e) {
            this.failedQueries.recordEvent();
        }

        @Override
        public void onDnsQueryExpiration(DnsQuery query) {
            this.expirations.recordEvent();
        }

        @JmxAttribute
        public EventStats getQueries() {
            return this.queries;
        }

        @JmxAttribute
        public EventStats getFailedQueries() {
            return this.failedQueries;
        }

        @JmxAttribute
        public EventStats getExpirations() {
            return this.expirations;
        }
    }
}

