/*
 * Decompiled with CFR 0.152.
 */
package io.activej.bytebuf;

import io.activej.bytebuf.ByteBuf;
import io.activej.common.ApplicationSettings;
import io.activej.common.Checks;
import io.activej.common.MemSize;
import io.activej.common.concurrent.ObjectPool;
import java.io.Serializable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

public final class ByteBufPool {
    private static final int NUMBER_OF_SLABS = 33;
    private static final int MIN_SIZE;
    private static final int MAX_SIZE;
    private static final boolean MIN_MAX_CHECKS;
    static final boolean REGISTRY;
    static final boolean STATS;
    static final boolean CLEAR_ON_RECYCLE;
    static final boolean USE_WATCHDOG;
    static final Duration WATCHDOG_INTERVAL;
    static final Duration WATCHDOG_SMOOTHING_WINDOW;
    static final double WATCHDOG_ERROR_MARGIN;
    private static final double SMOOTHING_COEFF;
    static final ObjectPool<ByteBuf>[] slabs;
    static final SlabStats[] slabStats;
    static final AtomicInteger[] created;
    static final AtomicInteger[] reused;
    private static final ByteBufPoolStats stats;
    private static final Map<ByteBuf, Entry> allocateRegistry;
    private static final Map<ByteBuf, Entry> recycleRegistry;

    private ByteBufPool() {
    }

    public static ByteBuf allocate(int size) {
        assert (size >= 0) : "Allocating ByteBuf with negative size";
        if (MIN_MAX_CHECKS && (MIN_SIZE != 0 && size < MIN_SIZE || MAX_SIZE != 0 && size >= MAX_SIZE)) {
            return ByteBuf.wrapForWriting(new byte[size]);
        }
        int index = 32 - Integer.numberOfLeadingZeros(size - 1);
        ObjectPool<ByteBuf> slab = slabs[index];
        ByteBuf buf = (ByteBuf)slab.poll();
        if (buf != null) {
            if (USE_WATCHDOG) {
                int slabSize = slab.size();
                ByteBufPool.slabStats[index].min = Math.min(ByteBufPool.slabStats[index].min, slabSize);
            }
            if (ByteBuf.CHECK_RECYCLE && buf.refs != -1) {
                throw ByteBufPool.onByteBufRecycled(buf);
            }
            buf.tail = 0;
            buf.head = 0;
            buf.refs = 1;
            if (STATS) {
                ByteBufPool.recordReuse(index);
            }
        } else {
            if (USE_WATCHDOG) {
                ByteBufPool.slabStats[index].min = 0;
            }
            buf = ByteBuf.wrapForWriting(new byte[index == 32 ? 0 : 1 << index]);
            buf.refs = 1;
            if (STATS) {
                ByteBufPool.recordNew(index);
            }
        }
        if (REGISTRY) {
            allocateRegistry.put(buf, ByteBufPool.buildRegistryEntry(buf));
        }
        return buf;
    }

    private static void recordNew(int index) {
        created[index].incrementAndGet();
    }

    private static void recordReuse(int index) {
        reused[index].incrementAndGet();
    }

    private static Entry buildRegistryEntry(ByteBuf buf) {
        return new Entry(buf.array.length, System.currentTimeMillis(), Thread.currentThread(), StackWalker.getInstance().walk(frames -> frames.skip(3L).toList()));
    }

    static AssertionError onByteBufRecycled(ByteBuf buf) {
        int slab = 32 - Integer.numberOfLeadingZeros(buf.array.length - 1);
        ObjectPool<ByteBuf> pool = slabs[slab];
        pool.clear();
        if (USE_WATCHDOG) {
            slabStats[slab].clear();
        }
        return new AssertionError((Object)("Attempt to use recycled ByteBuf" + (REGISTRY ? ByteBufPool.getByteBufTrace(buf) : "")));
    }

    static String getByteBufTrace(ByteBuf buf) {
        Entry allocated = allocateRegistry.get(buf);
        Entry recycled = recycleRegistry.get(buf);
        if (allocated == null && recycled == null) {
            return "";
        }
        return "\nAllocated: " + allocated + "\nRecycled: " + recycled;
    }

    public static ByteBuf allocateExact(int size) {
        ByteBuf buf = ByteBufPool.allocate(size);
        int d = buf.writeRemaining() - size;
        buf.tail(d);
        buf.head(d);
        return buf;
    }

    public static ByteBuf allocate(MemSize size) {
        return ByteBufPool.allocate(size.toInt());
    }

    public static ByteBuf allocateExact(MemSize size) {
        return ByteBufPool.allocateExact(size.toInt());
    }

    static void recycle(ByteBuf buf) {
        int slab = 32 - Integer.numberOfLeadingZeros(buf.array.length - 1);
        ObjectPool<ByteBuf> pool = slabs[slab];
        if (CLEAR_ON_RECYCLE) {
            Arrays.fill(buf.array(), (byte)0);
        }
        if (REGISTRY) {
            recycleRegistry.put(buf, ByteBufPool.buildRegistryEntry(buf));
            allocateRegistry.remove(buf);
        }
        pool.offer((Object)buf);
    }

    public static ByteBuf ensureWriteRemaining(ByteBuf buf, int newWriteRemaining) {
        return ByteBufPool.ensureWriteRemaining(buf, 0, newWriteRemaining);
    }

    public static ByteBuf ensureWriteRemaining(ByteBuf buf, int minSize, int newWriteRemaining) {
        if (newWriteRemaining == 0) {
            return buf;
        }
        if (buf.writeRemaining() < newWriteRemaining || buf instanceof ByteBuf.ByteBufSlice) {
            ByteBuf newBuf = ByteBufPool.allocate(Math.max(minSize, newWriteRemaining + buf.readRemaining()));
            newBuf.put(buf);
            buf.recycle();
            return newBuf;
        }
        return buf;
    }

    public static ByteBuf append(ByteBuf to, ByteBuf from) {
        Checks.checkArgument((!to.isRecycled() && !from.isRecycled() ? 1 : 0) != 0);
        if (to.readRemaining() == 0) {
            to.recycle();
            return from;
        }
        to = ByteBufPool.ensureWriteRemaining(to, from.readRemaining());
        to.put(from);
        from.recycle();
        return to;
    }

    public static ByteBuf append(ByteBuf to, byte[] from, int offset, int length) {
        Checks.checkArgument((!to.isRecycled() ? 1 : 0) != 0);
        to = ByteBufPool.ensureWriteRemaining(to, length);
        to.put(from, offset, length);
        return to;
    }

    public static ByteBuf append(ByteBuf to, byte[] from) {
        return ByteBufPool.append(to, from, 0, from.length);
    }

    public static void clear() {
        for (int i = 0; i < 33; ++i) {
            slabs[i].clear();
            created[i].set(0);
            reused[i].set(0);
            if (!USE_WATCHDOG) continue;
            slabStats[i].clear();
        }
        allocateRegistry.clear();
        recycleRegistry.clear();
    }

    public static ByteBufPoolStats getStats() {
        return stats;
    }

    private static void updateStats() {
        for (int i = 0; i < slabs.length; ++i) {
            SlabStats stats = slabStats[i];
            ObjectPool<ByteBuf> slab = slabs[i];
            int min = stats.min;
            stats.min = slab.size();
            double error = Math.abs(stats.estimatedMin - (double)min);
            stats.estimatedError += (error - stats.estimatedError) * SMOOTHING_COEFF;
            if ((double)min < stats.estimatedMin) {
                stats.estimatedMin = min;
                continue;
            }
            stats.estimatedMin += ((double)min - stats.estimatedMin) * SMOOTHING_COEFF;
        }
    }

    private static void evict() {
        for (int i = 0; i < slabs.length; ++i) {
            ObjectPool<ByteBuf> slab = slabs[i];
            SlabStats stats = slabStats[i];
            int evictCount = (int)Math.round(stats.estimatedMin - stats.estimatedError * WATCHDOG_ERROR_MARGIN);
            stats.evictedLast = 0;
            for (int j = 0; j < evictCount; ++j) {
                ByteBuf buf = (ByteBuf)slab.poll();
                if (buf == null) {
                    if (!USE_WATCHDOG) break;
                    stats.min = 0;
                    break;
                }
                if (USE_WATCHDOG) {
                    int slabSize = slab.size();
                    stats.min = Math.min(stats.min, slabSize);
                }
                stats.estimatedMin -= 1.0;
                ++stats.evictedLast;
                if (!REGISTRY) continue;
                recycleRegistry.remove(buf);
            }
            stats.evictedTotal += stats.evictedLast;
            stats.evictedMax = Math.max(stats.evictedLast, stats.evictedMax);
        }
    }

    static {
        int i;
        MIN_SIZE = ApplicationSettings.getMemSize(ByteBufPool.class, (String)"minSize", (MemSize)MemSize.ZERO).toInt();
        MAX_SIZE = ApplicationSettings.getMemSize(ByteBufPool.class, (String)"maxSize", (MemSize)MemSize.ZERO).toInt();
        MIN_MAX_CHECKS = MIN_SIZE != 0 || MAX_SIZE != 0;
        REGISTRY = ApplicationSettings.getBoolean(ByteBufPool.class, (String)"registry", (Boolean)false);
        STATS = ApplicationSettings.getBoolean(ByteBufPool.class, (String)"stats", (Boolean)false);
        CLEAR_ON_RECYCLE = ApplicationSettings.getBoolean(ByteBufPool.class, (String)"clearOnRecycle", (Boolean)false);
        USE_WATCHDOG = ApplicationSettings.getBoolean(ByteBufPool.class, (String)"useWatchdog", (Boolean)false);
        WATCHDOG_INTERVAL = ApplicationSettings.getDuration(ByteBufPool.class, (String)"watchdogInterval", (Duration)Duration.ofSeconds(2L));
        WATCHDOG_SMOOTHING_WINDOW = ApplicationSettings.getDuration(ByteBufPool.class, (String)"watchdogSmoothingWindow", (Duration)Duration.ofSeconds(10L));
        WATCHDOG_ERROR_MARGIN = ApplicationSettings.getDouble(ByteBufPool.class, (String)"watchdogErrorMargin", (Double)4.0);
        SMOOTHING_COEFF = 1.0 - Math.pow(0.5, (double)WATCHDOG_INTERVAL.toMillis() / (double)WATCHDOG_SMOOTHING_WINDOW.toMillis());
        stats = new ByteBufPoolStats();
        allocateRegistry = Collections.synchronizedMap(new WeakHashMap());
        recycleRegistry = Collections.synchronizedMap(new WeakHashMap());
        slabs = new ObjectPool[33];
        slabStats = new SlabStats[33];
        created = new AtomicInteger[33];
        reused = new AtomicInteger[33];
        for (i = 0; i < 33; ++i) {
            ByteBufPool.slabs[i] = new ObjectPool();
            ByteBufPool.created[i] = new AtomicInteger();
            ByteBufPool.reused[i] = new AtomicInteger();
        }
        if (USE_WATCHDOG) {
            for (i = 0; i < 33; ++i) {
                ByteBufPool.slabStats[i] = new SlabStats();
            }
            Thread watchdogThread = new Thread(() -> {
                while (true) {
                    ByteBufPool.updateStats();
                    ByteBufPool.evict();
                    try {
                        Thread.sleep(WATCHDOG_INTERVAL.toMillis());
                    }
                    catch (InterruptedException ignored) {
                        Thread.currentThread().interrupt();
                        return;
                    }
                }
            }, "bytebufpool-watchdog-thread");
            watchdogThread.setDaemon(true);
            watchdogThread.setPriority(1);
            watchdogThread.start();
        }
    }

    public static final class SlabStats {
        volatile int min = 0;
        double estimatedMin;
        int evictedTotal;
        int evictedLast;
        int evictedMax;
        double estimatedError;

        void clear() {
            this.evictedMax = 0;
            this.evictedLast = 0;
            this.evictedTotal = 0;
            this.estimatedMin = this.estimatedError = (double)0;
            this.min = 0;
        }

        public String toString() {
            return "SlabStats{estimatedMin=" + this.estimatedMin + ", estimatedError=" + this.estimatedError + ", evictedTotal=" + this.evictedTotal + ", evictedLast=" + this.evictedLast + ", evictedMax=" + this.evictedMax + "}";
        }
    }

    public static final class Entry {
        final int size;
        final long timestamp;
        final Thread thread;
        final List<StackWalker.StackFrame> stackTrace;

        Entry(int size, long timestamp, Thread thread, List<StackWalker.StackFrame> stackTrace) {
            this.size = size;
            this.timestamp = timestamp;
            this.thread = thread;
            this.stackTrace = stackTrace;
        }

        public int getSize() {
            return this.size;
        }

        public long getTimestamp() {
            return this.timestamp;
        }

        public String getAge() {
            return Duration.ofMillis(System.currentTimeMillis() - this.timestamp).toString();
        }

        public String getThread() {
            return this.thread.toString();
        }

        public List<String> getStackTrace() {
            return this.stackTrace.stream().map(Object::toString).toList();
        }

        public String toString() {
            return "{size=" + this.size + ", timestamp=" + this.timestamp + ", thread=" + this.thread + ", stackTrace=" + this.stackTrace + "}";
        }
    }

    public static final class ByteBufPoolStats
    implements ByteBufPoolStatsMXBean {
        @Override
        public int getCreatedItems() {
            return Arrays.stream(created).mapToInt(AtomicInteger::get).sum();
        }

        @Override
        public int getReusedItems() {
            return Arrays.stream(reused).mapToInt(AtomicInteger::get).sum();
        }

        @Override
        public int getPoolItems() {
            return Arrays.stream(slabs).mapToInt(ObjectPool::size).sum();
        }

        public String getPoolItemsString() {
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < 33; ++i) {
                int poolItems;
                int createdItems = created[i].get();
                if (createdItems == (poolItems = slabs[i].size())) continue;
                sb.append(String.format("Slab %d (%d) ", i, 1 << i)).append(" created: " + createdItems).append(" pool: " + poolItems + "\n");
            }
            return sb.toString();
        }

        @Override
        public long getPoolSize() {
            long result = 0L;
            for (int i = 0; i < slabs.length - 1; ++i) {
                long slabSize = 1L << i;
                result += slabSize * (long)slabs[i].size();
            }
            return result;
        }

        @Override
        public long getPoolSizeKB() {
            return this.getPoolSize() / 1024L;
        }

        @Override
        public long getTotalSlabMins() {
            if (!USE_WATCHDOG) {
                return -1L;
            }
            long totalSlabMins = 0L;
            for (SlabStats slabStat : slabStats) {
                totalSlabMins += (long)slabStat.min;
            }
            return totalSlabMins;
        }

        @Override
        public long getTotalEvicted() {
            if (!USE_WATCHDOG) {
                return -1L;
            }
            long totalEvicted = 0L;
            for (SlabStats slabStat : slabStats) {
                totalEvicted += (long)slabStat.evictedTotal;
            }
            return totalEvicted;
        }

        public Map<ByteBuf, Entry> getUnrecycledBufs() {
            return new HashMap<ByteBuf, Entry>(allocateRegistry);
        }

        @Override
        public List<Entry> queryUnrecycledBufs(int limit) {
            if (limit < 1) {
                throw new IllegalArgumentException("Limit must be >= 1");
            }
            Map<ByteBuf, Entry> danglingBufs = this.getUnrecycledBufs();
            return danglingBufs.values().stream().sorted(Comparator.comparingLong(Entry::getTimestamp)).limit(limit).collect(Collectors.toList());
        }

        @Override
        public List<String> getPoolSlabs() {
            ArrayList<String> result = new ArrayList<String>(slabs.length + 1);
            Object header = "SlotSize,Created,Reused,InPool,Total(Kb)";
            if (USE_WATCHDOG) {
                header = (String)header + ",Min,EstMin,Error,Evicted";
            }
            result.add((String)header);
            for (int i = 0; i < slabs.length; ++i) {
                int idx = (i + 32) % slabs.length;
                long slabSize = idx == 32 ? 0L : 1L << idx;
                ObjectPool<ByteBuf> slab = slabs[idx];
                int count = slab.size();
                String slabInfo = slabSize + "," + (Serializable)(STATS ? created[idx] : "-") + "," + (Serializable)(STATS ? reused[idx] : "-") + "," + count + "," + slabSize * (long)count / 1024L;
                if (USE_WATCHDOG) {
                    SlabStats slabStat = slabStats[idx];
                    slabInfo = slabInfo + "," + slabStat.min + "," + String.format("%.1f", slabStat.estimatedMin) + "," + String.format("%.1f", slabStat.estimatedError) + "," + slabStat.evictedTotal;
                }
                result.add(slabInfo);
            }
            return result;
        }

        @Override
        public void clear() {
            ByteBufPool.clear();
        }

        @Override
        public void clearRegistry() {
            allocateRegistry.clear();
            recycleRegistry.clear();
        }
    }

    public static interface ByteBufPoolStatsMXBean {
        public int getCreatedItems();

        public int getReusedItems();

        public int getPoolItems();

        public long getPoolSize();

        public long getPoolSizeKB();

        public long getTotalSlabMins();

        public long getTotalEvicted();

        public List<String> getPoolSlabs();

        public List<Entry> queryUnrecycledBufs(int var1);

        public void clear();

        public void clearRegistry();
    }
}

