/*
 * Decompiled with CFR 0.152.
 */
package dev.langchain4j.community.store.embedding.alloydb;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.alloydb.ConnectorConfig;
import com.google.cloud.alloydb.ConnectorRegistry;
import com.google.cloud.alloydb.RefreshStrategy;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import dev.langchain4j.community.store.embedding.alloydb.EmbeddingStoreConfig;
import dev.langchain4j.community.store.embedding.alloydb.MetadataColumn;
import dev.langchain4j.internal.Utils;
import dev.langchain4j.internal.ValidationUtils;
import java.io.IOException;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Map;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AlloyDBEngine {
    private static final String USER_AGENT = "langchain4j-alloydb-pg";
    private static final Logger log = LoggerFactory.getLogger((String)AlloyDBEngine.class.getName());
    private ConnectorConfig namedConnectorConfig;
    private final HikariDataSource dataSource;
    static final ObjectMapper OBJECT_MAPPER = new ObjectMapper().enable(SerializationFeature.INDENT_OUTPUT);

    public AlloyDBEngine(Builder builder) {
        if (Utils.isNotNullOrBlank((String)builder.host) && (Utils.isNotNullOrBlank((String)builder.projectId) || Utils.isNotNullOrBlank((String)builder.cluster))) {
            throw new IllegalStateException("Connect directly to an instance using projectId, region, cluster, instance, and database params or connect via an IP Address using host, user, password, and database params");
        }
        if (Utils.isNotNullOrBlank((String)builder.cluster)) {
            boolean enableIAMAuth;
            String authId = builder.user;
            if (Utils.isNullOrBlank((String)authId) && Utils.isNullOrBlank((String)builder.password)) {
                enableIAMAuth = true;
                if (Utils.isNotNullOrBlank((String)builder.iamAccountEmail)) {
                    log.debug("Found iamAccountEmail");
                    authId = builder.iamAccountEmail;
                } else {
                    log.debug("Retrieving IAM principal email");
                    authId = this.getIAMPrincipalEmail().replace(".gserviceaccount.com", "");
                }
            } else if (Utils.isNotNullOrBlank((String)authId) && Utils.isNotNullOrBlank((String)builder.password)) {
                enableIAMAuth = false;
                log.debug("Found user and password, IAM Auth disabled");
            } else {
                throw new IllegalStateException("Either one of user or password is blank, expected both user and password to be valid credentials or empty");
            }
            String instanceName = "projects/" + ValidationUtils.ensureNotBlank((String)builder.projectId, (String)"projectId") + "/locations/" + ValidationUtils.ensureNotBlank((String)builder.region, (String)"region") + "/clusters/" + ValidationUtils.ensureNotBlank((String)builder.cluster, (String)"cluster") + "/instances/" + ValidationUtils.ensureNotBlank((String)builder.instance, (String)"instance");
            this.dataSource = this.createConnectorDataSource(builder.database, authId, builder.password, instanceName, builder.ipType, enableIAMAuth);
        } else {
            this.dataSource = this.createUrlDataSource(builder.database, builder.user, builder.password, builder.host, builder.port);
        }
    }

    private HikariDataSource createConnectorDataSource(String database, String user, String password, String instanceName, String ipType, boolean enableIAMAuth) {
        if (this.namedConnectorConfig == null) {
            this.namedConnectorConfig = new ConnectorConfig.Builder().withRefreshStrategy(RefreshStrategy.LAZY).build();
            ConnectorRegistry.addArtifactId((String)USER_AGENT);
            ConnectorRegistry.register((String)"langchain-connector", (ConnectorConfig)this.namedConnectorConfig);
        }
        HikariConfig config = new HikariConfig();
        config.setJdbcUrl(String.format("jdbc:postgresql:///%s", ValidationUtils.ensureNotBlank((String)database, (String)"database")));
        config.setUsername(ValidationUtils.ensureNotBlank((String)user, (String)"user"));
        if (enableIAMAuth) {
            config.addDataSourceProperty("alloydbEnableIAMAuth", (Object)"true");
        } else {
            config.setPassword(ValidationUtils.ensureNotBlank((String)password, (String)"password"));
        }
        config.addDataSourceProperty("socketFactory", (Object)"com.google.cloud.alloydb.SocketFactory");
        config.addDataSourceProperty("alloydbInstanceName", (Object)ValidationUtils.ensureNotBlank((String)instanceName, (String)"instanceName"));
        config.addDataSourceProperty("alloydbIpType", (Object)ValidationUtils.ensureNotBlank((String)ipType, (String)"ipType"));
        config.addDataSourceProperty("alloydbNamedConnector", (Object)"langchain-connector");
        return new HikariDataSource(config);
    }

    private HikariDataSource createUrlDataSource(String database, String user, String password, String host, Integer port) {
        HikariConfig config = new HikariConfig();
        config.setJdbcUrl(String.format("jdbc:postgresql://%s:%d/%s", ValidationUtils.ensureNotBlank((String)host, (String)"host"), port, ValidationUtils.ensureNotBlank((String)database, (String)"database")));
        config.setUsername(ValidationUtils.ensureNotBlank((String)user, (String)"user"));
        config.setPassword(ValidationUtils.ensureNotBlank((String)password, (String)"password"));
        return new HikariDataSource(config);
    }

    private String getIAMPrincipalEmail() {
        try {
            GoogleCredentials credentials = GoogleCredentials.getApplicationDefault();
            String accessToken = credentials.refreshAccessToken().getTokenValue();
            String oauth2APIURL = "https://oauth2.googleapis.com/tokeninfo?access_token=" + accessToken;
            byte[] responseBytes = Utils.readBytes((String)oauth2APIURL);
            Map responseJson = (Map)OBJECT_MAPPER.readValue(new String(responseBytes), Map.class);
            if (responseJson.containsKey("email")) {
                return (String)responseJson.get("email");
            }
            throw new RuntimeException("unable to load IAM principal email");
        }
        catch (IOException e) {
            throw new RuntimeException("unable to load IAM principal email", e);
        }
    }

    public Connection getConnection() throws SQLException {
        Connection connection = this.dataSource.getConnection();
        return connection;
    }

    public void initVectorStoreTable(EmbeddingStoreConfig embeddingStoreConfig) {
        try (Connection connection = this.getConnection();
             Statement statement = connection.createStatement();){
            statement.executeUpdate("CREATE EXTENSION IF NOT EXISTS vector");
            if (embeddingStoreConfig.getOverwriteExisting().booleanValue()) {
                statement.executeUpdate(String.format("DROP TABLE IF EXISTS \"%s\".\"%s\"", embeddingStoreConfig.getSchemaName(), embeddingStoreConfig.getTableName()));
            }
            Object metadataClause = "";
            if (embeddingStoreConfig.getMetadataColumns() != null && !embeddingStoreConfig.getMetadataColumns().isEmpty()) {
                metadataClause = (String)metadataClause + String.format(", %s", embeddingStoreConfig.getMetadataColumns().stream().map(MetadataColumn::generateColumnString).collect(Collectors.joining(", ")));
            }
            if (embeddingStoreConfig.getStoreMetadata().booleanValue()) {
                metadataClause = (String)metadataClause + String.format(", %s", new MetadataColumn(embeddingStoreConfig.getMetadataJsonColumn(), "JSON", true).generateColumnString());
            }
            String query = String.format("CREATE TABLE \"%s\".\"%s\" (\"%s\" UUID PRIMARY KEY, \"%s\" TEXT NULL, \"%s\" vector(%d) NOT NULL%s)", embeddingStoreConfig.getSchemaName(), embeddingStoreConfig.getTableName(), embeddingStoreConfig.getIdColumn(), embeddingStoreConfig.getContentColumn(), embeddingStoreConfig.getEmbeddingColumn(), embeddingStoreConfig.getVectorSize(), metadataClause);
            statement.executeUpdate(query);
        }
        catch (SQLException ex) {
            throw new RuntimeException(String.format("Failed to initialize vector store table: \"%s\".\"%s\"", embeddingStoreConfig.getSchemaName(), embeddingStoreConfig.getTableName()), ex);
        }
    }

    public void close() {
        this.dataSource.close();
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String projectId;
        private String region;
        private String cluster;
        private String instance;
        private String database;
        private String host;
        private Integer port = 5432;
        private String user;
        private String password;
        private String ipType = "public";
        private String iamAccountEmail;

        public Builder projectId(String projectId) {
            this.projectId = projectId;
            return this;
        }

        public Builder instance(String instance) {
            this.instance = instance;
            return this;
        }

        public Builder region(String region) {
            this.region = region;
            return this;
        }

        public Builder cluster(String cluster) {
            this.cluster = cluster;
            return this;
        }

        public Builder database(String database) {
            this.database = database;
            return this;
        }

        public Builder user(String user) {
            this.user = user;
            return this;
        }

        public Builder password(String password) {
            this.password = password;
            return this;
        }

        public Builder ipType(String ipType) {
            this.ipType = ipType;
            return this;
        }

        public Builder iamAccountEmail(String iamAccountEmail) {
            this.iamAccountEmail = iamAccountEmail;
            return this;
        }

        public Builder host(String host) {
            this.host = host;
            return this;
        }

        public Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public AlloyDBEngine build() {
            return new AlloyDBEngine(this);
        }
    }
}

