package de.flapdoodle.embed.process.transitions;

import de.flapdoodle.embed.process.archives.ExtractedFileSet;
import de.flapdoodle.embed.process.config.SupportConfig;
import de.flapdoodle.embed.process.io.ProcessOutput;
import de.flapdoodle.embed.process.types.*;
import de.flapdoodle.embed.process.types.ExecutedProcess;
import de.flapdoodle.embed.process.types.ExecutedProcessFactory;
import de.flapdoodle.embed.process.types.ProcessConfig;
import de.flapdoodle.embed.process.types.RunningProcess;
import de.flapdoodle.embed.process.types.RunningProcessFactory;
import de.flapdoodle.reverse.StateID;
import de.flapdoodle.reverse.naming.HasLabel;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link Executer}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableExecuter.builder()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableExecuter<R extends RunningProcess, T extends ExecutedProcess>
    extends Executer<R, T> {
  private final StateID<ProcessWorkingDir> processWorkingDir;
  private final String transitionLabel;
  private final StateID<ExtractedFileSet> processExecutable;
  private final StateID<ProcessConfig> processConfig;
  private final StateID<ProcessEnv> processEnv;
  private final StateID<ProcessArguments> arguments;
  private final StateID<ProcessOutput> processOutput;
  private final StateID<SupportConfig> supportConfig;
  private final StateID<T> destination;
  private final RunningProcessFactory<R> runningProcessFactory;
  private final ExecutedProcessFactory<R, T> executedProcessFactory;

  private ImmutableExecuter(ImmutableExecuter.Builder<R, T> builder) {
    this.destination = builder.destination;
    this.runningProcessFactory = builder.runningProcessFactory;
    this.executedProcessFactory = builder.executedProcessFactory;
    if (builder.processWorkingDir != null) {
      initShim.processWorkingDir(builder.processWorkingDir);
    }
    if (builder.transitionLabel != null) {
      initShim.transitionLabel(builder.transitionLabel);
    }
    if (builder.processExecutable != null) {
      initShim.processExecutable(builder.processExecutable);
    }
    if (builder.processConfig != null) {
      initShim.processConfig(builder.processConfig);
    }
    if (builder.processEnv != null) {
      initShim.processEnv(builder.processEnv);
    }
    if (builder.arguments != null) {
      initShim.arguments(builder.arguments);
    }
    if (builder.processOutput != null) {
      initShim.processOutput(builder.processOutput);
    }
    if (builder.supportConfig != null) {
      initShim.supportConfig(builder.supportConfig);
    }
    this.processWorkingDir = initShim.processWorkingDir();
    this.transitionLabel = initShim.transitionLabel();
    this.processExecutable = initShim.processExecutable();
    this.processConfig = initShim.processConfig();
    this.processEnv = initShim.processEnv();
    this.arguments = initShim.arguments();
    this.processOutput = initShim.processOutput();
    this.supportConfig = initShim.supportConfig();
    this.initShim = null;
  }

  private ImmutableExecuter(
      StateID<ProcessWorkingDir> processWorkingDir,
      String transitionLabel,
      StateID<ExtractedFileSet> processExecutable,
      StateID<ProcessConfig> processConfig,
      StateID<ProcessEnv> processEnv,
      StateID<ProcessArguments> arguments,
      StateID<ProcessOutput> processOutput,
      StateID<SupportConfig> supportConfig,
      StateID<T> destination,
      RunningProcessFactory<R> runningProcessFactory,
      ExecutedProcessFactory<R, T> executedProcessFactory) {
    this.processWorkingDir = processWorkingDir;
    this.transitionLabel = transitionLabel;
    this.processExecutable = processExecutable;
    this.processConfig = processConfig;
    this.processEnv = processEnv;
    this.arguments = arguments;
    this.processOutput = processOutput;
    this.supportConfig = supportConfig;
    this.destination = destination;
    this.runningProcessFactory = runningProcessFactory;
    this.executedProcessFactory = executedProcessFactory;
    this.initShim = null;
  }

  private static final byte STAGE_INITIALIZING = -1;
  private static final byte STAGE_UNINITIALIZED = 0;
  private static final byte STAGE_INITIALIZED = 1;
  private transient volatile InitShim initShim = new InitShim();

  private final class InitShim {
    private byte processWorkingDirBuildStage = STAGE_UNINITIALIZED;
    private StateID<ProcessWorkingDir> processWorkingDir;

    StateID<ProcessWorkingDir> processWorkingDir() {
      if (processWorkingDirBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (processWorkingDirBuildStage == STAGE_UNINITIALIZED) {
        processWorkingDirBuildStage = STAGE_INITIALIZING;
        this.processWorkingDir = Objects.requireNonNull(ImmutableExecuter.super.processWorkingDir(), "processWorkingDir");
        processWorkingDirBuildStage = STAGE_INITIALIZED;
      }
      return this.processWorkingDir;
    }

    void processWorkingDir(StateID<ProcessWorkingDir> processWorkingDir) {
      this.processWorkingDir = processWorkingDir;
      processWorkingDirBuildStage = STAGE_INITIALIZED;
    }

    private byte transitionLabelBuildStage = STAGE_UNINITIALIZED;
    private String transitionLabel;

    String transitionLabel() {
      if (transitionLabelBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (transitionLabelBuildStage == STAGE_UNINITIALIZED) {
        transitionLabelBuildStage = STAGE_INITIALIZING;
        this.transitionLabel = Objects.requireNonNull(ImmutableExecuter.super.transitionLabel(), "transitionLabel");
        transitionLabelBuildStage = STAGE_INITIALIZED;
      }
      return this.transitionLabel;
    }

    void transitionLabel(String transitionLabel) {
      this.transitionLabel = transitionLabel;
      transitionLabelBuildStage = STAGE_INITIALIZED;
    }

    private byte processExecutableBuildStage = STAGE_UNINITIALIZED;
    private StateID<ExtractedFileSet> processExecutable;

    StateID<ExtractedFileSet> processExecutable() {
      if (processExecutableBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (processExecutableBuildStage == STAGE_UNINITIALIZED) {
        processExecutableBuildStage = STAGE_INITIALIZING;
        this.processExecutable = Objects.requireNonNull(ImmutableExecuter.super.processExecutable(), "processExecutable");
        processExecutableBuildStage = STAGE_INITIALIZED;
      }
      return this.processExecutable;
    }

    void processExecutable(StateID<ExtractedFileSet> processExecutable) {
      this.processExecutable = processExecutable;
      processExecutableBuildStage = STAGE_INITIALIZED;
    }

    private byte processConfigBuildStage = STAGE_UNINITIALIZED;
    private StateID<ProcessConfig> processConfig;

    StateID<ProcessConfig> processConfig() {
      if (processConfigBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (processConfigBuildStage == STAGE_UNINITIALIZED) {
        processConfigBuildStage = STAGE_INITIALIZING;
        this.processConfig = Objects.requireNonNull(ImmutableExecuter.super.processConfig(), "processConfig");
        processConfigBuildStage = STAGE_INITIALIZED;
      }
      return this.processConfig;
    }

    void processConfig(StateID<ProcessConfig> processConfig) {
      this.processConfig = processConfig;
      processConfigBuildStage = STAGE_INITIALIZED;
    }

    private byte processEnvBuildStage = STAGE_UNINITIALIZED;
    private StateID<ProcessEnv> processEnv;

    StateID<ProcessEnv> processEnv() {
      if (processEnvBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (processEnvBuildStage == STAGE_UNINITIALIZED) {
        processEnvBuildStage = STAGE_INITIALIZING;
        this.processEnv = Objects.requireNonNull(ImmutableExecuter.super.processEnv(), "processEnv");
        processEnvBuildStage = STAGE_INITIALIZED;
      }
      return this.processEnv;
    }

    void processEnv(StateID<ProcessEnv> processEnv) {
      this.processEnv = processEnv;
      processEnvBuildStage = STAGE_INITIALIZED;
    }

    private byte argumentsBuildStage = STAGE_UNINITIALIZED;
    private StateID<ProcessArguments> arguments;

    StateID<ProcessArguments> arguments() {
      if (argumentsBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (argumentsBuildStage == STAGE_UNINITIALIZED) {
        argumentsBuildStage = STAGE_INITIALIZING;
        this.arguments = Objects.requireNonNull(ImmutableExecuter.super.arguments(), "arguments");
        argumentsBuildStage = STAGE_INITIALIZED;
      }
      return this.arguments;
    }

    void arguments(StateID<ProcessArguments> arguments) {
      this.arguments = arguments;
      argumentsBuildStage = STAGE_INITIALIZED;
    }

    private byte processOutputBuildStage = STAGE_UNINITIALIZED;
    private StateID<ProcessOutput> processOutput;

    StateID<ProcessOutput> processOutput() {
      if (processOutputBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (processOutputBuildStage == STAGE_UNINITIALIZED) {
        processOutputBuildStage = STAGE_INITIALIZING;
        this.processOutput = Objects.requireNonNull(ImmutableExecuter.super.processOutput(), "processOutput");
        processOutputBuildStage = STAGE_INITIALIZED;
      }
      return this.processOutput;
    }

    void processOutput(StateID<ProcessOutput> processOutput) {
      this.processOutput = processOutput;
      processOutputBuildStage = STAGE_INITIALIZED;
    }

    private byte supportConfigBuildStage = STAGE_UNINITIALIZED;
    private StateID<SupportConfig> supportConfig;

    StateID<SupportConfig> supportConfig() {
      if (supportConfigBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (supportConfigBuildStage == STAGE_UNINITIALIZED) {
        supportConfigBuildStage = STAGE_INITIALIZING;
        this.supportConfig = Objects.requireNonNull(ImmutableExecuter.super.supportConfig(), "supportConfig");
        supportConfigBuildStage = STAGE_INITIALIZED;
      }
      return this.supportConfig;
    }

    void supportConfig(StateID<SupportConfig> supportConfig) {
      this.supportConfig = supportConfig;
      supportConfigBuildStage = STAGE_INITIALIZED;
    }

    private String formatInitCycleMessage() {
      List<String> attributes = new ArrayList<>();
      if (processWorkingDirBuildStage == STAGE_INITIALIZING) attributes.add("processWorkingDir");
      if (transitionLabelBuildStage == STAGE_INITIALIZING) attributes.add("transitionLabel");
      if (processExecutableBuildStage == STAGE_INITIALIZING) attributes.add("processExecutable");
      if (processConfigBuildStage == STAGE_INITIALIZING) attributes.add("processConfig");
      if (processEnvBuildStage == STAGE_INITIALIZING) attributes.add("processEnv");
      if (argumentsBuildStage == STAGE_INITIALIZING) attributes.add("arguments");
      if (processOutputBuildStage == STAGE_INITIALIZING) attributes.add("processOutput");
      if (supportConfigBuildStage == STAGE_INITIALIZING) attributes.add("supportConfig");
      return "Cannot build Executer, attribute initializers form cycle " + attributes;
    }
  }

  /**
   * @return The value of the {@code processWorkingDir} attribute
   */
  @Override
  public StateID<ProcessWorkingDir> processWorkingDir() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.processWorkingDir()
        : this.processWorkingDir;
  }

  /**
   * @return The value of the {@code transitionLabel} attribute
   */
  @Override
  public String transitionLabel() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.transitionLabel()
        : this.transitionLabel;
  }

  /**
   * @return The value of the {@code processExecutable} attribute
   */
  @Override
  public StateID<ExtractedFileSet> processExecutable() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.processExecutable()
        : this.processExecutable;
  }

  /**
   * @return The value of the {@code processConfig} attribute
   */
  @Override
  public StateID<ProcessConfig> processConfig() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.processConfig()
        : this.processConfig;
  }

  /**
   * @return The value of the {@code processEnv} attribute
   */
  @Override
  public StateID<ProcessEnv> processEnv() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.processEnv()
        : this.processEnv;
  }

  /**
   * @return The value of the {@code arguments} attribute
   */
  @Override
  public StateID<ProcessArguments> arguments() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.arguments()
        : this.arguments;
  }

  /**
   * @return The value of the {@code processOutput} attribute
   */
  @Override
  public StateID<ProcessOutput> processOutput() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.processOutput()
        : this.processOutput;
  }

  /**
   * @return The value of the {@code supportConfig} attribute
   */
  @Override
  public StateID<SupportConfig> supportConfig() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.supportConfig()
        : this.supportConfig;
  }

  /**
   * @return The value of the {@code destination} attribute
   */
  @Override
  public StateID<T> destination() {
    return destination;
  }

  /**
   * @return The value of the {@code runningProcessFactory} attribute
   */
  @Override
  protected RunningProcessFactory<R> runningProcessFactory() {
    return runningProcessFactory;
  }

  /**
   * @return The value of the {@code executedProcessFactory} attribute
   */
  @Override
  protected ExecutedProcessFactory<R, T> executedProcessFactory() {
    return executedProcessFactory;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Executer#processWorkingDir() processWorkingDir} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for processWorkingDir
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExecuter<R, T> withProcessWorkingDir(StateID<ProcessWorkingDir> value) {
    if (this.processWorkingDir == value) return this;
    StateID<ProcessWorkingDir> newValue = Objects.requireNonNull(value, "processWorkingDir");
    return new ImmutableExecuter<>(
        newValue,
        this.transitionLabel,
        this.processExecutable,
        this.processConfig,
        this.processEnv,
        this.arguments,
        this.processOutput,
        this.supportConfig,
        this.destination,
        this.runningProcessFactory,
        this.executedProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Executer#transitionLabel() transitionLabel} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for transitionLabel
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExecuter<R, T> withTransitionLabel(String value) {
    String newValue = Objects.requireNonNull(value, "transitionLabel");
    if (this.transitionLabel.equals(newValue)) return this;
    return new ImmutableExecuter<>(
        this.processWorkingDir,
        newValue,
        this.processExecutable,
        this.processConfig,
        this.processEnv,
        this.arguments,
        this.processOutput,
        this.supportConfig,
        this.destination,
        this.runningProcessFactory,
        this.executedProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Executer#processExecutable() processExecutable} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for processExecutable
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExecuter<R, T> withProcessExecutable(StateID<ExtractedFileSet> value) {
    if (this.processExecutable == value) return this;
    StateID<ExtractedFileSet> newValue = Objects.requireNonNull(value, "processExecutable");
    return new ImmutableExecuter<>(
        this.processWorkingDir,
        this.transitionLabel,
        newValue,
        this.processConfig,
        this.processEnv,
        this.arguments,
        this.processOutput,
        this.supportConfig,
        this.destination,
        this.runningProcessFactory,
        this.executedProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Executer#processConfig() processConfig} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for processConfig
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExecuter<R, T> withProcessConfig(StateID<ProcessConfig> value) {
    if (this.processConfig == value) return this;
    StateID<ProcessConfig> newValue = Objects.requireNonNull(value, "processConfig");
    return new ImmutableExecuter<>(
        this.processWorkingDir,
        this.transitionLabel,
        this.processExecutable,
        newValue,
        this.processEnv,
        this.arguments,
        this.processOutput,
        this.supportConfig,
        this.destination,
        this.runningProcessFactory,
        this.executedProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Executer#processEnv() processEnv} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for processEnv
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExecuter<R, T> withProcessEnv(StateID<ProcessEnv> value) {
    if (this.processEnv == value) return this;
    StateID<ProcessEnv> newValue = Objects.requireNonNull(value, "processEnv");
    return new ImmutableExecuter<>(
        this.processWorkingDir,
        this.transitionLabel,
        this.processExecutable,
        this.processConfig,
        newValue,
        this.arguments,
        this.processOutput,
        this.supportConfig,
        this.destination,
        this.runningProcessFactory,
        this.executedProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Executer#arguments() arguments} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for arguments
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExecuter<R, T> withArguments(StateID<ProcessArguments> value) {
    if (this.arguments == value) return this;
    StateID<ProcessArguments> newValue = Objects.requireNonNull(value, "arguments");
    return new ImmutableExecuter<>(
        this.processWorkingDir,
        this.transitionLabel,
        this.processExecutable,
        this.processConfig,
        this.processEnv,
        newValue,
        this.processOutput,
        this.supportConfig,
        this.destination,
        this.runningProcessFactory,
        this.executedProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Executer#processOutput() processOutput} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for processOutput
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExecuter<R, T> withProcessOutput(StateID<ProcessOutput> value) {
    if (this.processOutput == value) return this;
    StateID<ProcessOutput> newValue = Objects.requireNonNull(value, "processOutput");
    return new ImmutableExecuter<>(
        this.processWorkingDir,
        this.transitionLabel,
        this.processExecutable,
        this.processConfig,
        this.processEnv,
        this.arguments,
        newValue,
        this.supportConfig,
        this.destination,
        this.runningProcessFactory,
        this.executedProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Executer#supportConfig() supportConfig} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for supportConfig
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExecuter<R, T> withSupportConfig(StateID<SupportConfig> value) {
    if (this.supportConfig == value) return this;
    StateID<SupportConfig> newValue = Objects.requireNonNull(value, "supportConfig");
    return new ImmutableExecuter<>(
        this.processWorkingDir,
        this.transitionLabel,
        this.processExecutable,
        this.processConfig,
        this.processEnv,
        this.arguments,
        this.processOutput,
        newValue,
        this.destination,
        this.runningProcessFactory,
        this.executedProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Executer#destination() destination} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for destination
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExecuter<R, T> withDestination(StateID<T> value) {
    if (this.destination == value) return this;
    StateID<T> newValue = Objects.requireNonNull(value, "destination");
    return new ImmutableExecuter<>(
        this.processWorkingDir,
        this.transitionLabel,
        this.processExecutable,
        this.processConfig,
        this.processEnv,
        this.arguments,
        this.processOutput,
        this.supportConfig,
        newValue,
        this.runningProcessFactory,
        this.executedProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Executer#runningProcessFactory() runningProcessFactory} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for runningProcessFactory
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExecuter<R, T> withRunningProcessFactory(RunningProcessFactory<R> value) {
    if (this.runningProcessFactory == value) return this;
    RunningProcessFactory<R> newValue = Objects.requireNonNull(value, "runningProcessFactory");
    return new ImmutableExecuter<>(
        this.processWorkingDir,
        this.transitionLabel,
        this.processExecutable,
        this.processConfig,
        this.processEnv,
        this.arguments,
        this.processOutput,
        this.supportConfig,
        this.destination,
        newValue,
        this.executedProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Executer#executedProcessFactory() executedProcessFactory} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for executedProcessFactory
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExecuter<R, T> withExecutedProcessFactory(ExecutedProcessFactory<R, T> value) {
    if (this.executedProcessFactory == value) return this;
    ExecutedProcessFactory<R, T> newValue = Objects.requireNonNull(value, "executedProcessFactory");
    return new ImmutableExecuter<>(
        this.processWorkingDir,
        this.transitionLabel,
        this.processExecutable,
        this.processConfig,
        this.processEnv,
        this.arguments,
        this.processOutput,
        this.supportConfig,
        this.destination,
        this.runningProcessFactory,
        newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableExecuter} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableExecuter<?, ?>
        && equalTo(0, (ImmutableExecuter<?, ?>) another);
  }

  private boolean equalTo(int synthetic, ImmutableExecuter<?, ?> another) {
    return processWorkingDir.equals(another.processWorkingDir)
        && transitionLabel.equals(another.transitionLabel)
        && processExecutable.equals(another.processExecutable)
        && processConfig.equals(another.processConfig)
        && processEnv.equals(another.processEnv)
        && arguments.equals(another.arguments)
        && processOutput.equals(another.processOutput)
        && supportConfig.equals(another.supportConfig)
        && destination.equals(another.destination)
        && runningProcessFactory.equals(another.runningProcessFactory)
        && executedProcessFactory.equals(another.executedProcessFactory);
  }

  /**
   * Computes a hash code from attributes: {@code processWorkingDir}, {@code transitionLabel}, {@code processExecutable}, {@code processConfig}, {@code processEnv}, {@code arguments}, {@code processOutput}, {@code supportConfig}, {@code destination}, {@code runningProcessFactory}, {@code executedProcessFactory}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + processWorkingDir.hashCode();
    h += (h << 5) + transitionLabel.hashCode();
    h += (h << 5) + processExecutable.hashCode();
    h += (h << 5) + processConfig.hashCode();
    h += (h << 5) + processEnv.hashCode();
    h += (h << 5) + arguments.hashCode();
    h += (h << 5) + processOutput.hashCode();
    h += (h << 5) + supportConfig.hashCode();
    h += (h << 5) + destination.hashCode();
    h += (h << 5) + runningProcessFactory.hashCode();
    h += (h << 5) + executedProcessFactory.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Executer} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Executer{"
        + "processWorkingDir=" + processWorkingDir
        + ", transitionLabel=" + transitionLabel
        + ", processExecutable=" + processExecutable
        + ", processConfig=" + processConfig
        + ", processEnv=" + processEnv
        + ", arguments=" + arguments
        + ", processOutput=" + processOutput
        + ", supportConfig=" + supportConfig
        + ", destination=" + destination
        + ", runningProcessFactory=" + runningProcessFactory
        + ", executedProcessFactory=" + executedProcessFactory
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Executer} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param <R> generic parameter R
   * @param <T> generic parameter T
   * @param instance The instance to copy
   * @return A copied immutable Executer instance
   */
  public static <R extends RunningProcess, T extends ExecutedProcess> ImmutableExecuter<R, T> copyOf(Executer<R, T> instance) {
    if (instance instanceof ImmutableExecuter<?, ?>) {
      return (ImmutableExecuter<R, T>) instance;
    }
    return ImmutableExecuter.<R, T>builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableExecuter ImmutableExecuter}.
   * <pre>
   * ImmutableExecuter.&amp;lt;R, T&amp;gt;builder()
   *    .processWorkingDir(de.flapdoodle.reverse.StateID&amp;lt;ProcessWorkingDir&amp;gt;) // optional {@link Executer#processWorkingDir() processWorkingDir}
   *    .transitionLabel(String) // optional {@link Executer#transitionLabel() transitionLabel}
   *    .processExecutable(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.archives.ExtractedFileSet&amp;gt;) // optional {@link Executer#processExecutable() processExecutable}
   *    .processConfig(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.types.ProcessConfig&amp;gt;) // optional {@link Executer#processConfig() processConfig}
   *    .processEnv(de.flapdoodle.reverse.StateID&amp;lt;ProcessEnv&amp;gt;) // optional {@link Executer#processEnv() processEnv}
   *    .arguments(de.flapdoodle.reverse.StateID&amp;lt;ProcessArguments&amp;gt;) // optional {@link Executer#arguments() arguments}
   *    .processOutput(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.io.ProcessOutput&amp;gt;) // optional {@link Executer#processOutput() processOutput}
   *    .supportConfig(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.config.SupportConfig&amp;gt;) // optional {@link Executer#supportConfig() supportConfig}
   *    .destination(de.flapdoodle.reverse.StateID&amp;lt;T&amp;gt;) // required {@link Executer#destination() destination}
   *    .runningProcessFactory(de.flapdoodle.embed.process.types.RunningProcessFactory&amp;lt;R&amp;gt;) // required {@link Executer#runningProcessFactory() runningProcessFactory}
   *    .executedProcessFactory(de.flapdoodle.embed.process.types.ExecutedProcessFactory&amp;lt;R, T&amp;gt;) // required {@link Executer#executedProcessFactory() executedProcessFactory}
   *    .build();
   * </pre>
   * @param <R> generic parameter R
   * @param <T> generic parameter T
   * @param runningProcessFactory {@code runningProcessFactory} parameter
   * @param executedProcessFactory {@code executedProcessFactory} parameter
   * @return A new ImmutableExecuter builder
   */
  public static <R extends RunningProcess, T extends ExecutedProcess> ImmutableExecuter.Builder<R, T> builder(RunningProcessFactory<R> runningProcessFactory, ExecutedProcessFactory<R, T> executedProcessFactory) {
    return new ImmutableExecuter.Builder<>(runningProcessFactory, executedProcessFactory);
  }

  static <R extends RunningProcess, T extends ExecutedProcess> ImmutableExecuter.Builder<R, T> builder() {
    return new ImmutableExecuter.Builder<>();
  }

  /**
   * Builds instances of type {@link ImmutableExecuter ImmutableExecuter}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder<R extends RunningProcess, T extends ExecutedProcess>
     {
    private static final long INIT_BIT_DESTINATION = 0x1L;
    private static final long INIT_BIT_RUNNING_PROCESS_FACTORY = 0x2L;
    private static final long INIT_BIT_EXECUTED_PROCESS_FACTORY = 0x4L;
    private long initBits = 0x7L;

    private StateID<ProcessWorkingDir> processWorkingDir;
    private String transitionLabel;
    private StateID<ExtractedFileSet> processExecutable;
    private StateID<ProcessConfig> processConfig;
    private StateID<ProcessEnv> processEnv;
    private StateID<ProcessArguments> arguments;
    private StateID<ProcessOutput> processOutput;
    private StateID<SupportConfig> supportConfig;
    private StateID<T> destination;
    private RunningProcessFactory<R> runningProcessFactory;
    private ExecutedProcessFactory<R, T> executedProcessFactory;

    private Builder(RunningProcessFactory<R> runningProcessFactory, ExecutedProcessFactory<R, T> executedProcessFactory) {
      runningProcessFactory(runningProcessFactory);
      executedProcessFactory(executedProcessFactory);
    }

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code de.flapdoodle.embed.process.transitions.Executer} instance.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<R, T> from(Executer<R, T> instance) {
      Objects.requireNonNull(instance, "instance");
      from((Object) instance);
      return this;
    }

    /**
     * Fill a builder with attribute values from the provided {@code de.flapdoodle.reverse.naming.HasLabel} instance.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<R, T> from(HasLabel instance) {
      Objects.requireNonNull(instance, "instance");
      from((Object) instance);
      return this;
    }

    @SuppressWarnings("unchecked")
    private void from(Object object) {
      long bits = 0;
      if (object instanceof Executer<?, ?>) {
        Executer<R, T> instance = (Executer<R, T>) object;
        supportConfig(instance.supportConfig());
        if ((bits & 0x1L) == 0) {
          transitionLabel(instance.transitionLabel());
          bits |= 0x1L;
        }
        processExecutable(instance.processExecutable());
        runningProcessFactory(instance.runningProcessFactory());
        executedProcessFactory(instance.executedProcessFactory());
        processConfig(instance.processConfig());
        processEnv(instance.processEnv());
        processOutput(instance.processOutput());
        destination(instance.destination());
        arguments(instance.arguments());
        processWorkingDir(instance.processWorkingDir());
      }
      if (object instanceof HasLabel) {
        HasLabel instance = (HasLabel) object;
        if ((bits & 0x1L) == 0) {
          transitionLabel(instance.transitionLabel());
          bits |= 0x1L;
        }
      }
    }

    /**
     * Initializes the value for the {@link Executer#processWorkingDir() processWorkingDir} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Executer#processWorkingDir() processWorkingDir}.</em>
     * @param processWorkingDir The value for processWorkingDir 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<R, T> processWorkingDir(StateID<ProcessWorkingDir> processWorkingDir) {
      this.processWorkingDir = Objects.requireNonNull(processWorkingDir, "processWorkingDir");
      return this;
    }

    /**
     * Initializes the value for the {@link Executer#transitionLabel() transitionLabel} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Executer#transitionLabel() transitionLabel}.</em>
     * @param transitionLabel The value for transitionLabel 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<R, T> transitionLabel(String transitionLabel) {
      this.transitionLabel = Objects.requireNonNull(transitionLabel, "transitionLabel");
      return this;
    }

    /**
     * Initializes the value for the {@link Executer#processExecutable() processExecutable} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Executer#processExecutable() processExecutable}.</em>
     * @param processExecutable The value for processExecutable 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<R, T> processExecutable(StateID<ExtractedFileSet> processExecutable) {
      this.processExecutable = Objects.requireNonNull(processExecutable, "processExecutable");
      return this;
    }

    /**
     * Initializes the value for the {@link Executer#processConfig() processConfig} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Executer#processConfig() processConfig}.</em>
     * @param processConfig The value for processConfig 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<R, T> processConfig(StateID<ProcessConfig> processConfig) {
      this.processConfig = Objects.requireNonNull(processConfig, "processConfig");
      return this;
    }

    /**
     * Initializes the value for the {@link Executer#processEnv() processEnv} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Executer#processEnv() processEnv}.</em>
     * @param processEnv The value for processEnv 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<R, T> processEnv(StateID<ProcessEnv> processEnv) {
      this.processEnv = Objects.requireNonNull(processEnv, "processEnv");
      return this;
    }

    /**
     * Initializes the value for the {@link Executer#arguments() arguments} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Executer#arguments() arguments}.</em>
     * @param arguments The value for arguments 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<R, T> arguments(StateID<ProcessArguments> arguments) {
      this.arguments = Objects.requireNonNull(arguments, "arguments");
      return this;
    }

    /**
     * Initializes the value for the {@link Executer#processOutput() processOutput} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Executer#processOutput() processOutput}.</em>
     * @param processOutput The value for processOutput 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<R, T> processOutput(StateID<ProcessOutput> processOutput) {
      this.processOutput = Objects.requireNonNull(processOutput, "processOutput");
      return this;
    }

    /**
     * Initializes the value for the {@link Executer#supportConfig() supportConfig} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Executer#supportConfig() supportConfig}.</em>
     * @param supportConfig The value for supportConfig 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<R, T> supportConfig(StateID<SupportConfig> supportConfig) {
      this.supportConfig = Objects.requireNonNull(supportConfig, "supportConfig");
      return this;
    }

    /**
     * Initializes the value for the {@link Executer#destination() destination} attribute.
     * @param destination The value for destination 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<R, T> destination(StateID<T> destination) {
      this.destination = Objects.requireNonNull(destination, "destination");
      initBits &= ~INIT_BIT_DESTINATION;
      return this;
    }

    /**
     * Initializes the value for the {@link Executer#runningProcessFactory() runningProcessFactory} attribute.
     * @param runningProcessFactory The value for runningProcessFactory 
     * @return {@code this} builder for use in a chained invocation
     */
    final Builder<R, T> runningProcessFactory(RunningProcessFactory<R> runningProcessFactory) {
      this.runningProcessFactory = Objects.requireNonNull(runningProcessFactory, "runningProcessFactory");
      initBits &= ~INIT_BIT_RUNNING_PROCESS_FACTORY;
      return this;
    }

    /**
     * Initializes the value for the {@link Executer#executedProcessFactory() executedProcessFactory} attribute.
     * @param executedProcessFactory The value for executedProcessFactory 
     * @return {@code this} builder for use in a chained invocation
     */
    final Builder<R, T> executedProcessFactory(ExecutedProcessFactory<R, T> executedProcessFactory) {
      this.executedProcessFactory = Objects.requireNonNull(executedProcessFactory, "executedProcessFactory");
      initBits &= ~INIT_BIT_EXECUTED_PROCESS_FACTORY;
      return this;
    }

    /**
     * Builds a new {@link ImmutableExecuter ImmutableExecuter}.
     * @return An immutable instance of Executer
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableExecuter<R, T> build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableExecuter<R, T>(this);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_DESTINATION) != 0) attributes.add("destination");
      if ((initBits & INIT_BIT_RUNNING_PROCESS_FACTORY) != 0) attributes.add("runningProcessFactory");
      if ((initBits & INIT_BIT_EXECUTED_PROCESS_FACTORY) != 0) attributes.add("executedProcessFactory");
      return "Cannot build Executer, some of required attributes are not set " + attributes;
    }
  }
}
