/*
 * Decompiled with CFR 0.152.
 */
package com.sourceclear.util.io;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.sourceclear.api.data.git.GitCommit;
import com.sourceclear.api.data.git.MetaGit;
import com.sourceclear.api.data.git.RefType;
import com.sourceclear.util.config.RepoUtils;
import com.veracode.security.logging.SecureLogger;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.eclipse.jgit.annotations.NonNull;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.Status;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.lib.AnyObjectId;
import org.eclipse.jgit.lib.BranchConfig;
import org.eclipse.jgit.lib.Config;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.lib.RepositoryBuilder;
import org.eclipse.jgit.lib.StoredConfig;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevSort;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.revwalk.filter.CommitTimeRevFilter;
import org.eclipse.jgit.storage.file.FileRepositoryBuilder;
import org.eclipse.jgit.transport.RemoteConfig;
import org.eclipse.jgit.transport.URIish;

public final class GitUtils {
    private static final SecureLogger LOGGER = SecureLogger.getLogger(GitUtils.class);
    private static final Set<String> CI_BRANCHES = ImmutableSet.of((Object)"GIT_BRANCH", (Object)"bamboo_planRepository_branch", (Object)"CI_COMMIT_REF_NAME");
    private static final ImmutableList<String> TRAVIS_BRANCHES_ORDER_PREFERENCE = ImmutableList.of((Object)"TRAVIS_PULL_REQUEST_BRANCH", (Object)"TRAVIS_BRANCH");
    private static final String TRAVIS_PULL_REQUEST_SLUG = "TRAVIS_PULL_REQUEST_SLUG";
    private static final String TRAVIS_REPO_SLUG = "TRAVIS_REPO_SLUG";
    private static final Set<String> CI_ORIGINS = ImmutableSet.of((Object)"GIT_URL", (Object)"bamboo_planRepository_repositoryUrl");
    public static final String GENERIC_SCM_ENV_PREFIX = "SRCCLR_SCM_";
    public static final String GITLESS_ENV_KEY = "__WITHOUT_GIT__";
    public static final String GIT_CEILING_DIR = "SRCCLR_GIT_CEILING_DIR";
    private static final String HEADS = "heads/";

    private static Optional<File> goUpToGitRepo(File origFile, Optional<File> ceilingDir) {
        FileRepositoryBuilder fileRepositoryBuilder = new FileRepositoryBuilder();
        ceilingDir.ifPresent(dir -> fileRepositoryBuilder.addCeilingDirectory(dir));
        fileRepositoryBuilder.findGitDir(origFile);
        return Optional.ofNullable(fileRepositoryBuilder.getGitDir()).map(File::getParentFile);
    }

    @VisibleForTesting
    static Optional<Repository> getRepoFromProjDir(@Nonnull File projectDir, Optional<File> ceilingDir) throws IOException {
        Optional<File> repoDir = GitUtils.goUpToGitRepo(projectDir, ceilingDir);
        if (repoDir.isPresent()) {
            return Optional.ofNullable(((RepositoryBuilder)new RepositoryBuilder().setWorkTree(repoDir.get())).build());
        }
        return Optional.empty();
    }

    @VisibleForTesting
    public static Optional<Repository> getRepoFromProjDir(@Nonnull File projectDir) throws IOException {
        Optional<File> ceilingDir = Optional.ofNullable(System.getenv(GIT_CEILING_DIR)).filter(s -> !Strings.isNullOrEmpty((String)s)).map(File::new);
        LOGGER.debug("SRCCLR_GIT_CEILING_DIR={}", (Object)ceilingDir);
        return GitUtils.getRepoFromProjDir(projectDir, ceilingDir);
    }

    protected static String convertGitToHttps(String urlIsh) {
        if (urlIsh.startsWith("http")) {
            return urlIsh;
        }
        if (urlIsh.startsWith("git@")) {
            return urlIsh.replaceFirst(":", "/").replaceFirst("^git@", "https://");
        }
        if (urlIsh.startsWith("ssh:")) {
            String tmp = urlIsh.replaceFirst("^ssh:", "").replaceFirst("git@", "").replaceFirst(":", "/");
            tmp = "https:" + tmp;
            return tmp;
        }
        return "https://" + urlIsh;
    }

    public static MetaGit getGitMetaData(@Nonnull File projectDir) throws Exception {
        return GitUtils.getGitMetaData(projectDir, System.getenv());
    }

    @Nonnull
    public static MetaGit getGitMetaData(@Nonnull File projectDir, @Nonnull Map<String, String> env) throws Exception {
        Optional<String> branchName;
        Optional<Repository> repoOpt;
        GitUtils.verifyOrThrow(projectDir);
        if (env.containsKey(GITLESS_ENV_KEY)) {
            String projectUri;
            String subPath;
            String refType;
            boolean haveRef = false;
            boolean haveRefName = false;
            boolean haveUri = false;
            MetaGit.Builder builder = new MetaGit.Builder();
            String ref = GenericScmEnvNames.REF.getOrNull(env);
            if (ref != null) {
                builder.withRefName(ref);
                haveRefName = true;
            }
            if ((refType = GenericScmEnvNames.REF_TYPE.getOrNull(env)) != null) {
                builder.withRefType(RefType.valueOf(refType.toUpperCase()));
            } else {
                builder.withRefType(RefType.COMMIT);
            }
            String rev = GenericScmEnvNames.REV.getOrNull(env);
            if (rev != null) {
                builder.withHead(rev);
                haveRef = true;
            }
            if ((subPath = GenericScmEnvNames.SUB_PATH.getOrNull(env)) != null) {
                builder.withSubPath(subPath);
            }
            if ((projectUri = GenericScmEnvNames.URI.getOrNull(env)) != null) {
                builder.withRemote(GitUtils.removeWindowsDomain(new URIish(projectUri)));
                haveUri = true;
            }
            if (haveRef && haveRefName && haveUri) {
                builder.withLocalPath(projectDir.toURI());
                MetaGit result = builder.build();
                LOGGER.debug("Short-circuiting MetaGit in favor of the environment, result = {}", (Object)result);
                return result;
            }
        }
        if (!(repoOpt = GitUtils.getRepoFromProjDir(projectDir)).isPresent()) {
            throw new RuntimeException(String.format("%s is not in a git repo", projectDir.toString()));
        }
        Repository repo = repoOpt.get();
        Path repoPath = repo.getDirectory().toPath();
        Path scanPath = projectDir.toPath();
        Path subPath = repoPath.getParent().toAbsolutePath().relativize(scanPath.toAbsolutePath());
        Pair<String, RefType> refPair = GitUtils.findRef(repo, env);
        URIish remoteUri = GitUtils.getRemoteUrl(repo, env);
        RefType refType = (RefType)((Object)refPair.getRight());
        String refName = (String)refPair.getLeft();
        URI remote = remoteUri == null ? null : GitUtils.removeWindowsDomain(remoteUri);
        if (refType == RefType.COMMIT && (branchName = GitUtils.tryGetBranchName(repo)).isPresent()) {
            refType = RefType.BRANCH;
            refName = branchName.get();
        }
        return new MetaGit.Builder().withSubPath(subPath.toString()).withLocalPath(projectDir.toURI()).withRemote(remote).withHead(GitUtils.getCommitHash(repo)).withRefName(refName).withRefType(refType).build();
    }

    @Deprecated
    @Nonnull
    public static Pair<String, String> parseRepoUrl(@Nullable String repoUrl) {
        return RepoUtils.parseRepoUrl(repoUrl);
    }

    @NonNull
    public static Pair<String, RefType> findRef(@Nonnull Repository repo, @Nonnull Map<String, String> env) throws Exception {
        for (String var : CI_BRANCHES) {
            if (!env.containsKey(var)) continue;
            String branch = env.get(var).replaceAll("^.+/", "");
            return Pair.of((Object)branch, (Object)((Object)RefType.BRANCH));
        }
        Optional<String> branch = GitUtils.tryTravisBranchNames(env);
        if (branch.isPresent()) {
            Optional<String> prSlug = GitUtils.getTravisPrSlugIfDifferentFromRepoSlug(env);
            String fullBranchName = prSlug.isPresent() ? String.format("%s/%s", prSlug.get(), branch.get()) : branch.get();
            return Pair.of((Object)fullBranchName, (Object)((Object)RefType.BRANCH));
        }
        String tagName = GitUtils.getTagName(repo);
        if (tagName != null) {
            return Pair.of((Object)tagName, (Object)((Object)RefType.TAG));
        }
        String name = repo.getBranch();
        String fullName = repo.getFullBranch();
        if (name == null || fullName == null) {
            return Pair.of((Object)"Unknown", (Object)((Object)RefType.COMMIT));
        }
        RefType refType = name.equals(fullName) ? RefType.COMMIT : RefType.BRANCH;
        return Pair.of((Object)name, (Object)((Object)refType));
    }

    @Nullable
    private static String getTagName(@Nonnull Repository repository) throws Exception {
        String commitHash = GitUtils.getCommitHash(repository);
        if (commitHash == null) {
            return null;
        }
        ObjectId commitId = ObjectId.fromString((String)commitHash);
        List list = Git.wrap((Repository)repository).tagList().call();
        for (Ref tag : list) {
            Ref peeledTag = repository.peel(tag);
            ObjectId object = peeledTag.getPeeledObjectId() != null ? peeledTag.getPeeledObjectId() : peeledTag.getObjectId();
            if (!commitId.equals((AnyObjectId)object)) continue;
            return Repository.shortenRefName((String)peeledTag.getName());
        }
        return null;
    }

    @Nullable
    public static URIish getRemoteUrl(@Nonnull Repository repo, @Nonnull Map<String, String> env) throws Exception {
        URIish uri = null;
        for (String var : CI_ORIGINS) {
            if (!env.containsKey(var)) continue;
            uri = new URIish(env.get(var));
            break;
        }
        if (uri == null) {
            RemoteConfig remoteConfig;
            List uris;
            String branch = repo.getBranch();
            StoredConfig config = repo.getConfig();
            BranchConfig branchConfig = new BranchConfig((Config)config, branch);
            String remote = branchConfig.getRemote();
            if (remote == null) {
                remote = "origin";
            }
            if (!(uris = (remoteConfig = new RemoteConfig((Config)config, remote)).getURIs()).isEmpty()) {
                uri = ((URIish)uris.get(0)).setPass(null);
            }
        }
        return uri == null ? null : GitUtils.ensureScheme(uri);
    }

    @Nullable
    public static String getRepoUrl(@Nonnull Repository repo, @Nonnull Map<String, String> env) throws Exception {
        URIish uri = GitUtils.getRemoteUrl(repo, env);
        if (uri == null) {
            return null;
        }
        String host = uri.getHost();
        if (host != null) {
            String path = uri.getPath();
            if (path.charAt(0) != '/') {
                path = '/' + path;
            }
            return host + path;
        }
        return uri.toString();
    }

    @Nullable
    public static String getCommitHash(@Nonnull Repository repo) throws Exception {
        ObjectId head = repo.resolve("HEAD");
        if (head == null) {
            return null;
        }
        return head.getName();
    }

    private static void verifyOrThrow(File projectDir) {
        if (!projectDir.exists()) {
            throw new IllegalArgumentException(projectDir + " does not exist");
        }
        if (!projectDir.canRead()) {
            throw new IllegalArgumentException(projectDir + " cannot be read");
        }
        if (!projectDir.isDirectory()) {
            throw new IllegalArgumentException(projectDir + " is not a directory");
        }
    }

    @Nonnull
    static URIish ensureScheme(@Nonnull URIish uri) {
        if (Strings.isNullOrEmpty((String)uri.getScheme())) {
            if (uri.toString().matches("^[^/]+:.+")) {
                return uri.setScheme("ssh");
            }
            return uri.setScheme("file");
        }
        return uri;
    }

    private static URI removeWindowsDomain(URIish remoteUri) {
        return URI.create(Optional.ofNullable(remoteUri.getUser()).map(remoteUser -> remoteUser.substring(remoteUser.indexOf("\\") + 1)).map(remoteUser -> remoteUri.setUser(remoteUser).toString()).orElse(remoteUri.toString()));
    }

    public static boolean isRepoClean(File projectDir) {
        try {
            Optional<Repository> optRepo = GitUtils.getRepoFromProjDir(projectDir);
            if (!optRepo.isPresent()) {
                throw new RuntimeException(String.format("%s is not in a git repository", projectDir));
            }
            Status status = Git.wrap((Repository)optRepo.get()).status().call();
            boolean noUncommittedChanges = status.getUncommittedChanges().isEmpty();
            return noUncommittedChanges;
        }
        catch (IOException | GitAPIException e) {
            throw new RuntimeException(String.format("Unable to obtain status of git repo:", e.getMessage()));
        }
    }

    public static ImmutableSet<GitCommit> getLog(File dir) throws IOException, GitAPIException {
        Git git = GitUtils.getCloseableGit(dir);
        return GitUtils.getLog(git);
    }

    public static ImmutableSet<GitCommit> getLog(File dir, Date from, Date to) throws Exception {
        Git git = GitUtils.getCloseableGit(dir);
        return GitUtils.getLog(git, from, to);
    }

    public static ImmutableSet<GitCommit> getLog(Git git) throws GitAPIException {
        LinkedHashSet<GitCommit> log = new LinkedHashSet<GitCommit>();
        Iterable commits = git.log().call();
        for (RevCommit commit : commits) {
            log.add(new GitCommit(commit.getId().getName(), commit.getAuthorIdent().getName(), commit.getAuthorIdent().getEmailAddress(), new Date((long)commit.getCommitTime() * 1000L)));
        }
        return ImmutableSet.copyOf(log);
    }

    private static Git getCloseableGit(File dir) throws IOException {
        Optional<Repository> optRepo = GitUtils.getRepoFromProjDir(dir);
        if (!optRepo.isPresent()) {
            throw new RuntimeException(String.format("%s is not in a git repo", dir));
        }
        return Git.wrap((Repository)optRepo.get());
    }

    public static ImmutableSet<GitCommit> getLog(Git git, Date from, Date to) throws Exception {
        LinkedHashSet<GitCommit> log = new LinkedHashSet<GitCommit>();
        Repository repo = git.getRepository();
        try (RevWalk walk = new RevWalk(repo);){
            walk.markStart(walk.parseCommit((AnyObjectId)repo.resolve("HEAD")));
            walk.sort(RevSort.REVERSE);
            walk.setRevFilter(CommitTimeRevFilter.between((Date)from, (Date)to));
            for (RevCommit commit : walk) {
                log.add(new GitCommit(commit.getId().getName(), commit.getAuthorIdent().getName(), commit.getAuthorIdent().getEmailAddress(), new Date((long)commit.getCommitTime() * 1000L)));
            }
        }
        return ImmutableSet.copyOf(log);
    }

    public static boolean isValidRemote(URI remote) {
        return remote != null && !"file".equals(remote.getScheme());
    }

    private GitUtils() {
    }

    @VisibleForTesting
    static Optional<String> tryTravisBranchNames(@Nonnull Map<String, String> env) {
        for (String var : TRAVIS_BRANCHES_ORDER_PREFERENCE) {
            String branch;
            if (!env.containsKey(var) || !StringUtils.isNotBlank((CharSequence)(branch = env.get(var)))) continue;
            return Optional.of(branch);
        }
        return Optional.empty();
    }

    @VisibleForTesting
    static Optional<String> getTravisPrSlugIfDifferentFromRepoSlug(@Nonnull Map<String, String> env) {
        String prSlug = Strings.nullToEmpty((String)env.get(TRAVIS_PULL_REQUEST_SLUG)).trim();
        String repoSLug = Strings.nullToEmpty((String)env.get(TRAVIS_REPO_SLUG)).trim();
        return !prSlug.isEmpty() && !prSlug.equals(repoSLug) ? Optional.of(prSlug) : Optional.empty();
    }

    @VisibleForTesting
    static Optional<String> tryGetBranchName(@Nonnull Repository repository) throws Exception {
        Optional<String> desc = Optional.ofNullable(Git.wrap((Repository)repository).describe().setAll(true).call());
        return desc.map(d -> d.startsWith(HEADS) ? d.substring(HEADS.length()) : d);
    }

    static boolean shouldNotProxyForUrl(URL url, Collection<String> noProxyDomainList) {
        String host = url.getHost().toLowerCase();
        for (String domain : noProxyDomainList) {
            if (domain.equals("*")) {
                return true;
            }
            if (domain.startsWith(".")) {
                if (host.endsWith(domain)) {
                    return true;
                }
                if (!domain.substring(1).equalsIgnoreCase(host)) continue;
                return true;
            }
            if (!host.equalsIgnoreCase(domain)) continue;
            return true;
        }
        return false;
    }

    public static enum GenericScmEnvNames {
        REF("REF"),
        REF_TYPE("REF_TYPE"),
        REV("REV"),
        SUB_PATH("SUB_PATH"),
        URI("URI");

        private String envName;

        private GenericScmEnvNames(String name) {
            this.envName = GitUtils.GENERIC_SCM_ENV_PREFIX + name;
        }

        @Nonnull
        public String getEnvironmentName() {
            return this.envName;
        }

        @Nullable
        public String getOrNull(@Nonnull Map<String, String> env) {
            return Strings.emptyToNull((String)env.get(this.envName));
        }
    }
}

