//[okio](../../../index.md)/[okio](../index.md)/[ForwardingFileSystem](index.md)

# ForwardingFileSystem

[common]\
abstract class [ForwardingFileSystem](index.md)(val delegate: [FileSystem](../-file-system/index.md)) : [FileSystem](../-file-system/index.md)

A [FileSystem](../-file-system/index.md) that forwards calls to another, intended for subclassing.

### Fault Injection

You can use this to deterministically trigger file system failures in tests. This is useful to confirm that your program behaves correctly even if its file system operations fail. For example, this subclass fails every access of files named `unlucky.txt`:

```kotlin
val faultyFileSystem = object : ForwardingFileSystem(FileSystem.SYSTEM) {
  override fun onPathParameter(path: Path, functionName: String, parameterName: String): Path {
    if (path.name == "unlucky.txt") throw IOException("synthetic failure!")
    return path
  }
}
```

You can fail specific operations by overriding them directly:

```kotlin
val faultyFileSystem = object : ForwardingFileSystem(FileSystem.SYSTEM) {
  override fun delete(path: Path) {
    throw IOException("synthetic failure!")
  }
}
```

### Observability

You can extend this to verify which files your program accesses. This is a testing file system that records accesses as they happen:

```kotlin
class LoggingFileSystem : ForwardingFileSystem(FileSystem.SYSTEM) {
  val log = mutableListOf<String>()

  override fun onPathParameter(path: Path, functionName: String, parameterName: String): Path {
    log += "$functionName($parameterName=$path)"
    return path
  }
}
```

This makes it easy for tests to assert exactly which files were accessed.

```kotlin
@Test
fun testMergeJsonReports() {
  createSampleJsonReports()
  loggingFileSystem.log.clear()

  mergeJsonReports()

  assertThat(loggingFileSystem.log).containsExactly(
    "list(dir=json_reports)",
    "source(file=json_reports/2020-10.json)",
    "source(file=json_reports/2020-12.json)",
    "source(file=json_reports/2020-11.json)",
    "sink(file=json_reports/2020-all.json)"
  )
}
```

### Transformations

Subclasses can transform file names and content.

For example, your program may be written to operate on a well-known directory like `/etc/` or `/System`. You can rewrite paths to make such operations safer to test.

You may also transform file content to apply application-layer encryption or compression. This is particularly useful in situations where it's difficult or impossible to enable those features in the underlying file system.

### Abstract Functions Only

Some file system functions like copy are implemented by using other features. These are the non-abstract functions in the [FileSystem](../-file-system/index.md) interface.

**This class forwards only the abstract functions;** non-abstract functions delegate to the other functions of this class. If desired, subclasses may override non-abstract functions to forward them.

### Closeable

Closing this file system closes the delegate file system.

## Constructors

| | |
|---|---|
| [ForwardingFileSystem](-forwarding-file-system.md) | [common]<br>constructor(delegate: [FileSystem](../-file-system/index.md)) |

## Properties

| Name | Summary |
|---|---|
| [delegate](delegate.md) | [common]<br>@get:[JvmName](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin.jvm/-jvm-name/index.html)(name = &quot;delegate&quot;)<br>val [delegate](delegate.md): [FileSystem](../-file-system/index.md)<br>[FileSystem](../-file-system/index.md) to which this instance is delegating. |

## Functions

| Name | Summary |
|---|---|
| [appendingSink](appending-sink.md) | [common]<br>open override fun [appendingSink](appending-sink.md)(file: [Path](../-path/index.md), mustExist: [Boolean](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin/-boolean/index.html)): [Sink](../-sink/index.md)<br>Returns a sink that appends bytes to the end of [file](appending-sink.md), creating it if it doesn't already exist. |
| [atomicMove](atomic-move.md) | [common]<br>open override fun [atomicMove](atomic-move.md)(source: [Path](../-path/index.md), target: [Path](../-path/index.md))<br>Moves [source](atomic-move.md) to [target](atomic-move.md) in-place if the underlying file system supports it. If [target](atomic-move.md) exists, it is first removed. If `source == target`, this operation does nothing. This may be used to move a file or a directory. |
| [canonicalize](canonicalize.md) | [common]<br>open override fun [canonicalize](canonicalize.md)(path: [Path](../-path/index.md)): [Path](../-path/index.md)<br>Resolves [path](canonicalize.md) against the current working directory and symlinks in this file system. The returned path identifies the same file as [path](canonicalize.md), but with an absolute path that does not include any symbolic links. |
| [close](close.md) | [common]<br>open override fun [close](close.md)()<br>Closes this object and releases the resources it holds. It is an error to use an object after it has been closed. It is safe to close an object more than once. |
| [copy](../-file-system/copy.md) | [common]<br>expect open fun [copy](../-file-system/copy.md)(source: [Path](../-path/index.md), target: [Path](../-path/index.md))<br>Copies all the bytes from the file at [source](../-file-system/copy.md) to the file at [target](../-file-system/copy.md). This does not copy file metadata like last modified time, permissions, or extended attributes. |
| [createDirectories](../-file-system/create-directories.md) | [common]<br>expect fun [createDirectories](../-file-system/create-directories.md)(dir: [Path](../-path/index.md), mustCreate: [Boolean](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin/-boolean/index.html) = false)<br>Creates a directory at the path identified by [dir](../-file-system/create-directories.md), and any enclosing parent path directories, recursively. |
| [createDirectory](create-directory.md) | [common]<br>open override fun [createDirectory](create-directory.md)(dir: [Path](../-path/index.md), mustCreate: [Boolean](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin/-boolean/index.html))<br>Creates a directory at the path identified by [dir](create-directory.md). |
| [createSymlink](create-symlink.md) | [common]<br>open override fun [createSymlink](create-symlink.md)(source: [Path](../-path/index.md), target: [Path](../-path/index.md))<br>Creates a symbolic link at [source](create-symlink.md) that resolves to [target](create-symlink.md). If [target](create-symlink.md) is a relative path, it is relative to `source.parent`. |
| [delete](delete.md) | [common]<br>open override fun [delete](delete.md)(path: [Path](../-path/index.md), mustExist: [Boolean](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin/-boolean/index.html))<br>Deletes the file or directory at [path](delete.md). |
| [deleteRecursively](../-file-system/delete-recursively.md) | [common]<br>expect open fun [deleteRecursively](../-file-system/delete-recursively.md)(fileOrDirectory: [Path](../-path/index.md), mustExist: [Boolean](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin/-boolean/index.html) = false)<br>Recursively deletes all children of [fileOrDirectory](../-file-system/delete-recursively.md) if it is a directory, then deletes [fileOrDirectory](../-file-system/delete-recursively.md) itself. |
| [exists](../-file-system/exists.md) | [common]<br>expect fun [exists](../-file-system/exists.md)(path: [Path](../-path/index.md)): [Boolean](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin/-boolean/index.html)<br>Returns true if [path](../-file-system/exists.md) identifies an object on this file system. |
| [list](list.md) | [common]<br>open override fun [list](list.md)(dir: [Path](../-path/index.md)): [List](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin.collections/-list/index.html)&lt;[Path](../-path/index.md)&gt;<br>Returns the children of [dir](list.md). The returned list is sorted using natural ordering. If [dir](list.md) is a relative path, the returned elements will also be relative paths. If it is an absolute path, the returned elements will also be absolute paths. |
| [listOrNull](list-or-null.md) | [common]<br>open override fun [listOrNull](list-or-null.md)(dir: [Path](../-path/index.md)): [List](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin.collections/-list/index.html)&lt;[Path](../-path/index.md)&gt;?<br>Returns the children of the directory identified by [dir](list-or-null.md). The returned list is sorted using natural ordering. If [dir](list-or-null.md) is a relative path, the returned elements will also be relative paths. If it is an absolute path, the returned elements will also be absolute paths. |
| [listRecursively](list-recursively.md) | [common]<br>open override fun [listRecursively](list-recursively.md)(dir: [Path](../-path/index.md), followSymlinks: [Boolean](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin/-boolean/index.html)): [Sequence](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin.sequences/-sequence/index.html)&lt;[Path](../-path/index.md)&gt;<br>Returns a sequence that **lazily** traverses the children of [dir](list-recursively.md) using repeated calls to [list](list.md). If none of [dir](list-recursively.md)'s children are directories this returns the same elements as [list](list.md). |
| [metadata](../-file-system/metadata.md) | [common]<br>expect fun [metadata](../-file-system/metadata.md)(path: [Path](../-path/index.md)): [FileMetadata](../-file-metadata/index.md)<br>Returns metadata of the file, directory, or object identified by [path](../-file-system/metadata.md). |
| [metadataOrNull](metadata-or-null.md) | [common]<br>open override fun [metadataOrNull](metadata-or-null.md)(path: [Path](../-path/index.md)): [FileMetadata](../-file-metadata/index.md)?<br>Returns metadata of the file, directory, or object identified by [path](metadata-or-null.md). This returns null if there is no file at [path](metadata-or-null.md). |
| [onPathParameter](on-path-parameter.md) | [common]<br>open fun [onPathParameter](on-path-parameter.md)(path: [Path](../-path/index.md), functionName: [String](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin/-string/index.html), parameterName: [String](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin/-string/index.html)): [Path](../-path/index.md)<br>Invoked each time a path is passed as a parameter to this file system. This returns the path to pass to [delegate](delegate.md), which should be [path](on-path-parameter.md) itself or a path on [delegate](delegate.md) that corresponds to it. |
| [onPathResult](on-path-result.md) | [common]<br>open fun [onPathResult](on-path-result.md)(path: [Path](../-path/index.md), functionName: [String](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin/-string/index.html)): [Path](../-path/index.md)<br>Invoked each time a path is returned by [delegate](delegate.md). This returns the path to return to the caller, which should be [path](on-path-result.md) itself or a path on this that corresponds to it. |
| [openReadOnly](open-read-only.md) | [common]<br>open override fun [openReadOnly](open-read-only.md)(file: [Path](../-path/index.md)): [FileHandle](../-file-handle/index.md)<br>Returns a handle to read [file](open-read-only.md). This will fail if the file doesn't already exist. |
| [openReadWrite](open-read-write.md) | [common]<br>open override fun [openReadWrite](open-read-write.md)(file: [Path](../-path/index.md), mustCreate: [Boolean](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin/-boolean/index.html), mustExist: [Boolean](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin/-boolean/index.html)): [FileHandle](../-file-handle/index.md)<br>Returns a handle to read and write [file](open-read-write.md). This will create the file if it doesn't already exist. |
| [openZip](../open-zip.md) | [zlib]<br>fun [FileSystem](../-file-system/index.md#-199436775%2FMain%2F-1717108311).[openZip](../open-zip.md)(zipPath: Path): [FileSystem](../-file-system/index.md#-199436775%2FMain%2F-1717108311)<br>Returns a new read-only file system. |
| [read](../-file-system/read.md) | [common]<br>expect inline fun &lt;[T](../-file-system/read.md)&gt; [read](../-file-system/read.md)(file: [Path](../-path/index.md), readerAction: [BufferedSource](../-buffered-source/index.md).() -&gt; [T](../-file-system/read.md)): [T](../-file-system/read.md)<br>Creates a source to read [file](../-file-system/read.md), executes [readerAction](../-file-system/read.md) to read it, and then closes the source. This is a compact way to read the contents of a file. |
| [sink](sink.md) | [common]<br>open override fun [sink](sink.md)(file: [Path](../-path/index.md), mustCreate: [Boolean](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin/-boolean/index.html)): [Sink](../-sink/index.md)<br>Returns a sink that writes bytes to [file](sink.md) from beginning to end. If [file](sink.md) already exists it will be replaced with the new data. |
| [source](source.md) | [common]<br>open override fun [source](source.md)(file: [Path](../-path/index.md)): [Source](../-source/index.md)<br>Returns a source that reads the bytes of [file](source.md) from beginning to end. |
| [toString](to-string.md) | [common]<br>open override fun [toString](to-string.md)(): [String](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin/-string/index.html) |
| [write](../-file-system/write.md) | [common]<br>expect inline fun &lt;[T](../-file-system/write.md)&gt; [write](../-file-system/write.md)(file: [Path](../-path/index.md), mustCreate: [Boolean](https://kotlinlang.org/api/core/kotlin-stdlib/kotlin/-boolean/index.html) = false, writerAction: [BufferedSink](../-buffered-sink/index.md).() -&gt; [T](../-file-system/write.md)): [T](../-file-system/write.md)<br>Creates a sink to write [file](../-file-system/write.md), executes [writerAction](../-file-system/write.md) to write it, and then closes the sink. This is a compact way to write a file. |