// This file is generated by /utils/generate_types/index.js
/**
 * Copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { ChildProcess } from 'child_process';
import { EventEmitter } from 'events';
import { Readable } from 'stream';
import { ReadStream } from 'fs';
import { Protocol } from './protocol';
import { Serializable, EvaluationArgument, PageFunction, PageFunctionOn, SmartHandle, ElementHandleForTag, BindingSource } from './structs';

type PageWaitForSelectorOptionsNotHidden = PageWaitForSelectorOptions & {
  state?: 'visible'|'attached';
};
type ElementHandleWaitForSelectorOptionsNotHidden = ElementHandleWaitForSelectorOptions & {
  state?: 'visible'|'attached';
};

/**
 * - extends: [EventEmitter]
 *
 * Page provides methods to interact with a single tab in a [Browser], or an
 * [extension background page](https://developer.chrome.com/extensions/background_pages) in Chromium. One [Browser]
 * instance might have multiple [Page] instances.
 *
 * This example creates a page, navigates it to a URL, and then saves a screenshot:
 *
 * ```js
 * const { webkit } = require('playwright');  // Or 'chromium' or 'firefox'.
 *
 * (async () => {
 *   const browser = await webkit.launch();
 *   const context = await browser.newContext();
 *   const page = await context.newPage();
 *   await page.goto('https://example.com');
 *   await page.screenshot({path: 'screenshot.png'});
 *   await browser.close();
 * })();
 * ```
 *
 * The Page class emits various events (described below) which can be handled using any of Node's native
 * [`EventEmitter`](https://nodejs.org/api/events.html#events_class_eventemitter) methods, such as `on`, `once` or
 * `removeListener`.
 *
 * This example logs a message for a single page `load` event:
 *
 * ```js
 * page.once('load', () => console.log('Page loaded!'));
 * ```
 *
 * To unsubscribe from events use the `removeListener` method:
 *
 * ```js
 * function logRequest(interceptedRequest) {
 *   console.log('A request was made:', interceptedRequest.url());
 * }
 * page.on('request', logRequest);
 * // Sometime later...
 * page.removeListener('request', logRequest);
 * ```
 *
 */
export interface Page {
  /**
   * Returns the value of the `pageFunction` invocation.
   *
   * If the function passed to the
   * [page.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate) returns a [Promise],
   * then [page.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate) would wait for
   * the promise to resolve and return its value.
   *
   * If the function passed to the
   * [page.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate) returns a
   * non-[Serializable] value, then
   * [page.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate) resolves to
   * `undefined`. Playwright also supports transferring some additional values that are not serializable by `JSON`:
   * `-0`, `NaN`, `Infinity`, `-Infinity`.
   *
   * **Usage**
   *
   * Passing argument to `pageFunction`:
   *
   * ```js
   * const result = await page.evaluate(([x, y]) => {
   *   return Promise.resolve(x * y);
   * }, [7, 8]);
   * console.log(result); // prints "56"
   * ```
   *
   * A string can also be passed in instead of a function:
   *
   * ```js
   * console.log(await page.evaluate('1 + 2')); // prints "3"
   * const x = 10;
   * console.log(await page.evaluate(`1 + ${x}`)); // prints "11"
   * ```
   *
   * [ElementHandle] instances can be passed as an argument to the
   * [page.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate):
   *
   * ```js
   * const bodyHandle = await page.evaluate('document.body');
   * const html = await page.evaluate(([body, suffix]) => body.innerHTML + suffix, [bodyHandle, 'hello']);
   * await bodyHandle.dispose();
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluate<R, Arg>(pageFunction: PageFunction<Arg, R>, arg: Arg): Promise<R>;
  /**
   * Returns the value of the `pageFunction` invocation.
   *
   * If the function passed to the
   * [page.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate) returns a [Promise],
   * then [page.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate) would wait for
   * the promise to resolve and return its value.
   *
   * If the function passed to the
   * [page.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate) returns a
   * non-[Serializable] value, then
   * [page.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate) resolves to
   * `undefined`. Playwright also supports transferring some additional values that are not serializable by `JSON`:
   * `-0`, `NaN`, `Infinity`, `-Infinity`.
   *
   * **Usage**
   *
   * Passing argument to `pageFunction`:
   *
   * ```js
   * const result = await page.evaluate(([x, y]) => {
   *   return Promise.resolve(x * y);
   * }, [7, 8]);
   * console.log(result); // prints "56"
   * ```
   *
   * A string can also be passed in instead of a function:
   *
   * ```js
   * console.log(await page.evaluate('1 + 2')); // prints "3"
   * const x = 10;
   * console.log(await page.evaluate(`1 + ${x}`)); // prints "11"
   * ```
   *
   * [ElementHandle] instances can be passed as an argument to the
   * [page.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate):
   *
   * ```js
   * const bodyHandle = await page.evaluate('document.body');
   * const html = await page.evaluate(([body, suffix]) => body.innerHTML + suffix, [bodyHandle, 'hello']);
   * await bodyHandle.dispose();
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluate<R>(pageFunction: PageFunction<void, R>, arg?: any): Promise<R>;

  /**
   * Returns the value of the `pageFunction` invocation as a [JSHandle].
   *
   * The only difference between
   * [page.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate) and
   * [page.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate-handle) is that
   * [page.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate-handle) returns
   * [JSHandle].
   *
   * If the function passed to the
   * [page.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate-handle) returns
   * a [Promise], then
   * [page.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate-handle) would
   * wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const aWindowHandle = await page.evaluateHandle(() => Promise.resolve(window));
   * aWindowHandle; // Handle for the window object.
   * ```
   *
   * A string can also be passed in instead of a function:
   *
   * ```js
   * const aHandle = await page.evaluateHandle('document'); // Handle for the 'document'
   * ```
   *
   * [JSHandle] instances can be passed as an argument to the
   * [page.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate-handle):
   *
   * ```js
   * const aHandle = await page.evaluateHandle(() => document.body);
   * const resultHandle = await page.evaluateHandle(body => body.innerHTML, aHandle);
   * console.log(await resultHandle.jsonValue());
   * await resultHandle.dispose();
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluateHandle<R, Arg>(pageFunction: PageFunction<Arg, R>, arg: Arg): Promise<SmartHandle<R>>;
  /**
   * Returns the value of the `pageFunction` invocation as a [JSHandle].
   *
   * The only difference between
   * [page.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate) and
   * [page.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate-handle) is that
   * [page.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate-handle) returns
   * [JSHandle].
   *
   * If the function passed to the
   * [page.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate-handle) returns
   * a [Promise], then
   * [page.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate-handle) would
   * wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const aWindowHandle = await page.evaluateHandle(() => Promise.resolve(window));
   * aWindowHandle; // Handle for the window object.
   * ```
   *
   * A string can also be passed in instead of a function:
   *
   * ```js
   * const aHandle = await page.evaluateHandle('document'); // Handle for the 'document'
   * ```
   *
   * [JSHandle] instances can be passed as an argument to the
   * [page.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate-handle):
   *
   * ```js
   * const aHandle = await page.evaluateHandle(() => document.body);
   * const resultHandle = await page.evaluateHandle(body => body.innerHTML, aHandle);
   * console.log(await resultHandle.jsonValue());
   * await resultHandle.dispose();
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluateHandle<R>(pageFunction: PageFunction<void, R>, arg?: any): Promise<SmartHandle<R>>;

  /**
   * Adds a script which would be evaluated in one of the following scenarios:
   * - Whenever the page is navigated.
   * - Whenever the child frame is attached or navigated. In this case, the script is evaluated in the context of the
   *   newly attached frame.
   *
   * The script is evaluated after the document was created but before any of its scripts were run. This is useful to
   * amend the JavaScript environment, e.g. to seed `Math.random`.
   *
   * **Usage**
   *
   * An example of overriding `Math.random` before the page loads:
   *
   * ```js
   * // preload.js
   * Math.random = () => 42;
   * ```
   *
   * ```js
   * // In your playwright script, assuming the preload.js file is in same directory
   * await page.addInitScript({ path: './preload.js' });
   * ```
   *
   * **NOTE** The order of evaluation of multiple scripts installed via
   * [browserContext.addInitScript(script[, arg])](https://playwright.dev/docs/api/class-browsercontext#browser-context-add-init-script)
   * and [page.addInitScript(script[, arg])](https://playwright.dev/docs/api/class-page#page-add-init-script) is not
   * defined.
   * @param script Script to be evaluated in the page.
   * @param arg Optional argument to pass to `script` (only supported when passing a function).
   */
  addInitScript<Arg>(script: PageFunction<Arg, any> | { path?: string, content?: string }, arg?: Arg): Promise<void>;

  /**
   * **NOTE** Use locator-based [page.locator(selector[, options])](https://playwright.dev/docs/api/class-page#page-locator)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * The method finds an element matching the specified selector within the page. If no elements match the selector, the
   * return value resolves to `null`. To wait for an element on the page, use
   * [locator.waitFor([options])](https://playwright.dev/docs/api/class-locator#locator-wait-for).
   * @param selector A selector to query for.
   * @param options
   */
  $<K extends keyof HTMLElementTagNameMap>(selector: K, options?: { strict: boolean }): Promise<ElementHandleForTag<K> | null>;
  /**
   * **NOTE** Use locator-based [page.locator(selector[, options])](https://playwright.dev/docs/api/class-page#page-locator)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * The method finds an element matching the specified selector within the page. If no elements match the selector, the
   * return value resolves to `null`. To wait for an element on the page, use
   * [locator.waitFor([options])](https://playwright.dev/docs/api/class-locator#locator-wait-for).
   * @param selector A selector to query for.
   * @param options
   */
  $(selector: string, options?: { strict: boolean }): Promise<ElementHandle<SVGElement | HTMLElement> | null>;

  /**
   * **NOTE** Use locator-based [page.locator(selector[, options])](https://playwright.dev/docs/api/class-page#page-locator)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * The method finds all elements matching the specified selector within the page. If no elements match the selector,
   * the return value resolves to `[]`.
   * @param selector A selector to query for.
   */
  $$<K extends keyof HTMLElementTagNameMap>(selector: K): Promise<ElementHandleForTag<K>[]>;
  /**
   * **NOTE** Use locator-based [page.locator(selector[, options])](https://playwright.dev/docs/api/class-page#page-locator)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * The method finds all elements matching the specified selector within the page. If no elements match the selector,
   * the return value resolves to `[]`.
   * @param selector A selector to query for.
   */
  $$(selector: string): Promise<ElementHandle<SVGElement | HTMLElement>[]>;

  /**
   * **NOTE** This method does not wait for the element to pass actionability checks and therefore can lead to the flaky tests.
   * Use
   * [locator.evaluate(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-locator#locator-evaluate),
   * other [Locator] helper methods or web-first assertions instead.
   *
   * The method finds an element matching the specified selector within the page and passes it as a first argument to
   * `pageFunction`. If no elements match the selector, the method throws an error. Returns the value of `pageFunction`.
   *
   * If `pageFunction` returns a [Promise], then
   * [page.$eval(selector, pageFunction[, arg, options])](https://playwright.dev/docs/api/class-page#page-eval-on-selector)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const searchValue = await page.$eval('#search', el => el.value);
   * const preloadHref = await page.$eval('link[rel=preload]', el => el.href);
   * const html = await page.$eval('.main-container', (e, suffix) => e.outerHTML + suffix, 'hello');
   * // In TypeScript, this example requires an explicit type annotation (HTMLLinkElement) on el:
   * const preloadHrefTS = await page.$eval('link[rel=preload]', (el: HTMLLinkElement) => el.href);
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   * @param options
   */
  $eval<K extends keyof HTMLElementTagNameMap, R, Arg>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K], Arg, R>, arg: Arg): Promise<R>;
  /**
   * **NOTE** This method does not wait for the element to pass actionability checks and therefore can lead to the flaky tests.
   * Use
   * [locator.evaluate(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-locator#locator-evaluate),
   * other [Locator] helper methods or web-first assertions instead.
   *
   * The method finds an element matching the specified selector within the page and passes it as a first argument to
   * `pageFunction`. If no elements match the selector, the method throws an error. Returns the value of `pageFunction`.
   *
   * If `pageFunction` returns a [Promise], then
   * [page.$eval(selector, pageFunction[, arg, options])](https://playwright.dev/docs/api/class-page#page-eval-on-selector)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const searchValue = await page.$eval('#search', el => el.value);
   * const preloadHref = await page.$eval('link[rel=preload]', el => el.href);
   * const html = await page.$eval('.main-container', (e, suffix) => e.outerHTML + suffix, 'hello');
   * // In TypeScript, this example requires an explicit type annotation (HTMLLinkElement) on el:
   * const preloadHrefTS = await page.$eval('link[rel=preload]', (el: HTMLLinkElement) => el.href);
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   * @param options
   */
  $eval<R, Arg, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E, Arg, R>, arg: Arg): Promise<R>;
  /**
   * **NOTE** This method does not wait for the element to pass actionability checks and therefore can lead to the flaky tests.
   * Use
   * [locator.evaluate(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-locator#locator-evaluate),
   * other [Locator] helper methods or web-first assertions instead.
   *
   * The method finds an element matching the specified selector within the page and passes it as a first argument to
   * `pageFunction`. If no elements match the selector, the method throws an error. Returns the value of `pageFunction`.
   *
   * If `pageFunction` returns a [Promise], then
   * [page.$eval(selector, pageFunction[, arg, options])](https://playwright.dev/docs/api/class-page#page-eval-on-selector)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const searchValue = await page.$eval('#search', el => el.value);
   * const preloadHref = await page.$eval('link[rel=preload]', el => el.href);
   * const html = await page.$eval('.main-container', (e, suffix) => e.outerHTML + suffix, 'hello');
   * // In TypeScript, this example requires an explicit type annotation (HTMLLinkElement) on el:
   * const preloadHrefTS = await page.$eval('link[rel=preload]', (el: HTMLLinkElement) => el.href);
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   * @param options
   */
  $eval<K extends keyof HTMLElementTagNameMap, R>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K], void, R>, arg?: any): Promise<R>;
  /**
   * **NOTE** This method does not wait for the element to pass actionability checks and therefore can lead to the flaky tests.
   * Use
   * [locator.evaluate(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-locator#locator-evaluate),
   * other [Locator] helper methods or web-first assertions instead.
   *
   * The method finds an element matching the specified selector within the page and passes it as a first argument to
   * `pageFunction`. If no elements match the selector, the method throws an error. Returns the value of `pageFunction`.
   *
   * If `pageFunction` returns a [Promise], then
   * [page.$eval(selector, pageFunction[, arg, options])](https://playwright.dev/docs/api/class-page#page-eval-on-selector)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const searchValue = await page.$eval('#search', el => el.value);
   * const preloadHref = await page.$eval('link[rel=preload]', el => el.href);
   * const html = await page.$eval('.main-container', (e, suffix) => e.outerHTML + suffix, 'hello');
   * // In TypeScript, this example requires an explicit type annotation (HTMLLinkElement) on el:
   * const preloadHrefTS = await page.$eval('link[rel=preload]', (el: HTMLLinkElement) => el.href);
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   * @param options
   */
  $eval<R, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E, void, R>, arg?: any): Promise<R>;

  /**
   * **NOTE** In most cases,
   * [locator.evaluateAll(pageFunction[, arg])](https://playwright.dev/docs/api/class-locator#locator-evaluate-all),
   * other [Locator] helper methods and web-first assertions do a better job.
   *
   * The method finds all elements matching the specified selector within the page and passes an array of matched
   * elements as a first argument to `pageFunction`. Returns the result of `pageFunction` invocation.
   *
   * If `pageFunction` returns a [Promise], then
   * [page.$$eval(selector, pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-eval-on-selector-all)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const divCounts = await page.$$eval('div', (divs, min) => divs.length >= min, 10);
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  $$eval<K extends keyof HTMLElementTagNameMap, R, Arg>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K][], Arg, R>, arg: Arg): Promise<R>;
  /**
   * **NOTE** In most cases,
   * [locator.evaluateAll(pageFunction[, arg])](https://playwright.dev/docs/api/class-locator#locator-evaluate-all),
   * other [Locator] helper methods and web-first assertions do a better job.
   *
   * The method finds all elements matching the specified selector within the page and passes an array of matched
   * elements as a first argument to `pageFunction`. Returns the result of `pageFunction` invocation.
   *
   * If `pageFunction` returns a [Promise], then
   * [page.$$eval(selector, pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-eval-on-selector-all)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const divCounts = await page.$$eval('div', (divs, min) => divs.length >= min, 10);
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  $$eval<R, Arg, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E[], Arg, R>, arg: Arg): Promise<R>;
  /**
   * **NOTE** In most cases,
   * [locator.evaluateAll(pageFunction[, arg])](https://playwright.dev/docs/api/class-locator#locator-evaluate-all),
   * other [Locator] helper methods and web-first assertions do a better job.
   *
   * The method finds all elements matching the specified selector within the page and passes an array of matched
   * elements as a first argument to `pageFunction`. Returns the result of `pageFunction` invocation.
   *
   * If `pageFunction` returns a [Promise], then
   * [page.$$eval(selector, pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-eval-on-selector-all)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const divCounts = await page.$$eval('div', (divs, min) => divs.length >= min, 10);
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  $$eval<K extends keyof HTMLElementTagNameMap, R>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K][], void, R>, arg?: any): Promise<R>;
  /**
   * **NOTE** In most cases,
   * [locator.evaluateAll(pageFunction[, arg])](https://playwright.dev/docs/api/class-locator#locator-evaluate-all),
   * other [Locator] helper methods and web-first assertions do a better job.
   *
   * The method finds all elements matching the specified selector within the page and passes an array of matched
   * elements as a first argument to `pageFunction`. Returns the result of `pageFunction` invocation.
   *
   * If `pageFunction` returns a [Promise], then
   * [page.$$eval(selector, pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-eval-on-selector-all)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const divCounts = await page.$$eval('div', (divs, min) => divs.length >= min, 10);
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  $$eval<R, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E[], void, R>, arg?: any): Promise<R>;

  /**
   * Returns when the `pageFunction` returns a truthy value. It resolves to a JSHandle of the truthy value.
   *
   * **Usage**
   *
   * The
   * [page.waitForFunction(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-page#page-wait-for-function)
   * can be used to observe viewport size change:
   *
   * ```js
   * const { webkit } = require('playwright');  // Or 'chromium' or 'firefox'.
   *
   * (async () => {
   *   const browser = await webkit.launch();
   *   const page = await browser.newPage();
   *   const watchDog = page.waitForFunction(() => window.innerWidth < 100);
   *   await page.setViewportSize({width: 50, height: 50});
   *   await watchDog;
   *   await browser.close();
   * })();
   * ```
   *
   * To pass an argument to the predicate of
   * [page.waitForFunction(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-page#page-wait-for-function)
   * function:
   *
   * ```js
   * const selector = '.foo';
   * await page.waitForFunction(selector => !!document.querySelector(selector), selector);
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   * @param options
   */
  waitForFunction<R, Arg>(pageFunction: PageFunction<Arg, R>, arg: Arg, options?: PageWaitForFunctionOptions): Promise<SmartHandle<R>>;
  /**
   * Returns when the `pageFunction` returns a truthy value. It resolves to a JSHandle of the truthy value.
   *
   * **Usage**
   *
   * The
   * [page.waitForFunction(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-page#page-wait-for-function)
   * can be used to observe viewport size change:
   *
   * ```js
   * const { webkit } = require('playwright');  // Or 'chromium' or 'firefox'.
   *
   * (async () => {
   *   const browser = await webkit.launch();
   *   const page = await browser.newPage();
   *   const watchDog = page.waitForFunction(() => window.innerWidth < 100);
   *   await page.setViewportSize({width: 50, height: 50});
   *   await watchDog;
   *   await browser.close();
   * })();
   * ```
   *
   * To pass an argument to the predicate of
   * [page.waitForFunction(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-page#page-wait-for-function)
   * function:
   *
   * ```js
   * const selector = '.foo';
   * await page.waitForFunction(selector => !!document.querySelector(selector), selector);
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   * @param options
   */
  waitForFunction<R>(pageFunction: PageFunction<void, R>, arg?: any, options?: PageWaitForFunctionOptions): Promise<SmartHandle<R>>;

  /**
   * Returns when element specified by selector satisfies `state` option. Returns `null` if waiting for `hidden` or
   * `detached`.
   *
   * **NOTE** Playwright automatically waits for element to be ready before performing an action. Using [Locator]
   * objects and web-first assertions makes the code wait-for-selector-free.
   *
   * Wait for the `selector` to satisfy `state` option (either appear/disappear from dom, or become visible/hidden). If
   * at the moment of calling the method `selector` already satisfies the condition, the method will return immediately.
   * If the selector doesn't satisfy the condition for the `timeout` milliseconds, the function will throw.
   *
   * **Usage**
   *
   * This method works across navigations:
   *
   * ```js
   * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
   *
   * (async () => {
   *   const browser = await chromium.launch();
   *   const page = await browser.newPage();
   *   for (let currentURL of ['https://google.com', 'https://bbc.com']) {
   *     await page.goto(currentURL);
   *     const element = await page.waitForSelector('img');
   *     console.log('Loaded image: ' + await element.getAttribute('src'));
   *   }
   *   await browser.close();
   * })();
   * ```
   *
   * @param selector A selector to query for.
   * @param options
   */
  waitForSelector<K extends keyof HTMLElementTagNameMap>(selector: K, options?: PageWaitForSelectorOptionsNotHidden): Promise<ElementHandleForTag<K>>;
  /**
   * Returns when element specified by selector satisfies `state` option. Returns `null` if waiting for `hidden` or
   * `detached`.
   *
   * **NOTE** Playwright automatically waits for element to be ready before performing an action. Using [Locator]
   * objects and web-first assertions makes the code wait-for-selector-free.
   *
   * Wait for the `selector` to satisfy `state` option (either appear/disappear from dom, or become visible/hidden). If
   * at the moment of calling the method `selector` already satisfies the condition, the method will return immediately.
   * If the selector doesn't satisfy the condition for the `timeout` milliseconds, the function will throw.
   *
   * **Usage**
   *
   * This method works across navigations:
   *
   * ```js
   * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
   *
   * (async () => {
   *   const browser = await chromium.launch();
   *   const page = await browser.newPage();
   *   for (let currentURL of ['https://google.com', 'https://bbc.com']) {
   *     await page.goto(currentURL);
   *     const element = await page.waitForSelector('img');
   *     console.log('Loaded image: ' + await element.getAttribute('src'));
   *   }
   *   await browser.close();
   * })();
   * ```
   *
   * @param selector A selector to query for.
   * @param options
   */
  waitForSelector(selector: string, options?: PageWaitForSelectorOptionsNotHidden): Promise<ElementHandle<SVGElement | HTMLElement>>;
  /**
   * Returns when element specified by selector satisfies `state` option. Returns `null` if waiting for `hidden` or
   * `detached`.
   *
   * **NOTE** Playwright automatically waits for element to be ready before performing an action. Using [Locator]
   * objects and web-first assertions makes the code wait-for-selector-free.
   *
   * Wait for the `selector` to satisfy `state` option (either appear/disappear from dom, or become visible/hidden). If
   * at the moment of calling the method `selector` already satisfies the condition, the method will return immediately.
   * If the selector doesn't satisfy the condition for the `timeout` milliseconds, the function will throw.
   *
   * **Usage**
   *
   * This method works across navigations:
   *
   * ```js
   * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
   *
   * (async () => {
   *   const browser = await chromium.launch();
   *   const page = await browser.newPage();
   *   for (let currentURL of ['https://google.com', 'https://bbc.com']) {
   *     await page.goto(currentURL);
   *     const element = await page.waitForSelector('img');
   *     console.log('Loaded image: ' + await element.getAttribute('src'));
   *   }
   *   await browser.close();
   * })();
   * ```
   *
   * @param selector A selector to query for.
   * @param options
   */
  waitForSelector<K extends keyof HTMLElementTagNameMap>(selector: K, options: PageWaitForSelectorOptions): Promise<ElementHandleForTag<K> | null>;
  /**
   * Returns when element specified by selector satisfies `state` option. Returns `null` if waiting for `hidden` or
   * `detached`.
   *
   * **NOTE** Playwright automatically waits for element to be ready before performing an action. Using [Locator]
   * objects and web-first assertions makes the code wait-for-selector-free.
   *
   * Wait for the `selector` to satisfy `state` option (either appear/disappear from dom, or become visible/hidden). If
   * at the moment of calling the method `selector` already satisfies the condition, the method will return immediately.
   * If the selector doesn't satisfy the condition for the `timeout` milliseconds, the function will throw.
   *
   * **Usage**
   *
   * This method works across navigations:
   *
   * ```js
   * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
   *
   * (async () => {
   *   const browser = await chromium.launch();
   *   const page = await browser.newPage();
   *   for (let currentURL of ['https://google.com', 'https://bbc.com']) {
   *     await page.goto(currentURL);
   *     const element = await page.waitForSelector('img');
   *     console.log('Loaded image: ' + await element.getAttribute('src'));
   *   }
   *   await browser.close();
   * })();
   * ```
   *
   * @param selector A selector to query for.
   * @param options
   */
  waitForSelector(selector: string, options: PageWaitForSelectorOptions): Promise<null|ElementHandle<SVGElement | HTMLElement>>;

  /**
   * The method adds a function called `name` on the `window` object of every frame in this page. When called, the
   * function executes `callback` and returns a [Promise] which resolves to the return value of `callback`. If the
   * `callback` returns a [Promise], it will be awaited.
   *
   * The first argument of the `callback` function contains information about the caller: `{ browserContext:
   * BrowserContext, page: Page, frame: Frame }`.
   *
   * See
   * [browserContext.exposeBinding(name, callback[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-expose-binding)
   * for the context-wide version.
   *
   * **NOTE** Functions installed via
   * [page.exposeBinding(name, callback[, options])](https://playwright.dev/docs/api/class-page#page-expose-binding)
   * survive navigations.
   *
   * **Usage**
   *
   * An example of exposing page URL to all frames in a page:
   *
   * ```js
   * const { webkit } = require('playwright');  // Or 'chromium' or 'firefox'.
   *
   * (async () => {
   *   const browser = await webkit.launch({ headless: false });
   *   const context = await browser.newContext();
   *   const page = await context.newPage();
   *   await page.exposeBinding('pageURL', ({ page }) => page.url());
   *   await page.setContent(`
   *     <script>
   *       async function onClick() {
   *         document.querySelector('div').textContent = await window.pageURL();
   *       }
   *     </script>
   *     <button onclick="onClick()">Click me</button>
   *     <div></div>
   *   `);
   *   await page.click('button');
   * })();
   * ```
   *
   * An example of passing an element handle:
   *
   * ```js
   * await page.exposeBinding('clicked', async (source, element) => {
   *   console.log(await element.textContent());
   * }, { handle: true });
   * await page.setContent(`
   *   <script>
   *     document.addEventListener('click', event => window.clicked(event.target));
   *   </script>
   *   <div>Click me</div>
   *   <div>Or click me</div>
   * `);
   * ```
   *
   * @param name Name of the function on the window object.
   * @param callback Callback function that will be called in the Playwright's context.
   * @param options
   */
  exposeBinding(name: string, playwrightBinding: (source: BindingSource, arg: JSHandle) => any, options: { handle: true }): Promise<void>;
  /**
   * The method adds a function called `name` on the `window` object of every frame in this page. When called, the
   * function executes `callback` and returns a [Promise] which resolves to the return value of `callback`. If the
   * `callback` returns a [Promise], it will be awaited.
   *
   * The first argument of the `callback` function contains information about the caller: `{ browserContext:
   * BrowserContext, page: Page, frame: Frame }`.
   *
   * See
   * [browserContext.exposeBinding(name, callback[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-expose-binding)
   * for the context-wide version.
   *
   * **NOTE** Functions installed via
   * [page.exposeBinding(name, callback[, options])](https://playwright.dev/docs/api/class-page#page-expose-binding)
   * survive navigations.
   *
   * **Usage**
   *
   * An example of exposing page URL to all frames in a page:
   *
   * ```js
   * const { webkit } = require('playwright');  // Or 'chromium' or 'firefox'.
   *
   * (async () => {
   *   const browser = await webkit.launch({ headless: false });
   *   const context = await browser.newContext();
   *   const page = await context.newPage();
   *   await page.exposeBinding('pageURL', ({ page }) => page.url());
   *   await page.setContent(`
   *     <script>
   *       async function onClick() {
   *         document.querySelector('div').textContent = await window.pageURL();
   *       }
   *     </script>
   *     <button onclick="onClick()">Click me</button>
   *     <div></div>
   *   `);
   *   await page.click('button');
   * })();
   * ```
   *
   * An example of passing an element handle:
   *
   * ```js
   * await page.exposeBinding('clicked', async (source, element) => {
   *   console.log(await element.textContent());
   * }, { handle: true });
   * await page.setContent(`
   *   <script>
   *     document.addEventListener('click', event => window.clicked(event.target));
   *   </script>
   *   <div>Click me</div>
   *   <div>Or click me</div>
   * `);
   * ```
   *
   * @param name Name of the function on the window object.
   * @param callback Callback function that will be called in the Playwright's context.
   * @param options
   */
  exposeBinding(name: string, playwrightBinding: (source: BindingSource, ...args: any[]) => any, options?: { handle?: boolean }): Promise<void>;
  /**
   * Emitted when the page closes.
   */
  on(event: 'close', listener: (page: Page) => void): this;

  /**
   * Emitted when JavaScript within the page calls one of console API methods, e.g. `console.log` or `console.dir`. Also
   * emitted if the page throws an error or a warning.
   *
   * The arguments passed into `console.log` appear as arguments on the event handler.
   *
   * An example of handling `console` event:
   *
   * ```js
   * page.on('console', async msg => {
   *   const values = [];
   *   for (const arg of msg.args())
   *     values.push(await arg.jsonValue());
   *   console.log(...values);
   * });
   * await page.evaluate(() => console.log('hello', 5, {foo: 'bar'}));
   * ```
   *
   */
  on(event: 'console', listener: (consoleMessage: ConsoleMessage) => void): this;

  /**
   * Emitted when the page crashes. Browser pages might crash if they try to allocate too much memory. When the page
   * crashes, ongoing and subsequent operations will throw.
   *
   * The most common way to deal with crashes is to catch an exception:
   *
   * ```js
   * try {
   *   // Crash might happen during a click.
   *   await page.click('button');
   *   // Or while waiting for an event.
   *   await page.waitForEvent('popup');
   * } catch (e) {
   *   // When the page crashes, exception message contains 'crash'.
   * }
   * ```
   *
   */
  on(event: 'crash', listener: (page: Page) => void): this;

  /**
   * Emitted when a JavaScript dialog appears, such as `alert`, `prompt`, `confirm` or `beforeunload`. Listener **must**
   * either [dialog.accept([promptText])](https://playwright.dev/docs/api/class-dialog#dialog-accept) or
   * [dialog.dismiss()](https://playwright.dev/docs/api/class-dialog#dialog-dismiss) the dialog - otherwise the page
   * will [freeze](https://developer.mozilla.org/en-US/docs/Web/JavaScript/EventLoop#never_blocking) waiting for the
   * dialog, and actions like click will never finish.
   *
   * ```js
   * page.on('dialog', dialog => {
   *   dialog.accept();
   * });
   * ```
   *
   * **NOTE** When no [page.on('dialog')](https://playwright.dev/docs/api/class-page#page-event-dialog) listeners are
   * present, all dialogs are automatically dismissed.
   */
  on(event: 'dialog', listener: (dialog: Dialog) => void): this;

  /**
   * Emitted when the JavaScript
   * [`DOMContentLoaded`](https://developer.mozilla.org/en-US/docs/Web/Events/DOMContentLoaded) event is dispatched.
   */
  on(event: 'domcontentloaded', listener: (page: Page) => void): this;

  /**
   * Emitted when attachment download started. User can access basic file operations on downloaded content via the
   * passed [Download] instance.
   */
  on(event: 'download', listener: (download: Download) => void): this;

  /**
   * Emitted when a file chooser is supposed to appear, such as after clicking the  `<input type=file>`. Playwright can
   * respond to it via setting the input files using
   * [fileChooser.setFiles(files[, options])](https://playwright.dev/docs/api/class-filechooser#file-chooser-set-files)
   * that can be uploaded after that.
   *
   * ```js
   * page.on('filechooser', async (fileChooser) => {
   *   await fileChooser.setFiles('/tmp/myfile.pdf');
   * });
   * ```
   *
   */
  on(event: 'filechooser', listener: (fileChooser: FileChooser) => void): this;

  /**
   * Emitted when a frame is attached.
   */
  on(event: 'frameattached', listener: (frame: Frame) => void): this;

  /**
   * Emitted when a frame is detached.
   */
  on(event: 'framedetached', listener: (frame: Frame) => void): this;

  /**
   * Emitted when a frame is navigated to a new url.
   */
  on(event: 'framenavigated', listener: (frame: Frame) => void): this;

  /**
   * Emitted when the JavaScript [`load`](https://developer.mozilla.org/en-US/docs/Web/Events/load) event is dispatched.
   */
  on(event: 'load', listener: (page: Page) => void): this;

  /**
   * Emitted when an uncaught exception happens within the page.
   *
   * ```js
   * // Log all uncaught errors to the terminal
   * page.on('pageerror', exception => {
   *   console.log(`Uncaught exception: "${exception}"`);
   * });
   *
   * // Navigate to a page with an exception.
   * await page.goto('data:text/html,<script>throw new Error("Test")</script>');
   * ```
   *
   */
  on(event: 'pageerror', listener: (error: Error) => void): this;

  /**
   * Emitted when the page opens a new tab or window. This event is emitted in addition to the
   * [browserContext.on('page')](https://playwright.dev/docs/api/class-browsercontext#browser-context-event-page), but
   * only for popups relevant to this page.
   *
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to
   * "http://example.com" is done and its response has started loading in the popup.
   *
   * ```js
   * // Start waiting for popup before clicking. Note no await.
   * const popupPromise = page.waitForEvent('popup');
   * await page.getByText('open the popup').click();
   * const popup = await popupPromise;
   * console.log(await popup.evaluate('location.href'));
   * ```
   *
   * **NOTE** Use
   * [page.waitForLoadState([state, options])](https://playwright.dev/docs/api/class-page#page-wait-for-load-state) to
   * wait until the page gets to a particular state (you should not need it in most cases).
   */
  on(event: 'popup', listener: (page: Page) => void): this;

  /**
   * Emitted when a page issues a request. The [request] object is read-only. In order to intercept and mutate requests,
   * see [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route) or
   * [browserContext.route(url, handler[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-route).
   */
  on(event: 'request', listener: (request: Request) => void): this;

  /**
   * Emitted when a request fails, for example by timing out.
   *
   * ```js
   * page.on('requestfailed', request => {
   *   console.log(request.url() + ' ' + request.failure().errorText);
   * });
   * ```
   *
   * **NOTE** HTTP Error responses, such as 404 or 503, are still successful responses from HTTP standpoint, so request
   * will complete with
   * [page.on('requestfinished')](https://playwright.dev/docs/api/class-page#page-event-request-finished) event and not
   * with [page.on('requestfailed')](https://playwright.dev/docs/api/class-page#page-event-request-failed). A request
   * will only be considered failed when the client cannot get an HTTP response from the server, e.g. due to network
   * error net::ERR_FAILED.
   */
  on(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Emitted when a request finishes successfully after downloading the response body. For a successful response, the
   * sequence of events is `request`, `response` and `requestfinished`.
   */
  on(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Emitted when [response] status and headers are received for a request. For a successful response, the sequence of
   * events is `request`, `response` and `requestfinished`.
   */
  on(event: 'response', listener: (response: Response) => void): this;

  /**
   * Emitted when [WebSocket] request is sent.
   */
  on(event: 'websocket', listener: (webSocket: WebSocket) => void): this;

  /**
   * Emitted when a dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is spawned
   * by the page.
   */
  on(event: 'worker', listener: (worker: Worker) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'close', listener: (page: Page) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'console', listener: (consoleMessage: ConsoleMessage) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'crash', listener: (page: Page) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'dialog', listener: (dialog: Dialog) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'domcontentloaded', listener: (page: Page) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'download', listener: (download: Download) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'filechooser', listener: (fileChooser: FileChooser) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'frameattached', listener: (frame: Frame) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'framedetached', listener: (frame: Frame) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'framenavigated', listener: (frame: Frame) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'load', listener: (page: Page) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'pageerror', listener: (error: Error) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'popup', listener: (page: Page) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'request', listener: (request: Request) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'response', listener: (response: Response) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'websocket', listener: (webSocket: WebSocket) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'worker', listener: (worker: Worker) => void): this;

  /**
   * Emitted when the page closes.
   */
  addListener(event: 'close', listener: (page: Page) => void): this;

  /**
   * Emitted when JavaScript within the page calls one of console API methods, e.g. `console.log` or `console.dir`. Also
   * emitted if the page throws an error or a warning.
   *
   * The arguments passed into `console.log` appear as arguments on the event handler.
   *
   * An example of handling `console` event:
   *
   * ```js
   * page.on('console', async msg => {
   *   const values = [];
   *   for (const arg of msg.args())
   *     values.push(await arg.jsonValue());
   *   console.log(...values);
   * });
   * await page.evaluate(() => console.log('hello', 5, {foo: 'bar'}));
   * ```
   *
   */
  addListener(event: 'console', listener: (consoleMessage: ConsoleMessage) => void): this;

  /**
   * Emitted when the page crashes. Browser pages might crash if they try to allocate too much memory. When the page
   * crashes, ongoing and subsequent operations will throw.
   *
   * The most common way to deal with crashes is to catch an exception:
   *
   * ```js
   * try {
   *   // Crash might happen during a click.
   *   await page.click('button');
   *   // Or while waiting for an event.
   *   await page.waitForEvent('popup');
   * } catch (e) {
   *   // When the page crashes, exception message contains 'crash'.
   * }
   * ```
   *
   */
  addListener(event: 'crash', listener: (page: Page) => void): this;

  /**
   * Emitted when a JavaScript dialog appears, such as `alert`, `prompt`, `confirm` or `beforeunload`. Listener **must**
   * either [dialog.accept([promptText])](https://playwright.dev/docs/api/class-dialog#dialog-accept) or
   * [dialog.dismiss()](https://playwright.dev/docs/api/class-dialog#dialog-dismiss) the dialog - otherwise the page
   * will [freeze](https://developer.mozilla.org/en-US/docs/Web/JavaScript/EventLoop#never_blocking) waiting for the
   * dialog, and actions like click will never finish.
   *
   * ```js
   * page.on('dialog', dialog => {
   *   dialog.accept();
   * });
   * ```
   *
   * **NOTE** When no [page.on('dialog')](https://playwright.dev/docs/api/class-page#page-event-dialog) listeners are
   * present, all dialogs are automatically dismissed.
   */
  addListener(event: 'dialog', listener: (dialog: Dialog) => void): this;

  /**
   * Emitted when the JavaScript
   * [`DOMContentLoaded`](https://developer.mozilla.org/en-US/docs/Web/Events/DOMContentLoaded) event is dispatched.
   */
  addListener(event: 'domcontentloaded', listener: (page: Page) => void): this;

  /**
   * Emitted when attachment download started. User can access basic file operations on downloaded content via the
   * passed [Download] instance.
   */
  addListener(event: 'download', listener: (download: Download) => void): this;

  /**
   * Emitted when a file chooser is supposed to appear, such as after clicking the  `<input type=file>`. Playwright can
   * respond to it via setting the input files using
   * [fileChooser.setFiles(files[, options])](https://playwright.dev/docs/api/class-filechooser#file-chooser-set-files)
   * that can be uploaded after that.
   *
   * ```js
   * page.on('filechooser', async (fileChooser) => {
   *   await fileChooser.setFiles('/tmp/myfile.pdf');
   * });
   * ```
   *
   */
  addListener(event: 'filechooser', listener: (fileChooser: FileChooser) => void): this;

  /**
   * Emitted when a frame is attached.
   */
  addListener(event: 'frameattached', listener: (frame: Frame) => void): this;

  /**
   * Emitted when a frame is detached.
   */
  addListener(event: 'framedetached', listener: (frame: Frame) => void): this;

  /**
   * Emitted when a frame is navigated to a new url.
   */
  addListener(event: 'framenavigated', listener: (frame: Frame) => void): this;

  /**
   * Emitted when the JavaScript [`load`](https://developer.mozilla.org/en-US/docs/Web/Events/load) event is dispatched.
   */
  addListener(event: 'load', listener: (page: Page) => void): this;

  /**
   * Emitted when an uncaught exception happens within the page.
   *
   * ```js
   * // Log all uncaught errors to the terminal
   * page.on('pageerror', exception => {
   *   console.log(`Uncaught exception: "${exception}"`);
   * });
   *
   * // Navigate to a page with an exception.
   * await page.goto('data:text/html,<script>throw new Error("Test")</script>');
   * ```
   *
   */
  addListener(event: 'pageerror', listener: (error: Error) => void): this;

  /**
   * Emitted when the page opens a new tab or window. This event is emitted in addition to the
   * [browserContext.on('page')](https://playwright.dev/docs/api/class-browsercontext#browser-context-event-page), but
   * only for popups relevant to this page.
   *
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to
   * "http://example.com" is done and its response has started loading in the popup.
   *
   * ```js
   * // Start waiting for popup before clicking. Note no await.
   * const popupPromise = page.waitForEvent('popup');
   * await page.getByText('open the popup').click();
   * const popup = await popupPromise;
   * console.log(await popup.evaluate('location.href'));
   * ```
   *
   * **NOTE** Use
   * [page.waitForLoadState([state, options])](https://playwright.dev/docs/api/class-page#page-wait-for-load-state) to
   * wait until the page gets to a particular state (you should not need it in most cases).
   */
  addListener(event: 'popup', listener: (page: Page) => void): this;

  /**
   * Emitted when a page issues a request. The [request] object is read-only. In order to intercept and mutate requests,
   * see [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route) or
   * [browserContext.route(url, handler[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-route).
   */
  addListener(event: 'request', listener: (request: Request) => void): this;

  /**
   * Emitted when a request fails, for example by timing out.
   *
   * ```js
   * page.on('requestfailed', request => {
   *   console.log(request.url() + ' ' + request.failure().errorText);
   * });
   * ```
   *
   * **NOTE** HTTP Error responses, such as 404 or 503, are still successful responses from HTTP standpoint, so request
   * will complete with
   * [page.on('requestfinished')](https://playwright.dev/docs/api/class-page#page-event-request-finished) event and not
   * with [page.on('requestfailed')](https://playwright.dev/docs/api/class-page#page-event-request-failed). A request
   * will only be considered failed when the client cannot get an HTTP response from the server, e.g. due to network
   * error net::ERR_FAILED.
   */
  addListener(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Emitted when a request finishes successfully after downloading the response body. For a successful response, the
   * sequence of events is `request`, `response` and `requestfinished`.
   */
  addListener(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Emitted when [response] status and headers are received for a request. For a successful response, the sequence of
   * events is `request`, `response` and `requestfinished`.
   */
  addListener(event: 'response', listener: (response: Response) => void): this;

  /**
   * Emitted when [WebSocket] request is sent.
   */
  addListener(event: 'websocket', listener: (webSocket: WebSocket) => void): this;

  /**
   * Emitted when a dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is spawned
   * by the page.
   */
  addListener(event: 'worker', listener: (worker: Worker) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'close', listener: (page: Page) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'console', listener: (consoleMessage: ConsoleMessage) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'crash', listener: (page: Page) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'dialog', listener: (dialog: Dialog) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'domcontentloaded', listener: (page: Page) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'download', listener: (download: Download) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'filechooser', listener: (fileChooser: FileChooser) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'frameattached', listener: (frame: Frame) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'framedetached', listener: (frame: Frame) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'framenavigated', listener: (frame: Frame) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'load', listener: (page: Page) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'pageerror', listener: (error: Error) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'popup', listener: (page: Page) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'request', listener: (request: Request) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'response', listener: (response: Response) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'websocket', listener: (webSocket: WebSocket) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'worker', listener: (worker: Worker) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'close', listener: (page: Page) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'console', listener: (consoleMessage: ConsoleMessage) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'crash', listener: (page: Page) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'dialog', listener: (dialog: Dialog) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'domcontentloaded', listener: (page: Page) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'download', listener: (download: Download) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'filechooser', listener: (fileChooser: FileChooser) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'frameattached', listener: (frame: Frame) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'framedetached', listener: (frame: Frame) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'framenavigated', listener: (frame: Frame) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'load', listener: (page: Page) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'pageerror', listener: (error: Error) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'popup', listener: (page: Page) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'request', listener: (request: Request) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'response', listener: (response: Response) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'websocket', listener: (webSocket: WebSocket) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'worker', listener: (worker: Worker) => void): this;

  /**
   * Emitted when the page closes.
   */
  prependListener(event: 'close', listener: (page: Page) => void): this;

  /**
   * Emitted when JavaScript within the page calls one of console API methods, e.g. `console.log` or `console.dir`. Also
   * emitted if the page throws an error or a warning.
   *
   * The arguments passed into `console.log` appear as arguments on the event handler.
   *
   * An example of handling `console` event:
   *
   * ```js
   * page.on('console', async msg => {
   *   const values = [];
   *   for (const arg of msg.args())
   *     values.push(await arg.jsonValue());
   *   console.log(...values);
   * });
   * await page.evaluate(() => console.log('hello', 5, {foo: 'bar'}));
   * ```
   *
   */
  prependListener(event: 'console', listener: (consoleMessage: ConsoleMessage) => void): this;

  /**
   * Emitted when the page crashes. Browser pages might crash if they try to allocate too much memory. When the page
   * crashes, ongoing and subsequent operations will throw.
   *
   * The most common way to deal with crashes is to catch an exception:
   *
   * ```js
   * try {
   *   // Crash might happen during a click.
   *   await page.click('button');
   *   // Or while waiting for an event.
   *   await page.waitForEvent('popup');
   * } catch (e) {
   *   // When the page crashes, exception message contains 'crash'.
   * }
   * ```
   *
   */
  prependListener(event: 'crash', listener: (page: Page) => void): this;

  /**
   * Emitted when a JavaScript dialog appears, such as `alert`, `prompt`, `confirm` or `beforeunload`. Listener **must**
   * either [dialog.accept([promptText])](https://playwright.dev/docs/api/class-dialog#dialog-accept) or
   * [dialog.dismiss()](https://playwright.dev/docs/api/class-dialog#dialog-dismiss) the dialog - otherwise the page
   * will [freeze](https://developer.mozilla.org/en-US/docs/Web/JavaScript/EventLoop#never_blocking) waiting for the
   * dialog, and actions like click will never finish.
   *
   * ```js
   * page.on('dialog', dialog => {
   *   dialog.accept();
   * });
   * ```
   *
   * **NOTE** When no [page.on('dialog')](https://playwright.dev/docs/api/class-page#page-event-dialog) listeners are
   * present, all dialogs are automatically dismissed.
   */
  prependListener(event: 'dialog', listener: (dialog: Dialog) => void): this;

  /**
   * Emitted when the JavaScript
   * [`DOMContentLoaded`](https://developer.mozilla.org/en-US/docs/Web/Events/DOMContentLoaded) event is dispatched.
   */
  prependListener(event: 'domcontentloaded', listener: (page: Page) => void): this;

  /**
   * Emitted when attachment download started. User can access basic file operations on downloaded content via the
   * passed [Download] instance.
   */
  prependListener(event: 'download', listener: (download: Download) => void): this;

  /**
   * Emitted when a file chooser is supposed to appear, such as after clicking the  `<input type=file>`. Playwright can
   * respond to it via setting the input files using
   * [fileChooser.setFiles(files[, options])](https://playwright.dev/docs/api/class-filechooser#file-chooser-set-files)
   * that can be uploaded after that.
   *
   * ```js
   * page.on('filechooser', async (fileChooser) => {
   *   await fileChooser.setFiles('/tmp/myfile.pdf');
   * });
   * ```
   *
   */
  prependListener(event: 'filechooser', listener: (fileChooser: FileChooser) => void): this;

  /**
   * Emitted when a frame is attached.
   */
  prependListener(event: 'frameattached', listener: (frame: Frame) => void): this;

  /**
   * Emitted when a frame is detached.
   */
  prependListener(event: 'framedetached', listener: (frame: Frame) => void): this;

  /**
   * Emitted when a frame is navigated to a new url.
   */
  prependListener(event: 'framenavigated', listener: (frame: Frame) => void): this;

  /**
   * Emitted when the JavaScript [`load`](https://developer.mozilla.org/en-US/docs/Web/Events/load) event is dispatched.
   */
  prependListener(event: 'load', listener: (page: Page) => void): this;

  /**
   * Emitted when an uncaught exception happens within the page.
   *
   * ```js
   * // Log all uncaught errors to the terminal
   * page.on('pageerror', exception => {
   *   console.log(`Uncaught exception: "${exception}"`);
   * });
   *
   * // Navigate to a page with an exception.
   * await page.goto('data:text/html,<script>throw new Error("Test")</script>');
   * ```
   *
   */
  prependListener(event: 'pageerror', listener: (error: Error) => void): this;

  /**
   * Emitted when the page opens a new tab or window. This event is emitted in addition to the
   * [browserContext.on('page')](https://playwright.dev/docs/api/class-browsercontext#browser-context-event-page), but
   * only for popups relevant to this page.
   *
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to
   * "http://example.com" is done and its response has started loading in the popup.
   *
   * ```js
   * // Start waiting for popup before clicking. Note no await.
   * const popupPromise = page.waitForEvent('popup');
   * await page.getByText('open the popup').click();
   * const popup = await popupPromise;
   * console.log(await popup.evaluate('location.href'));
   * ```
   *
   * **NOTE** Use
   * [page.waitForLoadState([state, options])](https://playwright.dev/docs/api/class-page#page-wait-for-load-state) to
   * wait until the page gets to a particular state (you should not need it in most cases).
   */
  prependListener(event: 'popup', listener: (page: Page) => void): this;

  /**
   * Emitted when a page issues a request. The [request] object is read-only. In order to intercept and mutate requests,
   * see [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route) or
   * [browserContext.route(url, handler[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-route).
   */
  prependListener(event: 'request', listener: (request: Request) => void): this;

  /**
   * Emitted when a request fails, for example by timing out.
   *
   * ```js
   * page.on('requestfailed', request => {
   *   console.log(request.url() + ' ' + request.failure().errorText);
   * });
   * ```
   *
   * **NOTE** HTTP Error responses, such as 404 or 503, are still successful responses from HTTP standpoint, so request
   * will complete with
   * [page.on('requestfinished')](https://playwright.dev/docs/api/class-page#page-event-request-finished) event and not
   * with [page.on('requestfailed')](https://playwright.dev/docs/api/class-page#page-event-request-failed). A request
   * will only be considered failed when the client cannot get an HTTP response from the server, e.g. due to network
   * error net::ERR_FAILED.
   */
  prependListener(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Emitted when a request finishes successfully after downloading the response body. For a successful response, the
   * sequence of events is `request`, `response` and `requestfinished`.
   */
  prependListener(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Emitted when [response] status and headers are received for a request. For a successful response, the sequence of
   * events is `request`, `response` and `requestfinished`.
   */
  prependListener(event: 'response', listener: (response: Response) => void): this;

  /**
   * Emitted when [WebSocket] request is sent.
   */
  prependListener(event: 'websocket', listener: (webSocket: WebSocket) => void): this;

  /**
   * Emitted when a dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is spawned
   * by the page.
   */
  prependListener(event: 'worker', listener: (worker: Worker) => void): this;

  /**
   * Adds a `<script>` tag into the page with the desired url or content. Returns the added tag when the script's onload
   * fires or when the script content was injected into frame.
   * @param options
   */
  addScriptTag(options?: {
    /**
     * Raw JavaScript content to be injected into frame.
     */
    content?: string;

    /**
     * Path to the JavaScript file to be injected into frame. If `path` is a relative path, then it is resolved relative
     * to the current working directory.
     */
    path?: string;

    /**
     * Script type. Use 'module' in order to load a Javascript ES6 module. See
     * [script](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/script) for more details.
     */
    type?: string;

    /**
     * URL of a script to be added.
     */
    url?: string;
  }): Promise<ElementHandle>;

  /**
   * Adds a `<link rel="stylesheet">` tag into the page with the desired url or a `<style type="text/css">` tag with the
   * content. Returns the added tag when the stylesheet's onload fires or when the CSS content was injected into frame.
   * @param options
   */
  addStyleTag(options?: {
    /**
     * Raw CSS content to be injected into frame.
     */
    content?: string;

    /**
     * Path to the CSS file to be injected into frame. If `path` is a relative path, then it is resolved relative to the
     * current working directory.
     */
    path?: string;

    /**
     * URL of the `<link>` tag.
     */
    url?: string;
  }): Promise<ElementHandle>;

  /**
   * Brings page to front (activates tab).
   */
  bringToFront(): Promise<void>;

  /**
   * **NOTE** Use locator-based [locator.check([options])](https://playwright.dev/docs/api/class-locator#locator-check) instead.
   * Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method checks an element matching `selector` by performing the following steps:
   * 1. Find an element matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Ensure that matched element is a checkbox or a radio input. If not, this method throws. If the element is
   *    already checked, this method returns immediately.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the matched element, unless `force` option is set. If
   *    the element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to click in the center of the
   *    element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now checked. If not, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  check(selector: string, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * **NOTE** Use locator-based [locator.click([options])](https://playwright.dev/docs/api/class-locator#locator-click) instead.
   * Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method clicks an element matching `selector` by performing the following steps:
   * 1. Find an element matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the matched element, unless `force` option is set. If
   *    the element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to click in the center of the
   *    element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  click(selector: string, options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * defaults to 1. See [UIEvent.detail].
     */
    clickCount?: number;

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores
     * current modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * If `runBeforeUnload` is `false`, does not run any unload handlers and waits for the page to be closed. If
   * `runBeforeUnload` is `true` the method will run unload handlers, but will **not** wait for the page to close.
   *
   * By default, `page.close()` **does not** run `beforeunload` handlers.
   *
   * **NOTE** if `runBeforeUnload` is passed as true, a `beforeunload` dialog might be summoned and should be handled
   * manually via [page.on('dialog')](https://playwright.dev/docs/api/class-page#page-event-dialog) event.
   * @param options
   */
  close(options?: {
    /**
     * Defaults to `false`. Whether to run the
     * [before unload](https://developer.mozilla.org/en-US/docs/Web/Events/beforeunload) page handlers.
     */
    runBeforeUnload?: boolean;
  }): Promise<void>;

  /**
   * Gets the full HTML contents of the page, including the doctype.
   */
  content(): Promise<string>;

  /**
   * Get the browser context that the page belongs to.
   */
  context(): BrowserContext;

  /**
   * **NOTE** Use locator-based [locator.dblclick([options])](https://playwright.dev/docs/api/class-locator#locator-dblclick)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method double clicks an element matching `selector` by performing the following steps:
   * 1. Find an element matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the matched element, unless `force` option is set. If
   *    the element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to double click in the center of the
   *    element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set. Note that if
   *    the first click of the `dblclick()` triggers a navigation event, this method will throw.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   *
   * **NOTE** `page.dblclick()` dispatches two `click` events and a single `dblclick` event.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  dblclick(selector: string, options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores
     * current modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * **NOTE** Use locator-based
   * [locator.dispatchEvent(type[, eventInit, options])](https://playwright.dev/docs/api/class-locator#locator-dispatch-event)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * The snippet below dispatches the `click` event on the element. Regardless of the visibility state of the element,
   * `click` is dispatched. This is equivalent to calling
   * [element.click()](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/click).
   *
   * **Usage**
   *
   * ```js
   * await page.dispatchEvent('button#submit', 'click');
   * ```
   *
   * Under the hood, it creates an instance of an event based on the given `type`, initializes it with `eventInit`
   * properties and dispatches it on the element. Events are `composed`, `cancelable` and bubble by default.
   *
   * Since `eventInit` is event-specific, please refer to the events documentation for the lists of initial properties:
   * - [DragEvent](https://developer.mozilla.org/en-US/docs/Web/API/DragEvent/DragEvent)
   * - [FocusEvent](https://developer.mozilla.org/en-US/docs/Web/API/FocusEvent/FocusEvent)
   * - [KeyboardEvent](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/KeyboardEvent)
   * - [MouseEvent](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/MouseEvent)
   * - [PointerEvent](https://developer.mozilla.org/en-US/docs/Web/API/PointerEvent/PointerEvent)
   * - [TouchEvent](https://developer.mozilla.org/en-US/docs/Web/API/TouchEvent/TouchEvent)
   * - [Event](https://developer.mozilla.org/en-US/docs/Web/API/Event/Event)
   *
   * You can also specify `JSHandle` as the property value if you want live objects to be passed into the event:
   *
   * ```js
   * // Note you can only create DataTransfer in Chromium and Firefox
   * const dataTransfer = await page.evaluateHandle(() => new DataTransfer());
   * await page.dispatchEvent('#source', 'dragstart', { dataTransfer });
   * ```
   *
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param type DOM event type: `"click"`, `"dragstart"`, etc.
   * @param eventInit Optional event-specific initialization properties.
   * @param options
   */
  dispatchEvent(selector: string, type: string, eventInit?: EvaluationArgument, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method drags the source element to the target element. It will first move to the source element, perform a
   * `mousedown`, then move to the target element and perform a `mouseup`.
   *
   * **Usage**
   *
   * ```js
   * await page.dragAndDrop('#source', '#target');
   * // or specify exact positions relative to the top-left corners of the elements:
   * await page.dragAndDrop('#source', '#target', {
   *   sourcePosition: { x: 34, y: 7 },
   *   targetPosition: { x: 10, y: 20 },
   * });
   * ```
   *
   * @param source A selector to search for an element to drag. If there are multiple elements satisfying the selector, the first will
   * be used.
   * @param target A selector to search for an element to drop onto. If there are multiple elements satisfying the selector, the first
   * will be used.
   * @param options
   */
  dragAndDrop(source: string, target: string, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Clicks on the source element at this point relative to the top-left corner of the element's padding box. If not
     * specified, some visible point of the element is used.
     */
    sourcePosition?: {
      x: number;

      y: number;
    };

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Drops on the target element at this point relative to the top-left corner of the element's padding box. If not
     * specified, some visible point of the element is used.
     */
    targetPosition?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * This method changes the `CSS media type` through the `media` argument, and/or the `'prefers-colors-scheme'` media
   * feature, using the `colorScheme` argument.
   *
   * **Usage**
   *
   * ```js
   * await page.evaluate(() => matchMedia('screen').matches);
   * // → true
   * await page.evaluate(() => matchMedia('print').matches);
   * // → false
   *
   * await page.emulateMedia({ media: 'print' });
   * await page.evaluate(() => matchMedia('screen').matches);
   * // → false
   * await page.evaluate(() => matchMedia('print').matches);
   * // → true
   *
   * await page.emulateMedia({});
   * await page.evaluate(() => matchMedia('screen').matches);
   * // → true
   * await page.evaluate(() => matchMedia('print').matches);
   * // → false
   * ```
   *
   * ```js
   * await page.emulateMedia({ colorScheme: 'dark' });
   * await page.evaluate(() => matchMedia('(prefers-color-scheme: dark)').matches);
   * // → true
   * await page.evaluate(() => matchMedia('(prefers-color-scheme: light)').matches);
   * // → false
   * await page.evaluate(() => matchMedia('(prefers-color-scheme: no-preference)').matches);
   * // → false
   * ```
   *
   * @param options
   */
  emulateMedia(options?: {
    /**
     * Emulates `'prefers-colors-scheme'` media feature, supported values are `'light'`, `'dark'`, `'no-preference'`.
     * Passing `null` disables color scheme emulation.
     */
    colorScheme?: null|"light"|"dark"|"no-preference";

    /**
     * Emulates `'forced-colors'` media feature, supported values are `'active'` and `'none'`. Passing `null` disables
     * forced colors emulation.
     */
    forcedColors?: null|"active"|"none";

    /**
     * Changes the CSS media type of the page. The only allowed values are `'screen'`, `'print'` and `null`. Passing
     * `null` disables CSS media emulation.
     */
    media?: null|"screen"|"print";

    /**
     * Emulates `'prefers-reduced-motion'` media feature, supported values are `'reduce'`, `'no-preference'`. Passing
     * `null` disables reduced motion emulation.
     */
    reducedMotion?: null|"reduce"|"no-preference";
  }): Promise<void>;

  /**
   * The method adds a function called `name` on the `window` object of every frame in the page. When called, the
   * function executes `callback` and returns a [Promise] which resolves to the return value of `callback`.
   *
   * If the `callback` returns a [Promise], it will be awaited.
   *
   * See
   * [browserContext.exposeFunction(name, callback)](https://playwright.dev/docs/api/class-browsercontext#browser-context-expose-function)
   * for context-wide exposed function.
   *
   * **NOTE** Functions installed via
   * [page.exposeFunction(name, callback)](https://playwright.dev/docs/api/class-page#page-expose-function) survive
   * navigations.
   *
   * **Usage**
   *
   * An example of adding a `sha256` function to the page:
   *
   * ```js
   * const { webkit } = require('playwright');  // Or 'chromium' or 'firefox'.
   * const crypto = require('crypto');
   *
   * (async () => {
   *   const browser = await webkit.launch({ headless: false });
   *   const page = await browser.newPage();
   *   await page.exposeFunction('sha256', text => crypto.createHash('sha256').update(text).digest('hex'));
   *   await page.setContent(`
   *     <script>
   *       async function onClick() {
   *         document.querySelector('div').textContent = await window.sha256('PLAYWRIGHT');
   *       }
   *     </script>
   *     <button onclick="onClick()">Click me</button>
   *     <div></div>
   *   `);
   *   await page.click('button');
   * })();
   * ```
   *
   * @param name Name of the function on the window object
   * @param callback Callback function which will be called in Playwright's context.
   */
  exposeFunction(name: string, callback: Function): Promise<void>;

  /**
   * **NOTE** Use locator-based [locator.fill(value[, options])](https://playwright.dev/docs/api/class-locator#locator-fill)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method waits for an element matching `selector`, waits for [actionability](https://playwright.dev/docs/actionability) checks,
   * focuses the element, fills it and triggers an `input` event after filling. Note that you can pass an empty string
   * to clear the input field.
   *
   * If the target element is not an `<input>`, `<textarea>` or `[contenteditable]` element, this method throws an
   * error. However, if the element is inside the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), the control will be filled
   * instead.
   *
   * To send fine-grained keyboard events, use
   * [page.type(selector, text[, options])](https://playwright.dev/docs/api/class-page#page-type).
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param value Value to fill for the `<input>`, `<textarea>` or `[contenteditable]` element.
   * @param options
   */
  fill(selector: string, value: string, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * **NOTE** Use locator-based [locator.focus([options])](https://playwright.dev/docs/api/class-locator#locator-focus) instead.
   * Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method fetches an element with `selector` and focuses it. If there's no element matching `selector`, the
   * method waits until a matching element appears in the DOM.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  focus(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns frame matching the specified criteria. Either `name` or `url` must be specified.
   *
   * **Usage**
   *
   * ```js
   * const frame = page.frame('frame-name');
   * ```
   *
   * ```js
   * const frame = page.frame({ url: /.*domain.*\/ });
   * ```
   *
   * @param frameSelector Frame name or other frame lookup options.
   */
  frame(frameSelector: string|{
    /**
     * Frame name specified in the `iframe`'s `name` attribute. Optional.
     */
    name?: string;

    /**
     * A glob pattern, regex pattern or predicate receiving frame's `url` as a [URL] object. Optional.
     */
    url?: string|RegExp|((url: URL) => boolean);
  }): null|Frame;

  /**
   * When working with iframes, you can create a frame locator that will enter the iframe and allow selecting elements
   * in that iframe.
   *
   * **Usage**
   *
   * Following snippet locates element with text "Submit" in the iframe with id `my-frame`, like `<iframe
   * id="my-frame">`:
   *
   * ```js
   * const locator = page.frameLocator('#my-iframe').getByText('Submit');
   * await locator.click();
   * ```
   *
   * @param selector A selector to use when resolving DOM element.
   */
  frameLocator(selector: string): FrameLocator;

  /**
   * An array of all frames attached to the page.
   */
  frames(): Array<Frame>;

  /**
   * **NOTE** Use locator-based
   * [locator.getAttribute(name[, options])](https://playwright.dev/docs/api/class-locator#locator-get-attribute)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns element attribute value.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param name Attribute name to get the value for.
   * @param options
   */
  getAttribute(selector: string, name: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<null|string>;

  /**
   * Allows locating elements by their alt text.
   *
   * **Usage**
   *
   * For example, this method will find the image by alt text "Playwright logo":
   *
   * ```html
   * <img alt='Playwright logo'>
   * ```
   *
   * ```js
   * await page.getByAltText('Playwright logo').click();
   * ```
   *
   * @param text Text to locate the element for.
   * @param options
   */
  getByAltText(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Allows locating input elements by the text of the associated `<label>` or `aria-labelledby` element, or by the
   * `aria-label` attribute.
   *
   * **Usage**
   *
   * For example, this method will find inputs by label "Username" and "Password" in the following DOM:
   *
   * ```html
   * <input aria-label="Username">
   * <label for="password-input">Password:</label>
   * <input id="password-input">
   * ```
   *
   * ```js
   * await page.getByLabel('Username').fill('john');
   * await page.getByLabel('Password').fill('secret');
   * ```
   *
   * @param text Text to locate the element for.
   * @param options
   */
  getByLabel(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Allows locating input elements by the placeholder text.
   *
   * **Usage**
   *
   * For example, consider the following DOM structure.
   *
   * ```html
   * <input type="email" placeholder="name@example.com" />
   * ```
   *
   * You can fill the input after locating it by the placeholder text:
   *
   * ```js
   * await page
   *     .getByPlaceholder("name@example.com")
   *     .fill("playwright@microsoft.com");
   * ```
   *
   * @param text Text to locate the element for.
   * @param options
   */
  getByPlaceholder(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Allows locating elements by their [ARIA role](https://www.w3.org/TR/wai-aria-1.2/#roles),
   * [ARIA attributes](https://www.w3.org/TR/wai-aria-1.2/#aria-attributes) and
   * [accessible name](https://w3c.github.io/accname/#dfn-accessible-name).
   *
   * **Usage**
   *
   * Consider the following DOM structure.
   *
   * ```html
   * <h3>Sign up</h3>
   * <label>
   *   <input type="checkbox" /> Subscribe
   * </label>
   * <br/>
   * <button>Submit</button>
   * ```
   *
   * You can locate each element by it's implicit role:
   *
   * ```js
   * await expect(page.getByRole('heading', { name: 'Sign up' })).toBeVisible();
   *
   * await page.getByRole('checkbox', { name: 'Subscribe' }).check();
   *
   * await page.getByRole('button', { name: /submit/i }).click();
   * ```
   *
   * **Details**
   *
   * Role selector **does not replace** accessibility audits and conformance tests, but rather gives early feedback
   * about the ARIA guidelines.
   *
   * Many html elements have an implicitly [defined role](https://w3c.github.io/html-aam/#html-element-role-mappings)
   * that is recognized by the role selector. You can find all the
   * [supported roles here](https://www.w3.org/TR/wai-aria-1.2/#role_definitions). ARIA guidelines **do not recommend**
   * duplicating implicit roles and attributes by setting `role` and/or `aria-*` attributes to default values.
   * @param role Required aria role.
   * @param options
   */
  getByRole(role: "alert"|"alertdialog"|"application"|"article"|"banner"|"blockquote"|"button"|"caption"|"cell"|"checkbox"|"code"|"columnheader"|"combobox"|"complementary"|"contentinfo"|"definition"|"deletion"|"dialog"|"directory"|"document"|"emphasis"|"feed"|"figure"|"form"|"generic"|"grid"|"gridcell"|"group"|"heading"|"img"|"insertion"|"link"|"list"|"listbox"|"listitem"|"log"|"main"|"marquee"|"math"|"meter"|"menu"|"menubar"|"menuitem"|"menuitemcheckbox"|"menuitemradio"|"navigation"|"none"|"note"|"option"|"paragraph"|"presentation"|"progressbar"|"radio"|"radiogroup"|"region"|"row"|"rowgroup"|"rowheader"|"scrollbar"|"search"|"searchbox"|"separator"|"slider"|"spinbutton"|"status"|"strong"|"subscript"|"superscript"|"switch"|"tab"|"table"|"tablist"|"tabpanel"|"term"|"textbox"|"time"|"timer"|"toolbar"|"tooltip"|"tree"|"treegrid"|"treeitem", options?: {
    /**
     * An attribute that is usually set by `aria-checked` or native `<input type=checkbox>` controls.
     *
     * Learn more about [`aria-checked`](https://www.w3.org/TR/wai-aria-1.2/#aria-checked).
     */
    checked?: boolean;

    /**
     * An attribute that is usually set by `aria-disabled` or `disabled`.
     *
     * **NOTE** Unlike most other attributes, `disabled` is inherited through the DOM hierarchy. Learn more about
     * [`aria-disabled`](https://www.w3.org/TR/wai-aria-1.2/#aria-disabled).
     */
    disabled?: boolean;

    /**
     * Whether `name` is matched exactly: case-sensitive and whole-string. Defaults to false. Ignored when `name` is a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;

    /**
     * An attribute that is usually set by `aria-expanded`.
     *
     * Learn more about [`aria-expanded`](https://www.w3.org/TR/wai-aria-1.2/#aria-expanded).
     */
    expanded?: boolean;

    /**
     * Option that controls whether hidden elements are matched. By default, only non-hidden elements, as
     * [defined by ARIA](https://www.w3.org/TR/wai-aria-1.2/#tree_exclusion), are matched by role selector.
     *
     * Learn more about [`aria-hidden`](https://www.w3.org/TR/wai-aria-1.2/#aria-hidden).
     */
    includeHidden?: boolean;

    /**
     * A number attribute that is usually present for roles `heading`, `listitem`, `row`, `treeitem`, with default values
     * for `<h1>-<h6>` elements.
     *
     * Learn more about [`aria-level`](https://www.w3.org/TR/wai-aria-1.2/#aria-level).
     */
    level?: number;

    /**
     * Option to match the [accessible name](https://w3c.github.io/accname/#dfn-accessible-name). By default, matching is
     * case-insensitive and searches for a substring, use `exact` to control this behavior.
     *
     * Learn more about [accessible name](https://w3c.github.io/accname/#dfn-accessible-name).
     */
    name?: string|RegExp;

    /**
     * An attribute that is usually set by `aria-pressed`.
     *
     * Learn more about [`aria-pressed`](https://www.w3.org/TR/wai-aria-1.2/#aria-pressed).
     */
    pressed?: boolean;

    /**
     * An attribute that is usually set by `aria-selected`.
     *
     * Learn more about [`aria-selected`](https://www.w3.org/TR/wai-aria-1.2/#aria-selected).
     */
    selected?: boolean;
  }): Locator;

  /**
   * Locate element by the test id.
   *
   * **Usage**
   *
   * Consider the following DOM structure.
   *
   * ```html
   * <button data-testid="directions">Itinéraire</button>
   * ```
   *
   * You can locate the element by it's test id:
   *
   * ```js
   * await page.getByTestId('directions').click();
   * ```
   *
   * **Details**
   *
   * By default, the `data-testid` attribute is used as a test id. Use
   * [selectors.setTestIdAttribute(attributeName)](https://playwright.dev/docs/api/class-selectors#selectors-set-test-id-attribute)
   * to configure a different test id attribute if necessary.
   *
   * ```js
   * // Set custom test id attribute from @playwright/test config:
   * use: {
   *   testIdAttribute: 'data-pw'
   * }
   * ```
   *
   * @param testId Id to locate the element by.
   */
  getByTestId(testId: string|RegExp): Locator;

  /**
   * Allows locating elements that contain given text.
   *
   * See also [locator.filter([options])](https://playwright.dev/docs/api/class-locator#locator-filter) that allows to
   * match by another criteria, like an accessible role, and then filter by the text content.
   *
   * **Usage**
   *
   * Consider the following DOM structure:
   *
   * ```html
   * <div>Hello <span>world</span></div>
   * <div>Hello</div>
   * ```
   *
   * You can locate by text substring, exact string, or a regular expression:
   *
   * ```js
   * // Matches <span>
   * page.getByText('world')
   *
   * // Matches first <div>
   * page.getByText('Hello world')
   *
   * // Matches second <div>
   * page.getByText('Hello', { exact: true })
   *
   * // Matches both <div>s
   * page.getByText(/Hello/)
   *
   * // Matches second <div>
   * page.getByText(/^hello$/i)
   * ```
   *
   * **Details**
   *
   * Matching by text always normalizes whitespace, even with exact match. For example, it turns multiple spaces into
   * one, turns line breaks into spaces and ignores leading and trailing whitespace.
   *
   * Input elements of the type `button` and `submit` are matched by their `value` instead of the text content. For
   * example, locating by text `"Log in"` matches `<input type=button value="Log in">`.
   * @param text Text to locate the element for.
   * @param options
   */
  getByText(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Allows locating elements by their title attribute.
   *
   * **Usage**
   *
   * Consider the following DOM structure.
   *
   * ```html
   * <span title='Issues count'>25 issues</span>
   * ```
   *
   * You can check the issues count after locating it by the title text:
   *
   * ```js
   * await expect(page.getByTitle('Issues count')).toHaveText('25 issues');
   * ```
   *
   * @param text Text to locate the element for.
   * @param options
   */
  getByTitle(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Returns the main resource response. In case of multiple redirects, the navigation will resolve with the response of
   * the last redirect. If can not go back, returns `null`.
   *
   * Navigate to the previous page in history.
   * @param options
   */
  goBack(options?: {
    /**
     * Maximum operation time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via
     * `navigationTimeout` option in the config, or by using the
     * [browserContext.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-navigation-timeout),
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout),
     * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500`
     *   ms.
     * - `'commit'` - consider operation to be finished when network response is received and the document started
     *   loading.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle"|"commit";
  }): Promise<null|Response>;

  /**
   * Returns the main resource response. In case of multiple redirects, the navigation will resolve with the response of
   * the last redirect. If can not go forward, returns `null`.
   *
   * Navigate to the next page in history.
   * @param options
   */
  goForward(options?: {
    /**
     * Maximum operation time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via
     * `navigationTimeout` option in the config, or by using the
     * [browserContext.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-navigation-timeout),
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout),
     * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500`
     *   ms.
     * - `'commit'` - consider operation to be finished when network response is received and the document started
     *   loading.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle"|"commit";
  }): Promise<null|Response>;

  /**
   * Returns the main resource response. In case of multiple redirects, the navigation will resolve with the first
   * non-redirect response.
   *
   * The method will throw an error if:
   * - there's an SSL error (e.g. in case of self-signed certificates).
   * - target URL is invalid.
   * - the `timeout` is exceeded during navigation.
   * - the remote server does not respond or is unreachable.
   * - the main resource failed to load.
   *
   * The method will not throw an error when any valid HTTP status code is returned by the remote server, including 404
   * "Not Found" and 500 "Internal Server Error".  The status code for such responses can be retrieved by calling
   * [response.status()](https://playwright.dev/docs/api/class-response#response-status).
   *
   * **NOTE** The method either throws an error or returns a main resource response. The only exceptions are navigation
   * to `about:blank` or navigation to the same URL with a different hash, which would succeed and return `null`.
   *
   * **NOTE** Headless mode doesn't support navigation to a PDF document. See the
   * [upstream issue](https://bugs.chromium.org/p/chromium/issues/detail?id=761295).
   * @param url URL to navigate page to. The url should include scheme, e.g. `https://`. When a `baseURL` via the context options
   * was provided and the passed URL is a path, it gets merged via the
   * [`new URL()`](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL) constructor.
   * @param options
   */
  goto(url: string, options?: {
    /**
     * Referer header value. If provided it will take preference over the referer header value set by
     * [page.setExtraHTTPHeaders(headers)](https://playwright.dev/docs/api/class-page#page-set-extra-http-headers).
     */
    referer?: string;

    /**
     * Maximum operation time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via
     * `navigationTimeout` option in the config, or by using the
     * [browserContext.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-navigation-timeout),
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout),
     * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500`
     *   ms.
     * - `'commit'` - consider operation to be finished when network response is received and the document started
     *   loading.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle"|"commit";
  }): Promise<null|Response>;

  /**
   * **NOTE** Use locator-based [locator.hover([options])](https://playwright.dev/docs/api/class-locator#locator-hover) instead.
   * Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method hovers over an element matching `selector` by performing the following steps:
   * 1. Find an element matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the matched element, unless `force` option is set. If
   *    the element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to hover over the center of the
   *    element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  hover(selector: string, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores
     * current modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * **NOTE** Use locator-based [locator.innerHTML([options])](https://playwright.dev/docs/api/class-locator#locator-inner-html)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns `element.innerHTML`.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  innerHTML(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<string>;

  /**
   * **NOTE** Use locator-based [locator.innerText([options])](https://playwright.dev/docs/api/class-locator#locator-inner-text)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns `element.innerText`.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  innerText(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<string>;

  /**
   * **NOTE** Use locator-based
   * [locator.inputValue([options])](https://playwright.dev/docs/api/class-locator#locator-input-value) instead. Read
   * more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns `input.value` for the selected `<input>` or `<textarea>` or `<select>` element.
   *
   * Throws for non-input elements. However, if the element is inside the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), returns the value of the
   * control.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  inputValue(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<string>;

  /**
   * **NOTE** Use locator-based [locator.isChecked([options])](https://playwright.dev/docs/api/class-locator#locator-is-checked)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns whether the element is checked. Throws if the element is not a checkbox or radio input.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  isChecked(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Indicates that the page has been closed.
   */
  isClosed(): boolean;

  /**
   * **NOTE** Use locator-based
   * [locator.isDisabled([options])](https://playwright.dev/docs/api/class-locator#locator-is-disabled) instead. Read
   * more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns whether the element is disabled, the opposite of [enabled](https://playwright.dev/docs/actionability#enabled).
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  isDisabled(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * **NOTE** Use locator-based
   * [locator.isEditable([options])](https://playwright.dev/docs/api/class-locator#locator-is-editable) instead. Read
   * more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns whether the element is [editable](https://playwright.dev/docs/actionability#editable).
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  isEditable(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * **NOTE** Use locator-based [locator.isEnabled([options])](https://playwright.dev/docs/api/class-locator#locator-is-enabled)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns whether the element is [enabled](https://playwright.dev/docs/actionability#enabled).
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  isEnabled(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * **NOTE** Use locator-based [locator.isHidden([options])](https://playwright.dev/docs/api/class-locator#locator-is-hidden)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns whether the element is hidden, the opposite of [visible](https://playwright.dev/docs/actionability#visible).  `selector` that
   * does not match any elements is considered hidden.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  isHidden(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * @deprecated This option is ignored.
     * [page.isHidden(selector[, options])](https://playwright.dev/docs/api/class-page#page-is-hidden) does not wait for
     * the element to become hidden and returns immediately.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * **NOTE** Use locator-based [locator.isVisible([options])](https://playwright.dev/docs/api/class-locator#locator-is-visible)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns whether the element is [visible](https://playwright.dev/docs/actionability#visible). `selector` that does not match any elements
   * is considered not visible.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  isVisible(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * @deprecated This option is ignored.
     * [page.isVisible(selector[, options])](https://playwright.dev/docs/api/class-page#page-is-visible) does not wait for
     * the element to become visible and returns immediately.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * The method returns an element locator that can be used to perform actions on this page / frame. Locator is resolved
   * to the element immediately before performing an action, so a series of actions on the same locator can in fact be
   * performed on different DOM elements. That would happen if the DOM structure between those actions has changed.
   *
   * [Learn more about locators](https://playwright.dev/docs/locators).
   * @param selector A selector to use when resolving DOM element.
   * @param options
   */
  locator(selector: string, options?: {
    /**
     * Matches elements containing an element that matches an inner locator. Inner locator is queried against the outer
     * one. For example, `article` that has `text=Playwright` matches `<article><div>Playwright</div></article>`.
     *
     * Note that outer and inner locators must belong to the same frame. Inner locator must not contain [FrameLocator]s.
     */
    has?: Locator;

    /**
     * Matches elements containing specified text somewhere inside, possibly in a child or a descendant element. When
     * passed a [string], matching is case-insensitive and searches for a substring. For example, `"Playwright"` matches
     * `<article><div>Playwright</div></article>`.
     */
    hasText?: string|RegExp;
  }): Locator;

  /**
   * The page's main frame. Page is guaranteed to have a main frame which persists during navigations.
   */
  mainFrame(): Frame;

  /**
   * Returns the opener for popup pages and `null` for others. If the opener has been closed already the returns `null`.
   */
  opener(): Promise<null|Page>;

  /**
   * Pauses script execution. Playwright will stop executing the script and wait for the user to either press 'Resume'
   * button in the page overlay or to call `playwright.resume()` in the DevTools console.
   *
   * User can inspect selectors or perform manual steps while paused. Resume will continue running the original script
   * from the place it was paused.
   *
   * **NOTE** This method requires Playwright to be started in a headed mode, with a falsy `headless` value in the
   * [browserType.launch([options])](https://playwright.dev/docs/api/class-browsertype#browser-type-launch).
   */
  pause(): Promise<void>;

  /**
   * Returns the PDF buffer.
   *
   * **NOTE** Generating a pdf is currently only supported in Chromium headless.
   *
   * `page.pdf()` generates a pdf of the page with `print` css media. To generate a pdf with `screen` media, call
   * [page.emulateMedia([options])](https://playwright.dev/docs/api/class-page#page-emulate-media) before calling
   * `page.pdf()`:
   *
   * **NOTE** By default, `page.pdf()` generates a pdf with modified colors for printing. Use the
   * [`-webkit-print-color-adjust`](https://developer.mozilla.org/en-US/docs/Web/CSS/-webkit-print-color-adjust)
   * property to force rendering of exact colors.
   *
   * **Usage**
   *
   * ```js
   * // Generates a PDF with 'screen' media type.
   * await page.emulateMedia({media: 'screen'});
   * await page.pdf({path: 'page.pdf'});
   * ```
   *
   * The `width`, `height`, and `margin` options accept values labeled with units. Unlabeled values are treated as
   * pixels.
   *
   * A few examples:
   * - `page.pdf({width: 100})` - prints with width set to 100 pixels
   * - `page.pdf({width: '100px'})` - prints with width set to 100 pixels
   * - `page.pdf({width: '10cm'})` - prints with width set to 10 centimeters.
   *
   * All possible units are:
   * - `px` - pixel
   * - `in` - inch
   * - `cm` - centimeter
   * - `mm` - millimeter
   *
   * The `format` options are:
   * - `Letter`: 8.5in x 11in
   * - `Legal`: 8.5in x 14in
   * - `Tabloid`: 11in x 17in
   * - `Ledger`: 17in x 11in
   * - `A0`: 33.1in x 46.8in
   * - `A1`: 23.4in x 33.1in
   * - `A2`: 16.54in x 23.4in
   * - `A3`: 11.7in x 16.54in
   * - `A4`: 8.27in x 11.7in
   * - `A5`: 5.83in x 8.27in
   * - `A6`: 4.13in x 5.83in
   *
   * **NOTE** `headerTemplate` and `footerTemplate` markup have the following limitations: > 1. Script tags inside
   * templates are not evaluated. > 2. Page styles are not visible inside templates.
   * @param options
   */
  pdf(options?: {
    /**
     * Display header and footer. Defaults to `false`.
     */
    displayHeaderFooter?: boolean;

    /**
     * HTML template for the print footer. Should use the same format as the `headerTemplate`.
     */
    footerTemplate?: string;

    /**
     * Paper format. If set, takes priority over `width` or `height` options. Defaults to 'Letter'.
     */
    format?: string;

    /**
     * HTML template for the print header. Should be valid HTML markup with following classes used to inject printing
     * values into them:
     * - `'date'` formatted print date
     * - `'title'` document title
     * - `'url'` document location
     * - `'pageNumber'` current page number
     * - `'totalPages'` total pages in the document
     */
    headerTemplate?: string;

    /**
     * Paper height, accepts values labeled with units.
     */
    height?: string|number;

    /**
     * Paper orientation. Defaults to `false`.
     */
    landscape?: boolean;

    /**
     * Paper margins, defaults to none.
     */
    margin?: {
      /**
       * Top margin, accepts values labeled with units. Defaults to `0`.
       */
      top?: string|number;

      /**
       * Right margin, accepts values labeled with units. Defaults to `0`.
       */
      right?: string|number;

      /**
       * Bottom margin, accepts values labeled with units. Defaults to `0`.
       */
      bottom?: string|number;

      /**
       * Left margin, accepts values labeled with units. Defaults to `0`.
       */
      left?: string|number;
    };

    /**
     * Paper ranges to print, e.g., '1-5, 8, 11-13'. Defaults to the empty string, which means print all pages.
     */
    pageRanges?: string;

    /**
     * The file path to save the PDF to. If `path` is a relative path, then it is resolved relative to the current working
     * directory. If no path is provided, the PDF won't be saved to the disk.
     */
    path?: string;

    /**
     * Give any CSS `@page` size declared in the page priority over what is declared in `width` and `height` or `format`
     * options. Defaults to `false`, which will scale the content to fit the paper size.
     */
    preferCSSPageSize?: boolean;

    /**
     * Print background graphics. Defaults to `false`.
     */
    printBackground?: boolean;

    /**
     * Scale of the webpage rendering. Defaults to `1`. Scale amount must be between 0.1 and 2.
     */
    scale?: number;

    /**
     * Paper width, accepts values labeled with units.
     */
    width?: string|number;
  }): Promise<Buffer>;

  /**
   * **NOTE** Use locator-based [locator.press(key[, options])](https://playwright.dev/docs/api/class-locator#locator-press)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Focuses the element, and then uses
   * [keyboard.down(key)](https://playwright.dev/docs/api/class-keyboard#keyboard-down) and
   * [keyboard.up(key)](https://playwright.dev/docs/api/class-keyboard#keyboard-up).
   *
   * `key` can specify the intended
   * [keyboardEvent.key](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key) value or a single character
   * to generate the text for. A superset of the `key` values can be found
   * [here](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key/Key_Values). Examples of the keys are:
   *
   * `F1` - `F12`, `Digit0`- `Digit9`, `KeyA`- `KeyZ`, `Backquote`, `Minus`, `Equal`, `Backslash`, `Backspace`, `Tab`,
   * `Delete`, `Escape`, `ArrowDown`, `End`, `Enter`, `Home`, `Insert`, `PageDown`, `PageUp`, `ArrowRight`, `ArrowUp`,
   * etc.
   *
   * Following modification shortcuts are also supported: `Shift`, `Control`, `Alt`, `Meta`, `ShiftLeft`.
   *
   * Holding down `Shift` will type the text that corresponds to the `key` in the upper case.
   *
   * If `key` is a single character, it is case-sensitive, so the values `a` and `A` will generate different respective
   * texts.
   *
   * Shortcuts such as `key: "Control+o"` or `key: "Control+Shift+T"` are supported as well. When specified with the
   * modifier, modifier is pressed and being held while the subsequent key is being pressed.
   *
   * **Usage**
   *
   * ```js
   * const page = await browser.newPage();
   * await page.goto('https://keycode.info');
   * await page.press('body', 'A');
   * await page.screenshot({ path: 'A.png' });
   * await page.press('body', 'ArrowLeft');
   * await page.screenshot({ path: 'ArrowLeft.png' });
   * await page.press('body', 'Shift+O');
   * await page.screenshot({ path: 'O.png' });
   * await browser.close();
   * ```
   *
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param key Name of the key to press or a character to generate, such as `ArrowLeft` or `a`.
   * @param options
   */
  press(selector: string, key: string, options?: {
    /**
     * Time to wait between `keydown` and `keyup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method reloads the current page, in the same way as if the user had triggered a browser refresh. Returns the
   * main resource response. In case of multiple redirects, the navigation will resolve with the response of the last
   * redirect.
   * @param options
   */
  reload(options?: {
    /**
     * Maximum operation time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via
     * `navigationTimeout` option in the config, or by using the
     * [browserContext.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-navigation-timeout),
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout),
     * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500`
     *   ms.
     * - `'commit'` - consider operation to be finished when network response is received and the document started
     *   loading.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle"|"commit";
  }): Promise<null|Response>;

  /**
   * Routing provides the capability to modify network requests that are made by a page.
   *
   * Once routing is enabled, every request matching the url pattern will stall unless it's continued, fulfilled or
   * aborted.
   *
   * **NOTE** The handler will only be called for the first url if the response is a redirect.
   *
   * **NOTE** [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route) will not
   * intercept requests intercepted by Service Worker. See [this](https://github.com/microsoft/playwright/issues/1090)
   * issue. We recommend disabling Service Workers when using request interception by setting
   * `Browser.newContext.serviceWorkers` to `'block'`.
   *
   * **Usage**
   *
   * An example of a naive handler that aborts all image requests:
   *
   * ```js
   * const page = await browser.newPage();
   * await page.route('**\/*.{png,jpg,jpeg}', route => route.abort());
   * await page.goto('https://example.com');
   * await browser.close();
   * ```
   *
   * or the same snippet using a regex pattern instead:
   *
   * ```js
   * const page = await browser.newPage();
   * await page.route(/(\.png$)|(\.jpg$)/, route => route.abort());
   * await page.goto('https://example.com');
   * await browser.close();
   * ```
   *
   * It is possible to examine the request to decide the route action. For example, mocking all requests that contain
   * some post data, and leaving all other requests as is:
   *
   * ```js
   * await page.route('/api/**', route => {
   *   if (route.request().postData().includes('my-string'))
   *     route.fulfill({ body: 'mocked-data' });
   *   else
   *     route.continue();
   * });
   * ```
   *
   * Page routes take precedence over browser context routes (set up with
   * [browserContext.route(url, handler[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-route))
   * when request matches both handlers.
   *
   * To remove a route with its handler you can use
   * [page.unroute(url[, handler])](https://playwright.dev/docs/api/class-page#page-unroute).
   *
   * **NOTE** Enabling routing disables http cache.
   * @param url A glob pattern, regex pattern or predicate receiving [URL] to match while routing. When a `baseURL` via the context
   * options was provided and the passed URL is a path, it gets merged via the
   * [`new URL()`](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL) constructor.
   * @param handler handler function to route the request.
   * @param options
   */
  route(url: string|RegExp|((url: URL) => boolean), handler: ((route: Route, request: Request) => Promise<any>|any), options?: {
    /**
     * How often a route should be used. By default it will be used every time.
     */
    times?: number;
  }): Promise<void>;

  /**
   * If specified the network requests that are made in the page will be served from the HAR file. Read more about
   * [Replaying from HAR](https://playwright.dev/docs/network#replaying-from-har).
   *
   * Playwright will not serve requests intercepted by Service Worker from the HAR file. See
   * [this](https://github.com/microsoft/playwright/issues/1090) issue. We recommend disabling Service Workers when
   * using request interception by setting `Browser.newContext.serviceWorkers` to `'block'`.
   * @param har Path to a [HAR](http://www.softwareishard.com/blog/har-12-spec) file with prerecorded network data. If `path` is a
   * relative path, then it is resolved relative to the current working directory.
   * @param options
   */
  routeFromHAR(har: string, options?: {
    /**
     * - If set to 'abort' any request not found in the HAR file will be aborted.
     * - If set to 'fallback' missing requests will be sent to the network.
     *
     * Defaults to abort.
     */
    notFound?: "abort"|"fallback";

    /**
     * If specified, updates the given HAR with the actual network information instead of serving from file. The file is
     * written to disk when
     * [browserContext.close()](https://playwright.dev/docs/api/class-browsercontext#browser-context-close) is called.
     */
    update?: boolean;

    /**
     * Optional setting to control resource content management. If `attach` is specified, resources are persisted as
     * separate files or entries in the ZIP archive. If `embed` is specified, content is stored inline the HAR file.
     */
    updateContent?: "embed"|"attach";

    /**
     * When set to `minimal`, only record information necessary for routing from HAR. This omits sizes, timing, page,
     * cookies, security and other types of HAR information that are not used when replaying from HAR. Defaults to `full`.
     */
    updateMode?: "full"|"minimal";

    /**
     * A glob pattern, regular expression or predicate to match the request URL. Only requests with URL matching the
     * pattern will be served from the HAR file. If not specified, all requests are served from the HAR file.
     */
    url?: string|RegExp;
  }): Promise<void>;

  /**
   * Returns the buffer with the captured screenshot.
   * @param options
   */
  screenshot(options?: PageScreenshotOptions): Promise<Buffer>;

  /**
   * **NOTE** Use locator-based
   * [locator.selectOption(values[, options])](https://playwright.dev/docs/api/class-locator#locator-select-option)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method waits for an element matching `selector`, waits for [actionability](https://playwright.dev/docs/actionability) checks, waits
   * until all specified options are present in the `<select>` element and selects these options.
   *
   * If the target element is not a `<select>` element, this method throws an error. However, if the element is inside
   * the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), the control will be used
   * instead.
   *
   * Returns the array of option values that have been successfully selected.
   *
   * Triggers a `change` and `input` event once all the provided options have been selected.
   *
   * **Usage**
   *
   * ```js
   * // single selection matching the value
   * page.selectOption('select#colors', 'blue');
   *
   * // single selection matching the label
   * page.selectOption('select#colors', { label: 'Blue' });
   *
   * // multiple selection
   * page.selectOption('select#colors', ['red', 'green', 'blue']);
   *
   * ```
   *
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param values Options to select. If the `<select>` has the `multiple` attribute, all matching options are selected, otherwise
   * only the first option matching one of the passed options is selected. String values are matching both values and
   * labels. Option is considered matching if all specified properties match.
   * @param options
   */
  selectOption(selector: string, values: null|string|ElementHandle|Array<string>|{
    /**
     * Matches by `option.value`. Optional.
     */
    value?: string;

    /**
     * Matches by `option.label`. Optional.
     */
    label?: string;

    /**
     * Matches by the index. Optional.
     */
    index?: number;
  }|Array<ElementHandle>|Array<{
    /**
     * Matches by `option.value`. Optional.
     */
    value?: string;

    /**
     * Matches by `option.label`. Optional.
     */
    label?: string;

    /**
     * Matches by the index. Optional.
     */
    index?: number;
  }>, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<Array<string>>;

  /**
   * **NOTE** Use locator-based
   * [locator.setChecked(checked[, options])](https://playwright.dev/docs/api/class-locator#locator-set-checked)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method checks or unchecks an element matching `selector` by performing the following steps:
   * 1. Find an element matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Ensure that matched element is a checkbox or a radio input. If not, this method throws.
   * 1. If the element already has the right checked state, this method returns immediately.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the matched element, unless `force` option is set. If
   *    the element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to click in the center of the
   *    element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now checked or unchecked. If not, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param checked Whether to check or uncheck the checkbox.
   * @param options
   */
  setChecked(selector: string, checked: boolean, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * @param html HTML markup to assign to the page.
   * @param options
   */
  setContent(html: string, options?: {
    /**
     * Maximum operation time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via
     * `navigationTimeout` option in the config, or by using the
     * [browserContext.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-navigation-timeout),
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout),
     * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500`
     *   ms.
     * - `'commit'` - consider operation to be finished when network response is received and the document started
     *   loading.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle"|"commit";
  }): Promise<void>;

  /**
   * This setting will change the default maximum navigation time for the following methods and related shortcuts:
   * - [page.goBack([options])](https://playwright.dev/docs/api/class-page#page-go-back)
   * - [page.goForward([options])](https://playwright.dev/docs/api/class-page#page-go-forward)
   * - [page.goto(url[, options])](https://playwright.dev/docs/api/class-page#page-goto)
   * - [page.reload([options])](https://playwright.dev/docs/api/class-page#page-reload)
   * - [page.setContent(html[, options])](https://playwright.dev/docs/api/class-page#page-set-content)
   * - [page.waitForNavigation([options])](https://playwright.dev/docs/api/class-page#page-wait-for-navigation)
   * - [page.waitForURL(url[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-url)
   *
   * **NOTE**
   * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout)
   * takes priority over
   * [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout),
   * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
   * and
   * [browserContext.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-navigation-timeout).
   * @param timeout Maximum navigation time in milliseconds
   */
  setDefaultNavigationTimeout(timeout: number): void;

  /**
   * This setting will change the default maximum time for all the methods accepting `timeout` option.
   *
   * **NOTE**
   * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout)
   * takes priority over
   * [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout).
   * @param timeout Maximum time in milliseconds
   */
  setDefaultTimeout(timeout: number): void;

  /**
   * The extra HTTP headers will be sent with every request the page initiates.
   *
   * **NOTE**
   * [page.setExtraHTTPHeaders(headers)](https://playwright.dev/docs/api/class-page#page-set-extra-http-headers) does
   * not guarantee the order of headers in the outgoing requests.
   * @param headers An object containing additional HTTP headers to be sent with every request. All header values must be strings.
   */
  setExtraHTTPHeaders(headers: { [key: string]: string; }): Promise<void>;

  /**
   * **NOTE** Use locator-based
   * [locator.setInputFiles(files[, options])](https://playwright.dev/docs/api/class-locator#locator-set-input-files)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Sets the value of the file input to these file paths or files. If some of the `filePaths` are relative paths, then
   * they are resolved relative to the current working directory. For empty array, clears the selected files.
   *
   * This method expects `selector` to point to an
   * [input element](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input). However, if the element is inside
   * the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), targets the control instead.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param files
   * @param options
   */
  setInputFiles(selector: string, files: string|Array<string>|{
    /**
     * File name
     */
    name: string;

    /**
     * File type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }|Array<{
    /**
     * File name
     */
    name: string;

    /**
     * File type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }>, options?: {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * In the case of multiple pages in a single browser, each page can have its own viewport size. However,
   * [browser.newContext([options])](https://playwright.dev/docs/api/class-browser#browser-new-context) allows to set
   * viewport size (and more) for all pages in the context at once.
   *
   * [page.setViewportSize(viewportSize)](https://playwright.dev/docs/api/class-page#page-set-viewport-size) will resize
   * the page. A lot of websites don't expect phones to change size, so you should set the viewport size before
   * navigating to the page.
   * [page.setViewportSize(viewportSize)](https://playwright.dev/docs/api/class-page#page-set-viewport-size) will also
   * reset `screen` size, use
   * [browser.newContext([options])](https://playwright.dev/docs/api/class-browser#browser-new-context) with `screen`
   * and `viewport` parameters if you need better control of these properties.
   *
   * **Usage**
   *
   * ```js
   * const page = await browser.newPage();
   * await page.setViewportSize({
   *   width: 640,
   *   height: 480,
   * });
   * await page.goto('https://example.com');
   * ```
   *
   * @param viewportSize
   */
  setViewportSize(viewportSize: {
    /**
     * page width in pixels.
     */
    width: number;

    /**
     * page height in pixels.
     */
    height: number;
  }): Promise<void>;

  /**
   * **NOTE** Use locator-based [locator.tap([options])](https://playwright.dev/docs/api/class-locator#locator-tap) instead. Read
   * more about [locators](https://playwright.dev/docs/locators).
   *
   * This method taps an element matching `selector` by performing the following steps:
   * 1. Find an element matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the matched element, unless `force` option is set. If
   *    the element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.touchscreen](https://playwright.dev/docs/api/class-page#page-touchscreen) to tap the center of the
   *    element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   *
   * **NOTE** [page.tap(selector[, options])](https://playwright.dev/docs/api/class-page#page-tap) the method will throw
   * if `hasTouch` option of the browser context is false.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  tap(selector: string, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores
     * current modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * **NOTE** Use locator-based
   * [locator.textContent([options])](https://playwright.dev/docs/api/class-locator#locator-text-content) instead. Read
   * more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns `element.textContent`.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  textContent(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<null|string>;

  /**
   * Returns the page's title.
   */
  title(): Promise<string>;

  /**
   * **NOTE** Use locator-based [locator.type(text[, options])](https://playwright.dev/docs/api/class-locator#locator-type)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Sends a `keydown`, `keypress`/`input`, and `keyup` event for each character in the text. `page.type` can be used to
   * send fine-grained keyboard events. To fill values in form fields, use
   * [page.fill(selector, value[, options])](https://playwright.dev/docs/api/class-page#page-fill).
   *
   * To press a special key, like `Control` or `ArrowDown`, use
   * [keyboard.press(key[, options])](https://playwright.dev/docs/api/class-keyboard#keyboard-press).
   *
   * **Usage**
   *
   * ```js
   * await page.type('#mytextarea', 'Hello'); // Types instantly
   * await page.type('#mytextarea', 'World', {delay: 100}); // Types slower, like a user
   * ```
   *
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param text A text to type into a focused element.
   * @param options
   */
  type(selector: string, text: string, options?: {
    /**
     * Time to wait between key presses in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * **NOTE** Use locator-based [locator.uncheck([options])](https://playwright.dev/docs/api/class-locator#locator-uncheck)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method unchecks an element matching `selector` by performing the following steps:
   * 1. Find an element matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Ensure that matched element is a checkbox or a radio input. If not, this method throws. If the element is
   *    already unchecked, this method returns immediately.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the matched element, unless `force` option is set. If
   *    the element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to click in the center of the
   *    element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now unchecked. If not, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  uncheck(selector: string, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * Removes a route created with
   * [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route). When `handler` is not
   * specified, removes all routes for the `url`.
   * @param url A glob pattern, regex pattern or predicate receiving [URL] to match while routing.
   * @param handler Optional handler function to route the request.
   */
  unroute(url: string|RegExp|((url: URL) => boolean), handler?: ((route: Route, request: Request) => Promise<any>|any)): Promise<void>;

  url(): string;

  /**
   * Video object associated with this page.
   */
  video(): null|Video;

  viewportSize(): null|{
    /**
     * page width in pixels.
     */
    width: number;

    /**
     * page height in pixels.
     */
    height: number;
  };

  /**
   * Emitted when the page closes.
   */
  waitForEvent(event: 'close', optionsOrPredicate?: { predicate?: (page: Page) => boolean | Promise<boolean>, timeout?: number } | ((page: Page) => boolean | Promise<boolean>)): Promise<Page>;

  /**
   * Emitted when JavaScript within the page calls one of console API methods, e.g. `console.log` or `console.dir`. Also
   * emitted if the page throws an error or a warning.
   *
   * The arguments passed into `console.log` appear as arguments on the event handler.
   *
   * An example of handling `console` event:
   *
   * ```js
   * page.on('console', async msg => {
   *   const values = [];
   *   for (const arg of msg.args())
   *     values.push(await arg.jsonValue());
   *   console.log(...values);
   * });
   * await page.evaluate(() => console.log('hello', 5, {foo: 'bar'}));
   * ```
   *
   */
  waitForEvent(event: 'console', optionsOrPredicate?: { predicate?: (consoleMessage: ConsoleMessage) => boolean | Promise<boolean>, timeout?: number } | ((consoleMessage: ConsoleMessage) => boolean | Promise<boolean>)): Promise<ConsoleMessage>;

  /**
   * Emitted when the page crashes. Browser pages might crash if they try to allocate too much memory. When the page
   * crashes, ongoing and subsequent operations will throw.
   *
   * The most common way to deal with crashes is to catch an exception:
   *
   * ```js
   * try {
   *   // Crash might happen during a click.
   *   await page.click('button');
   *   // Or while waiting for an event.
   *   await page.waitForEvent('popup');
   * } catch (e) {
   *   // When the page crashes, exception message contains 'crash'.
   * }
   * ```
   *
   */
  waitForEvent(event: 'crash', optionsOrPredicate?: { predicate?: (page: Page) => boolean | Promise<boolean>, timeout?: number } | ((page: Page) => boolean | Promise<boolean>)): Promise<Page>;

  /**
   * Emitted when a JavaScript dialog appears, such as `alert`, `prompt`, `confirm` or `beforeunload`. Listener **must**
   * either [dialog.accept([promptText])](https://playwright.dev/docs/api/class-dialog#dialog-accept) or
   * [dialog.dismiss()](https://playwright.dev/docs/api/class-dialog#dialog-dismiss) the dialog - otherwise the page
   * will [freeze](https://developer.mozilla.org/en-US/docs/Web/JavaScript/EventLoop#never_blocking) waiting for the
   * dialog, and actions like click will never finish.
   *
   * ```js
   * page.on('dialog', dialog => {
   *   dialog.accept();
   * });
   * ```
   *
   * **NOTE** When no [page.on('dialog')](https://playwright.dev/docs/api/class-page#page-event-dialog) listeners are
   * present, all dialogs are automatically dismissed.
   */
  waitForEvent(event: 'dialog', optionsOrPredicate?: { predicate?: (dialog: Dialog) => boolean | Promise<boolean>, timeout?: number } | ((dialog: Dialog) => boolean | Promise<boolean>)): Promise<Dialog>;

  /**
   * Emitted when the JavaScript
   * [`DOMContentLoaded`](https://developer.mozilla.org/en-US/docs/Web/Events/DOMContentLoaded) event is dispatched.
   */
  waitForEvent(event: 'domcontentloaded', optionsOrPredicate?: { predicate?: (page: Page) => boolean | Promise<boolean>, timeout?: number } | ((page: Page) => boolean | Promise<boolean>)): Promise<Page>;

  /**
   * Emitted when attachment download started. User can access basic file operations on downloaded content via the
   * passed [Download] instance.
   */
  waitForEvent(event: 'download', optionsOrPredicate?: { predicate?: (download: Download) => boolean | Promise<boolean>, timeout?: number } | ((download: Download) => boolean | Promise<boolean>)): Promise<Download>;

  /**
   * Emitted when a file chooser is supposed to appear, such as after clicking the  `<input type=file>`. Playwright can
   * respond to it via setting the input files using
   * [fileChooser.setFiles(files[, options])](https://playwright.dev/docs/api/class-filechooser#file-chooser-set-files)
   * that can be uploaded after that.
   *
   * ```js
   * page.on('filechooser', async (fileChooser) => {
   *   await fileChooser.setFiles('/tmp/myfile.pdf');
   * });
   * ```
   *
   */
  waitForEvent(event: 'filechooser', optionsOrPredicate?: { predicate?: (fileChooser: FileChooser) => boolean | Promise<boolean>, timeout?: number } | ((fileChooser: FileChooser) => boolean | Promise<boolean>)): Promise<FileChooser>;

  /**
   * Emitted when a frame is attached.
   */
  waitForEvent(event: 'frameattached', optionsOrPredicate?: { predicate?: (frame: Frame) => boolean | Promise<boolean>, timeout?: number } | ((frame: Frame) => boolean | Promise<boolean>)): Promise<Frame>;

  /**
   * Emitted when a frame is detached.
   */
  waitForEvent(event: 'framedetached', optionsOrPredicate?: { predicate?: (frame: Frame) => boolean | Promise<boolean>, timeout?: number } | ((frame: Frame) => boolean | Promise<boolean>)): Promise<Frame>;

  /**
   * Emitted when a frame is navigated to a new url.
   */
  waitForEvent(event: 'framenavigated', optionsOrPredicate?: { predicate?: (frame: Frame) => boolean | Promise<boolean>, timeout?: number } | ((frame: Frame) => boolean | Promise<boolean>)): Promise<Frame>;

  /**
   * Emitted when the JavaScript [`load`](https://developer.mozilla.org/en-US/docs/Web/Events/load) event is dispatched.
   */
  waitForEvent(event: 'load', optionsOrPredicate?: { predicate?: (page: Page) => boolean | Promise<boolean>, timeout?: number } | ((page: Page) => boolean | Promise<boolean>)): Promise<Page>;

  /**
   * Emitted when an uncaught exception happens within the page.
   *
   * ```js
   * // Log all uncaught errors to the terminal
   * page.on('pageerror', exception => {
   *   console.log(`Uncaught exception: "${exception}"`);
   * });
   *
   * // Navigate to a page with an exception.
   * await page.goto('data:text/html,<script>throw new Error("Test")</script>');
   * ```
   *
   */
  waitForEvent(event: 'pageerror', optionsOrPredicate?: { predicate?: (error: Error) => boolean | Promise<boolean>, timeout?: number } | ((error: Error) => boolean | Promise<boolean>)): Promise<Error>;

  /**
   * Emitted when the page opens a new tab or window. This event is emitted in addition to the
   * [browserContext.on('page')](https://playwright.dev/docs/api/class-browsercontext#browser-context-event-page), but
   * only for popups relevant to this page.
   *
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to
   * "http://example.com" is done and its response has started loading in the popup.
   *
   * ```js
   * // Start waiting for popup before clicking. Note no await.
   * const popupPromise = page.waitForEvent('popup');
   * await page.getByText('open the popup').click();
   * const popup = await popupPromise;
   * console.log(await popup.evaluate('location.href'));
   * ```
   *
   * **NOTE** Use
   * [page.waitForLoadState([state, options])](https://playwright.dev/docs/api/class-page#page-wait-for-load-state) to
   * wait until the page gets to a particular state (you should not need it in most cases).
   */
  waitForEvent(event: 'popup', optionsOrPredicate?: { predicate?: (page: Page) => boolean | Promise<boolean>, timeout?: number } | ((page: Page) => boolean | Promise<boolean>)): Promise<Page>;

  /**
   * Emitted when a page issues a request. The [request] object is read-only. In order to intercept and mutate requests,
   * see [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route) or
   * [browserContext.route(url, handler[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-route).
   */
  waitForEvent(event: 'request', optionsOrPredicate?: { predicate?: (request: Request) => boolean | Promise<boolean>, timeout?: number } | ((request: Request) => boolean | Promise<boolean>)): Promise<Request>;

  /**
   * Emitted when a request fails, for example by timing out.
   *
   * ```js
   * page.on('requestfailed', request => {
   *   console.log(request.url() + ' ' + request.failure().errorText);
   * });
   * ```
   *
   * **NOTE** HTTP Error responses, such as 404 or 503, are still successful responses from HTTP standpoint, so request
   * will complete with
   * [page.on('requestfinished')](https://playwright.dev/docs/api/class-page#page-event-request-finished) event and not
   * with [page.on('requestfailed')](https://playwright.dev/docs/api/class-page#page-event-request-failed). A request
   * will only be considered failed when the client cannot get an HTTP response from the server, e.g. due to network
   * error net::ERR_FAILED.
   */
  waitForEvent(event: 'requestfailed', optionsOrPredicate?: { predicate?: (request: Request) => boolean | Promise<boolean>, timeout?: number } | ((request: Request) => boolean | Promise<boolean>)): Promise<Request>;

  /**
   * Emitted when a request finishes successfully after downloading the response body. For a successful response, the
   * sequence of events is `request`, `response` and `requestfinished`.
   */
  waitForEvent(event: 'requestfinished', optionsOrPredicate?: { predicate?: (request: Request) => boolean | Promise<boolean>, timeout?: number } | ((request: Request) => boolean | Promise<boolean>)): Promise<Request>;

  /**
   * Emitted when [response] status and headers are received for a request. For a successful response, the sequence of
   * events is `request`, `response` and `requestfinished`.
   */
  waitForEvent(event: 'response', optionsOrPredicate?: { predicate?: (response: Response) => boolean | Promise<boolean>, timeout?: number } | ((response: Response) => boolean | Promise<boolean>)): Promise<Response>;

  /**
   * Emitted when [WebSocket] request is sent.
   */
  waitForEvent(event: 'websocket', optionsOrPredicate?: { predicate?: (webSocket: WebSocket) => boolean | Promise<boolean>, timeout?: number } | ((webSocket: WebSocket) => boolean | Promise<boolean>)): Promise<WebSocket>;

  /**
   * Emitted when a dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is spawned
   * by the page.
   */
  waitForEvent(event: 'worker', optionsOrPredicate?: { predicate?: (worker: Worker) => boolean | Promise<boolean>, timeout?: number } | ((worker: Worker) => boolean | Promise<boolean>)): Promise<Worker>;


  /**
   * Returns when the required load state has been reached.
   *
   * This resolves when the page reaches a required load state, `load` by default. The navigation must have been
   * committed when this method is called. If current document has already reached the required state, resolves
   * immediately.
   *
   * **Usage**
   *
   * ```js
   * await page.getByRole('button').click(); // Click triggers navigation.
   * await page.waitForLoadState(); // The promise resolves after 'load' event.
   * ```
   *
   * ```js
   * const popupPromise = page.waitForEvent('popup');
   * await page.getByRole('button').click(); // Click triggers a popup.
   * const popup = await popupPromise;
   * await popup.waitForLoadState('domcontentloaded'); // Wait for the 'DOMContentLoaded' event.
   * console.log(await popup.title()); // Popup is ready to use.
   * ```
   *
   * @param state Optional load state to wait for, defaults to `load`. If the state has been already reached while loading current
   * document, the method resolves immediately. Can be one of:
   * - `'load'` - wait for the `load` event to be fired.
   * - `'domcontentloaded'` - wait for the `DOMContentLoaded` event to be fired.
   * - `'networkidle'` - wait until there are no network connections for at least `500` ms.
   * @param options
   */
  waitForLoadState(state?: "load"|"domcontentloaded"|"networkidle", options?: {
    /**
     * Maximum operation time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via
     * `navigationTimeout` option in the config, or by using the
     * [browserContext.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-navigation-timeout),
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout),
     * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Waits for the main frame navigation and returns the main resource response. In case of multiple redirects, the
   * navigation will resolve with the response of the last redirect. In case of navigation to a different anchor or
   * navigation due to History API usage, the navigation will resolve with `null`.
   *
   * **Usage**
   *
   * This resolves when the page navigates to a new URL or reloads. It is useful for when you run code which will
   * indirectly cause the page to navigate. e.g. The click target has an `onclick` handler that triggers navigation from
   * a `setTimeout`. Consider this example:
   *
   * ```js
   * // Start waiting for navigation before clicking. Note no await.
   * const navigationPromise = page.waitForNavigation();
   * await page.getByText('Navigate after timeout').click();
   * await navigationPromise;
   * ```
   *
   * **NOTE** Usage of the [History API](https://developer.mozilla.org/en-US/docs/Web/API/History_API) to change the URL
   * is considered a navigation.
   * @deprecated This method is inherently racy, please use
   * [page.waitForURL(url[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-url) instead.
   * @param options
   */
  waitForNavigation(options?: {
    /**
     * Maximum operation time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via
     * `navigationTimeout` option in the config, or by using the
     * [browserContext.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-navigation-timeout),
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout),
     * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * A glob pattern, regex pattern or predicate receiving [URL] to match while waiting for the navigation. Note that if
     * the parameter is a string without wildcard characters, the method will wait for navigation to URL that is exactly
     * equal to the string.
     */
    url?: string|RegExp|((url: URL) => boolean);

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500`
     *   ms.
     * - `'commit'` - consider operation to be finished when network response is received and the document started
     *   loading.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle"|"commit";
  }): Promise<null|Response>;

  /**
   * Waits for the matching request and returns it. See [waiting for event](https://playwright.dev/docs/events#waiting-for-event) for more
   * details about events.
   *
   * **Usage**
   *
   * ```js
   * // Start waiting for request before clicking. Note no await.
   * const requestPromise = page.waitForRequest('https://example.com/resource');
   * await page.getByText('trigger request').click();
   * const request = await requestPromise;
   *
   * // Alternative way with a predicate. Note no await.
   * const requestPromise = page.waitForRequest(request => request.url() === 'https://example.com' && request.method() === 'GET');
   * await page.getByText('trigger request').click();
   * const request = await requestPromise;
   * ```
   *
   * @param urlOrPredicate Request URL string, regex or predicate receiving [Request] object.
   * @param options
   */
  waitForRequest(urlOrPredicate: string|RegExp|((request: Request) => boolean|Promise<boolean>), options?: {
    /**
     * Maximum wait time in milliseconds, defaults to 30 seconds, pass `0` to disable the timeout. The default value can
     * be changed by using the
     * [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) method.
     */
    timeout?: number;
  }): Promise<Request>;

  /**
   * Returns the matched response. See [waiting for event](https://playwright.dev/docs/events#waiting-for-event) for more details about
   * events.
   *
   * **Usage**
   *
   * ```js
   * // Start waiting for response before clicking. Note no await.
   * const responsePromise = page.waitForResponse('https://example.com/resource');
   * await page.getByText('trigger response').click();
   * const response = await responsePromise;
   *
   * // Alternative way with a predicate. Note no await.
   * const responsePromise = page.waitForResponse(response => response.url() === 'https://example.com' && response.status() === 200);
   * await page.getByText('trigger response').click();
   * const response = await responsePromise;
   * ```
   *
   * @param urlOrPredicate Request URL string, regex or predicate receiving [Response] object. When a `baseURL` via the context options was
   * provided and the passed URL is a path, it gets merged via the
   * [`new URL()`](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL) constructor.
   * @param options
   */
  waitForResponse(urlOrPredicate: string|RegExp|((response: Response) => boolean|Promise<boolean>), options?: {
    /**
     * Maximum wait time in milliseconds, defaults to 30 seconds, pass `0` to disable the timeout. The default value can
     * be changed by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<Response>;

  /**
   * Waits for the given `timeout` in milliseconds.
   *
   * Note that `page.waitForTimeout()` should only be used for debugging. Tests using the timer in production are going
   * to be flaky. Use signals such as network events, selectors becoming visible and others instead.
   *
   * **Usage**
   *
   * ```js
   * // wait for 1 second
   * await page.waitForTimeout(1000);
   * ```
   *
   * @param timeout A timeout to wait for
   */
  waitForTimeout(timeout: number): Promise<void>;

  /**
   * Waits for the main frame to navigate to the given URL.
   *
   * **Usage**
   *
   * ```js
   * await page.click('a.delayed-navigation'); // Clicking the link will indirectly cause a navigation
   * await page.waitForURL('**\/target.html');
   * ```
   *
   * @param url A glob pattern, regex pattern or predicate receiving [URL] to match while waiting for the navigation. Note that if
   * the parameter is a string without wildcard characters, the method will wait for navigation to URL that is exactly
   * equal to the string.
   * @param options
   */
  waitForURL(url: string|RegExp|((url: URL) => boolean), options?: {
    /**
     * Maximum operation time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via
     * `navigationTimeout` option in the config, or by using the
     * [browserContext.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-navigation-timeout),
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout),
     * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500`
     *   ms.
     * - `'commit'` - consider operation to be finished when network response is received and the document started
     *   loading.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle"|"commit";
  }): Promise<void>;

  /**
   * This method returns all of the dedicated
   * [WebWorkers](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) associated with the page.
   *
   * **NOTE** This does not contain ServiceWorkers
   */
  workers(): Array<Worker>;

  /**
   * @deprecated This property is discouraged. Please use other libraries such as [Axe](https://www.deque.com/axe/) if you need to
   * test page accessibility. See our Node.js [guide](https://playwright.dev/docs/accessibility-testing) for integration
   * with Axe.
   */
  accessibility: Accessibility;

  /**
   * **NOTE** Only available for Chromium atm.
   *
   * Browser-specific Coverage implementation. See [Coverage] for more details.
   */
  coverage: Coverage;

  keyboard: Keyboard;

  mouse: Mouse;

  /**
   * API testing helper associated with this page. This method returns the same instance as
   * [browserContext.request](https://playwright.dev/docs/api/class-browsercontext#browser-context-request) on the
   * page's context. See
   * [browserContext.request](https://playwright.dev/docs/api/class-browsercontext#browser-context-request) for more
   * details.
   */
  request: APIRequestContext;

  touchscreen: Touchscreen;
}

/**
 * At every point of time, page exposes its current frame tree via the
 * [page.mainFrame()](https://playwright.dev/docs/api/class-page#page-main-frame) and
 * [frame.childFrames()](https://playwright.dev/docs/api/class-frame#frame-child-frames) methods.
 *
 * [Frame] object's lifecycle is controlled by three events, dispatched on the page object:
 * - [page.on('frameattached')](https://playwright.dev/docs/api/class-page#page-event-frame-attached) - fired when
 *   the frame gets attached to the page. A Frame can be attached to the page only once.
 * - [page.on('framenavigated')](https://playwright.dev/docs/api/class-page#page-event-frame-navigated) - fired when
 *   the frame commits navigation to a different URL.
 * - [page.on('framedetached')](https://playwright.dev/docs/api/class-page#page-event-frame-detached) - fired when
 *   the frame gets detached from the page.  A Frame can be detached from the page only once.
 *
 * An example of dumping frame tree:
 *
 * ```js
 * const { firefox } = require('playwright');  // Or 'chromium' or 'webkit'.
 *
 * (async () => {
 *   const browser = await firefox.launch();
 *   const page = await browser.newPage();
 *   await page.goto('https://www.google.com/chrome/browser/canary.html');
 *   dumpFrameTree(page.mainFrame(), '');
 *   await browser.close();
 *
 *   function dumpFrameTree(frame, indent) {
 *     console.log(indent + frame.url());
 *     for (const child of frame.childFrames()) {
 *       dumpFrameTree(child, indent + '  ');
 *     }
 *   }
 * })();
 * ```
 *
 */
export interface Frame {
  /**
   * Returns the return value of `pageFunction`.
   *
   * If the function passed to the
   * [frame.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate) returns a
   * [Promise], then [frame.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate)
   * would wait for the promise to resolve and return its value.
   *
   * If the function passed to the
   * [frame.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate) returns a
   * non-[Serializable] value, then
   * [frame.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate) returns
   * `undefined`. Playwright also supports transferring some additional values that are not serializable by `JSON`:
   * `-0`, `NaN`, `Infinity`, `-Infinity`.
   *
   * **Usage**
   *
   * ```js
   * const result = await frame.evaluate(([x, y]) => {
   *   return Promise.resolve(x * y);
   * }, [7, 8]);
   * console.log(result); // prints "56"
   * ```
   *
   * A string can also be passed in instead of a function.
   *
   * ```js
   * console.log(await frame.evaluate('1 + 2')); // prints "3"
   * ```
   *
   * [ElementHandle] instances can be passed as an argument to the
   * [frame.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate):
   *
   * ```js
   * const bodyHandle = await frame.evaluate('document.body');
   * const html = await frame.evaluate(([body, suffix]) => body.innerHTML + suffix, [bodyHandle, 'hello']);
   * await bodyHandle.dispose();
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluate<R, Arg>(pageFunction: PageFunction<Arg, R>, arg: Arg): Promise<R>;
  /**
   * Returns the return value of `pageFunction`.
   *
   * If the function passed to the
   * [frame.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate) returns a
   * [Promise], then [frame.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate)
   * would wait for the promise to resolve and return its value.
   *
   * If the function passed to the
   * [frame.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate) returns a
   * non-[Serializable] value, then
   * [frame.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate) returns
   * `undefined`. Playwright also supports transferring some additional values that are not serializable by `JSON`:
   * `-0`, `NaN`, `Infinity`, `-Infinity`.
   *
   * **Usage**
   *
   * ```js
   * const result = await frame.evaluate(([x, y]) => {
   *   return Promise.resolve(x * y);
   * }, [7, 8]);
   * console.log(result); // prints "56"
   * ```
   *
   * A string can also be passed in instead of a function.
   *
   * ```js
   * console.log(await frame.evaluate('1 + 2')); // prints "3"
   * ```
   *
   * [ElementHandle] instances can be passed as an argument to the
   * [frame.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate):
   *
   * ```js
   * const bodyHandle = await frame.evaluate('document.body');
   * const html = await frame.evaluate(([body, suffix]) => body.innerHTML + suffix, [bodyHandle, 'hello']);
   * await bodyHandle.dispose();
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluate<R>(pageFunction: PageFunction<void, R>, arg?: any): Promise<R>;

  /**
   * Returns the return value of `pageFunction` as a [JSHandle].
   *
   * The only difference between
   * [frame.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate) and
   * [frame.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate-handle) is
   * that [frame.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate-handle)
   * returns [JSHandle].
   *
   * If the function, passed to the
   * [frame.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate-handle),
   * returns a [Promise], then
   * [frame.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate-handle)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const aWindowHandle = await frame.evaluateHandle(() => Promise.resolve(window));
   * aWindowHandle; // Handle for the window object.
   * ```
   *
   * A string can also be passed in instead of a function.
   *
   * ```js
   * const aHandle = await frame.evaluateHandle('document'); // Handle for the 'document'.
   * ```
   *
   * [JSHandle] instances can be passed as an argument to the
   * [frame.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate-handle):
   *
   * ```js
   * const aHandle = await frame.evaluateHandle(() => document.body);
   * const resultHandle = await frame.evaluateHandle(([body, suffix]) => body.innerHTML + suffix, [aHandle, 'hello']);
   * console.log(await resultHandle.jsonValue());
   * await resultHandle.dispose();
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluateHandle<R, Arg>(pageFunction: PageFunction<Arg, R>, arg: Arg): Promise<SmartHandle<R>>;
  /**
   * Returns the return value of `pageFunction` as a [JSHandle].
   *
   * The only difference between
   * [frame.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate) and
   * [frame.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate-handle) is
   * that [frame.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate-handle)
   * returns [JSHandle].
   *
   * If the function, passed to the
   * [frame.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate-handle),
   * returns a [Promise], then
   * [frame.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate-handle)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const aWindowHandle = await frame.evaluateHandle(() => Promise.resolve(window));
   * aWindowHandle; // Handle for the window object.
   * ```
   *
   * A string can also be passed in instead of a function.
   *
   * ```js
   * const aHandle = await frame.evaluateHandle('document'); // Handle for the 'document'.
   * ```
   *
   * [JSHandle] instances can be passed as an argument to the
   * [frame.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-evaluate-handle):
   *
   * ```js
   * const aHandle = await frame.evaluateHandle(() => document.body);
   * const resultHandle = await frame.evaluateHandle(([body, suffix]) => body.innerHTML + suffix, [aHandle, 'hello']);
   * console.log(await resultHandle.jsonValue());
   * await resultHandle.dispose();
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluateHandle<R>(pageFunction: PageFunction<void, R>, arg?: any): Promise<SmartHandle<R>>;

  /**
   * **NOTE** Use locator-based [frame.locator(selector[, options])](https://playwright.dev/docs/api/class-frame#frame-locator)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns the ElementHandle pointing to the frame element.
   *
   * **NOTE** The use of [ElementHandle] is discouraged, use [Locator] objects and web-first assertions instead.
   *
   * The method finds an element matching the specified selector within the frame. If no elements match the selector,
   * returns `null`.
   * @param selector A selector to query for.
   * @param options
   */
  $<K extends keyof HTMLElementTagNameMap>(selector: K, options?: { strict: boolean }): Promise<ElementHandleForTag<K> | null>;
  /**
   * **NOTE** Use locator-based [frame.locator(selector[, options])](https://playwright.dev/docs/api/class-frame#frame-locator)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns the ElementHandle pointing to the frame element.
   *
   * **NOTE** The use of [ElementHandle] is discouraged, use [Locator] objects and web-first assertions instead.
   *
   * The method finds an element matching the specified selector within the frame. If no elements match the selector,
   * returns `null`.
   * @param selector A selector to query for.
   * @param options
   */
  $(selector: string, options?: { strict: boolean }): Promise<ElementHandle<SVGElement | HTMLElement> | null>;

  /**
   * **NOTE** Use locator-based [frame.locator(selector[, options])](https://playwright.dev/docs/api/class-frame#frame-locator)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns the ElementHandles pointing to the frame elements.
   *
   * **NOTE** The use of [ElementHandle] is discouraged, use [Locator] objects instead.
   *
   * The method finds all elements matching the specified selector within the frame. If no elements match the selector,
   * returns empty array.
   * @param selector A selector to query for.
   */
  $$<K extends keyof HTMLElementTagNameMap>(selector: K): Promise<ElementHandleForTag<K>[]>;
  /**
   * **NOTE** Use locator-based [frame.locator(selector[, options])](https://playwright.dev/docs/api/class-frame#frame-locator)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns the ElementHandles pointing to the frame elements.
   *
   * **NOTE** The use of [ElementHandle] is discouraged, use [Locator] objects instead.
   *
   * The method finds all elements matching the specified selector within the frame. If no elements match the selector,
   * returns empty array.
   * @param selector A selector to query for.
   */
  $$(selector: string): Promise<ElementHandle<SVGElement | HTMLElement>[]>;

  /**
   * **NOTE** This method does not wait for the element to pass the actionability checks and therefore can lead to the flaky
   * tests. Use
   * [locator.evaluate(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-locator#locator-evaluate),
   * other [Locator] helper methods or web-first assertions instead.
   *
   * Returns the return value of `pageFunction`.
   *
   * The method finds an element matching the specified selector within the frame and passes it as a first argument to
   * `pageFunction`. If no elements match the selector, the method throws an error.
   *
   * If `pageFunction` returns a [Promise], then
   * [frame.$eval(selector, pageFunction[, arg, options])](https://playwright.dev/docs/api/class-frame#frame-eval-on-selector)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const searchValue = await frame.$eval('#search', el => el.value);
   * const preloadHref = await frame.$eval('link[rel=preload]', el => el.href);
   * const html = await frame.$eval('.main-container', (e, suffix) => e.outerHTML + suffix, 'hello');
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   * @param options
   */
  $eval<K extends keyof HTMLElementTagNameMap, R, Arg>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K], Arg, R>, arg: Arg): Promise<R>;
  /**
   * **NOTE** This method does not wait for the element to pass the actionability checks and therefore can lead to the flaky
   * tests. Use
   * [locator.evaluate(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-locator#locator-evaluate),
   * other [Locator] helper methods or web-first assertions instead.
   *
   * Returns the return value of `pageFunction`.
   *
   * The method finds an element matching the specified selector within the frame and passes it as a first argument to
   * `pageFunction`. If no elements match the selector, the method throws an error.
   *
   * If `pageFunction` returns a [Promise], then
   * [frame.$eval(selector, pageFunction[, arg, options])](https://playwright.dev/docs/api/class-frame#frame-eval-on-selector)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const searchValue = await frame.$eval('#search', el => el.value);
   * const preloadHref = await frame.$eval('link[rel=preload]', el => el.href);
   * const html = await frame.$eval('.main-container', (e, suffix) => e.outerHTML + suffix, 'hello');
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   * @param options
   */
  $eval<R, Arg, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E, Arg, R>, arg: Arg): Promise<R>;
  /**
   * **NOTE** This method does not wait for the element to pass the actionability checks and therefore can lead to the flaky
   * tests. Use
   * [locator.evaluate(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-locator#locator-evaluate),
   * other [Locator] helper methods or web-first assertions instead.
   *
   * Returns the return value of `pageFunction`.
   *
   * The method finds an element matching the specified selector within the frame and passes it as a first argument to
   * `pageFunction`. If no elements match the selector, the method throws an error.
   *
   * If `pageFunction` returns a [Promise], then
   * [frame.$eval(selector, pageFunction[, arg, options])](https://playwright.dev/docs/api/class-frame#frame-eval-on-selector)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const searchValue = await frame.$eval('#search', el => el.value);
   * const preloadHref = await frame.$eval('link[rel=preload]', el => el.href);
   * const html = await frame.$eval('.main-container', (e, suffix) => e.outerHTML + suffix, 'hello');
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   * @param options
   */
  $eval<K extends keyof HTMLElementTagNameMap, R>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K], void, R>, arg?: any): Promise<R>;
  /**
   * **NOTE** This method does not wait for the element to pass the actionability checks and therefore can lead to the flaky
   * tests. Use
   * [locator.evaluate(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-locator#locator-evaluate),
   * other [Locator] helper methods or web-first assertions instead.
   *
   * Returns the return value of `pageFunction`.
   *
   * The method finds an element matching the specified selector within the frame and passes it as a first argument to
   * `pageFunction`. If no elements match the selector, the method throws an error.
   *
   * If `pageFunction` returns a [Promise], then
   * [frame.$eval(selector, pageFunction[, arg, options])](https://playwright.dev/docs/api/class-frame#frame-eval-on-selector)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const searchValue = await frame.$eval('#search', el => el.value);
   * const preloadHref = await frame.$eval('link[rel=preload]', el => el.href);
   * const html = await frame.$eval('.main-container', (e, suffix) => e.outerHTML + suffix, 'hello');
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   * @param options
   */
  $eval<R, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E, void, R>, arg?: any): Promise<R>;

  /**
   * **NOTE** In most cases,
   * [locator.evaluateAll(pageFunction[, arg])](https://playwright.dev/docs/api/class-locator#locator-evaluate-all),
   * other [Locator] helper methods and web-first assertions do a better job.
   *
   * Returns the return value of `pageFunction`.
   *
   * The method finds all elements matching the specified selector within the frame and passes an array of matched
   * elements as a first argument to `pageFunction`.
   *
   * If `pageFunction` returns a [Promise], then
   * [frame.$$eval(selector, pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-eval-on-selector-all)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const divsCounts = await frame.$$eval('div', (divs, min) => divs.length >= min, 10);
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  $$eval<K extends keyof HTMLElementTagNameMap, R, Arg>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K][], Arg, R>, arg: Arg): Promise<R>;
  /**
   * **NOTE** In most cases,
   * [locator.evaluateAll(pageFunction[, arg])](https://playwright.dev/docs/api/class-locator#locator-evaluate-all),
   * other [Locator] helper methods and web-first assertions do a better job.
   *
   * Returns the return value of `pageFunction`.
   *
   * The method finds all elements matching the specified selector within the frame and passes an array of matched
   * elements as a first argument to `pageFunction`.
   *
   * If `pageFunction` returns a [Promise], then
   * [frame.$$eval(selector, pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-eval-on-selector-all)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const divsCounts = await frame.$$eval('div', (divs, min) => divs.length >= min, 10);
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  $$eval<R, Arg, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E[], Arg, R>, arg: Arg): Promise<R>;
  /**
   * **NOTE** In most cases,
   * [locator.evaluateAll(pageFunction[, arg])](https://playwright.dev/docs/api/class-locator#locator-evaluate-all),
   * other [Locator] helper methods and web-first assertions do a better job.
   *
   * Returns the return value of `pageFunction`.
   *
   * The method finds all elements matching the specified selector within the frame and passes an array of matched
   * elements as a first argument to `pageFunction`.
   *
   * If `pageFunction` returns a [Promise], then
   * [frame.$$eval(selector, pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-eval-on-selector-all)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const divsCounts = await frame.$$eval('div', (divs, min) => divs.length >= min, 10);
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  $$eval<K extends keyof HTMLElementTagNameMap, R>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K][], void, R>, arg?: any): Promise<R>;
  /**
   * **NOTE** In most cases,
   * [locator.evaluateAll(pageFunction[, arg])](https://playwright.dev/docs/api/class-locator#locator-evaluate-all),
   * other [Locator] helper methods and web-first assertions do a better job.
   *
   * Returns the return value of `pageFunction`.
   *
   * The method finds all elements matching the specified selector within the frame and passes an array of matched
   * elements as a first argument to `pageFunction`.
   *
   * If `pageFunction` returns a [Promise], then
   * [frame.$$eval(selector, pageFunction[, arg])](https://playwright.dev/docs/api/class-frame#frame-eval-on-selector-all)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const divsCounts = await frame.$$eval('div', (divs, min) => divs.length >= min, 10);
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  $$eval<R, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E[], void, R>, arg?: any): Promise<R>;

  /**
   * Returns when the `pageFunction` returns a truthy value, returns that value.
   *
   * **Usage**
   *
   * The
   * [frame.waitForFunction(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-frame#frame-wait-for-function)
   * can be used to observe viewport size change:
   *
   * ```js
   * const { firefox } = require('playwright');  // Or 'chromium' or 'webkit'.
   *
   * (async () => {
   *   const browser = await firefox.launch();
   *   const page = await browser.newPage();
   *   const watchDog = page.mainFrame().waitForFunction('window.innerWidth < 100');
   *   page.setViewportSize({width: 50, height: 50});
   *   await watchDog;
   *   await browser.close();
   * })();
   * ```
   *
   * To pass an argument to the predicate of `frame.waitForFunction` function:
   *
   * ```js
   * const selector = '.foo';
   * await frame.waitForFunction(selector => !!document.querySelector(selector), selector);
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   * @param options
   */
  waitForFunction<R, Arg>(pageFunction: PageFunction<Arg, R>, arg: Arg, options?: PageWaitForFunctionOptions): Promise<SmartHandle<R>>;
  /**
   * Returns when the `pageFunction` returns a truthy value, returns that value.
   *
   * **Usage**
   *
   * The
   * [frame.waitForFunction(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-frame#frame-wait-for-function)
   * can be used to observe viewport size change:
   *
   * ```js
   * const { firefox } = require('playwright');  // Or 'chromium' or 'webkit'.
   *
   * (async () => {
   *   const browser = await firefox.launch();
   *   const page = await browser.newPage();
   *   const watchDog = page.mainFrame().waitForFunction('window.innerWidth < 100');
   *   page.setViewportSize({width: 50, height: 50});
   *   await watchDog;
   *   await browser.close();
   * })();
   * ```
   *
   * To pass an argument to the predicate of `frame.waitForFunction` function:
   *
   * ```js
   * const selector = '.foo';
   * await frame.waitForFunction(selector => !!document.querySelector(selector), selector);
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   * @param options
   */
  waitForFunction<R>(pageFunction: PageFunction<void, R>, arg?: any, options?: PageWaitForFunctionOptions): Promise<SmartHandle<R>>;

  /**
   * Returns when element specified by selector satisfies `state` option. Returns `null` if waiting for `hidden` or
   * `detached`.
   *
   * **NOTE** Playwright automatically waits for element to be ready before performing an action. Using [Locator]
   * objects and web-first assertions make the code wait-for-selector-free.
   *
   * Wait for the `selector` to satisfy `state` option (either appear/disappear from dom, or become visible/hidden). If
   * at the moment of calling the method `selector` already satisfies the condition, the method will return immediately.
   * If the selector doesn't satisfy the condition for the `timeout` milliseconds, the function will throw.
   *
   * **Usage**
   *
   * This method works across navigations:
   *
   * ```js
   * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
   *
   * (async () => {
   *   const browser = await chromium.launch();
   *   const page = await browser.newPage();
   *   for (let currentURL of ['https://google.com', 'https://bbc.com']) {
   *     await page.goto(currentURL);
   *     const element = await page.mainFrame().waitForSelector('img');
   *     console.log('Loaded image: ' + await element.getAttribute('src'));
   *   }
   *   await browser.close();
   * })();
   * ```
   *
   * @param selector A selector to query for.
   * @param options
   */
  waitForSelector<K extends keyof HTMLElementTagNameMap>(selector: K, options?: PageWaitForSelectorOptionsNotHidden): Promise<ElementHandleForTag<K>>;
  /**
   * Returns when element specified by selector satisfies `state` option. Returns `null` if waiting for `hidden` or
   * `detached`.
   *
   * **NOTE** Playwright automatically waits for element to be ready before performing an action. Using [Locator]
   * objects and web-first assertions make the code wait-for-selector-free.
   *
   * Wait for the `selector` to satisfy `state` option (either appear/disappear from dom, or become visible/hidden). If
   * at the moment of calling the method `selector` already satisfies the condition, the method will return immediately.
   * If the selector doesn't satisfy the condition for the `timeout` milliseconds, the function will throw.
   *
   * **Usage**
   *
   * This method works across navigations:
   *
   * ```js
   * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
   *
   * (async () => {
   *   const browser = await chromium.launch();
   *   const page = await browser.newPage();
   *   for (let currentURL of ['https://google.com', 'https://bbc.com']) {
   *     await page.goto(currentURL);
   *     const element = await page.mainFrame().waitForSelector('img');
   *     console.log('Loaded image: ' + await element.getAttribute('src'));
   *   }
   *   await browser.close();
   * })();
   * ```
   *
   * @param selector A selector to query for.
   * @param options
   */
  waitForSelector(selector: string, options?: PageWaitForSelectorOptionsNotHidden): Promise<ElementHandle<SVGElement | HTMLElement>>;
  /**
   * Returns when element specified by selector satisfies `state` option. Returns `null` if waiting for `hidden` or
   * `detached`.
   *
   * **NOTE** Playwright automatically waits for element to be ready before performing an action. Using [Locator]
   * objects and web-first assertions make the code wait-for-selector-free.
   *
   * Wait for the `selector` to satisfy `state` option (either appear/disappear from dom, or become visible/hidden). If
   * at the moment of calling the method `selector` already satisfies the condition, the method will return immediately.
   * If the selector doesn't satisfy the condition for the `timeout` milliseconds, the function will throw.
   *
   * **Usage**
   *
   * This method works across navigations:
   *
   * ```js
   * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
   *
   * (async () => {
   *   const browser = await chromium.launch();
   *   const page = await browser.newPage();
   *   for (let currentURL of ['https://google.com', 'https://bbc.com']) {
   *     await page.goto(currentURL);
   *     const element = await page.mainFrame().waitForSelector('img');
   *     console.log('Loaded image: ' + await element.getAttribute('src'));
   *   }
   *   await browser.close();
   * })();
   * ```
   *
   * @param selector A selector to query for.
   * @param options
   */
  waitForSelector<K extends keyof HTMLElementTagNameMap>(selector: K, options: PageWaitForSelectorOptions): Promise<ElementHandleForTag<K> | null>;
  /**
   * Returns when element specified by selector satisfies `state` option. Returns `null` if waiting for `hidden` or
   * `detached`.
   *
   * **NOTE** Playwright automatically waits for element to be ready before performing an action. Using [Locator]
   * objects and web-first assertions make the code wait-for-selector-free.
   *
   * Wait for the `selector` to satisfy `state` option (either appear/disappear from dom, or become visible/hidden). If
   * at the moment of calling the method `selector` already satisfies the condition, the method will return immediately.
   * If the selector doesn't satisfy the condition for the `timeout` milliseconds, the function will throw.
   *
   * **Usage**
   *
   * This method works across navigations:
   *
   * ```js
   * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
   *
   * (async () => {
   *   const browser = await chromium.launch();
   *   const page = await browser.newPage();
   *   for (let currentURL of ['https://google.com', 'https://bbc.com']) {
   *     await page.goto(currentURL);
   *     const element = await page.mainFrame().waitForSelector('img');
   *     console.log('Loaded image: ' + await element.getAttribute('src'));
   *   }
   *   await browser.close();
   * })();
   * ```
   *
   * @param selector A selector to query for.
   * @param options
   */
  waitForSelector(selector: string, options: PageWaitForSelectorOptions): Promise<null|ElementHandle<SVGElement | HTMLElement>>;
  /**
   * Returns the added tag when the script's onload fires or when the script content was injected into frame.
   *
   * Adds a `<script>` tag into the page with the desired url or content.
   * @param options
   */
  addScriptTag(options?: {
    /**
     * Raw JavaScript content to be injected into frame.
     */
    content?: string;

    /**
     * Path to the JavaScript file to be injected into frame. If `path` is a relative path, then it is resolved relative
     * to the current working directory.
     */
    path?: string;

    /**
     * Script type. Use 'module' in order to load a Javascript ES6 module. See
     * [script](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/script) for more details.
     */
    type?: string;

    /**
     * URL of a script to be added.
     */
    url?: string;
  }): Promise<ElementHandle>;

  /**
   * Returns the added tag when the stylesheet's onload fires or when the CSS content was injected into frame.
   *
   * Adds a `<link rel="stylesheet">` tag into the page with the desired url or a `<style type="text/css">` tag with the
   * content.
   * @param options
   */
  addStyleTag(options?: {
    /**
     * Raw CSS content to be injected into frame.
     */
    content?: string;

    /**
     * Path to the CSS file to be injected into frame. If `path` is a relative path, then it is resolved relative to the
     * current working directory.
     */
    path?: string;

    /**
     * URL of the `<link>` tag.
     */
    url?: string;
  }): Promise<ElementHandle>;

  /**
   * **NOTE** Use locator-based [locator.check([options])](https://playwright.dev/docs/api/class-locator#locator-check) instead.
   * Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method checks an element matching `selector` by performing the following steps:
   * 1. Find an element matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Ensure that matched element is a checkbox or a radio input. If not, this method throws. If the element is
   *    already checked, this method returns immediately.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the matched element, unless `force` option is set. If
   *    the element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to click in the center of the
   *    element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now checked. If not, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  check(selector: string, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  childFrames(): Array<Frame>;

  /**
   * **NOTE** Use locator-based [locator.click([options])](https://playwright.dev/docs/api/class-locator#locator-click) instead.
   * Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method clicks an element matching `selector` by performing the following steps:
   * 1. Find an element matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the matched element, unless `force` option is set. If
   *    the element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to click in the center of the
   *    element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  click(selector: string, options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * defaults to 1. See [UIEvent.detail].
     */
    clickCount?: number;

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores
     * current modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * Gets the full HTML contents of the frame, including the doctype.
   */
  content(): Promise<string>;

  /**
   * **NOTE** Use locator-based [locator.dblclick([options])](https://playwright.dev/docs/api/class-locator#locator-dblclick)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method double clicks an element matching `selector` by performing the following steps:
   * 1. Find an element matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the matched element, unless `force` option is set. If
   *    the element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to double click in the center of the
   *    element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set. Note that if
   *    the first click of the `dblclick()` triggers a navigation event, this method will throw.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   *
   * **NOTE** `frame.dblclick()` dispatches two `click` events and a single `dblclick` event.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  dblclick(selector: string, options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores
     * current modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * **NOTE** Use locator-based
   * [locator.dispatchEvent(type[, eventInit, options])](https://playwright.dev/docs/api/class-locator#locator-dispatch-event)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * The snippet below dispatches the `click` event on the element. Regardless of the visibility state of the element,
   * `click` is dispatched. This is equivalent to calling
   * [element.click()](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/click).
   *
   * **Usage**
   *
   * ```js
   * await frame.dispatchEvent('button#submit', 'click');
   * ```
   *
   * Under the hood, it creates an instance of an event based on the given `type`, initializes it with `eventInit`
   * properties and dispatches it on the element. Events are `composed`, `cancelable` and bubble by default.
   *
   * Since `eventInit` is event-specific, please refer to the events documentation for the lists of initial properties:
   * - [DragEvent](https://developer.mozilla.org/en-US/docs/Web/API/DragEvent/DragEvent)
   * - [FocusEvent](https://developer.mozilla.org/en-US/docs/Web/API/FocusEvent/FocusEvent)
   * - [KeyboardEvent](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/KeyboardEvent)
   * - [MouseEvent](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/MouseEvent)
   * - [PointerEvent](https://developer.mozilla.org/en-US/docs/Web/API/PointerEvent/PointerEvent)
   * - [TouchEvent](https://developer.mozilla.org/en-US/docs/Web/API/TouchEvent/TouchEvent)
   * - [Event](https://developer.mozilla.org/en-US/docs/Web/API/Event/Event)
   *
   * You can also specify `JSHandle` as the property value if you want live objects to be passed into the event:
   *
   * ```js
   * // Note you can only create DataTransfer in Chromium and Firefox
   * const dataTransfer = await frame.evaluateHandle(() => new DataTransfer());
   * await frame.dispatchEvent('#source', 'dragstart', { dataTransfer });
   * ```
   *
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param type DOM event type: `"click"`, `"dragstart"`, etc.
   * @param eventInit Optional event-specific initialization properties.
   * @param options
   */
  dispatchEvent(selector: string, type: string, eventInit?: EvaluationArgument, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * @param source A selector to search for an element to drag. If there are multiple elements satisfying the selector, the first will
   * be used.
   * @param target A selector to search for an element to drop onto. If there are multiple elements satisfying the selector, the first
   * will be used.
   * @param options
   */
  dragAndDrop(source: string, target: string, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Clicks on the source element at this point relative to the top-left corner of the element's padding box. If not
     * specified, some visible point of the element is used.
     */
    sourcePosition?: {
      x: number;

      y: number;
    };

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Drops on the target element at this point relative to the top-left corner of the element's padding box. If not
     * specified, some visible point of the element is used.
     */
    targetPosition?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * **NOTE** Use locator-based [locator.fill(value[, options])](https://playwright.dev/docs/api/class-locator#locator-fill)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method waits for an element matching `selector`, waits for [actionability](https://playwright.dev/docs/actionability) checks,
   * focuses the element, fills it and triggers an `input` event after filling. Note that you can pass an empty string
   * to clear the input field.
   *
   * If the target element is not an `<input>`, `<textarea>` or `[contenteditable]` element, this method throws an
   * error. However, if the element is inside the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), the control will be filled
   * instead.
   *
   * To send fine-grained keyboard events, use
   * [frame.type(selector, text[, options])](https://playwright.dev/docs/api/class-frame#frame-type).
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param value Value to fill for the `<input>`, `<textarea>` or `[contenteditable]` element.
   * @param options
   */
  fill(selector: string, value: string, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * **NOTE** Use locator-based [locator.focus([options])](https://playwright.dev/docs/api/class-locator#locator-focus) instead.
   * Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method fetches an element with `selector` and focuses it. If there's no element matching `selector`, the
   * method waits until a matching element appears in the DOM.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  focus(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns the `frame` or `iframe` element handle which corresponds to this frame.
   *
   * This is an inverse of
   * [elementHandle.contentFrame()](https://playwright.dev/docs/api/class-elementhandle#element-handle-content-frame).
   * Note that returned handle actually belongs to the parent frame.
   *
   * This method throws an error if the frame has been detached before `frameElement()` returns.
   *
   * **Usage**
   *
   * ```js
   * const frameElement = await frame.frameElement();
   * const contentFrame = await frameElement.contentFrame();
   * console.log(frame === contentFrame);  // -> true
   * ```
   *
   */
  frameElement(): Promise<ElementHandle>;

  /**
   * When working with iframes, you can create a frame locator that will enter the iframe and allow selecting elements
   * in that iframe.
   *
   * **Usage**
   *
   * Following snippet locates element with text "Submit" in the iframe with id `my-frame`, like `<iframe
   * id="my-frame">`:
   *
   * ```js
   * const locator = frame.frameLocator('#my-iframe').getByText('Submit');
   * await locator.click();
   * ```
   *
   * @param selector A selector to use when resolving DOM element.
   */
  frameLocator(selector: string): FrameLocator;

  /**
   * **NOTE** Use locator-based
   * [locator.getAttribute(name[, options])](https://playwright.dev/docs/api/class-locator#locator-get-attribute)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns element attribute value.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param name Attribute name to get the value for.
   * @param options
   */
  getAttribute(selector: string, name: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<null|string>;

  /**
   * Allows locating elements by their alt text.
   *
   * **Usage**
   *
   * For example, this method will find the image by alt text "Playwright logo":
   *
   * ```html
   * <img alt='Playwright logo'>
   * ```
   *
   * ```js
   * await page.getByAltText('Playwright logo').click();
   * ```
   *
   * @param text Text to locate the element for.
   * @param options
   */
  getByAltText(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Allows locating input elements by the text of the associated `<label>` or `aria-labelledby` element, or by the
   * `aria-label` attribute.
   *
   * **Usage**
   *
   * For example, this method will find inputs by label "Username" and "Password" in the following DOM:
   *
   * ```html
   * <input aria-label="Username">
   * <label for="password-input">Password:</label>
   * <input id="password-input">
   * ```
   *
   * ```js
   * await page.getByLabel('Username').fill('john');
   * await page.getByLabel('Password').fill('secret');
   * ```
   *
   * @param text Text to locate the element for.
   * @param options
   */
  getByLabel(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Allows locating input elements by the placeholder text.
   *
   * **Usage**
   *
   * For example, consider the following DOM structure.
   *
   * ```html
   * <input type="email" placeholder="name@example.com" />
   * ```
   *
   * You can fill the input after locating it by the placeholder text:
   *
   * ```js
   * await page
   *     .getByPlaceholder("name@example.com")
   *     .fill("playwright@microsoft.com");
   * ```
   *
   * @param text Text to locate the element for.
   * @param options
   */
  getByPlaceholder(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Allows locating elements by their [ARIA role](https://www.w3.org/TR/wai-aria-1.2/#roles),
   * [ARIA attributes](https://www.w3.org/TR/wai-aria-1.2/#aria-attributes) and
   * [accessible name](https://w3c.github.io/accname/#dfn-accessible-name).
   *
   * **Usage**
   *
   * Consider the following DOM structure.
   *
   * ```html
   * <h3>Sign up</h3>
   * <label>
   *   <input type="checkbox" /> Subscribe
   * </label>
   * <br/>
   * <button>Submit</button>
   * ```
   *
   * You can locate each element by it's implicit role:
   *
   * ```js
   * await expect(page.getByRole('heading', { name: 'Sign up' })).toBeVisible();
   *
   * await page.getByRole('checkbox', { name: 'Subscribe' }).check();
   *
   * await page.getByRole('button', { name: /submit/i }).click();
   * ```
   *
   * **Details**
   *
   * Role selector **does not replace** accessibility audits and conformance tests, but rather gives early feedback
   * about the ARIA guidelines.
   *
   * Many html elements have an implicitly [defined role](https://w3c.github.io/html-aam/#html-element-role-mappings)
   * that is recognized by the role selector. You can find all the
   * [supported roles here](https://www.w3.org/TR/wai-aria-1.2/#role_definitions). ARIA guidelines **do not recommend**
   * duplicating implicit roles and attributes by setting `role` and/or `aria-*` attributes to default values.
   * @param role Required aria role.
   * @param options
   */
  getByRole(role: "alert"|"alertdialog"|"application"|"article"|"banner"|"blockquote"|"button"|"caption"|"cell"|"checkbox"|"code"|"columnheader"|"combobox"|"complementary"|"contentinfo"|"definition"|"deletion"|"dialog"|"directory"|"document"|"emphasis"|"feed"|"figure"|"form"|"generic"|"grid"|"gridcell"|"group"|"heading"|"img"|"insertion"|"link"|"list"|"listbox"|"listitem"|"log"|"main"|"marquee"|"math"|"meter"|"menu"|"menubar"|"menuitem"|"menuitemcheckbox"|"menuitemradio"|"navigation"|"none"|"note"|"option"|"paragraph"|"presentation"|"progressbar"|"radio"|"radiogroup"|"region"|"row"|"rowgroup"|"rowheader"|"scrollbar"|"search"|"searchbox"|"separator"|"slider"|"spinbutton"|"status"|"strong"|"subscript"|"superscript"|"switch"|"tab"|"table"|"tablist"|"tabpanel"|"term"|"textbox"|"time"|"timer"|"toolbar"|"tooltip"|"tree"|"treegrid"|"treeitem", options?: {
    /**
     * An attribute that is usually set by `aria-checked` or native `<input type=checkbox>` controls.
     *
     * Learn more about [`aria-checked`](https://www.w3.org/TR/wai-aria-1.2/#aria-checked).
     */
    checked?: boolean;

    /**
     * An attribute that is usually set by `aria-disabled` or `disabled`.
     *
     * **NOTE** Unlike most other attributes, `disabled` is inherited through the DOM hierarchy. Learn more about
     * [`aria-disabled`](https://www.w3.org/TR/wai-aria-1.2/#aria-disabled).
     */
    disabled?: boolean;

    /**
     * Whether `name` is matched exactly: case-sensitive and whole-string. Defaults to false. Ignored when `name` is a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;

    /**
     * An attribute that is usually set by `aria-expanded`.
     *
     * Learn more about [`aria-expanded`](https://www.w3.org/TR/wai-aria-1.2/#aria-expanded).
     */
    expanded?: boolean;

    /**
     * Option that controls whether hidden elements are matched. By default, only non-hidden elements, as
     * [defined by ARIA](https://www.w3.org/TR/wai-aria-1.2/#tree_exclusion), are matched by role selector.
     *
     * Learn more about [`aria-hidden`](https://www.w3.org/TR/wai-aria-1.2/#aria-hidden).
     */
    includeHidden?: boolean;

    /**
     * A number attribute that is usually present for roles `heading`, `listitem`, `row`, `treeitem`, with default values
     * for `<h1>-<h6>` elements.
     *
     * Learn more about [`aria-level`](https://www.w3.org/TR/wai-aria-1.2/#aria-level).
     */
    level?: number;

    /**
     * Option to match the [accessible name](https://w3c.github.io/accname/#dfn-accessible-name). By default, matching is
     * case-insensitive and searches for a substring, use `exact` to control this behavior.
     *
     * Learn more about [accessible name](https://w3c.github.io/accname/#dfn-accessible-name).
     */
    name?: string|RegExp;

    /**
     * An attribute that is usually set by `aria-pressed`.
     *
     * Learn more about [`aria-pressed`](https://www.w3.org/TR/wai-aria-1.2/#aria-pressed).
     */
    pressed?: boolean;

    /**
     * An attribute that is usually set by `aria-selected`.
     *
     * Learn more about [`aria-selected`](https://www.w3.org/TR/wai-aria-1.2/#aria-selected).
     */
    selected?: boolean;
  }): Locator;

  /**
   * Locate element by the test id.
   *
   * **Usage**
   *
   * Consider the following DOM structure.
   *
   * ```html
   * <button data-testid="directions">Itinéraire</button>
   * ```
   *
   * You can locate the element by it's test id:
   *
   * ```js
   * await page.getByTestId('directions').click();
   * ```
   *
   * **Details**
   *
   * By default, the `data-testid` attribute is used as a test id. Use
   * [selectors.setTestIdAttribute(attributeName)](https://playwright.dev/docs/api/class-selectors#selectors-set-test-id-attribute)
   * to configure a different test id attribute if necessary.
   *
   * ```js
   * // Set custom test id attribute from @playwright/test config:
   * use: {
   *   testIdAttribute: 'data-pw'
   * }
   * ```
   *
   * @param testId Id to locate the element by.
   */
  getByTestId(testId: string|RegExp): Locator;

  /**
   * Allows locating elements that contain given text.
   *
   * See also [locator.filter([options])](https://playwright.dev/docs/api/class-locator#locator-filter) that allows to
   * match by another criteria, like an accessible role, and then filter by the text content.
   *
   * **Usage**
   *
   * Consider the following DOM structure:
   *
   * ```html
   * <div>Hello <span>world</span></div>
   * <div>Hello</div>
   * ```
   *
   * You can locate by text substring, exact string, or a regular expression:
   *
   * ```js
   * // Matches <span>
   * page.getByText('world')
   *
   * // Matches first <div>
   * page.getByText('Hello world')
   *
   * // Matches second <div>
   * page.getByText('Hello', { exact: true })
   *
   * // Matches both <div>s
   * page.getByText(/Hello/)
   *
   * // Matches second <div>
   * page.getByText(/^hello$/i)
   * ```
   *
   * **Details**
   *
   * Matching by text always normalizes whitespace, even with exact match. For example, it turns multiple spaces into
   * one, turns line breaks into spaces and ignores leading and trailing whitespace.
   *
   * Input elements of the type `button` and `submit` are matched by their `value` instead of the text content. For
   * example, locating by text `"Log in"` matches `<input type=button value="Log in">`.
   * @param text Text to locate the element for.
   * @param options
   */
  getByText(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Allows locating elements by their title attribute.
   *
   * **Usage**
   *
   * Consider the following DOM structure.
   *
   * ```html
   * <span title='Issues count'>25 issues</span>
   * ```
   *
   * You can check the issues count after locating it by the title text:
   *
   * ```js
   * await expect(page.getByTitle('Issues count')).toHaveText('25 issues');
   * ```
   *
   * @param text Text to locate the element for.
   * @param options
   */
  getByTitle(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Returns the main resource response. In case of multiple redirects, the navigation will resolve with the response of
   * the last redirect.
   *
   * The method will throw an error if:
   * - there's an SSL error (e.g. in case of self-signed certificates).
   * - target URL is invalid.
   * - the `timeout` is exceeded during navigation.
   * - the remote server does not respond or is unreachable.
   * - the main resource failed to load.
   *
   * The method will not throw an error when any valid HTTP status code is returned by the remote server, including 404
   * "Not Found" and 500 "Internal Server Error".  The status code for such responses can be retrieved by calling
   * [response.status()](https://playwright.dev/docs/api/class-response#response-status).
   *
   * **NOTE** The method either throws an error or returns a main resource response. The only exceptions are navigation
   * to `about:blank` or navigation to the same URL with a different hash, which would succeed and return `null`.
   *
   * **NOTE** Headless mode doesn't support navigation to a PDF document. See the
   * [upstream issue](https://bugs.chromium.org/p/chromium/issues/detail?id=761295).
   * @param url URL to navigate frame to. The url should include scheme, e.g. `https://`.
   * @param options
   */
  goto(url: string, options?: {
    /**
     * Referer header value. If provided it will take preference over the referer header value set by
     * [page.setExtraHTTPHeaders(headers)](https://playwright.dev/docs/api/class-page#page-set-extra-http-headers).
     */
    referer?: string;

    /**
     * Maximum operation time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via
     * `navigationTimeout` option in the config, or by using the
     * [browserContext.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-navigation-timeout),
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout),
     * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500`
     *   ms.
     * - `'commit'` - consider operation to be finished when network response is received and the document started
     *   loading.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle"|"commit";
  }): Promise<null|Response>;

  /**
   * **NOTE** Use locator-based [locator.hover([options])](https://playwright.dev/docs/api/class-locator#locator-hover) instead.
   * Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method hovers over an element matching `selector` by performing the following steps:
   * 1. Find an element matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the matched element, unless `force` option is set. If
   *    the element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to hover over the center of the
   *    element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  hover(selector: string, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores
     * current modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * **NOTE** Use locator-based [locator.innerHTML([options])](https://playwright.dev/docs/api/class-locator#locator-inner-html)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns `element.innerHTML`.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  innerHTML(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<string>;

  /**
   * **NOTE** Use locator-based [locator.innerText([options])](https://playwright.dev/docs/api/class-locator#locator-inner-text)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns `element.innerText`.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  innerText(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<string>;

  /**
   * **NOTE** Use locator-based
   * [locator.inputValue([options])](https://playwright.dev/docs/api/class-locator#locator-input-value) instead. Read
   * more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns `input.value` for the selected `<input>` or `<textarea>` or `<select>` element.
   *
   * Throws for non-input elements. However, if the element is inside the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), returns the value of the
   * control.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  inputValue(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<string>;

  /**
   * **NOTE** Use locator-based [locator.isChecked([options])](https://playwright.dev/docs/api/class-locator#locator-is-checked)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns whether the element is checked. Throws if the element is not a checkbox or radio input.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  isChecked(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns `true` if the frame has been detached, or `false` otherwise.
   */
  isDetached(): boolean;

  /**
   * **NOTE** Use locator-based
   * [locator.isDisabled([options])](https://playwright.dev/docs/api/class-locator#locator-is-disabled) instead. Read
   * more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns whether the element is disabled, the opposite of [enabled](https://playwright.dev/docs/actionability#enabled).
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  isDisabled(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * **NOTE** Use locator-based
   * [locator.isEditable([options])](https://playwright.dev/docs/api/class-locator#locator-is-editable) instead. Read
   * more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns whether the element is [editable](https://playwright.dev/docs/actionability#editable).
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  isEditable(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns whether the element is [enabled](https://playwright.dev/docs/actionability#enabled).
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  isEnabled(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * **NOTE** Use locator-based [locator.isHidden([options])](https://playwright.dev/docs/api/class-locator#locator-is-hidden)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns whether the element is hidden, the opposite of [visible](https://playwright.dev/docs/actionability#visible).  `selector` that
   * does not match any elements is considered hidden.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  isHidden(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * @deprecated This option is ignored.
     * [frame.isHidden(selector[, options])](https://playwright.dev/docs/api/class-frame#frame-is-hidden) does not wait
     * for the element to become hidden and returns immediately.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * **NOTE** Use locator-based [locator.isVisible([options])](https://playwright.dev/docs/api/class-locator#locator-is-visible)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns whether the element is [visible](https://playwright.dev/docs/actionability#visible). `selector` that does not match any elements
   * is considered not visible.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  isVisible(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * @deprecated This option is ignored.
     * [frame.isVisible(selector[, options])](https://playwright.dev/docs/api/class-frame#frame-is-visible) does not wait
     * for the element to become visible and returns immediately.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * The method returns an element locator that can be used to perform actions on this page / frame. Locator is resolved
   * to the element immediately before performing an action, so a series of actions on the same locator can in fact be
   * performed on different DOM elements. That would happen if the DOM structure between those actions has changed.
   *
   * [Learn more about locators](https://playwright.dev/docs/locators).
   *
   * [Learn more about locators](https://playwright.dev/docs/locators).
   * @param selector A selector to use when resolving DOM element.
   * @param options
   */
  locator(selector: string, options?: {
    /**
     * Matches elements containing an element that matches an inner locator. Inner locator is queried against the outer
     * one. For example, `article` that has `text=Playwright` matches `<article><div>Playwright</div></article>`.
     *
     * Note that outer and inner locators must belong to the same frame. Inner locator must not contain [FrameLocator]s.
     */
    has?: Locator;

    /**
     * Matches elements containing specified text somewhere inside, possibly in a child or a descendant element. When
     * passed a [string], matching is case-insensitive and searches for a substring. For example, `"Playwright"` matches
     * `<article><div>Playwright</div></article>`.
     */
    hasText?: string|RegExp;
  }): Locator;

  /**
   * Returns frame's name attribute as specified in the tag.
   *
   * If the name is empty, returns the id attribute instead.
   *
   * **NOTE** This value is calculated once when the frame is created, and will not update if the attribute is changed
   * later.
   */
  name(): string;

  /**
   * Returns the page containing this frame.
   */
  page(): Page;

  /**
   * Parent frame, if any. Detached frames and main frames return `null`.
   */
  parentFrame(): null|Frame;

  /**
   * **NOTE** Use locator-based [locator.press(key[, options])](https://playwright.dev/docs/api/class-locator#locator-press)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * `key` can specify the intended
   * [keyboardEvent.key](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key) value or a single character
   * to generate the text for. A superset of the `key` values can be found
   * [here](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key/Key_Values). Examples of the keys are:
   *
   * `F1` - `F12`, `Digit0`- `Digit9`, `KeyA`- `KeyZ`, `Backquote`, `Minus`, `Equal`, `Backslash`, `Backspace`, `Tab`,
   * `Delete`, `Escape`, `ArrowDown`, `End`, `Enter`, `Home`, `Insert`, `PageDown`, `PageUp`, `ArrowRight`, `ArrowUp`,
   * etc.
   *
   * Following modification shortcuts are also supported: `Shift`, `Control`, `Alt`, `Meta`, `ShiftLeft`.
   *
   * Holding down `Shift` will type the text that corresponds to the `key` in the upper case.
   *
   * If `key` is a single character, it is case-sensitive, so the values `a` and `A` will generate different respective
   * texts.
   *
   * Shortcuts such as `key: "Control+o"` or `key: "Control+Shift+T"` are supported as well. When specified with the
   * modifier, modifier is pressed and being held while the subsequent key is being pressed.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param key Name of the key to press or a character to generate, such as `ArrowLeft` or `a`.
   * @param options
   */
  press(selector: string, key: string, options?: {
    /**
     * Time to wait between `keydown` and `keyup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * **NOTE** Use locator-based
   * [locator.selectOption(values[, options])](https://playwright.dev/docs/api/class-locator#locator-select-option)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method waits for an element matching `selector`, waits for [actionability](https://playwright.dev/docs/actionability) checks, waits
   * until all specified options are present in the `<select>` element and selects these options.
   *
   * If the target element is not a `<select>` element, this method throws an error. However, if the element is inside
   * the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), the control will be used
   * instead.
   *
   * Returns the array of option values that have been successfully selected.
   *
   * Triggers a `change` and `input` event once all the provided options have been selected.
   *
   * **Usage**
   *
   * ```js
   * // single selection matching the value
   * frame.selectOption('select#colors', 'blue');
   *
   * // single selection matching both the value and the label
   * frame.selectOption('select#colors', { label: 'Blue' });
   *
   * // multiple selection
   * frame.selectOption('select#colors', 'red', 'green', 'blue');
   * ```
   *
   * @param selector A selector to query for.
   * @param values Options to select. If the `<select>` has the `multiple` attribute, all matching options are selected, otherwise
   * only the first option matching one of the passed options is selected. String values are matching both values and
   * labels. Option is considered matching if all specified properties match.
   * @param options
   */
  selectOption(selector: string, values: null|string|ElementHandle|Array<string>|{
    /**
     * Matches by `option.value`. Optional.
     */
    value?: string;

    /**
     * Matches by `option.label`. Optional.
     */
    label?: string;

    /**
     * Matches by the index. Optional.
     */
    index?: number;
  }|Array<ElementHandle>|Array<{
    /**
     * Matches by `option.value`. Optional.
     */
    value?: string;

    /**
     * Matches by `option.label`. Optional.
     */
    label?: string;

    /**
     * Matches by the index. Optional.
     */
    index?: number;
  }>, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<Array<string>>;

  /**
   * **NOTE** Use locator-based
   * [locator.setChecked(checked[, options])](https://playwright.dev/docs/api/class-locator#locator-set-checked)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method checks or unchecks an element matching `selector` by performing the following steps:
   * 1. Find an element matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Ensure that matched element is a checkbox or a radio input. If not, this method throws.
   * 1. If the element already has the right checked state, this method returns immediately.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the matched element, unless `force` option is set. If
   *    the element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to click in the center of the
   *    element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now checked or unchecked. If not, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param checked Whether to check or uncheck the checkbox.
   * @param options
   */
  setChecked(selector: string, checked: boolean, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * @param html HTML markup to assign to the page.
   * @param options
   */
  setContent(html: string, options?: {
    /**
     * Maximum operation time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via
     * `navigationTimeout` option in the config, or by using the
     * [browserContext.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-navigation-timeout),
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout),
     * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500`
     *   ms.
     * - `'commit'` - consider operation to be finished when network response is received and the document started
     *   loading.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle"|"commit";
  }): Promise<void>;

  /**
   * **NOTE** Use locator-based
   * [locator.setInputFiles(files[, options])](https://playwright.dev/docs/api/class-locator#locator-set-input-files)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Sets the value of the file input to these file paths or files. If some of the `filePaths` are relative paths, then
   * they are resolved relative to the current working directory. For empty array, clears the selected files.
   *
   * This method expects `selector` to point to an
   * [input element](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input). However, if the element is inside
   * the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), targets the control instead.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param files
   * @param options
   */
  setInputFiles(selector: string, files: string|Array<string>|{
    /**
     * File name
     */
    name: string;

    /**
     * File type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }|Array<{
    /**
     * File name
     */
    name: string;

    /**
     * File type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }>, options?: {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * **NOTE** Use locator-based [locator.tap([options])](https://playwright.dev/docs/api/class-locator#locator-tap) instead. Read
   * more about [locators](https://playwright.dev/docs/locators).
   *
   * This method taps an element matching `selector` by performing the following steps:
   * 1. Find an element matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the matched element, unless `force` option is set. If
   *    the element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.touchscreen](https://playwright.dev/docs/api/class-page#page-touchscreen) to tap the center of the
   *    element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   *
   * **NOTE** `frame.tap()` requires that the `hasTouch` option of the browser context be set to true.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  tap(selector: string, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores
     * current modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * **NOTE** Use locator-based
   * [locator.textContent([options])](https://playwright.dev/docs/api/class-locator#locator-text-content) instead. Read
   * more about [locators](https://playwright.dev/docs/locators).
   *
   * Returns `element.textContent`.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  textContent(selector: string, options?: {
    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<null|string>;

  /**
   * Returns the page title.
   */
  title(): Promise<string>;

  /**
   * **NOTE** Use locator-based [locator.type(text[, options])](https://playwright.dev/docs/api/class-locator#locator-type)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * Sends a `keydown`, `keypress`/`input`, and `keyup` event for each character in the text. `frame.type` can be used
   * to send fine-grained keyboard events. To fill values in form fields, use
   * [frame.fill(selector, value[, options])](https://playwright.dev/docs/api/class-frame#frame-fill).
   *
   * To press a special key, like `Control` or `ArrowDown`, use
   * [keyboard.press(key[, options])](https://playwright.dev/docs/api/class-keyboard#keyboard-press).
   *
   * **Usage**
   *
   * ```js
   * await frame.type('#mytextarea', 'Hello'); // Types instantly
   * await frame.type('#mytextarea', 'World', {delay: 100}); // Types slower, like a user
   * ```
   *
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param text A text to type into a focused element.
   * @param options
   */
  type(selector: string, text: string, options?: {
    /**
     * Time to wait between key presses in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * **NOTE** Use locator-based [locator.uncheck([options])](https://playwright.dev/docs/api/class-locator#locator-uncheck)
   * instead. Read more about [locators](https://playwright.dev/docs/locators).
   *
   * This method checks an element matching `selector` by performing the following steps:
   * 1. Find an element matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Ensure that matched element is a checkbox or a radio input. If not, this method throws. If the element is
   *    already unchecked, this method returns immediately.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the matched element, unless `force` option is set. If
   *    the element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to click in the center of the
   *    element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now unchecked. If not, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param selector A selector to search for an element. If there are multiple elements satisfying the selector, the first will be
   * used.
   * @param options
   */
  uncheck(selector: string, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
     * element, the call throws an exception.
     */
    strict?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * Returns frame's url.
   */
  url(): string;

  /**
   * Waits for the required load state to be reached.
   *
   * This returns when the frame reaches a required load state, `load` by default. The navigation must have been
   * committed when this method is called. If current document has already reached the required state, resolves
   * immediately.
   *
   * **Usage**
   *
   * ```js
   * await frame.click('button'); // Click triggers navigation.
   * await frame.waitForLoadState(); // Waits for 'load' state by default.
   * ```
   *
   * @param state Optional load state to wait for, defaults to `load`. If the state has been already reached while loading current
   * document, the method resolves immediately. Can be one of:
   * - `'load'` - wait for the `load` event to be fired.
   * - `'domcontentloaded'` - wait for the `DOMContentLoaded` event to be fired.
   * - `'networkidle'` - wait until there are no network connections for at least `500` ms.
   * @param options
   */
  waitForLoadState(state?: "load"|"domcontentloaded"|"networkidle", options?: {
    /**
     * Maximum operation time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via
     * `navigationTimeout` option in the config, or by using the
     * [browserContext.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-navigation-timeout),
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout),
     * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Waits for the frame navigation and returns the main resource response. In case of multiple redirects, the
   * navigation will resolve with the response of the last redirect. In case of navigation to a different anchor or
   * navigation due to History API usage, the navigation will resolve with `null`.
   *
   * **Usage**
   *
   * This method waits for the frame to navigate to a new URL. It is useful for when you run code which will indirectly
   * cause the frame to navigate. Consider this example:
   *
   * ```js
   * // Start waiting for navigation before clicking. Note no await.
   * const navigationPromise = page.waitForNavigation();
   * await page.getByText('Navigate after timeout').click();
   * await navigationPromise;
   * ```
   *
   * **NOTE** Usage of the [History API](https://developer.mozilla.org/en-US/docs/Web/API/History_API) to change the URL
   * is considered a navigation.
   * @deprecated This method is inherently racy, please use
   * [frame.waitForURL(url[, options])](https://playwright.dev/docs/api/class-frame#frame-wait-for-url) instead.
   * @param options
   */
  waitForNavigation(options?: {
    /**
     * Maximum operation time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via
     * `navigationTimeout` option in the config, or by using the
     * [browserContext.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-navigation-timeout),
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout),
     * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * A glob pattern, regex pattern or predicate receiving [URL] to match while waiting for the navigation. Note that if
     * the parameter is a string without wildcard characters, the method will wait for navigation to URL that is exactly
     * equal to the string.
     */
    url?: string|RegExp|((url: URL) => boolean);

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500`
     *   ms.
     * - `'commit'` - consider operation to be finished when network response is received and the document started
     *   loading.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle"|"commit";
  }): Promise<null|Response>;

  /**
   * Waits for the given `timeout` in milliseconds.
   *
   * Note that `frame.waitForTimeout()` should only be used for debugging. Tests using the timer in production are going
   * to be flaky. Use signals such as network events, selectors becoming visible and others instead.
   * @param timeout A timeout to wait for
   */
  waitForTimeout(timeout: number): Promise<void>;

  /**
   * Waits for the frame to navigate to the given URL.
   *
   * **Usage**
   *
   * ```js
   * await frame.click('a.delayed-navigation'); // Clicking the link will indirectly cause a navigation
   * await frame.waitForURL('**\/target.html');
   * ```
   *
   * @param url A glob pattern, regex pattern or predicate receiving [URL] to match while waiting for the navigation. Note that if
   * the parameter is a string without wildcard characters, the method will wait for navigation to URL that is exactly
   * equal to the string.
   * @param options
   */
  waitForURL(url: string|RegExp|((url: URL) => boolean), options?: {
    /**
     * Maximum operation time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via
     * `navigationTimeout` option in the config, or by using the
     * [browserContext.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-navigation-timeout),
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout),
     * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500`
     *   ms.
     * - `'commit'` - consider operation to be finished when network response is received and the document started
     *   loading.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle"|"commit";
  }): Promise<void>;
}

/**
 * - extends: [EventEmitter]
 *
 * BrowserContexts provide a way to operate multiple independent browser sessions.
 *
 * If a page opens another page, e.g. with a `window.open` call, the popup will belong to the parent page's browser
 * context.
 *
 * Playwright allows creating "incognito" browser contexts with
 * [browser.newContext([options])](https://playwright.dev/docs/api/class-browser#browser-new-context) method.
 * "Incognito" browser contexts don't write any browsing data to disk.
 *
 * ```js
 * // Create a new incognito browser context
 * const context = await browser.newContext();
 * // Create a new page inside context.
 * const page = await context.newPage();
 * await page.goto('https://example.com');
 * // Dispose context once it's no longer needed.
 * await context.close();
 * ```
 *
 */
export interface BrowserContext {
  /**
   * The method adds a function called `name` on the `window` object of every frame in every page in the context. When
   * called, the function executes `callback` and returns a [Promise] which resolves to the return value of `callback`.
   * If the `callback` returns a [Promise], it will be awaited.
   *
   * The first argument of the `callback` function contains information about the caller: `{ browserContext:
   * BrowserContext, page: Page, frame: Frame }`.
   *
   * See [page.exposeBinding(name, callback[, options])](https://playwright.dev/docs/api/class-page#page-expose-binding)
   * for page-only version.
   *
   * **Usage**
   *
   * An example of exposing page URL to all frames in all pages in the context:
   *
   * ```js
   * const { webkit } = require('playwright');  // Or 'chromium' or 'firefox'.
   *
   * (async () => {
   *   const browser = await webkit.launch({ headless: false });
   *   const context = await browser.newContext();
   *   await context.exposeBinding('pageURL', ({ page }) => page.url());
   *   const page = await context.newPage();
   *   await page.setContent(`
   *     <script>
   *       async function onClick() {
   *         document.querySelector('div').textContent = await window.pageURL();
   *       }
   *     </script>
   *     <button onclick="onClick()">Click me</button>
   *     <div></div>
   *   `);
   *   await page.getByRole('button').click();
   * })();
   * ```
   *
   * An example of passing an element handle:
   *
   * ```js
   * await context.exposeBinding('clicked', async (source, element) => {
   *   console.log(await element.textContent());
   * }, { handle: true });
   * await page.setContent(`
   *   <script>
   *     document.addEventListener('click', event => window.clicked(event.target));
   *   </script>
   *   <div>Click me</div>
   *   <div>Or click me</div>
   * `);
   * ```
   *
   * @param name Name of the function on the window object.
   * @param callback Callback function that will be called in the Playwright's context.
   * @param options
   */
  exposeBinding(name: string, playwrightBinding: (source: BindingSource, arg: JSHandle) => any, options: { handle: true }): Promise<void>;
  /**
   * The method adds a function called `name` on the `window` object of every frame in every page in the context. When
   * called, the function executes `callback` and returns a [Promise] which resolves to the return value of `callback`.
   * If the `callback` returns a [Promise], it will be awaited.
   *
   * The first argument of the `callback` function contains information about the caller: `{ browserContext:
   * BrowserContext, page: Page, frame: Frame }`.
   *
   * See [page.exposeBinding(name, callback[, options])](https://playwright.dev/docs/api/class-page#page-expose-binding)
   * for page-only version.
   *
   * **Usage**
   *
   * An example of exposing page URL to all frames in all pages in the context:
   *
   * ```js
   * const { webkit } = require('playwright');  // Or 'chromium' or 'firefox'.
   *
   * (async () => {
   *   const browser = await webkit.launch({ headless: false });
   *   const context = await browser.newContext();
   *   await context.exposeBinding('pageURL', ({ page }) => page.url());
   *   const page = await context.newPage();
   *   await page.setContent(`
   *     <script>
   *       async function onClick() {
   *         document.querySelector('div').textContent = await window.pageURL();
   *       }
   *     </script>
   *     <button onclick="onClick()">Click me</button>
   *     <div></div>
   *   `);
   *   await page.getByRole('button').click();
   * })();
   * ```
   *
   * An example of passing an element handle:
   *
   * ```js
   * await context.exposeBinding('clicked', async (source, element) => {
   *   console.log(await element.textContent());
   * }, { handle: true });
   * await page.setContent(`
   *   <script>
   *     document.addEventListener('click', event => window.clicked(event.target));
   *   </script>
   *   <div>Click me</div>
   *   <div>Or click me</div>
   * `);
   * ```
   *
   * @param name Name of the function on the window object.
   * @param callback Callback function that will be called in the Playwright's context.
   * @param options
   */
  exposeBinding(name: string, playwrightBinding: (source: BindingSource, ...args: any[]) => any, options?: { handle?: boolean }): Promise<void>;

  /**
   * Adds a script which would be evaluated in one of the following scenarios:
   * - Whenever a page is created in the browser context or is navigated.
   * - Whenever a child frame is attached or navigated in any page in the browser context. In this case, the script is
   *   evaluated in the context of the newly attached frame.
   *
   * The script is evaluated after the document was created but before any of its scripts were run. This is useful to
   * amend the JavaScript environment, e.g. to seed `Math.random`.
   *
   * **Usage**
   *
   * An example of overriding `Math.random` before the page loads:
   *
   * ```js
   * // preload.js
   * Math.random = () => 42;
   * ```
   *
   * ```js
   * // In your playwright script, assuming the preload.js file is in same directory.
   * await browserContext.addInitScript({
   *   path: 'preload.js'
   * });
   * ```
   *
   * **NOTE** The order of evaluation of multiple scripts installed via
   * [browserContext.addInitScript(script[, arg])](https://playwright.dev/docs/api/class-browsercontext#browser-context-add-init-script)
   * and [page.addInitScript(script[, arg])](https://playwright.dev/docs/api/class-page#page-add-init-script) is not
   * defined.
   * @param script Script to be evaluated in all pages in the browser context.
   * @param arg Optional argument to pass to `script` (only supported when passing a function).
   */
  addInitScript<Arg>(script: PageFunction<Arg, any> | { path?: string, content?: string }, arg?: Arg): Promise<void>;
  /**
   * **NOTE** Only works with Chromium browser's persistent context.
   *
   * Emitted when new background page is created in the context.
   *
   * ```js
   * const backgroundPage = await context.waitForEvent('backgroundpage');
   * ```
   *
   */
  on(event: 'backgroundpage', listener: (page: Page) => void): this;

  /**
   * Emitted when Browser context gets closed. This might happen because of one of the following:
   * - Browser context is closed.
   * - Browser application is closed or crashed.
   * - The [browser.close()](https://playwright.dev/docs/api/class-browser#browser-close) method was called.
   */
  on(event: 'close', listener: (browserContext: BrowserContext) => void): this;

  /**
   * The event is emitted when a new Page is created in the BrowserContext. The page may still be loading. The event
   * will also fire for popup pages. See also
   * [page.on('popup')](https://playwright.dev/docs/api/class-page#page-event-popup) to receive events about popups
   * relevant to a specific page.
   *
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to
   * "http://example.com" is done and its response has started loading in the popup.
   *
   * ```js
   * const newPagePromise = context.waitForEvent('page');
   * await page.getByText('open new page').click();
   * const newPage = await newPagePromise;
   * console.log(await newPage.evaluate('location.href'));
   * ```
   *
   * **NOTE** Use
   * [page.waitForLoadState([state, options])](https://playwright.dev/docs/api/class-page#page-wait-for-load-state) to
   * wait until the page gets to a particular state (you should not need it in most cases).
   */
  on(event: 'page', listener: (page: Page) => void): this;

  /**
   * Emitted when a request is issued from any pages created through this context. The [request] object is read-only. To
   * only listen for requests from a particular page, use
   * [page.on('request')](https://playwright.dev/docs/api/class-page#page-event-request).
   *
   * In order to intercept and mutate requests, see
   * [browserContext.route(url, handler[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-route)
   * or [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route).
   */
  on(event: 'request', listener: (request: Request) => void): this;

  /**
   * Emitted when a request fails, for example by timing out. To only listen for failed requests from a particular page,
   * use [page.on('requestfailed')](https://playwright.dev/docs/api/class-page#page-event-request-failed).
   *
   * **NOTE** HTTP Error responses, such as 404 or 503, are still successful responses from HTTP standpoint, so request
   * will complete with
   * [browserContext.on('requestfinished')](https://playwright.dev/docs/api/class-browsercontext#browser-context-event-request-finished)
   * event and not with
   * [browserContext.on('requestfailed')](https://playwright.dev/docs/api/class-browsercontext#browser-context-event-request-failed).
   */
  on(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Emitted when a request finishes successfully after downloading the response body. For a successful response, the
   * sequence of events is `request`, `response` and `requestfinished`. To listen for successful requests from a
   * particular page, use
   * [page.on('requestfinished')](https://playwright.dev/docs/api/class-page#page-event-request-finished).
   */
  on(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Emitted when [response] status and headers are received for a request. For a successful response, the sequence of
   * events is `request`, `response` and `requestfinished`. To listen for response events from a particular page, use
   * [page.on('response')](https://playwright.dev/docs/api/class-page#page-event-response).
   */
  on(event: 'response', listener: (response: Response) => void): this;

  /**
   * **NOTE** Service workers are only supported on Chromium-based browsers.
   *
   * Emitted when new service worker is created in the context.
   */
  on(event: 'serviceworker', listener: (worker: Worker) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'backgroundpage', listener: (page: Page) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'close', listener: (browserContext: BrowserContext) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'page', listener: (page: Page) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'request', listener: (request: Request) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'response', listener: (response: Response) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'serviceworker', listener: (worker: Worker) => void): this;

  /**
   * **NOTE** Only works with Chromium browser's persistent context.
   *
   * Emitted when new background page is created in the context.
   *
   * ```js
   * const backgroundPage = await context.waitForEvent('backgroundpage');
   * ```
   *
   */
  addListener(event: 'backgroundpage', listener: (page: Page) => void): this;

  /**
   * Emitted when Browser context gets closed. This might happen because of one of the following:
   * - Browser context is closed.
   * - Browser application is closed or crashed.
   * - The [browser.close()](https://playwright.dev/docs/api/class-browser#browser-close) method was called.
   */
  addListener(event: 'close', listener: (browserContext: BrowserContext) => void): this;

  /**
   * The event is emitted when a new Page is created in the BrowserContext. The page may still be loading. The event
   * will also fire for popup pages. See also
   * [page.on('popup')](https://playwright.dev/docs/api/class-page#page-event-popup) to receive events about popups
   * relevant to a specific page.
   *
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to
   * "http://example.com" is done and its response has started loading in the popup.
   *
   * ```js
   * const newPagePromise = context.waitForEvent('page');
   * await page.getByText('open new page').click();
   * const newPage = await newPagePromise;
   * console.log(await newPage.evaluate('location.href'));
   * ```
   *
   * **NOTE** Use
   * [page.waitForLoadState([state, options])](https://playwright.dev/docs/api/class-page#page-wait-for-load-state) to
   * wait until the page gets to a particular state (you should not need it in most cases).
   */
  addListener(event: 'page', listener: (page: Page) => void): this;

  /**
   * Emitted when a request is issued from any pages created through this context. The [request] object is read-only. To
   * only listen for requests from a particular page, use
   * [page.on('request')](https://playwright.dev/docs/api/class-page#page-event-request).
   *
   * In order to intercept and mutate requests, see
   * [browserContext.route(url, handler[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-route)
   * or [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route).
   */
  addListener(event: 'request', listener: (request: Request) => void): this;

  /**
   * Emitted when a request fails, for example by timing out. To only listen for failed requests from a particular page,
   * use [page.on('requestfailed')](https://playwright.dev/docs/api/class-page#page-event-request-failed).
   *
   * **NOTE** HTTP Error responses, such as 404 or 503, are still successful responses from HTTP standpoint, so request
   * will complete with
   * [browserContext.on('requestfinished')](https://playwright.dev/docs/api/class-browsercontext#browser-context-event-request-finished)
   * event and not with
   * [browserContext.on('requestfailed')](https://playwright.dev/docs/api/class-browsercontext#browser-context-event-request-failed).
   */
  addListener(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Emitted when a request finishes successfully after downloading the response body. For a successful response, the
   * sequence of events is `request`, `response` and `requestfinished`. To listen for successful requests from a
   * particular page, use
   * [page.on('requestfinished')](https://playwright.dev/docs/api/class-page#page-event-request-finished).
   */
  addListener(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Emitted when [response] status and headers are received for a request. For a successful response, the sequence of
   * events is `request`, `response` and `requestfinished`. To listen for response events from a particular page, use
   * [page.on('response')](https://playwright.dev/docs/api/class-page#page-event-response).
   */
  addListener(event: 'response', listener: (response: Response) => void): this;

  /**
   * **NOTE** Service workers are only supported on Chromium-based browsers.
   *
   * Emitted when new service worker is created in the context.
   */
  addListener(event: 'serviceworker', listener: (worker: Worker) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'backgroundpage', listener: (page: Page) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'close', listener: (browserContext: BrowserContext) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'page', listener: (page: Page) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'request', listener: (request: Request) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'response', listener: (response: Response) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'serviceworker', listener: (worker: Worker) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'backgroundpage', listener: (page: Page) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'close', listener: (browserContext: BrowserContext) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'page', listener: (page: Page) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'request', listener: (request: Request) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'response', listener: (response: Response) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'serviceworker', listener: (worker: Worker) => void): this;

  /**
   * **NOTE** Only works with Chromium browser's persistent context.
   *
   * Emitted when new background page is created in the context.
   *
   * ```js
   * const backgroundPage = await context.waitForEvent('backgroundpage');
   * ```
   *
   */
  prependListener(event: 'backgroundpage', listener: (page: Page) => void): this;

  /**
   * Emitted when Browser context gets closed. This might happen because of one of the following:
   * - Browser context is closed.
   * - Browser application is closed or crashed.
   * - The [browser.close()](https://playwright.dev/docs/api/class-browser#browser-close) method was called.
   */
  prependListener(event: 'close', listener: (browserContext: BrowserContext) => void): this;

  /**
   * The event is emitted when a new Page is created in the BrowserContext. The page may still be loading. The event
   * will also fire for popup pages. See also
   * [page.on('popup')](https://playwright.dev/docs/api/class-page#page-event-popup) to receive events about popups
   * relevant to a specific page.
   *
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to
   * "http://example.com" is done and its response has started loading in the popup.
   *
   * ```js
   * const newPagePromise = context.waitForEvent('page');
   * await page.getByText('open new page').click();
   * const newPage = await newPagePromise;
   * console.log(await newPage.evaluate('location.href'));
   * ```
   *
   * **NOTE** Use
   * [page.waitForLoadState([state, options])](https://playwright.dev/docs/api/class-page#page-wait-for-load-state) to
   * wait until the page gets to a particular state (you should not need it in most cases).
   */
  prependListener(event: 'page', listener: (page: Page) => void): this;

  /**
   * Emitted when a request is issued from any pages created through this context. The [request] object is read-only. To
   * only listen for requests from a particular page, use
   * [page.on('request')](https://playwright.dev/docs/api/class-page#page-event-request).
   *
   * In order to intercept and mutate requests, see
   * [browserContext.route(url, handler[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-route)
   * or [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route).
   */
  prependListener(event: 'request', listener: (request: Request) => void): this;

  /**
   * Emitted when a request fails, for example by timing out. To only listen for failed requests from a particular page,
   * use [page.on('requestfailed')](https://playwright.dev/docs/api/class-page#page-event-request-failed).
   *
   * **NOTE** HTTP Error responses, such as 404 or 503, are still successful responses from HTTP standpoint, so request
   * will complete with
   * [browserContext.on('requestfinished')](https://playwright.dev/docs/api/class-browsercontext#browser-context-event-request-finished)
   * event and not with
   * [browserContext.on('requestfailed')](https://playwright.dev/docs/api/class-browsercontext#browser-context-event-request-failed).
   */
  prependListener(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Emitted when a request finishes successfully after downloading the response body. For a successful response, the
   * sequence of events is `request`, `response` and `requestfinished`. To listen for successful requests from a
   * particular page, use
   * [page.on('requestfinished')](https://playwright.dev/docs/api/class-page#page-event-request-finished).
   */
  prependListener(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Emitted when [response] status and headers are received for a request. For a successful response, the sequence of
   * events is `request`, `response` and `requestfinished`. To listen for response events from a particular page, use
   * [page.on('response')](https://playwright.dev/docs/api/class-page#page-event-response).
   */
  prependListener(event: 'response', listener: (response: Response) => void): this;

  /**
   * **NOTE** Service workers are only supported on Chromium-based browsers.
   *
   * Emitted when new service worker is created in the context.
   */
  prependListener(event: 'serviceworker', listener: (worker: Worker) => void): this;

  /**
   * Adds cookies into this browser context. All pages within this context will have these cookies installed. Cookies
   * can be obtained via
   * [browserContext.cookies([urls])](https://playwright.dev/docs/api/class-browsercontext#browser-context-cookies).
   *
   * **Usage**
   *
   * ```js
   * await browserContext.addCookies([cookieObject1, cookieObject2]);
   * ```
   *
   * @param cookies
   */
  addCookies(cookies: Array<{
    name: string;

    value: string;

    /**
     * either url or domain / path are required. Optional.
     */
    url?: string;

    /**
     * either url or domain / path are required Optional.
     */
    domain?: string;

    /**
     * either url or domain / path are required Optional.
     */
    path?: string;

    /**
     * Unix time in seconds. Optional.
     */
    expires?: number;

    /**
     * Optional.
     */
    httpOnly?: boolean;

    /**
     * Optional.
     */
    secure?: boolean;

    /**
     * Optional.
     */
    sameSite?: "Strict"|"Lax"|"None";
  }>): Promise<void>;

  /**
   * **NOTE** Background pages are only supported on Chromium-based browsers.
   *
   * All existing background pages in the context.
   */
  backgroundPages(): Array<Page>;

  /**
   * Returns the browser instance of the context. If it was launched as a persistent context null gets returned.
   */
  browser(): null|Browser;

  /**
   * Clears context cookies.
   */
  clearCookies(): Promise<void>;

  /**
   * Clears all permission overrides for the browser context.
   *
   * **Usage**
   *
   * ```js
   * const context = await browser.newContext();
   * await context.grantPermissions(['clipboard-read']);
   * // do stuff ..
   * context.clearPermissions();
   * ```
   *
   */
  clearPermissions(): Promise<void>;

  /**
   * Closes the browser context. All the pages that belong to the browser context will be closed.
   *
   * **NOTE** The default browser context cannot be closed.
   */
  close(): Promise<void>;

  /**
   * If no URLs are specified, this method returns all cookies. If URLs are specified, only cookies that affect those
   * URLs are returned.
   * @param urls Optional list of URLs.
   */
  cookies(urls?: string|Array<string>): Promise<Array<Cookie>>;

  /**
   * The method adds a function called `name` on the `window` object of every frame in every page in the context. When
   * called, the function executes `callback` and returns a [Promise] which resolves to the return value of `callback`.
   *
   * If the `callback` returns a [Promise], it will be awaited.
   *
   * See [page.exposeFunction(name, callback)](https://playwright.dev/docs/api/class-page#page-expose-function) for
   * page-only version.
   *
   * **Usage**
   *
   * An example of adding a `sha256` function to all pages in the context:
   *
   * ```js
   * const { webkit } = require('playwright');  // Or 'chromium' or 'firefox'.
   * const crypto = require('crypto');
   *
   * (async () => {
   *   const browser = await webkit.launch({ headless: false });
   *   const context = await browser.newContext();
   *   await context.exposeFunction('sha256', text => crypto.createHash('sha256').update(text).digest('hex'));
   *   const page = await context.newPage();
   *   await page.setContent(`
   *     <script>
   *       async function onClick() {
   *         document.querySelector('div').textContent = await window.sha256('PLAYWRIGHT');
   *       }
   *     </script>
   *     <button onclick="onClick()">Click me</button>
   *     <div></div>
   *   `);
   *   await page.getByRole('button').click();
   * })();
   * ```
   *
   * @param name Name of the function on the window object.
   * @param callback Callback function that will be called in the Playwright's context.
   */
  exposeFunction(name: string, callback: Function): Promise<void>;

  /**
   * Grants specified permissions to the browser context. Only grants corresponding permissions to the given origin if
   * specified.
   * @param permissions A permission or an array of permissions to grant. Permissions can be one of the following values:
   * - `'geolocation'`
   * - `'midi'`
   * - `'midi-sysex'` (system-exclusive midi)
   * - `'notifications'`
   * - `'camera'`
   * - `'microphone'`
   * - `'background-sync'`
   * - `'ambient-light-sensor'`
   * - `'accelerometer'`
   * - `'gyroscope'`
   * - `'magnetometer'`
   * - `'accessibility-events'`
   * - `'clipboard-read'`
   * - `'clipboard-write'`
   * - `'payment-handler'`
   * @param options
   */
  grantPermissions(permissions: Array<string>, options?: {
    /**
     * The [origin] to grant permissions to, e.g. "https://example.com".
     */
    origin?: string;
  }): Promise<void>;

  /**
   * **NOTE** CDP sessions are only supported on Chromium-based browsers.
   *
   * Returns the newly created session.
   * @param page Target to create new session for. For backwards-compatibility, this parameter is named `page`, but it can be a
   * `Page` or `Frame` type.
   */
  newCDPSession(page: Page|Frame): Promise<CDPSession>;

  /**
   * Creates a new page in the browser context.
   */
  newPage(): Promise<Page>;

  /**
   * Returns all open pages in the context.
   */
  pages(): Array<Page>;

  /**
   * Routing provides the capability to modify network requests that are made by any page in the browser context. Once
   * route is enabled, every request matching the url pattern will stall unless it's continued, fulfilled or aborted.
   *
   * **NOTE**
   * [browserContext.route(url, handler[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-route)
   * will not intercept requests intercepted by Service Worker. See
   * [this](https://github.com/microsoft/playwright/issues/1090) issue. We recommend disabling Service Workers when
   * using request interception by setting `Browser.newContext.serviceWorkers` to `'block'`.
   *
   * **Usage**
   *
   * An example of a naive handler that aborts all image requests:
   *
   * ```js
   * const context = await browser.newContext();
   * await context.route('**\/*.{png,jpg,jpeg}', route => route.abort());
   * const page = await context.newPage();
   * await page.goto('https://example.com');
   * await browser.close();
   * ```
   *
   * or the same snippet using a regex pattern instead:
   *
   * ```js
   * const context = await browser.newContext();
   * await context.route(/(\.png$)|(\.jpg$)/, route => route.abort());
   * const page = await context.newPage();
   * await page.goto('https://example.com');
   * await browser.close();
   * ```
   *
   * It is possible to examine the request to decide the route action. For example, mocking all requests that contain
   * some post data, and leaving all other requests as is:
   *
   * ```js
   * await context.route('/api/**', route => {
   *   if (route.request().postData().includes('my-string'))
   *     route.fulfill({ body: 'mocked-data' });
   *   else
   *     route.continue();
   * });
   * ```
   *
   * Page routes (set up with
   * [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route)) take precedence over
   * browser context routes when request matches both handlers.
   *
   * To remove a route with its handler you can use
   * [browserContext.unroute(url[, handler])](https://playwright.dev/docs/api/class-browsercontext#browser-context-unroute).
   *
   * **NOTE** Enabling routing disables http cache.
   * @param url A glob pattern, regex pattern or predicate receiving [URL] to match while routing. When a `baseURL` via the context
   * options was provided and the passed URL is a path, it gets merged via the
   * [`new URL()`](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL) constructor.
   * @param handler handler function to route the request.
   * @param options
   */
  route(url: string|RegExp|((url: URL) => boolean), handler: ((route: Route, request: Request) => Promise<any>|any), options?: {
    /**
     * How often a route should be used. By default it will be used every time.
     */
    times?: number;
  }): Promise<void>;

  /**
   * If specified the network requests that are made in the context will be served from the HAR file. Read more about
   * [Replaying from HAR](https://playwright.dev/docs/network#replaying-from-har).
   *
   * Playwright will not serve requests intercepted by Service Worker from the HAR file. See
   * [this](https://github.com/microsoft/playwright/issues/1090) issue. We recommend disabling Service Workers when
   * using request interception by setting `Browser.newContext.serviceWorkers` to `'block'`.
   * @param har Path to a [HAR](http://www.softwareishard.com/blog/har-12-spec) file with prerecorded network data. If `path` is a
   * relative path, then it is resolved relative to the current working directory.
   * @param options
   */
  routeFromHAR(har: string, options?: {
    /**
     * - If set to 'abort' any request not found in the HAR file will be aborted.
     * - If set to 'fallback' falls through to the next route handler in the handler chain.
     *
     * Defaults to abort.
     */
    notFound?: "abort"|"fallback";

    /**
     * If specified, updates the given HAR with the actual network information instead of serving from file. The file is
     * written to disk when
     * [browserContext.close()](https://playwright.dev/docs/api/class-browsercontext#browser-context-close) is called.
     */
    update?: boolean;

    /**
     * Optional setting to control resource content management. If `attach` is specified, resources are persisted as
     * separate files or entries in the ZIP archive. If `embed` is specified, content is stored inline the HAR file.
     */
    updateContent?: "embed"|"attach";

    /**
     * When set to `minimal`, only record information necessary for routing from HAR. This omits sizes, timing, page,
     * cookies, security and other types of HAR information that are not used when replaying from HAR. Defaults to
     * `minimal`.
     */
    updateMode?: "full"|"minimal";

    /**
     * A glob pattern, regular expression or predicate to match the request URL. Only requests with URL matching the
     * pattern will be served from the HAR file. If not specified, all requests are served from the HAR file.
     */
    url?: string|RegExp;
  }): Promise<void>;

  /**
   * **NOTE** Service workers are only supported on Chromium-based browsers.
   *
   * All existing service workers in the context.
   */
  serviceWorkers(): Array<Worker>;

  /**
   * This setting will change the default maximum navigation time for the following methods and related shortcuts:
   * - [page.goBack([options])](https://playwright.dev/docs/api/class-page#page-go-back)
   * - [page.goForward([options])](https://playwright.dev/docs/api/class-page#page-go-forward)
   * - [page.goto(url[, options])](https://playwright.dev/docs/api/class-page#page-goto)
   * - [page.reload([options])](https://playwright.dev/docs/api/class-page#page-reload)
   * - [page.setContent(html[, options])](https://playwright.dev/docs/api/class-page#page-set-content)
   * - [page.waitForNavigation([options])](https://playwright.dev/docs/api/class-page#page-wait-for-navigation)
   *
   * **NOTE**
   * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout)
   * and [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) take
   * priority over
   * [browserContext.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-navigation-timeout).
   * @param timeout Maximum navigation time in milliseconds
   */
  setDefaultNavigationTimeout(timeout: number): void;

  /**
   * This setting will change the default maximum time for all the methods accepting `timeout` option.
   *
   * **NOTE**
   * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout),
   * [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) and
   * [browserContext.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-navigation-timeout)
   * take priority over
   * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout).
   * @param timeout Maximum time in milliseconds
   */
  setDefaultTimeout(timeout: number): void;

  /**
   * The extra HTTP headers will be sent with every request initiated by any page in the context. These headers are
   * merged with page-specific extra HTTP headers set with
   * [page.setExtraHTTPHeaders(headers)](https://playwright.dev/docs/api/class-page#page-set-extra-http-headers). If
   * page overrides a particular header, page-specific header value will be used instead of the browser context header
   * value.
   *
   * **NOTE**
   * [browserContext.setExtraHTTPHeaders(headers)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-extra-http-headers)
   * does not guarantee the order of headers in the outgoing requests.
   * @param headers An object containing additional HTTP headers to be sent with every request. All header values must be strings.
   */
  setExtraHTTPHeaders(headers: { [key: string]: string; }): Promise<void>;

  /**
   * Sets the context's geolocation. Passing `null` or `undefined` emulates position unavailable.
   *
   * **Usage**
   *
   * ```js
   * await browserContext.setGeolocation({latitude: 59.95, longitude: 30.31667});
   * ```
   *
   * **NOTE** Consider using
   * [browserContext.grantPermissions(permissions[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-grant-permissions)
   * to grant permissions for the browser context pages to read its geolocation.
   * @param geolocation
   */
  setGeolocation(geolocation: null|{
    /**
     * Latitude between -90 and 90.
     */
    latitude: number;

    /**
     * Longitude between -180 and 180.
     */
    longitude: number;

    /**
     * Non-negative accuracy value. Defaults to `0`.
     */
    accuracy?: number;
  }): Promise<void>;

  /**
   * @deprecated Browsers may cache credentials after successful authentication. Create a new browser context instead.
   * @param httpCredentials
   */
  setHTTPCredentials(httpCredentials: null|{
    username: string;

    password: string;
  }): Promise<void>;

  /**
   * @param offline Whether to emulate network being offline for the browser context.
   */
  setOffline(offline: boolean): Promise<void>;

  /**
   * Returns storage state for this browser context, contains current cookies and local storage snapshot.
   * @param options
   */
  storageState(options?: {
    /**
     * The file path to save the storage state to. If `path` is a relative path, then it is resolved relative to current
     * working directory. If no path is provided, storage state is still returned, but won't be saved to the disk.
     */
    path?: string;
  }): Promise<{
    cookies: Array<{
      name: string;

      value: string;

      domain: string;

      path: string;

      /**
       * Unix time in seconds.
       */
      expires: number;

      httpOnly: boolean;

      secure: boolean;

      sameSite: "Strict"|"Lax"|"None";
    }>;

    origins: Array<{
      origin: string;

      localStorage: Array<{
        name: string;

        value: string;
      }>;
    }>;
  }>;

  /**
   * Removes a route created with
   * [browserContext.route(url, handler[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-route).
   * When `handler` is not specified, removes all routes for the `url`.
   * @param url A glob pattern, regex pattern or predicate receiving [URL] used to register a routing with
   * [browserContext.route(url, handler[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-route).
   * @param handler Optional handler function used to register a routing with
   * [browserContext.route(url, handler[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-route).
   */
  unroute(url: string|RegExp|((url: URL) => boolean), handler?: ((route: Route, request: Request) => Promise<any>|any)): Promise<void>;

  /**
   * **NOTE** Only works with Chromium browser's persistent context.
   *
   * Emitted when new background page is created in the context.
   *
   * ```js
   * const backgroundPage = await context.waitForEvent('backgroundpage');
   * ```
   *
   */
  waitForEvent(event: 'backgroundpage', optionsOrPredicate?: { predicate?: (page: Page) => boolean | Promise<boolean>, timeout?: number } | ((page: Page) => boolean | Promise<boolean>)): Promise<Page>;

  /**
   * Emitted when Browser context gets closed. This might happen because of one of the following:
   * - Browser context is closed.
   * - Browser application is closed or crashed.
   * - The [browser.close()](https://playwright.dev/docs/api/class-browser#browser-close) method was called.
   */
  waitForEvent(event: 'close', optionsOrPredicate?: { predicate?: (browserContext: BrowserContext) => boolean | Promise<boolean>, timeout?: number } | ((browserContext: BrowserContext) => boolean | Promise<boolean>)): Promise<BrowserContext>;

  /**
   * The event is emitted when a new Page is created in the BrowserContext. The page may still be loading. The event
   * will also fire for popup pages. See also
   * [page.on('popup')](https://playwright.dev/docs/api/class-page#page-event-popup) to receive events about popups
   * relevant to a specific page.
   *
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to
   * "http://example.com" is done and its response has started loading in the popup.
   *
   * ```js
   * const newPagePromise = context.waitForEvent('page');
   * await page.getByText('open new page').click();
   * const newPage = await newPagePromise;
   * console.log(await newPage.evaluate('location.href'));
   * ```
   *
   * **NOTE** Use
   * [page.waitForLoadState([state, options])](https://playwright.dev/docs/api/class-page#page-wait-for-load-state) to
   * wait until the page gets to a particular state (you should not need it in most cases).
   */
  waitForEvent(event: 'page', optionsOrPredicate?: { predicate?: (page: Page) => boolean | Promise<boolean>, timeout?: number } | ((page: Page) => boolean | Promise<boolean>)): Promise<Page>;

  /**
   * Emitted when a request is issued from any pages created through this context. The [request] object is read-only. To
   * only listen for requests from a particular page, use
   * [page.on('request')](https://playwright.dev/docs/api/class-page#page-event-request).
   *
   * In order to intercept and mutate requests, see
   * [browserContext.route(url, handler[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-route)
   * or [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route).
   */
  waitForEvent(event: 'request', optionsOrPredicate?: { predicate?: (request: Request) => boolean | Promise<boolean>, timeout?: number } | ((request: Request) => boolean | Promise<boolean>)): Promise<Request>;

  /**
   * Emitted when a request fails, for example by timing out. To only listen for failed requests from a particular page,
   * use [page.on('requestfailed')](https://playwright.dev/docs/api/class-page#page-event-request-failed).
   *
   * **NOTE** HTTP Error responses, such as 404 or 503, are still successful responses from HTTP standpoint, so request
   * will complete with
   * [browserContext.on('requestfinished')](https://playwright.dev/docs/api/class-browsercontext#browser-context-event-request-finished)
   * event and not with
   * [browserContext.on('requestfailed')](https://playwright.dev/docs/api/class-browsercontext#browser-context-event-request-failed).
   */
  waitForEvent(event: 'requestfailed', optionsOrPredicate?: { predicate?: (request: Request) => boolean | Promise<boolean>, timeout?: number } | ((request: Request) => boolean | Promise<boolean>)): Promise<Request>;

  /**
   * Emitted when a request finishes successfully after downloading the response body. For a successful response, the
   * sequence of events is `request`, `response` and `requestfinished`. To listen for successful requests from a
   * particular page, use
   * [page.on('requestfinished')](https://playwright.dev/docs/api/class-page#page-event-request-finished).
   */
  waitForEvent(event: 'requestfinished', optionsOrPredicate?: { predicate?: (request: Request) => boolean | Promise<boolean>, timeout?: number } | ((request: Request) => boolean | Promise<boolean>)): Promise<Request>;

  /**
   * Emitted when [response] status and headers are received for a request. For a successful response, the sequence of
   * events is `request`, `response` and `requestfinished`. To listen for response events from a particular page, use
   * [page.on('response')](https://playwright.dev/docs/api/class-page#page-event-response).
   */
  waitForEvent(event: 'response', optionsOrPredicate?: { predicate?: (response: Response) => boolean | Promise<boolean>, timeout?: number } | ((response: Response) => boolean | Promise<boolean>)): Promise<Response>;

  /**
   * **NOTE** Service workers are only supported on Chromium-based browsers.
   *
   * Emitted when new service worker is created in the context.
   */
  waitForEvent(event: 'serviceworker', optionsOrPredicate?: { predicate?: (worker: Worker) => boolean | Promise<boolean>, timeout?: number } | ((worker: Worker) => boolean | Promise<boolean>)): Promise<Worker>;


  /**
   * API testing helper associated with this context. Requests made with this API will use context cookies.
   */
  request: APIRequestContext;

  tracing: Tracing;
}

/**
 * The Worker class represents a [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API).
 * `worker` event is emitted on the page object to signal a worker creation. `close` event is emitted on the worker
 * object when the worker is gone.
 *
 * ```js
 * page.on('worker', worker => {
 *   console.log('Worker created: ' + worker.url());
 *   worker.on('close', worker => console.log('Worker destroyed: ' + worker.url()));
 * });
 *
 * console.log('Current workers:');
 * for (const worker of page.workers())
 *   console.log('  ' + worker.url());
 * ```
 *
 */
export interface Worker {
  /**
   * Returns the return value of `pageFunction`.
   *
   * If the function passed to the
   * [worker.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate) returns a
   * [Promise], then
   * [worker.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate) would wait for
   * the promise to resolve and return its value.
   *
   * If the function passed to the
   * [worker.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate) returns a
   * non-[Serializable] value, then
   * [worker.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate) returns
   * `undefined`. Playwright also supports transferring some additional values that are not serializable by `JSON`:
   * `-0`, `NaN`, `Infinity`, `-Infinity`.
   * @param pageFunction Function to be evaluated in the worker context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluate<R, Arg>(pageFunction: PageFunction<Arg, R>, arg: Arg): Promise<R>;
  /**
   * Returns the return value of `pageFunction`.
   *
   * If the function passed to the
   * [worker.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate) returns a
   * [Promise], then
   * [worker.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate) would wait for
   * the promise to resolve and return its value.
   *
   * If the function passed to the
   * [worker.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate) returns a
   * non-[Serializable] value, then
   * [worker.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate) returns
   * `undefined`. Playwright also supports transferring some additional values that are not serializable by `JSON`:
   * `-0`, `NaN`, `Infinity`, `-Infinity`.
   * @param pageFunction Function to be evaluated in the worker context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluate<R>(pageFunction: PageFunction<void, R>, arg?: any): Promise<R>;

  /**
   * Returns the return value of `pageFunction` as a [JSHandle].
   *
   * The only difference between
   * [worker.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate) and
   * [worker.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate-handle)
   * is that
   * [worker.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate-handle)
   * returns [JSHandle].
   *
   * If the function passed to the
   * [worker.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate-handle)
   * returns a [Promise], then
   * [worker.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate-handle)
   * would wait for the promise to resolve and return its value.
   * @param pageFunction Function to be evaluated in the worker context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluateHandle<R, Arg>(pageFunction: PageFunction<Arg, R>, arg: Arg): Promise<SmartHandle<R>>;
  /**
   * Returns the return value of `pageFunction` as a [JSHandle].
   *
   * The only difference between
   * [worker.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate) and
   * [worker.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate-handle)
   * is that
   * [worker.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate-handle)
   * returns [JSHandle].
   *
   * If the function passed to the
   * [worker.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate-handle)
   * returns a [Promise], then
   * [worker.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-worker#worker-evaluate-handle)
   * would wait for the promise to resolve and return its value.
   * @param pageFunction Function to be evaluated in the worker context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluateHandle<R>(pageFunction: PageFunction<void, R>, arg?: any): Promise<SmartHandle<R>>;
  /**
   * Emitted when this dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is
   * terminated.
   */
  on(event: 'close', listener: (worker: Worker) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'close', listener: (worker: Worker) => void): this;

  /**
   * Emitted when this dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is
   * terminated.
   */
  addListener(event: 'close', listener: (worker: Worker) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'close', listener: (worker: Worker) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'close', listener: (worker: Worker) => void): this;

  /**
   * Emitted when this dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is
   * terminated.
   */
  prependListener(event: 'close', listener: (worker: Worker) => void): this;

  url(): string;
}

/**
 * JSHandle represents an in-page JavaScript object. JSHandles can be created with the
 * [page.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate-handle) method.
 *
 * ```js
 * const windowHandle = await page.evaluateHandle(() => window);
 * // ...
 * ```
 *
 * JSHandle prevents the referenced JavaScript object being garbage collected unless the handle is exposed with
 * [jsHandle.dispose()](https://playwright.dev/docs/api/class-jshandle#js-handle-dispose). JSHandles are auto-disposed
 * when their origin frame gets navigated or the parent context gets destroyed.
 *
 * JSHandle instances can be used as an argument in
 * [page.$eval(selector, pageFunction[, arg, options])](https://playwright.dev/docs/api/class-page#page-eval-on-selector),
 * [page.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate) and
 * [page.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate-handle)
 * methods.
 */
export interface JSHandle<T = any> {
  /**
   * Returns the return value of `pageFunction`.
   *
   * This method passes this handle as the first argument to `pageFunction`.
   *
   * If `pageFunction` returns a [Promise], then `handle.evaluate` would wait for the promise to resolve and return its
   * value.
   *
   * **Usage**
   *
   * ```js
   * const tweetHandle = await page.$('.tweet .retweets');
   * expect(await tweetHandle.evaluate(node => node.innerText)).toBe('10 retweets');
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluate<R, Arg, O extends T = T>(pageFunction: PageFunctionOn<O, Arg, R>, arg: Arg): Promise<R>;
  /**
   * Returns the return value of `pageFunction`.
   *
   * This method passes this handle as the first argument to `pageFunction`.
   *
   * If `pageFunction` returns a [Promise], then `handle.evaluate` would wait for the promise to resolve and return its
   * value.
   *
   * **Usage**
   *
   * ```js
   * const tweetHandle = await page.$('.tweet .retweets');
   * expect(await tweetHandle.evaluate(node => node.innerText)).toBe('10 retweets');
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluate<R, O extends T = T>(pageFunction: PageFunctionOn<O, void, R>, arg?: any): Promise<R>;

  /**
   * Returns the return value of `pageFunction` as a [JSHandle].
   *
   * This method passes this handle as the first argument to `pageFunction`.
   *
   * The only difference between `jsHandle.evaluate` and `jsHandle.evaluateHandle` is that `jsHandle.evaluateHandle`
   * returns [JSHandle].
   *
   * If the function passed to the `jsHandle.evaluateHandle` returns a [Promise], then `jsHandle.evaluateHandle` would
   * wait for the promise to resolve and return its value.
   *
   * See [page.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate-handle) for
   * more details.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluateHandle<R, Arg, O extends T = T>(pageFunction: PageFunctionOn<O, Arg, R>, arg: Arg): Promise<SmartHandle<R>>;
  /**
   * Returns the return value of `pageFunction` as a [JSHandle].
   *
   * This method passes this handle as the first argument to `pageFunction`.
   *
   * The only difference between `jsHandle.evaluate` and `jsHandle.evaluateHandle` is that `jsHandle.evaluateHandle`
   * returns [JSHandle].
   *
   * If the function passed to the `jsHandle.evaluateHandle` returns a [Promise], then `jsHandle.evaluateHandle` would
   * wait for the promise to resolve and return its value.
   *
   * See [page.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate-handle) for
   * more details.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluateHandle<R, O extends T = T>(pageFunction: PageFunctionOn<O, void, R>, arg?: any): Promise<SmartHandle<R>>;

  /**
   * Returns a JSON representation of the object. If the object has a `toJSON` function, it **will not be called**.
   *
   * **NOTE** The method will return an empty JSON object if the referenced object is not stringifiable. It will throw
   * an error if the object has circular references.
   */
  jsonValue(): Promise<T>;
  /**
   * Returns either `null` or the object handle itself, if the object handle is an instance of [ElementHandle].
   */
  asElement(): T extends Node ? ElementHandle<T> : null;
  /**
   * The `jsHandle.dispose` method stops referencing the element handle.
   */
  dispose(): Promise<void>;

  /**
   * The method returns a map with **own property names** as keys and JSHandle instances for the property values.
   *
   * **Usage**
   *
   * ```js
   * const handle = await page.evaluateHandle(() => ({window, document}));
   * const properties = await handle.getProperties();
   * const windowHandle = properties.get('window');
   * const documentHandle = properties.get('document');
   * await handle.dispose();
   * ```
   *
   */
  getProperties(): Promise<Map<string, JSHandle>>;

  /**
   * Fetches a single property from the referenced object.
   * @param propertyName property to get
   */
  getProperty(propertyName: string): Promise<JSHandle>;
}

/**
 * - extends: [JSHandle]
 *
 * ElementHandle represents an in-page DOM element. ElementHandles can be created with the
 * [page.$(selector[, options])](https://playwright.dev/docs/api/class-page#page-query-selector) method.
 *
 * **NOTE** The use of ElementHandle is discouraged, use [Locator] objects and web-first assertions instead.
 *
 * ```js
 * const hrefElement = await page.$('a');
 * await hrefElement.click();
 * ```
 *
 * ElementHandle prevents DOM element from garbage collection unless the handle is disposed with
 * [jsHandle.dispose()](https://playwright.dev/docs/api/class-jshandle#js-handle-dispose). ElementHandles are
 * auto-disposed when their origin frame gets navigated.
 *
 * ElementHandle instances can be used as an argument in
 * [page.$eval(selector, pageFunction[, arg, options])](https://playwright.dev/docs/api/class-page#page-eval-on-selector)
 * and [page.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate) methods.
 *
 * The difference between the [Locator] and ElementHandle is that the ElementHandle points to a particular element,
 * while [Locator] captures the logic of how to retrieve an element.
 *
 * In the example below, handle points to a particular DOM element on page. If that element changes text or is used by
 * React to render an entirely different component, handle is still pointing to that very DOM element. This can lead
 * to unexpected behaviors.
 *
 * ```js
 * const handle = await page.$('text=Submit');
 * // ...
 * await handle.hover();
 * await handle.click();
 * ```
 *
 * With the locator, every time the `element` is used, up-to-date DOM element is located in the page using the
 * selector. So in the snippet below, underlying DOM element is going to be located twice.
 *
 * ```js
 * const locator = page.getByText('Submit');
 * // ...
 * await locator.hover();
 * await locator.click();
 * ```
 *
 */
export interface ElementHandle<T=Node> extends JSHandle<T> {
  /**
   * The method finds an element matching the specified selector in the `ElementHandle`'s subtree. If no elements match
   * the selector, returns `null`.
   * @param selector A selector to query for.
   */
  $<K extends keyof HTMLElementTagNameMap>(selector: K, options?: { strict: boolean }): Promise<ElementHandleForTag<K> | null>;
  /**
   * The method finds an element matching the specified selector in the `ElementHandle`'s subtree. If no elements match
   * the selector, returns `null`.
   * @param selector A selector to query for.
   */
  $(selector: string, options?: { strict: boolean }): Promise<ElementHandle<SVGElement | HTMLElement> | null>;

  /**
   * The method finds all elements matching the specified selector in the `ElementHandle`s subtree. If no elements match
   * the selector, returns empty array.
   * @param selector A selector to query for.
   */
  $$<K extends keyof HTMLElementTagNameMap>(selector: K): Promise<ElementHandleForTag<K>[]>;
  /**
   * The method finds all elements matching the specified selector in the `ElementHandle`s subtree. If no elements match
   * the selector, returns empty array.
   * @param selector A selector to query for.
   */
  $$(selector: string): Promise<ElementHandle<SVGElement | HTMLElement>[]>;

  /**
   * Returns the return value of `pageFunction`.
   *
   * The method finds an element matching the specified selector in the `ElementHandle`s subtree and passes it as a
   * first argument to `pageFunction`. If no elements match the selector, the method throws an error.
   *
   * If `pageFunction` returns a [Promise], then
   * [elementHandle.$eval(selector, pageFunction[, arg])](https://playwright.dev/docs/api/class-elementhandle#element-handle-eval-on-selector)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const tweetHandle = await page.$('.tweet');
   * expect(await tweetHandle.$eval('.like', node => node.innerText)).toBe('100');
   * expect(await tweetHandle.$eval('.retweets', node => node.innerText)).toBe('10');
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  $eval<K extends keyof HTMLElementTagNameMap, R, Arg>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K], Arg, R>, arg: Arg): Promise<R>;
  /**
   * Returns the return value of `pageFunction`.
   *
   * The method finds an element matching the specified selector in the `ElementHandle`s subtree and passes it as a
   * first argument to `pageFunction`. If no elements match the selector, the method throws an error.
   *
   * If `pageFunction` returns a [Promise], then
   * [elementHandle.$eval(selector, pageFunction[, arg])](https://playwright.dev/docs/api/class-elementhandle#element-handle-eval-on-selector)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const tweetHandle = await page.$('.tweet');
   * expect(await tweetHandle.$eval('.like', node => node.innerText)).toBe('100');
   * expect(await tweetHandle.$eval('.retweets', node => node.innerText)).toBe('10');
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  $eval<R, Arg, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E, Arg, R>, arg: Arg): Promise<R>;
  /**
   * Returns the return value of `pageFunction`.
   *
   * The method finds an element matching the specified selector in the `ElementHandle`s subtree and passes it as a
   * first argument to `pageFunction`. If no elements match the selector, the method throws an error.
   *
   * If `pageFunction` returns a [Promise], then
   * [elementHandle.$eval(selector, pageFunction[, arg])](https://playwright.dev/docs/api/class-elementhandle#element-handle-eval-on-selector)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const tweetHandle = await page.$('.tweet');
   * expect(await tweetHandle.$eval('.like', node => node.innerText)).toBe('100');
   * expect(await tweetHandle.$eval('.retweets', node => node.innerText)).toBe('10');
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  $eval<K extends keyof HTMLElementTagNameMap, R>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K], void, R>, arg?: any): Promise<R>;
  /**
   * Returns the return value of `pageFunction`.
   *
   * The method finds an element matching the specified selector in the `ElementHandle`s subtree and passes it as a
   * first argument to `pageFunction`. If no elements match the selector, the method throws an error.
   *
   * If `pageFunction` returns a [Promise], then
   * [elementHandle.$eval(selector, pageFunction[, arg])](https://playwright.dev/docs/api/class-elementhandle#element-handle-eval-on-selector)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```js
   * const tweetHandle = await page.$('.tweet');
   * expect(await tweetHandle.$eval('.like', node => node.innerText)).toBe('100');
   * expect(await tweetHandle.$eval('.retweets', node => node.innerText)).toBe('10');
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  $eval<R, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E, void, R>, arg?: any): Promise<R>;

  /**
   * Returns the return value of `pageFunction`.
   *
   * The method finds all elements matching the specified selector in the `ElementHandle`'s subtree and passes an array
   * of matched elements as a first argument to `pageFunction`.
   *
   * If `pageFunction` returns a [Promise], then
   * [elementHandle.$$eval(selector, pageFunction[, arg])](https://playwright.dev/docs/api/class-elementhandle#element-handle-eval-on-selector-all)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```html
   * <div class="feed">
   *   <div class="tweet">Hello!</div>
   *   <div class="tweet">Hi!</div>
   * </div>
   * ```
   *
   * ```js
   * const feedHandle = await page.$('.feed');
   * expect(await feedHandle.$$eval('.tweet', nodes => nodes.map(n => n.innerText))).toEqual(['Hello!', 'Hi!']);
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  $$eval<K extends keyof HTMLElementTagNameMap, R, Arg>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K][], Arg, R>, arg: Arg): Promise<R>;
  /**
   * Returns the return value of `pageFunction`.
   *
   * The method finds all elements matching the specified selector in the `ElementHandle`'s subtree and passes an array
   * of matched elements as a first argument to `pageFunction`.
   *
   * If `pageFunction` returns a [Promise], then
   * [elementHandle.$$eval(selector, pageFunction[, arg])](https://playwright.dev/docs/api/class-elementhandle#element-handle-eval-on-selector-all)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```html
   * <div class="feed">
   *   <div class="tweet">Hello!</div>
   *   <div class="tweet">Hi!</div>
   * </div>
   * ```
   *
   * ```js
   * const feedHandle = await page.$('.feed');
   * expect(await feedHandle.$$eval('.tweet', nodes => nodes.map(n => n.innerText))).toEqual(['Hello!', 'Hi!']);
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  $$eval<R, Arg, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E[], Arg, R>, arg: Arg): Promise<R>;
  /**
   * Returns the return value of `pageFunction`.
   *
   * The method finds all elements matching the specified selector in the `ElementHandle`'s subtree and passes an array
   * of matched elements as a first argument to `pageFunction`.
   *
   * If `pageFunction` returns a [Promise], then
   * [elementHandle.$$eval(selector, pageFunction[, arg])](https://playwright.dev/docs/api/class-elementhandle#element-handle-eval-on-selector-all)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```html
   * <div class="feed">
   *   <div class="tweet">Hello!</div>
   *   <div class="tweet">Hi!</div>
   * </div>
   * ```
   *
   * ```js
   * const feedHandle = await page.$('.feed');
   * expect(await feedHandle.$$eval('.tweet', nodes => nodes.map(n => n.innerText))).toEqual(['Hello!', 'Hi!']);
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  $$eval<K extends keyof HTMLElementTagNameMap, R>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K][], void, R>, arg?: any): Promise<R>;
  /**
   * Returns the return value of `pageFunction`.
   *
   * The method finds all elements matching the specified selector in the `ElementHandle`'s subtree and passes an array
   * of matched elements as a first argument to `pageFunction`.
   *
   * If `pageFunction` returns a [Promise], then
   * [elementHandle.$$eval(selector, pageFunction[, arg])](https://playwright.dev/docs/api/class-elementhandle#element-handle-eval-on-selector-all)
   * would wait for the promise to resolve and return its value.
   *
   * **Usage**
   *
   * ```html
   * <div class="feed">
   *   <div class="tweet">Hello!</div>
   *   <div class="tweet">Hi!</div>
   * </div>
   * ```
   *
   * ```js
   * const feedHandle = await page.$('.feed');
   * expect(await feedHandle.$$eval('.tweet', nodes => nodes.map(n => n.innerText))).toEqual(['Hello!', 'Hi!']);
   * ```
   *
   * @param selector A selector to query for.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  $$eval<R, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E[], void, R>, arg?: any): Promise<R>;

  /**
   * Returns element specified by selector when it satisfies `state` option. Returns `null` if waiting for `hidden` or
   * `detached`.
   *
   * Wait for the `selector` relative to the element handle to satisfy `state` option (either appear/disappear from dom,
   * or become visible/hidden). If at the moment of calling the method `selector` already satisfies the condition, the
   * method will return immediately. If the selector doesn't satisfy the condition for the `timeout` milliseconds, the
   * function will throw.
   *
   * **Usage**
   *
   * ```js
   * await page.setContent(`<div><span></span></div>`);
   * const div = await page.$('div');
   * // Waiting for the 'span' selector relative to the div.
   * const span = await div.waitForSelector('span', { state: 'attached' });
   * ```
   *
   * **NOTE** This method does not work across navigations, use
   * [page.waitForSelector(selector[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-selector)
   * instead.
   * @param selector A selector to query for.
   * @param options
   */
  waitForSelector<K extends keyof HTMLElementTagNameMap>(selector: K, options?: ElementHandleWaitForSelectorOptionsNotHidden): Promise<ElementHandleForTag<K>>;
  /**
   * Returns element specified by selector when it satisfies `state` option. Returns `null` if waiting for `hidden` or
   * `detached`.
   *
   * Wait for the `selector` relative to the element handle to satisfy `state` option (either appear/disappear from dom,
   * or become visible/hidden). If at the moment of calling the method `selector` already satisfies the condition, the
   * method will return immediately. If the selector doesn't satisfy the condition for the `timeout` milliseconds, the
   * function will throw.
   *
   * **Usage**
   *
   * ```js
   * await page.setContent(`<div><span></span></div>`);
   * const div = await page.$('div');
   * // Waiting for the 'span' selector relative to the div.
   * const span = await div.waitForSelector('span', { state: 'attached' });
   * ```
   *
   * **NOTE** This method does not work across navigations, use
   * [page.waitForSelector(selector[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-selector)
   * instead.
   * @param selector A selector to query for.
   * @param options
   */
  waitForSelector(selector: string, options?: ElementHandleWaitForSelectorOptionsNotHidden): Promise<ElementHandle<SVGElement | HTMLElement>>;
  /**
   * Returns element specified by selector when it satisfies `state` option. Returns `null` if waiting for `hidden` or
   * `detached`.
   *
   * Wait for the `selector` relative to the element handle to satisfy `state` option (either appear/disappear from dom,
   * or become visible/hidden). If at the moment of calling the method `selector` already satisfies the condition, the
   * method will return immediately. If the selector doesn't satisfy the condition for the `timeout` milliseconds, the
   * function will throw.
   *
   * **Usage**
   *
   * ```js
   * await page.setContent(`<div><span></span></div>`);
   * const div = await page.$('div');
   * // Waiting for the 'span' selector relative to the div.
   * const span = await div.waitForSelector('span', { state: 'attached' });
   * ```
   *
   * **NOTE** This method does not work across navigations, use
   * [page.waitForSelector(selector[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-selector)
   * instead.
   * @param selector A selector to query for.
   * @param options
   */
  waitForSelector<K extends keyof HTMLElementTagNameMap>(selector: K, options: ElementHandleWaitForSelectorOptions): Promise<ElementHandleForTag<K> | null>;
  /**
   * Returns element specified by selector when it satisfies `state` option. Returns `null` if waiting for `hidden` or
   * `detached`.
   *
   * Wait for the `selector` relative to the element handle to satisfy `state` option (either appear/disappear from dom,
   * or become visible/hidden). If at the moment of calling the method `selector` already satisfies the condition, the
   * method will return immediately. If the selector doesn't satisfy the condition for the `timeout` milliseconds, the
   * function will throw.
   *
   * **Usage**
   *
   * ```js
   * await page.setContent(`<div><span></span></div>`);
   * const div = await page.$('div');
   * // Waiting for the 'span' selector relative to the div.
   * const span = await div.waitForSelector('span', { state: 'attached' });
   * ```
   *
   * **NOTE** This method does not work across navigations, use
   * [page.waitForSelector(selector[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-selector)
   * instead.
   * @param selector A selector to query for.
   * @param options
   */
  waitForSelector(selector: string, options: ElementHandleWaitForSelectorOptions): Promise<null|ElementHandle<SVGElement | HTMLElement>>;
  /**
   * This method returns the bounding box of the element, or `null` if the element is not visible. The bounding box is
   * calculated relative to the main frame viewport - which is usually the same as the browser window.
   *
   * Scrolling affects the returned bounding box, similarly to
   * [Element.getBoundingClientRect](https://developer.mozilla.org/en-US/docs/Web/API/Element/getBoundingClientRect).
   * That means `x` and/or `y` may be negative.
   *
   * Elements from child frames return the bounding box relative to the main frame, unlike the
   * [Element.getBoundingClientRect](https://developer.mozilla.org/en-US/docs/Web/API/Element/getBoundingClientRect).
   *
   * Assuming the page is static, it is safe to use bounding box coordinates to perform input. For example, the
   * following snippet should click the center of the element.
   *
   * **Usage**
   *
   * ```js
   * const box = await elementHandle.boundingBox();
   * await page.mouse.click(box.x + box.width / 2, box.y + box.height / 2);
   * ```
   *
   */
  boundingBox(): Promise<null|{
    /**
     * the x coordinate of the element in pixels.
     */
    x: number;

    /**
     * the y coordinate of the element in pixels.
     */
    y: number;

    /**
     * the width of the element in pixels.
     */
    width: number;

    /**
     * the height of the element in pixels.
     */
    height: number;
  }>;

  /**
   * This method checks the element by performing the following steps:
   * 1. Ensure that element is a checkbox or a radio input. If not, this method throws. If the element is already
   *    checked, this method returns immediately.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to click in the center of the
   *    element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now checked. If not, this method throws.
   *
   * If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param options
   */
  check(options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * This method clicks the element by performing the following steps:
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to click in the center of the
   *    element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   *
   * If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param options
   */
  click(options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * defaults to 1. See [UIEvent.detail].
     */
    clickCount?: number;

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores
     * current modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * Returns the content frame for element handles referencing iframe nodes, or `null` otherwise
   */
  contentFrame(): Promise<null|Frame>;

  /**
   * This method double clicks the element by performing the following steps:
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to double click in the center of the
   *    element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set. Note that if
   *    the first click of the `dblclick()` triggers a navigation event, this method will throw.
   *
   * If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   *
   * **NOTE** `elementHandle.dblclick()` dispatches two `click` events and a single `dblclick` event.
   * @param options
   */
  dblclick(options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores
     * current modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * The snippet below dispatches the `click` event on the element. Regardless of the visibility state of the element,
   * `click` is dispatched. This is equivalent to calling
   * [element.click()](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/click).
   *
   * **Usage**
   *
   * ```js
   * await elementHandle.dispatchEvent('click');
   * ```
   *
   * Under the hood, it creates an instance of an event based on the given `type`, initializes it with `eventInit`
   * properties and dispatches it on the element. Events are `composed`, `cancelable` and bubble by default.
   *
   * Since `eventInit` is event-specific, please refer to the events documentation for the lists of initial properties:
   * - [DragEvent](https://developer.mozilla.org/en-US/docs/Web/API/DragEvent/DragEvent)
   * - [FocusEvent](https://developer.mozilla.org/en-US/docs/Web/API/FocusEvent/FocusEvent)
   * - [KeyboardEvent](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/KeyboardEvent)
   * - [MouseEvent](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/MouseEvent)
   * - [PointerEvent](https://developer.mozilla.org/en-US/docs/Web/API/PointerEvent/PointerEvent)
   * - [TouchEvent](https://developer.mozilla.org/en-US/docs/Web/API/TouchEvent/TouchEvent)
   * - [Event](https://developer.mozilla.org/en-US/docs/Web/API/Event/Event)
   *
   * You can also specify `JSHandle` as the property value if you want live objects to be passed into the event:
   *
   * ```js
   * // Note you can only create DataTransfer in Chromium and Firefox
   * const dataTransfer = await page.evaluateHandle(() => new DataTransfer());
   * await elementHandle.dispatchEvent('dragstart', { dataTransfer });
   * ```
   *
   * @param type DOM event type: `"click"`, `"dragstart"`, etc.
   * @param eventInit Optional event-specific initialization properties.
   */
  dispatchEvent(type: string, eventInit?: EvaluationArgument): Promise<void>;

  /**
   * This method waits for [actionability](https://playwright.dev/docs/actionability) checks, focuses the element, fills it and triggers an
   * `input` event after filling. Note that you can pass an empty string to clear the input field.
   *
   * If the target element is not an `<input>`, `<textarea>` or `[contenteditable]` element, this method throws an
   * error. However, if the element is inside the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), the control will be filled
   * instead.
   *
   * To send fine-grained keyboard events, use
   * [elementHandle.type(text[, options])](https://playwright.dev/docs/api/class-elementhandle#element-handle-type).
   * @param value Value to set for the `<input>`, `<textarea>` or `[contenteditable]` element.
   * @param options
   */
  fill(value: string, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Calls [focus](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/focus) on the element.
   */
  focus(): Promise<void>;

  /**
   * Returns element attribute value.
   * @param name Attribute name to get the value for.
   */
  getAttribute(name: string): Promise<null|string>;

  /**
   * This method hovers over the element by performing the following steps:
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to hover over the center of the
   *    element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   *
   * If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param options
   */
  hover(options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores
     * current modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * Returns the `element.innerHTML`.
   */
  innerHTML(): Promise<string>;

  /**
   * Returns the `element.innerText`.
   */
  innerText(): Promise<string>;

  /**
   * Returns `input.value` for the selected `<input>` or `<textarea>` or `<select>` element.
   *
   * Throws for non-input elements. However, if the element is inside the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), returns the value of the
   * control.
   * @param options
   */
  inputValue(options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<string>;

  /**
   * Returns whether the element is checked. Throws if the element is not a checkbox or radio input.
   */
  isChecked(): Promise<boolean>;

  /**
   * Returns whether the element is disabled, the opposite of [enabled](https://playwright.dev/docs/actionability#enabled).
   */
  isDisabled(): Promise<boolean>;

  /**
   * Returns whether the element is [editable](https://playwright.dev/docs/actionability#editable).
   */
  isEditable(): Promise<boolean>;

  /**
   * Returns whether the element is [enabled](https://playwright.dev/docs/actionability#enabled).
   */
  isEnabled(): Promise<boolean>;

  /**
   * Returns whether the element is hidden, the opposite of [visible](https://playwright.dev/docs/actionability#visible).
   */
  isHidden(): Promise<boolean>;

  /**
   * Returns whether the element is [visible](https://playwright.dev/docs/actionability#visible).
   */
  isVisible(): Promise<boolean>;

  /**
   * Returns the frame containing the given element.
   */
  ownerFrame(): Promise<null|Frame>;

  /**
   * Focuses the element, and then uses
   * [keyboard.down(key)](https://playwright.dev/docs/api/class-keyboard#keyboard-down) and
   * [keyboard.up(key)](https://playwright.dev/docs/api/class-keyboard#keyboard-up).
   *
   * `key` can specify the intended
   * [keyboardEvent.key](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key) value or a single character
   * to generate the text for. A superset of the `key` values can be found
   * [here](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key/Key_Values). Examples of the keys are:
   *
   * `F1` - `F12`, `Digit0`- `Digit9`, `KeyA`- `KeyZ`, `Backquote`, `Minus`, `Equal`, `Backslash`, `Backspace`, `Tab`,
   * `Delete`, `Escape`, `ArrowDown`, `End`, `Enter`, `Home`, `Insert`, `PageDown`, `PageUp`, `ArrowRight`, `ArrowUp`,
   * etc.
   *
   * Following modification shortcuts are also supported: `Shift`, `Control`, `Alt`, `Meta`, `ShiftLeft`.
   *
   * Holding down `Shift` will type the text that corresponds to the `key` in the upper case.
   *
   * If `key` is a single character, it is case-sensitive, so the values `a` and `A` will generate different respective
   * texts.
   *
   * Shortcuts such as `key: "Control+o"` or `key: "Control+Shift+T"` are supported as well. When specified with the
   * modifier, modifier is pressed and being held while the subsequent key is being pressed.
   * @param key Name of the key to press or a character to generate, such as `ArrowLeft` or `a`.
   * @param options
   */
  press(key: string, options?: {
    /**
     * Time to wait between `keydown` and `keyup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method captures a screenshot of the page, clipped to the size and position of this particular element. If the
   * element is covered by other elements, it will not be actually visible on the screenshot. If the element is a
   * scrollable container, only the currently scrolled content will be visible on the screenshot.
   *
   * This method waits for the [actionability](https://playwright.dev/docs/actionability) checks, then scrolls element into view before taking
   * a screenshot. If the element is detached from DOM, the method throws an error.
   *
   * Returns the buffer with the captured screenshot.
   * @param options
   */
  screenshot(options?: {
    /**
     * When set to `"disabled"`, stops CSS animations, CSS transitions and Web Animations. Animations get different
     * treatment depending on their duration:
     * - finite animations are fast-forwarded to completion, so they'll fire `transitionend` event.
     * - infinite animations are canceled to initial state, and then played over after the screenshot.
     *
     * Defaults to `"allow"` that leaves animations untouched.
     */
    animations?: "disabled"|"allow";

    /**
     * When set to `"hide"`, screenshot will hide text caret. When set to `"initial"`, text caret behavior will not be
     * changed.  Defaults to `"hide"`.
     */
    caret?: "hide"|"initial";

    /**
     * Specify locators that should be masked when the screenshot is taken. Masked elements will be overlaid with a pink
     * box `#FF00FF` that completely covers its bounding box.
     */
    mask?: Array<Locator>;

    /**
     * Hides default white background and allows capturing screenshots with transparency. Not applicable to `jpeg` images.
     * Defaults to `false`.
     */
    omitBackground?: boolean;

    /**
     * The file path to save the image to. The screenshot type will be inferred from file extension. If `path` is a
     * relative path, then it is resolved relative to the current working directory. If no path is provided, the image
     * won't be saved to the disk.
     */
    path?: string;

    /**
     * The quality of the image, between 0-100. Not applicable to `png` images.
     */
    quality?: number;

    /**
     * When set to `"css"`, screenshot will have a single pixel per each css pixel on the page. For high-dpi devices, this
     * will keep screenshots small. Using `"device"` option will produce a single pixel per each device pixel, so
     * screenshots of high-dpi devices will be twice as large or even larger.
     *
     * Defaults to `"device"`.
     */
    scale?: "css"|"device";

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * Specify screenshot type, defaults to `png`.
     */
    type?: "png"|"jpeg";
  }): Promise<Buffer>;

  /**
   * This method waits for [actionability](https://playwright.dev/docs/actionability) checks, then tries to scroll element into view, unless
   * it is completely visible as defined by
   * [IntersectionObserver](https://developer.mozilla.org/en-US/docs/Web/API/Intersection_Observer_API)'s `ratio`.
   *
   * Throws when `elementHandle` does not point to an element
   * [connected](https://developer.mozilla.org/en-US/docs/Web/API/Node/isConnected) to a Document or a ShadowRoot.
   * @param options
   */
  scrollIntoViewIfNeeded(options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method waits for [actionability](https://playwright.dev/docs/actionability) checks, waits until all specified options are present in
   * the `<select>` element and selects these options.
   *
   * If the target element is not a `<select>` element, this method throws an error. However, if the element is inside
   * the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), the control will be used
   * instead.
   *
   * Returns the array of option values that have been successfully selected.
   *
   * Triggers a `change` and `input` event once all the provided options have been selected.
   *
   * **Usage**
   *
   * ```js
   * // single selection matching the value
   * handle.selectOption('blue');
   *
   * // single selection matching the label
   * handle.selectOption({ label: 'Blue' });
   *
   * // multiple selection
   * handle.selectOption(['red', 'green', 'blue']);
   * ```
   *
   * @param values Options to select. If the `<select>` has the `multiple` attribute, all matching options are selected, otherwise
   * only the first option matching one of the passed options is selected. String values are matching both values and
   * labels. Option is considered matching if all specified properties match.
   * @param options
   */
  selectOption(values: null|string|ElementHandle|Array<string>|{
    /**
     * Matches by `option.value`. Optional.
     */
    value?: string;

    /**
     * Matches by `option.label`. Optional.
     */
    label?: string;

    /**
     * Matches by the index. Optional.
     */
    index?: number;
  }|Array<ElementHandle>|Array<{
    /**
     * Matches by `option.value`. Optional.
     */
    value?: string;

    /**
     * Matches by `option.label`. Optional.
     */
    label?: string;

    /**
     * Matches by the index. Optional.
     */
    index?: number;
  }>, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<Array<string>>;

  /**
   * This method waits for [actionability](https://playwright.dev/docs/actionability) checks, then focuses the element and selects all its
   * text content.
   *
   * If the element is inside the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), focuses and selects text in
   * the control instead.
   * @param options
   */
  selectText(options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method checks or unchecks an element by performing the following steps:
   * 1. Ensure that element is a checkbox or a radio input. If not, this method throws.
   * 1. If the element already has the right checked state, this method returns immediately.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the matched element, unless `force` option is set. If
   *    the element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to click in the center of the
   *    element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now checked or unchecked. If not, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param checked Whether to check or uncheck the checkbox.
   * @param options
   */
  setChecked(checked: boolean, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * Sets the value of the file input to these file paths or files. If some of the `filePaths` are relative paths, then
   * they are resolved relative to the current working directory. For empty array, clears the selected files.
   *
   * This method expects [ElementHandle] to point to an
   * [input element](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input). However, if the element is inside
   * the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), targets the control instead.
   * @param files
   * @param options
   */
  setInputFiles(files: string|Array<string>|{
    /**
     * File name
     */
    name: string;

    /**
     * File type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }|Array<{
    /**
     * File name
     */
    name: string;

    /**
     * File type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }>, options?: {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method taps the element by performing the following steps:
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.touchscreen](https://playwright.dev/docs/api/class-page#page-touchscreen) to tap the center of the
   *    element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   *
   * If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   *
   * **NOTE** `elementHandle.tap()` requires that the `hasTouch` option of the browser context be set to true.
   * @param options
   */
  tap(options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores
     * current modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * Returns the `node.textContent`.
   */
  textContent(): Promise<null|string>;

  /**
   * Focuses the element, and then sends a `keydown`, `keypress`/`input`, and `keyup` event for each character in the
   * text.
   *
   * To press a special key, like `Control` or `ArrowDown`, use
   * [elementHandle.press(key[, options])](https://playwright.dev/docs/api/class-elementhandle#element-handle-press).
   *
   * **Usage**
   *
   * ```js
   * await elementHandle.type('Hello'); // Types instantly
   * await elementHandle.type('World', {delay: 100}); // Types slower, like a user
   * ```
   *
   * An example of typing into a text field and then submitting the form:
   *
   * ```js
   * const elementHandle = await page.$('input');
   * await elementHandle.type('some text');
   * await elementHandle.press('Enter');
   * ```
   *
   * @param text A text to type into a focused element.
   * @param options
   */
  type(text: string, options?: {
    /**
     * Time to wait between key presses in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method checks the element by performing the following steps:
   * 1. Ensure that element is a checkbox or a radio input. If not, this method throws. If the element is already
   *    unchecked, this method returns immediately.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to click in the center of the
   *    element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now unchecked. If not, this method throws.
   *
   * If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param options
   */
  uncheck(options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * Returns when the element satisfies the `state`.
   *
   * Depending on the `state` parameter, this method waits for one of the [actionability](https://playwright.dev/docs/actionability) checks to
   * pass. This method throws when the element is detached while waiting, unless waiting for the `"hidden"` state.
   * - `"visible"` Wait until the element is [visible](https://playwright.dev/docs/actionability#visible).
   * - `"hidden"` Wait until the element is [not visible](https://playwright.dev/docs/actionability#visible) or
   *   [not attached](https://playwright.dev/docs/actionability#attached). Note that waiting for hidden does not throw when the element
   *   detaches.
   * - `"stable"` Wait until the element is both [visible](https://playwright.dev/docs/actionability#visible) and
   *   [stable](https://playwright.dev/docs/actionability#stable).
   * - `"enabled"` Wait until the element is [enabled](https://playwright.dev/docs/actionability#enabled).
   * - `"disabled"` Wait until the element is [not enabled](https://playwright.dev/docs/actionability#enabled).
   * - `"editable"` Wait until the element is [editable](https://playwright.dev/docs/actionability#editable).
   *
   * If the element does not satisfy the condition for the `timeout` milliseconds, this method will throw.
   * @param state A state to wait for, see below for more details.
   * @param options
   */
  waitForElementState(state: "visible"|"hidden"|"stable"|"enabled"|"disabled"|"editable", options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;
}

/**
 * Locators are the central piece of Playwright's auto-waiting and retry-ability. In a nutshell, locators represent a
 * way to find element(s) on the page at any moment. Locator can be created with the
 * [page.locator(selector[, options])](https://playwright.dev/docs/api/class-page#page-locator) method.
 *
 * [Learn more about locators](https://playwright.dev/docs/locators).
 */
export interface Locator {
  /**
   * Execute JavaScript code in the page, taking the matching element as an argument.
   *
   * **Details**
   *
   * Returns the return value of `pageFunction`, called with the matching element as a first argument, and `arg` as a
   * second argument.
   *
   * If `pageFunction` returns a [Promise], this method will wait for the promise to resolve and return its value.
   *
   * If `pageFunction` throws or rejects, this method throws.
   *
   * **Usage**
   *
   * ```js
   * const tweets = page.locator('.tweet .retweets');
   * expect(await tweets.evaluate(node => node.innerText)).toBe('10 retweets');
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   * @param options
   */
  evaluate<R, Arg, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(pageFunction: PageFunctionOn<E, Arg, R>, arg: Arg, options?: {
    timeout?: number;
  }): Promise<R>;
  /**
   * Execute JavaScript code in the page, taking the matching element as an argument.
   *
   * **Details**
   *
   * Returns the return value of `pageFunction`, called with the matching element as a first argument, and `arg` as a
   * second argument.
   *
   * If `pageFunction` returns a [Promise], this method will wait for the promise to resolve and return its value.
   *
   * If `pageFunction` throws or rejects, this method throws.
   *
   * **Usage**
   *
   * ```js
   * const tweets = page.locator('.tweet .retweets');
   * expect(await tweets.evaluate(node => node.innerText)).toBe('10 retweets');
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   * @param options
   */
  evaluate<R, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(pageFunction: PageFunctionOn<E, void, R>, options?: {
    timeout?: number;
  }): Promise<R>;
  /**
   * Execute JavaScript code in the page, taking all matching elements as an argument.
   *
   * **Details**
   *
   * Returns the return value of `pageFunction`, called with an array of all matching elements as a first argument, and
   * `arg` as a second argument.
   *
   * If `pageFunction` returns a [Promise], this method will wait for the promise to resolve and return its value.
   *
   * If `pageFunction` throws or rejects, this method throws.
   *
   * **Usage**
   *
   * ```js
   * const locator = page.locator('div');
   * const moreThanTen = await locator.evaluateAll((divs, min) => divs.length > min, 10);
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluateAll<R, Arg, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(pageFunction: PageFunctionOn<E[], Arg, R>, arg: Arg): Promise<R>;
  /**
   * Execute JavaScript code in the page, taking all matching elements as an argument.
   *
   * **Details**
   *
   * Returns the return value of `pageFunction`, called with an array of all matching elements as a first argument, and
   * `arg` as a second argument.
   *
   * If `pageFunction` returns a [Promise], this method will wait for the promise to resolve and return its value.
   *
   * If `pageFunction` throws or rejects, this method throws.
   *
   * **Usage**
   *
   * ```js
   * const locator = page.locator('div');
   * const moreThanTen = await locator.evaluateAll((divs, min) => divs.length > min, 10);
   * ```
   *
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluateAll<R, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(pageFunction: PageFunctionOn<E[], void, R>): Promise<R>;
  /**
   * **NOTE** Always prefer using [Locator]s and web assertions over [ElementHandle]s because latter are inherently racy.
   *
   * Resolves given locator to the first matching DOM element. If there are no matching elements, waits for one. If
   * multiple elements match the locator, throws.
   * @param options
   */
  elementHandle(options?: {
    timeout?: number;
  }): Promise<null|ElementHandle<SVGElement | HTMLElement>>;
  /**
   * When locator points to a list of elements, returns array of locators, pointing to respective elements.
   *
   * Note that [locator.all()](https://playwright.dev/docs/api/class-locator#locator-all) does not wait for elements to
   * match the locator, and instead immediately returns whatever is present in the page. To avoid flakiness when
   * elements are loaded dynamically, wait for the loading to finish before calling
   * [locator.all()](https://playwright.dev/docs/api/class-locator#locator-all).
   *
   * **Usage**
   *
   * ```js
   * for (const li of await page.getByRole('listitem').all())
   *   await li.click();
   * ```
   *
   */
  all(): Promise<Array<Locator>>;

  /**
   * Returns an array of `node.innerText` values for all matching nodes.
   *
   * **Usage**
   *
   * ```js
   * const texts = await page.getByRole('link').allInnerTexts();
   * ```
   *
   */
  allInnerTexts(): Promise<Array<string>>;

  /**
   * Returns an array of `node.textContent` values for all matching nodes.
   *
   * **Usage**
   *
   * ```js
   * const texts = await page.getByRole('link').allTextContents();
   * ```
   *
   */
  allTextContents(): Promise<Array<string>>;

  /**
   * Calls [blur](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/blur) on the element.
   * @param options
   */
  blur(options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method returns the bounding box of the element matching the locator, or `null` if the element is not visible.
   * The bounding box is calculated relative to the main frame viewport - which is usually the same as the browser
   * window.
   *
   * **Details**
   *
   * Scrolling affects the returned bounding box, similarly to
   * [Element.getBoundingClientRect](https://developer.mozilla.org/en-US/docs/Web/API/Element/getBoundingClientRect).
   * That means `x` and/or `y` may be negative.
   *
   * Elements from child frames return the bounding box relative to the main frame, unlike the
   * [Element.getBoundingClientRect](https://developer.mozilla.org/en-US/docs/Web/API/Element/getBoundingClientRect).
   *
   * Assuming the page is static, it is safe to use bounding box coordinates to perform input. For example, the
   * following snippet should click the center of the element.
   *
   * **Usage**
   *
   * ```js
   * const box = await page.getByRole('button').boundingBox();
   * await page.mouse.click(box.x + box.width / 2, box.y + box.height / 2);
   * ```
   *
   * @param options
   */
  boundingBox(options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<null|{
    /**
     * the x coordinate of the element in pixels.
     */
    x: number;

    /**
     * the y coordinate of the element in pixels.
     */
    y: number;

    /**
     * the width of the element in pixels.
     */
    width: number;

    /**
     * the height of the element in pixels.
     */
    height: number;
  }>;

  /**
   * Ensure that checkbox or radio element is checked.
   *
   * **Details**
   *
   * Performs the following steps:
   * 1. Ensure that element is a checkbox or a radio input. If not, this method throws. If the element is already
   *    checked, this method returns immediately.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to click in the center of the
   *    element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now checked. If not, this method throws.
   *
   * If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   *
   * **Usage**
   *
   * ```js
   * await page.getByRole('checkbox').check();
   * ```
   *
   * @param options
   */
  check(options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * Clear the input field.
   *
   * **Details**
   *
   * This method waits for [actionability](https://playwright.dev/docs/actionability) checks, focuses the element, clears it and triggers an
   * `input` event after clearing.
   *
   * If the target element is not an `<input>`, `<textarea>` or `[contenteditable]` element, this method throws an
   * error. However, if the element is inside the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), the control will be cleared
   * instead.
   *
   * **Usage**
   *
   * ```js
   * await page.getByRole('textbox').clear();
   * ```
   *
   * @param options
   */
  clear(options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Click an element.
   *
   * **Details**
   *
   * This method clicks the element by performing the following steps:
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to click in the center of the
   *    element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   *
   * If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   *
   * **Usage**
   *
   * Click a button:
   *
   * ```js
   * await page.getByRole('button').click();
   * ```
   *
   * Shift-right-click at a specific position on a canvas:
   *
   * ```js
   * await page.locator('canvas').click({
   *   button: 'right',
   *   modifiers: ['Shift'],
   *   position: { x: 23, y: 32 },
   * });
   * ```
   *
   * @param options
   */
  click(options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * defaults to 1. See [UIEvent.detail].
     */
    clickCount?: number;

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores
     * current modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * Returns the number of elements matching the locator.
   *
   * **Usage**
   *
   * ```js
   * const count = await page.getByRole('listitem').count();
   * ```
   *
   */
  count(): Promise<number>;

  /**
   * Double-click an element.
   *
   * **Details**
   *
   * This method double clicks the element by performing the following steps:
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to double click in the center of the
   *    element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set. Note that if
   *    the first click of the `dblclick()` triggers a navigation event, this method will throw.
   *
   * If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   *
   * **NOTE** `element.dblclick()` dispatches two `click` events and a single `dblclick` event.
   * @param options
   */
  dblclick(options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores
     * current modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * Programmatically dispatch an event on the matching element.
   *
   * **Usage**
   *
   * ```js
   * await locator.dispatchEvent('click');
   * ```
   *
   * **Details**
   *
   * The snippet above dispatches the `click` event on the element. Regardless of the visibility state of the element,
   * `click` is dispatched. This is equivalent to calling
   * [element.click()](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/click).
   *
   * Under the hood, it creates an instance of an event based on the given `type`, initializes it with `eventInit`
   * properties and dispatches it on the element. Events are `composed`, `cancelable` and bubble by default.
   *
   * Since `eventInit` is event-specific, please refer to the events documentation for the lists of initial properties:
   * - [DragEvent](https://developer.mozilla.org/en-US/docs/Web/API/DragEvent/DragEvent)
   * - [FocusEvent](https://developer.mozilla.org/en-US/docs/Web/API/FocusEvent/FocusEvent)
   * - [KeyboardEvent](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/KeyboardEvent)
   * - [MouseEvent](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/MouseEvent)
   * - [PointerEvent](https://developer.mozilla.org/en-US/docs/Web/API/PointerEvent/PointerEvent)
   * - [TouchEvent](https://developer.mozilla.org/en-US/docs/Web/API/TouchEvent/TouchEvent)
   * - [Event](https://developer.mozilla.org/en-US/docs/Web/API/Event/Event)
   *
   * You can also specify [JSHandle] as the property value if you want live objects to be passed into the event:
   *
   * ```js
   * // Note you can only create DataTransfer in Chromium and Firefox
   * const dataTransfer = await page.evaluateHandle(() => new DataTransfer());
   * await locator.dispatchEvent('dragstart', { dataTransfer });
   * ```
   *
   * @param type DOM event type: `"click"`, `"dragstart"`, etc.
   * @param eventInit Optional event-specific initialization properties.
   * @param options
   */
  dispatchEvent(type: string, eventInit?: EvaluationArgument, options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Drag the source element towards the target element and drop it.
   *
   * **Details**
   *
   * This method drags the locator to another target locator or target position. It will first move to the source
   * element, perform a `mousedown`, then move to the target element or position and perform a `mouseup`.
   *
   * **Usage**
   *
   * ```js
   * const source = page.locator('#source');
   * const target = page.locator('#target');
   *
   * await source.dragTo(target);
   * // or specify exact positions relative to the top-left corners of the elements:
   * await source.dragTo(target, {
   *   sourcePosition: { x: 34, y: 7 },
   *   targetPosition: { x: 10, y: 20 },
   * });
   * ```
   *
   * @param target Locator of the element to drag to.
   * @param options
   */
  dragTo(target: Locator, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Clicks on the source element at this point relative to the top-left corner of the element's padding box. If not
     * specified, some visible point of the element is used.
     */
    sourcePosition?: {
      x: number;

      y: number;
    };

    /**
     * Drops on the target element at this point relative to the top-left corner of the element's padding box. If not
     * specified, some visible point of the element is used.
     */
    targetPosition?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * **NOTE** Always prefer using [Locator]s and web assertions over [ElementHandle]s because latter are inherently racy.
   *
   * Resolves given locator to all matching DOM elements. If there are no matching elements, returns an empty list.
   */
  elementHandles(): Promise<Array<ElementHandle>>;

  /**
   * Execute JavaScript code in the page, taking the matching element as an argument, and return a [JSHandle] with the
   * result.
   *
   * **Details**
   *
   * Returns the return value of `pageFunction` as a[JSHandle], called with the matching element as a first argument,
   * and `arg` as a second argument.
   *
   * The only difference between
   * [locator.evaluate(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-locator#locator-evaluate)
   * and
   * [locator.evaluateHandle(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-locator#locator-evaluate-handle)
   * is that
   * [locator.evaluateHandle(pageFunction[, arg, options])](https://playwright.dev/docs/api/class-locator#locator-evaluate-handle)
   * returns [JSHandle].
   *
   * If `pageFunction` returns a [Promise], this method will wait for the promise to resolve and return its value.
   *
   * If `pageFunction` throws or rejects, this method throws.
   *
   * See [page.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-page#page-evaluate-handle) for
   * more details.
   * @param pageFunction Function to be evaluated in the page context.
   * @param arg Optional argument to pass to `pageFunction`.
   * @param options
   */
  evaluateHandle(pageFunction: Function|string, arg?: EvaluationArgument, options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<JSHandle>;

  /**
   * Set a value to the input field.
   *
   * **Usage**
   *
   * ```js
   * await page.getByRole('textbox').fill('example value');
   * ```
   *
   * **Details**
   *
   * This method waits for [actionability](https://playwright.dev/docs/actionability) checks, focuses the element, fills it and triggers an
   * `input` event after filling. Note that you can pass an empty string to clear the input field.
   *
   * If the target element is not an `<input>`, `<textarea>` or `[contenteditable]` element, this method throws an
   * error. However, if the element is inside the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), the control will be filled
   * instead.
   *
   * To send fine-grained keyboard events, use
   * [locator.type(text[, options])](https://playwright.dev/docs/api/class-locator#locator-type).
   * @param value Value to set for the `<input>`, `<textarea>` or `[contenteditable]` element.
   * @param options
   */
  fill(value: string, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method narrows existing locator according to the options, for example filters by text. It can be chained to
   * filter multiple times.
   *
   * **Usage**
   *
   * ```js
   * const rowLocator = page.locator('tr');
   * // ...
   * await rowLocator
   *     .filter({ hasText: 'text in column 1' })
   *     .filter({ has: page.getByRole('button', { name: 'column 2 button' }) })
   *     .screenshot();
   * ```
   *
   * @param options
   */
  filter(options?: {
    /**
     * Matches elements containing an element that matches an inner locator. Inner locator is queried against the outer
     * one. For example, `article` that has `text=Playwright` matches `<article><div>Playwright</div></article>`.
     *
     * Note that outer and inner locators must belong to the same frame. Inner locator must not contain [FrameLocator]s.
     */
    has?: Locator;

    /**
     * Matches elements containing specified text somewhere inside, possibly in a child or a descendant element. When
     * passed a [string], matching is case-insensitive and searches for a substring. For example, `"Playwright"` matches
     * `<article><div>Playwright</div></article>`.
     */
    hasText?: string|RegExp;
  }): Locator;

  /**
   * Returns locator to the first matching element.
   */
  first(): Locator;

  /**
   * Calls [focus](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/focus) on the matching element.
   * @param options
   */
  focus(options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * When working with iframes, you can create a frame locator that will enter the iframe and allow locating elements in
   * that iframe:
   *
   * **Usage**
   *
   * ```js
   * const locator = page.frameLocator('iframe').getByText('Submit');
   * await locator.click();
   * ```
   *
   * @param selector A selector to use when resolving DOM element.
   */
  frameLocator(selector: string): FrameLocator;

  /**
   * Returns the matching element's attribute value.
   * @param name Attribute name to get the value for.
   * @param options
   */
  getAttribute(name: string, options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<null|string>;

  /**
   * Allows locating elements by their alt text.
   *
   * **Usage**
   *
   * For example, this method will find the image by alt text "Playwright logo":
   *
   * ```html
   * <img alt='Playwright logo'>
   * ```
   *
   * ```js
   * await page.getByAltText('Playwright logo').click();
   * ```
   *
   * @param text Text to locate the element for.
   * @param options
   */
  getByAltText(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Allows locating input elements by the text of the associated `<label>` or `aria-labelledby` element, or by the
   * `aria-label` attribute.
   *
   * **Usage**
   *
   * For example, this method will find inputs by label "Username" and "Password" in the following DOM:
   *
   * ```html
   * <input aria-label="Username">
   * <label for="password-input">Password:</label>
   * <input id="password-input">
   * ```
   *
   * ```js
   * await page.getByLabel('Username').fill('john');
   * await page.getByLabel('Password').fill('secret');
   * ```
   *
   * @param text Text to locate the element for.
   * @param options
   */
  getByLabel(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Allows locating input elements by the placeholder text.
   *
   * **Usage**
   *
   * For example, consider the following DOM structure.
   *
   * ```html
   * <input type="email" placeholder="name@example.com" />
   * ```
   *
   * You can fill the input after locating it by the placeholder text:
   *
   * ```js
   * await page
   *     .getByPlaceholder("name@example.com")
   *     .fill("playwright@microsoft.com");
   * ```
   *
   * @param text Text to locate the element for.
   * @param options
   */
  getByPlaceholder(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Allows locating elements by their [ARIA role](https://www.w3.org/TR/wai-aria-1.2/#roles),
   * [ARIA attributes](https://www.w3.org/TR/wai-aria-1.2/#aria-attributes) and
   * [accessible name](https://w3c.github.io/accname/#dfn-accessible-name).
   *
   * **Usage**
   *
   * Consider the following DOM structure.
   *
   * ```html
   * <h3>Sign up</h3>
   * <label>
   *   <input type="checkbox" /> Subscribe
   * </label>
   * <br/>
   * <button>Submit</button>
   * ```
   *
   * You can locate each element by it's implicit role:
   *
   * ```js
   * await expect(page.getByRole('heading', { name: 'Sign up' })).toBeVisible();
   *
   * await page.getByRole('checkbox', { name: 'Subscribe' }).check();
   *
   * await page.getByRole('button', { name: /submit/i }).click();
   * ```
   *
   * **Details**
   *
   * Role selector **does not replace** accessibility audits and conformance tests, but rather gives early feedback
   * about the ARIA guidelines.
   *
   * Many html elements have an implicitly [defined role](https://w3c.github.io/html-aam/#html-element-role-mappings)
   * that is recognized by the role selector. You can find all the
   * [supported roles here](https://www.w3.org/TR/wai-aria-1.2/#role_definitions). ARIA guidelines **do not recommend**
   * duplicating implicit roles and attributes by setting `role` and/or `aria-*` attributes to default values.
   * @param role Required aria role.
   * @param options
   */
  getByRole(role: "alert"|"alertdialog"|"application"|"article"|"banner"|"blockquote"|"button"|"caption"|"cell"|"checkbox"|"code"|"columnheader"|"combobox"|"complementary"|"contentinfo"|"definition"|"deletion"|"dialog"|"directory"|"document"|"emphasis"|"feed"|"figure"|"form"|"generic"|"grid"|"gridcell"|"group"|"heading"|"img"|"insertion"|"link"|"list"|"listbox"|"listitem"|"log"|"main"|"marquee"|"math"|"meter"|"menu"|"menubar"|"menuitem"|"menuitemcheckbox"|"menuitemradio"|"navigation"|"none"|"note"|"option"|"paragraph"|"presentation"|"progressbar"|"radio"|"radiogroup"|"region"|"row"|"rowgroup"|"rowheader"|"scrollbar"|"search"|"searchbox"|"separator"|"slider"|"spinbutton"|"status"|"strong"|"subscript"|"superscript"|"switch"|"tab"|"table"|"tablist"|"tabpanel"|"term"|"textbox"|"time"|"timer"|"toolbar"|"tooltip"|"tree"|"treegrid"|"treeitem", options?: {
    /**
     * An attribute that is usually set by `aria-checked` or native `<input type=checkbox>` controls.
     *
     * Learn more about [`aria-checked`](https://www.w3.org/TR/wai-aria-1.2/#aria-checked).
     */
    checked?: boolean;

    /**
     * An attribute that is usually set by `aria-disabled` or `disabled`.
     *
     * **NOTE** Unlike most other attributes, `disabled` is inherited through the DOM hierarchy. Learn more about
     * [`aria-disabled`](https://www.w3.org/TR/wai-aria-1.2/#aria-disabled).
     */
    disabled?: boolean;

    /**
     * Whether `name` is matched exactly: case-sensitive and whole-string. Defaults to false. Ignored when `name` is a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;

    /**
     * An attribute that is usually set by `aria-expanded`.
     *
     * Learn more about [`aria-expanded`](https://www.w3.org/TR/wai-aria-1.2/#aria-expanded).
     */
    expanded?: boolean;

    /**
     * Option that controls whether hidden elements are matched. By default, only non-hidden elements, as
     * [defined by ARIA](https://www.w3.org/TR/wai-aria-1.2/#tree_exclusion), are matched by role selector.
     *
     * Learn more about [`aria-hidden`](https://www.w3.org/TR/wai-aria-1.2/#aria-hidden).
     */
    includeHidden?: boolean;

    /**
     * A number attribute that is usually present for roles `heading`, `listitem`, `row`, `treeitem`, with default values
     * for `<h1>-<h6>` elements.
     *
     * Learn more about [`aria-level`](https://www.w3.org/TR/wai-aria-1.2/#aria-level).
     */
    level?: number;

    /**
     * Option to match the [accessible name](https://w3c.github.io/accname/#dfn-accessible-name). By default, matching is
     * case-insensitive and searches for a substring, use `exact` to control this behavior.
     *
     * Learn more about [accessible name](https://w3c.github.io/accname/#dfn-accessible-name).
     */
    name?: string|RegExp;

    /**
     * An attribute that is usually set by `aria-pressed`.
     *
     * Learn more about [`aria-pressed`](https://www.w3.org/TR/wai-aria-1.2/#aria-pressed).
     */
    pressed?: boolean;

    /**
     * An attribute that is usually set by `aria-selected`.
     *
     * Learn more about [`aria-selected`](https://www.w3.org/TR/wai-aria-1.2/#aria-selected).
     */
    selected?: boolean;
  }): Locator;

  /**
   * Locate element by the test id.
   *
   * **Usage**
   *
   * Consider the following DOM structure.
   *
   * ```html
   * <button data-testid="directions">Itinéraire</button>
   * ```
   *
   * You can locate the element by it's test id:
   *
   * ```js
   * await page.getByTestId('directions').click();
   * ```
   *
   * **Details**
   *
   * By default, the `data-testid` attribute is used as a test id. Use
   * [selectors.setTestIdAttribute(attributeName)](https://playwright.dev/docs/api/class-selectors#selectors-set-test-id-attribute)
   * to configure a different test id attribute if necessary.
   *
   * ```js
   * // Set custom test id attribute from @playwright/test config:
   * use: {
   *   testIdAttribute: 'data-pw'
   * }
   * ```
   *
   * @param testId Id to locate the element by.
   */
  getByTestId(testId: string|RegExp): Locator;

  /**
   * Allows locating elements that contain given text.
   *
   * See also [locator.filter([options])](https://playwright.dev/docs/api/class-locator#locator-filter) that allows to
   * match by another criteria, like an accessible role, and then filter by the text content.
   *
   * **Usage**
   *
   * Consider the following DOM structure:
   *
   * ```html
   * <div>Hello <span>world</span></div>
   * <div>Hello</div>
   * ```
   *
   * You can locate by text substring, exact string, or a regular expression:
   *
   * ```js
   * // Matches <span>
   * page.getByText('world')
   *
   * // Matches first <div>
   * page.getByText('Hello world')
   *
   * // Matches second <div>
   * page.getByText('Hello', { exact: true })
   *
   * // Matches both <div>s
   * page.getByText(/Hello/)
   *
   * // Matches second <div>
   * page.getByText(/^hello$/i)
   * ```
   *
   * **Details**
   *
   * Matching by text always normalizes whitespace, even with exact match. For example, it turns multiple spaces into
   * one, turns line breaks into spaces and ignores leading and trailing whitespace.
   *
   * Input elements of the type `button` and `submit` are matched by their `value` instead of the text content. For
   * example, locating by text `"Log in"` matches `<input type=button value="Log in">`.
   * @param text Text to locate the element for.
   * @param options
   */
  getByText(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Allows locating elements by their title attribute.
   *
   * **Usage**
   *
   * Consider the following DOM structure.
   *
   * ```html
   * <span title='Issues count'>25 issues</span>
   * ```
   *
   * You can check the issues count after locating it by the title text:
   *
   * ```js
   * await expect(page.getByTitle('Issues count')).toHaveText('25 issues');
   * ```
   *
   * @param text Text to locate the element for.
   * @param options
   */
  getByTitle(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Highlight the corresponding element(s) on the screen. Useful for debugging, don't commit the code that uses
   * [locator.highlight()](https://playwright.dev/docs/api/class-locator#locator-highlight).
   */
  highlight(): Promise<void>;

  /**
   * Hover over the matching element.
   *
   * **Usage**
   *
   * ```js
   * await page.getByRole('link').hover();
   * ```
   *
   * **Details**
   *
   * This method hovers over the element by performing the following steps:
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to hover over the center of the
   *    element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   *
   * If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param options
   */
  hover(options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores
     * current modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * Returns the [`element.innerHTML`](https://developer.mozilla.org/en-US/docs/Web/API/Element/innerHTML).
   * @param options
   */
  innerHTML(options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<string>;

  /**
   * Returns the [`element.innerText`](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/innerText).
   * @param options
   */
  innerText(options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<string>;

  /**
   * Returns the value for the matching `<input>` or `<textarea>` or `<select>` element.
   *
   * **Usage**
   *
   * ```js
   * const value = await page.getByRole('textbox').inputValue();
   * ```
   *
   * **Details**
   *
   * Throws elements that are not an input, textarea or a select. However, if the element is inside the `<label>`
   * element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), returns the value of the
   * control.
   * @param options
   */
  inputValue(options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<string>;

  /**
   * Returns whether the element is checked. Throws if the element is not a checkbox or radio input.
   *
   * **Usage**
   *
   * ```js
   * const checked = await page.getByRole('checkbox').isChecked();
   * ```
   *
   * @param options
   */
  isChecked(options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns whether the element is disabled, the opposite of [enabled](https://playwright.dev/docs/actionability#enabled).
   *
   * **Usage**
   *
   * ```js
   * const disabled = await page.getByRole('button').isDisabled();
   * ```
   *
   * @param options
   */
  isDisabled(options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns whether the element is [editable](https://playwright.dev/docs/actionability#editable).
   *
   * **Usage**
   *
   * ```js
   * const editable = await page.getByRole('textbox').isEditable();
   * ```
   *
   * @param options
   */
  isEditable(options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns whether the element is [enabled](https://playwright.dev/docs/actionability#enabled).
   *
   * **Usage**
   *
   * ```js
   * const enabled = await page.getByRole('button').isEnabled();
   * ```
   *
   * @param options
   */
  isEnabled(options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns whether the element is hidden, the opposite of [visible](https://playwright.dev/docs/actionability#visible).
   *
   * **Usage**
   *
   * ```js
   * const hidden = await page.getByRole('button').isHidden();
   * ```
   *
   * @param options
   */
  isHidden(options?: {
    /**
     * @deprecated This option is ignored.
     * [locator.isHidden([options])](https://playwright.dev/docs/api/class-locator#locator-is-hidden) does not wait for
     * the element to become hidden and returns immediately.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns whether the element is [visible](https://playwright.dev/docs/actionability#visible).
   *
   * **Usage**
   *
   * ```js
   * const visible = await page.getByRole('button').isVisible();
   * ```
   *
   * @param options
   */
  isVisible(options?: {
    /**
     * @deprecated This option is ignored.
     * [locator.isVisible([options])](https://playwright.dev/docs/api/class-locator#locator-is-visible) does not wait for
     * the element to become visible and returns immediately.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns locator to the last matching element.
   *
   * **Usage**
   *
   * ```js
   * const banana = await page.getByRole('listitem').last();
   * ```
   *
   */
  last(): Locator;

  /**
   * The method finds an element matching the specified selector in the locator's subtree. It also accepts filter
   * options, similar to [locator.filter([options])](https://playwright.dev/docs/api/class-locator#locator-filter)
   * method.
   *
   * [Learn more about locators](https://playwright.dev/docs/locators).
   * @param selectorOrLocator A selector or locator to use when resolving DOM element.
   * @param options
   */
  locator(selectorOrLocator: string|Locator, options?: {
    /**
     * Matches elements containing an element that matches an inner locator. Inner locator is queried against the outer
     * one. For example, `article` that has `text=Playwright` matches `<article><div>Playwright</div></article>`.
     *
     * Note that outer and inner locators must belong to the same frame. Inner locator must not contain [FrameLocator]s.
     */
    has?: Locator;

    /**
     * Matches elements containing specified text somewhere inside, possibly in a child or a descendant element. When
     * passed a [string], matching is case-insensitive and searches for a substring. For example, `"Playwright"` matches
     * `<article><div>Playwright</div></article>`.
     */
    hasText?: string|RegExp;
  }): Locator;

  /**
   * Returns locator to the n-th matching element. It's zero based, `nth(0)` selects the first element.
   *
   * **Usage**
   *
   * ```js
   * const banana = await page.getByRole('listitem').nth(2);
   * ```
   *
   * @param index
   */
  nth(index: number): Locator;

  /**
   * A page this locator belongs to.
   */
  page(): Page;

  /**
   * Focuses the matching element and presses a combination of the keys.
   *
   * **Usage**
   *
   * ```js
   * await page.getByRole('textbox').press('Backspace');
   * ```
   *
   * **Details**
   *
   * Focuses the element, and then uses
   * [keyboard.down(key)](https://playwright.dev/docs/api/class-keyboard#keyboard-down) and
   * [keyboard.up(key)](https://playwright.dev/docs/api/class-keyboard#keyboard-up).
   *
   * `key` can specify the intended
   * [keyboardEvent.key](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key) value or a single character
   * to generate the text for. A superset of the `key` values can be found
   * [here](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key/Key_Values). Examples of the keys are:
   *
   * `F1` - `F12`, `Digit0`- `Digit9`, `KeyA`- `KeyZ`, `Backquote`, `Minus`, `Equal`, `Backslash`, `Backspace`, `Tab`,
   * `Delete`, `Escape`, `ArrowDown`, `End`, `Enter`, `Home`, `Insert`, `PageDown`, `PageUp`, `ArrowRight`, `ArrowUp`,
   * etc.
   *
   * Following modification shortcuts are also supported: `Shift`, `Control`, `Alt`, `Meta`, `ShiftLeft`.
   *
   * Holding down `Shift` will type the text that corresponds to the `key` in the upper case.
   *
   * If `key` is a single character, it is case-sensitive, so the values `a` and `A` will generate different respective
   * texts.
   *
   * Shortcuts such as `key: "Control+o"` or `key: "Control+Shift+T"` are supported as well. When specified with the
   * modifier, modifier is pressed and being held while the subsequent key is being pressed.
   * @param key Name of the key to press or a character to generate, such as `ArrowLeft` or `a`.
   * @param options
   */
  press(key: string, options?: {
    /**
     * Time to wait between `keydown` and `keyup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Take a screenshot of the element matching the locator.
   *
   * **Usage**
   *
   * ```js
   * await page.getByRole('link').screenshot();
   * ```
   *
   * Disable animations and save screenshot to a file:
   *
   * ```js
   * await page.getByRole('link').screenshot({ animations: 'disabled', path: 'link.png' });
   * ```
   *
   * **Details**
   *
   * This method captures a screenshot of the page, clipped to the size and position of a particular element matching
   * the locator. If the element is covered by other elements, it will not be actually visible on the screenshot. If the
   * element is a scrollable container, only the currently scrolled content will be visible on the screenshot.
   *
   * This method waits for the [actionability](https://playwright.dev/docs/actionability) checks, then scrolls element into view before taking
   * a screenshot. If the element is detached from DOM, the method throws an error.
   *
   * Returns the buffer with the captured screenshot.
   * @param options
   */
  screenshot(options?: LocatorScreenshotOptions): Promise<Buffer>;

  /**
   * This method waits for [actionability](https://playwright.dev/docs/actionability) checks, then tries to scroll element into view, unless
   * it is completely visible as defined by
   * [IntersectionObserver](https://developer.mozilla.org/en-US/docs/Web/API/Intersection_Observer_API)'s `ratio`.
   * @param options
   */
  scrollIntoViewIfNeeded(options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Selects option or options in `<select>`.
   *
   * **Details**
   *
   * This method waits for [actionability](https://playwright.dev/docs/actionability) checks, waits until all specified options are present in
   * the `<select>` element and selects these options.
   *
   * If the target element is not a `<select>` element, this method throws an error. However, if the element is inside
   * the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), the control will be used
   * instead.
   *
   * Returns the array of option values that have been successfully selected.
   *
   * Triggers a `change` and `input` event once all the provided options have been selected.
   *
   * **Usage**
   *
   * ```html
   * <select multiple>
   *   <option value="red">Red</div>
   *   <option value="green">Green</div>
   *   <option value="blue">Blue</div>
   * </select>
   * ```
   *
   * ```js
   * // single selection matching the value or label
   * element.selectOption('blue');
   *
   * // single selection matching the label
   * element.selectOption({ label: 'Blue' });
   *
   * // multiple selection for red, green and blue options
   * element.selectOption(['red', 'green', 'blue']);
   * ```
   *
   * @param values Options to select. If the `<select>` has the `multiple` attribute, all matching options are selected, otherwise
   * only the first option matching one of the passed options is selected. String values are matching both values and
   * labels. Option is considered matching if all specified properties match.
   * @param options
   */
  selectOption(values: null|string|ElementHandle|Array<string>|{
    /**
     * Matches by `option.value`. Optional.
     */
    value?: string;

    /**
     * Matches by `option.label`. Optional.
     */
    label?: string;

    /**
     * Matches by the index. Optional.
     */
    index?: number;
  }|Array<ElementHandle>|Array<{
    /**
     * Matches by `option.value`. Optional.
     */
    value?: string;

    /**
     * Matches by `option.label`. Optional.
     */
    label?: string;

    /**
     * Matches by the index. Optional.
     */
    index?: number;
  }>, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<Array<string>>;

  /**
   * This method waits for [actionability](https://playwright.dev/docs/actionability) checks, then focuses the element and selects all its
   * text content.
   *
   * If the element is inside the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), focuses and selects text in
   * the control instead.
   * @param options
   */
  selectText(options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Set the state of a checkbox or a radio element.
   *
   * **Usage**
   *
   * ```js
   * await page.getByRole('checkbox').setChecked(true);
   * ```
   *
   * **Details**
   *
   * This method checks or unchecks an element by performing the following steps:
   * 1. Ensure that matched element is a checkbox or a radio input. If not, this method throws.
   * 1. If the element already has the right checked state, this method returns immediately.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the matched element, unless `force` option is set. If
   *    the element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to click in the center of the
   *    element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now checked or unchecked. If not, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param checked Whether to check or uncheck the checkbox.
   * @param options
   */
  setChecked(checked: boolean, options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * Upload file or multiple files into `<input type=file>`.
   *
   * **Usage**
   *
   * ```js
   * // Select one file
   * await page.getByLabel('Upload file').setInputFiles('myfile.pdf');
   *
   * // Select multiple files
   * await page.getByLabel('Upload files').setInputFiles(['file1.txt', 'file2.txt']);
   *
   * // Remove all the selected files
   * await page.getByLabel('Upload file').setInputFiles([]);
   *
   * // Upload buffer from memory
   * await page.getByLabel('Upload file').setInputFiles({
   *   name: 'file.txt',
   *   mimeType: 'text/plain',
   *   buffer: Buffer.from('this is test')
   * });
   * ```
   *
   * **Details**
   *
   * Sets the value of the file input to these file paths or files. If some of the `filePaths` are relative paths, then
   * they are resolved relative to the current working directory. For empty array, clears the selected files.
   *
   * This method expects [Locator] to point to an
   * [input element](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input). However, if the element is inside
   * the `<label>` element that has an associated
   * [control](https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control), targets the control instead.
   * @param files
   * @param options
   */
  setInputFiles(files: string|Array<string>|{
    /**
     * File name
     */
    name: string;

    /**
     * File type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }|Array<{
    /**
     * File name
     */
    name: string;

    /**
     * File type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }>, options?: {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Perform a tap gesture on the element matching the locator.
   *
   * **Details**
   *
   * This method taps the element by performing the following steps:
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.touchscreen](https://playwright.dev/docs/api/class-page#page-touchscreen) to tap the center of the
   *    element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   *
   * If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   *
   * **NOTE** `element.tap()` requires that the `hasTouch` option of the browser context be set to true.
   * @param options
   */
  tap(options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores
     * current modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * Returns the [`node.textContent`](https://developer.mozilla.org/en-US/docs/Web/API/Node/textContent).
   * @param options
   */
  textContent(options?: {
    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<null|string>;

  /**
   * Focuses the element, and then sends a `keydown`, `keypress`/`input`, and `keyup` event for each character in the
   * text.
   *
   * To press a special key, like `Control` or `ArrowDown`, use
   * [locator.press(key[, options])](https://playwright.dev/docs/api/class-locator#locator-press).
   *
   * **Usage**
   *
   * ```js
   * await element.type('Hello'); // Types instantly
   * await element.type('World', {delay: 100}); // Types slower, like a user
   * ```
   *
   * An example of typing into a text field and then submitting the form:
   *
   * ```js
   * const element = page.getByLabel('Password');
   * await element.type('my password');
   * await element.press('Enter');
   * ```
   *
   * @param text A text to type into a focused element.
   * @param options
   */
  type(text: string, options?: {
    /**
     * Time to wait between key presses in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensure that checkbox or radio element is unchecked.
   *
   * **Usage**
   *
   * ```js
   * await page.getByRole('checkbox').uncheck();
   * ```
   *
   * **Details**
   *
   * This method unchecks the element by performing the following steps:
   * 1. Ensure that element is a checkbox or a radio input. If not, this method throws. If the element is already
   *    unchecked, this method returns immediately.
   * 1. Wait for [actionability](https://playwright.dev/docs/actionability) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse) to click in the center of the
   *    element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now unchecked. If not, this method throws.
   *
   * If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * When all steps combined have not finished during the specified `timeout`, this method throws a [TimeoutError].
   * Passing zero timeout disables this.
   * @param options
   */
  uncheck(options?: {
    /**
     * Whether to bypass the [actionability](https://playwright.dev/docs/actionability) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of
     * the element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;

    /**
     * When set, this method only performs the [actionability](https://playwright.dev/docs/actionability) checks and skips the action. Defaults
     * to `false`. Useful to wait until the element is ready for the action without performing it.
     */
    trial?: boolean;
  }): Promise<void>;

  /**
   * Returns when element specified by locator satisfies the `state` option.
   *
   * If target element already satisfies the condition, the method returns immediately. Otherwise, waits for up to
   * `timeout` milliseconds until the condition is met.
   *
   * **Usage**
   *
   * ```js
   * const orderSent = page.locator('#order-sent');
   * await orderSent.waitFor();
   * ```
   *
   * @param options
   */
  waitFor(options?: {
    /**
     * Defaults to `'visible'`. Can be either:
     * - `'attached'` - wait for element to be present in DOM.
     * - `'detached'` - wait for element to not be present in DOM.
     * - `'visible'` - wait for element to have non-empty bounding box and no `visibility:hidden`. Note that element
     *   without any content or with `display:none` has an empty bounding box and is not considered visible.
     * - `'hidden'` - wait for element to be either detached from DOM, or have an empty bounding box or
     *   `visibility:hidden`. This is opposite to the `'visible'` option.
     */
    state?: "attached"|"detached"|"visible"|"hidden";

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;
}

/**
 * BrowserType provides methods to launch a specific browser instance or connect to an existing one. The following is
 * a typical example of using Playwright to drive automation:
 *
 * ```js
 * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
 *
 * (async () => {
 *   const browser = await chromium.launch();
 *   const page = await browser.newPage();
 *   await page.goto('https://example.com');
 *   // other actions...
 *   await browser.close();
 * })();
 * ```
 *
 */
export interface BrowserType<Unused = {}> {
  /**
   * This method attaches Playwright to an existing browser instance using the Chrome DevTools Protocol.
   *
   * The default browser context is accessible via
   * [browser.contexts()](https://playwright.dev/docs/api/class-browser#browser-contexts).
   *
   * **NOTE** Connecting over the Chrome DevTools Protocol is only supported for Chromium-based browsers.
   *
   * **Usage**
   *
   * ```js
   * const browser = await playwright.chromium.connectOverCDP('http://localhost:9222');
   * const defaultContext = browser.contexts()[0];
   * const page = defaultContext.pages()[0];
   * ```
   *
   * @param endpointURL A CDP websocket endpoint or http url to connect to. For example `http://localhost:9222/` or
   * `ws://127.0.0.1:9222/devtools/browser/387adf4c-243f-4051-a181-46798f4a46f4`.
   * @param options
   */
  connectOverCDP(endpointURL: string, options?: ConnectOverCDPOptions): Promise<Browser>;
  /**
   * Option `wsEndpoint` is deprecated. Instead use `endpointURL`.
   * @deprecated
   */
  /**
   * This method attaches Playwright to an existing browser instance using the Chrome DevTools Protocol.
   *
   * The default browser context is accessible via
   * [browser.contexts()](https://playwright.dev/docs/api/class-browser#browser-contexts).
   *
   * **NOTE** Connecting over the Chrome DevTools Protocol is only supported for Chromium-based browsers.
   *
   * **Usage**
   *
   * ```js
   * const browser = await playwright.chromium.connectOverCDP('http://localhost:9222');
   * const defaultContext = browser.contexts()[0];
   * const page = defaultContext.pages()[0];
   * ```
   *
   * @param endpointURL A CDP websocket endpoint or http url to connect to. For example `http://localhost:9222/` or
   * `ws://127.0.0.1:9222/devtools/browser/387adf4c-243f-4051-a181-46798f4a46f4`.
   * @param options
   */
  connectOverCDP(options: ConnectOverCDPOptions & { wsEndpoint?: string }): Promise<Browser>;
  /**
   * This method attaches Playwright to an existing browser instance. When connecting to another browser launched via
   * `BrowserType.launchServer` in Node.js, the major and minor version needs to match the client version (1.2.3 → is
   * compatible with 1.2.x).
   * @param wsEndpoint A browser websocket endpoint to connect to.
   * @param options
   */
  connect(wsEndpoint: string, options?: ConnectOptions): Promise<Browser>;
  /**
   * wsEndpoint in options is deprecated. Instead use `wsEndpoint`.
   * @param wsEndpoint A browser websocket endpoint to connect to.
   * @param options
   * @deprecated
   */
  /**
   * This method attaches Playwright to an existing browser instance. When connecting to another browser launched via
   * `BrowserType.launchServer` in Node.js, the major and minor version needs to match the client version (1.2.3 → is
   * compatible with 1.2.x).
   * @param wsEndpoint A browser websocket endpoint to connect to.
   * @param options
   */
  connect(options: ConnectOptions & { wsEndpoint?: string }): Promise<Browser>;
  /**
   * A path where Playwright expects to find a bundled browser executable.
   */
  executablePath(): string;

  /**
   * Returns the browser instance.
   *
   * **Usage**
   *
   * You can use `ignoreDefaultArgs` to filter out `--mute-audio` from default arguments:
   *
   * ```js
   * const browser = await chromium.launch({  // Or 'firefox' or 'webkit'.
   *   ignoreDefaultArgs: ['--mute-audio']
   * });
   * ```
   *
   * > **Chromium-only** Playwright can also be used to control the Google Chrome or Microsoft Edge browsers, but it
   * works best with the version of Chromium it is bundled with. There is no guarantee it will work with any other
   * version. Use `executablePath` option with extreme caution.
   * >
   * > If Google Chrome (rather than Chromium) is preferred, a
   * [Chrome Canary](https://www.google.com/chrome/browser/canary.html) or
   * [Dev Channel](https://www.chromium.org/getting-involved/dev-channel) build is suggested.
   * >
   * > Stock browsers like Google Chrome and Microsoft Edge are suitable for tests that require proprietary media codecs
   * for video playback. See
   * [this article](https://www.howtogeek.com/202825/what%E2%80%99s-the-difference-between-chromium-and-chrome/) for
   * other differences between Chromium and Chrome.
   * [This article](https://chromium.googlesource.com/chromium/src/+/lkgr/docs/chromium_browser_vs_google_chrome.md)
   * describes some differences for Linux users.
   * @param options
   */
  launch(options?: LaunchOptions): Promise<Browser>;

  /**
   * Returns the persistent browser context instance.
   *
   * Launches browser that uses persistent storage located at `userDataDir` and returns the only context. Closing this
   * context will automatically close the browser.
   * @param userDataDir Path to a User Data Directory, which stores browser session data like cookies and local storage. More details for
   * [Chromium](https://chromium.googlesource.com/chromium/src/+/master/docs/user_data_dir.md#introduction) and
   * [Firefox](https://developer.mozilla.org/en-US/docs/Mozilla/Command_Line_Options#User_Profile). Note that Chromium's
   * user data directory is the **parent** directory of the "Profile Path" seen at `chrome://version`. Pass an empty
   * string to use a temporary directory instead.
   * @param options
   */
  launchPersistentContext(userDataDir: string, options?: {
    /**
     * Whether to automatically download all the attachments. Defaults to `true` where all the downloads are accepted.
     */
    acceptDownloads?: boolean;

    /**
     * Additional arguments to pass to the browser instance. The list of Chromium flags can be found
     * [here](http://peter.sh/experiments/chromium-command-line-switches/).
     */
    args?: Array<string>;

    /**
     * When using [page.goto(url[, options])](https://playwright.dev/docs/api/class-page#page-goto),
     * [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route),
     * [page.waitForURL(url[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-url),
     * [page.waitForRequest(urlOrPredicate[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-request),
     * or
     * [page.waitForResponse(urlOrPredicate[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-response)
     * it takes the base URL in consideration by using the
     * [`URL()`](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL) constructor for building the corresponding URL.
     * Examples:
     * - baseURL: `http://localhost:3000` and navigating to `/bar.html` results in `http://localhost:3000/bar.html`
     * - baseURL: `http://localhost:3000/foo/` and navigating to `./bar.html` results in
     *   `http://localhost:3000/foo/bar.html`
     * - baseURL: `http://localhost:3000/foo` (without trailing slash) and navigating to `./bar.html` results in
     *   `http://localhost:3000/bar.html`
     */
    baseURL?: string;

    /**
     * Toggles bypassing page's Content-Security-Policy.
     */
    bypassCSP?: boolean;

    /**
     * Browser distribution channel.  Supported values are "chrome", "chrome-beta", "chrome-dev", "chrome-canary",
     * "msedge", "msedge-beta", "msedge-dev", "msedge-canary". Read more about using
     * [Google Chrome and Microsoft Edge](https://playwright.dev/docs/browsers#google-chrome--microsoft-edge).
     */
    channel?: string;

    /**
     * Enable Chromium sandboxing. Defaults to `false`.
     */
    chromiumSandbox?: boolean;

    /**
     * Emulates `'prefers-colors-scheme'` media feature, supported values are `'light'`, `'dark'`, `'no-preference'`. See
     * [page.emulateMedia([options])](https://playwright.dev/docs/api/class-page#page-emulate-media) for more details.
     * Passing `null` resets emulation to system defaults. Defaults to `'light'`.
     */
    colorScheme?: null|"light"|"dark"|"no-preference";

    /**
     * Specify device scale factor (can be thought of as dpr). Defaults to `1`.
     */
    deviceScaleFactor?: number;

    /**
     * **Chromium-only** Whether to auto-open a Developer Tools panel for each tab. If this option is `true`, the
     * `headless` option will be set `false`.
     */
    devtools?: boolean;

    /**
     * If specified, accepted downloads are downloaded into this directory. Otherwise, temporary directory is created and
     * is deleted when browser is closed. In either case, the downloads are deleted when the browser context they were
     * created in is closed.
     */
    downloadsPath?: string;

    /**
     * Specify environment variables that will be visible to the browser. Defaults to `process.env`.
     */
    env?: { [key: string]: string|number|boolean; };

    /**
     * Path to a browser executable to run instead of the bundled one. If `executablePath` is a relative path, then it is
     * resolved relative to the current working directory. Note that Playwright only works with the bundled Chromium,
     * Firefox or WebKit, use at your own risk.
     */
    executablePath?: string;

    /**
     * An object containing additional HTTP headers to be sent with every request.
     */
    extraHTTPHeaders?: { [key: string]: string; };

    /**
     * Emulates `'forced-colors'` media feature, supported values are `'active'`, `'none'`. See
     * [page.emulateMedia([options])](https://playwright.dev/docs/api/class-page#page-emulate-media) for more details.
     * Passing `null` resets emulation to system defaults. Defaults to `'none'`.
     */
    forcedColors?: null|"active"|"none";

    geolocation?: {
      /**
       * Latitude between -90 and 90.
       */
      latitude: number;

      /**
       * Longitude between -180 and 180.
       */
      longitude: number;

      /**
       * Non-negative accuracy value. Defaults to `0`.
       */
      accuracy?: number;
    };

    /**
     * Close the browser process on SIGHUP. Defaults to `true`.
     */
    handleSIGHUP?: boolean;

    /**
     * Close the browser process on Ctrl-C. Defaults to `true`.
     */
    handleSIGINT?: boolean;

    /**
     * Close the browser process on SIGTERM. Defaults to `true`.
     */
    handleSIGTERM?: boolean;

    /**
     * Specifies if viewport supports touch events. Defaults to false.
     */
    hasTouch?: boolean;

    /**
     * Whether to run browser in headless mode. More details for
     * [Chromium](https://developers.google.com/web/updates/2017/04/headless-chrome) and
     * [Firefox](https://developer.mozilla.org/en-US/docs/Mozilla/Firefox/Headless_mode). Defaults to `true` unless the
     * `devtools` option is `true`.
     */
    headless?: boolean;

    /**
     * Credentials for [HTTP authentication](https://developer.mozilla.org/en-US/docs/Web/HTTP/Authentication).
     */
    httpCredentials?: {
      username: string;

      password: string;
    };

    /**
     * If `true`, Playwright does not pass its own configurations args and only uses the ones from `args`. If an array is
     * given, then filters out the given default arguments. Dangerous option; use with care. Defaults to `false`.
     */
    ignoreDefaultArgs?: boolean|Array<string>;

    /**
     * Whether to ignore HTTPS errors when sending network requests. Defaults to `false`.
     */
    ignoreHTTPSErrors?: boolean;

    /**
     * Whether the `meta viewport` tag is taken into account and touch events are enabled. Defaults to `false`. Not
     * supported in Firefox.
     */
    isMobile?: boolean;

    /**
     * Whether or not to enable JavaScript in the context. Defaults to `true`.
     */
    javaScriptEnabled?: boolean;

    /**
     * Specify user locale, for example `en-GB`, `de-DE`, etc. Locale will affect `navigator.language` value,
     * `Accept-Language` request header value as well as number and date formatting rules.
     */
    locale?: string;

    /**
     * Logger sink for Playwright logging.
     */
    logger?: Logger;

    /**
     * Whether to emulate network being offline. Defaults to `false`.
     */
    offline?: boolean;

    /**
     * A list of permissions to grant to all pages in this context. See
     * [browserContext.grantPermissions(permissions[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-grant-permissions)
     * for more details.
     */
    permissions?: Array<string>;

    /**
     * Network proxy settings.
     */
    proxy?: {
      /**
       * Proxy to be used for all requests. HTTP and SOCKS proxies are supported, for example `http://myproxy.com:3128` or
       * `socks5://myproxy.com:3128`. Short form `myproxy.com:3128` is considered an HTTP proxy.
       */
      server: string;

      /**
       * Optional comma-separated domains to bypass proxy, for example `".com, chromium.org, .domain.com"`.
       */
      bypass?: string;

      /**
       * Optional username to use if HTTP proxy requires authentication.
       */
      username?: string;

      /**
       * Optional password to use if HTTP proxy requires authentication.
       */
      password?: string;
    };

    /**
     * Enables [HAR](http://www.softwareishard.com/blog/har-12-spec) recording for all pages into `recordHar.path` file.
     * If not specified, the HAR is not recorded. Make sure to await
     * [browserContext.close()](https://playwright.dev/docs/api/class-browsercontext#browser-context-close) for the HAR to
     * be saved.
     */
    recordHar?: {
      /**
       * Optional setting to control whether to omit request content from the HAR. Defaults to `false`. Deprecated, use
       * `content` policy instead.
       */
      omitContent?: boolean;

      /**
       * Optional setting to control resource content management. If `omit` is specified, content is not persisted. If
       * `attach` is specified, resources are persisted as separate files or entries in the ZIP archive. If `embed` is
       * specified, content is stored inline the HAR file as per HAR specification. Defaults to `attach` for `.zip` output
       * files and to `embed` for all other file extensions.
       */
      content?: "omit"|"embed"|"attach";

      /**
       * Path on the filesystem to write the HAR file to. If the file name ends with `.zip`, `content: 'attach'` is used by
       * default.
       */
      path: string;

      /**
       * When set to `minimal`, only record information necessary for routing from HAR. This omits sizes, timing, page,
       * cookies, security and other types of HAR information that are not used when replaying from HAR. Defaults to `full`.
       */
      mode?: "full"|"minimal";

      /**
       * A glob or regex pattern to filter requests that are stored in the HAR. When a `baseURL` via the context options was
       * provided and the passed URL is a path, it gets merged via the
       * [`new URL()`](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL) constructor.
       */
      urlFilter?: string|RegExp;
    };

    /**
     * Enables video recording for all pages into `recordVideo.dir` directory. If not specified videos are not recorded.
     * Make sure to await
     * [browserContext.close()](https://playwright.dev/docs/api/class-browsercontext#browser-context-close) for videos to
     * be saved.
     */
    recordVideo?: {
      /**
       * Path to the directory to put videos into.
       */
      dir: string;

      /**
       * Optional dimensions of the recorded videos. If not specified the size will be equal to `viewport` scaled down to
       * fit into 800x800. If `viewport` is not configured explicitly the video size defaults to 800x450. Actual picture of
       * each page will be scaled down if necessary to fit the specified size.
       */
      size?: {
        /**
         * Video frame width.
         */
        width: number;

        /**
         * Video frame height.
         */
        height: number;
      };
    };

    /**
     * Emulates `'prefers-reduced-motion'` media feature, supported values are `'reduce'`, `'no-preference'`. See
     * [page.emulateMedia([options])](https://playwright.dev/docs/api/class-page#page-emulate-media) for more details.
     * Passing `null` resets emulation to system defaults. Defaults to `'no-preference'`.
     */
    reducedMotion?: null|"reduce"|"no-preference";

    /**
     * Emulates consistent window screen size available inside web page via `window.screen`. Is only used when the
     * `viewport` is set.
     */
    screen?: {
      /**
       * page width in pixels.
       */
      width: number;

      /**
       * page height in pixels.
       */
      height: number;
    };

    /**
     * Whether to allow sites to register Service workers. Defaults to `'allow'`.
     * - `'allow'`: [Service Workers](https://developer.mozilla.org/en-US/docs/Web/API/Service_Worker_API) can be
     *   registered.
     * - `'block'`: Playwright will block all registration of Service Workers.
     */
    serviceWorkers?: "allow"|"block";

    /**
     * Slows down Playwright operations by the specified amount of milliseconds. Useful so that you can see what is going
     * on.
     */
    slowMo?: number;

    /**
     * If set to true, enables strict selectors mode for this context. In the strict selectors mode all operations on
     * selectors that imply single target DOM element will throw when more than one element matches the selector. This
     * option does not affect any Locator APIs (Locators are always strict). See [Locator] to learn more about the strict
     * mode.
     */
    strictSelectors?: boolean;

    /**
     * Maximum time in milliseconds to wait for the browser instance to start. Defaults to `30000` (30 seconds). Pass `0`
     * to disable timeout.
     */
    timeout?: number;

    /**
     * Changes the timezone of the context. See
     * [ICU's metaZones.txt](https://cs.chromium.org/chromium/src/third_party/icu/source/data/misc/metaZones.txt?rcl=faee8bc70570192d82d2978a71e2a615788597d1)
     * for a list of supported timezone IDs.
     */
    timezoneId?: string;

    /**
     * If specified, traces are saved into this directory.
     */
    tracesDir?: string;

    /**
     * Specific user agent to use in this context.
     */
    userAgent?: string;

    /**
     * @deprecated Use `recordVideo` instead.
     */
    videoSize?: {
      /**
       * Video frame width.
       */
      width: number;

      /**
       * Video frame height.
       */
      height: number;
    };

    /**
     * @deprecated Use `recordVideo` instead.
     */
    videosPath?: string;

    /**
     * Emulates consistent viewport for each page. Defaults to an 1280x720 viewport. Use `null` to disable the consistent
     * viewport emulation.
     *
     * **NOTE** The `null` value opts out from the default presets, makes viewport depend on the host window size defined
     * by the operating system. It makes the execution of the tests non-deterministic.
     */
    viewport?: null|{
      /**
       * page width in pixels.
       */
      width: number;

      /**
       * page height in pixels.
       */
      height: number;
    };
  }): Promise<BrowserContext>;

  /**
   * Returns the browser app instance. You can connect to it via
   * [browserType.connect(wsEndpoint[, options])](https://playwright.dev/docs/api/class-browsertype#browser-type-connect),
   * which requires the major/minor client/server version to match (1.2.3 → is compatible with 1.2.x).
   *
   * **Usage**
   *
   * Launches browser server that client can connect to. An example of launching a browser executable and connecting to
   * it later:
   *
   * ```js
   * const { chromium } = require('playwright');  // Or 'webkit' or 'firefox'.
   *
   * (async () => {
   *   const browserServer = await chromium.launchServer();
   *   const wsEndpoint = browserServer.wsEndpoint();
   *   // Use web socket endpoint later to establish a connection.
   *   const browser = await chromium.connect(wsEndpoint);
   *   // Close browser instance.
   *   await browserServer.close();
   * })();
   * ```
   *
   * @param options
   */
  launchServer(options?: {
    /**
     * Additional arguments to pass to the browser instance. The list of Chromium flags can be found
     * [here](http://peter.sh/experiments/chromium-command-line-switches/).
     */
    args?: Array<string>;

    /**
     * Browser distribution channel.  Supported values are "chrome", "chrome-beta", "chrome-dev", "chrome-canary",
     * "msedge", "msedge-beta", "msedge-dev", "msedge-canary". Read more about using
     * [Google Chrome and Microsoft Edge](https://playwright.dev/docs/browsers#google-chrome--microsoft-edge).
     */
    channel?: string;

    /**
     * Enable Chromium sandboxing. Defaults to `false`.
     */
    chromiumSandbox?: boolean;

    /**
     * **Chromium-only** Whether to auto-open a Developer Tools panel for each tab. If this option is `true`, the
     * `headless` option will be set `false`.
     */
    devtools?: boolean;

    /**
     * If specified, accepted downloads are downloaded into this directory. Otherwise, temporary directory is created and
     * is deleted when browser is closed. In either case, the downloads are deleted when the browser context they were
     * created in is closed.
     */
    downloadsPath?: string;

    /**
     * Specify environment variables that will be visible to the browser. Defaults to `process.env`.
     */
    env?: { [key: string]: string|number|boolean; };

    /**
     * Path to a browser executable to run instead of the bundled one. If `executablePath` is a relative path, then it is
     * resolved relative to the current working directory. Note that Playwright only works with the bundled Chromium,
     * Firefox or WebKit, use at your own risk.
     */
    executablePath?: string;

    /**
     * Firefox user preferences. Learn more about the Firefox user preferences at
     * [`about:config`](https://support.mozilla.org/en-US/kb/about-config-editor-firefox).
     */
    firefoxUserPrefs?: { [key: string]: string|number|boolean; };

    /**
     * Close the browser process on SIGHUP. Defaults to `true`.
     */
    handleSIGHUP?: boolean;

    /**
     * Close the browser process on Ctrl-C. Defaults to `true`.
     */
    handleSIGINT?: boolean;

    /**
     * Close the browser process on SIGTERM. Defaults to `true`.
     */
    handleSIGTERM?: boolean;

    /**
     * Whether to run browser in headless mode. More details for
     * [Chromium](https://developers.google.com/web/updates/2017/04/headless-chrome) and
     * [Firefox](https://developer.mozilla.org/en-US/docs/Mozilla/Firefox/Headless_mode). Defaults to `true` unless the
     * `devtools` option is `true`.
     */
    headless?: boolean;

    /**
     * If `true`, Playwright does not pass its own configurations args and only uses the ones from `args`. If an array is
     * given, then filters out the given default arguments. Dangerous option; use with care. Defaults to `false`.
     */
    ignoreDefaultArgs?: boolean|Array<string>;

    /**
     * Logger sink for Playwright logging.
     */
    logger?: Logger;

    /**
     * Port to use for the web socket. Defaults to 0 that picks any available port.
     */
    port?: number;

    /**
     * Network proxy settings.
     */
    proxy?: {
      /**
       * Proxy to be used for all requests. HTTP and SOCKS proxies are supported, for example `http://myproxy.com:3128` or
       * `socks5://myproxy.com:3128`. Short form `myproxy.com:3128` is considered an HTTP proxy.
       */
      server: string;

      /**
       * Optional comma-separated domains to bypass proxy, for example `".com, chromium.org, .domain.com"`.
       */
      bypass?: string;

      /**
       * Optional username to use if HTTP proxy requires authentication.
       */
      username?: string;

      /**
       * Optional password to use if HTTP proxy requires authentication.
       */
      password?: string;
    };

    /**
     * Maximum time in milliseconds to wait for the browser instance to start. Defaults to `30000` (30 seconds). Pass `0`
     * to disable timeout.
     */
    timeout?: number;

    /**
     * If specified, traces are saved into this directory.
     */
    tracesDir?: string;

    /**
     * Path at which to serve the Browser Server. For security, this defaults to an unguessable string.
     *
     * **NOTE** Any process or web page (including those running in Playwright) with knowledge of the `wsPath` can take
     * control of the OS user. For this reason, you should use an unguessable token when using this option.
     */
    wsPath?: string;
  }): Promise<BrowserServer>;

  /**
   * Returns browser name. For example: `'chromium'`, `'webkit'` or `'firefox'`.
   */
  name(): string;
}

/**
 * - extends: [EventEmitter]
 *
 * The `CDPSession` instances are used to talk raw Chrome Devtools Protocol:
 * - protocol methods can be called with `session.send` method.
 * - protocol events can be subscribed to with `session.on` method.
 *
 * Useful links:
 * - Documentation on DevTools Protocol can be found here:
 *   [DevTools Protocol Viewer](https://chromedevtools.github.io/devtools-protocol/).
 * - Getting Started with DevTools Protocol:
 *   https://github.com/aslushnikov/getting-started-with-cdp/blob/master/README.md
 *
 * ```js
 * const client = await page.context().newCDPSession(page);
 * await client.send('Animation.enable');
 * client.on('Animation.animationCreated', () => console.log('Animation created!'));
 * const response = await client.send('Animation.getPlaybackRate');
 * console.log('playback rate is ' + response.playbackRate);
 * await client.send('Animation.setPlaybackRate', {
 *   playbackRate: response.playbackRate / 2
 * });
 * ```
 *
 */
export interface CDPSession {
  on: <T extends keyof Protocol.Events | symbol>(event: T, listener: (payload: T extends symbol ? any : Protocol.Events[T extends keyof Protocol.Events ? T : never]) => void) => this;
  addListener: <T extends keyof Protocol.Events | symbol>(event: T, listener: (payload: T extends symbol ? any : Protocol.Events[T extends keyof Protocol.Events ? T : never]) => void) => this;
  off: <T extends keyof Protocol.Events | symbol>(event: T, listener: (payload: T extends symbol ? any : Protocol.Events[T extends keyof Protocol.Events ? T : never]) => void) => this;
  removeListener: <T extends keyof Protocol.Events | symbol>(event: T, listener: (payload: T extends symbol ? any : Protocol.Events[T extends keyof Protocol.Events ? T : never]) => void) => this;
  once: <T extends keyof Protocol.Events | symbol>(event: T, listener: (payload: T extends symbol ? any : Protocol.Events[T extends keyof Protocol.Events ? T : never]) => void) => this;
  /**
   * @param method Protocol method name.
   * @param params Optional method parameters.
   */
  send<T extends keyof Protocol.CommandParameters>(
    method: T,
    params?: Protocol.CommandParameters[T]
  ): Promise<Protocol.CommandReturnValues[T]>;
  /**
   * Detaches the CDPSession from the target. Once detached, the CDPSession object won't emit any events and can't be
   * used to send messages.
   */
  detach(): Promise<void>;
}

type DeviceDescriptor = {
  viewport: ViewportSize;
  userAgent: string;
  deviceScaleFactor: number;
  isMobile: boolean;
  hasTouch: boolean;
  defaultBrowserType: 'chromium' | 'firefox' | 'webkit';
};

export namespace errors {

/**
 * - extends: [Error]
 *
 * TimeoutError is emitted whenever certain operations are terminated due to timeout, e.g.
 * [locator.waitFor([options])](https://playwright.dev/docs/api/class-locator#locator-wait-for) or
 * [browserType.launch([options])](https://playwright.dev/docs/api/class-browsertype#browser-type-launch).
 *
 * ```js
 * const playwright = require('playwright');
 *
 * (async () => {
 *   const browser = await playwright.chromium.launch();
 *   const context = await browser.newContext();
 *   const page = await context.newPage();
 *   try {
 *     await page.locator("text=Foo").click({
 *       timeout: 100,
 *     })
 *   } catch (error) {
 *     if (error instanceof playwright.errors.TimeoutError)
 *       console.log("Timeout!")
 *   }
 *   await browser.close();
 * })();
 * ```
 *
 */
class TimeoutError extends Error {
}

}

/**
 * The Accessibility class provides methods for inspecting Chromium's accessibility tree. The accessibility tree is
 * used by assistive technology such as [screen readers](https://en.wikipedia.org/wiki/Screen_reader) or
 * [switches](https://en.wikipedia.org/wiki/Switch_access).
 *
 * Accessibility is a very platform-specific thing. On different platforms, there are different screen readers that
 * might have wildly different output.
 *
 * Rendering engines of Chromium, Firefox and WebKit have a concept of "accessibility tree", which is then translated
 * into different platform-specific APIs. Accessibility namespace gives access to this Accessibility Tree.
 *
 * Most of the accessibility tree gets filtered out when converting from internal browser AX Tree to Platform-specific
 * AX-Tree or by assistive technologies themselves. By default, Playwright tries to approximate this filtering,
 * exposing only the "interesting" nodes of the tree.
 */
export interface Accessibility {
  /**
   * Captures the current state of the accessibility tree. The returned object represents the root accessible node of
   * the page.
   *
   * **NOTE** The Chromium accessibility tree contains nodes that go unused on most platforms and by most screen
   * readers. Playwright will discard them as well for an easier to process tree, unless `interestingOnly` is set to
   * `false`.
   *
   * **Usage**
   *
   * An example of dumping the entire accessibility tree:
   *
   * ```js
   * const snapshot = await page.accessibility.snapshot();
   * console.log(snapshot);
   * ```
   *
   * An example of logging the focused node's name:
   *
   * ```js
   * const snapshot = await page.accessibility.snapshot();
   * const node = findFocusedNode(snapshot);
   * console.log(node && node.name);
   *
   * function findFocusedNode(node) {
   *   if (node.focused)
   *     return node;
   *   for (const child of node.children || []) {
   *     const foundNode = findFocusedNode(child);
   *     if (foundNode)
   *       return foundNode;
   *   }
   *   return null;
   * }
   * ```
   *
   * @deprecated This method is deprecated. Please use other libraries such as [Axe](https://www.deque.com/axe/) if you need to test
   * page accessibility. See our Node.js [guide](https://playwright.dev/docs/accessibility-testing) for integration with
   * Axe.
   * @param options
   */
  snapshot(options?: AccessibilitySnapshotOptions): Promise<null|AccessibilityNode>;

}

type AccessibilityNode = {
  role: string;
  name: string;
  value?: string|number;
  description?: string;
  keyshortcuts?: string;
  roledescription?: string;
  valuetext?: string;
  disabled?: boolean;
  expanded?: boolean;
  focused?: boolean;
  modal?: boolean;
  multiline?: boolean;
  multiselectable?: boolean;
  readonly?: boolean;
  required?: boolean;
  selected?: boolean;
  checked?: boolean|"mixed";
  pressed?: boolean|"mixed";
  level?: number;
  valuemin?: number;
  valuemax?: number;
  autocomplete?: string;
  haspopup?: string;
  invalid?: string;
  orientation?: string;
  children?: AccessibilityNode[];
}

export const devices: Devices;

//@ts-ignore this will be any if electron is not installed
type ElectronType = typeof import('electron');

/**
 * Electron application representation. You can use
 * [electron.launch([options])](https://playwright.dev/docs/api/class-electron#electron-launch) to obtain the
 * application instance. This instance you can control main electron process as well as work with Electron windows:
 *
 * ```js
 * const { _electron: electron } = require('playwright');
 *
 * (async () => {
 *   // Launch Electron app.
 *   const electronApp = await electron.launch({ args: ['main.js'] });
 *
 *   // Evaluation expression in the Electron context.
 *   const appPath = await electronApp.evaluate(async ({ app }) => {
 *     // This runs in the main Electron process, parameter here is always
 *     // the result of the require('electron') in the main app script.
 *     return app.getAppPath();
 *   });
 *   console.log(appPath);
 *
 *   // Get the first window that the app opens, wait if necessary.
 *   const window = await electronApp.firstWindow();
 *   // Print the title.
 *   console.log(await window.title());
 *   // Capture a screenshot.
 *   await window.screenshot({ path: 'intro.png' });
 *   // Direct Electron console to Node terminal.
 *   window.on('console', console.log);
 *   // Click button.
 *   await window.click('text=Click me');
 *   // Exit app.
 *   await electronApp.close();
 * })();
 * ```
 *
 */
export interface ElectronApplication {
  /**
   * Returns the return value of `pageFunction`.
   *
   * If the function passed to the
   * [electronApplication.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate)
   * returns a [Promise], then
   * [electronApplication.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate)
   * would wait for the promise to resolve and return its value.
   *
   * If the function passed to the
   * [electronApplication.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate)
   * returns a non-[Serializable] value, then
   * [electronApplication.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate)
   * returns `undefined`. Playwright also supports transferring some additional values that are not serializable by
   * `JSON`: `-0`, `NaN`, `Infinity`, `-Infinity`.
   * @param pageFunction Function to be evaluated in the main Electron process.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluate<R, Arg>(pageFunction: PageFunctionOn<ElectronType, Arg, R>, arg: Arg): Promise<R>;
  /**
   * Returns the return value of `pageFunction`.
   *
   * If the function passed to the
   * [electronApplication.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate)
   * returns a [Promise], then
   * [electronApplication.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate)
   * would wait for the promise to resolve and return its value.
   *
   * If the function passed to the
   * [electronApplication.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate)
   * returns a non-[Serializable] value, then
   * [electronApplication.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate)
   * returns `undefined`. Playwright also supports transferring some additional values that are not serializable by
   * `JSON`: `-0`, `NaN`, `Infinity`, `-Infinity`.
   * @param pageFunction Function to be evaluated in the main Electron process.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluate<R>(pageFunction: PageFunctionOn<ElectronType, void, R>, arg?: any): Promise<R>;

  /**
   * Returns the return value of `pageFunction` as a [JSHandle].
   *
   * The only difference between
   * [electronApplication.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate)
   * and
   * [electronApplication.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate-handle)
   * is that
   * [electronApplication.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate-handle)
   * returns [JSHandle].
   *
   * If the function passed to the
   * [electronApplication.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate-handle)
   * returns a [Promise], then
   * [electronApplication.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate-handle)
   * would wait for the promise to resolve and return its value.
   * @param pageFunction Function to be evaluated in the main Electron process.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluateHandle<R, Arg>(pageFunction: PageFunctionOn<ElectronType, Arg, R>, arg: Arg): Promise<SmartHandle<R>>;
  /**
   * Returns the return value of `pageFunction` as a [JSHandle].
   *
   * The only difference between
   * [electronApplication.evaluate(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate)
   * and
   * [electronApplication.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate-handle)
   * is that
   * [electronApplication.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate-handle)
   * returns [JSHandle].
   *
   * If the function passed to the
   * [electronApplication.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate-handle)
   * returns a [Promise], then
   * [electronApplication.evaluateHandle(pageFunction[, arg])](https://playwright.dev/docs/api/class-electronapplication#electron-application-evaluate-handle)
   * would wait for the promise to resolve and return its value.
   * @param pageFunction Function to be evaluated in the main Electron process.
   * @param arg Optional argument to pass to `pageFunction`.
   */
  evaluateHandle<R>(pageFunction: PageFunctionOn<ElectronType, void, R>, arg?: any): Promise<SmartHandle<R>>;
  /**
   * This event is issued when the application closes.
   */
  on(event: 'close', listener: () => void): this;

  /**
   * This event is issued for every window that is created **and loaded** in Electron. It contains a [Page] that can be
   * used for Playwright automation.
   */
  on(event: 'window', listener: (page: Page) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'close', listener: () => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'window', listener: (page: Page) => void): this;

  /**
   * This event is issued when the application closes.
   */
  addListener(event: 'close', listener: () => void): this;

  /**
   * This event is issued for every window that is created **and loaded** in Electron. It contains a [Page] that can be
   * used for Playwright automation.
   */
  addListener(event: 'window', listener: (page: Page) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'close', listener: () => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'window', listener: (page: Page) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'close', listener: () => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'window', listener: (page: Page) => void): this;

  /**
   * This event is issued when the application closes.
   */
  prependListener(event: 'close', listener: () => void): this;

  /**
   * This event is issued for every window that is created **and loaded** in Electron. It contains a [Page] that can be
   * used for Playwright automation.
   */
  prependListener(event: 'window', listener: (page: Page) => void): this;

  /**
   * Returns the BrowserWindow object that corresponds to the given Playwright page.
   * @param page Page to retrieve the window for.
   */
  browserWindow(page: Page): Promise<JSHandle>;

  /**
   * Closes Electron application.
   */
  close(): Promise<void>;

  /**
   * This method returns browser context that can be used for setting up context-wide routing, etc.
   */
  context(): BrowserContext;

  /**
   * Convenience method that waits for the first application window to be opened.
   *
   * **Usage**
   *
   * ```js
   *   const electronApp = await electron.launch({
   *     args: ['main.js']
   *   });
   *   const window = await electronApp.firstWindow();
   *   // ...
   * ```
   *
   */
  firstWindow(): Promise<Page>;

  /**
   * Returns the main process for this Electron Application.
   */
  process(): ChildProcess;

  /**
   * This event is issued when the application closes.
   */
  waitForEvent(event: 'close', optionsOrPredicate?: { predicate?: () => boolean | Promise<boolean>, timeout?: number } | (() => boolean | Promise<boolean>)): Promise<void>;

  /**
   * This event is issued for every window that is created **and loaded** in Electron. It contains a [Page] that can be
   * used for Playwright automation.
   */
  waitForEvent(event: 'window', optionsOrPredicate?: { predicate?: (page: Page) => boolean | Promise<boolean>, timeout?: number } | ((page: Page) => boolean | Promise<boolean>)): Promise<Page>;


  /**
   * Convenience method that returns all the opened windows.
   */
  windows(): Array<Page>;
}

export type AndroidElementInfo = {
  clazz: string;
  desc: string;
  res: string;
  pkg: string;
  text: string;
  bounds: { x: number, y: number, width: number, height: number };
  checkable: boolean;
  checked: boolean;
  clickable: boolean;
  enabled: boolean;
  focusable: boolean;
  focused: boolean;
  longClickable: boolean;
  scrollable: boolean;
  selected: boolean;
};

export type AndroidSelector = {
  checkable?: boolean,
  checked?: boolean,
  clazz?: string | RegExp,
  clickable?: boolean,
  depth?: number,
  desc?: string | RegExp,
  enabled?: boolean,
  focusable?: boolean,
  focused?: boolean,
  hasChild?: { selector: AndroidSelector },
  hasDescendant?: { selector: AndroidSelector, maxDepth?: number },
  longClickable?: boolean,
  pkg?: string | RegExp,
  res?: string | RegExp,
  scrollable?: boolean,
  selected?: boolean,
  text?: string | RegExp,
};

export type AndroidKey =
  'Unknown' |
  'SoftLeft' | 'SoftRight' |
  'Home' |
  'Back' |
  'Call' | 'EndCall' |
  '0' |  '1' |  '2' |  '3' |  '4' |  '5' |  '6' |  '7' |  '8' |  '9' |
  'Star' | 'Pound' | '*' | '#' |
  'DialUp' | 'DialDown' | 'DialLeft' | 'DialRight' | 'DialCenter' |
  'VolumeUp' | 'VolumeDown' |
  'Power' |
  'Camera' |
  'Clear' |
  'A' | 'B' | 'C' | 'D' | 'E' | 'F' | 'G' | 'H' | 'I' | 'J' | 'K' | 'L' | 'M' |
  'N' | 'O' | 'P' | 'Q' | 'R' | 'S' | 'T' | 'U' | 'V' | 'W' | 'X' | 'Y' | 'Z' |
  'Comma' | ',' |
  'Period' | '.' |
  'AltLeft' | 'AltRight' |
  'ShiftLeft' | 'ShiftRight' |
  'Tab' | '\t' |
  'Space' | ' ' |
  'Sym' |
  'Explorer' |
  'Envelop' |
  'Enter' | '\n' |
  'Del' |
  'Grave' |
  'Minus' | '-' |
  'Equals' | '=' |
  'LeftBracket' | '(' |
  'RightBracket' | ')' |
  'Backslash' | '\\' |
  'Semicolon' | ';' |
  'Apostrophe' | '`' |
  'Slash' | '/' |
  'At' | '@' |
  'Num' |
  'HeadsetHook' |
  'Focus' |
  'Plus' | '+' |
  'Menu' |
  'Notification' |
  'Search' |
  'RecentApps' |
  'AppSwitch' |
  'Assist' |
  'Cut' |
  'Copy' |
  'Paste';

export const _electron: Electron;
export const _android: Android;

// This is required to not export everything by default. See https://github.com/Microsoft/TypeScript/issues/19545#issuecomment-340490459
export {};


/**
 * Playwright has **experimental** support for Android automation. This includes Chrome for Android and Android
 * WebView.
 *
 * *Requirements*
 * - Android device or AVD Emulator.
 * - [ADB daemon](https://developer.android.com/studio/command-line/adb) running and authenticated with your device.
 *   Typically running `adb devices` is all you need to do.
 * - [`Chrome 87`](https://play.google.com/store/apps/details?id=com.android.chrome) or newer installed on the
 *   device
 * - "Enable command line on non-rooted devices" enabled in `chrome://flags`.
 *
 * *Known limitations*
 * - Raw USB operation is not yet supported, so you need ADB.
 * - Device needs to be awake to produce screenshots. Enabling "Stay awake" developer mode will help.
 * - We didn't run all the tests against the device, so not everything works.
 *
 * *How to run*
 *
 * An example of the Android automation script would be:
 *
 * ```js
 * const { _android: android } = require('playwright');
 *
 * (async () => {
 *   // Connect to the device.
 *   const [device] = await android.devices();
 *   console.log(`Model: ${device.model()}`);
 *   console.log(`Serial: ${device.serial()}`);
 *   // Take screenshot of the whole device.
 *   await device.screenshot({ path: 'device.png' });
 *
 *   {
 *     // --------------------- WebView -----------------------
 *
 *     // Launch an application with WebView.
 *     await device.shell('am force-stop org.chromium.webview_shell');
 *     await device.shell('am start org.chromium.webview_shell/.WebViewBrowserActivity');
 *     // Get the WebView.
 *     const webview = await device.webView({ pkg: 'org.chromium.webview_shell' });
 *
 *     // Fill the input box.
 *     await device.fill({ res: 'org.chromium.webview_shell:id/url_field' }, 'github.com/microsoft/playwright');
 *     await device.press({ res: 'org.chromium.webview_shell:id/url_field' }, 'Enter');
 *
 *     // Work with WebView's page as usual.
 *     const page = await webview.page();
 *     await page.waitForNavigation({ url: /.*microsoft\/playwright.*\/ });
 *     console.log(await page.title());
 *   }
 *
 *   {
 *     // --------------------- Browser -----------------------
 *
 *     // Launch Chrome browser.
 *     await device.shell('am force-stop com.android.chrome');
 *     const context = await device.launchBrowser();
 *
 *     // Use BrowserContext as usual.
 *     const page = await context.newPage();
 *     await page.goto('https://webkit.org/');
 *     console.log(await page.evaluate(() => window.location.href));
 *     await page.screenshot({ path: 'page.png' });
 *
 *     await context.close();
 *   }
 *
 *   // Close the device.
 *   await device.close();
 * })();
 * ```
 *
 * Note that since you don't need Playwright to install web browsers when testing Android, you can omit browser
 * download via setting the following environment variable when installing Playwright:
 *
 * ```bash
 * PLAYWRIGHT_SKIP_BROWSER_DOWNLOAD=1 npm i -D playwright
 * ```
 *
 */
export interface Android {
  /**
   * This methods attaches Playwright to an existing Android device. Use
   * [android.launchServer([options])](https://playwright.dev/docs/api/class-android#android-launch-server) to launch a
   * new Android server instance.
   * @param wsEndpoint A browser websocket endpoint to connect to.
   * @param options
   */
  connect(wsEndpoint: string, options?: {
    /**
     * Additional HTTP headers to be sent with web socket connect request. Optional.
     */
    headers?: { [key: string]: string; };

    /**
     * Slows down Playwright operations by the specified amount of milliseconds. Useful so that you can see what is going
     * on. Defaults to `0`.
     */
    slowMo?: number;

    /**
     * Maximum time in milliseconds to wait for the connection to be established. Defaults to `30000` (30 seconds). Pass
     * `0` to disable timeout.
     */
    timeout?: number;
  }): Promise<AndroidDevice>;

  /**
   * Returns the list of detected Android devices.
   * @param options
   */
  devices(options?: {
    /**
     * Optional host to establish ADB server connection. Default to `127.0.0.1`.
     */
    host?: string;

    /**
     * Prevents automatic playwright driver installation on attach. Assumes that the drivers have been installed already.
     */
    omitDriverInstall?: boolean;

    /**
     * Optional port to establish ADB server connection. Default to `5037`.
     */
    port?: number;
  }): Promise<Array<AndroidDevice>>;

  /**
   * Launches Playwright Android server that clients can connect to. See the following example:
   *
   * **Usage**
   *
   * Server Side:
   *
   * ```js
   * const { _android } = require('playwright');
   *
   * (async () => {
   *   const browserServer = await _android.launchServer({
   *     // If you have multiple devices connected and want to use a specific one.
   *     // deviceSerialNumber: '<deviceSerialNumber>',
   *   });
   *   const wsEndpoint = browserServer.wsEndpoint();
   *   console.log(wsEndpoint);
   * })();
   * ```
   *
   * Client Side:
   *
   * ```js
   * const { _android } = require('playwright');
   *
   * (async () => {
   *   const device = await _android.connect('<wsEndpoint>');
   *
   *   console.log(device.model());
   *   console.log(device.serial());
   *   await device.shell('am force-stop com.android.chrome');
   *   const context = await device.launchBrowser();
   *
   *   const page = await context.newPage();
   *   await page.goto('https://webkit.org/');
   *   console.log(await page.evaluate(() => window.location.href));
   *   await page.screenshot({ path: 'page-chrome-1.png' });
   *
   *   await context.close();
   * })();
   * ```
   *
   * @param options
   */
  launchServer(options?: {
    /**
     * Optional host to establish ADB server connection. Default to `127.0.0.1`.
     */
    adbHost?: string;

    /**
     * Optional port to establish ADB server connection. Default to `5037`.
     */
    adbPort?: number;

    /**
     * Optional device serial number to launch the browser on. If not specified, it will throw if multiple devices are
     * connected.
     */
    deviceSerialNumber?: string;

    /**
     * Prevents automatic playwright driver installation on attach. Assumes that the drivers have been installed already.
     */
    omitDriverInstall?: boolean;

    /**
     * Port to use for the web socket. Defaults to 0 that picks any available port.
     */
    port?: number;

    /**
     * Path at which to serve the Android Server. For security, this defaults to an unguessable string.
     *
     * **NOTE** Any process or web page (including those running in Playwright) with knowledge of the `wsPath` can take
     * control of the OS user. For this reason, you should use an unguessable token when using this option.
     */
    wsPath?: string;
  }): Promise<BrowserServer>;

  /**
   * This setting will change the default maximum time for all the methods accepting `timeout` option.
   * @param timeout Maximum time in milliseconds
   */
  setDefaultTimeout(timeout: number): void;
}

/**
 * [AndroidDevice] represents a connected device, either real hardware or emulated. Devices can be obtained using
 * [android.devices([options])](https://playwright.dev/docs/api/class-android#android-devices).
 */
export interface AndroidDevice {
  /**
   * Emitted when the device connection gets closed.
   */
  on(event: 'close', listener: (androidDevice: AndroidDevice) => void): this;

  /**
   * Emitted when a new WebView instance is detected.
   */
  on(event: 'webview', listener: (androidWebView: AndroidWebView) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'close', listener: (androidDevice: AndroidDevice) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'webview', listener: (androidWebView: AndroidWebView) => void): this;

  /**
   * Emitted when the device connection gets closed.
   */
  addListener(event: 'close', listener: (androidDevice: AndroidDevice) => void): this;

  /**
   * Emitted when a new WebView instance is detected.
   */
  addListener(event: 'webview', listener: (androidWebView: AndroidWebView) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'close', listener: (androidDevice: AndroidDevice) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'webview', listener: (androidWebView: AndroidWebView) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'close', listener: (androidDevice: AndroidDevice) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'webview', listener: (androidWebView: AndroidWebView) => void): this;

  /**
   * Emitted when the device connection gets closed.
   */
  prependListener(event: 'close', listener: (androidDevice: AndroidDevice) => void): this;

  /**
   * Emitted when a new WebView instance is detected.
   */
  prependListener(event: 'webview', listener: (androidWebView: AndroidWebView) => void): this;

  /**
   * Disconnects from the device.
   */
  close(): Promise<void>;

  /**
   * Drags the widget defined by `selector` towards `dest` point.
   * @param selector Selector to drag.
   * @param dest Point to drag to.
   * @param options
   */
  drag(selector: AndroidSelector, dest: {
    x: number;

    y: number;
  }, options?: {
    /**
     * Optional speed of the drag in pixels per second.
     */
    speed?: number;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed
     * by using the
     * [androidDevice.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-androiddevice#android-device-set-default-timeout)
     * method.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Fills the specific `selector` input box with `text`.
   * @param selector Selector to fill.
   * @param text Text to be filled in the input box.
   * @param options
   */
  fill(selector: AndroidSelector, text: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed
     * by using the
     * [androidDevice.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-androiddevice#android-device-set-default-timeout)
     * method.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Flings the widget defined by `selector` in  the specified `direction`.
   * @param selector Selector to fling.
   * @param direction Fling direction.
   * @param options
   */
  fling(selector: AndroidSelector, direction: "down"|"up"|"left"|"right", options?: {
    /**
     * Optional speed of the fling in pixels per second.
     */
    speed?: number;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed
     * by using the
     * [androidDevice.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-androiddevice#android-device-set-default-timeout)
     * method.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns information about a widget defined by `selector`.
   * @param selector Selector to return information about.
   */
  info(selector: AndroidSelector): Promise<AndroidElementInfo>;

  /**
   * Installs an apk on the device.
   * @param file Either a path to the apk file, or apk file content.
   * @param options
   */
  installApk(file: string|Buffer, options?: {
    /**
     * Optional arguments to pass to the `shell:cmd package install` call. Defaults to `-r -t -S`.
     */
    args?: Array<string>;
  }): Promise<void>;

  /**
   * Launches Chrome browser on the device, and returns its persistent context.
   * @param options
   */
  launchBrowser(options?: {
    /**
     * Whether to automatically download all the attachments. Defaults to `true` where all the downloads are accepted.
     */
    acceptDownloads?: boolean;

    /**
     * Additional arguments to pass to the browser instance. The list of Chromium flags can be found
     * [here](http://peter.sh/experiments/chromium-command-line-switches/).
     */
    args?: Array<string>;

    /**
     * When using [page.goto(url[, options])](https://playwright.dev/docs/api/class-page#page-goto),
     * [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route),
     * [page.waitForURL(url[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-url),
     * [page.waitForRequest(urlOrPredicate[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-request),
     * or
     * [page.waitForResponse(urlOrPredicate[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-response)
     * it takes the base URL in consideration by using the
     * [`URL()`](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL) constructor for building the corresponding URL.
     * Examples:
     * - baseURL: `http://localhost:3000` and navigating to `/bar.html` results in `http://localhost:3000/bar.html`
     * - baseURL: `http://localhost:3000/foo/` and navigating to `./bar.html` results in
     *   `http://localhost:3000/foo/bar.html`
     * - baseURL: `http://localhost:3000/foo` (without trailing slash) and navigating to `./bar.html` results in
     *   `http://localhost:3000/bar.html`
     */
    baseURL?: string;

    /**
     * Toggles bypassing page's Content-Security-Policy.
     */
    bypassCSP?: boolean;

    /**
     * Emulates `'prefers-colors-scheme'` media feature, supported values are `'light'`, `'dark'`, `'no-preference'`. See
     * [page.emulateMedia([options])](https://playwright.dev/docs/api/class-page#page-emulate-media) for more details.
     * Passing `null` resets emulation to system defaults. Defaults to `'light'`.
     */
    colorScheme?: null|"light"|"dark"|"no-preference";

    /**
     * Optional package name to launch instead of default Chrome for Android.
     */
    command?: string;

    /**
     * Specify device scale factor (can be thought of as dpr). Defaults to `1`.
     */
    deviceScaleFactor?: number;

    /**
     * An object containing additional HTTP headers to be sent with every request.
     */
    extraHTTPHeaders?: { [key: string]: string; };

    /**
     * Emulates `'forced-colors'` media feature, supported values are `'active'`, `'none'`. See
     * [page.emulateMedia([options])](https://playwright.dev/docs/api/class-page#page-emulate-media) for more details.
     * Passing `null` resets emulation to system defaults. Defaults to `'none'`.
     */
    forcedColors?: null|"active"|"none";

    geolocation?: {
      /**
       * Latitude between -90 and 90.
       */
      latitude: number;

      /**
       * Longitude between -180 and 180.
       */
      longitude: number;

      /**
       * Non-negative accuracy value. Defaults to `0`.
       */
      accuracy?: number;
    };

    /**
     * Specifies if viewport supports touch events. Defaults to false.
     */
    hasTouch?: boolean;

    /**
     * Credentials for [HTTP authentication](https://developer.mozilla.org/en-US/docs/Web/HTTP/Authentication).
     */
    httpCredentials?: {
      username: string;

      password: string;
    };

    /**
     * Whether to ignore HTTPS errors when sending network requests. Defaults to `false`.
     */
    ignoreHTTPSErrors?: boolean;

    /**
     * Whether the `meta viewport` tag is taken into account and touch events are enabled. Defaults to `false`. Not
     * supported in Firefox.
     */
    isMobile?: boolean;

    /**
     * Whether or not to enable JavaScript in the context. Defaults to `true`.
     */
    javaScriptEnabled?: boolean;

    /**
     * Specify user locale, for example `en-GB`, `de-DE`, etc. Locale will affect `navigator.language` value,
     * `Accept-Language` request header value as well as number and date formatting rules.
     */
    locale?: string;

    /**
     * Logger sink for Playwright logging.
     */
    logger?: Logger;

    /**
     * Whether to emulate network being offline. Defaults to `false`.
     */
    offline?: boolean;

    /**
     * A list of permissions to grant to all pages in this context. See
     * [browserContext.grantPermissions(permissions[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-grant-permissions)
     * for more details.
     */
    permissions?: Array<string>;

    /**
     * Network proxy settings.
     */
    proxy?: {
      /**
       * Proxy to be used for all requests. HTTP and SOCKS proxies are supported, for example `http://myproxy.com:3128` or
       * `socks5://myproxy.com:3128`. Short form `myproxy.com:3128` is considered an HTTP proxy.
       */
      server: string;

      /**
       * Optional comma-separated domains to bypass proxy, for example `".com, chromium.org, .domain.com"`.
       */
      bypass?: string;

      /**
       * Optional username to use if HTTP proxy requires authentication.
       */
      username?: string;

      /**
       * Optional password to use if HTTP proxy requires authentication.
       */
      password?: string;
    };

    /**
     * Enables [HAR](http://www.softwareishard.com/blog/har-12-spec) recording for all pages into `recordHar.path` file.
     * If not specified, the HAR is not recorded. Make sure to await
     * [browserContext.close()](https://playwright.dev/docs/api/class-browsercontext#browser-context-close) for the HAR to
     * be saved.
     */
    recordHar?: {
      /**
       * Optional setting to control whether to omit request content from the HAR. Defaults to `false`. Deprecated, use
       * `content` policy instead.
       */
      omitContent?: boolean;

      /**
       * Optional setting to control resource content management. If `omit` is specified, content is not persisted. If
       * `attach` is specified, resources are persisted as separate files or entries in the ZIP archive. If `embed` is
       * specified, content is stored inline the HAR file as per HAR specification. Defaults to `attach` for `.zip` output
       * files and to `embed` for all other file extensions.
       */
      content?: "omit"|"embed"|"attach";

      /**
       * Path on the filesystem to write the HAR file to. If the file name ends with `.zip`, `content: 'attach'` is used by
       * default.
       */
      path: string;

      /**
       * When set to `minimal`, only record information necessary for routing from HAR. This omits sizes, timing, page,
       * cookies, security and other types of HAR information that are not used when replaying from HAR. Defaults to `full`.
       */
      mode?: "full"|"minimal";

      /**
       * A glob or regex pattern to filter requests that are stored in the HAR. When a `baseURL` via the context options was
       * provided and the passed URL is a path, it gets merged via the
       * [`new URL()`](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL) constructor.
       */
      urlFilter?: string|RegExp;
    };

    /**
     * Enables video recording for all pages into `recordVideo.dir` directory. If not specified videos are not recorded.
     * Make sure to await
     * [browserContext.close()](https://playwright.dev/docs/api/class-browsercontext#browser-context-close) for videos to
     * be saved.
     */
    recordVideo?: {
      /**
       * Path to the directory to put videos into.
       */
      dir: string;

      /**
       * Optional dimensions of the recorded videos. If not specified the size will be equal to `viewport` scaled down to
       * fit into 800x800. If `viewport` is not configured explicitly the video size defaults to 800x450. Actual picture of
       * each page will be scaled down if necessary to fit the specified size.
       */
      size?: {
        /**
         * Video frame width.
         */
        width: number;

        /**
         * Video frame height.
         */
        height: number;
      };
    };

    /**
     * Emulates `'prefers-reduced-motion'` media feature, supported values are `'reduce'`, `'no-preference'`. See
     * [page.emulateMedia([options])](https://playwright.dev/docs/api/class-page#page-emulate-media) for more details.
     * Passing `null` resets emulation to system defaults. Defaults to `'no-preference'`.
     */
    reducedMotion?: null|"reduce"|"no-preference";

    /**
     * Emulates consistent window screen size available inside web page via `window.screen`. Is only used when the
     * `viewport` is set.
     */
    screen?: {
      /**
       * page width in pixels.
       */
      width: number;

      /**
       * page height in pixels.
       */
      height: number;
    };

    /**
     * Whether to allow sites to register Service workers. Defaults to `'allow'`.
     * - `'allow'`: [Service Workers](https://developer.mozilla.org/en-US/docs/Web/API/Service_Worker_API) can be
     *   registered.
     * - `'block'`: Playwright will block all registration of Service Workers.
     */
    serviceWorkers?: "allow"|"block";

    /**
     * If set to true, enables strict selectors mode for this context. In the strict selectors mode all operations on
     * selectors that imply single target DOM element will throw when more than one element matches the selector. This
     * option does not affect any Locator APIs (Locators are always strict). See [Locator] to learn more about the strict
     * mode.
     */
    strictSelectors?: boolean;

    /**
     * Changes the timezone of the context. See
     * [ICU's metaZones.txt](https://cs.chromium.org/chromium/src/third_party/icu/source/data/misc/metaZones.txt?rcl=faee8bc70570192d82d2978a71e2a615788597d1)
     * for a list of supported timezone IDs.
     */
    timezoneId?: string;

    /**
     * Specific user agent to use in this context.
     */
    userAgent?: string;

    /**
     * @deprecated Use `recordVideo` instead.
     */
    videoSize?: {
      /**
       * Video frame width.
       */
      width: number;

      /**
       * Video frame height.
       */
      height: number;
    };

    /**
     * @deprecated Use `recordVideo` instead.
     */
    videosPath?: string;

    /**
     * Emulates consistent viewport for each page. Defaults to an 1280x720 viewport. Use `null` to disable the consistent
     * viewport emulation.
     *
     * **NOTE** The `null` value opts out from the default presets, makes viewport depend on the host window size defined
     * by the operating system. It makes the execution of the tests non-deterministic.
     */
    viewport?: null|{
      /**
       * page width in pixels.
       */
      width: number;

      /**
       * page height in pixels.
       */
      height: number;
    };
  }): Promise<BrowserContext>;

  /**
   * Performs a long tap on the widget defined by `selector`.
   * @param selector Selector to tap on.
   * @param options
   */
  longTap(selector: AndroidSelector, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed
     * by using the
     * [androidDevice.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-androiddevice#android-device-set-default-timeout)
     * method.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Device model.
   */
  model(): string;

  /**
   * Launches a process in the shell on the device and returns a socket to communicate with the launched process.
   * @param command
   */
  open(command: string): Promise<AndroidSocket>;

  /**
   * Pinches the widget defined by `selector` in the closing direction.
   * @param selector Selector to pinch close.
   * @param percent The size of the pinch as a percentage of the widget's size.
   * @param options
   */
  pinchClose(selector: AndroidSelector, percent: number, options?: {
    /**
     * Optional speed of the pinch in pixels per second.
     */
    speed?: number;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed
     * by using the
     * [androidDevice.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-androiddevice#android-device-set-default-timeout)
     * method.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Pinches the widget defined by `selector` in the open direction.
   * @param selector Selector to pinch open.
   * @param percent The size of the pinch as a percentage of the widget's size.
   * @param options
   */
  pinchOpen(selector: AndroidSelector, percent: number, options?: {
    /**
     * Optional speed of the pinch in pixels per second.
     */
    speed?: number;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed
     * by using the
     * [androidDevice.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-androiddevice#android-device-set-default-timeout)
     * method.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Presses the specific `key` in the widget defined by `selector`.
   * @param selector Selector to press the key in.
   * @param key The key to press.
   * @param options
   */
  press(selector: AndroidSelector, key: AndroidKey, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed
     * by using the
     * [androidDevice.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-androiddevice#android-device-set-default-timeout)
     * method.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Copies a file to the device.
   * @param file Either a path to the file, or file content.
   * @param path Path to the file on the device.
   * @param options
   */
  push(file: string|Buffer, path: string, options?: {
    /**
     * Optional file mode, defaults to `644` (`rw-r--r--`).
     */
    mode?: number;
  }): Promise<void>;

  /**
   * Returns the buffer with the captured screenshot of the device.
   * @param options
   */
  screenshot(options?: {
    /**
     * The file path to save the image to. If `path` is a relative path, then it is resolved relative to the current
     * working directory. If no path is provided, the image won't be saved to the disk.
     */
    path?: string;
  }): Promise<Buffer>;

  /**
   * Scrolls the widget defined by `selector` in  the specified `direction`.
   * @param selector Selector to scroll.
   * @param direction Scroll direction.
   * @param percent Distance to scroll as a percentage of the widget's size.
   * @param options
   */
  scroll(selector: AndroidSelector, direction: "down"|"up"|"left"|"right", percent: number, options?: {
    /**
     * Optional speed of the scroll in pixels per second.
     */
    speed?: number;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed
     * by using the
     * [androidDevice.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-androiddevice#android-device-set-default-timeout)
     * method.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Device serial number.
   */
  serial(): string;

  /**
   * This setting will change the default maximum time for all the methods accepting `timeout` option.
   * @param timeout Maximum time in milliseconds
   */
  setDefaultTimeout(timeout: number): void;

  /**
   * Executes a shell command on the device and returns its output.
   * @param command Shell command to execute.
   */
  shell(command: string): Promise<Buffer>;

  /**
   * Swipes the widget defined by `selector` in  the specified `direction`.
   * @param selector Selector to swipe.
   * @param direction Swipe direction.
   * @param percent Distance to swipe as a percentage of the widget's size.
   * @param options
   */
  swipe(selector: AndroidSelector, direction: "down"|"up"|"left"|"right", percent: number, options?: {
    /**
     * Optional speed of the swipe in pixels per second.
     */
    speed?: number;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed
     * by using the
     * [androidDevice.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-androiddevice#android-device-set-default-timeout)
     * method.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Taps on the widget defined by `selector`.
   * @param selector Selector to tap on.
   * @param options
   */
  tap(selector: AndroidSelector, options?: {
    /**
     * Optional duration of the tap in milliseconds.
     */
    duration?: number;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed
     * by using the
     * [androidDevice.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-androiddevice#android-device-set-default-timeout)
     * method.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Waits for the specific `selector` to either appear or disappear, depending on the `state`.
   * @param selector Selector to wait for.
   * @param options
   */
  wait(selector: AndroidSelector, options?: {
    /**
     * Optional state. Can be either:
     * - default - wait for element to be present.
     * - `'gone'` - wait for element to not be present.
     */
    state?: "gone";

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed
     * by using the
     * [androidDevice.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-androiddevice#android-device-set-default-timeout)
     * method.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Emitted when the device connection gets closed.
   */
  waitForEvent(event: 'close', optionsOrPredicate?: { predicate?: (androidDevice: AndroidDevice) => boolean | Promise<boolean>, timeout?: number } | ((androidDevice: AndroidDevice) => boolean | Promise<boolean>)): Promise<AndroidDevice>;

  /**
   * Emitted when a new WebView instance is detected.
   */
  waitForEvent(event: 'webview', optionsOrPredicate?: { predicate?: (androidWebView: AndroidWebView) => boolean | Promise<boolean>, timeout?: number } | ((androidWebView: AndroidWebView) => boolean | Promise<boolean>)): Promise<AndroidWebView>;


  /**
   * This method waits until [AndroidWebView] matching the `selector` is opened and returns it. If there is already an
   * open [AndroidWebView] matching the `selector`, returns immediately.
   * @param selector
   * @param options
   */
  webView(selector: {
    /**
     * Optional Package identifier.
     */
    pkg?: string;

    /**
     * Optional webview socket name.
     */
    socketName?: string;
  }, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed
     * by using the
     * [androidDevice.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-androiddevice#android-device-set-default-timeout)
     * method.
     */
    timeout?: number;
  }): Promise<AndroidWebView>;

  /**
   * Currently open WebViews.
   */
  webViews(): Array<AndroidWebView>;

  input: AndroidInput;
}

export interface AndroidInput {
  /**
   * Performs a drag between `from` and `to` points.
   * @param from The start point of the drag.
   * @param to The end point of the drag.
   * @param steps The number of steps in the drag. Each step takes 5 milliseconds to complete.
   */
  drag(from: {
    x: number;

    y: number;
  }, to: {
    x: number;

    y: number;
  }, steps: number): Promise<void>;

  /**
   * Presses the `key`.
   * @param key Key to press.
   */
  press(key: AndroidKey): Promise<void>;

  /**
   * Swipes following the path defined by `segments`.
   * @param from The point to start swiping from.
   * @param segments Points following the `from` point in the swipe gesture.
   * @param steps The number of steps for each segment. Each step takes 5 milliseconds to complete, so 100 steps means half a second
   * per each segment.
   */
  swipe(from: {
    x: number;

    y: number;
  }, segments: Array<{
    x: number;

    y: number;
  }>, steps: number): Promise<void>;

  /**
   * Taps at the specified `point`.
   * @param point The point to tap at.
   */
  tap(point: {
    x: number;

    y: number;
  }): Promise<void>;

  /**
   * Types `text` into currently focused widget.
   * @param text Text to type.
   */
  type(text: string): Promise<void>;
}

/**
 * [AndroidSocket] is a way to communicate with a process launched on the [AndroidDevice]. Use
 * [androidDevice.open(command)](https://playwright.dev/docs/api/class-androiddevice#android-device-open) to open a
 * socket.
 */
export interface AndroidSocket {
  /**
   * Emitted when the socket is closed.
   */
  on(event: 'close', listener: () => void): this;

  /**
   * Emitted when data is available to read from the socket.
   */
  on(event: 'data', listener: (buffer: Buffer) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'close', listener: () => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'data', listener: (buffer: Buffer) => void): this;

  /**
   * Emitted when the socket is closed.
   */
  addListener(event: 'close', listener: () => void): this;

  /**
   * Emitted when data is available to read from the socket.
   */
  addListener(event: 'data', listener: (buffer: Buffer) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'close', listener: () => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'data', listener: (buffer: Buffer) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'close', listener: () => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'data', listener: (buffer: Buffer) => void): this;

  /**
   * Emitted when the socket is closed.
   */
  prependListener(event: 'close', listener: () => void): this;

  /**
   * Emitted when data is available to read from the socket.
   */
  prependListener(event: 'data', listener: (buffer: Buffer) => void): this;

  /**
   * Closes the socket.
   */
  close(): Promise<void>;

  /**
   * Writes some `data` to the socket.
   * @param data Data to write.
   */
  write(data: Buffer): Promise<void>;
}

/**
 * [AndroidWebView] represents a WebView open on the [AndroidDevice]. WebView is usually obtained using
 * [androidDevice.webView(selector[, options])](https://playwright.dev/docs/api/class-androiddevice#android-device-web-view).
 */
export interface AndroidWebView {
  /**
   * Emitted when the WebView is closed.
   */
  on(event: 'close', listener: () => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'close', listener: () => void): this;

  /**
   * Emitted when the WebView is closed.
   */
  addListener(event: 'close', listener: () => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'close', listener: () => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'close', listener: () => void): this;

  /**
   * Emitted when the WebView is closed.
   */
  prependListener(event: 'close', listener: () => void): this;

  /**
   * Connects to the WebView and returns a regular Playwright [Page] to interact with.
   */
  page(): Promise<Page>;

  /**
   * WebView process PID.
   */
  pid(): number;

  /**
   * WebView package identifier.
   */
  pkg(): string;
}

/**
 * Exposes API that can be used for the Web API testing. This class is used for creating [APIRequestContext] instance
 * which in turn can be used for sending web requests. An instance of this class can be obtained via
 * [playwright.request](https://playwright.dev/docs/api/class-playwright#playwright-request). For more information see
 * [APIRequestContext].
 */
export interface APIRequest {
  /**
   * Creates new instances of [APIRequestContext].
   * @param options
   */
  newContext(options?: {
    /**
     * Methods like
     * [apiRequestContext.get(url[, options])](https://playwright.dev/docs/api/class-apirequestcontext#api-request-context-get)
     * take the base URL into consideration by using the
     * [`URL()`](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL) constructor for building the corresponding URL.
     * Examples:
     * - baseURL: `http://localhost:3000` and sending request to `/bar.html` results in `http://localhost:3000/bar.html`
     * - baseURL: `http://localhost:3000/foo/` and sending request to `./bar.html` results in
     *   `http://localhost:3000/foo/bar.html`
     * - baseURL: `http://localhost:3000/foo` (without trailing slash) and navigating to `./bar.html` results in
     *   `http://localhost:3000/bar.html`
     */
    baseURL?: string;

    /**
     * An object containing additional HTTP headers to be sent with every request.
     */
    extraHTTPHeaders?: { [key: string]: string; };

    /**
     * Credentials for [HTTP authentication](https://developer.mozilla.org/en-US/docs/Web/HTTP/Authentication).
     */
    httpCredentials?: {
      username: string;

      password: string;
    };

    /**
     * Whether to ignore HTTPS errors when sending network requests. Defaults to `false`.
     */
    ignoreHTTPSErrors?: boolean;

    /**
     * Network proxy settings.
     */
    proxy?: {
      /**
       * Proxy to be used for all requests. HTTP and SOCKS proxies are supported, for example `http://myproxy.com:3128` or
       * `socks5://myproxy.com:3128`. Short form `myproxy.com:3128` is considered an HTTP proxy.
       */
      server: string;

      /**
       * Optional comma-separated domains to bypass proxy, for example `".com, chromium.org, .domain.com"`.
       */
      bypass?: string;

      /**
       * Optional username to use if HTTP proxy requires authentication.
       */
      username?: string;

      /**
       * Optional password to use if HTTP proxy requires authentication.
       */
      password?: string;
    };

    /**
     * Populates context with given storage state. This option can be used to initialize context with logged-in
     * information obtained via
     * [browserContext.storageState([options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-storage-state)
     * or
     * [apiRequestContext.storageState([options])](https://playwright.dev/docs/api/class-apirequestcontext#api-request-context-storage-state).
     * Either a path to the file with saved storage, or the value returned by one of
     * [browserContext.storageState([options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-storage-state)
     * or
     * [apiRequestContext.storageState([options])](https://playwright.dev/docs/api/class-apirequestcontext#api-request-context-storage-state)
     * methods.
     */
    storageState?: string|{
      cookies: Array<{
        name: string;

        value: string;

        domain: string;

        path: string;

        /**
         * Unix time in seconds.
         */
        expires: number;

        httpOnly: boolean;

        secure: boolean;

        sameSite: "Strict"|"Lax"|"None";
      }>;

      origins: Array<{
        origin: string;

        localStorage: Array<{
          name: string;

          value: string;
        }>;
      }>;
    };

    /**
     * Maximum time in milliseconds to wait for the response. Defaults to `30000` (30 seconds). Pass `0` to disable
     * timeout.
     */
    timeout?: number;

    /**
     * Specific user agent to use in this context.
     */
    userAgent?: string;
  }): Promise<APIRequestContext>;
}

/**
 * This API is used for the Web API testing. You can use it to trigger API endpoints, configure micro-services,
 * prepare environment or the service to your e2e test.
 *
 * Each Playwright browser context has associated with it [APIRequestContext] instance which shares cookie storage
 * with the browser context and can be accessed via
 * [browserContext.request](https://playwright.dev/docs/api/class-browsercontext#browser-context-request) or
 * [page.request](https://playwright.dev/docs/api/class-page#page-request). It is also possible to create a new
 * APIRequestContext instance manually by calling
 * [apiRequest.newContext([options])](https://playwright.dev/docs/api/class-apirequest#api-request-new-context).
 *
 * **Cookie management**
 *
 * [APIRequestContext] returned by
 * [browserContext.request](https://playwright.dev/docs/api/class-browsercontext#browser-context-request) and
 * [page.request](https://playwright.dev/docs/api/class-page#page-request) shares cookie storage with the
 * corresponding [BrowserContext]. Each API request will have `Cookie` header populated with the values from the
 * browser context. If the API response contains `Set-Cookie` header it will automatically update [BrowserContext]
 * cookies and requests made from the page will pick them up. This means that if you log in using this API, your e2e
 * test will be logged in and vice versa.
 *
 * If you want API requests to not interfere with the browser cookies you should create a new [APIRequestContext] by
 * calling
 * [apiRequest.newContext([options])](https://playwright.dev/docs/api/class-apirequest#api-request-new-context). Such
 * `APIRequestContext` object will have its own isolated cookie storage.
 */
export interface APIRequestContext {
  /**
   * Sends HTTP(S) [DELETE](https://developer.mozilla.org/en-US/docs/Web/HTTP/Methods/DELETE) request and returns its
   * response. The method will populate request cookies from the context and update context cookies from the response.
   * The method will automatically follow redirects.
   * @param url Target URL.
   * @param options
   */
  delete(url: string, options?: {
    /**
     * Allows to set post data of the request. If the data parameter is an object, it will be serialized to json string
     * and `content-type` header will be set to `application/json` if not explicitly set. Otherwise the `content-type`
     * header will be set to `application/octet-stream` if not explicitly set.
     */
    data?: string|Buffer|Serializable;

    /**
     * Whether to throw on response codes other than 2xx and 3xx. By default response object is returned for all status
     * codes.
     */
    failOnStatusCode?: boolean;

    /**
     * Provides an object that will be serialized as html form using `application/x-www-form-urlencoded` encoding and sent
     * as this request body. If this parameter is specified `content-type` header will be set to
     * `application/x-www-form-urlencoded` unless explicitly provided.
     */
    form?: { [key: string]: string|number|boolean; };

    /**
     * Allows to set HTTP headers. These headers will apply to the fetched request as well as any redirects initiated by
     * it.
     */
    headers?: { [key: string]: string; };

    /**
     * Whether to ignore HTTPS errors when sending network requests. Defaults to `false`.
     */
    ignoreHTTPSErrors?: boolean;

    /**
     * Maximum number of request redirects that will be followed automatically. An error will be thrown if the number is
     * exceeded. Defaults to `20`. Pass `0` to not follow redirects.
     */
    maxRedirects?: number;

    /**
     * Provides an object that will be serialized as html form using `multipart/form-data` encoding and sent as this
     * request body. If this parameter is specified `content-type` header will be set to `multipart/form-data` unless
     * explicitly provided. File values can be passed either as
     * [`fs.ReadStream`](https://nodejs.org/api/fs.html#fs_class_fs_readstream) or as file-like object containing file
     * name, mime-type and its content.
     */
    multipart?: { [key: string]: string|number|boolean|ReadStream|{
      /**
       * File name
       */
      name: string;

      /**
       * File type
       */
      mimeType: string;

      /**
       * File content
       */
      buffer: Buffer;
    }; };

    /**
     * Query parameters to be sent with the URL.
     */
    params?: { [key: string]: string|number|boolean; };

    /**
     * Request timeout in milliseconds. Defaults to `30000` (30 seconds). Pass `0` to disable timeout.
     */
    timeout?: number;
  }): Promise<APIResponse>;

  /**
   * All responses returned by
   * [apiRequestContext.get(url[, options])](https://playwright.dev/docs/api/class-apirequestcontext#api-request-context-get)
   * and similar methods are stored in the memory, so that you can later call
   * [apiResponse.body()](https://playwright.dev/docs/api/class-apiresponse#api-response-body). This method discards all
   * stored responses, and makes
   * [apiResponse.body()](https://playwright.dev/docs/api/class-apiresponse#api-response-body) throw "Response disposed"
   * error.
   */
  dispose(): Promise<void>;

  /**
   * Sends HTTP(S) request and returns its response. The method will populate request cookies from the context and
   * update context cookies from the response. The method will automatically follow redirects. JSON objects can be
   * passed directly to the request.
   *
   * **Usage**
   *
   * ```js
   * await request.fetch('https://example.com/api/createBook', {
   *   method: 'post',
   *   data: {
   *     title: 'Book Title',
   *     author: 'John Doe',
   *   }
   * });
   * ```
   *
   * The common way to send file(s) in the body of a request is to encode it as form fields with `multipart/form-data`
   * encoding. You can achieve that with Playwright API like this:
   *
   * ```js
   * // Open file as a stream and pass it to the request:
   * const stream = fs.createReadStream('team.csv');
   * await request.fetch('https://example.com/api/uploadTeamList', {
   *   method: 'post',
   *   multipart: {
   *     fileField: stream
   *   }
   * });
   *
   * // Or you can pass the file content directly as an object:
   * await request.fetch('https://example.com/api/uploadScript', {
   *   method: 'post',
   *   multipart: {
   *     fileField: {
   *       name: 'f.js',
   *       mimeType: 'text/javascript',
   *       buffer: Buffer.from('console.log(2022);')
   *     }
   *   }
   * });
   * ```
   *
   * @param urlOrRequest Target URL or Request to get all parameters from.
   * @param options
   */
  fetch(urlOrRequest: string|Request, options?: {
    /**
     * Allows to set post data of the request. If the data parameter is an object, it will be serialized to json string
     * and `content-type` header will be set to `application/json` if not explicitly set. Otherwise the `content-type`
     * header will be set to `application/octet-stream` if not explicitly set.
     */
    data?: string|Buffer|Serializable;

    /**
     * Whether to throw on response codes other than 2xx and 3xx. By default response object is returned for all status
     * codes.
     */
    failOnStatusCode?: boolean;

    /**
     * Provides an object that will be serialized as html form using `application/x-www-form-urlencoded` encoding and sent
     * as this request body. If this parameter is specified `content-type` header will be set to
     * `application/x-www-form-urlencoded` unless explicitly provided.
     */
    form?: { [key: string]: string|number|boolean; };

    /**
     * Allows to set HTTP headers. These headers will apply to the fetched request as well as any redirects initiated by
     * it.
     */
    headers?: { [key: string]: string; };

    /**
     * Whether to ignore HTTPS errors when sending network requests. Defaults to `false`.
     */
    ignoreHTTPSErrors?: boolean;

    /**
     * Maximum number of request redirects that will be followed automatically. An error will be thrown if the number is
     * exceeded. Defaults to `20`. Pass `0` to not follow redirects.
     */
    maxRedirects?: number;

    /**
     * If set changes the fetch method (e.g. [PUT](https://developer.mozilla.org/en-US/docs/Web/HTTP/Methods/PUT) or
     * [POST](https://developer.mozilla.org/en-US/docs/Web/HTTP/Methods/POST)). If not specified, GET method is used.
     */
    method?: string;

    /**
     * Provides an object that will be serialized as html form using `multipart/form-data` encoding and sent as this
     * request body. If this parameter is specified `content-type` header will be set to `multipart/form-data` unless
     * explicitly provided. File values can be passed either as
     * [`fs.ReadStream`](https://nodejs.org/api/fs.html#fs_class_fs_readstream) or as file-like object containing file
     * name, mime-type and its content.
     */
    multipart?: { [key: string]: string|number|boolean|ReadStream|{
      /**
       * File name
       */
      name: string;

      /**
       * File type
       */
      mimeType: string;

      /**
       * File content
       */
      buffer: Buffer;
    }; };

    /**
     * Query parameters to be sent with the URL.
     */
    params?: { [key: string]: string|number|boolean; };

    /**
     * Request timeout in milliseconds. Defaults to `30000` (30 seconds). Pass `0` to disable timeout.
     */
    timeout?: number;
  }): Promise<APIResponse>;

  /**
   * Sends HTTP(S) [GET](https://developer.mozilla.org/en-US/docs/Web/HTTP/Methods/GET) request and returns its
   * response. The method will populate request cookies from the context and update context cookies from the response.
   * The method will automatically follow redirects.
   *
   * **Usage**
   *
   * Request parameters can be configured with `params` option, they will be serialized into the URL search parameters:
   *
   * ```js
   * await request.get('https://example.com/api/getText', {
   *   params: {
   *     'isbn': '1234',
   *     'page': 23,
   *   }
   * });
   * ```
   *
   * @param url Target URL.
   * @param options
   */
  get(url: string, options?: {
    /**
     * Allows to set post data of the request. If the data parameter is an object, it will be serialized to json string
     * and `content-type` header will be set to `application/json` if not explicitly set. Otherwise the `content-type`
     * header will be set to `application/octet-stream` if not explicitly set.
     */
    data?: string|Buffer|Serializable;

    /**
     * Whether to throw on response codes other than 2xx and 3xx. By default response object is returned for all status
     * codes.
     */
    failOnStatusCode?: boolean;

    /**
     * Provides an object that will be serialized as html form using `application/x-www-form-urlencoded` encoding and sent
     * as this request body. If this parameter is specified `content-type` header will be set to
     * `application/x-www-form-urlencoded` unless explicitly provided.
     */
    form?: { [key: string]: string|number|boolean; };

    /**
     * Allows to set HTTP headers. These headers will apply to the fetched request as well as any redirects initiated by
     * it.
     */
    headers?: { [key: string]: string; };

    /**
     * Whether to ignore HTTPS errors when sending network requests. Defaults to `false`.
     */
    ignoreHTTPSErrors?: boolean;

    /**
     * Maximum number of request redirects that will be followed automatically. An error will be thrown if the number is
     * exceeded. Defaults to `20`. Pass `0` to not follow redirects.
     */
    maxRedirects?: number;

    /**
     * Provides an object that will be serialized as html form using `multipart/form-data` encoding and sent as this
     * request body. If this parameter is specified `content-type` header will be set to `multipart/form-data` unless
     * explicitly provided. File values can be passed either as
     * [`fs.ReadStream`](https://nodejs.org/api/fs.html#fs_class_fs_readstream) or as file-like object containing file
     * name, mime-type and its content.
     */
    multipart?: { [key: string]: string|number|boolean|ReadStream|{
      /**
       * File name
       */
      name: string;

      /**
       * File type
       */
      mimeType: string;

      /**
       * File content
       */
      buffer: Buffer;
    }; };

    /**
     * Query parameters to be sent with the URL.
     */
    params?: { [key: string]: string|number|boolean; };

    /**
     * Request timeout in milliseconds. Defaults to `30000` (30 seconds). Pass `0` to disable timeout.
     */
    timeout?: number;
  }): Promise<APIResponse>;

  /**
   * Sends HTTP(S) [HEAD](https://developer.mozilla.org/en-US/docs/Web/HTTP/Methods/HEAD) request and returns its
   * response. The method will populate request cookies from the context and update context cookies from the response.
   * The method will automatically follow redirects.
   * @param url Target URL.
   * @param options
   */
  head(url: string, options?: {
    /**
     * Allows to set post data of the request. If the data parameter is an object, it will be serialized to json string
     * and `content-type` header will be set to `application/json` if not explicitly set. Otherwise the `content-type`
     * header will be set to `application/octet-stream` if not explicitly set.
     */
    data?: string|Buffer|Serializable;

    /**
     * Whether to throw on response codes other than 2xx and 3xx. By default response object is returned for all status
     * codes.
     */
    failOnStatusCode?: boolean;

    /**
     * Provides an object that will be serialized as html form using `application/x-www-form-urlencoded` encoding and sent
     * as this request body. If this parameter is specified `content-type` header will be set to
     * `application/x-www-form-urlencoded` unless explicitly provided.
     */
    form?: { [key: string]: string|number|boolean; };

    /**
     * Allows to set HTTP headers. These headers will apply to the fetched request as well as any redirects initiated by
     * it.
     */
    headers?: { [key: string]: string; };

    /**
     * Whether to ignore HTTPS errors when sending network requests. Defaults to `false`.
     */
    ignoreHTTPSErrors?: boolean;

    /**
     * Maximum number of request redirects that will be followed automatically. An error will be thrown if the number is
     * exceeded. Defaults to `20`. Pass `0` to not follow redirects.
     */
    maxRedirects?: number;

    /**
     * Provides an object that will be serialized as html form using `multipart/form-data` encoding and sent as this
     * request body. If this parameter is specified `content-type` header will be set to `multipart/form-data` unless
     * explicitly provided. File values can be passed either as
     * [`fs.ReadStream`](https://nodejs.org/api/fs.html#fs_class_fs_readstream) or as file-like object containing file
     * name, mime-type and its content.
     */
    multipart?: { [key: string]: string|number|boolean|ReadStream|{
      /**
       * File name
       */
      name: string;

      /**
       * File type
       */
      mimeType: string;

      /**
       * File content
       */
      buffer: Buffer;
    }; };

    /**
     * Query parameters to be sent with the URL.
     */
    params?: { [key: string]: string|number|boolean; };

    /**
     * Request timeout in milliseconds. Defaults to `30000` (30 seconds). Pass `0` to disable timeout.
     */
    timeout?: number;
  }): Promise<APIResponse>;

  /**
   * Sends HTTP(S) [PATCH](https://developer.mozilla.org/en-US/docs/Web/HTTP/Methods/PATCH) request and returns its
   * response. The method will populate request cookies from the context and update context cookies from the response.
   * The method will automatically follow redirects.
   * @param url Target URL.
   * @param options
   */
  patch(url: string, options?: {
    /**
     * Allows to set post data of the request. If the data parameter is an object, it will be serialized to json string
     * and `content-type` header will be set to `application/json` if not explicitly set. Otherwise the `content-type`
     * header will be set to `application/octet-stream` if not explicitly set.
     */
    data?: string|Buffer|Serializable;

    /**
     * Whether to throw on response codes other than 2xx and 3xx. By default response object is returned for all status
     * codes.
     */
    failOnStatusCode?: boolean;

    /**
     * Provides an object that will be serialized as html form using `application/x-www-form-urlencoded` encoding and sent
     * as this request body. If this parameter is specified `content-type` header will be set to
     * `application/x-www-form-urlencoded` unless explicitly provided.
     */
    form?: { [key: string]: string|number|boolean; };

    /**
     * Allows to set HTTP headers. These headers will apply to the fetched request as well as any redirects initiated by
     * it.
     */
    headers?: { [key: string]: string; };

    /**
     * Whether to ignore HTTPS errors when sending network requests. Defaults to `false`.
     */
    ignoreHTTPSErrors?: boolean;

    /**
     * Maximum number of request redirects that will be followed automatically. An error will be thrown if the number is
     * exceeded. Defaults to `20`. Pass `0` to not follow redirects.
     */
    maxRedirects?: number;

    /**
     * Provides an object that will be serialized as html form using `multipart/form-data` encoding and sent as this
     * request body. If this parameter is specified `content-type` header will be set to `multipart/form-data` unless
     * explicitly provided. File values can be passed either as
     * [`fs.ReadStream`](https://nodejs.org/api/fs.html#fs_class_fs_readstream) or as file-like object containing file
     * name, mime-type and its content.
     */
    multipart?: { [key: string]: string|number|boolean|ReadStream|{
      /**
       * File name
       */
      name: string;

      /**
       * File type
       */
      mimeType: string;

      /**
       * File content
       */
      buffer: Buffer;
    }; };

    /**
     * Query parameters to be sent with the URL.
     */
    params?: { [key: string]: string|number|boolean; };

    /**
     * Request timeout in milliseconds. Defaults to `30000` (30 seconds). Pass `0` to disable timeout.
     */
    timeout?: number;
  }): Promise<APIResponse>;

  /**
   * Sends HTTP(S) [POST](https://developer.mozilla.org/en-US/docs/Web/HTTP/Methods/POST) request and returns its
   * response. The method will populate request cookies from the context and update context cookies from the response.
   * The method will automatically follow redirects.
   *
   * **Usage**
   *
   * JSON objects can be passed directly to the request:
   *
   * ```js
   * await request.post('https://example.com/api/createBook', {
   *   data: {
   *     title: 'Book Title',
   *     author: 'John Doe',
   *   }
   * });
   * ```
   *
   * To send form data to the server use `form` option. Its value will be encoded into the request body with
   * `application/x-www-form-urlencoded` encoding (see below how to use `multipart/form-data` form encoding to send
   * files):
   *
   * ```js
   * await request.post('https://example.com/api/findBook', {
   *   form: {
   *     title: 'Book Title',
   *     author: 'John Doe',
   *   }
   * });
   * ```
   *
   * The common way to send file(s) in the body of a request is to upload them as form fields with `multipart/form-data`
   * encoding. You can achieve that with Playwright API like this:
   *
   * ```js
   * // Open file as a stream and pass it to the request:
   * const stream = fs.createReadStream('team.csv');
   * await request.post('https://example.com/api/uploadTeamList', {
   *   multipart: {
   *     fileField: stream
   *   }
   * });
   *
   * // Or you can pass the file content directly as an object:
   * await request.post('https://example.com/api/uploadScript', {
   *   multipart: {
   *     fileField: {
   *       name: 'f.js',
   *       mimeType: 'text/javascript',
   *       buffer: Buffer.from('console.log(2022);')
   *     }
   *   }
   * });
   * ```
   *
   * @param url Target URL.
   * @param options
   */
  post(url: string, options?: {
    /**
     * Allows to set post data of the request. If the data parameter is an object, it will be serialized to json string
     * and `content-type` header will be set to `application/json` if not explicitly set. Otherwise the `content-type`
     * header will be set to `application/octet-stream` if not explicitly set.
     */
    data?: string|Buffer|Serializable;

    /**
     * Whether to throw on response codes other than 2xx and 3xx. By default response object is returned for all status
     * codes.
     */
    failOnStatusCode?: boolean;

    /**
     * Provides an object that will be serialized as html form using `application/x-www-form-urlencoded` encoding and sent
     * as this request body. If this parameter is specified `content-type` header will be set to
     * `application/x-www-form-urlencoded` unless explicitly provided.
     */
    form?: { [key: string]: string|number|boolean; };

    /**
     * Allows to set HTTP headers. These headers will apply to the fetched request as well as any redirects initiated by
     * it.
     */
    headers?: { [key: string]: string; };

    /**
     * Whether to ignore HTTPS errors when sending network requests. Defaults to `false`.
     */
    ignoreHTTPSErrors?: boolean;

    /**
     * Maximum number of request redirects that will be followed automatically. An error will be thrown if the number is
     * exceeded. Defaults to `20`. Pass `0` to not follow redirects.
     */
    maxRedirects?: number;

    /**
     * Provides an object that will be serialized as html form using `multipart/form-data` encoding and sent as this
     * request body. If this parameter is specified `content-type` header will be set to `multipart/form-data` unless
     * explicitly provided. File values can be passed either as
     * [`fs.ReadStream`](https://nodejs.org/api/fs.html#fs_class_fs_readstream) or as file-like object containing file
     * name, mime-type and its content.
     */
    multipart?: { [key: string]: string|number|boolean|ReadStream|{
      /**
       * File name
       */
      name: string;

      /**
       * File type
       */
      mimeType: string;

      /**
       * File content
       */
      buffer: Buffer;
    }; };

    /**
     * Query parameters to be sent with the URL.
     */
    params?: { [key: string]: string|number|boolean; };

    /**
     * Request timeout in milliseconds. Defaults to `30000` (30 seconds). Pass `0` to disable timeout.
     */
    timeout?: number;
  }): Promise<APIResponse>;

  /**
   * Sends HTTP(S) [PUT](https://developer.mozilla.org/en-US/docs/Web/HTTP/Methods/PUT) request and returns its
   * response. The method will populate request cookies from the context and update context cookies from the response.
   * The method will automatically follow redirects.
   * @param url Target URL.
   * @param options
   */
  put(url: string, options?: {
    /**
     * Allows to set post data of the request. If the data parameter is an object, it will be serialized to json string
     * and `content-type` header will be set to `application/json` if not explicitly set. Otherwise the `content-type`
     * header will be set to `application/octet-stream` if not explicitly set.
     */
    data?: string|Buffer|Serializable;

    /**
     * Whether to throw on response codes other than 2xx and 3xx. By default response object is returned for all status
     * codes.
     */
    failOnStatusCode?: boolean;

    /**
     * Provides an object that will be serialized as html form using `application/x-www-form-urlencoded` encoding and sent
     * as this request body. If this parameter is specified `content-type` header will be set to
     * `application/x-www-form-urlencoded` unless explicitly provided.
     */
    form?: { [key: string]: string|number|boolean; };

    /**
     * Allows to set HTTP headers. These headers will apply to the fetched request as well as any redirects initiated by
     * it.
     */
    headers?: { [key: string]: string; };

    /**
     * Whether to ignore HTTPS errors when sending network requests. Defaults to `false`.
     */
    ignoreHTTPSErrors?: boolean;

    /**
     * Maximum number of request redirects that will be followed automatically. An error will be thrown if the number is
     * exceeded. Defaults to `20`. Pass `0` to not follow redirects.
     */
    maxRedirects?: number;

    /**
     * Provides an object that will be serialized as html form using `multipart/form-data` encoding and sent as this
     * request body. If this parameter is specified `content-type` header will be set to `multipart/form-data` unless
     * explicitly provided. File values can be passed either as
     * [`fs.ReadStream`](https://nodejs.org/api/fs.html#fs_class_fs_readstream) or as file-like object containing file
     * name, mime-type and its content.
     */
    multipart?: { [key: string]: string|number|boolean|ReadStream|{
      /**
       * File name
       */
      name: string;

      /**
       * File type
       */
      mimeType: string;

      /**
       * File content
       */
      buffer: Buffer;
    }; };

    /**
     * Query parameters to be sent with the URL.
     */
    params?: { [key: string]: string|number|boolean; };

    /**
     * Request timeout in milliseconds. Defaults to `30000` (30 seconds). Pass `0` to disable timeout.
     */
    timeout?: number;
  }): Promise<APIResponse>;

  /**
   * Returns storage state for this request context, contains current cookies and local storage snapshot if it was
   * passed to the constructor.
   * @param options
   */
  storageState(options?: {
    /**
     * The file path to save the storage state to. If `path` is a relative path, then it is resolved relative to current
     * working directory. If no path is provided, storage state is still returned, but won't be saved to the disk.
     */
    path?: string;
  }): Promise<{
    cookies: Array<{
      name: string;

      value: string;

      domain: string;

      path: string;

      /**
       * Unix time in seconds.
       */
      expires: number;

      httpOnly: boolean;

      secure: boolean;

      sameSite: "Strict"|"Lax"|"None";
    }>;

    origins: Array<{
      origin: string;

      localStorage: Array<{
        name: string;

        value: string;
      }>;
    }>;
  }>;
}

/**
 * [APIResponse] class represents responses returned by
 * [apiRequestContext.get(url[, options])](https://playwright.dev/docs/api/class-apirequestcontext#api-request-context-get)
 * and similar methods.
 */
export interface APIResponse {
  /**
   * Returns the buffer with response body.
   */
  body(): Promise<Buffer>;

  /**
   * Disposes the body of this response. If not called then the body will stay in memory until the context closes.
   */
  dispose(): Promise<void>;

  /**
   * An object with all the response HTTP headers associated with this response.
   */
  headers(): { [key: string]: string; };

  /**
   * An array with all the request HTTP headers associated with this response. Header names are not lower-cased. Headers
   * with multiple entries, such as `Set-Cookie`, appear in the array multiple times.
   */
  headersArray(): Array<{
    /**
     * Name of the header.
     */
    name: string;

    /**
     * Value of the header.
     */
    value: string;
  }>;

  /**
   * Returns the JSON representation of response body.
   *
   * This method will throw if the response body is not parsable via `JSON.parse`.
   */
  json(): Promise<Serializable>;

  /**
   * Contains a boolean stating whether the response was successful (status in the range 200-299) or not.
   */
  ok(): boolean;

  /**
   * Contains the status code of the response (e.g., 200 for a success).
   */
  status(): number;

  /**
   * Contains the status text of the response (e.g. usually an "OK" for a success).
   */
  statusText(): string;

  /**
   * Returns the text representation of response body.
   */
  text(): Promise<string>;

  /**
   * Contains the URL of the response.
   */
  url(): string;
}

/**
 * - extends: [EventEmitter]
 *
 * A Browser is created via
 * [browserType.launch([options])](https://playwright.dev/docs/api/class-browsertype#browser-type-launch). An example
 * of using a [Browser] to create a [Page]:
 *
 * ```js
 * const { firefox } = require('playwright');  // Or 'chromium' or 'webkit'.
 *
 * (async () => {
 *   const browser = await firefox.launch();
 *   const page = await browser.newPage();
 *   await page.goto('https://example.com');
 *   await browser.close();
 * })();
 * ```
 *
 */
export interface Browser extends EventEmitter {
  /**
   * Emitted when Browser gets disconnected from the browser application. This might happen because of one of the
   * following:
   * - Browser application is closed or crashed.
   * - The [browser.close()](https://playwright.dev/docs/api/class-browser#browser-close) method was called.
   */
  on(event: 'disconnected', listener: (browser: Browser) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'disconnected', listener: (browser: Browser) => void): this;

  /**
   * Emitted when Browser gets disconnected from the browser application. This might happen because of one of the
   * following:
   * - Browser application is closed or crashed.
   * - The [browser.close()](https://playwright.dev/docs/api/class-browser#browser-close) method was called.
   */
  addListener(event: 'disconnected', listener: (browser: Browser) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'disconnected', listener: (browser: Browser) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'disconnected', listener: (browser: Browser) => void): this;

  /**
   * Emitted when Browser gets disconnected from the browser application. This might happen because of one of the
   * following:
   * - Browser application is closed or crashed.
   * - The [browser.close()](https://playwright.dev/docs/api/class-browser#browser-close) method was called.
   */
  prependListener(event: 'disconnected', listener: (browser: Browser) => void): this;

  /**
   * Get the browser type (chromium, firefox or webkit) that the browser belongs to.
   */
  browserType(): BrowserType;

  /**
   * In case this browser is obtained using
   * [browserType.launch([options])](https://playwright.dev/docs/api/class-browsertype#browser-type-launch), closes the
   * browser and all of its pages (if any were opened).
   *
   * In case this browser is connected to, clears all created contexts belonging to this browser and disconnects from
   * the browser server.
   *
   * **NOTE** This is similar to force quitting the browser. Therefore, you should call
   * [browserContext.close()](https://playwright.dev/docs/api/class-browsercontext#browser-context-close) on any
   * [BrowserContext]'s you explicitly created earlier with
   * [browser.newContext([options])](https://playwright.dev/docs/api/class-browser#browser-new-context) **before**
   * calling [browser.close()](https://playwright.dev/docs/api/class-browser#browser-close).
   *
   * The [Browser] object itself is considered to be disposed and cannot be used anymore.
   */
  close(): Promise<void>;

  /**
   * Returns an array of all open browser contexts. In a newly created browser, this will return zero browser contexts.
   *
   * **Usage**
   *
   * ```js
   * const browser = await pw.webkit.launch();
   * console.log(browser.contexts().length); // prints `0`
   *
   * const context = await browser.newContext();
   * console.log(browser.contexts().length); // prints `1`
   * ```
   *
   */
  contexts(): Array<BrowserContext>;

  /**
   * Indicates that the browser is connected.
   */
  isConnected(): boolean;

  /**
   * **NOTE** CDP Sessions are only supported on Chromium-based browsers.
   *
   * Returns the newly created browser session.
   */
  newBrowserCDPSession(): Promise<CDPSession>;

  /**
   * Creates a new browser context. It won't share cookies/cache with other browser contexts.
   *
   * **NOTE** If directly using this method to create [BrowserContext]s, it is best practice to explicitly close the
   * returned context via
   * [browserContext.close()](https://playwright.dev/docs/api/class-browsercontext#browser-context-close) when your code
   * is done with the [BrowserContext], and before calling
   * [browser.close()](https://playwright.dev/docs/api/class-browser#browser-close). This will ensure the `context` is
   * closed gracefully and any artifacts—like HARs and videos—are fully flushed and saved.
   *
   * **Usage**
   *
   * ```js
   * (async () => {
   *   const browser = await playwright.firefox.launch();  // Or 'chromium' or 'webkit'.
   *   // Create a new incognito browser context.
   *   const context = await browser.newContext();
   *   // Create a new page in a pristine context.
   *   const page = await context.newPage();
   *   await page.goto('https://example.com');
   *
   *   // Gracefully close up everything
   *   await context.close();
   *   await browser.close();
   * })();
   * ```
   *
   * @param options
   */
  newContext(options?: BrowserContextOptions): Promise<BrowserContext>;

  /**
   * Creates a new page in a new browser context. Closing this page will close the context as well.
   *
   * This is a convenience API that should only be used for the single-page scenarios and short snippets. Production
   * code and testing frameworks should explicitly create
   * [browser.newContext([options])](https://playwright.dev/docs/api/class-browser#browser-new-context) followed by the
   * [browserContext.newPage()](https://playwright.dev/docs/api/class-browsercontext#browser-context-new-page) to
   * control their exact life times.
   * @param options
   */
  newPage(options?: {
    /**
     * Whether to automatically download all the attachments. Defaults to `true` where all the downloads are accepted.
     */
    acceptDownloads?: boolean;

    /**
     * When using [page.goto(url[, options])](https://playwright.dev/docs/api/class-page#page-goto),
     * [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route),
     * [page.waitForURL(url[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-url),
     * [page.waitForRequest(urlOrPredicate[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-request),
     * or
     * [page.waitForResponse(urlOrPredicate[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-response)
     * it takes the base URL in consideration by using the
     * [`URL()`](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL) constructor for building the corresponding URL.
     * Examples:
     * - baseURL: `http://localhost:3000` and navigating to `/bar.html` results in `http://localhost:3000/bar.html`
     * - baseURL: `http://localhost:3000/foo/` and navigating to `./bar.html` results in
     *   `http://localhost:3000/foo/bar.html`
     * - baseURL: `http://localhost:3000/foo` (without trailing slash) and navigating to `./bar.html` results in
     *   `http://localhost:3000/bar.html`
     */
    baseURL?: string;

    /**
     * Toggles bypassing page's Content-Security-Policy.
     */
    bypassCSP?: boolean;

    /**
     * Emulates `'prefers-colors-scheme'` media feature, supported values are `'light'`, `'dark'`, `'no-preference'`. See
     * [page.emulateMedia([options])](https://playwright.dev/docs/api/class-page#page-emulate-media) for more details.
     * Passing `null` resets emulation to system defaults. Defaults to `'light'`.
     */
    colorScheme?: null|"light"|"dark"|"no-preference";

    /**
     * Specify device scale factor (can be thought of as dpr). Defaults to `1`.
     */
    deviceScaleFactor?: number;

    /**
     * An object containing additional HTTP headers to be sent with every request.
     */
    extraHTTPHeaders?: { [key: string]: string; };

    /**
     * Emulates `'forced-colors'` media feature, supported values are `'active'`, `'none'`. See
     * [page.emulateMedia([options])](https://playwright.dev/docs/api/class-page#page-emulate-media) for more details.
     * Passing `null` resets emulation to system defaults. Defaults to `'none'`.
     */
    forcedColors?: null|"active"|"none";

    geolocation?: {
      /**
       * Latitude between -90 and 90.
       */
      latitude: number;

      /**
       * Longitude between -180 and 180.
       */
      longitude: number;

      /**
       * Non-negative accuracy value. Defaults to `0`.
       */
      accuracy?: number;
    };

    /**
     * Specifies if viewport supports touch events. Defaults to false.
     */
    hasTouch?: boolean;

    /**
     * Credentials for [HTTP authentication](https://developer.mozilla.org/en-US/docs/Web/HTTP/Authentication).
     */
    httpCredentials?: {
      username: string;

      password: string;
    };

    /**
     * Whether to ignore HTTPS errors when sending network requests. Defaults to `false`.
     */
    ignoreHTTPSErrors?: boolean;

    /**
     * Whether the `meta viewport` tag is taken into account and touch events are enabled. Defaults to `false`. Not
     * supported in Firefox.
     */
    isMobile?: boolean;

    /**
     * Whether or not to enable JavaScript in the context. Defaults to `true`.
     */
    javaScriptEnabled?: boolean;

    /**
     * Specify user locale, for example `en-GB`, `de-DE`, etc. Locale will affect `navigator.language` value,
     * `Accept-Language` request header value as well as number and date formatting rules.
     */
    locale?: string;

    /**
     * Logger sink for Playwright logging.
     */
    logger?: Logger;

    /**
     * Whether to emulate network being offline. Defaults to `false`.
     */
    offline?: boolean;

    /**
     * A list of permissions to grant to all pages in this context. See
     * [browserContext.grantPermissions(permissions[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-grant-permissions)
     * for more details.
     */
    permissions?: Array<string>;

    /**
     * Network proxy settings to use with this context.
     *
     * **NOTE** For Chromium on Windows the browser needs to be launched with the global proxy for this option to work. If
     * all contexts override the proxy, global proxy will be never used and can be any string, for example `launch({
     * proxy: { server: 'http://per-context' } })`.
     */
    proxy?: {
      /**
       * Proxy to be used for all requests. HTTP and SOCKS proxies are supported, for example `http://myproxy.com:3128` or
       * `socks5://myproxy.com:3128`. Short form `myproxy.com:3128` is considered an HTTP proxy.
       */
      server: string;

      /**
       * Optional comma-separated domains to bypass proxy, for example `".com, chromium.org, .domain.com"`.
       */
      bypass?: string;

      /**
       * Optional username to use if HTTP proxy requires authentication.
       */
      username?: string;

      /**
       * Optional password to use if HTTP proxy requires authentication.
       */
      password?: string;
    };

    /**
     * Enables [HAR](http://www.softwareishard.com/blog/har-12-spec) recording for all pages into `recordHar.path` file.
     * If not specified, the HAR is not recorded. Make sure to await
     * [browserContext.close()](https://playwright.dev/docs/api/class-browsercontext#browser-context-close) for the HAR to
     * be saved.
     */
    recordHar?: {
      /**
       * Optional setting to control whether to omit request content from the HAR. Defaults to `false`. Deprecated, use
       * `content` policy instead.
       */
      omitContent?: boolean;

      /**
       * Optional setting to control resource content management. If `omit` is specified, content is not persisted. If
       * `attach` is specified, resources are persisted as separate files or entries in the ZIP archive. If `embed` is
       * specified, content is stored inline the HAR file as per HAR specification. Defaults to `attach` for `.zip` output
       * files and to `embed` for all other file extensions.
       */
      content?: "omit"|"embed"|"attach";

      /**
       * Path on the filesystem to write the HAR file to. If the file name ends with `.zip`, `content: 'attach'` is used by
       * default.
       */
      path: string;

      /**
       * When set to `minimal`, only record information necessary for routing from HAR. This omits sizes, timing, page,
       * cookies, security and other types of HAR information that are not used when replaying from HAR. Defaults to `full`.
       */
      mode?: "full"|"minimal";

      /**
       * A glob or regex pattern to filter requests that are stored in the HAR. When a `baseURL` via the context options was
       * provided and the passed URL is a path, it gets merged via the
       * [`new URL()`](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL) constructor.
       */
      urlFilter?: string|RegExp;
    };

    /**
     * Enables video recording for all pages into `recordVideo.dir` directory. If not specified videos are not recorded.
     * Make sure to await
     * [browserContext.close()](https://playwright.dev/docs/api/class-browsercontext#browser-context-close) for videos to
     * be saved.
     */
    recordVideo?: {
      /**
       * Path to the directory to put videos into.
       */
      dir: string;

      /**
       * Optional dimensions of the recorded videos. If not specified the size will be equal to `viewport` scaled down to
       * fit into 800x800. If `viewport` is not configured explicitly the video size defaults to 800x450. Actual picture of
       * each page will be scaled down if necessary to fit the specified size.
       */
      size?: {
        /**
         * Video frame width.
         */
        width: number;

        /**
         * Video frame height.
         */
        height: number;
      };
    };

    /**
     * Emulates `'prefers-reduced-motion'` media feature, supported values are `'reduce'`, `'no-preference'`. See
     * [page.emulateMedia([options])](https://playwright.dev/docs/api/class-page#page-emulate-media) for more details.
     * Passing `null` resets emulation to system defaults. Defaults to `'no-preference'`.
     */
    reducedMotion?: null|"reduce"|"no-preference";

    /**
     * Emulates consistent window screen size available inside web page via `window.screen`. Is only used when the
     * `viewport` is set.
     */
    screen?: {
      /**
       * page width in pixels.
       */
      width: number;

      /**
       * page height in pixels.
       */
      height: number;
    };

    /**
     * Whether to allow sites to register Service workers. Defaults to `'allow'`.
     * - `'allow'`: [Service Workers](https://developer.mozilla.org/en-US/docs/Web/API/Service_Worker_API) can be
     *   registered.
     * - `'block'`: Playwright will block all registration of Service Workers.
     */
    serviceWorkers?: "allow"|"block";

    /**
     * Populates context with given storage state. This option can be used to initialize context with logged-in
     * information obtained via
     * [browserContext.storageState([options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-storage-state).
     * Either a path to the file with saved storage, or an object with the following fields:
     */
    storageState?: string|{
      /**
       * cookies to set for context
       */
      cookies: Array<{
        name: string;

        value: string;

        /**
         * domain and path are required
         */
        domain: string;

        /**
         * domain and path are required
         */
        path: string;

        /**
         * Unix time in seconds.
         */
        expires: number;

        httpOnly: boolean;

        secure: boolean;

        /**
         * sameSite flag
         */
        sameSite: "Strict"|"Lax"|"None";
      }>;

      /**
       * localStorage to set for context
       */
      origins: Array<{
        origin: string;

        localStorage: Array<{
          name: string;

          value: string;
        }>;
      }>;
    };

    /**
     * If set to true, enables strict selectors mode for this context. In the strict selectors mode all operations on
     * selectors that imply single target DOM element will throw when more than one element matches the selector. This
     * option does not affect any Locator APIs (Locators are always strict). See [Locator] to learn more about the strict
     * mode.
     */
    strictSelectors?: boolean;

    /**
     * Changes the timezone of the context. See
     * [ICU's metaZones.txt](https://cs.chromium.org/chromium/src/third_party/icu/source/data/misc/metaZones.txt?rcl=faee8bc70570192d82d2978a71e2a615788597d1)
     * for a list of supported timezone IDs.
     */
    timezoneId?: string;

    /**
     * Specific user agent to use in this context.
     */
    userAgent?: string;

    /**
     * @deprecated Use `recordVideo` instead.
     */
    videoSize?: {
      /**
       * Video frame width.
       */
      width: number;

      /**
       * Video frame height.
       */
      height: number;
    };

    /**
     * @deprecated Use `recordVideo` instead.
     */
    videosPath?: string;

    /**
     * Emulates consistent viewport for each page. Defaults to an 1280x720 viewport. Use `null` to disable the consistent
     * viewport emulation.
     *
     * **NOTE** The `null` value opts out from the default presets, makes viewport depend on the host window size defined
     * by the operating system. It makes the execution of the tests non-deterministic.
     */
    viewport?: null|{
      /**
       * page width in pixels.
       */
      width: number;

      /**
       * page height in pixels.
       */
      height: number;
    };
  }): Promise<Page>;

  /**
   * **NOTE** This API controls
   * [Chromium Tracing](https://www.chromium.org/developers/how-tos/trace-event-profiling-tool) which is a low-level
   * chromium-specific debugging tool. API to control [Playwright Tracing](https://playwright.dev/docs/trace-viewer) could be found
   * [here](https://playwright.dev/docs/api/class-tracing).
   *
   * You can use
   * [browser.startTracing([page, options])](https://playwright.dev/docs/api/class-browser#browser-start-tracing) and
   * [browser.stopTracing()](https://playwright.dev/docs/api/class-browser#browser-stop-tracing) to create a trace file
   * that can be opened in Chrome DevTools performance panel.
   *
   * **Usage**
   *
   * ```js
   * await browser.startTracing(page, {path: 'trace.json'});
   * await page.goto('https://www.google.com');
   * await browser.stopTracing();
   * ```
   *
   * @param page Optional, if specified, tracing includes screenshots of the given page.
   * @param options
   */
  startTracing(page?: Page, options?: {
    /**
     * specify custom categories to use instead of default.
     */
    categories?: Array<string>;

    /**
     * A path to write the trace file to.
     */
    path?: string;

    /**
     * captures screenshots in the trace.
     */
    screenshots?: boolean;
  }): Promise<void>;

  /**
   * **NOTE** This API controls
   * [Chromium Tracing](https://www.chromium.org/developers/how-tos/trace-event-profiling-tool) which is a low-level
   * chromium-specific debugging tool. API to control [Playwright Tracing](https://playwright.dev/docs/trace-viewer) could be found
   * [here](https://playwright.dev/docs/api/class-tracing).
   *
   * Returns the buffer with trace data.
   */
  stopTracing(): Promise<Buffer>;

  /**
   * Returns the browser version.
   */
  version(): string;
}

export interface BrowserServer {
  /**
   * Emitted when the browser server closes.
   */
  on(event: 'close', listener: () => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'close', listener: () => void): this;

  /**
   * Emitted when the browser server closes.
   */
  addListener(event: 'close', listener: () => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'close', listener: () => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'close', listener: () => void): this;

  /**
   * Emitted when the browser server closes.
   */
  prependListener(event: 'close', listener: () => void): this;

  /**
   * Closes the browser gracefully and makes sure the process is terminated.
   */
  close(): Promise<void>;

  /**
   * Kills the browser process and waits for the process to exit.
   */
  kill(): Promise<void>;

  /**
   * Spawned browser application process.
   */
  process(): ChildProcess;

  /**
   * Browser websocket url.
   *
   * Browser websocket endpoint which can be used as an argument to
   * [browserType.connect(wsEndpoint[, options])](https://playwright.dev/docs/api/class-browsertype#browser-type-connect)
   * to establish connection to the browser.
   */
  wsEndpoint(): string;
}

/**
 * [ConsoleMessage] objects are dispatched by page via the
 * [page.on('console')](https://playwright.dev/docs/api/class-page#page-event-console) event. For each console
 * messages logged in the page there will be corresponding event in the Playwright context.
 *
 * ```js
 * // Listen for all console logs
 * page.on('console', msg => console.log(msg.text()))
 *
 * // Listen for all console events and handle errors
 * page.on('console', msg => {
 *   if (msg.type() === 'error')
 *     console.log(`Error text: "${msg.text()}"`);
 * });
 *
 * // Get the next console log
 * const msgPromise = page.waitForEvent('console');
 * await page.evaluate(() => {
 *   console.log('hello', 42, { foo: 'bar' });  // Issue console.log inside the page
 * });
 * const msg = await msgPromise;
 *
 * // Deconstruct console log arguments
 * await msg.args[0].jsonValue() // hello
 * await msg.args[1].jsonValue() // 42
 * ```
 *
 */
export interface ConsoleMessage {
  /**
   * List of arguments passed to a `console` function call. See also
   * [page.on('console')](https://playwright.dev/docs/api/class-page#page-event-console).
   */
  args(): Array<JSHandle>;

  location(): {
    /**
     * URL of the resource.
     */
    url: string;

    /**
     * 0-based line number in the resource.
     */
    lineNumber: number;

    /**
     * 0-based column number in the resource.
     */
    columnNumber: number;
  };

  /**
   * The text of the console message.
   */
  text(): string;

  /**
   * One of the following values: `'log'`, `'debug'`, `'info'`, `'error'`, `'warning'`, `'dir'`, `'dirxml'`, `'table'`,
   * `'trace'`, `'clear'`, `'startGroup'`, `'startGroupCollapsed'`, `'endGroup'`, `'assert'`, `'profile'`,
   * `'profileEnd'`, `'count'`, `'timeEnd'`.
   */
  type(): string;
}

/**
 * Coverage gathers information about parts of JavaScript and CSS that were used by the page.
 *
 * An example of using JavaScript coverage to produce Istanbul report for page load:
 *
 * **NOTE** Coverage APIs are only supported on Chromium-based browsers.
 *
 * ```js
 * const { chromium } = require('playwright');
 * const v8toIstanbul = require('v8-to-istanbul');
 *
 * (async() => {
 *   const browser = await chromium.launch();
 *   const page = await browser.newPage();
 *   await page.coverage.startJSCoverage();
 *   await page.goto('https://chromium.org');
 *   const coverage = await page.coverage.stopJSCoverage();
 *   for (const entry of coverage) {
 *     const converter = v8toIstanbul('', 0, { source: entry.source });
 *     await converter.load();
 *     converter.applyCoverage(entry.functions);
 *     console.log(JSON.stringify(converter.toIstanbul()));
 *   }
 *   await browser.close();
 * })();
 * ```
 *
 */
export interface Coverage {
  /**
   * Returns coverage is started
   * @param options
   */
  startCSSCoverage(options?: {
    /**
     * Whether to reset coverage on every navigation. Defaults to `true`.
     */
    resetOnNavigation?: boolean;
  }): Promise<void>;

  /**
   * Returns coverage is started
   *
   * **NOTE** Anonymous scripts are ones that don't have an associated url. These are scripts that are dynamically
   * created on the page using `eval` or `new Function`. If `reportAnonymousScripts` is set to `true`, anonymous scripts
   * will have `__playwright_evaluation_script__` as their URL.
   * @param options
   */
  startJSCoverage(options?: {
    /**
     * Whether anonymous scripts generated by the page should be reported. Defaults to `false`.
     */
    reportAnonymousScripts?: boolean;

    /**
     * Whether to reset coverage on every navigation. Defaults to `true`.
     */
    resetOnNavigation?: boolean;
  }): Promise<void>;

  /**
   * Returns the array of coverage reports for all stylesheets
   *
   * **NOTE** CSS Coverage doesn't include dynamically injected style tags without sourceURLs.
   */
  stopCSSCoverage(): Promise<Array<{
    /**
     * StyleSheet URL
     */
    url: string;

    /**
     * StyleSheet content, if available.
     */
    text?: string;

    /**
     * StyleSheet ranges that were used. Ranges are sorted and non-overlapping.
     */
    ranges: Array<{
      /**
       * A start offset in text, inclusive
       */
      start: number;

      /**
       * An end offset in text, exclusive
       */
      end: number;
    }>;
  }>>;

  /**
   * Returns the array of coverage reports for all scripts
   *
   * **NOTE** JavaScript Coverage doesn't include anonymous scripts by default. However, scripts with sourceURLs are
   * reported.
   */
  stopJSCoverage(): Promise<Array<{
    /**
     * Script URL
     */
    url: string;

    /**
     * Script ID
     */
    scriptId: string;

    /**
     * Script content, if applicable.
     */
    source?: string;

    /**
     * V8-specific coverage format.
     */
    functions: Array<{
      functionName: string;

      isBlockCoverage: boolean;

      ranges: Array<{
        count: number;

        startOffset: number;

        endOffset: number;
      }>;
    }>;
  }>>;
}

/**
 * [Dialog] objects are dispatched by page via the
 * [page.on('dialog')](https://playwright.dev/docs/api/class-page#page-event-dialog) event.
 *
 * An example of using `Dialog` class:
 *
 * ```js
 * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
 *
 * (async () => {
 *   const browser = await chromium.launch();
 *   const page = await browser.newPage();
 *   page.on('dialog', async dialog => {
 *     console.log(dialog.message());
 *     await dialog.dismiss();
 *   });
 *   await page.evaluate(() => alert('1'));
 *   await browser.close();
 * })();
 * ```
 *
 * **NOTE** Dialogs are dismissed automatically, unless there is a
 * [page.on('dialog')](https://playwright.dev/docs/api/class-page#page-event-dialog) listener. When listener is
 * present, it **must** either
 * [dialog.accept([promptText])](https://playwright.dev/docs/api/class-dialog#dialog-accept) or
 * [dialog.dismiss()](https://playwright.dev/docs/api/class-dialog#dialog-dismiss) the dialog - otherwise the page
 * will [freeze](https://developer.mozilla.org/en-US/docs/Web/JavaScript/EventLoop#never_blocking) waiting for the
 * dialog, and actions like click will never finish.
 */
export interface Dialog {
  /**
   * Returns when the dialog has been accepted.
   * @param promptText A text to enter in prompt. Does not cause any effects if the dialog's `type` is not prompt. Optional.
   */
  accept(promptText?: string): Promise<void>;

  /**
   * If dialog is prompt, returns default prompt value. Otherwise, returns empty string.
   */
  defaultValue(): string;

  /**
   * Returns when the dialog has been dismissed.
   */
  dismiss(): Promise<void>;

  /**
   * A message displayed in the dialog.
   */
  message(): string;

  /**
   * Returns dialog's type, can be one of `alert`, `beforeunload`, `confirm` or `prompt`.
   */
  type(): string;
}

/**
 * [Download] objects are dispatched by page via the
 * [page.on('download')](https://playwright.dev/docs/api/class-page#page-event-download) event.
 *
 * All the downloaded files belonging to the browser context are deleted when the browser context is closed.
 *
 * Download event is emitted once the download starts. Download path becomes available once download completes:
 *
 * ```js
 * // Start waiting for download before clicking. Note no await.
 * const downloadPromise = page.waitForEvent('download');
 * await page.getByText('Download file').click();
 * const download = await downloadPromise;
 *
 * // Wait for the download process to complete.
 * console.log(await download.path());
 * ```
 *
 */
export interface Download {
  /**
   * Cancels a download. Will not fail if the download is already finished or canceled. Upon successful cancellations,
   * `download.failure()` would resolve to `'canceled'`.
   */
  cancel(): Promise<void>;

  /**
   * Returns readable stream for current download or `null` if download failed.
   */
  createReadStream(): Promise<null|Readable>;

  /**
   * Deletes the downloaded file. Will wait for the download to finish if necessary.
   */
  delete(): Promise<void>;

  /**
   * Returns download error if any. Will wait for the download to finish if necessary.
   */
  failure(): Promise<null|string>;

  /**
   * Get the page that the download belongs to.
   */
  page(): Page;

  /**
   * Returns path to the downloaded file in case of successful download. The method will wait for the download to finish
   * if necessary. The method throws when connected remotely.
   *
   * Note that the download's file name is a random GUID, use
   * [download.suggestedFilename()](https://playwright.dev/docs/api/class-download#download-suggested-filename) to get
   * suggested file name.
   */
  path(): Promise<null|string>;

  /**
   * Copy the download to a user-specified path. It is safe to call this method while the download is still in progress.
   * Will wait for the download to finish if necessary.
   * @param path Path where the download should be copied.
   */
  saveAs(path: string): Promise<void>;

  /**
   * Returns suggested filename for this download. It is typically computed by the browser from the
   * [`Content-Disposition`](https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Content-Disposition) response
   * header or the `download` attribute. See the spec on [whatwg](https://html.spec.whatwg.org/#downloading-resources).
   * Different browsers can use different logic for computing it.
   */
  suggestedFilename(): string;

  /**
   * Returns downloaded url.
   */
  url(): string;
}

/**
 * Playwright has **experimental** support for Electron automation. You can access electron namespace via:
 *
 * ```js
 * const { _electron } = require('playwright');
 * ```
 *
 * An example of the Electron automation script would be:
 *
 * ```js
 * const { _electron: electron } = require('playwright');
 *
 * (async () => {
 *   // Launch Electron app.
 *   const electronApp = await electron.launch({ args: ['main.js'] });
 *
 *   // Evaluation expression in the Electron context.
 *   const appPath = await electronApp.evaluate(async ({ app }) => {
 *     // This runs in the main Electron process, parameter here is always
 *     // the result of the require('electron') in the main app script.
 *     return app.getAppPath();
 *   });
 *   console.log(appPath);
 *
 *   // Get the first window that the app opens, wait if necessary.
 *   const window = await electronApp.firstWindow();
 *   // Print the title.
 *   console.log(await window.title());
 *   // Capture a screenshot.
 *   await window.screenshot({ path: 'intro.png' });
 *   // Direct Electron console to Node terminal.
 *   window.on('console', console.log);
 *   // Click button.
 *   await window.click('text=Click me');
 *   // Exit app.
 *   await electronApp.close();
 * })();
 * ```
 *
 * Note that since you don't need Playwright to install web browsers when testing Electron, you can omit browser
 * download via setting the following environment variable when installing Playwright:
 *
 * ```bash
 * PLAYWRIGHT_SKIP_BROWSER_DOWNLOAD=1 npm i -D playwright
 * ```
 *
 * **Supported Electron versions are:**
 * - v12.2.0+
 * - v13.4.0+
 * - v14+
 *
 * **Known issues:**
 *
 * If you are not able to launch Electron and it will end up in timeouts during launch, try the following:
 * - Ensure that `nodeCliInspect`
 *   ([FuseV1Options.EnableNodeCliInspectArguments](https://www.electronjs.org/docs/latest/tutorial/fuses#nodecliinspect))
 *   fuse is **not** set to `false`.
 */
export interface Electron {
  /**
   * Launches electron application specified with the `executablePath`.
   * @param options
   */
  launch(options?: {
    /**
     * Whether to automatically download all the attachments. Defaults to `true` where all the downloads are accepted.
     */
    acceptDownloads?: boolean;

    /**
     * Additional arguments to pass to the application when launching. You typically pass the main script name here.
     */
    args?: Array<string>;

    /**
     * Toggles bypassing page's Content-Security-Policy.
     */
    bypassCSP?: boolean;

    /**
     * Emulates `'prefers-colors-scheme'` media feature, supported values are `'light'`, `'dark'`, `'no-preference'`. See
     * [page.emulateMedia([options])](https://playwright.dev/docs/api/class-page#page-emulate-media) for more details.
     * Passing `null` resets emulation to system defaults. Defaults to `'light'`.
     */
    colorScheme?: null|"light"|"dark"|"no-preference";

    /**
     * Current working directory to launch application from.
     */
    cwd?: string;

    /**
     * Specifies environment variables that will be visible to Electron. Defaults to `process.env`.
     */
    env?: { [key: string]: string; };

    /**
     * Launches given Electron application. If not specified, launches the default Electron executable installed in this
     * package, located at `node_modules/.bin/electron`.
     */
    executablePath?: string;

    /**
     * An object containing additional HTTP headers to be sent with every request.
     */
    extraHTTPHeaders?: { [key: string]: string; };

    geolocation?: {
      /**
       * Latitude between -90 and 90.
       */
      latitude: number;

      /**
       * Longitude between -180 and 180.
       */
      longitude: number;

      /**
       * Non-negative accuracy value. Defaults to `0`.
       */
      accuracy?: number;
    };

    /**
     * Credentials for [HTTP authentication](https://developer.mozilla.org/en-US/docs/Web/HTTP/Authentication).
     */
    httpCredentials?: {
      username: string;

      password: string;
    };

    /**
     * Whether to ignore HTTPS errors when sending network requests. Defaults to `false`.
     */
    ignoreHTTPSErrors?: boolean;

    /**
     * Specify user locale, for example `en-GB`, `de-DE`, etc. Locale will affect `navigator.language` value,
     * `Accept-Language` request header value as well as number and date formatting rules.
     */
    locale?: string;

    /**
     * Whether to emulate network being offline. Defaults to `false`.
     */
    offline?: boolean;

    /**
     * Enables [HAR](http://www.softwareishard.com/blog/har-12-spec) recording for all pages into `recordHar.path` file.
     * If not specified, the HAR is not recorded. Make sure to await
     * [browserContext.close()](https://playwright.dev/docs/api/class-browsercontext#browser-context-close) for the HAR to
     * be saved.
     */
    recordHar?: {
      /**
       * Optional setting to control whether to omit request content from the HAR. Defaults to `false`. Deprecated, use
       * `content` policy instead.
       */
      omitContent?: boolean;

      /**
       * Optional setting to control resource content management. If `omit` is specified, content is not persisted. If
       * `attach` is specified, resources are persisted as separate files or entries in the ZIP archive. If `embed` is
       * specified, content is stored inline the HAR file as per HAR specification. Defaults to `attach` for `.zip` output
       * files and to `embed` for all other file extensions.
       */
      content?: "omit"|"embed"|"attach";

      /**
       * Path on the filesystem to write the HAR file to. If the file name ends with `.zip`, `content: 'attach'` is used by
       * default.
       */
      path: string;

      /**
       * When set to `minimal`, only record information necessary for routing from HAR. This omits sizes, timing, page,
       * cookies, security and other types of HAR information that are not used when replaying from HAR. Defaults to `full`.
       */
      mode?: "full"|"minimal";

      /**
       * A glob or regex pattern to filter requests that are stored in the HAR. When a `baseURL` via the context options was
       * provided and the passed URL is a path, it gets merged via the
       * [`new URL()`](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL) constructor.
       */
      urlFilter?: string|RegExp;
    };

    /**
     * Enables video recording for all pages into `recordVideo.dir` directory. If not specified videos are not recorded.
     * Make sure to await
     * [browserContext.close()](https://playwright.dev/docs/api/class-browsercontext#browser-context-close) for videos to
     * be saved.
     */
    recordVideo?: {
      /**
       * Path to the directory to put videos into.
       */
      dir: string;

      /**
       * Optional dimensions of the recorded videos. If not specified the size will be equal to `viewport` scaled down to
       * fit into 800x800. If `viewport` is not configured explicitly the video size defaults to 800x450. Actual picture of
       * each page will be scaled down if necessary to fit the specified size.
       */
      size?: {
        /**
         * Video frame width.
         */
        width: number;

        /**
         * Video frame height.
         */
        height: number;
      };
    };

    /**
     * Maximum time in milliseconds to wait for the application to start. Defaults to `30000` (30 seconds). Pass `0` to
     * disable timeout.
     */
    timeout?: number;

    /**
     * Changes the timezone of the context. See
     * [ICU's metaZones.txt](https://cs.chromium.org/chromium/src/third_party/icu/source/data/misc/metaZones.txt?rcl=faee8bc70570192d82d2978a71e2a615788597d1)
     * for a list of supported timezone IDs.
     */
    timezoneId?: string;
  }): Promise<ElectronApplication>;
}

/**
 * [FileChooser] objects are dispatched by the page in the
 * [page.on('filechooser')](https://playwright.dev/docs/api/class-page#page-event-file-chooser) event.
 *
 * ```js
 * // Start waiting for file chooser before clicking. Note no await.
 * const fileChooserPromise = page.waitForEvent('filechooser');
 * await page.getByText('Upload file').click();
 * const fileChooser = await fileChooserPromise;
 * await fileChooser.setFiles('myfile.pdf');
 * ```
 *
 */
export interface FileChooser {
  /**
   * Returns input element associated with this file chooser.
   */
  element(): ElementHandle;

  /**
   * Returns whether this file chooser accepts multiple files.
   */
  isMultiple(): boolean;

  /**
   * Returns page this file chooser belongs to.
   */
  page(): Page;

  /**
   * Sets the value of the file input this chooser is associated with. If some of the `filePaths` are relative paths,
   * then they are resolved relative to the current working directory. For empty array, clears the selected files.
   * @param files
   * @param options
   */
  setFiles(files: string|Array<string>|{
    /**
     * File name
     */
    name: string;

    /**
     * File type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }|Array<{
    /**
     * File name
     */
    name: string;

    /**
     * File type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }>, options?: {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You
     * can opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as
     * navigating to inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
     * option in the config, or by using the
     * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
     * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
     */
    timeout?: number;
  }): Promise<void>;
}

/**
 * FrameLocator represents a view to the `iframe` on the page. It captures the logic sufficient to retrieve the
 * `iframe` and locate elements in that iframe. FrameLocator can be created with either
 * [page.frameLocator(selector)](https://playwright.dev/docs/api/class-page#page-frame-locator) or
 * [locator.frameLocator(selector)](https://playwright.dev/docs/api/class-locator#locator-frame-locator) method.
 *
 * ```js
 * const locator = page.frameLocator('#my-frame').getByText('Submit');
 * await locator.click();
 * ```
 *
 * **Strictness**
 *
 * Frame locators are strict. This means that all operations on frame locators will throw if more than one element
 * matches a given selector.
 *
 * ```js
 * // Throws if there are several frames in DOM:
 * await page.frameLocator('.result-frame').getByRole('button').click();
 *
 * // Works because we explicitly tell locator to pick the first frame:
 * await page.frameLocator('.result-frame').first().getByRole('button').click();
 * ```
 *
 * **Converting Locator to FrameLocator**
 *
 * If you have a [Locator] object pointing to an `iframe` it can be converted to [FrameLocator] using
 * [`:scope`](https://developer.mozilla.org/en-US/docs/Web/CSS/:scope) CSS selector:
 *
 * ```js
 * const frameLocator = locator.frameLocator(':scope');
 * ```
 *
 */
export interface FrameLocator {
  /**
   * Returns locator to the first matching frame.
   */
  first(): FrameLocator;

  /**
   * When working with iframes, you can create a frame locator that will enter the iframe and allow selecting elements
   * in that iframe.
   * @param selector A selector to use when resolving DOM element.
   */
  frameLocator(selector: string): FrameLocator;

  /**
   * Allows locating elements by their alt text.
   *
   * **Usage**
   *
   * For example, this method will find the image by alt text "Playwright logo":
   *
   * ```html
   * <img alt='Playwright logo'>
   * ```
   *
   * ```js
   * await page.getByAltText('Playwright logo').click();
   * ```
   *
   * @param text Text to locate the element for.
   * @param options
   */
  getByAltText(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Allows locating input elements by the text of the associated `<label>` or `aria-labelledby` element, or by the
   * `aria-label` attribute.
   *
   * **Usage**
   *
   * For example, this method will find inputs by label "Username" and "Password" in the following DOM:
   *
   * ```html
   * <input aria-label="Username">
   * <label for="password-input">Password:</label>
   * <input id="password-input">
   * ```
   *
   * ```js
   * await page.getByLabel('Username').fill('john');
   * await page.getByLabel('Password').fill('secret');
   * ```
   *
   * @param text Text to locate the element for.
   * @param options
   */
  getByLabel(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Allows locating input elements by the placeholder text.
   *
   * **Usage**
   *
   * For example, consider the following DOM structure.
   *
   * ```html
   * <input type="email" placeholder="name@example.com" />
   * ```
   *
   * You can fill the input after locating it by the placeholder text:
   *
   * ```js
   * await page
   *     .getByPlaceholder("name@example.com")
   *     .fill("playwright@microsoft.com");
   * ```
   *
   * @param text Text to locate the element for.
   * @param options
   */
  getByPlaceholder(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Allows locating elements by their [ARIA role](https://www.w3.org/TR/wai-aria-1.2/#roles),
   * [ARIA attributes](https://www.w3.org/TR/wai-aria-1.2/#aria-attributes) and
   * [accessible name](https://w3c.github.io/accname/#dfn-accessible-name).
   *
   * **Usage**
   *
   * Consider the following DOM structure.
   *
   * ```html
   * <h3>Sign up</h3>
   * <label>
   *   <input type="checkbox" /> Subscribe
   * </label>
   * <br/>
   * <button>Submit</button>
   * ```
   *
   * You can locate each element by it's implicit role:
   *
   * ```js
   * await expect(page.getByRole('heading', { name: 'Sign up' })).toBeVisible();
   *
   * await page.getByRole('checkbox', { name: 'Subscribe' }).check();
   *
   * await page.getByRole('button', { name: /submit/i }).click();
   * ```
   *
   * **Details**
   *
   * Role selector **does not replace** accessibility audits and conformance tests, but rather gives early feedback
   * about the ARIA guidelines.
   *
   * Many html elements have an implicitly [defined role](https://w3c.github.io/html-aam/#html-element-role-mappings)
   * that is recognized by the role selector. You can find all the
   * [supported roles here](https://www.w3.org/TR/wai-aria-1.2/#role_definitions). ARIA guidelines **do not recommend**
   * duplicating implicit roles and attributes by setting `role` and/or `aria-*` attributes to default values.
   * @param role Required aria role.
   * @param options
   */
  getByRole(role: "alert"|"alertdialog"|"application"|"article"|"banner"|"blockquote"|"button"|"caption"|"cell"|"checkbox"|"code"|"columnheader"|"combobox"|"complementary"|"contentinfo"|"definition"|"deletion"|"dialog"|"directory"|"document"|"emphasis"|"feed"|"figure"|"form"|"generic"|"grid"|"gridcell"|"group"|"heading"|"img"|"insertion"|"link"|"list"|"listbox"|"listitem"|"log"|"main"|"marquee"|"math"|"meter"|"menu"|"menubar"|"menuitem"|"menuitemcheckbox"|"menuitemradio"|"navigation"|"none"|"note"|"option"|"paragraph"|"presentation"|"progressbar"|"radio"|"radiogroup"|"region"|"row"|"rowgroup"|"rowheader"|"scrollbar"|"search"|"searchbox"|"separator"|"slider"|"spinbutton"|"status"|"strong"|"subscript"|"superscript"|"switch"|"tab"|"table"|"tablist"|"tabpanel"|"term"|"textbox"|"time"|"timer"|"toolbar"|"tooltip"|"tree"|"treegrid"|"treeitem", options?: {
    /**
     * An attribute that is usually set by `aria-checked` or native `<input type=checkbox>` controls.
     *
     * Learn more about [`aria-checked`](https://www.w3.org/TR/wai-aria-1.2/#aria-checked).
     */
    checked?: boolean;

    /**
     * An attribute that is usually set by `aria-disabled` or `disabled`.
     *
     * **NOTE** Unlike most other attributes, `disabled` is inherited through the DOM hierarchy. Learn more about
     * [`aria-disabled`](https://www.w3.org/TR/wai-aria-1.2/#aria-disabled).
     */
    disabled?: boolean;

    /**
     * Whether `name` is matched exactly: case-sensitive and whole-string. Defaults to false. Ignored when `name` is a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;

    /**
     * An attribute that is usually set by `aria-expanded`.
     *
     * Learn more about [`aria-expanded`](https://www.w3.org/TR/wai-aria-1.2/#aria-expanded).
     */
    expanded?: boolean;

    /**
     * Option that controls whether hidden elements are matched. By default, only non-hidden elements, as
     * [defined by ARIA](https://www.w3.org/TR/wai-aria-1.2/#tree_exclusion), are matched by role selector.
     *
     * Learn more about [`aria-hidden`](https://www.w3.org/TR/wai-aria-1.2/#aria-hidden).
     */
    includeHidden?: boolean;

    /**
     * A number attribute that is usually present for roles `heading`, `listitem`, `row`, `treeitem`, with default values
     * for `<h1>-<h6>` elements.
     *
     * Learn more about [`aria-level`](https://www.w3.org/TR/wai-aria-1.2/#aria-level).
     */
    level?: number;

    /**
     * Option to match the [accessible name](https://w3c.github.io/accname/#dfn-accessible-name). By default, matching is
     * case-insensitive and searches for a substring, use `exact` to control this behavior.
     *
     * Learn more about [accessible name](https://w3c.github.io/accname/#dfn-accessible-name).
     */
    name?: string|RegExp;

    /**
     * An attribute that is usually set by `aria-pressed`.
     *
     * Learn more about [`aria-pressed`](https://www.w3.org/TR/wai-aria-1.2/#aria-pressed).
     */
    pressed?: boolean;

    /**
     * An attribute that is usually set by `aria-selected`.
     *
     * Learn more about [`aria-selected`](https://www.w3.org/TR/wai-aria-1.2/#aria-selected).
     */
    selected?: boolean;
  }): Locator;

  /**
   * Locate element by the test id.
   *
   * **Usage**
   *
   * Consider the following DOM structure.
   *
   * ```html
   * <button data-testid="directions">Itinéraire</button>
   * ```
   *
   * You can locate the element by it's test id:
   *
   * ```js
   * await page.getByTestId('directions').click();
   * ```
   *
   * **Details**
   *
   * By default, the `data-testid` attribute is used as a test id. Use
   * [selectors.setTestIdAttribute(attributeName)](https://playwright.dev/docs/api/class-selectors#selectors-set-test-id-attribute)
   * to configure a different test id attribute if necessary.
   *
   * ```js
   * // Set custom test id attribute from @playwright/test config:
   * use: {
   *   testIdAttribute: 'data-pw'
   * }
   * ```
   *
   * @param testId Id to locate the element by.
   */
  getByTestId(testId: string|RegExp): Locator;

  /**
   * Allows locating elements that contain given text.
   *
   * See also [locator.filter([options])](https://playwright.dev/docs/api/class-locator#locator-filter) that allows to
   * match by another criteria, like an accessible role, and then filter by the text content.
   *
   * **Usage**
   *
   * Consider the following DOM structure:
   *
   * ```html
   * <div>Hello <span>world</span></div>
   * <div>Hello</div>
   * ```
   *
   * You can locate by text substring, exact string, or a regular expression:
   *
   * ```js
   * // Matches <span>
   * page.getByText('world')
   *
   * // Matches first <div>
   * page.getByText('Hello world')
   *
   * // Matches second <div>
   * page.getByText('Hello', { exact: true })
   *
   * // Matches both <div>s
   * page.getByText(/Hello/)
   *
   * // Matches second <div>
   * page.getByText(/^hello$/i)
   * ```
   *
   * **Details**
   *
   * Matching by text always normalizes whitespace, even with exact match. For example, it turns multiple spaces into
   * one, turns line breaks into spaces and ignores leading and trailing whitespace.
   *
   * Input elements of the type `button` and `submit` are matched by their `value` instead of the text content. For
   * example, locating by text `"Log in"` matches `<input type=button value="Log in">`.
   * @param text Text to locate the element for.
   * @param options
   */
  getByText(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Allows locating elements by their title attribute.
   *
   * **Usage**
   *
   * Consider the following DOM structure.
   *
   * ```html
   * <span title='Issues count'>25 issues</span>
   * ```
   *
   * You can check the issues count after locating it by the title text:
   *
   * ```js
   * await expect(page.getByTitle('Issues count')).toHaveText('25 issues');
   * ```
   *
   * @param text Text to locate the element for.
   * @param options
   */
  getByTitle(text: string|RegExp, options?: {
    /**
     * Whether to find an exact match: case-sensitive and whole-string. Default to false. Ignored when locating by a
     * regular expression. Note that exact match still trims whitespace.
     */
    exact?: boolean;
  }): Locator;

  /**
   * Returns locator to the last matching frame.
   */
  last(): FrameLocator;

  /**
   * The method finds an element matching the specified selector in the locator's subtree. It also accepts filter
   * options, similar to [locator.filter([options])](https://playwright.dev/docs/api/class-locator#locator-filter)
   * method.
   *
   * [Learn more about locators](https://playwright.dev/docs/locators).
   * @param selectorOrLocator A selector or locator to use when resolving DOM element.
   * @param options
   */
  locator(selectorOrLocator: string|Locator, options?: {
    /**
     * Matches elements containing an element that matches an inner locator. Inner locator is queried against the outer
     * one. For example, `article` that has `text=Playwright` matches `<article><div>Playwright</div></article>`.
     *
     * Note that outer and inner locators must belong to the same frame. Inner locator must not contain [FrameLocator]s.
     */
    has?: Locator;

    /**
     * Matches elements containing specified text somewhere inside, possibly in a child or a descendant element. When
     * passed a [string], matching is case-insensitive and searches for a substring. For example, `"Playwright"` matches
     * `<article><div>Playwright</div></article>`.
     */
    hasText?: string|RegExp;
  }): Locator;

  /**
   * Returns locator to the n-th matching frame. It's zero based, `nth(0)` selects the first frame.
   * @param index
   */
  nth(index: number): FrameLocator;
}

/**
 * Keyboard provides an api for managing a virtual keyboard. The high level api is
 * [keyboard.type(text[, options])](https://playwright.dev/docs/api/class-keyboard#keyboard-type), which takes raw
 * characters and generates proper `keydown`, `keypress`/`input`, and `keyup` events on your page.
 *
 * For finer control, you can use [keyboard.down(key)](https://playwright.dev/docs/api/class-keyboard#keyboard-down),
 * [keyboard.up(key)](https://playwright.dev/docs/api/class-keyboard#keyboard-up), and
 * [keyboard.insertText(text)](https://playwright.dev/docs/api/class-keyboard#keyboard-insert-text) to manually fire
 * events as if they were generated from a real keyboard.
 *
 * An example of holding down `Shift` in order to select and delete some text:
 *
 * ```js
 * await page.keyboard.type('Hello World!');
 * await page.keyboard.press('ArrowLeft');
 *
 * await page.keyboard.down('Shift');
 * for (let i = 0; i < ' World'.length; i++)
 *   await page.keyboard.press('ArrowLeft');
 * await page.keyboard.up('Shift');
 *
 * await page.keyboard.press('Backspace');
 * // Result text will end up saying 'Hello!'
 * ```
 *
 * An example of pressing uppercase `A`
 *
 * ```js
 * await page.keyboard.press('Shift+KeyA');
 * // or
 * await page.keyboard.press('Shift+A');
 * ```
 *
 * An example to trigger select-all with the keyboard
 *
 * ```js
 * // on Windows and Linux
 * await page.keyboard.press('Control+A');
 * // on macOS
 * await page.keyboard.press('Meta+A');
 * ```
 *
 */
export interface Keyboard {
  /**
   * Dispatches a `keydown` event.
   *
   * `key` can specify the intended
   * [keyboardEvent.key](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key) value or a single character
   * to generate the text for. A superset of the `key` values can be found
   * [here](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key/Key_Values). Examples of the keys are:
   *
   * `F1` - `F12`, `Digit0`- `Digit9`, `KeyA`- `KeyZ`, `Backquote`, `Minus`, `Equal`, `Backslash`, `Backspace`, `Tab`,
   * `Delete`, `Escape`, `ArrowDown`, `End`, `Enter`, `Home`, `Insert`, `PageDown`, `PageUp`, `ArrowRight`, `ArrowUp`,
   * etc.
   *
   * Following modification shortcuts are also supported: `Shift`, `Control`, `Alt`, `Meta`, `ShiftLeft`.
   *
   * Holding down `Shift` will type the text that corresponds to the `key` in the upper case.
   *
   * If `key` is a single character, it is case-sensitive, so the values `a` and `A` will generate different respective
   * texts.
   *
   * If `key` is a modifier key, `Shift`, `Meta`, `Control`, or `Alt`, subsequent key presses will be sent with that
   * modifier active. To release the modifier key, use
   * [keyboard.up(key)](https://playwright.dev/docs/api/class-keyboard#keyboard-up).
   *
   * After the key is pressed once, subsequent calls to
   * [keyboard.down(key)](https://playwright.dev/docs/api/class-keyboard#keyboard-down) will have
   * [repeat](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/repeat) set to true. To release the key,
   * use [keyboard.up(key)](https://playwright.dev/docs/api/class-keyboard#keyboard-up).
   *
   * **NOTE** Modifier keys DO influence `keyboard.down`. Holding down `Shift` will type the text in upper case.
   * @param key Name of the key to press or a character to generate, such as `ArrowLeft` or `a`.
   */
  down(key: string): Promise<void>;

  /**
   * Dispatches only `input` event, does not emit the `keydown`, `keyup` or `keypress` events.
   *
   * **Usage**
   *
   * ```js
   * page.keyboard.insertText('嗨');
   * ```
   *
   * **NOTE** Modifier keys DO NOT effect `keyboard.insertText`. Holding down `Shift` will not type the text in upper
   * case.
   * @param text Sets input to the specified text value.
   */
  insertText(text: string): Promise<void>;

  /**
   * `key` can specify the intended
   * [keyboardEvent.key](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key) value or a single character
   * to generate the text for. A superset of the `key` values can be found
   * [here](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key/Key_Values). Examples of the keys are:
   *
   * `F1` - `F12`, `Digit0`- `Digit9`, `KeyA`- `KeyZ`, `Backquote`, `Minus`, `Equal`, `Backslash`, `Backspace`, `Tab`,
   * `Delete`, `Escape`, `ArrowDown`, `End`, `Enter`, `Home`, `Insert`, `PageDown`, `PageUp`, `ArrowRight`, `ArrowUp`,
   * etc.
   *
   * Following modification shortcuts are also supported: `Shift`, `Control`, `Alt`, `Meta`, `ShiftLeft`.
   *
   * Holding down `Shift` will type the text that corresponds to the `key` in the upper case.
   *
   * If `key` is a single character, it is case-sensitive, so the values `a` and `A` will generate different respective
   * texts.
   *
   * Shortcuts such as `key: "Control+o"` or `key: "Control+Shift+T"` are supported as well. When specified with the
   * modifier, modifier is pressed and being held while the subsequent key is being pressed.
   *
   * **Usage**
   *
   * ```js
   * const page = await browser.newPage();
   * await page.goto('https://keycode.info');
   * await page.keyboard.press('A');
   * await page.screenshot({ path: 'A.png' });
   * await page.keyboard.press('ArrowLeft');
   * await page.screenshot({ path: 'ArrowLeft.png' });
   * await page.keyboard.press('Shift+O');
   * await page.screenshot({ path: 'O.png' });
   * await browser.close();
   * ```
   *
   * Shortcut for [keyboard.down(key)](https://playwright.dev/docs/api/class-keyboard#keyboard-down) and
   * [keyboard.up(key)](https://playwright.dev/docs/api/class-keyboard#keyboard-up).
   * @param key Name of the key to press or a character to generate, such as `ArrowLeft` or `a`.
   * @param options
   */
  press(key: string, options?: {
    /**
     * Time to wait between `keydown` and `keyup` in milliseconds. Defaults to 0.
     */
    delay?: number;
  }): Promise<void>;

  /**
   * Sends a `keydown`, `keypress`/`input`, and `keyup` event for each character in the text.
   *
   * To press a special key, like `Control` or `ArrowDown`, use
   * [keyboard.press(key[, options])](https://playwright.dev/docs/api/class-keyboard#keyboard-press).
   *
   * **Usage**
   *
   * ```js
   * await page.keyboard.type('Hello'); // Types instantly
   * await page.keyboard.type('World', {delay: 100}); // Types slower, like a user
   * ```
   *
   * **NOTE** Modifier keys DO NOT effect `keyboard.type`. Holding down `Shift` will not type the text in upper case.
   *
   * **NOTE** For characters that are not on a US keyboard, only an `input` event will be sent.
   * @param text A text to type into a focused element.
   * @param options
   */
  type(text: string, options?: {
    /**
     * Time to wait between key presses in milliseconds. Defaults to 0.
     */
    delay?: number;
  }): Promise<void>;

  /**
   * Dispatches a `keyup` event.
   * @param key Name of the key to press or a character to generate, such as `ArrowLeft` or `a`.
   */
  up(key: string): Promise<void>;
}

/**
 * Playwright generates a lot of logs and they are accessible via the pluggable logger sink.
 *
 * ```js
 * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
 *
 * (async () => {
 *   const browser = await chromium.launch({
 *     logger: {
 *       isEnabled: (name, severity) => name === 'browser',
 *       log: (name, severity, message, args) => console.log(`${name} ${message}`)
 *     }
 *   });
 *   ...
 * })();
 * ```
 *
 */
export interface Logger {
  /**
   * Determines whether sink is interested in the logger with the given name and severity.
   * @param name logger name
   * @param severity
   */
  isEnabled(name: string, severity: "verbose"|"info"|"warning"|"error"): boolean;

  /**
   * @param name logger name
   * @param severity
   * @param message log message format
   * @param args message arguments
   * @param hints optional formatting hints
   */
  log(name: string, severity: "verbose"|"info"|"warning"|"error", message: string|Error, args: Array<Object>, hints: {
    /**
     * Optional preferred logger color.
     */
    color?: string;
  }): void;
}

/**
 * The Mouse class operates in main-frame CSS pixels relative to the top-left corner of the viewport.
 *
 * Every `page` object has its own Mouse, accessible with
 * [page.mouse](https://playwright.dev/docs/api/class-page#page-mouse).
 *
 * ```js
 * // Using ‘page.mouse’ to trace a 100x100 square.
 * await page.mouse.move(0, 0);
 * await page.mouse.down();
 * await page.mouse.move(0, 100);
 * await page.mouse.move(100, 100);
 * await page.mouse.move(100, 0);
 * await page.mouse.move(0, 0);
 * await page.mouse.up();
 * ```
 *
 */
export interface Mouse {
  /**
   * Shortcut for [mouse.move(x, y[, options])](https://playwright.dev/docs/api/class-mouse#mouse-move),
   * [mouse.down([options])](https://playwright.dev/docs/api/class-mouse#mouse-down),
   * [mouse.up([options])](https://playwright.dev/docs/api/class-mouse#mouse-up).
   * @param x
   * @param y
   * @param options
   */
  click(x: number, y: number, options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * defaults to 1. See [UIEvent.detail].
     */
    clickCount?: number;

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;
  }): Promise<void>;

  /**
   * Shortcut for [mouse.move(x, y[, options])](https://playwright.dev/docs/api/class-mouse#mouse-move),
   * [mouse.down([options])](https://playwright.dev/docs/api/class-mouse#mouse-down),
   * [mouse.up([options])](https://playwright.dev/docs/api/class-mouse#mouse-up),
   * [mouse.down([options])](https://playwright.dev/docs/api/class-mouse#mouse-down) and
   * [mouse.up([options])](https://playwright.dev/docs/api/class-mouse#mouse-up).
   * @param x
   * @param y
   * @param options
   */
  dblclick(x: number, y: number, options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;
  }): Promise<void>;

  /**
   * Dispatches a `mousedown` event.
   * @param options
   */
  down(options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * defaults to 1. See [UIEvent.detail].
     */
    clickCount?: number;
  }): Promise<void>;

  /**
   * Dispatches a `mousemove` event.
   * @param x
   * @param y
   * @param options
   */
  move(x: number, y: number, options?: {
    /**
     * Defaults to 1. Sends intermediate `mousemove` events.
     */
    steps?: number;
  }): Promise<void>;

  /**
   * Dispatches a `mouseup` event.
   * @param options
   */
  up(options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * defaults to 1. See [UIEvent.detail].
     */
    clickCount?: number;
  }): Promise<void>;

  /**
   * Dispatches a `wheel` event.
   *
   * **NOTE** Wheel events may cause scrolling if they are not handled, and this method does not wait for the scrolling
   * to finish before returning.
   * @param deltaX Pixels to scroll horizontally.
   * @param deltaY Pixels to scroll vertically.
   */
  wheel(deltaX: number, deltaY: number): Promise<void>;
}

/**
 * This object can be used to launch or connect to Chromium, returning instances of [Browser].
 */
export const chromium: BrowserType;

/**
 * This object can be used to launch or connect to Firefox, returning instances of [Browser].
 */
export const firefox: BrowserType;

/**
 * Exposes API that can be used for the Web API testing.
 */
export const request: APIRequest;

/**
 * Selectors can be used to install custom selector engines. See [extensibility](https://playwright.dev/docs/extensibility) for more
 * information.
 */
export const selectors: Selectors;

/**
 * This object can be used to launch or connect to WebKit, returning instances of [Browser].
 */
export const webkit: BrowserType;

/**
 * Whenever the page sends a request for a network resource the following sequence of events are emitted by [Page]:
 * - [page.on('request')](https://playwright.dev/docs/api/class-page#page-event-request) emitted when the request is
 *   issued by the page.
 * - [page.on('response')](https://playwright.dev/docs/api/class-page#page-event-response) emitted when/if the
 *   response status and headers are received for the request.
 * - [page.on('requestfinished')](https://playwright.dev/docs/api/class-page#page-event-request-finished) emitted
 *   when the response body is downloaded and the request is complete.
 *
 * If request fails at some point, then instead of `'requestfinished'` event (and possibly instead of 'response'
 * event), the  [page.on('requestfailed')](https://playwright.dev/docs/api/class-page#page-event-request-failed) event
 * is emitted.
 *
 * **NOTE** HTTP Error responses, such as 404 or 503, are still successful responses from HTTP standpoint, so request
 * will complete with `'requestfinished'` event.
 *
 * If request gets a 'redirect' response, the request is successfully finished with the `requestfinished` event, and a
 * new request is  issued to a redirected url.
 */
export interface Request {
  /**
   * An object with all the request HTTP headers associated with this request. The header names are lower-cased.
   */
  allHeaders(): Promise<{ [key: string]: string; }>;

  /**
   * The method returns `null` unless this request has failed, as reported by `requestfailed` event.
   *
   * **Usage**
   *
   * Example of logging of all the failed requests:
   *
   * ```js
   * page.on('requestfailed', request => {
   *   console.log(request.url() + ' ' + request.failure().errorText);
   * });
   * ```
   *
   */
  failure(): null|{
    /**
     * Human-readable error message, e.g. `'net::ERR_FAILED'`.
     */
    errorText: string;
  };

  /**
   * Returns the [Frame] that initiated this request.
   */
  frame(): Frame;

  /**
   * An object with the request HTTP headers. The header names are lower-cased. Note that this method does not return
   * security-related headers, including cookie-related ones. You can use
   * [request.allHeaders()](https://playwright.dev/docs/api/class-request#request-all-headers) for complete list of
   * headers that include `cookie` information.
   */
  headers(): { [key: string]: string; };

  /**
   * An array with all the request HTTP headers associated with this request. Unlike
   * [request.allHeaders()](https://playwright.dev/docs/api/class-request#request-all-headers), header names are NOT
   * lower-cased. Headers with multiple entries, such as `Set-Cookie`, appear in the array multiple times.
   */
  headersArray(): Promise<Array<{
    /**
     * Name of the header.
     */
    name: string;

    /**
     * Value of the header.
     */
    value: string;
  }>>;

  /**
   * Returns the value of the header matching the name. The name is case insensitive.
   * @param name Name of the header.
   */
  headerValue(name: string): Promise<null|string>;

  /**
   * Whether this request is driving frame's navigation.
   */
  isNavigationRequest(): boolean;

  /**
   * Request's method (GET, POST, etc.)
   */
  method(): string;

  /**
   * Request's post body, if any.
   */
  postData(): null|string;

  /**
   * Request's post body in a binary form, if any.
   */
  postDataBuffer(): null|Buffer;

  /**
   * Returns parsed request's body for `form-urlencoded` and JSON as a fallback if any.
   *
   * When the response is `application/x-www-form-urlencoded` then a key/value object of the values will be returned.
   * Otherwise it will be parsed as JSON.
   */
  postDataJSON(): null|Serializable;

  /**
   * Request that was redirected by the server to this one, if any.
   *
   * When the server responds with a redirect, Playwright creates a new [Request] object. The two requests are connected
   * by `redirectedFrom()` and `redirectedTo()` methods. When multiple server redirects has happened, it is possible to
   * construct the whole redirect chain by repeatedly calling `redirectedFrom()`.
   *
   * **Usage**
   *
   * For example, if the website `http://example.com` redirects to `https://example.com`:
   *
   * ```js
   * const response = await page.goto('http://example.com');
   * console.log(response.request().redirectedFrom().url()); // 'http://example.com'
   * ```
   *
   * If the website `https://google.com` has no redirects:
   *
   * ```js
   * const response = await page.goto('https://google.com');
   * console.log(response.request().redirectedFrom()); // null
   * ```
   *
   */
  redirectedFrom(): null|Request;

  /**
   * New request issued by the browser if the server responded with redirect.
   *
   * **Usage**
   *
   * This method is the opposite of
   * [request.redirectedFrom()](https://playwright.dev/docs/api/class-request#request-redirected-from):
   *
   * ```js
   * console.log(request.redirectedFrom().redirectedTo() === request); // true
   * ```
   *
   */
  redirectedTo(): null|Request;

  /**
   * Contains the request's resource type as it was perceived by the rendering engine. ResourceType will be one of the
   * following: `document`, `stylesheet`, `image`, `media`, `font`, `script`, `texttrack`, `xhr`, `fetch`,
   * `eventsource`, `websocket`, `manifest`, `other`.
   */
  resourceType(): string;

  /**
   * Returns the matching [Response] object, or `null` if the response was not received due to error.
   */
  response(): Promise<null|Response>;

  /**
   * **NOTE** This field is Chromium only. It's safe to call when using other browsers, but it will always be `null`.
   *
   * The Service [Worker] that is performing the request.
   */
  serviceWorker(): null|Worker;

  /**
   * Returns resource size information for given request.
   */
  sizes(): Promise<{
    /**
     * Size of the request body (POST data payload) in bytes. Set to 0 if there was no body.
     */
    requestBodySize: number;

    /**
     * Total number of bytes from the start of the HTTP request message until (and including) the double CRLF before the
     * body.
     */
    requestHeadersSize: number;

    /**
     * Size of the received response body (encoded) in bytes.
     */
    responseBodySize: number;

    /**
     * Total number of bytes from the start of the HTTP response message until (and including) the double CRLF before the
     * body.
     */
    responseHeadersSize: number;
  }>;

  /**
   * Returns resource timing information for given request. Most of the timing values become available upon the
   * response, `responseEnd` becomes available when request finishes. Find more information at
   * [Resource Timing API](https://developer.mozilla.org/en-US/docs/Web/API/PerformanceResourceTiming).
   *
   * **Usage**
   *
   * ```js
   * const requestFinishedPromise = page.waitForEvent('requestfinished');
   * await page.goto('http://example.com');
   * const request = await requestFinishedPromise;
   * console.log(request.timing());
   * ```
   *
   */
  timing(): {
    /**
     * Request start time in milliseconds elapsed since January 1, 1970 00:00:00 UTC
     */
    startTime: number;

    /**
     * Time immediately before the browser starts the domain name lookup for the resource. The value is given in
     * milliseconds relative to `startTime`, -1 if not available.
     */
    domainLookupStart: number;

    /**
     * Time immediately after the browser starts the domain name lookup for the resource. The value is given in
     * milliseconds relative to `startTime`, -1 if not available.
     */
    domainLookupEnd: number;

    /**
     * Time immediately before the user agent starts establishing the connection to the server to retrieve the resource.
     * The value is given in milliseconds relative to `startTime`, -1 if not available.
     */
    connectStart: number;

    /**
     * Time immediately before the browser starts the handshake process to secure the current connection. The value is
     * given in milliseconds relative to `startTime`, -1 if not available.
     */
    secureConnectionStart: number;

    /**
     * Time immediately before the user agent starts establishing the connection to the server to retrieve the resource.
     * The value is given in milliseconds relative to `startTime`, -1 if not available.
     */
    connectEnd: number;

    /**
     * Time immediately before the browser starts requesting the resource from the server, cache, or local resource. The
     * value is given in milliseconds relative to `startTime`, -1 if not available.
     */
    requestStart: number;

    /**
     * Time immediately after the browser starts requesting the resource from the server, cache, or local resource. The
     * value is given in milliseconds relative to `startTime`, -1 if not available.
     */
    responseStart: number;

    /**
     * Time immediately after the browser receives the last byte of the resource or immediately before the transport
     * connection is closed, whichever comes first. The value is given in milliseconds relative to `startTime`, -1 if not
     * available.
     */
    responseEnd: number;
  };

  /**
   * URL of the request.
   */
  url(): string;
}

/**
 * [Response] class represents responses which are received by page.
 */
export interface Response {
  /**
   * An object with all the response HTTP headers associated with this response.
   */
  allHeaders(): Promise<{ [key: string]: string; }>;

  /**
   * Returns the buffer with response body.
   */
  body(): Promise<Buffer>;

  /**
   * Waits for this response to finish, returns always `null`.
   */
  finished(): Promise<null|Error>;

  /**
   * Returns the [Frame] that initiated this response.
   */
  frame(): Frame;

  /**
   * Indicates whether this Response was fulfilled by a Service Worker's Fetch Handler (i.e. via
   * [FetchEvent.respondWith](https://developer.mozilla.org/en-US/docs/Web/API/FetchEvent/respondWith)).
   */
  fromServiceWorker(): boolean;

  /**
   * An object with the response HTTP headers. The header names are lower-cased. Note that this method does not return
   * security-related headers, including cookie-related ones. You can use
   * [response.allHeaders()](https://playwright.dev/docs/api/class-response#response-all-headers) for complete list of
   * headers that include `cookie` information.
   */
  headers(): { [key: string]: string; };

  /**
   * An array with all the request HTTP headers associated with this response. Unlike
   * [response.allHeaders()](https://playwright.dev/docs/api/class-response#response-all-headers), header names are NOT
   * lower-cased. Headers with multiple entries, such as `Set-Cookie`, appear in the array multiple times.
   */
  headersArray(): Promise<Array<{
    /**
     * Name of the header.
     */
    name: string;

    /**
     * Value of the header.
     */
    value: string;
  }>>;

  /**
   * Returns the value of the header matching the name. The name is case insensitive. If multiple headers have the same
   * name (except `set-cookie`), they are returned as a list separated by `, `. For `set-cookie`, the `\n` separator is
   * used. If no headers are found, `null` is returned.
   * @param name Name of the header.
   */
  headerValue(name: string): Promise<null|string>;

  /**
   * Returns all values of the headers matching the name, for example `set-cookie`. The name is case insensitive.
   * @param name Name of the header.
   */
  headerValues(name: string): Promise<Array<string>>;

  /**
   * Returns the JSON representation of response body.
   *
   * This method will throw if the response body is not parsable via `JSON.parse`.
   */
  json(): Promise<Serializable>;

  /**
   * Contains a boolean stating whether the response was successful (status in the range 200-299) or not.
   */
  ok(): boolean;

  /**
   * Returns the matching [Request] object.
   */
  request(): Request;

  /**
   * Returns SSL and other security information.
   */
  securityDetails(): Promise<null|{
    /**
     * Common Name component of the Issuer field. from the certificate. This should only be used for informational
     * purposes. Optional.
     */
    issuer?: string;

    /**
     * The specific TLS protocol used. (e.g. `TLS 1.3`). Optional.
     */
    protocol?: string;

    /**
     * Common Name component of the Subject field from the certificate. This should only be used for informational
     * purposes. Optional.
     */
    subjectName?: string;

    /**
     * Unix timestamp (in seconds) specifying when this cert becomes valid. Optional.
     */
    validFrom?: number;

    /**
     * Unix timestamp (in seconds) specifying when this cert becomes invalid. Optional.
     */
    validTo?: number;
  }>;

  /**
   * Returns the IP address and port of the server.
   */
  serverAddr(): Promise<null|{
    /**
     * IPv4 or IPV6 address of the server.
     */
    ipAddress: string;

    port: number;
  }>;

  /**
   * Contains the status code of the response (e.g., 200 for a success).
   */
  status(): number;

  /**
   * Contains the status text of the response (e.g. usually an "OK" for a success).
   */
  statusText(): string;

  /**
   * Returns the text representation of response body.
   */
  text(): Promise<string>;

  /**
   * Contains the URL of the response.
   */
  url(): string;
}

/**
 * Whenever a network route is set up with
 * [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route) or
 * [browserContext.route(url, handler[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-route),
 * the `Route` object allows to handle the route.
 *
 * Learn more about [networking](https://playwright.dev/docs/network).
 */
export interface Route {
  /**
   * Aborts the route's request.
   * @param errorCode Optional error code. Defaults to `failed`, could be one of the following:
   * - `'aborted'` - An operation was aborted (due to user action)
   * - `'accessdenied'` - Permission to access a resource, other than the network, was denied
   * - `'addressunreachable'` - The IP address is unreachable. This usually means that there is no route to the
   * specified host or network.
   * - `'blockedbyclient'` - The client chose to block the request.
   * - `'blockedbyresponse'` - The request failed because the response was delivered along with requirements which are
   * not met ('X-Frame-Options' and 'Content-Security-Policy' ancestor checks, for instance).
   * - `'connectionaborted'` - A connection timed out as a result of not receiving an ACK for data sent.
   * - `'connectionclosed'` - A connection was closed (corresponding to a TCP FIN).
   * - `'connectionfailed'` - A connection attempt failed.
   * - `'connectionrefused'` - A connection attempt was refused.
   * - `'connectionreset'` - A connection was reset (corresponding to a TCP RST).
   * - `'internetdisconnected'` - The Internet connection has been lost.
   * - `'namenotresolved'` - The host name could not be resolved.
   * - `'timedout'` - An operation timed out.
   * - `'failed'` - A generic failure occurred.
   */
  abort(errorCode?: string): Promise<void>;

  /**
   * Continues route's request with optional overrides.
   *
   * **Usage**
   *
   * ```js
   * await page.route('**\/*', (route, request) => {
   *   // Override headers
   *   const headers = {
   *     ...request.headers(),
   *     foo: 'foo-value', // set "foo" header
   *     bar: undefined, // remove "bar" header
   *   };
   *   route.continue({headers});
   * });
   * ```
   *
   * **Details**
   *
   * Note that any overrides such as `url` or `headers` only apply to the request being routed. If this request results
   * in a redirect, overrides will not be applied to the new redirected request. If you want to propagate a header
   * through redirects, use the combination of
   * [route.fetch([options])](https://playwright.dev/docs/api/class-route#route-fetch) and
   * [route.fulfill([options])](https://playwright.dev/docs/api/class-route#route-fulfill) instead.
   * @param options
   */
  continue(options?: {
    /**
     * If set changes the request HTTP headers. Header values will be converted to a string.
     */
    headers?: { [key: string]: string; };

    /**
     * If set changes the request method (e.g. GET or POST).
     */
    method?: string;

    /**
     * If set changes the post data of request.
     */
    postData?: string|Buffer|Serializable;

    /**
     * If set changes the request URL. New URL must have same protocol as original one.
     */
    url?: string;
  }): Promise<void>;

  /**
   * When several routes match the given pattern, they run in the order opposite to their registration. That way the
   * last registered route can always override all the previous ones. In the example below, request will be handled by
   * the bottom-most handler first, then it'll fall back to the previous one and in the end will be aborted by the first
   * registered route.
   *
   * **Usage**
   *
   * ```js
   * await page.route('**\/*', route => {
   *   // Runs last.
   *   route.abort();
   * });
   * await page.route('**\/*', route => {
   *   // Runs second.
   *   route.fallback();
   * });
   * await page.route('**\/*', route => {
   *   // Runs first.
   *   route.fallback();
   * });
   * ```
   *
   * Registering multiple routes is useful when you want separate handlers to handle different kinds of requests, for
   * example API calls vs page resources or GET requests vs POST requests as in the example below.
   *
   * ```js
   * // Handle GET requests.
   * await page.route('**\/*', route => {
   *   if (route.request().method() !== 'GET') {
   *     route.fallback();
   *     return;
   *   }
   *   // Handling GET only.
   *   // ...
   * });
   *
   * // Handle POST requests.
   * await page.route('**\/*', route => {
   *   if (route.request().method() !== 'POST') {
   *     route.fallback();
   *     return;
   *   }
   *   // Handling POST only.
   *   // ...
   * });
   * ```
   *
   * One can also modify request while falling back to the subsequent handler, that way intermediate route handler can
   * modify url, method, headers and postData of the request.
   *
   * ```js
   * await page.route('**\/*', (route, request) => {
   *   // Override headers
   *   const headers = {
   *     ...request.headers(),
   *     foo: 'foo-value', // set "foo" header
   *     bar: undefined, // remove "bar" header
   *   };
   *   route.fallback({headers});
   * });
   * ```
   *
   * @param options
   */
  fallback(options?: {
    /**
     * If set changes the request HTTP headers. Header values will be converted to a string.
     */
    headers?: { [key: string]: string; };

    /**
     * If set changes the request method (e.g. GET or POST).
     */
    method?: string;

    /**
     * If set changes the post data of request.
     */
    postData?: string|Buffer|Serializable;

    /**
     * If set changes the request URL. New URL must have same protocol as original one. Changing the URL won't affect the
     * route matching, all the routes are matched using the original request URL.
     */
    url?: string;
  }): Promise<void>;

  /**
   * Performs the request and fetches result without fulfilling it, so that the response could be modified and then
   * fulfilled.
   *
   * **Usage**
   *
   * ```js
   * await page.route('https://dog.ceo/api/breeds/list/all', async route => {
   *   const response = await route.fetch();
   *   const json = await response.json();
   *   json.message['big_red_dog'] = [];
   *   await route.fulfill({ response, json });
   * });
   * ```
   *
   * **Details**
   *
   * Note that `headers` option will apply to the fetched request as well as any redirects initiated by it. If you want
   * to only apply `headers` to the original request, but not to redirects, look into
   * [route.continue([options])](https://playwright.dev/docs/api/class-route#route-continue) instead.
   * @param options
   */
  fetch(options?: {
    /**
     * If set changes the request HTTP headers. Header values will be converted to a string.
     */
    headers?: { [key: string]: string; };

    /**
     * Maximum number of request redirects that will be followed automatically. An error will be thrown if the number is
     * exceeded. Defaults to `20`. Pass `0` to not follow redirects.
     */
    maxRedirects?: number;

    /**
     * If set changes the request method (e.g. GET or POST).
     */
    method?: string;

    /**
     * Allows to set post data of the request. If the data parameter is an object, it will be serialized to json string
     * and `content-type` header will be set to `application/json` if not explicitly set. Otherwise the `content-type`
     * header will be set to `application/octet-stream` if not explicitly set.
     */
    postData?: string|Buffer|Serializable;

    /**
     * If set changes the request URL. New URL must have same protocol as original one.
     */
    url?: string;
  }): Promise<APIResponse>;

  /**
   * Fulfills route's request with given response.
   *
   * **Usage**
   *
   * An example of fulfilling all requests with 404 responses:
   *
   * ```js
   * await page.route('**\/*', route => {
   *   route.fulfill({
   *     status: 404,
   *     contentType: 'text/plain',
   *     body: 'Not Found!'
   *   });
   * });
   * ```
   *
   * An example of serving static file:
   *
   * ```js
   * await page.route('**\/xhr_endpoint', route => route.fulfill({ path: 'mock_data.json' }));
   * ```
   *
   * @param options
   */
  fulfill(options?: {
    /**
     * Response body.
     */
    body?: string|Buffer;

    /**
     * If set, equals to setting `Content-Type` response header.
     */
    contentType?: string;

    /**
     * Response headers. Header values will be converted to a string.
     */
    headers?: { [key: string]: string; };

    /**
     * JSON response. This method will set the content type to `application/json` if not set.
     */
    json?: Serializable;

    /**
     * File path to respond with. The content type will be inferred from file extension. If `path` is a relative path,
     * then it is resolved relative to the current working directory.
     */
    path?: string;

    /**
     * [APIResponse] to fulfill route's request with. Individual fields of the response (such as headers) can be
     * overridden using fulfill options.
     */
    response?: APIResponse;

    /**
     * Response status code, defaults to `200`.
     */
    status?: number;
  }): Promise<void>;

  /**
   * A request to be routed.
   */
  request(): Request;
}

/**
 * Selectors can be used to install custom selector engines. See [extensibility](https://playwright.dev/docs/extensibility) for more
 * information.
 */
export interface Selectors {
  /**
   * Selectors must be registered before creating the page.
   *
   * **Usage**
   *
   * An example of registering selector engine that queries elements based on a tag name:
   *
   * ```js
   * const { selectors, firefox } = require('playwright');  // Or 'chromium' or 'webkit'.
   *
   * (async () => {
   *   // Must be a function that evaluates to a selector engine instance.
   *   const createTagNameEngine = () => ({
   *     // Returns the first element matching given selector in the root's subtree.
   *     query(root, selector) {
   *       return root.querySelector(selector);
   *     },
   *
   *     // Returns all elements matching given selector in the root's subtree.
   *     queryAll(root, selector) {
   *       return Array.from(root.querySelectorAll(selector));
   *     }
   *   });
   *
   *   // Register the engine. Selectors will be prefixed with "tag=".
   *   await selectors.register('tag', createTagNameEngine);
   *
   *   const browser = await firefox.launch();
   *   const page = await browser.newPage();
   *   await page.setContent(`<div><button>Click me</button></div>`);
   *
   *   // Use the selector prefixed with its name.
   *   const button = page.locator('tag=button');
   *   // We can combine it with built-in locators.
   *   await page.locator('tag=div').getByText('Click me').click();
   *   // Can use it in any methods supporting selectors.
   *   const buttonCount = await page.locator('tag=button').count();
   *
   *   await browser.close();
   * })();
   * ```
   *
   * @param name Name that is used in selectors as a prefix, e.g. `{name: 'foo'}` enables `foo=myselectorbody` selectors. May only
   * contain `[a-zA-Z0-9_]` characters.
   * @param script Script that evaluates to a selector engine instance. The script is evaluated in the page context.
   * @param options
   */
  register(name: string, script: Function|string|{
    /**
     * Path to the JavaScript file. If `path` is a relative path, then it is resolved relative to the current working
     * directory. Optional.
     */
    path?: string;

    /**
     * Raw script content. Optional.
     */
    content?: string;
  }, options?: {
    /**
     * Whether to run this selector engine in isolated JavaScript environment. This environment has access to the same
     * DOM, but not any JavaScript objects from the frame's scripts. Defaults to `false`. Note that running as a content
     * script is not guaranteed when this engine is used together with other registered engines.
     */
    contentScript?: boolean;
  }): Promise<void>;

  /**
   * Defines custom attribute name to be used in
   * [page.getByTestId(testId)](https://playwright.dev/docs/api/class-page#page-get-by-test-id). `data-testid` is used
   * by default.
   * @param attributeName Test id attribute name.
   */
  setTestIdAttribute(attributeName: string): void;
}

/**
 * The Touchscreen class operates in main-frame CSS pixels relative to the top-left corner of the viewport. Methods on
 * the touchscreen can only be used in browser contexts that have been initialized with `hasTouch` set to true.
 */
export interface Touchscreen {
  /**
   * Dispatches a `touchstart` and `touchend` event with a single touch at the position (`x`,`y`).
   *
   * **NOTE** [page.tap(selector[, options])](https://playwright.dev/docs/api/class-page#page-tap) the method will throw
   * if `hasTouch` option of the browser context is false.
   * @param x
   * @param y
   */
  tap(x: number, y: number): Promise<void>;
}

/**
 * API for collecting and saving Playwright traces. Playwright traces can be opened in
 * [Trace Viewer](https://playwright.dev/docs/trace-viewer) after Playwright script runs.
 *
 * Start recording a trace before performing actions. At the end, stop tracing and save it to a file.
 *
 * ```js
 * const browser = await chromium.launch();
 * const context = await browser.newContext();
 * await context.tracing.start({ screenshots: true, snapshots: true });
 * const page = await context.newPage();
 * await page.goto('https://playwright.dev');
 * await context.tracing.stop({ path: 'trace.zip' });
 * ```
 *
 */
export interface Tracing {
  /**
   * Start tracing.
   *
   * **Usage**
   *
   * ```js
   * await context.tracing.start({ screenshots: true, snapshots: true });
   * const page = await context.newPage();
   * await page.goto('https://playwright.dev');
   * await context.tracing.stop({ path: 'trace.zip' });
   * ```
   *
   * @param options
   */
  start(options?: {
    /**
     * If specified, the trace is going to be saved into the file with the given name inside the `tracesDir` folder
     * specified in
     * [browserType.launch([options])](https://playwright.dev/docs/api/class-browsertype#browser-type-launch).
     */
    name?: string;

    /**
     * Whether to capture screenshots during tracing. Screenshots are used to build a timeline preview.
     */
    screenshots?: boolean;

    /**
     * If this option is true tracing will
     * - capture DOM snapshot on every action
     * - record network activity
     */
    snapshots?: boolean;

    /**
     * Whether to include source files for trace actions.
     */
    sources?: boolean;

    /**
     * Trace name to be shown in the Trace Viewer.
     */
    title?: string;
  }): Promise<void>;

  /**
   * Start a new trace chunk. If you'd like to record multiple traces on the same [BrowserContext], use
   * [tracing.start([options])](https://playwright.dev/docs/api/class-tracing#tracing-start) once, and then create
   * multiple trace chunks with
   * [tracing.startChunk([options])](https://playwright.dev/docs/api/class-tracing#tracing-start-chunk) and
   * [tracing.stopChunk([options])](https://playwright.dev/docs/api/class-tracing#tracing-stop-chunk).
   *
   * **Usage**
   *
   * ```js
   * await context.tracing.start({ screenshots: true, snapshots: true });
   * const page = await context.newPage();
   * await page.goto('https://playwright.dev');
   *
   * await context.tracing.startChunk();
   * await page.getByText('Get Started').click();
   * // Everything between startChunk and stopChunk will be recorded in the trace.
   * await context.tracing.stopChunk({ path: 'trace1.zip' });
   *
   * await context.tracing.startChunk();
   * await page.goto('http://example.com');
   * // Save a second trace file with different actions.
   * await context.tracing.stopChunk({ path: 'trace2.zip' });
   * ```
   *
   * @param options
   */
  startChunk(options?: {
    /**
     * If specified, the trace is going to be saved into the file with the given name inside the `tracesDir` folder
     * specified in
     * [browserType.launch([options])](https://playwright.dev/docs/api/class-browsertype#browser-type-launch).
     */
    name?: string;

    /**
     * Trace name to be shown in the Trace Viewer.
     */
    title?: string;
  }): Promise<void>;

  /**
   * Stop tracing.
   * @param options
   */
  stop(options?: {
    /**
     * Export trace into the file with the given path.
     */
    path?: string;
  }): Promise<void>;

  /**
   * Stop the trace chunk. See
   * [tracing.startChunk([options])](https://playwright.dev/docs/api/class-tracing#tracing-start-chunk) for more details
   * about multiple trace chunks.
   * @param options
   */
  stopChunk(options?: {
    /**
     * Export trace collected since the last
     * [tracing.startChunk([options])](https://playwright.dev/docs/api/class-tracing#tracing-start-chunk) call into the
     * file with the given path.
     */
    path?: string;
  }): Promise<void>;
}

/**
 * When browser context is created with the `recordVideo` option, each page has a video object associated with it.
 *
 * ```js
 * console.log(await page.video().path());
 * ```
 *
 */
export interface Video {
  /**
   * Deletes the video file. Will wait for the video to finish if necessary.
   */
  delete(): Promise<void>;

  /**
   * Returns the file system path this video will be recorded to. The video is guaranteed to be written to the
   * filesystem upon closing the browser context. This method throws when connected remotely.
   */
  path(): Promise<string>;

  /**
   * Saves the video to a user-specified path. It is safe to call this method while the video is still in progress, or
   * after the page has closed. This method waits until the page is closed and the video is fully saved.
   * @param path Path where the video should be saved.
   */
  saveAs(path: string): Promise<void>;
}

/**
 * The [WebSocket] class represents websocket connections in the page.
 */
export interface WebSocket {
  /**
   * Fired when the websocket closes.
   */
  on(event: 'close', listener: (webSocket: WebSocket) => void): this;

  /**
   * Fired when the websocket receives a frame.
   */
  on(event: 'framereceived', listener: (data: {
    /**
     * frame payload
     */
    payload: string|Buffer;
  }) => void): this;

  /**
   * Fired when the websocket sends a frame.
   */
  on(event: 'framesent', listener: (data: {
    /**
     * frame payload
     */
    payload: string|Buffer;
  }) => void): this;

  /**
   * Fired when the websocket has an error.
   */
  on(event: 'socketerror', listener: (string: String) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'close', listener: (webSocket: WebSocket) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'framereceived', listener: (data: {
    /**
     * frame payload
     */
    payload: string|Buffer;
  }) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'framesent', listener: (data: {
    /**
     * frame payload
     */
    payload: string|Buffer;
  }) => void): this;

  /**
   * Adds an event listener that will be automatically removed after it is triggered once. See `addListener` for more information about this event.
   */
  once(event: 'socketerror', listener: (string: String) => void): this;

  /**
   * Fired when the websocket closes.
   */
  addListener(event: 'close', listener: (webSocket: WebSocket) => void): this;

  /**
   * Fired when the websocket receives a frame.
   */
  addListener(event: 'framereceived', listener: (data: {
    /**
     * frame payload
     */
    payload: string|Buffer;
  }) => void): this;

  /**
   * Fired when the websocket sends a frame.
   */
  addListener(event: 'framesent', listener: (data: {
    /**
     * frame payload
     */
    payload: string|Buffer;
  }) => void): this;

  /**
   * Fired when the websocket has an error.
   */
  addListener(event: 'socketerror', listener: (string: String) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'close', listener: (webSocket: WebSocket) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'framereceived', listener: (data: {
    /**
     * frame payload
     */
    payload: string|Buffer;
  }) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'framesent', listener: (data: {
    /**
     * frame payload
     */
    payload: string|Buffer;
  }) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  removeListener(event: 'socketerror', listener: (string: String) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'close', listener: (webSocket: WebSocket) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'framereceived', listener: (data: {
    /**
     * frame payload
     */
    payload: string|Buffer;
  }) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'framesent', listener: (data: {
    /**
     * frame payload
     */
    payload: string|Buffer;
  }) => void): this;

  /**
   * Removes an event listener added by `on` or `addListener`.
   */
  off(event: 'socketerror', listener: (string: String) => void): this;

  /**
   * Fired when the websocket closes.
   */
  prependListener(event: 'close', listener: (webSocket: WebSocket) => void): this;

  /**
   * Fired when the websocket receives a frame.
   */
  prependListener(event: 'framereceived', listener: (data: {
    /**
     * frame payload
     */
    payload: string|Buffer;
  }) => void): this;

  /**
   * Fired when the websocket sends a frame.
   */
  prependListener(event: 'framesent', listener: (data: {
    /**
     * frame payload
     */
    payload: string|Buffer;
  }) => void): this;

  /**
   * Fired when the websocket has an error.
   */
  prependListener(event: 'socketerror', listener: (string: String) => void): this;

  /**
   * Indicates that the web socket has been closed.
   */
  isClosed(): boolean;

  /**
   * Contains the URL of the WebSocket.
   */
  url(): string;

  /**
   * Fired when the websocket closes.
   */
  waitForEvent(event: 'close', optionsOrPredicate?: { predicate?: (webSocket: WebSocket) => boolean | Promise<boolean>, timeout?: number } | ((webSocket: WebSocket) => boolean | Promise<boolean>)): Promise<WebSocket>;

  /**
   * Fired when the websocket receives a frame.
   */
  waitForEvent(event: 'framereceived', optionsOrPredicate?: { predicate?: (data: {
    /**
     * frame payload
     */
    payload: string|Buffer;
  }) => boolean | Promise<boolean>, timeout?: number } | ((data: {
    /**
     * frame payload
     */
    payload: string|Buffer;
  }) => boolean | Promise<boolean>)): Promise<{
    /**
     * frame payload
     */
    payload: string|Buffer;
  }>;

  /**
   * Fired when the websocket sends a frame.
   */
  waitForEvent(event: 'framesent', optionsOrPredicate?: { predicate?: (data: {
    /**
     * frame payload
     */
    payload: string|Buffer;
  }) => boolean | Promise<boolean>, timeout?: number } | ((data: {
    /**
     * frame payload
     */
    payload: string|Buffer;
  }) => boolean | Promise<boolean>)): Promise<{
    /**
     * frame payload
     */
    payload: string|Buffer;
  }>;

  /**
   * Fired when the websocket has an error.
   */
  waitForEvent(event: 'socketerror', optionsOrPredicate?: { predicate?: (string: String) => boolean | Promise<boolean>, timeout?: number } | ((string: String) => boolean | Promise<boolean>)): Promise<String>;

}

export interface BrowserContextOptions {
  /**
   * Whether to automatically download all the attachments. Defaults to `true` where all the downloads are accepted.
   */
  acceptDownloads?: boolean;

  /**
   * When using [page.goto(url[, options])](https://playwright.dev/docs/api/class-page#page-goto),
   * [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route),
   * [page.waitForURL(url[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-url),
   * [page.waitForRequest(urlOrPredicate[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-request),
   * or
   * [page.waitForResponse(urlOrPredicate[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-response)
   * it takes the base URL in consideration by using the
   * [`URL()`](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL) constructor for building the corresponding URL.
   * Examples:
   * - baseURL: `http://localhost:3000` and navigating to `/bar.html` results in `http://localhost:3000/bar.html`
   * - baseURL: `http://localhost:3000/foo/` and navigating to `./bar.html` results in
   *   `http://localhost:3000/foo/bar.html`
   * - baseURL: `http://localhost:3000/foo` (without trailing slash) and navigating to `./bar.html` results in
   *   `http://localhost:3000/bar.html`
   */
  baseURL?: string;

  /**
   * Toggles bypassing page's Content-Security-Policy.
   */
  bypassCSP?: boolean;

  /**
   * Emulates `'prefers-colors-scheme'` media feature, supported values are `'light'`, `'dark'`, `'no-preference'`. See
   * [page.emulateMedia([options])](https://playwright.dev/docs/api/class-page#page-emulate-media) for more details.
   * Passing `null` resets emulation to system defaults. Defaults to `'light'`.
   */
  colorScheme?: null|"light"|"dark"|"no-preference";

  /**
   * Specify device scale factor (can be thought of as dpr). Defaults to `1`.
   */
  deviceScaleFactor?: number;

  /**
   * An object containing additional HTTP headers to be sent with every request.
   */
  extraHTTPHeaders?: { [key: string]: string; };

  /**
   * Emulates `'forced-colors'` media feature, supported values are `'active'`, `'none'`. See
   * [page.emulateMedia([options])](https://playwright.dev/docs/api/class-page#page-emulate-media) for more details.
   * Passing `null` resets emulation to system defaults. Defaults to `'none'`.
   */
  forcedColors?: null|"active"|"none";

  geolocation?: Geolocation;

  /**
   * Specifies if viewport supports touch events. Defaults to false.
   */
  hasTouch?: boolean;

  /**
   * Credentials for [HTTP authentication](https://developer.mozilla.org/en-US/docs/Web/HTTP/Authentication).
   */
  httpCredentials?: HTTPCredentials;

  /**
   * Whether to ignore HTTPS errors when sending network requests. Defaults to `false`.
   */
  ignoreHTTPSErrors?: boolean;

  /**
   * Whether the `meta viewport` tag is taken into account and touch events are enabled. Defaults to `false`. Not
   * supported in Firefox.
   */
  isMobile?: boolean;

  /**
   * Whether or not to enable JavaScript in the context. Defaults to `true`.
   */
  javaScriptEnabled?: boolean;

  /**
   * Specify user locale, for example `en-GB`, `de-DE`, etc. Locale will affect `navigator.language` value,
   * `Accept-Language` request header value as well as number and date formatting rules.
   */
  locale?: string;

  /**
   * Logger sink for Playwright logging.
   */
  logger?: Logger;

  /**
   * Whether to emulate network being offline. Defaults to `false`.
   */
  offline?: boolean;

  /**
   * A list of permissions to grant to all pages in this context. See
   * [browserContext.grantPermissions(permissions[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-grant-permissions)
   * for more details.
   */
  permissions?: Array<string>;

  /**
   * Network proxy settings to use with this context.
   *
   * **NOTE** For Chromium on Windows the browser needs to be launched with the global proxy for this option to work. If
   * all contexts override the proxy, global proxy will be never used and can be any string, for example `launch({
   * proxy: { server: 'http://per-context' } })`.
   */
  proxy?: {
    /**
     * Proxy to be used for all requests. HTTP and SOCKS proxies are supported, for example `http://myproxy.com:3128` or
     * `socks5://myproxy.com:3128`. Short form `myproxy.com:3128` is considered an HTTP proxy.
     */
    server: string;

    /**
     * Optional comma-separated domains to bypass proxy, for example `".com, chromium.org, .domain.com"`.
     */
    bypass?: string;

    /**
     * Optional username to use if HTTP proxy requires authentication.
     */
    username?: string;

    /**
     * Optional password to use if HTTP proxy requires authentication.
     */
    password?: string;
  };

  /**
   * Enables [HAR](http://www.softwareishard.com/blog/har-12-spec) recording for all pages into `recordHar.path` file.
   * If not specified, the HAR is not recorded. Make sure to await
   * [browserContext.close()](https://playwright.dev/docs/api/class-browsercontext#browser-context-close) for the HAR to
   * be saved.
   */
  recordHar?: {
    /**
     * Optional setting to control whether to omit request content from the HAR. Defaults to `false`. Deprecated, use
     * `content` policy instead.
     */
    omitContent?: boolean;

    /**
     * Optional setting to control resource content management. If `omit` is specified, content is not persisted. If
     * `attach` is specified, resources are persisted as separate files or entries in the ZIP archive. If `embed` is
     * specified, content is stored inline the HAR file as per HAR specification. Defaults to `attach` for `.zip` output
     * files and to `embed` for all other file extensions.
     */
    content?: "omit"|"embed"|"attach";

    /**
     * Path on the filesystem to write the HAR file to. If the file name ends with `.zip`, `content: 'attach'` is used by
     * default.
     */
    path: string;

    /**
     * When set to `minimal`, only record information necessary for routing from HAR. This omits sizes, timing, page,
     * cookies, security and other types of HAR information that are not used when replaying from HAR. Defaults to `full`.
     */
    mode?: "full"|"minimal";

    /**
     * A glob or regex pattern to filter requests that are stored in the HAR. When a `baseURL` via the context options was
     * provided and the passed URL is a path, it gets merged via the
     * [`new URL()`](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL) constructor.
     */
    urlFilter?: string|RegExp;
  };

  /**
   * Enables video recording for all pages into `recordVideo.dir` directory. If not specified videos are not recorded.
   * Make sure to await
   * [browserContext.close()](https://playwright.dev/docs/api/class-browsercontext#browser-context-close) for videos to
   * be saved.
   */
  recordVideo?: {
    /**
     * Path to the directory to put videos into.
     */
    dir: string;

    /**
     * Optional dimensions of the recorded videos. If not specified the size will be equal to `viewport` scaled down to
     * fit into 800x800. If `viewport` is not configured explicitly the video size defaults to 800x450. Actual picture of
     * each page will be scaled down if necessary to fit the specified size.
     */
    size?: {
      /**
       * Video frame width.
       */
      width: number;

      /**
       * Video frame height.
       */
      height: number;
    };
  };

  /**
   * Emulates `'prefers-reduced-motion'` media feature, supported values are `'reduce'`, `'no-preference'`. See
   * [page.emulateMedia([options])](https://playwright.dev/docs/api/class-page#page-emulate-media) for more details.
   * Passing `null` resets emulation to system defaults. Defaults to `'no-preference'`.
   */
  reducedMotion?: null|"reduce"|"no-preference";

  /**
   * Emulates consistent window screen size available inside web page via `window.screen`. Is only used when the
   * `viewport` is set.
   */
  screen?: {
    /**
     * page width in pixels.
     */
    width: number;

    /**
     * page height in pixels.
     */
    height: number;
  };

  /**
   * Whether to allow sites to register Service workers. Defaults to `'allow'`.
   * - `'allow'`: [Service Workers](https://developer.mozilla.org/en-US/docs/Web/API/Service_Worker_API) can be
   *   registered.
   * - `'block'`: Playwright will block all registration of Service Workers.
   */
  serviceWorkers?: "allow"|"block";

  /**
   * Populates context with given storage state. This option can be used to initialize context with logged-in
   * information obtained via
   * [browserContext.storageState([options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-storage-state).
   * Either a path to the file with saved storage, or an object with the following fields:
   */
  storageState?: string|{
    /**
     * cookies to set for context
     */
    cookies: Array<{
      name: string;

      value: string;

      /**
       * domain and path are required
       */
      domain: string;

      /**
       * domain and path are required
       */
      path: string;

      /**
       * Unix time in seconds.
       */
      expires: number;

      httpOnly: boolean;

      secure: boolean;

      /**
       * sameSite flag
       */
      sameSite: "Strict"|"Lax"|"None";
    }>;

    /**
     * localStorage to set for context
     */
    origins: Array<{
      origin: string;

      localStorage: Array<{
        name: string;

        value: string;
      }>;
    }>;
  };

  /**
   * If set to true, enables strict selectors mode for this context. In the strict selectors mode all operations on
   * selectors that imply single target DOM element will throw when more than one element matches the selector. This
   * option does not affect any Locator APIs (Locators are always strict). See [Locator] to learn more about the strict
   * mode.
   */
  strictSelectors?: boolean;

  /**
   * Changes the timezone of the context. See
   * [ICU's metaZones.txt](https://cs.chromium.org/chromium/src/third_party/icu/source/data/misc/metaZones.txt?rcl=faee8bc70570192d82d2978a71e2a615788597d1)
   * for a list of supported timezone IDs.
   */
  timezoneId?: string;

  /**
   * Specific user agent to use in this context.
   */
  userAgent?: string;

  /**
   * @deprecated Use `recordVideo` instead.
   */
  videoSize?: {
    /**
     * Video frame width.
     */
    width: number;

    /**
     * Video frame height.
     */
    height: number;
  };

  /**
   * @deprecated Use `recordVideo` instead.
   */
  videosPath?: string;

  /**
   * Emulates consistent viewport for each page. Defaults to an 1280x720 viewport. Use `null` to disable the consistent
   * viewport emulation.
   *
   * **NOTE** The `null` value opts out from the default presets, makes viewport depend on the host window size defined
   * by the operating system. It makes the execution of the tests non-deterministic.
   */
  viewport?: null|ViewportSize;
}

export interface ViewportSize {
  /**
   * page width in pixels.
   */
  width: number;

  /**
   * page height in pixels.
   */
  height: number;
}

export interface HTTPCredentials {
  username: string;

  password: string;
}

export interface Geolocation {
  /**
   * Latitude between -90 and 90.
   */
  latitude: number;

  /**
   * Longitude between -180 and 180.
   */
  longitude: number;

  /**
   * Non-negative accuracy value. Defaults to `0`.
   */
  accuracy?: number;
}

interface AccessibilitySnapshotOptions {
  /**
   * Prune uninteresting nodes from the tree. Defaults to `true`.
   */
  interestingOnly?: boolean;

  /**
   * The root DOM element for the snapshot. Defaults to the whole page.
   */
  root?: ElementHandle;
}

export interface LaunchOptions {
  /**
   * Additional arguments to pass to the browser instance. The list of Chromium flags can be found
   * [here](http://peter.sh/experiments/chromium-command-line-switches/).
   */
  args?: Array<string>;

  /**
   * Browser distribution channel.  Supported values are "chrome", "chrome-beta", "chrome-dev", "chrome-canary",
   * "msedge", "msedge-beta", "msedge-dev", "msedge-canary". Read more about using
   * [Google Chrome and Microsoft Edge](https://playwright.dev/docs/browsers#google-chrome--microsoft-edge).
   */
  channel?: string;

  /**
   * Enable Chromium sandboxing. Defaults to `false`.
   */
  chromiumSandbox?: boolean;

  /**
   * **Chromium-only** Whether to auto-open a Developer Tools panel for each tab. If this option is `true`, the
   * `headless` option will be set `false`.
   */
  devtools?: boolean;

  /**
   * If specified, accepted downloads are downloaded into this directory. Otherwise, temporary directory is created and
   * is deleted when browser is closed. In either case, the downloads are deleted when the browser context they were
   * created in is closed.
   */
  downloadsPath?: string;

  /**
   * Specify environment variables that will be visible to the browser. Defaults to `process.env`.
   */
  env?: { [key: string]: string|number|boolean; };

  /**
   * Path to a browser executable to run instead of the bundled one. If `executablePath` is a relative path, then it is
   * resolved relative to the current working directory. Note that Playwright only works with the bundled Chromium,
   * Firefox or WebKit, use at your own risk.
   */
  executablePath?: string;

  /**
   * Firefox user preferences. Learn more about the Firefox user preferences at
   * [`about:config`](https://support.mozilla.org/en-US/kb/about-config-editor-firefox).
   */
  firefoxUserPrefs?: { [key: string]: string|number|boolean; };

  /**
   * Close the browser process on SIGHUP. Defaults to `true`.
   */
  handleSIGHUP?: boolean;

  /**
   * Close the browser process on Ctrl-C. Defaults to `true`.
   */
  handleSIGINT?: boolean;

  /**
   * Close the browser process on SIGTERM. Defaults to `true`.
   */
  handleSIGTERM?: boolean;

  /**
   * Whether to run browser in headless mode. More details for
   * [Chromium](https://developers.google.com/web/updates/2017/04/headless-chrome) and
   * [Firefox](https://developer.mozilla.org/en-US/docs/Mozilla/Firefox/Headless_mode). Defaults to `true` unless the
   * `devtools` option is `true`.
   */
  headless?: boolean;

  /**
   * If `true`, Playwright does not pass its own configurations args and only uses the ones from `args`. If an array is
   * given, then filters out the given default arguments. Dangerous option; use with care. Defaults to `false`.
   */
  ignoreDefaultArgs?: boolean|Array<string>;

  /**
   * Logger sink for Playwright logging.
   */
  logger?: Logger;

  /**
   * Network proxy settings.
   */
  proxy?: {
    /**
     * Proxy to be used for all requests. HTTP and SOCKS proxies are supported, for example `http://myproxy.com:3128` or
     * `socks5://myproxy.com:3128`. Short form `myproxy.com:3128` is considered an HTTP proxy.
     */
    server: string;

    /**
     * Optional comma-separated domains to bypass proxy, for example `".com, chromium.org, .domain.com"`.
     */
    bypass?: string;

    /**
     * Optional username to use if HTTP proxy requires authentication.
     */
    username?: string;

    /**
     * Optional password to use if HTTP proxy requires authentication.
     */
    password?: string;
  };

  /**
   * Slows down Playwright operations by the specified amount of milliseconds. Useful so that you can see what is going
   * on.
   */
  slowMo?: number;

  /**
   * Maximum time in milliseconds to wait for the browser instance to start. Defaults to `30000` (30 seconds). Pass `0`
   * to disable timeout.
   */
  timeout?: number;

  /**
   * If specified, traces are saved into this directory.
   */
  tracesDir?: string;
}

export interface ConnectOverCDPOptions {
  /**
   * Deprecated, use the first argument instead. Optional.
   */
  endpointURL?: string;

  /**
   * Additional HTTP headers to be sent with connect request. Optional.
   */
  headers?: { [key: string]: string; };

  /**
   * Logger sink for Playwright logging. Optional.
   */
  logger?: Logger;

  /**
   * Slows down Playwright operations by the specified amount of milliseconds. Useful so that you can see what is going
   * on. Defaults to 0.
   */
  slowMo?: number;

  /**
   * Maximum time in milliseconds to wait for the connection to be established. Defaults to `30000` (30 seconds). Pass
   * `0` to disable timeout.
   */
  timeout?: number;
}

export interface ConnectOptions {
  /**
   * Additional HTTP headers to be sent with web socket connect request. Optional.
   */
  headers?: { [key: string]: string; };

  /**
   * Logger sink for Playwright logging. Optional.
   */
  logger?: Logger;

  /**
   * Slows down Playwright operations by the specified amount of milliseconds. Useful so that you can see what is going
   * on. Defaults to 0.
   */
  slowMo?: number;

  /**
   * Maximum time in milliseconds to wait for the connection to be established. Defaults to `0` (no timeout).
   */
  timeout?: number;
}

export interface LocatorScreenshotOptions {
  /**
   * When set to `"disabled"`, stops CSS animations, CSS transitions and Web Animations. Animations get different
   * treatment depending on their duration:
   * - finite animations are fast-forwarded to completion, so they'll fire `transitionend` event.
   * - infinite animations are canceled to initial state, and then played over after the screenshot.
   *
   * Defaults to `"allow"` that leaves animations untouched.
   */
  animations?: "disabled"|"allow";

  /**
   * When set to `"hide"`, screenshot will hide text caret. When set to `"initial"`, text caret behavior will not be
   * changed.  Defaults to `"hide"`.
   */
  caret?: "hide"|"initial";

  /**
   * Specify locators that should be masked when the screenshot is taken. Masked elements will be overlaid with a pink
   * box `#FF00FF` that completely covers its bounding box.
   */
  mask?: Array<Locator>;

  /**
   * Hides default white background and allows capturing screenshots with transparency. Not applicable to `jpeg` images.
   * Defaults to `false`.
   */
  omitBackground?: boolean;

  /**
   * The file path to save the image to. The screenshot type will be inferred from file extension. If `path` is a
   * relative path, then it is resolved relative to the current working directory. If no path is provided, the image
   * won't be saved to the disk.
   */
  path?: string;

  /**
   * The quality of the image, between 0-100. Not applicable to `png` images.
   */
  quality?: number;

  /**
   * When set to `"css"`, screenshot will have a single pixel per each css pixel on the page. For high-dpi devices, this
   * will keep screenshots small. Using `"device"` option will produce a single pixel per each device pixel, so
   * screenshots of high-dpi devices will be twice as large or even larger.
   *
   * Defaults to `"device"`.
   */
  scale?: "css"|"device";

  /**
   * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
   * option in the config, or by using the
   * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
   * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
   */
  timeout?: number;

  /**
   * Specify screenshot type, defaults to `png`.
   */
  type?: "png"|"jpeg";
}

interface ElementHandleWaitForSelectorOptions {
  /**
   * Defaults to `'visible'`. Can be either:
   * - `'attached'` - wait for element to be present in DOM.
   * - `'detached'` - wait for element to not be present in DOM.
   * - `'visible'` - wait for element to have non-empty bounding box and no `visibility:hidden`. Note that element
   *   without any content or with `display:none` has an empty bounding box and is not considered visible.
   * - `'hidden'` - wait for element to be either detached from DOM, or have an empty bounding box or
   *   `visibility:hidden`. This is opposite to the `'visible'` option.
   */
  state?: "attached"|"detached"|"visible"|"hidden";

  /**
   * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
   * element, the call throws an exception.
   */
  strict?: boolean;

  /**
   * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
   * option in the config, or by using the
   * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
   * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
   */
  timeout?: number;
}

export interface Cookie {
  name: string;

  value: string;

  domain: string;

  path: string;

  /**
   * Unix time in seconds.
   */
  expires: number;

  httpOnly: boolean;

  secure: boolean;

  sameSite: "Strict"|"Lax"|"None";
}

interface PageWaitForSelectorOptions {
  /**
   * Defaults to `'visible'`. Can be either:
   * - `'attached'` - wait for element to be present in DOM.
   * - `'detached'` - wait for element to not be present in DOM.
   * - `'visible'` - wait for element to have non-empty bounding box and no `visibility:hidden`. Note that element
   *   without any content or with `display:none` has an empty bounding box and is not considered visible.
   * - `'hidden'` - wait for element to be either detached from DOM, or have an empty bounding box or
   *   `visibility:hidden`. This is opposite to the `'visible'` option.
   */
  state?: "attached"|"detached"|"visible"|"hidden";

  /**
   * When true, the call requires selector to resolve to a single element. If given selector resolves to more than one
   * element, the call throws an exception.
   */
  strict?: boolean;

  /**
   * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
   * option in the config, or by using the
   * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
   * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
   */
  timeout?: number;
}

interface PageWaitForFunctionOptions {
  /**
   * If `polling` is `'raf'`, then `pageFunction` is constantly executed in `requestAnimationFrame` callback. If
   * `polling` is a number, then it is treated as an interval in milliseconds at which the function would be executed.
   * Defaults to `raf`.
   */
  polling?: number|"raf";

  /**
   * Maximum time to wait for in milliseconds. Defaults to `0` - no timeout. The default value can be changed via
   * `actionTimeout` option in the config, or by using the
   * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
   * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
   */
  timeout?: number;
}

export interface PageScreenshotOptions {
  /**
   * When set to `"disabled"`, stops CSS animations, CSS transitions and Web Animations. Animations get different
   * treatment depending on their duration:
   * - finite animations are fast-forwarded to completion, so they'll fire `transitionend` event.
   * - infinite animations are canceled to initial state, and then played over after the screenshot.
   *
   * Defaults to `"allow"` that leaves animations untouched.
   */
  animations?: "disabled"|"allow";

  /**
   * When set to `"hide"`, screenshot will hide text caret. When set to `"initial"`, text caret behavior will not be
   * changed.  Defaults to `"hide"`.
   */
  caret?: "hide"|"initial";

  /**
   * An object which specifies clipping of the resulting image. Should have the following fields:
   */
  clip?: {
    /**
     * x-coordinate of top-left corner of clip area
     */
    x: number;

    /**
     * y-coordinate of top-left corner of clip area
     */
    y: number;

    /**
     * width of clipping area
     */
    width: number;

    /**
     * height of clipping area
     */
    height: number;
  };

  /**
   * When true, takes a screenshot of the full scrollable page, instead of the currently visible viewport. Defaults to
   * `false`.
   */
  fullPage?: boolean;

  /**
   * Specify locators that should be masked when the screenshot is taken. Masked elements will be overlaid with a pink
   * box `#FF00FF` that completely covers its bounding box.
   */
  mask?: Array<Locator>;

  /**
   * Hides default white background and allows capturing screenshots with transparency. Not applicable to `jpeg` images.
   * Defaults to `false`.
   */
  omitBackground?: boolean;

  /**
   * The file path to save the image to. The screenshot type will be inferred from file extension. If `path` is a
   * relative path, then it is resolved relative to the current working directory. If no path is provided, the image
   * won't be saved to the disk.
   */
  path?: string;

  /**
   * The quality of the image, between 0-100. Not applicable to `png` images.
   */
  quality?: number;

  /**
   * When set to `"css"`, screenshot will have a single pixel per each css pixel on the page. For high-dpi devices, this
   * will keep screenshots small. Using `"device"` option will produce a single pixel per each device pixel, so
   * screenshots of high-dpi devices will be twice as large or even larger.
   *
   * Defaults to `"device"`.
   */
  scale?: "css"|"device";

  /**
   * Maximum time in milliseconds. Defaults to `0` - no timeout. The default value can be changed via `actionTimeout`
   * option in the config, or by using the
   * [browserContext.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-browsercontext#browser-context-set-default-timeout)
   * or [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout) methods.
   */
  timeout?: number;

  /**
   * Specify screenshot type, defaults to `png`.
   */
  type?: "png"|"jpeg";
}

type Devices = {
  "Blackberry PlayBook": DeviceDescriptor;
  "Blackberry PlayBook landscape": DeviceDescriptor;
  "BlackBerry Z30": DeviceDescriptor;
  "BlackBerry Z30 landscape": DeviceDescriptor;
  "Galaxy Note 3": DeviceDescriptor;
  "Galaxy Note 3 landscape": DeviceDescriptor;
  "Galaxy Note II": DeviceDescriptor;
  "Galaxy Note II landscape": DeviceDescriptor;
  "Galaxy S III": DeviceDescriptor;
  "Galaxy S III landscape": DeviceDescriptor;
  "Galaxy S5": DeviceDescriptor;
  "Galaxy S5 landscape": DeviceDescriptor;
  "Galaxy S8": DeviceDescriptor;
  "Galaxy S8 landscape": DeviceDescriptor;
  "Galaxy S9+": DeviceDescriptor;
  "Galaxy S9+ landscape": DeviceDescriptor;
  "Galaxy Tab S4": DeviceDescriptor;
  "Galaxy Tab S4 landscape": DeviceDescriptor;
  "iPad (gen 6)": DeviceDescriptor;
  "iPad (gen 6) landscape": DeviceDescriptor;
  "iPad (gen 7)": DeviceDescriptor;
  "iPad (gen 7) landscape": DeviceDescriptor;
  "iPad Mini": DeviceDescriptor;
  "iPad Mini landscape": DeviceDescriptor;
  "iPad Pro 11": DeviceDescriptor;
  "iPad Pro 11 landscape": DeviceDescriptor;
  "iPhone 6": DeviceDescriptor;
  "iPhone 6 landscape": DeviceDescriptor;
  "iPhone 6 Plus": DeviceDescriptor;
  "iPhone 6 Plus landscape": DeviceDescriptor;
  "iPhone 7": DeviceDescriptor;
  "iPhone 7 landscape": DeviceDescriptor;
  "iPhone 7 Plus": DeviceDescriptor;
  "iPhone 7 Plus landscape": DeviceDescriptor;
  "iPhone 8": DeviceDescriptor;
  "iPhone 8 landscape": DeviceDescriptor;
  "iPhone 8 Plus": DeviceDescriptor;
  "iPhone 8 Plus landscape": DeviceDescriptor;
  "iPhone SE": DeviceDescriptor;
  "iPhone SE landscape": DeviceDescriptor;
  "iPhone X": DeviceDescriptor;
  "iPhone X landscape": DeviceDescriptor;
  "iPhone XR": DeviceDescriptor;
  "iPhone XR landscape": DeviceDescriptor;
  "iPhone 11": DeviceDescriptor;
  "iPhone 11 landscape": DeviceDescriptor;
  "iPhone 11 Pro": DeviceDescriptor;
  "iPhone 11 Pro landscape": DeviceDescriptor;
  "iPhone 11 Pro Max": DeviceDescriptor;
  "iPhone 11 Pro Max landscape": DeviceDescriptor;
  "iPhone 12": DeviceDescriptor;
  "iPhone 12 landscape": DeviceDescriptor;
  "iPhone 12 Pro": DeviceDescriptor;
  "iPhone 12 Pro landscape": DeviceDescriptor;
  "iPhone 12 Pro Max": DeviceDescriptor;
  "iPhone 12 Pro Max landscape": DeviceDescriptor;
  "iPhone 12 Mini": DeviceDescriptor;
  "iPhone 12 Mini landscape": DeviceDescriptor;
  "iPhone 13": DeviceDescriptor;
  "iPhone 13 landscape": DeviceDescriptor;
  "iPhone 13 Pro": DeviceDescriptor;
  "iPhone 13 Pro landscape": DeviceDescriptor;
  "iPhone 13 Pro Max": DeviceDescriptor;
  "iPhone 13 Pro Max landscape": DeviceDescriptor;
  "iPhone 13 Mini": DeviceDescriptor;
  "iPhone 13 Mini landscape": DeviceDescriptor;
  "Kindle Fire HDX": DeviceDescriptor;
  "Kindle Fire HDX landscape": DeviceDescriptor;
  "LG Optimus L70": DeviceDescriptor;
  "LG Optimus L70 landscape": DeviceDescriptor;
  "Microsoft Lumia 550": DeviceDescriptor;
  "Microsoft Lumia 550 landscape": DeviceDescriptor;
  "Microsoft Lumia 950": DeviceDescriptor;
  "Microsoft Lumia 950 landscape": DeviceDescriptor;
  "Nexus 10": DeviceDescriptor;
  "Nexus 10 landscape": DeviceDescriptor;
  "Nexus 4": DeviceDescriptor;
  "Nexus 4 landscape": DeviceDescriptor;
  "Nexus 5": DeviceDescriptor;
  "Nexus 5 landscape": DeviceDescriptor;
  "Nexus 5X": DeviceDescriptor;
  "Nexus 5X landscape": DeviceDescriptor;
  "Nexus 6": DeviceDescriptor;
  "Nexus 6 landscape": DeviceDescriptor;
  "Nexus 6P": DeviceDescriptor;
  "Nexus 6P landscape": DeviceDescriptor;
  "Nexus 7": DeviceDescriptor;
  "Nexus 7 landscape": DeviceDescriptor;
  "Nokia Lumia 520": DeviceDescriptor;
  "Nokia Lumia 520 landscape": DeviceDescriptor;
  "Nokia N9": DeviceDescriptor;
  "Nokia N9 landscape": DeviceDescriptor;
  "Pixel 2": DeviceDescriptor;
  "Pixel 2 landscape": DeviceDescriptor;
  "Pixel 2 XL": DeviceDescriptor;
  "Pixel 2 XL landscape": DeviceDescriptor;
  "Pixel 3": DeviceDescriptor;
  "Pixel 3 landscape": DeviceDescriptor;
  "Pixel 4": DeviceDescriptor;
  "Pixel 4 landscape": DeviceDescriptor;
  "Pixel 4a (5G)": DeviceDescriptor;
  "Pixel 4a (5G) landscape": DeviceDescriptor;
  "Pixel 5": DeviceDescriptor;
  "Pixel 5 landscape": DeviceDescriptor;
  "Moto G4": DeviceDescriptor;
  "Moto G4 landscape": DeviceDescriptor;
  "Desktop Chrome HiDPI": DeviceDescriptor;
  "Desktop Edge HiDPI": DeviceDescriptor;
  "Desktop Firefox HiDPI": DeviceDescriptor;
  "Desktop Safari": DeviceDescriptor;
  "Desktop Chrome": DeviceDescriptor;
  "Desktop Edge": DeviceDescriptor;
  "Desktop Firefox": DeviceDescriptor;
  [key: string]: DeviceDescriptor;
}

export interface ChromiumBrowserContext extends BrowserContext { }
export interface ChromiumBrowser extends Browser { }
export interface FirefoxBrowser extends Browser { }
export interface WebKitBrowser extends Browser { }
export interface ChromiumCoverage extends Coverage { }
