/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.identity.common.internal.broker;

import android.accounts.AccountManager;
import android.accounts.AuthenticatorDescription;
import android.annotation.SuppressLint;
import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.Signature;
import android.text.TextUtils;
import android.util.Base64;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import com.microsoft.identity.common.exception.ClientException;
import com.microsoft.identity.common.internal.broker.BrokerData;
import com.microsoft.identity.common.internal.broker.PackageHelper;
import com.microsoft.identity.common.internal.util.StringUtil;
import com.microsoft.identity.common.logging.Logger;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.PKIXParameters;
import java.security.cert.TrustAnchor;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

public class BrokerValidator {
    private static final String TAG = "BrokerValidator";
    private static boolean sShouldTrustDebugBrokers = false;
    private final Context mContext;

    public static void setShouldTrustDebugBrokers(boolean shouldTrustDebugBrokers) {
        if (shouldTrustDebugBrokers) {
            Logger.warn(TAG, "You are forcing to trust debug brokers in non-debug builds.");
        }
        sShouldTrustDebugBrokers = shouldTrustDebugBrokers;
    }

    public static boolean getShouldTrustDebugBrokers() {
        return sShouldTrustDebugBrokers;
    }

    public BrokerValidator(Context context) {
        this.mContext = context;
    }

    public String verifySignatureAndThrow(String brokerPackageName) throws ClientException {
        try {
            List<X509Certificate> certs = this.readCertDataForBrokerApp(brokerPackageName);
            String signatureHash = this.verifySignatureHash(certs);
            if (certs.size() > 1) {
                this.verifyCertificateChain(certs);
            }
            return signatureHash;
        }
        catch (PackageManager.NameNotFoundException e) {
            throw new ClientException("App package name is not found in the package manager", e.getMessage(), e);
        }
        catch (NoSuchAlgorithmException e) {
            throw new ClientException("no_such_algorithm", e.getMessage(), e);
        }
        catch (IOException | GeneralSecurityException e) {
            throw new ClientException("Signature could not be verified", e.getMessage(), e);
        }
    }

    public boolean verifySignature(String brokerPackageName) {
        String methodName = ":verifySignature";
        try {
            return this.verifySignatureAndThrow(brokerPackageName) != null;
        }
        catch (ClientException e) {
            Logger.error("BrokerValidator:verifySignature", e.getErrorCode() + ": " + e.getMessage(), e);
            return false;
        }
    }

    public Set<BrokerData> getValidBrokers() {
        Set<BrokerData> validBrokers = sShouldTrustDebugBrokers ? BrokerData.getAllBrokers() : BrokerData.getProdBrokers();
        return validBrokers;
    }

    public boolean isValidBrokerPackage(@NonNull String packageName) {
        Set<BrokerData> validBrokers = this.getValidBrokers();
        for (BrokerData brokerData : validBrokers) {
            if (!brokerData.packageName.equals(packageName) || !this.verifySignature(packageName)) continue;
            return true;
        }
        return false;
    }

    private String verifySignatureHash(List<X509Certificate> certs) throws NoSuchAlgorithmException, CertificateEncodingException, ClientException {
        StringBuilder hashListStringBuilder = new StringBuilder();
        for (X509Certificate x509Certificate : certs) {
            MessageDigest messageDigest = MessageDigest.getInstance("SHA");
            messageDigest.update(x509Certificate.getEncoded());
            String signatureHash = Base64.encodeToString((byte[])messageDigest.digest(), (int)2);
            hashListStringBuilder.append(signatureHash);
            hashListStringBuilder.append(',');
            for (BrokerData brokerData : this.getValidBrokers()) {
                if (TextUtils.isEmpty((CharSequence)brokerData.signatureHash) || !brokerData.signatureHash.equals(signatureHash)) continue;
                return signatureHash;
            }
        }
        throw new ClientException("Calling app could not be verified", "SignatureHashes: " + hashListStringBuilder.toString());
    }

    @SuppressLint(value={"PackageManagerGetSignatures"})
    private List<X509Certificate> readCertDataForBrokerApp(String brokerPackageName) throws PackageManager.NameNotFoundException, ClientException, IOException, GeneralSecurityException {
        PackageInfo packageInfo = this.mContext.getPackageManager().getPackageInfo(brokerPackageName, 64);
        if (packageInfo == null) {
            throw new ClientException("App package name is not found in the package manager", "No broker package existed.");
        }
        if (packageInfo.signatures == null || packageInfo.signatures.length == 0) {
            throw new ClientException("Calling app could not be verified", "No signature associated with the broker package.");
        }
        ArrayList<X509Certificate> certificates = new ArrayList<X509Certificate>(packageInfo.signatures.length);
        for (Signature signature : packageInfo.signatures) {
            byte[] rawCert = signature.toByteArray();
            ByteArrayInputStream certStream = new ByteArrayInputStream(rawCert);
            try {
                CertificateFactory certificateFactory = CertificateFactory.getInstance("X509");
                X509Certificate x509Certificate = (X509Certificate)certificateFactory.generateCertificate(certStream);
                certificates.add(x509Certificate);
            }
            catch (CertificateException e) {
                throw new ClientException("Calling app could not be verified");
            }
        }
        return certificates;
    }

    private void verifyCertificateChain(List<X509Certificate> certificates) throws GeneralSecurityException, ClientException {
        X509Certificate issuerCert = this.getSelfSignedCert(certificates);
        TrustAnchor trustAnchor = new TrustAnchor(issuerCert, null);
        PKIXParameters pkixParameters = new PKIXParameters(Collections.singleton(trustAnchor));
        pkixParameters.setRevocationEnabled(false);
        CertPath certPath = CertificateFactory.getInstance("X.509").generateCertPath(certificates);
        CertPathValidator certPathValidator = CertPathValidator.getInstance("PKIX");
        certPathValidator.validate(certPath, pkixParameters);
    }

    private X509Certificate getSelfSignedCert(List<X509Certificate> certs) throws ClientException {
        int count = 0;
        X509Certificate selfSignedCert = null;
        for (X509Certificate x509Certificate : certs) {
            if (!x509Certificate.getSubjectDN().equals(x509Certificate.getIssuerDN())) continue;
            selfSignedCert = x509Certificate;
            ++count;
        }
        if (count > 1 || selfSignedCert == null) {
            throw new ClientException("Calling app could not be verified", "Multiple self signed certs found or no self signed cert existed.");
        }
        return selfSignedCert;
    }

    @Nullable
    public String getCurrentActiveBrokerPackageName() {
        AuthenticatorDescription[] authenticators;
        for (AuthenticatorDescription authenticator : authenticators = AccountManager.get((Context)this.mContext).getAuthenticatorTypes()) {
            if (!authenticator.type.equals("com.microsoft.workaccount") || !this.verifySignature(authenticator.packageName)) continue;
            return authenticator.packageName;
        }
        return null;
    }

    public static boolean isValidBrokerRedirect(@Nullable String redirectUri, @NonNull Context context, @NonNull String packageName) {
        PackageHelper info;
        String signatureDigest;
        String methodName = ":isValidBrokerRedirect";
        String expectedBrokerRedirectUri = BrokerValidator.getBrokerRedirectUri(context, packageName);
        boolean isValidBrokerRedirect = StringUtil.equalsIgnoreCase(redirectUri, expectedBrokerRedirectUri);
        if (packageName.equals("com.azure.authenticator") && (BrokerData.MICROSOFT_AUTHENTICATOR_PROD.signatureHash.equals(signatureDigest = (info = new PackageHelper(context.getPackageManager())).getCurrentSignatureForPackage(packageName)) || BrokerData.MICROSOFT_AUTHENTICATOR_DEBUG.signatureHash.equals(signatureDigest))) {
            isValidBrokerRedirect |= StringUtil.equalsIgnoreCase(redirectUri, "urn:ietf:wg:oauth:2.0:oob");
        }
        if (!isValidBrokerRedirect) {
            Logger.error("BrokerValidator:isValidBrokerRedirect", "Broker redirect uri is invalid. Expected: " + expectedBrokerRedirectUri + " Actual: " + redirectUri, null);
        }
        return isValidBrokerRedirect;
    }

    public static String getBrokerRedirectUri(Context context, String packageName) {
        PackageHelper info = new PackageHelper(context.getPackageManager());
        String signatureDigest = info.getCurrentSignatureForPackage(packageName);
        return PackageHelper.getBrokerRedirectUrl(packageName, signatureDigest);
    }
}

