// Copyright (c) Microsoft Corporation.
// All rights reserved.
//
// This code is licensed under the MIT License.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files(the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and / or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions :
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
package com.microsoft.identity.common.internal.util;

import android.os.Parcel;
import android.os.Parcelable;

import androidx.annotation.NonNull;
import lombok.AccessLevel;
import lombok.NoArgsConstructor;

/**
 * Utility class to marshall and unmarshall Parcelable types.
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public final class ParcelableUtil {

    /**
     * Util method to transform a Parcelable to byte[].
     *
     * @param parcelable Parcelable type as an input
     * @return  byte[]
     */
    public static byte[] marshall(@NonNull final Parcelable parcelable) {
        final Parcel parcel = Parcel.obtain();
        parcelable.writeToParcel(parcel, 0);
        final byte[] bytes = parcel.marshall();
        parcel.recycle();
        return bytes;
    }

    /**
     * Util method to transform a byte[] to Parcel.
     *
     * @param bytes input as byte[]
     * @return Parcel
     */
    public static Parcel unmarshall(@NonNull byte[] bytes) {
        final Parcel parcel = Parcel.obtain();
        parcel.unmarshall(bytes, 0, bytes.length);
        parcel.setDataPosition(0);
        return parcel;
    }

    /**
     * Util method to transform a bytes to Parcelable type T.
     *
     * @param bytes input as bytes
     * @param creator<T> Creator of Parcelable Type T to which bytes need to be transformed
     * @return T
     */
    public static <T> T unmarshall(@NonNull byte[] bytes, @NonNull final Parcelable.Creator<T> creator) {
        final Parcel parcel = unmarshall(bytes);
        T result = creator.createFromParcel(parcel);
        parcel.recycle();
        return result;
    }
}
