//  Copyright (c) Microsoft Corporation.
//  All rights reserved.
//
//  This code is licensed under the MIT License.
//
//  Permission is hereby granted, free of charge, to any person obtaining a copy
//  of this software and associated documentation files(the "Software"), to deal
//  in the Software without restriction, including without limitation the rights
//  to use, copy, modify, merge, publish, distribute, sublicense, and / or sell
//  copies of the Software, and to permit persons to whom the Software is
//  furnished to do so, subject to the following conditions :
//
//  The above copyright notice and this permission notice shall be included in
//  all copies or substantial portions of the Software.
//
//  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//  THE SOFTWARE.
package com.microsoft.identity.common.internal.activebrokerdiscovery

import android.content.Context
import android.os.Bundle
import androidx.annotation.VisibleForTesting
import com.microsoft.identity.common.exception.BrokerCommunicationException
import com.microsoft.identity.common.internal.broker.BrokerData
import com.microsoft.identity.common.internal.broker.BrokerValidator
import com.microsoft.identity.common.internal.broker.PackageHelper
import com.microsoft.identity.common.internal.broker.ipc.BrokerOperationBundle
import com.microsoft.identity.common.internal.broker.ipc.ContentProviderStrategy
import com.microsoft.identity.common.internal.broker.ipc.IIpcStrategy
import com.microsoft.identity.common.internal.cache.IClientActiveBrokerCache
import com.microsoft.identity.common.java.exception.ClientException
import com.microsoft.identity.common.java.exception.ClientException.ONLY_SUPPORTS_ACCOUNT_MANAGER_ERROR_CODE
import com.microsoft.identity.common.java.interfaces.IPlatformComponents
import com.microsoft.identity.common.java.logging.Logger
import kotlinx.coroutines.*
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import java.util.concurrent.TimeUnit


/**
 * A class for figuring out which Broker app the caller should communicate with.
 *
 * This class will try pinging each installed apps provided in [brokerCandidates], which will
 * essentially trigger Broker Discovery on that side (and subsequently returned result).
 *
 * If none of the installed app supports the Broker Discovery protocol, this class will fall back
 * to the legacy AccountManager method.
 *
 * @param brokerCandidates                  list of (Broker hosting) candidate apps to perform discovery with.
 * @param getActiveBrokerFromAccountManager a function which returns a validated [BrokerData]
 *                                          based on Android's AccountManager API
 * @param ipcStrategy                       An [IIpcStrategy] to aggregate data with.
 * @param cache                             A local cache for storing active broker discovery results.
 * @param isPackageInstalled                a function to determine if any given broker app is installed.
 * @param isValidBroker                     a function to determine if the installed broker app contains a matching signature hash.
 **/
class BrokerDiscoveryClient(private val brokerCandidates: Set<BrokerData>,
                            private val getActiveBrokerFromAccountManager: () -> BrokerData?,
                            private val ipcStrategy: IIpcStrategy,
                            private val cache: IClientActiveBrokerCache,
                            private val isPackageInstalled: (BrokerData) -> Boolean,
                            private val isValidBroker: (BrokerData) -> Boolean) : IBrokerDiscoveryClient {

    companion object {
        val TAG = BrokerDiscoveryClient::class.simpleName

        @OptIn(ExperimentalCoroutinesApi::class)
        val dispatcher = Dispatchers.IO.limitedParallelism(10)

        const val ACTIVE_BROKER_PACKAGE_NAME_BUNDLE_KEY = "ACTIVE_BROKER_PACKAGE_NAME_BUNDLE_KEY"
        const val ACTIVE_BROKER_SIGNING_CERTIFICATE_THUMBPRINT_BUNDLE_KEY =
            "ACTIVE_BROKER_SIGNING_CERTIFICATE_THUMBPRINT_BUNDLE_KEY"

        const val FORCE_TRIGGER_BROKER_DISCOVERY_BUNDLE_KEY =
            "FORCE_TRIGGER_BROKER_DISCOVERY_BUNDLE_KEY"
        const val FORCE_TRIGGER_BROKER_DISCOVERY_RESULT_EXECUTED_BUNDLE_KEY =
            "FORCE_TRIGGER_BROKER_DISCOVERY_RESULT_EXECUTED_BUNDLE_KEY"

        // the broker service is too old and doesn't support this API.
        const val FORCE_TRIGGER_BROKER_DISCOVERY_RESULT_OPERATION_NOT_SUPPORTED =
            "OPERATION_NOT_SUPPORTED"

        // the broker service recognizes this API, but the feature is disabled (e.g. behind a flight).
        const val FORCE_TRIGGER_BROKER_DISCOVERY_RESULT_OPERATION_DISABLED = "OPERATION_DISABLED"

        // the broker app you're communicating to is not installed.
        const val FORCE_TRIGGER_BROKER_DISCOVERY_PACKAGE_NOT_INSTALLED = "PACKAGE_NOT_INSTALLED"

        // the broker service you're communicating to is not a valid broker app. (e.g. not signed by proper keys)
        const val FORCE_TRIGGER_BROKER_DISCOVERY_NOT_VALID_BROKER = "NOT_VALID_BROKER"

        // Unexpected error.
        const val FORCE_TRIGGER_BROKER_DISCOVERY_RESULT_UNEXPECTED_ERROR = "UNEXPECTED_ERROR"

        const val ERROR_BUNDLE_KEY = "ERROR_BUNDLE_KEY"

        /**
         * Per-process Thread-safe, coroutine-safe Mutex of this class.
         * This is to prevent the IPC mechanism from being unnecessarily triggered due to race condition.
         *
         * The object here must be both coroutine-safe and thread-safe.
         **/
        private val classLevelLock = Mutex()

        /**
         * Performs an IPC operation to get a result from the provided [brokerCandidates].
         *
         * @param brokerCandidates          the candidate(s) to query from.
         * @param ipcStrategy               the ipc mechanism to query with.
         * @param isPackageInstalled        a method which returns true if the provided [BrokerData] is installed.
         * @param shouldStopQueryForAWhile  a method which, if invoked, will force [BrokerDiscoveryClient]
         *                                  to skip the IPC discovery process for a while.
         **/
        internal suspend fun queryFromBroker(
            brokerCandidates: Set<BrokerData>,
            ipcStrategy: IIpcStrategy,
            isPackageInstalled: (BrokerData) -> Boolean,
            isValidBroker: (BrokerData) -> Boolean
        ): BrokerData? {
            return coroutineScope {
                val installedCandidates =
                    brokerCandidates.filter(isPackageInstalled).filter(isValidBroker)
                val deferredResults = installedCandidates.map { candidate ->
                    async(dispatcher) {
                        return@async makeRequest(candidate, ipcStrategy)
                    }
                }
                return@coroutineScope deferredResults.awaitAll().filterNotNull().firstOrNull()
            }
        }

        private fun makeRequest(
            candidate: BrokerData,
            ipcStrategy: IIpcStrategy
        ): BrokerData? {
            val methodTag = "$TAG:makeRequest"
            val operationBundle = BrokerOperationBundle(
                BrokerOperationBundle.Operation.BROKER_DISCOVERY_FROM_SDK,
                candidate.packageName,
                Bundle()
            )

            return try {
                val result = ipcStrategy.communicateToBroker(operationBundle)
                extractResult(result, forceTriggerDiscoveryFlow = false)
            } catch (t: Throwable) {
                if (t is BrokerCommunicationException &&
                    BrokerCommunicationException.Category.OPERATION_NOT_SUPPORTED_ON_SERVER_SIDE == t.category
                ) {
                    Logger.info(
                        methodTag,
                        "Tried broker discovery on ${candidate}. It doesn't support the IPC mechanism."
                    )
                } else if (t is ClientException && ONLY_SUPPORTS_ACCOUNT_MANAGER_ERROR_CODE == t.errorCode) {
                    Logger.info(
                        methodTag,
                        "Tried broker discovery on ${candidate}. " +
                                "The Broker side indicates that only AccountManager is supported."
                    )
                } else {
                    Logger.error(
                        methodTag,
                        "Tried broker discovery on ${candidate}, get an error", t
                    )
                }
                null
            }
        }

        /**
         * Extract the result returned via the IPC operation
         **/
        @Throws(NoSuchElementException::class)
        private fun extractResult(bundle: Bundle?,
                                  forceTriggerDiscoveryFlow: Boolean): BrokerData? {
            if (bundle == null) {
                return null
            }

            val errorData = bundle.getSerializable(ERROR_BUNDLE_KEY)
            if (errorData != null) {
                throw errorData as Throwable
            }

            if (forceTriggerDiscoveryFlow &&
                !bundle.containsKey(FORCE_TRIGGER_BROKER_DISCOVERY_RESULT_EXECUTED_BUNDLE_KEY)) {
                throw ClientException(
                    FORCE_TRIGGER_BROKER_DISCOVERY_RESULT_OPERATION_NOT_SUPPORTED,
                    "Force Broker Discovery is not supported by the broker side. Please update the app."
                )
            }

            val pkgName = bundle.getString(ACTIVE_BROKER_PACKAGE_NAME_BUNDLE_KEY)
                ?: throw NoSuchElementException("ACTIVE_BROKER_PACKAGE_NAME_BUNDLE_KEY must not be null")

            val signatureHash =
                bundle.getString(ACTIVE_BROKER_SIGNING_CERTIFICATE_THUMBPRINT_BUNDLE_KEY)
                    ?: throw NoSuchElementException("ACTIVE_BROKER_SIGNING_CERTIFICATE_THUMBPRINT_BUNDLE_KEY must not be null")

            return BrokerData(pkgName, signatureHash)
        }
    }

    data class CachedBrokerData (
        val brokerData: BrokerData?
    )

    /**
     * In-memory cache for the active broker data.
     * There are 3 possible states:
     * 1. null: the cache hasn't been initialized (needs to read from storage or query from broker first).
     * 2. CachedBrokerData with null brokerData: no active broker found.
     * 3. CachedBrokerData with non-null brokerData: found an active broker.
     **/
    @Volatile
    @VisibleForTesting
    var cachedData: CachedBrokerData? = null

    constructor(context: Context,
                components: IPlatformComponents,
                cache: IClientActiveBrokerCache) : this(
        brokerCandidates = BrokerData.getKnownBrokerApps(),
        getActiveBrokerFromAccountManager = {
            AccountManagerBrokerDiscoveryUtil(context).getActiveBrokerFromAccountManager()
        },
        ipcStrategy = ContentProviderStrategy(context, components),
        cache = cache,
        isPackageInstalled = { brokerData ->
            PackageHelper(context).isPackageInstalledAndEnabled(brokerData.packageName)
        },
        isValidBroker = { brokerData ->
            BrokerValidator(context).isSignedByKnownKeys(brokerData)
        })

    @kotlin.jvm.Throws(ClientException::class)
    override fun forceBrokerRediscovery(brokerCandidate: BrokerData): BrokerData {
        val methodTag = "$TAG:forceBrokerRediscovery"
        return runBlocking {
            classLevelLock.withLock {
                try {
                    if (!isPackageInstalled(brokerCandidate)) {
                        throw ClientException(
                            FORCE_TRIGGER_BROKER_DISCOVERY_PACKAGE_NOT_INSTALLED,
                            "${brokerCandidate.packageName} is not installed."
                        )
                    }

                    if (!isValidBroker(brokerCandidate)) {
                        throw ClientException(
                            FORCE_TRIGGER_BROKER_DISCOVERY_NOT_VALID_BROKER,
                            "${brokerCandidate.packageName} is not signed with valid key."
                        )
                    }

                    val operationBundle = BrokerOperationBundle(
                        BrokerOperationBundle.Operation.BROKER_DISCOVERY_FROM_SDK,
                        brokerCandidate.packageName,
                        Bundle().apply {
                            putBoolean(FORCE_TRIGGER_BROKER_DISCOVERY_BUNDLE_KEY, true)
                        }
                    )

                    val bundleResult = ipcStrategy.communicateToBroker(operationBundle)
                    val result = extractResult(bundleResult, forceTriggerDiscoveryFlow = true)
                        ?: throw ClientException(
                            FORCE_TRIGGER_BROKER_DISCOVERY_RESULT_UNEXPECTED_ERROR,
                            "Result bundle should not be null."
                        )
                    cache.setCachedActiveBroker(result)
                    return@runBlocking result
                } catch (c: ClientException) {
                    Logger.error(methodTag, "forceBrokerRediscovery Failed.", c)
                    throw c
                } catch (t: Throwable) {
                    Logger.error(methodTag, "forceBrokerRediscovery Failed with unknown error.", t)
                    throw ClientException(
                        FORCE_TRIGGER_BROKER_DISCOVERY_RESULT_UNEXPECTED_ERROR,
                        "Unexpected result: ${t.message}",
                        t
                    )
                }
            }
        }
    }

    override fun getActiveBrokerWithInMemoryCache(telemetryCallback: IBrokerDiscoveryClientTelemetryCallback?): BrokerData?{
        cachedData?.let {
            if (it.brokerData == null) {
                return null
            }
            if (validateInMemoryCacheValue(it.brokerData, telemetryCallback)){
                return it.brokerData
            }

            // Even if the value is not valid, we don't modify (invalidate) the class variable here since we're not in the lock.
            // The variable will be updated in the block below.
        }

        return runBlocking {
            val timeStartAcquiringLock = System.nanoTime()
            classLevelLock.withLock {
                telemetryCallback?.onLockAcquired(System.nanoTime() - timeStartAcquiringLock)

                // just in case the value is already populated while waiting for the lock.
                cachedData?.let {
                    if (it.brokerData == null) {
                        return@runBlocking null
                    }
                    if (validateInMemoryCacheValue(it.brokerData, telemetryCallback)){
                        return@runBlocking it.brokerData
                    }
                }

                val brokerData = getActiveBrokerAsync(shouldSkipCache = false, telemetryCallback)
                cachedData = CachedBrokerData(brokerData)
                return@runBlocking brokerData
            }
        }
    }

    /**
     * Make sure the [BrokerData] that stays in the memory cache (if any) is valid.
     **/
    private fun validateInMemoryCacheValue(
        data: BrokerData,
        telemetryCallback: IBrokerDiscoveryClientTelemetryCallback?,
    ): Boolean {
        val timeStartIsValidBroker = System.nanoTime()
        val isValidBroker = isValidBroker(data)
        telemetryCallback?.onFinishCheckingIfValidBroker(System.nanoTime() - timeStartIsValidBroker)
        return isValidBroker
    }

    override fun getActiveBroker(shouldSkipCache: Boolean): BrokerData? {
        return runBlocking {
            classLevelLock.withLock {
                return@runBlocking getActiveBrokerAsync(shouldSkipCache, null)
            }
        }
    }

    override fun getActiveBroker(
        shouldSkipCache: Boolean,
        telemetryCallback: IBrokerDiscoveryClientTelemetryCallback
    ): BrokerData? {
        return runBlocking {
            val timeStartAcquiringLock = System.nanoTime()
            classLevelLock.withLock {
                telemetryCallback.onLockAcquired(System.nanoTime() - timeStartAcquiringLock)
                return@runBlocking getActiveBrokerAsync(shouldSkipCache, telemetryCallback)
            }
        }
    }

    private suspend fun getActiveBrokerAsync(shouldSkipCache:Boolean,
                                             telemetryCallback: IBrokerDiscoveryClientTelemetryCallback?): BrokerData?{
        val methodTag = "$TAG:getActiveBrokerAsync"
        if (!shouldSkipCache) {
            if (cache.shouldUseAccountManager()) {
                telemetryCallback?.onUseAccountManager()
                return getActiveBrokerFromAccountManager()
            }
            val timeStartReadingFromCache = System.nanoTime()
            cache.getCachedActiveBroker()?.let {
                telemetryCallback?.onReadFromCache(System.nanoTime() - timeStartReadingFromCache)

                val timeStartIsPackageInstalled = System.nanoTime()
                val isPackageInstalled = isPackageInstalled(it)
                telemetryCallback?.onFinishCheckingIfPackageIsInstalled(System.nanoTime() - timeStartIsPackageInstalled)
                if (!isPackageInstalled) {
                    Logger.info(
                        methodTag,
                        "There is a cached broker: $it, but the app is no longer installed."
                    )
                    cache.clearCachedActiveBroker()
                    return@let
                }

                val timeStartIsValidBroker = System.nanoTime()
                val isValidBroker = isValidBroker(it)
                telemetryCallback?.onFinishCheckingIfValidBroker(System.nanoTime() - timeStartIsValidBroker)
                if (!isValidBroker) {
                    Logger.info(
                        methodTag,
                        "Clearing cache as the installed app does not have a matching signature hash."
                    )
                    cache.clearCachedActiveBroker()
                    return@let
                }

                Logger.info(methodTag, "Returning cached broker: $it")
                return it
            }
        }

        val timeStartQueryFromBroker = System.nanoTime()
        val brokerData = queryFromBroker(
            brokerCandidates = brokerCandidates,
            ipcStrategy = ipcStrategy,
            isPackageInstalled = isPackageInstalled,
            isValidBroker = isValidBroker
        )
        telemetryCallback?.onFinishQueryingResultFromBroker(System.nanoTime() - timeStartQueryFromBroker)

        if (brokerData != null) {
            cache.setCachedActiveBroker(brokerData)
            return brokerData
        }

        Logger.info(
            methodTag,
            "Will skip broker discovery via IPC and fall back to AccountManager " +
                    "for the next 60 minutes."
        )
        cache.clearCachedActiveBroker()
        cache.setShouldUseAccountManagerForTheNextMilliseconds(
            TimeUnit.MINUTES.toMillis(
                60
            )
        )

        telemetryCallback?.onUseAccountManager()
        val accountManagerResult = getActiveBrokerFromAccountManager()
        Logger.info(
            methodTag, "Tried getting active broker from account manager, " +
                    "get ${accountManagerResult?.packageName}."
        )

        return accountManagerResult
    }
}