/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.identity.common.crypto;

import android.content.Context;
import android.os.Build;
import android.security.KeyPairGeneratorSpec;
import android.security.keystore.KeyGenParameterSpec;
import androidx.annotation.RequiresApi;
import androidx.annotation.VisibleForTesting;
import com.microsoft.identity.common.internal.util.AndroidKeyStoreUtil;
import com.microsoft.identity.common.java.controllers.ExceptionAdapter;
import com.microsoft.identity.common.java.crypto.key.AES256SecretKeyGenerator;
import com.microsoft.identity.common.java.crypto.key.ISecretKeyProvider;
import com.microsoft.identity.common.java.crypto.key.KeyUtil;
import com.microsoft.identity.common.java.exception.ClientException;
import com.microsoft.identity.common.java.flighting.CommonFlight;
import com.microsoft.identity.common.java.flighting.CommonFlightsManager;
import com.microsoft.identity.common.java.flighting.IFlightConfig;
import com.microsoft.identity.common.java.opentelemetry.AttributeName;
import com.microsoft.identity.common.java.opentelemetry.OTelUtility;
import com.microsoft.identity.common.java.opentelemetry.SpanExtension;
import com.microsoft.identity.common.java.opentelemetry.SpanName;
import com.microsoft.identity.common.java.util.FileUtil;
import com.microsoft.identity.common.java.util.StringUtil;
import com.microsoft.identity.common.logging.Logger;
import edu.umd.cs.findbugs.annotations.Nullable;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.SpanContext;
import io.opentelemetry.api.trace.StatusCode;
import io.opentelemetry.context.Scope;
import java.io.File;
import java.math.BigInteger;
import java.security.KeyPair;
import java.security.spec.AlgorithmParameterSpec;
import java.util.Calendar;
import java.util.Locale;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import javax.crypto.SecretKey;
import javax.security.auth.x500.X500Principal;
import lombok.NonNull;

public class AndroidWrappedKeyProvider
implements ISecretKeyProvider {
    private static final String CIPHER_TRANSFORMATION = "AES/CBC/PKCS5Padding";
    private static final String TAG = AndroidWrappedKeyProvider.class.getSimpleName() + "#";
    @SuppressFBWarnings(value={"MS_SHOULD_BE_FINAL"})
    public static boolean sSkipKeyInvalidationCheck = false;
    private static final String WRAP_ALGORITHM = "RSA/ECB/PKCS1Padding";
    private static final String WRAP_KEY_ALGORITHM = "RSA";
    public static final String WRAPPED_KEY_KEY_IDENTIFIER = "A001";
    static final int KEY_FILE_SIZE = 1024;
    private static final ConcurrentMap<String, SecretKey> sKeyCacheMap = new ConcurrentHashMap<String, SecretKey>();
    private final Context mContext;
    private final String mAlias;
    private final String mFilePath;

    @Nullable
    @VisibleForTesting
    SecretKey getKeyFromCache() {
        String methodTag = TAG + ":getKeyFromCache";
        if (!(sSkipKeyInvalidationCheck || AndroidKeyStoreUtil.canLoadKey(this.mAlias) && this.getKeyFile().exists())) {
            Logger.warn(methodTag, "Key is invalid, removing from cache");
            this.clearKeyFromCache();
        }
        return (SecretKey)sKeyCacheMap.get(this.mFilePath);
    }

    @VisibleForTesting
    void clearKeyFromCache() {
        sKeyCacheMap.remove(this.mFilePath);
    }

    public AndroidWrappedKeyProvider(@NonNull String alias, @NonNull String filePath, @NonNull Context context) {
        if (alias == null) {
            throw new NullPointerException("alias is marked non-null but is null");
        }
        if (filePath == null) {
            throw new NullPointerException("filePath is marked non-null but is null");
        }
        if (context == null) {
            throw new NullPointerException("context is marked non-null but is null");
        }
        this.mAlias = alias;
        this.mFilePath = filePath;
        this.mContext = context;
    }

    @NonNull
    public String getAlias() {
        return this.mAlias;
    }

    @NonNull
    public String getKeyTypeIdentifier() {
        return WRAPPED_KEY_KEY_IDENTIFIER;
    }

    @NonNull
    public synchronized SecretKey getKey() throws ClientException {
        String methodTag = TAG + ":getKey";
        SecretKey key = this.getKeyFromCache();
        if (key != null) {
            return key;
        }
        Logger.info(methodTag, "Key not in cache or cache is empty, loading key from storage");
        key = this.readSecretKeyFromStorage();
        if (key == null) {
            Logger.info(methodTag, "Key does not exist in storage, generating a new key");
            key = this.generateRandomKey();
        }
        sKeyCacheMap.put(this.mFilePath, key);
        return key;
    }

    @NonNull
    protected SecretKey generateRandomKey() throws ClientException {
        String methodTag = TAG + ":generateRandomKey";
        SecretKey key = AES256SecretKeyGenerator.INSTANCE.generateRandomKey();
        this.saveSecretKeyToStorage(key);
        Logger.info(methodTag, "New key is generated with thumbprint: " + KeyUtil.getKeyThumbPrint((SecretKey)key));
        return key;
    }

    @Nullable
    synchronized SecretKey readSecretKeyFromStorage() throws ClientException {
        String methodTag = TAG + ":readSecretKeyFromStorage";
        try {
            KeyPair keyPair = AndroidKeyStoreUtil.readKey(this.mAlias);
            if (keyPair == null) {
                Logger.info(methodTag, "key does not exist in keystore");
                this.deleteSecretKeyFromStorage();
                return null;
            }
            byte[] wrappedSecretKey = FileUtil.readFromFile((File)this.getKeyFile(), (int)1024);
            if (wrappedSecretKey == null) {
                Logger.warn(methodTag, "Key file is empty");
                FileUtil.deleteFile((File)this.getKeyFile());
                sKeyCacheMap.remove(this.mFilePath);
                return null;
            }
            SecretKey key = AndroidKeyStoreUtil.unwrap(wrappedSecretKey, "AES", keyPair, WRAP_ALGORITHM);
            Logger.info(methodTag, "Key is loaded with thumbprint: " + KeyUtil.getKeyThumbPrint((SecretKey)key));
            return key;
        }
        catch (ClientException e) {
            Logger.warn(methodTag, "Error when loading key from Storage, wipe all existing key data ");
            this.deleteSecretKeyFromStorage();
            throw e;
        }
    }

    private void saveSecretKeyToStorage(@NonNull SecretKey unencryptedKey) throws ClientException {
        if (unencryptedKey == null) {
            throw new NullPointerException("unencryptedKey is marked non-null but is null");
        }
        String methodTag = TAG + ":saveSecretKeyToStorage";
        KeyPair keyPair = AndroidKeyStoreUtil.readKey(this.mAlias);
        if (keyPair == null) {
            Logger.info(methodTag, "No existing keypair. Generating a new one.");
            Span span2 = OTelUtility.createSpanFromParent((String)SpanName.KeyPairGeneration.name(), (SpanContext)SpanExtension.current().getSpanContext());
            try (Scope ignored = SpanExtension.makeCurrentSpan((Span)span2);){
                keyPair = this.generateNewKeyPair();
                span2.setStatus(StatusCode.OK);
            }
            catch (ClientException e) {
                span2.setStatus(StatusCode.ERROR);
                span2.recordException((Throwable)e);
                throw e;
            }
            finally {
                span2.end();
            }
        }
        byte[] keyWrapped = AndroidKeyStoreUtil.wrap(unencryptedKey, keyPair, WRAP_ALGORITHM);
        FileUtil.writeDataToFile((byte[])keyWrapped, (File)this.getKeyFile());
    }

    @NonNull
    private KeyPair generateNewKeyPair() throws ClientException {
        if (Build.VERSION.SDK_INT >= 28) {
            return this.generateNewKeyPairAPI28AndAbove();
        }
        if (Build.VERSION.SDK_INT >= 23) {
            return this.generateNewKeyPairAPI23AndAbove();
        }
        return this.generateKeyPairWithLegacySpec();
    }

    @RequiresApi(value=28)
    @NonNull
    private KeyPair generateNewKeyPairAPI28AndAbove() throws ClientException {
        if (CommonFlightsManager.INSTANCE.getFlightsProvider().isFlightEnabled((IFlightConfig)CommonFlight.ENABLE_NEW_KEY_GEN_SPEC_FOR_WRAP_WITH_PURPOSE_WRAP_KEY)) {
            return this.generateWrappingKeyPair_WithPurposeWrapKey();
        }
        if (CommonFlightsManager.INSTANCE.getFlightsProvider().isFlightEnabled((IFlightConfig)CommonFlight.ENABLE_NEW_KEY_GEN_SPEC_FOR_WRAP_WITHOUT_PURPOSE_WRAP_KEY)) {
            return this.generateWrappingKeyPair();
        }
        return this.generateKeyPairWithLegacySpec();
    }

    @RequiresApi(value=23)
    @NonNull
    private KeyPair generateNewKeyPairAPI23AndAbove() throws ClientException {
        if (CommonFlightsManager.INSTANCE.getFlightsProvider().isFlightEnabled((IFlightConfig)CommonFlight.ENABLE_NEW_KEY_GEN_SPEC_FOR_WRAP_WITHOUT_PURPOSE_WRAP_KEY)) {
            return this.generateWrappingKeyPair();
        }
        return this.generateKeyPairWithLegacySpec();
    }

    @NonNull
    private KeyPair generateKeyPairWithLegacySpec() throws ClientException {
        Span span2 = SpanExtension.current();
        try {
            AlgorithmParameterSpec keyPairGenSpec = this.getLegacySpecForKeyStoreKey();
            KeyPair keyPair = this.attemptKeyPairGeneration(keyPairGenSpec);
            span2.setAttribute(AttributeName.key_pair_gen_successful_method.name(), "legacy_key_gen_spec");
            return keyPair;
        }
        catch (Throwable e) {
            Logger.error(TAG + ":generateKeyPairWithLegacySpec", "Error generating keypair with legacy spec.", e);
            throw ExceptionAdapter.clientExceptionFromException((Throwable)e);
        }
    }

    @RequiresApi(value=28)
    private KeyPair generateWrappingKeyPair_WithPurposeWrapKey() throws ClientException {
        String methodTag = TAG + ":generateWrappingKeyPair_WithPurposeWrapKey";
        Span span2 = SpanExtension.current();
        try {
            Logger.info(methodTag, "Generating new keypair with new spec with purpose_wrap_key");
            int purposes = 35;
            AlgorithmParameterSpec keyPairGenSpec = this.getSpecForWrappingKey(purposes);
            KeyPair keyPair = this.attemptKeyPairGeneration(keyPairGenSpec);
            span2.setAttribute(AttributeName.key_pair_gen_successful_method.name(), "new_key_gen_spec_with_wrap");
            return keyPair;
        }
        catch (Throwable e) {
            Logger.error(methodTag, "Error generating keypair with new spec with purpose_wrap_key.Attempting without purpose_wrap_key.", e);
            if (!StringUtil.isNullOrEmpty((String)e.getMessage())) {
                span2.setAttribute(AttributeName.keypair_gen_exception.name(), e.getMessage());
            }
            return this.generateWrappingKeyPair();
        }
    }

    @RequiresApi(value=23)
    private KeyPair generateWrappingKeyPair() throws ClientException {
        String methodTag = TAG + ":generateWrappingKeyPair";
        Span span2 = SpanExtension.current();
        try {
            Logger.info(methodTag, "Generating new keypair with new spec without wrap key");
            int purposes = 3;
            AlgorithmParameterSpec keyPairGenSpec = this.getSpecForWrappingKey(purposes);
            KeyPair keyPair = this.attemptKeyPairGeneration(keyPairGenSpec);
            span2.setAttribute(AttributeName.key_pair_gen_successful_method.name(), "new_key_gen_spec_without_wrap");
            return keyPair;
        }
        catch (Throwable e) {
            Logger.error(methodTag, "Error generating keypair with new spec.Attempting with legacy spec.", e);
            if (!StringUtil.isNullOrEmpty((String)e.getMessage())) {
                span2.setAttribute(AttributeName.keypair_gen_exception.name(), e.getMessage());
            }
            return this.generateKeyPairWithLegacySpec();
        }
    }

    private KeyPair attemptKeyPairGeneration(@NonNull AlgorithmParameterSpec keyPairGenSpec) throws ClientException {
        if (keyPairGenSpec == null) {
            throw new NullPointerException("keyPairGenSpec is marked non-null but is null");
        }
        long keypairGenStartTime = System.currentTimeMillis();
        KeyPair keyPair = AndroidKeyStoreUtil.generateKeyPair(WRAP_KEY_ALGORITHM, keyPairGenSpec);
        this.recordKeyGenerationTime(keypairGenStartTime);
        return keyPair;
    }

    private void recordKeyGenerationTime(long keypairGenStartTime) {
        long elapsedTime = System.currentTimeMillis() - keypairGenStartTime;
        SpanExtension.current().setAttribute(AttributeName.elapsed_time_keypair_generation.name(), elapsedTime);
    }

    public void deleteSecretKeyFromStorage() throws ClientException {
        AndroidKeyStoreUtil.deleteKey(this.mAlias);
        FileUtil.deleteFile((File)this.getKeyFile());
        sKeyCacheMap.remove(this.mFilePath);
    }

    private AlgorithmParameterSpec getLegacySpecForKeyStoreKey() {
        String certInfo = String.format(Locale.ROOT, "CN=%s, OU=%s", this.mAlias, this.mContext.getPackageName());
        Calendar start = Calendar.getInstance();
        Calendar end = Calendar.getInstance();
        int certValidYears = 100;
        end.add(1, 100);
        return new KeyPairGeneratorSpec.Builder(this.mContext).setAlias(this.mAlias).setSubject(new X500Principal(certInfo)).setSerialNumber(BigInteger.ONE).setStartDate(start.getTime()).setEndDate(end.getTime()).build();
    }

    @RequiresApi(api=23)
    private AlgorithmParameterSpec getSpecForWrappingKey(int purposes) {
        return new KeyGenParameterSpec.Builder(this.mAlias, purposes).setKeySize(2048).setDigests(new String[]{"SHA-256", "SHA-512"}).setEncryptionPaddings(new String[]{"PKCS1Padding"}).build();
    }

    private File getKeyFile() {
        return new File(this.mContext.getDir(this.mContext.getPackageName(), 0), this.mFilePath);
    }

    @NonNull
    public String getCipherTransformation() {
        return CIPHER_TRANSFORMATION;
    }
}

