// Copyright (c) Microsoft Corporation.
// All rights reserved.
//
// This code is licensed under the MIT License.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files(the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and / or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions :
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
package com.microsoft.identity.common.internal.fido

import com.microsoft.identity.common.internal.fido.FidoChallengeField.Companion.throwIfInvalidOptionalListParameter
import com.microsoft.identity.common.internal.fido.FidoChallengeField.Companion.throwIfInvalidProtocolVersion
import com.microsoft.identity.common.internal.fido.FidoChallengeField.Companion.throwIfInvalidRelyingPartyIdentifier
import com.microsoft.identity.common.internal.fido.FidoChallengeField.Companion.throwIfInvalidRequiredParameter
import com.microsoft.identity.common.internal.fido.FidoChallengeField.Companion.throwIfInvalidSubmitUrl
import com.microsoft.identity.common.java.util.UrlUtil

/**
 * An object representing a FIDO challenge request from the server.
 *
 * @param challenge                 Random data string generated by the server. This is a required parameter.
 * @param relyingPartyIdentifier    The domain name of the identity provider. This is a required parameter.
 * @param userVerificationPolicy    Relying party's user verification preferences (required, preferred, none. For AAD, this should always be "required").
 *                                  This is a required parameter.
 * @param version                   Passkey Auth protocol version. This is a required parameter.
 * @param submitUrl                 The Url to which the client submits the response to the server's challenge. This is a required parameter.
 * @param context                   Server state that needs to be maintained between challenge and response. This is a required parameter.
 * @param keyTypes                  Array of allowed key types. This is an option auth parameter.
 * @param allowedCredentials        List of credential IDs the client will use for filtering. This is an option auth parameter.
 *
 */
data class FidoChallenge(
    val challenge: FidoChallengeField<String>,
    val relyingPartyIdentifier: FidoChallengeField<String>,
    val userVerificationPolicy: FidoChallengeField<String>,
    val version: FidoChallengeField<String>,
    val submitUrl: FidoChallengeField<String>,
    val context: FidoChallengeField<String>,
    val keyTypes: FidoChallengeField<List<String>?>,
    val allowedCredentials: FidoChallengeField<List<String>?>
) {
    companion object {
        const val DELIMITER = ","
        const val DEFAULT_USER_VERIFICATION_POLICY = "required"

        @JvmStatic
        fun createFromRedirectUri(redirectUri: String): FidoChallenge {
            val parameters: Map<String, String> = UrlUtil.getParameters(redirectUri)
            return FidoChallenge(
                challenge = FidoChallengeField(FidoRequestField.CHALLENGE, parameters[FidoRequestField.CHALLENGE.fieldName], ::throwIfInvalidRequiredParameter),
                relyingPartyIdentifier = FidoChallengeField(FidoRequestField.RELYING_PARTY_IDENTIFIER, parameters[FidoRequestField.RELYING_PARTY_IDENTIFIER.fieldName], ::throwIfInvalidRelyingPartyIdentifier),
                userVerificationPolicy = FidoChallengeField(FidoRequestField.USER_VERIFICATION_POLICY, parameters[FidoRequestField.USER_VERIFICATION_POLICY.fieldName] ?: DEFAULT_USER_VERIFICATION_POLICY, ::throwIfInvalidRequiredParameter),
                version = FidoChallengeField(FidoRequestField.VERSION, parameters[FidoRequestField.VERSION.fieldName], ::throwIfInvalidProtocolVersion),
                submitUrl = FidoChallengeField(FidoRequestField.SUBMIT_URL, parameters[FidoRequestField.SUBMIT_URL.fieldName], ::throwIfInvalidSubmitUrl),
                context = FidoChallengeField(FidoRequestField.CONTEXT, parameters[FidoRequestField.CONTEXT.fieldName], ::throwIfInvalidRequiredParameter),
                keyTypes = FidoChallengeField(FidoRequestField.KEY_TYPES, parameters[FidoRequestField.KEY_TYPES.fieldName]?.split(DELIMITER), ::throwIfInvalidOptionalListParameter),
                allowedCredentials = FidoChallengeField(FidoRequestField.ALLOWED_CREDENTIALS, parameters[FidoRequestField.ALLOWED_CREDENTIALS.fieldName]?.split(DELIMITER), ::throwIfInvalidOptionalListParameter)
            )
        }
    }
}
