// Copyright (c) Microsoft Corporation.
// All rights reserved.
//
// This code is licensed under the MIT License.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files(the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and / or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions :
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
package com.microsoft.identity.common.internal.providers.oauth2;

import com.microsoft.identity.common.adal.internal.util.StringExtensions;
import com.microsoft.identity.common.exception.ErrorStrings;
import com.microsoft.identity.common.exception.ServiceException;
import com.nimbusds.jwt.JWT;
import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.jwt.JWTParser;

import java.text.ParseException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * Represents the OpenID Connect Id Token.
 * Standard ID token Claims per spec
 * http://openid.net/specs/openid-connect-core-1_0.html#StandardClaims
 */
public class IDToken {

    /**
     * Subject - Identifier for the End-User at the Issuer.
     */
    public static final String SUBJECT = "sub";

    /**
     * End-User's full name in displayable form including all name parts, possibly including titles
     * and suffixes, ordered according to the End-User's locale and preferences.
     */
    public static final String NAME = "name";

    /**
     * Given name(s) or first name(s) of the End-User. Note that in some cultures, people can have
     * multiple given names; all can be present, with the names being separated by space characters.
     */
    public static final String GIVEN_NAME = "given_name";

    /**
     * Surname(s) or last name(s) of the End-User. Note that in some cultures, people can have
     * multiple family names or no family name; all can be present, with the names being separated
     * by space characters.
     */
    public static final String FAMILY_NAME = "family_name";

    /**
     * Middle name(s) of the End-User. Note that in some cultures, people can have multiple middle
     * names; all can be present, with the names being separated by space characters. Also note
     * that in some cultures, middle names are not used.
     */
    public static final String MIDDLE_NAME = "middle_name";

    /**
     * Casual name of the End-User that may or may not be the same as the given_name. For instance,
     * a nickname value of Mike might be returned alongside a given_name value of Michael.
     */
    public static final String NICKNAME = "nickname";

    /**
     * Shorthand name by which the End-User wishes to be referred to at the RP, such as janedoe or
     * j.doe. This value MAY be any valid JSON string including special characters such as @, /,
     * or whitespace. The RP MUST NOT rely upon this value being unique.
     */
    public static final String PREFERRED_USERNAME = "preferred_username";

    /**
     * URL of the End-User's profile page. The contents of this Web page SHOULD be about the
     * End-User.
     */
    public static final String PROFILE = "profile";

    /**
     * URL of the End-User's profile picture. This URL MUST refer to an image file (for example, a
     * PNG, JPEG, or GIF image file), rather than to a Web page containing an image. Note that this
     * URL SHOULD specifically reference a profile photo of the End-User suitable for displaying
     * when describing the End-User, rather than an arbitrary photo taken by the End-User.
     */
    public static final String PICTURE = "picture";

    /**
     * URL of the End-User's Web page or blog. This Web page SHOULD contain information published
     * by the End-User or an organization that the End-User is affiliated with.
     */
    public static final String WEBSITE = "website";

    /**
     * End-User's preferred e-mail address. Its value MUST conform to the RFC 5322 [RFC5322]
     * addr-spec syntax. The RP MUST NOT rely upon this value being unique.
     */
    public static final String EMAIL = "email";

    /**
     * True if the End-User's e-mail address has been verified; otherwise false. When this Claim
     * Value is true, this means that the OP took affirmative steps to ensure that this e-mail
     * address was controlled by the End-User at the time the verification was performed. The means
     * by which an e-mail address is verified is context-specific, and dependent upon the trust
     * framework or contractual agreements within which the parties are operating.
     */
    public static final String EMAIL_VERIFIED = "email_verified";

    /**
     * End-User's gender. Values defined by this specification are female and male. Other values
     * MAY be used when neither of the defined values are applicable.
     */
    public static final String GENDER = "gender";

    /**
     * End-User's birthday, represented as an ISO 8601:2004 [ISO8601‑2004] YYYY-MM-DD format. The
     * year MAY be 0000, indicating that it is omitted. To represent only the year, YYYY format is
     * allowed. Note that depending on the underlying platform's date related function, providing
     * just year can result in varying month and day, so the implementers need to take this factor
     * into account to correctly process the dates.
     */
    public static final String BIRTHDATE = "birthdate";

    /**
     * String from zoneinfo [zoneinfo] time zone database representing the End-User's time zone.
     * For example, Europe/Paris or America/Los_Angeles.
     */
    public static final String ZONEINFO = "zoneinfo";

    /**
     * End-User's locale, represented as a BCP47 [RFC5646] language tag. This is typically an
     * ISO 639-1 Alpha-2 [ISO639‑1] language code in lowercase and an ISO 3166-1 Alpha-2 [ISO3166‑1]
     * country code in uppercase, separated by a dash. For example, en-US or fr-CA. As a
     * compatibility note, some implementations have used an underscore as the separator rather
     * than a dash, for example, en_US; Relying Parties MAY choose to accept this locale syntax as
     * well.
     */
    public static final String LOCALE = "locale";

    /**
     * End-User's preferred telephone number. E.164 [E.164] is RECOMMENDED as the format of this
     * Claim, for example, +1 (425) 555-1212 or +56 (2) 687 2400. If the phone number contains an
     * extension, it is RECOMMENDED that the extension be represented using the RFC 3966 [RFC3966]
     * extension syntax, for example, +1 (604) 555-1234;ext=5678.
     */
    public static final String PHONE_NUMBER = "phone_number";

    /**
     * True if the End-User's phone number has been verified; otherwise false. When this Claim
     * Value is true, this means that the OP took affirmative steps to ensure that this phone
     * number was controlled by the End-User at the time the verification was performed. The means
     * by which a phone number is verified is context-specific, and dependent upon the trust
     * framework or contractual agreements within which the parties are operating. When true, the
     * phone_number Claim MUST be in E.164 format and any extensions MUST be represented in
     * RFC 3966 format.
     */
    public static final String PHONE_NUMBER_VERIFIED = "phone_number_verified";

    /**
     * End-User's preferred postal address. The value of the address member is a JSON [RFC4627]
     * structure containing some or all of the members defined in Section 5.1.1.
     */
    public static final String ADDRESS = "address";

    /**
     * Time the End-User's information was last updated. Its value is a JSON number representing
     * the number of seconds from 1970-01-01T0:0:0Z as measured in UTC until the date/time.
     */
    public static final String UPDATED_AT = "updated_at";


    private Map<String, String> mTokenClaims = null;
    private final String mRawIdToken;

    /**
     * Constructor of IDToken.
     *
     * @param rawIdToken raw ID token
     * @throws ServiceException if rawIdToken is malformed in JSON format.
     */
    public IDToken(final String rawIdToken) throws ServiceException {

        if (StringExtensions.isNullOrBlank(rawIdToken)) {
            throw new IllegalArgumentException("null or empty raw idtoken");
        }

        mRawIdToken = rawIdToken;

        // set all the instance variables.
        mTokenClaims = parseJWT(rawIdToken);
    }

    /**
     * @return mRawIdToken of IDToken object.
     */
    public String getRawIDToken() {
        return mRawIdToken;
    }

    /**
     * @return Token claims in Map<String, String>.
     */
    public Map<String, String> getTokenClaims() {
        return Collections.unmodifiableMap(mTokenClaims);
    }

    public static Map<String, String> parseJWT(final String idToken) throws ServiceException {
        final JWTClaimsSet claimsSet;
        try {
            // Create a SignedJWT from the input token String
            final JWT jwt = JWTParser.parse(idToken);
            claimsSet = jwt.getJWTClaimsSet();
        } catch (ParseException e) {
            throw new ServiceException("Failed to parse JWT", ErrorStrings.INVALID_JWT, e);
        }

        // Grab the claims and stick them into a Map<String, Object>
        final Map<String, Object> claimsMap = claimsSet.getClaims();

        // Convert that Map<String, Object> into Map<String, String>
        final Map<String, String> claimsMapStr = new HashMap<>();

        for (final Map.Entry<String, Object> entry : claimsMap.entrySet()) {
            claimsMapStr.put(entry.getKey(), entry.getValue().toString());
        }

        // Return our result
        return claimsMapStr;

    }

}
