/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.identity.common.internal.broker;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.Signature;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.Base64;
import com.microsoft.identity.common.adal.internal.AuthenticationSettings;
import com.microsoft.identity.common.exception.ClientException;
import com.microsoft.identity.common.internal.broker.PackageHelper;
import com.microsoft.identity.common.internal.logging.Logger;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.PKIXParameters;
import java.security.cert.TrustAnchor;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public class BrokerValidator {
    private static final String TAG = "BrokerValidator";
    private final Context mContext;
    private final String mCompanyPortalSignature;

    public BrokerValidator(Context context) {
        this.mContext = context;
        this.mCompanyPortalSignature = AuthenticationSettings.INSTANCE.getBrokerSignature();
    }

    public boolean verifySignature(String brokerPackageName) {
        String methodName = ":verifySignature";
        try {
            List<X509Certificate> certs = this.readCertDataForBrokerApp(brokerPackageName);
            this.verifySignatureHash(certs);
            if (certs.size() > 1) {
                this.verifyCertificateChain(certs);
            }
            return true;
        }
        catch (PackageManager.NameNotFoundException e) {
            Logger.error("BrokerValidator:verifySignature", "Broker related package does not exist", e);
        }
        catch (NoSuchAlgorithmException e) {
            Logger.error("BrokerValidator:verifySignature", "Digest SHA algorithm does not exists", e);
        }
        catch (ClientException | IOException | GeneralSecurityException e) {
            Logger.error("BrokerValidator:verifySignature", "Signature could not be verified", e);
        }
        return false;
    }

    private void verifySignatureHash(List<X509Certificate> certs) throws NoSuchAlgorithmException, CertificateEncodingException, ClientException {
        for (X509Certificate x509Certificate : certs) {
            MessageDigest messageDigest = MessageDigest.getInstance("SHA");
            messageDigest.update(x509Certificate.getEncoded());
            String signatureHash = Base64.encodeToString((byte[])messageDigest.digest(), (int)2);
            if (!this.mCompanyPortalSignature.equals(signatureHash) && !"ho040S3ffZkmxqtQrSwpTVOn9r0=".equals(signatureHash)) continue;
            return;
        }
        throw new ClientException("Calling app could not be verified");
    }

    @SuppressLint(value={"PackageManagerGetSignatures"})
    private List<X509Certificate> readCertDataForBrokerApp(String brokerPackageName) throws PackageManager.NameNotFoundException, ClientException, IOException, GeneralSecurityException {
        PackageInfo packageInfo = this.mContext.getPackageManager().getPackageInfo(brokerPackageName, 64);
        if (packageInfo == null) {
            throw new ClientException("App package name is not found in the package manager", "No broker package existed.");
        }
        if (packageInfo.signatures == null || packageInfo.signatures.length == 0) {
            throw new ClientException("Calling app could not be verified", "No signature associated with the broker package.");
        }
        ArrayList<X509Certificate> certificates = new ArrayList<X509Certificate>(packageInfo.signatures.length);
        for (Signature signature : packageInfo.signatures) {
            byte[] rawCert = signature.toByteArray();
            ByteArrayInputStream certStream = new ByteArrayInputStream(rawCert);
            try {
                CertificateFactory certificateFactory = CertificateFactory.getInstance("X509");
                X509Certificate x509Certificate = (X509Certificate)certificateFactory.generateCertificate(certStream);
                certificates.add(x509Certificate);
            }
            catch (CertificateException e) {
                throw new ClientException("Calling app could not be verified");
            }
        }
        return certificates;
    }

    private void verifyCertificateChain(List<X509Certificate> certificates) throws GeneralSecurityException, ClientException {
        X509Certificate issuerCert = this.getSelfSignedCert(certificates);
        TrustAnchor trustAnchor = new TrustAnchor(issuerCert, null);
        PKIXParameters pkixParameters = new PKIXParameters(Collections.singleton(trustAnchor));
        pkixParameters.setRevocationEnabled(false);
        CertPath certPath = CertificateFactory.getInstance("X.509").generateCertPath(certificates);
        CertPathValidator certPathValidator = CertPathValidator.getInstance("PKIX");
        certPathValidator.validate(certPath, pkixParameters);
    }

    private X509Certificate getSelfSignedCert(List<X509Certificate> certs) throws ClientException {
        int count = 0;
        X509Certificate selfSignedCert = null;
        for (X509Certificate x509Certificate : certs) {
            if (!x509Certificate.getSubjectDN().equals(x509Certificate.getIssuerDN())) continue;
            selfSignedCert = x509Certificate;
            ++count;
        }
        if (count > 1 || selfSignedCert == null) {
            throw new ClientException("Calling app could not be verified", "Multiple self signed certs found or no self signed cert existed.");
        }
        return selfSignedCert;
    }

    public static boolean isValidBrokerRedirect(@Nullable String redirectUri, @NonNull Context context, @NonNull String packageName) {
        return !TextUtils.isEmpty((CharSequence)redirectUri) && redirectUri.equalsIgnoreCase(BrokerValidator.getBrokerRedirectUri(context, packageName));
    }

    public static String getBrokerRedirectUri(Context context, String packageName) {
        PackageHelper info = new PackageHelper(context.getPackageManager());
        String signatureDigest = info.getCurrentSignatureForPackage(packageName);
        return PackageHelper.getBrokerRedirectUrl(packageName, signatureDigest);
    }
}

