/**
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for
 * license information.
 */

package com.microsoft.azure.management.compute;

import com.microsoft.azure.management.apigeneration.Fluent;
import com.microsoft.azure.management.compute.implementation.ResourceSkuInner;
import com.microsoft.azure.management.resources.fluentcore.arm.AvailabilityZoneId;
import com.microsoft.azure.management.resources.fluentcore.arm.Region;
import com.microsoft.azure.management.resources.fluentcore.model.HasInner;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Type representing sku for an Azure compute resource.
 */
@Fluent
public interface ComputeSku extends HasInner<ResourceSkuInner> {
    /**
     * @return the sku name
     */
    ComputeSkuName name();
    /**
     * @return the sku tier
     */
    ComputeSkuTier tier();
    /**
     * @return the compute resource type that the sku describes
     */
    ComputeResourceType resourceType();
    /**
     * The virtual machine size type if the sku describes sku for virtual machine resource type.
     *
     * The size can be used for {@link VirtualMachine.DefinitionStages.WithVMSize#withSize(VirtualMachineSizeTypes)}
     * and {@link VirtualMachine.Update#withSize(VirtualMachineSizeTypes)}.
     *
     * @return the virtual machine size type
     */
    VirtualMachineSizeTypes virtualMachineSizeType();
    /**
     * The managed disk or snapshot sku type if the sku describes sku for disk or snapshot resource type.
     *
     * The sku type can be used for {@link Disk.DefinitionStages.WithSku#withSku(DiskSkuTypes)},
     * {@link Disk.UpdateStages.WithSku#withSku(DiskSkuTypes)},
     * {@link Snapshot.DefinitionStages.WithSku#withSku(DiskSkuTypes)} and
     * {@link Snapshot.UpdateStages.WithSku#withSku(DiskSkuTypes)}.
     *
     * @return the managed disk or snapshot sku type
     */
    DiskSkuTypes diskSkuType();
    /**
     * The availability set sku type if the sku describes sku for availability set resource type.
     *
     * The sku type can be used for {@link AvailabilitySet.DefinitionStages.WithSku#withSku(AvailabilitySetSkuTypes)}
     * and {@link AvailabilitySet.UpdateStages.WithSku#withSku(AvailabilitySetSkuTypes)}.
     *
     * @return the availability set sku type
     */
    AvailabilitySetSkuTypes availabilitySetSkuType();
    /**
     * @return the regions that the sku is available
     */
    List<Region> regions();
    /**
     * @return the availability zones supported for this sku, index by region
     */
    Map<Region, Set<AvailabilityZoneId>>  zones();
    /**
     * @return the scaling information of the sku
     */
    ResourceSkuCapacity capacity();
    /**
     * @return the api versions that this sku supports
     */
    List<String> apiVersions();
    /**
     * @return the metadata for querying the sku pricing information
     */
    List<ResourceSkuCosts> costs();
    /**
     * @return the capabilities of the sku
     */
    List<ResourceSkuCapabilities> capabilities();
    /**
     * @return the restrictions because of which SKU cannot be used
     */
    List<ResourceSkuRestrictions> restrictions();
}
