/*
 * The MIT License (MIT)
 * Copyright (c) 2017 Microsoft Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.microsoft.azure.documentdb.internal.query;

import static com.microsoft.azure.documentdb.internal.query.ExceptionHelper.toRuntimeException;
import static com.microsoft.azure.documentdb.internal.query.ExceptionHelper.unwrap;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.PriorityBlockingQueue;

import com.microsoft.azure.documentdb.DocumentQueryClientInternal;
import com.microsoft.azure.documentdb.FeedOptions;
import com.microsoft.azure.documentdb.PartitionKeyRange;
import com.microsoft.azure.documentdb.SqlQuerySpec;
import com.microsoft.azure.documentdb.internal.query.funcs.Func1;
import com.microsoft.azure.documentdb.internal.query.orderbyquery.OrderByDocumentProducerConsumeComparator;
import com.microsoft.azure.documentdb.internal.query.orderbyquery.OrderByQueryResult;

public class OrderByQueryExecutionContext extends ParallelDocumentQueryExecutionContextBase<OrderByQueryResult> {

    private static final int DEFAULT_ORDER_BY_PAGE_SIZE = 1000;
    private static final int MINIMUM_PAGE_SIZE = 5;

    private final PriorityBlockingQueue<DocumentProducer<OrderByQueryResult>> documentProducerConsumeQueue;
    private final Comparator<DocumentProducer<OrderByQueryResult>> comparer;

    private DocumentProducer<OrderByQueryResult> currentDocumentProducer;

    public static OrderByQueryExecutionContext create(
            DocumentQueryClientInternal client,
            String collectionSelfLink,
            SqlQuerySpec querySpec,
            FeedOptions options, String resourceLink, PartitionedQueryExecutionInfo partitionedQueryExecutionInfo) {

        try {
            OrderByQueryExecutionContext context = new OrderByQueryExecutionContext(client, collectionSelfLink, querySpec, options,
                    resourceLink, partitionedQueryExecutionInfo);

            Integer pageSizeForOrderBy = options.getPageSize() == null || options.getPageSize() < 1
                    ? DEFAULT_ORDER_BY_PAGE_SIZE : Math.max(options.getPageSize(), MINIMUM_PAGE_SIZE);

            // TODO: we can refactor this and move querying partition key ranges out of orderby/parallel execution context
            // similar to .NET
            Collection<PartitionKeyRange> ranges = context
                    .getTargetPartitionKeyRanges(partitionedQueryExecutionInfo.getQueryRanges());

            context.initializationFuture = context.initializeAsync(partitionedQueryExecutionInfo, pageSizeForOrderBy,
                    OrderByQueryResult.class, ranges, options);

            return context;
        } catch (Exception e) {
            throw toRuntimeException(unwrap(e));
        }
    }

    private OrderByQueryExecutionContext(DocumentQueryClientInternal client, String collectionSelfLink, SqlQuerySpec querySpec, FeedOptions options,
            String resourceLink, PartitionedQueryExecutionInfo partitionedQueryExecutionInfo) {
        super(client, collectionSelfLink, querySpec, options, resourceLink, partitionedQueryExecutionInfo, OrderByQueryResult.class);

        comparer =  new OrderByDocumentProducerConsumeComparator(
                partitionedQueryExecutionInfo.getQueryInfo().getOrderBy());
        this.documentProducerConsumeQueue = new PriorityBlockingQueue<DocumentProducer<OrderByQueryResult>>(
                this.documentProducers.capacity(), comparer);
    }

    /* (non-Javadoc)
     * @see com.microsoft.azure.documentdb.internal.query.AbstractQueryExecutionContext#hasNextInternal()
     */
    @Override
    protected boolean hasNextInternal() {
        return !isDone();
    }

    @Override
    protected Future<Void> initializeAsync(PartitionedQueryExecutionInfo partitionedQueryExecutionInfo, int initialPageSize,
            final Class<OrderByQueryResult> documentProducerClassT,
            final Collection<PartitionKeyRange> ranges,
            final FeedOptions options) throws Exception {

        super.initializeAsync(partitionedQueryExecutionInfo, initialPageSize, documentProducerClassT, ranges, options).get();

        final OrderByQueryExecutionContext that = this;
        Callable<Void> callable = new Callable<Void>() {

            @Override
            public Void call() throws Exception {
                // The Foreach loop below is an optimization for the following While loop.
                // The While loop is made single-threaded as the base.DocumentProducers object can change during runtime due to split.
                // Even though the While loop is single threaded, the Foreach loop below makes the producers fetch documents concurrently.
                // If any of the producers fails to produce due to split (i.e., encounters PartitionKeyRangeGoneException),
                // then the while loop below will take out the failed document producers and replace it appropriate ones and then call fetchAsync() on them.
                for (DocumentProducer<OrderByQueryResult> producer: that.documentProducers) {
                    producer.tryScheduleFetch();
                }

                // Fetch one item from each of the producers to initialize the priority-queue. "tryMoveNextProducer()" has
                // a side-effect that, if Split is encountered while trying to move, related parent producer will be taken out and child
                // producers will be added to "base.DocumentProducers".

                for (int index = 0; index < that.documentProducers.size(); ++index) {
                    DocumentProducer<OrderByQueryResult> producer = that.documentProducers.get(index);

                    if (that.tryMoveNextProducer(producer)) {

                        producer = that.documentProducers.get(index);
                        that.documentProducerConsumeQueue.put(producer);
                    }
                }
                return null;
            }
        };

        return super.executorService.submit(callable);
    }

    public boolean isDone() {
        return this.currentDocumentProducer == null && this.documentProducerConsumeQueue.size() <= 0;
    }

    private void updateCurrentDocumentProducer() {
        if (this.documentProducerConsumeQueue.size() > 0) {

            if (this.currentDocumentProducer == null) {
                this.currentDocumentProducer = this.documentProducerConsumeQueue.poll();

            } else if (this.comparer.compare(this.currentDocumentProducer,
                    this.documentProducerConsumeQueue.peek()) > 0) {

                this.documentProducerConsumeQueue.put(this.currentDocumentProducer);
                this.currentDocumentProducer = this.documentProducerConsumeQueue.poll();
            }
        }
    }

    /* (non-Javadoc)
     * @see com.microsoft.azure.documentdb.internal.query.ParallelDocumentQueryExecutionContextBase#nextInternal()
     */
    @Override
    public OrderByQueryResult nextInternal() throws Exception {

        OrderByQueryResult result = null;

        while (!this.isDone() && result == null) {
            this.updateCurrentDocumentProducer();

            OrderByQueryResult orderByResult = (OrderByQueryResult)this.currentDocumentProducer.peek();
            result = orderByResult;

            if (!this.tryMoveNextProducer(this.currentDocumentProducer)) {
                this.currentDocumentProducer = null;
            }
        }
        if (isDone()) {
            super.onFinish();
        }

        return result;
    }

    private boolean tryMoveNextProducer(
            DocumentProducer<OrderByQueryResult> producer) throws Exception {

        final OrderByQueryExecutionContext that = this;
        return super.tryMoveNextProducer(producer,
                new Func1<DocumentProducer<OrderByQueryResult>, DocumentProducer<OrderByQueryResult>>() {

            @Override
            public DocumentProducer<OrderByQueryResult> apply(
                    DocumentProducer<OrderByQueryResult> currentProducer) {
                return that.repairOrderByContext(currentProducer);
            }
        });
    }

    private DocumentProducer<OrderByQueryResult> repairOrderByContext(
            DocumentProducer<OrderByQueryResult> parentProducer) {
        List<PartitionKeyRange> replacementRanges = super.getReplacementRanges(parentProducer.getTargetRange(), this.collectionSelfLink);
        int indexOfCurrentDocumentProducer = Collections.binarySearch(
                super.documentProducers,
                parentProducer,
                new Comparator<DocumentProducer<OrderByQueryResult>>() {
                    @Override
                    public int compare(DocumentProducer<OrderByQueryResult> producer1,
                            DocumentProducer<OrderByQueryResult> producer2) {
                        return producer1.getTargetRange().getMinInclusive().compareTo(producer2.getTargetRange().getMinInclusive());
                    }
                });

        super.repairContext(
                this.collectionSelfLink,
                indexOfCurrentDocumentProducer,
                super.defaultComparator,
                replacementRanges,
                super.querySpec);

        return super.documentProducers.get(indexOfCurrentDocumentProducer);
    }
}
