package com.microsoft.azure.documentdb.internal.query;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Queue;

import com.microsoft.azure.documentdb.internal.BridgeInternal;
import org.apache.commons.lang3.StringUtils;

import com.microsoft.azure.documentdb.ChangeFeedOptions;
import com.microsoft.azure.documentdb.DocumentClientException;
import com.microsoft.azure.documentdb.DocumentCollection;
import com.microsoft.azure.documentdb.DocumentQueryClientInternal;
import com.microsoft.azure.documentdb.FeedOptions;
import com.microsoft.azure.documentdb.FeedOptionsBase;
import com.microsoft.azure.documentdb.PartitionKeyRange;
import com.microsoft.azure.documentdb.Resource;
import com.microsoft.azure.documentdb.SqlQuerySpec;
import com.microsoft.azure.documentdb.internal.DocumentServiceRequest;
import com.microsoft.azure.documentdb.internal.DocumentServiceResponse;
import com.microsoft.azure.documentdb.internal.HttpConstants;
import com.microsoft.azure.documentdb.internal.ResourceType;
import com.microsoft.azure.documentdb.internal.ServiceJNIWrapper;
import com.microsoft.azure.documentdb.internal.Utils;
import com.microsoft.azure.documentdb.internal.routing.CollectionCache;
import com.microsoft.azure.documentdb.internal.routing.PartitionKeyInternal;
import com.microsoft.azure.documentdb.internal.routing.PartitionKeyInternalHelper;
import com.microsoft.azure.documentdb.internal.routing.PartitionKeyRangeIdentity;
import com.microsoft.azure.documentdb.internal.routing.PartitionRoutingHelper;
import com.microsoft.azure.documentdb.internal.routing.Range;
import com.microsoft.azure.documentdb.internal.routing.RoutingMapProvider;

final class DefaultQueryExecutionContext<T extends Resource> extends AbstractQueryExecutionContext<T> {
    private final Queue<T> buffer;
    private PartitionedQueryExecutionInfo queryExecutionInfo;
    private List<Range<String>> providedRanges;
    private DocumentCollection collection;
    private DocumentServiceResponse prefetchedResponse;
    private String partitionKeyRangeId;
    private boolean isChangeFeedRequest;
    private boolean isUserProvidedPartitionKeyRangeIdentity;

    public DefaultQueryExecutionContext(
            DocumentQueryClientInternal client, ResourceType resourceType, Class<T> classT,
            SqlQuerySpec querySpec, PartitionedQueryExecutionInfo queryExecutionInfo,
            FeedOptionsBase options, String resourceLink) {
        super(client, resourceType, classT, querySpec, options, resourceLink);
        this.buffer = new ArrayDeque<T>();
        ChangeFeedOptions changeFeedOptions = options instanceof ChangeFeedOptions ? ((ChangeFeedOptions) options) : null;

        this.partitionKeyRangeId = changeFeedOptions != null ?
                changeFeedOptions.getPartitionKeyRangeId() :
                ((FeedOptions)options).getPartitionKeyRangeIdInternal();
        if (this.partitionKeyRangeId != null) {
            this.isUserProvidedPartitionKeyRangeIdentity = true;
        }

        this.queryExecutionInfo = queryExecutionInfo;
        this.isChangeFeedRequest = options instanceof ChangeFeedOptions;
    }

    @Override
    public List<T> fetchNextBlock() throws DocumentClientException {
        if (!this.hasNext()) {
            return null;
        }

        List<T> result = new ArrayList<T>(this.buffer);
        this.buffer.clear();
        return result;
    }

    public boolean hasNext() {
        try {
            this.fillBuffer();
        } catch (DocumentClientException e) {
            throw new IllegalStateException(e);
        }
        if (this.buffer.isEmpty()) {
            return false;
        }
        return true;
    }

    @Override
    public T next() {
        if (this.buffer.isEmpty()) {
            throw new NoSuchElementException("next");
        }

        return this.buffer.poll();
    }

    @Override
    public void onNotifyStop() {
        // Nothing to do
    }

    private void fillBuffer() throws DocumentClientException {
        while (super.hasNextInternal() && this.buffer.isEmpty()) {
            DocumentServiceResponse response = null;
            if (!this.isChangeFeedRequest || this.prefetchedResponse == null) {
                response = this.executeOnce();
            } else {
                response = this.prefetchedResponse;
                this.prefetchedResponse = null;
            }
            processResponse(response);

            // For ChangeFeed, we need to prefetch the next batch in order to do hasNext() correctly
            if (this.isChangeFeedRequest && super.hasNextInternal()) {
                this.prefetchedResponse = this.executeOnce();
                this.prefetchedStatusCode = this.prefetchedResponse.getStatusCode();
            }
        }
    }

    private void processResponse(DocumentServiceResponse response) {
        super.responseHeaders = response.getResponseHeaders();
        List<T> queryResponse = response.getQueryResponse(this.classT);
        this.buffer.addAll(queryResponse);
    }

    private DocumentServiceResponse executeOnce() throws DocumentClientException {
        // Don't reuse request, the code leaves some temporary garbage in request
        // which should be erased during execution.
        DocumentServiceRequest request = this.partitionKeyRangeId != null
                ? super.createRequest(this.getFeedHeaders(this.options), this.querySpec, this.partitionKeyRangeId)
                : super.createRequest(this.getFeedHeaders(this.options), this.querySpec, this.getPartitionKeyInternal());

        BridgeInternal.setUserProvidedPartitionKeyRangeIdentity(request, this.isUserProvidedPartitionKeyRangeIdentity);

        String continuationToken = super.getContinuationToken();
        if (continuationToken != null) {
            request.getHeaders().put(!this.isChangeFeedRequest ?
                    HttpConstants.HttpHeaders.CONTINUATION :
                    HttpConstants.HttpHeaders.IF_NONE_MATCH, continuationToken);
        }

        String partitionKey = request.getHeaders().get(HttpConstants.HttpHeaders.PARTITION_KEY);
        if (!StringUtils.isEmpty(partitionKey) || !this.resourceType.isPartitioned()) {
            return this.executeRequest(request);
        }

        if (this.partitionKeyRangeId != null) {
            String partitionKeyRangeId = request.getHeaders().get(HttpConstants.HttpHeaders.PARTITION_KEY_RANGE_ID);
            if (!StringUtils.isEmpty(partitionKeyRangeId)) {
                return this.executeRequest(request);
            } else {
                throw new IllegalStateException("For partitioned collection, PartitionKeyRangeId must be specified.");
            }
        }

        if (this.collection == null) {
            this.collection = this.client.getCollectionCache().resolveCollection(request);
        }

        if (!Utils.isCollectionPartitioned(this.collection)) {
            request.routeTo(new PartitionKeyRangeIdentity(this.collection.getResourceId(), "0"));
            return this.executeRequest(request);
        }

        if (!ServiceJNIWrapper.isServiceJNIAvailable()
                && this.shouldExecuteQuery()) {
            return this.executeRequest(request);
        }

        Range<String> rangeFromContinuationToken = PartitionRoutingHelper
                .extractPartitionKeyRangeFromContinuationToken(request.getHeaders());

        RoutingMapProvider routingMapProvider = this.client.getPartitionKeyRangeCache();
        CollectionCache collectionCache = this.client.getCollectionCache();

        this.populateProvidedRanges(request);

        PartitionKeyRange targetPartitionKeyRange = this.tryGetTargetPartitionKeyRange(rangeFromContinuationToken);

        if (request.getIsNameBased() && targetPartitionKeyRange == null) {
            request.setForceNameCacheRefresh(true);
            this.collection = collectionCache.resolveCollection(request);
            targetPartitionKeyRange = this.tryGetTargetPartitionKeyRange(rangeFromContinuationToken);
        }

        if (targetPartitionKeyRange == null) {
            throw new DocumentClientException(HttpConstants.StatusCodes.NOTFOUND, "Target range information not found.");
        }

        request.routeTo(new PartitionKeyRangeIdentity(this.collection.getResourceId(), targetPartitionKeyRange.getId()));

        String globalSessionToken = request.getHeaders().get(HttpConstants.HttpHeaders.SESSION_TOKEN);
        DocumentServiceResponse response = this.executeRequest(request);
        request.getHeaders().put(HttpConstants.HttpHeaders.SESSION_TOKEN, globalSessionToken);

        if (!PartitionRoutingHelper.tryAddPartitionKeyRangeToContinuationToken(
                response.getResponseHeaders(),
                this.providedRanges,
                routingMapProvider,
                this.collection.getSelfLink(),
                targetPartitionKeyRange)) {
            throw new DocumentClientException(HttpConstants.StatusCodes.NOTFOUND, "Collection not found");
        }

        return response;
    }

    private PartitionKeyRange tryGetTargetPartitionKeyRange(
            Range<String> rangeFromContinuationToken) throws DocumentClientException {

        PartitionKeyRange targetPartitionKeyRange = PartitionRoutingHelper.tryGetTargetRangeFromContinuationTokenRange(
                this.providedRanges,
                this.client.getPartitionKeyRangeCache(),
                this.collection.getSelfLink(),
                rangeFromContinuationToken);

        return targetPartitionKeyRange;
    }

    /**
     * Derive the ranges for the query of this execution context.
     * @return                          a boolean value indicating whether the population has been taken place or not
     * @throws DocumentClientException
     */
    private boolean populateProvidedRanges(DocumentServiceRequest request) throws DocumentClientException {

        if (this.providedRanges != null) {
            return false;
        }

        if (this.providedRanges == null && this.queryExecutionInfo != null) {
            this.providedRanges = this.queryExecutionInfo.getQueryRanges();
            return true;
        }

        String version = request.getHeaders().get(HttpConstants.HttpHeaders.VERSION);
        version = version == null || version.isEmpty() ? HttpConstants.Versions.CURRENT_VERSION : version;

        String enableCrossPartitionQueryHeader = request.getHeaders().get(HttpConstants.HttpHeaders.ENABLE_CROSS_PARTITION_QUERY);
        boolean enableCrossPartitionQuery = Boolean.parseBoolean(enableCrossPartitionQueryHeader);

        if (this.shouldExecuteQuery()) {
            this.providedRanges = PartitionRoutingHelper.getProvidedPartitionKeyRanges(
                    this.querySpec,
                    enableCrossPartitionQuery,
                    false,
                    this.collection.getPartitionKey(),
                    this.client.getQueryPartitionProvider(),
                    version);
        } else {
            this.providedRanges = new ArrayList<Range<String>>() {{
                add(new Range<>(
                        PartitionKeyInternalHelper.MinimumInclusiveEffectivePartitionKey,
                        PartitionKeyInternalHelper.MaximumExclusiveEffectivePartitionKey,
                        true,
                        false
                        ));
            }};
        }

        return true;
    }
}
