package com.microsoft.azure.documentdb.internal.directconnectivity;


import com.microsoft.azure.documentdb.DocumentClientException;
import com.microsoft.azure.documentdb.internal.BridgeInternal;
import com.microsoft.azure.documentdb.internal.DocumentServiceRequest;
import com.microsoft.azure.documentdb.internal.HttpConstants;
import com.microsoft.azure.documentdb.internal.RetryPolicy;
import org.apache.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Used internally to represents a retry policy for Gone exception in the Azure Cosmos DB database service.
 */
public class GoneAndRetryWithRetryPolicy implements RetryPolicy {
    private final static int WAIT_TIME_IN_SECONDS = 30;
    private final static int INITIALI_BACKOFF_SEONDS = 1;
    private final static int BACKOFF_MULTIPLIER = 2;

    private final Logger LOGGER = LoggerFactory.getLogger(GoneAndRetryWithRetryPolicy.class);

    private int attemptCount = 1;
    private DocumentClientException lastRetryWithException;
    private int currentBackOffSeconds = INITIALI_BACKOFF_SEONDS;
    private DocumentServiceRequest request;

    private long startTimeMilliSeconds;
    private long retryAfterSeconds;
    private boolean retryPartitionKeyRangeIdentity = true;

    public GoneAndRetryWithRetryPolicy(DocumentServiceRequest request) {
        this.request = request;
        this.startTimeMilliSeconds = System.currentTimeMillis();
    }

    @Override
    public boolean shouldRetry(DocumentClientException exception) throws DocumentClientException {
        if (exception.getStatusCode() != HttpStatus.SC_GONE && exception.getStatusCode() != HttpConstants.StatusCodes.RETRY_WITH) {
            return false;
        }

        Integer substatusCode = exception.getSubStatusCode();

	    if (substatusCode != null &&
	    substatusCode.intValue() == HttpConstants.SubStatusCodes.PARTITION_KEY_RANGE_GONE &&
        BridgeInternal.isUserProvidedPartitionKeyRangeIdentity(this.request)) {
            if (retryPartitionKeyRangeIdentity) {
                request.setForcePartitionKeyRangeRefresh(true);
                retryPartitionKeyRangeIdentity = false;
            } else {
			    LOGGER.warn("Received invalid partition id from user, retry won't happen, failing the request", exception);
			    return false;
		    }
	    }

        if (exception.getStatusCode() == HttpConstants.StatusCodes.RETRY_WITH) {
            this.lastRetryWithException = exception;
        }

        this.retryAfterSeconds = 0;
        long remainingSeconds = WAIT_TIME_IN_SECONDS - (int) ((System.currentTimeMillis() - this.startTimeMilliSeconds) * 1.0 / 1000.0);
        if (this.attemptCount++ > 1) {
            if (remainingSeconds <= 0) {
                if (exception.getStatusCode() == HttpStatus.SC_GONE) {
                    // Differentiate between Gone and InvalidPartition exceptions
                    String exceptionName = substatusCode != null
                            && substatusCode == HttpConstants.SubStatusCodes.NAME_CACHE_IS_STALE
                            ? "invalid partition exception"
                            : "gone exception";
                    if (this.lastRetryWithException != null) {
                        LOGGER.warn("Received {} after backoff/retry including at least one " +
                                "RetryWithException. Will fail the request with RetryWithException", exceptionName, this.lastRetryWithException);
                        throw this.lastRetryWithException;
                    } else {
                        LOGGER.warn("Received {} after backoff/retry. Will fail the request", exceptionName, exception);
                        throw new DocumentClientException(HttpStatus.SC_SERVICE_UNAVAILABLE, exception);
                    }
                } else {
                    LOGGER.warn("Received retryWith exception after backoff/retry. Will fail the request", exception);
                }

                return false;
            }

            this.retryAfterSeconds = Math.min(this.currentBackOffSeconds, remainingSeconds);
            this.currentBackOffSeconds *= BACKOFF_MULTIPLIER;
        }

        if (exception.getStatusCode() == HttpStatus.SC_GONE) {

            if (substatusCode != null && substatusCode.intValue() == HttpConstants.SubStatusCodes.NAME_CACHE_IS_STALE) {
                request.setQuorumSelectedLSN(-1);
                request.setQuorumSelectedStoreResponse(null);
                request.setForceNameCacheRefresh(true);
                LOGGER.debug("Received Gone exception with sub status name_cach_ is_stale "+substatusCode.intValue()+", will retry with name cache refresh");
            } else if (substatusCode != null &&
                      (substatusCode.intValue() == HttpConstants.SubStatusCodes.COMPLETING_SPLIT ||
                       substatusCode.intValue() == HttpConstants.SubStatusCodes.COMPLETING_PARTITION_MIGRATION ||
                       substatusCode.intValue() == HttpConstants.SubStatusCodes.PARTITION_KEY_RANGE_GONE)) {
                this.request.setForcePartitionKeyRangeRefresh(true);
                LOGGER.debug("Received Gone exception with sub status "+substatusCode.intValue()+", will retry with partition key range cache refresh");
            }

            request.setForceAddressRefresh(true);
            LOGGER.debug("Received Gone exception, will retry with address cache refresh");
        } else {
            LOGGER.debug("Received retryWith exception, will retry", exception);
        }

        return true;
    }

    @Override
    public long getRetryAfterInMilliseconds() {
        return this.retryAfterSeconds * 1000;
    }
}
