package com.microsoft.azure.documentdb.internal.directconnectivity;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;

import org.apache.http.client.HttpClient;

import com.microsoft.azure.documentdb.ConnectionPolicy;
import com.microsoft.azure.documentdb.DocumentClient;
import com.microsoft.azure.documentdb.internal.AuthorizationTokenProvider;
import com.microsoft.azure.documentdb.internal.DocumentServiceRequest;
import com.microsoft.azure.documentdb.internal.EndpointManager;
import com.microsoft.azure.documentdb.internal.UserAgentContainer;
import com.microsoft.azure.documentdb.internal.routing.ClientCollectionCache;
import com.microsoft.azure.documentdb.internal.routing.PartitionKeyRangeCache;

/**
 * Used internally to maintain a region endpoint to GatewayAddressCache mapping,
 * in the Azure Cosmos DB database service.
 */
public class GlobalAddressResolver {

    private ConnectionPolicy connectionPolicy;
    ClientCollectionCache collectionCache;
    PartitionKeyRangeCache partitionKeyRangeCache;
    UserAgentContainer userAgentContainer;
    AuthorizationTokenProvider authorizationTokenProvider;
    HttpClient httpClient;
    EndpointManager endpointManager;
    DocumentClient documentClient;
    ExecutorService executorService;
    private ConcurrentHashMap<String, AddressCache> addressCaches;

    public GlobalAddressResolver(String serviceEndpoint,
            ConcurrentHashMap<String,
            AddressCache> addressCaches,
            ConnectionPolicy connectionPolicy,
            ClientCollectionCache collectionCache,
            PartitionKeyRangeCache partitionKeyRangeCache,
            UserAgentContainer userAgentContainer,
            AuthorizationTokenProvider authorizationTokenProvider,
            HttpClient httpClient,
            EndpointManager endpointManager,
            DocumentClient documentClient,
            ExecutorService executorService) {
        this.connectionPolicy = connectionPolicy;
        this.collectionCache = collectionCache;
        this.partitionKeyRangeCache = partitionKeyRangeCache;
        this.userAgentContainer = userAgentContainer;
        this.authorizationTokenProvider = authorizationTokenProvider;
        this.httpClient = httpClient;
        this.endpointManager = endpointManager;
        this.documentClient = documentClient;
        this.executorService = executorService;
        this.addressCaches = addressCaches;
        if (this.addressCaches == null) {
            this.addressCaches = new ConcurrentHashMap<String, AddressCache>();
            this.addressCaches.put(serviceEndpoint,
                    new GatewayAddressCache(serviceEndpoint,
                            connectionPolicy,
                            collectionCache,
                            partitionKeyRangeCache,
                            userAgentContainer,
                            authorizationTokenProvider,
                            httpClient,
                            endpointManager,
                            documentClient,
                            executorService));
        }
    }

    public AddressCache resolve(DocumentServiceRequest request) {
        String endpoint = request.getLocationEndpointToRoute().toString();
        AddressCache newAddressCache = new GatewayAddressCache(endpoint,
                this.connectionPolicy,
                this.collectionCache,
                this.partitionKeyRangeCache,
                this.userAgentContainer,
                this.authorizationTokenProvider,
                this.httpClient,
                this.endpointManager,
                this.documentClient,
                this.executorService);
        AddressCache existingAddressCache = this.addressCaches.putIfAbsent(endpoint, newAddressCache);
        if (existingAddressCache == null) {
            existingAddressCache = newAddressCache;
        }
        return existingAddressCache;
    }

    public ConcurrentHashMap<String, AddressCache> getAddressCaches() {
        return this.addressCaches;
    }
}
