package com.microsoft.azure.documentdb.internal.directconnectivity;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.message.BasicNameValuePair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.microsoft.azure.documentdb.ConnectionPolicy;
import com.microsoft.azure.documentdb.DocumentClient;
import com.microsoft.azure.documentdb.DocumentClientException;
import com.microsoft.azure.documentdb.DocumentCollection;
import com.microsoft.azure.documentdb.Error;
import com.microsoft.azure.documentdb.PartitionKeyRange;
import com.microsoft.azure.documentdb.internal.AsyncCache;
import com.microsoft.azure.documentdb.internal.AuthorizationTokenProvider;
import com.microsoft.azure.documentdb.internal.Constants;
import com.microsoft.azure.documentdb.internal.DocumentServiceRequest;
import com.microsoft.azure.documentdb.internal.DocumentServiceResponse;
import com.microsoft.azure.documentdb.internal.EndpointManager;
import com.microsoft.azure.documentdb.internal.ErrorUtils;
import com.microsoft.azure.documentdb.internal.HttpConstants;
import com.microsoft.azure.documentdb.internal.Paths;
import com.microsoft.azure.documentdb.internal.PathsHelper;
import com.microsoft.azure.documentdb.internal.ResourceType;
import com.microsoft.azure.documentdb.internal.RetryRequestDelegate;
import com.microsoft.azure.documentdb.internal.RetryUtility;
import com.microsoft.azure.documentdb.internal.UserAgentContainer;
import com.microsoft.azure.documentdb.internal.Utils;
import com.microsoft.azure.documentdb.internal.routing.ClientCollectionCache;
import com.microsoft.azure.documentdb.internal.routing.CollectionCache;
import com.microsoft.azure.documentdb.internal.routing.PartitionKeyInternal;
import com.microsoft.azure.documentdb.internal.routing.PartitionKeyInternalHelper;
import com.microsoft.azure.documentdb.internal.routing.PartitionKeyRangeIdentity;
import com.microsoft.azure.documentdb.internal.routing.RoutingMapProvider;

/**
 * Used internally to cache the physical addresses of the collections' partitions in the Azure Cosmos DB database service.
 */
public class GatewayAddressCache extends AddressCache {
    private final static Logger LOGGER = LoggerFactory.getLogger(GatewayAddressCache.class);
    private final static String PROTOCOL_HTTPS = "https";
    private final static String PROTOCOL_FILTER_FORMAT = "%1$s eq %2$s";
    private final Logger logger = LoggerFactory.getLogger(GatewayAddressCache.class);
    private AsyncCache<PartitionKeyRangeIdentity, AddressInformation[]> serverPartitionAddressCache;
    private String protocolFilter;
    private HttpClient httpClient;
    private List<NameValuePair> defaultHeaders;
    private String addressEndpoint;
    private AuthorizationTokenProvider authorizationTokenProvider;
    private final CollectionCache collectionCache;
    private RoutingMapProvider partitionKeyRangeCache;
    private ImmutablePair<String, AddressInformation[]> masterPartitionAddressCache;
    private HttpHost proxy;
    private final EndpointManager globalEndpointManager;
    private final DocumentClient documentClient;

    public GatewayAddressCache(
            String serviceEndpoint,
            ConnectionPolicy connectionPolicy,
            CollectionCache collectionCache,
            RoutingMapProvider partitionKeyRangeCache,
            UserAgentContainer userAgent,
            AuthorizationTokenProvider authorizationTokenProvider,
            HttpClient httpClient,
            EndpointManager globalEndpointManager,
            DocumentClient documentClient,
            ExecutorService executorService) {
        this.serverPartitionAddressCache = new AsyncCache<>(executorService);
        this.authorizationTokenProvider = authorizationTokenProvider;
        this.collectionCache = collectionCache;
        this.partitionKeyRangeCache = partitionKeyRangeCache;

        this.httpClient = httpClient;

        this.proxy = connectionPolicy.getProxy();

        if (userAgent == null) {
            userAgent = new UserAgentContainer();
        }

        this.protocolFilter = String.format(PROTOCOL_FILTER_FORMAT, Constants.Properties.PROTOCOL, PROTOCOL_HTTPS);
        this.defaultHeaders = new LinkedList<NameValuePair>();
        this.defaultHeaders
        .add(new BasicNameValuePair(HttpConstants.HttpHeaders.VERSION, HttpConstants.Versions.CURRENT_VERSION));
        this.defaultHeaders
        .add(new BasicNameValuePair(HttpConstants.HttpHeaders.USER_AGENT, userAgent.getUserAgent()));
        retargetEndpoint(serviceEndpoint);
        this.globalEndpointManager = globalEndpointManager;
        this.documentClient = documentClient;
    }

    public void retargetEndpoint(String serviceEndpoint) {
        int portStartIndex = serviceEndpoint.lastIndexOf(":");
        String host = serviceEndpoint.substring(0, portStartIndex);
        String hostSuffix = serviceEndpoint.substring(portStartIndex + 1);
        int portEndIndex = hostSuffix.indexOf("/");
        String port = portEndIndex == -1? hostSuffix : hostSuffix.substring(0, portEndIndex);
        
        this.addressEndpoint = host + ":" + port + "//" + Paths.ADDRESS_PATH_SEGMENT;
        this.serverPartitionAddressCache.clear();
        this.masterPartitionAddressCache = null;
    }

    @Override
    public AddressInformation[] resolve(DocumentServiceRequest request) throws DocumentClientException {
        boolean isMasterResource = request.isReadingFromMaster();
        return isMasterResource ? this.resolveMaster(request) : this.resolveServer(request);
    }

    private AddressInformation[] resolveMaster(DocumentServiceRequest request) throws DocumentClientException {
        if (request.isForceNameCacheRefresh() || this.masterPartitionAddressCache == null) {
            List<Address> response = this.resolveAddressesViaGatewayAsync(
                    request,
                    null,
                    request.isForceAddressRefresh());

            this.masterPartitionAddressCache = this.toPartitionAddressAndRange(response);
        }

        PartitionKeyRange partitionKeyRange = new PartitionKeyRange();
        partitionKeyRange.setId(PartitionKeyRange.MASTER_PARTITION_KEY_RANGE_ID);
        request.setResolvedPartitionKeyRange(partitionKeyRange);

        request.setResolvedCollectionRid(this.masterPartitionAddressCache.getLeft());
        return this.masterPartitionAddressCache.getRight();
    }

    private AddressInformation[] resolveServer(DocumentServiceRequest request) throws DocumentClientException {

        DocumentCollection collection = this.collectionCache.resolveCollection(request);
        String partitionKeyRangeId = null;
        PartitionKeyRange partitionKeyRange = null;

        if (request.isForceAddressRefresh() || request.getResolvedPartitionKeyRange() == null) {
            // if the partition key range id is already resolved skips this

            if (request.getHeaders().get(HttpConstants.HttpHeaders.PARTITION_KEY) != null) {
                partitionKeyRange = GatewayAddressCache.tryResolveServerPartitionByPartitionKey(this.partitionKeyRangeCache,
                        request.getHeaders().get(HttpConstants.HttpHeaders.PARTITION_KEY),
                        collection,
                        request.isForcePartitionKeyRangeRefresh());

                if(partitionKeyRange != null)
                {
                    partitionKeyRangeId = partitionKeyRange.getId();
                }
            } else if (request.getPartitionKeyRangeIdentity() != null) {
                partitionKeyRangeId = request.getPartitionKeyRangeIdentity().getPartitionKeyRangeId();
                partitionKeyRange = partitionKeyRangeCache.getPartitionKeyRangeById(
                collection.getSelfLink(), partitionKeyRangeId, request.isForcePartitionKeyRangeRefresh());
            }

            if (partitionKeyRangeId == null) {
                this.logger.warn("Request contains neither partition key nor partition range Id.");
                throw new IllegalStateException();
            }

            this.logger.debug("request.isForcePartitionKeyRangeRefresh={}, partitionKeyRange={}",
                    request.isForcePartitionKeyRangeRefresh(), partitionKeyRange);

            if (partitionKeyRange == null) {
                Map<String, String> responseHeaders = new HashMap<String, String>();
                responseHeaders.put(HttpConstants.HttpHeaders.SUB_STATUS,
                        String.valueOf(HttpConstants.SubStatusCodes.PARTITION_KEY_RANGE_GONE));
                this.logger.debug("Invalid partition key range Id '{}'", partitionKeyRangeId);
                throw new DocumentClientException(HttpConstants.StatusCodes.GONE,
                        new Error("{ 'message': 'Invalid partition key range' }"), responseHeaders);
            }

            request.setResolvedPartitionKeyRange(partitionKeyRange);
        }

        if (request.getIsNameBased()) {
            request.getHeaders().put(WFConstants.BackendHeaders.COLLECTION_RID, collection.getResourceId());
        }
        
        return resolveAddressesForRangeId(request, collection,
                request.getResolvedPartitionKeyRange().getId());
    }

    private AddressInformation[] resolveAddressesForRangeId(
            final DocumentServiceRequest request,
            final DocumentCollection collection,
            final String partitionKeyRangeId) throws DocumentClientException {

        PartitionKeyRangeIdentity partitionKeyRangeIdentity = new PartitionKeyRangeIdentity(
                collection.getResourceId(), partitionKeyRangeId);
        final GatewayAddressCache that = this;
        if (request.isForceAddressRefresh()) {
            this.serverPartitionAddressCache.refresh(partitionKeyRangeIdentity, new Callable<AddressInformation[]>() {
                @Override
                public AddressInformation[] call() throws Exception {
                    return that.queryAddressesViaGateway(
                            request,
                            collection,
                            partitionKeyRangeId,
                            true
                    );
                }
            });
        }

        try {
            LOGGER.trace("Resolving address for range Id {}, {}, obsoleteValue: {}",
                    partitionKeyRangeIdentity.getCollectionRid(),
                    partitionKeyRangeIdentity.getPartitionKeyRangeId(),
                    null);
            return this.serverPartitionAddressCache.get(partitionKeyRangeIdentity, null, new Callable<AddressInformation[]>() {
                @Override
                public AddressInformation[] call() throws Exception {
                    return that.queryAddressesViaGateway(
                            request,
                            collection,
                            partitionKeyRangeId,
                            false
                    );
                }
            }).get();
        } catch (InterruptedException|ExecutionException e) {
            if (e.getCause() instanceof DocumentClientException) {
                throw (DocumentClientException) e.getCause();
            }
            throw new IllegalStateException("Failed to get address from cache", e);
        }
    }

    private AddressInformation[] queryAddressesViaGateway(
            DocumentServiceRequest request,
            DocumentCollection collection,
            String partitionKeyRangeId,
            boolean forceRefresh) throws DocumentClientException {

        List<Address> addresses = this.resolveAddressesViaGatewayAsync(
                request,
                new String[] { partitionKeyRangeId },
                forceRefresh);

        HashMap<String, List<Address>> addressesByPartitionRangeId = new HashMap<>();
        for (Address address : addresses) {
            String partitionRangeId = address.getParitionKeyRangeId();
            if (!addressesByPartitionRangeId.containsKey(partitionRangeId)) {
                addressesByPartitionRangeId.put(partitionRangeId, new ArrayList<Address>());
            }
            addressesByPartitionRangeId.get(partitionRangeId).add(address);
        }

        ArrayList<ImmutablePair<String, AddressInformation[]>> addressInfos = new ArrayList<>();
        for (Map.Entry<String, List<Address>> entry : addressesByPartitionRangeId.entrySet()) {
            addressInfos.add(this.toPartitionAddressAndRange(entry.getValue()));
        }

        ImmutablePair<String, AddressInformation[]> result = null;
        for (ImmutablePair<String, AddressInformation[]> addressInfo : addressInfos) {
            if (addressInfo.getLeft().equals(partitionKeyRangeId)) {
                result = addressInfo;
                break;
            }
        }

        if (result == null) {
            String errorMessage = String.format("PartitionKeyRange with id %s in collection %s doesn't exist.",
                    partitionKeyRangeId,
                    collection.getSelfLink());
            logger.debug(errorMessage);
            Map<String, String> responseHeaders = new HashMap<String, String>();
            responseHeaders.put(HttpConstants.HttpHeaders.SUB_STATUS,
            String.valueOf(HttpConstants.SubStatusCodes.PARTITION_KEY_RANGE_GONE));
            this.logger.debug("Invalid partition key range Id '{}'", partitionKeyRangeId);
            throw new DocumentClientException(HttpConstants.StatusCodes.GONE,
            new Error("{ 'message': 'Invalid partition key range' }"), responseHeaders);
        }

        return result.getRight();
    }

    public static PartitionKeyRange tryResolveServerPartitionByPartitionKey(
            RoutingMapProvider partitionKeyRangeCache,
            String partitionKeyString,
            DocumentCollection collection,
            boolean forcePartitionKeyRangeRefresh) {
        PartitionKeyInternal partitionKey = null;
        try {
            partitionKey = Utils.getSimpleObjectMapper().readValue(
                    partitionKeyString,
                    PartitionKeyInternal.class);
        } catch (IOException e) {
            throw new IllegalStateException("Unable to deserialize PartitionKeyInternal due to I/O error");
        }

        if (partitionKey.getComponents().size() == collection.getPartitionKey().getPaths().size()) {

            String effectivePartitionKey = PartitionKeyInternalHelper.getEffectivePartitionKeyString(partitionKey,collection.getPartitionKey(), true);

            // There should be exactly one range which contains a partition key. Always.
            Collection<PartitionKeyRange> ranges = partitionKeyRangeCache.getOverlappingRanges(collection.getSelfLink(),
                    com.microsoft.azure.documentdb.internal.routing.Range.getPointRange(effectivePartitionKey), forcePartitionKeyRangeRefresh);

            assert ranges.size() <= 1 : "There should be exactly one range which contains a partition key. Always.";
            
            return ranges.size() > 0 ? ranges.iterator().next() : null;
        }

        return null;
    }

    private DocumentServiceResponse getPartitionAddresses(DocumentServiceRequest request,
                                                          String[] partitionKeyRangeIds,
                                                          boolean forceRefresh) throws DocumentClientException {
        String entryUrl = PathsHelper.generatePath(
                request.getResourceType(),
                request,
                Utils.isFeedRequest(request.getOperationType()));

        ResourceType resourceType = request.getResourceType();
        List<NameValuePair> addressQuery = new LinkedList<NameValuePair>();

        try {
            addressQuery.add(new BasicNameValuePair(
                    HttpConstants.QueryStrings.URL,
                    URLEncoder.encode(entryUrl, "UTF-8")));
            addressQuery.add(new BasicNameValuePair(
                    HttpConstants.QueryStrings.FILTER,
                    new URI(null, null, null, this.protocolFilter, null).toString().substring(1)));

            if (partitionKeyRangeIds != null && partitionKeyRangeIds.length > 0) {
                addressQuery.add(new BasicNameValuePair(
                        HttpConstants.QueryStrings.PARTITION_KEY_RANGE_IDS,
                        StringUtils.join(partitionKeyRangeIds, ",")
                ));
            }
        } catch (UnsupportedEncodingException e1) {
            this.logger.warn(e1.toString(), e1);
        } catch (URISyntaxException e) {
            this.logger.warn(e.toString(), e);
        }

        URL targetEndpoint = Utils.setQuery(this.addressEndpoint, Utils.createQuery(addressQuery));

        HttpGet httpGet = new HttpGet(targetEndpoint.toString());
        Map<String, String> headers = new HashMap<String, String>();

        for (NameValuePair nameValuePair : this.defaultHeaders) {
            httpGet.addHeader(nameValuePair.getName(), nameValuePair.getValue());
            headers.put(nameValuePair.getName(), nameValuePair.getValue());
        }

        if (forceRefresh) {
            httpGet.addHeader(HttpConstants.HttpHeaders.FORCE_REFRESH, Boolean.TRUE.toString());
            headers.put(HttpConstants.HttpHeaders.FORCE_REFRESH, Boolean.TRUE.toString());
        }

        if (request.isReadingFromMaster()) {
            httpGet.addHeader(HttpConstants.HttpHeaders.USE_MASTER_COLLECTION_RESOLVER, Boolean.TRUE.toString());
            headers.put(HttpConstants.HttpHeaders.USE_MASTER_COLLECTION_RESOLVER, Boolean.TRUE.toString());
        }

        String xDate = Utils.getCurrentTimeGMT();
        httpGet.addHeader(HttpConstants.HttpHeaders.X_DATE, xDate);
        headers.put(HttpConstants.HttpHeaders.X_DATE, xDate);

        String token = null;
        if (this.authorizationTokenProvider.getMasterKey() != null) {
            if (!request.getIsNameBased()) {
                token = this.authorizationTokenProvider.generateKeyAuthorizationSignature("get", request.getResourceAddress().toLowerCase(),
                        resourceType, headers);
            } else {
                token = this.authorizationTokenProvider.generateKeyAuthorizationSignature("get", request.getResourceFullName(),
                        resourceType, headers);
            }
        } else if (this.authorizationTokenProvider.getResourceTokens() != null) {
            if (!request.getIsNameBased()) {
                token = this.authorizationTokenProvider.getAuthorizationTokenUsingResourceTokens(request.getPath(),
                        request.getResourceAddress().toLowerCase());
                } else {
                token = this.authorizationTokenProvider.getAuthorizationTokenUsingResourceTokens(request.getPath(),
                        request.getResourceFullName());
            }
        }


        try {
            httpGet.addHeader(HttpConstants.HttpHeaders.AUTHORIZATION, URLEncoder.encode(token, "UTF-8"));
        } catch (UnsupportedEncodingException e) {
            throw new IllegalStateException("Unsupported encoding", e);
        }

        // Add proxy
        if (this.proxy != null) {
            RequestConfig requestConfig = RequestConfig.custom().setProxy(this.proxy).build();
            httpGet.setConfig(requestConfig);
        }

        HttpResponse httpResponse;
        String identifier;
        try {
            identifier = logAddressResolutionStart(request, targetEndpoint.toURI());
            httpResponse = this.httpClient.execute(httpGet);
            ErrorUtils.maybeThrowException(targetEndpoint.getPath(), httpResponse, true, this.logger);
            DocumentServiceResponse documentServiceResponse = new DocumentServiceResponse(httpResponse, request.getIsMedia(), request.getClientSideRequestStatistics());
            logAddressResolutionEnd(request, identifier);
            return documentServiceResponse;
        } catch (IOException e) {
            httpGet.releaseConnection();
            throw new DocumentClientException(HttpConstants.StatusCodes.FORBIDDEN, e);
        } catch (URISyntaxException e) {
            throw new DocumentClientException(HttpStatus.SC_INTERNAL_SERVER_ERROR, e);
        } catch (DocumentClientException e) {
            throw e;
        } catch (Exception e) {
            httpGet.releaseConnection();
            throw e;
        }
    }

    private ImmutablePair<String, AddressInformation[]> toPartitionAddressAndRange(List<Address> addresses) {
        // The addressList should always contains at least one address
        List<AddressInformation> addressInfos = new ArrayList<>();
        for (Address address : addresses) {
            addressInfos.add(new AddressInformation(true, address.IsPrimary(), address.getPhyicalUri()));
        }

        AddressInformation[] addressInfoArr = new AddressInformation[addressInfos.size()];
        return new ImmutablePair<>(addresses.get(0).getParitionKeyRangeId(), addressInfos.toArray(addressInfoArr));
    }

    private List<Address> resolveAddressesViaGatewayAsync(
            final DocumentServiceRequest request,
            final String[] partitionKeyRangeIds,
            final boolean forceRefresh) throws DocumentClientException {

        RetryRequestDelegate readDelegate = new RetryRequestDelegate() {

            @Override
            public DocumentServiceResponse apply(DocumentServiceRequest requestInner) throws DocumentClientException {
                return getPartitionAddresses(request, partitionKeyRangeIds, forceRefresh);
            }
        };

        DocumentServiceResponse response = RetryUtility.executeDocumentClientRequest(
                readDelegate, this.documentClient, this.globalEndpointManager, request, (ClientCollectionCache) collectionCache);

        List<Address> addresses = response.getQueryResponse(Address.class);

        return addresses;
    }

    private static String logAddressResolutionStart(DocumentServiceRequest request, URI targetEndpoint) {
        if (request.getClientSideRequestStatistics() != null) {
            return request.getClientSideRequestStatistics().recordAddressResolutionStart(targetEndpoint);
        }

        return null;
    }

    private static void logAddressResolutionEnd(DocumentServiceRequest request, String identifier) {
        if (request.getClientSideRequestStatistics() != null) {
            request.getClientSideRequestStatistics().recordAddressResolutionEnd(identifier);
        }
    }
}
