/*
 * Copyright (c) Microsoft Corporation.  All rights reserved.
 */

package com.microsoft.azure.documentdb.internal.directconnectivity;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

import com.microsoft.azure.documentdb.internal.AuthorizationTokenProvider;
import com.microsoft.azure.documentdb.internal.DocumentServiceRequest;
import com.microsoft.azure.documentdb.internal.HttpConstants;
import com.microsoft.azure.documentdb.internal.OperationType;
import com.microsoft.azure.documentdb.internal.PathsHelper;
import com.microsoft.azure.documentdb.internal.ResourceId;
import com.microsoft.azure.documentdb.internal.ResourceType;
import com.microsoft.azure.documentdb.internal.Utils;

class BarrierRequestHelper {
    static DocumentServiceRequest create(DocumentServiceRequest request, AuthorizationTokenProvider authorizationTokenProvider) {
        DocumentServiceRequest barrierRequest;

        if(request.isReadingFromMaster() || request.isWritingToMaster()) {
            //DB Feed
            barrierRequest = DocumentServiceRequest.create(OperationType.HeadFeed, null, ResourceType.Database, null);
        } else if (request.getIsNameBased()) {
            // Name based server request
            String collectionLink = PathsHelper.getCollectionPath(request.getResourceAddress());
            barrierRequest = DocumentServiceRequest.create(OperationType.Head, ResourceType.DocumentCollection, collectionLink,
                    null);
        } else {
            // RID based Server requestI
            barrierRequest = DocumentServiceRequest.create(
                    OperationType.Head,
                    ResourceId.parse(request.getResourceId()).getDocumentCollectionId().toString(),
                    ResourceType.DocumentCollection,
                    null);
        }

        String xDate = Utils.getCurrentTimeGMT();
        barrierRequest.getHeaders().put(HttpConstants.HttpHeaders.X_DATE, xDate);

        String token = null;
        if (authorizationTokenProvider.getMasterKey() != null) {
            token = authorizationTokenProvider.generateKeyAuthorizationSignature(HttpConstants.HttpMethods.HEAD, barrierRequest.getResourceAddress(),
                barrierRequest.getResourceType(), barrierRequest.getHeaders());
        } else if (authorizationTokenProvider.getResourceTokens() != null) {
            token = authorizationTokenProvider.getAuthorizationTokenUsingResourceTokens(barrierRequest.getPath(), barrierRequest.getResourceAddress());
        }

        try {
            barrierRequest.getHeaders().put(HttpConstants.HttpHeaders.AUTHORIZATION, URLEncoder.encode(token, "UTF-8"));
        } catch (UnsupportedEncodingException e) {
            throw new IllegalStateException("Unsupported encoding", e);
        }

        barrierRequest.setForceAddressRefresh(request.isForceAddressRefresh());
        barrierRequest.setRequestChargeTracker(request.getRequestChargeTracker());
        barrierRequest.setEndpointOverride(request.getEndpointOverride());
        barrierRequest.setLocationEndpointToRoute(request.getLocationEndpointToRoute());
        barrierRequest.setShouldClearSessionTokenOnSessionReadFailure(request.shouldClearSessionTokenOnSessionReadFailure());

        if (request.getPartitionKeyRangeIdentity() != null) {
            barrierRequest.routeTo(request.getPartitionKeyRangeIdentity());
        }
        String partitionKey = request.getHeaders().get(HttpConstants.HttpHeaders.PARTITION_KEY);
        if (partitionKey != null) {
            barrierRequest.getHeaders().put(HttpConstants.HttpHeaders.PARTITION_KEY, partitionKey);
        }

        if (request.getClientSideRequestStatistics() != null) {
            barrierRequest.setClientSideRequestStatistics(request.getClientSideRequestStatistics());
        }

        return barrierRequest;
    }
}
