package com.microsoft.azure.documentdb.internal;

import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.List;
import java.util.Locale;
import java.util.UUID;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.NameValuePair;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.uuid.EthernetAddress;
import com.fasterxml.uuid.Generators;
import com.fasterxml.uuid.impl.RandomBasedGenerator;
import com.fasterxml.uuid.impl.TimeBasedGenerator;
import com.microsoft.azure.documentdb.ConsistencyLevel;
import com.microsoft.azure.documentdb.DocumentCollection;

/**
 * Used internally to provides utility functions for the Azure Cosmos DB database service Java SDK.
 */
public class Utils {
    private static final ObjectMapper simpleObjectMapper = new ObjectMapper();
    private static RandomBasedGenerator RD_GENERATOR = Generators.randomBasedGenerator();
    private static TimeBasedGenerator TB_GENERATOR = Generators.timeBasedGenerator(EthernetAddress.constructMulticastAddress());
    private static DateTimeFormatter dateTimeFormatGMT;

    static {
        Utils.simpleObjectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        Utils.dateTimeFormatGMT = DateTimeFormat.forPattern("EEE, dd MMM yyyy HH:mm:ss zzz").withLocale(Locale.US);
    }

    public static String encodeBase64String(byte[] binaryData) {
        String encodedString = Base64.encodeBase64String(binaryData);

        if (encodedString.endsWith("\r\n")) {
            encodedString = encodedString.substring(0, encodedString.length() - 2);
        }
        return encodedString;
    }

    /**
     * Checks whether the specified link is Name based or not
     *
     * @param link the link to analyze.
     * @return true or false
     */
    public static boolean isNameBased(String link) {
        if (StringUtils.isEmpty(link)) {
            return false;
        }

        // trimming the leading "/"
        if (link.startsWith("/") && link.length() > 1) {
            link = link.substring(1);
        }

        // Splitting the link(separated by "/") into parts
        String[] parts = StringUtils.split(link, '/');

        // First part should be "dbs"
        if (parts.length == 0 || StringUtils.isEmpty(parts[0])
                || !parts[0].equalsIgnoreCase(Paths.DATABASES_PATH_SEGMENT)) {
            return false;
        }

        // The second part is the database id(ResourceID or Name) and cannot be
        // empty
        if (parts.length < 2 || StringUtils.isEmpty(parts[1])) {
            return false;
        }

        // Either ResourceID or database name
        String databaseID = parts[1];

        // Length of databaseID(in case of ResourceID) is always 8
        if (databaseID.length() != 8) {
            return true;
        }

        // Decoding the databaseID
        byte[] buffer = ResourceId.fromBase64String(databaseID);

        // Length of decoded buffer(in case of ResourceID) is always 4
        if (buffer.length != 4) {
            return true;
        }

        return false;
    }

    /**
     * Checks whether the specified link is a Database Self Link or a Database
     * ID based link
     *
     * @param link the link to analyze.
     * @return true or false
     */
    public static boolean isDatabaseLink(String link) {
        if (StringUtils.isEmpty(link)) {
            return false;
        }

        // trimming the leading and trailing "/" from the input string
        link = trimBeginingAndEndingSlashes(link);

        // Splitting the link(separated by "/") into parts
        String[] parts = StringUtils.split(link, '/');

        if (parts.length != 2) {
            return false;
        }

        // First part should be "dbs"
        if (StringUtils.isEmpty(parts[0]) || !parts[0].equalsIgnoreCase(Paths.DATABASES_PATH_SEGMENT)) {
            return false;
        }

        // The second part is the database id(ResourceID or Name) and cannot be
        // empty
        if (StringUtils.isEmpty(parts[1])) {
            return false;
        }

        return true;
    }

    /**
     * Checks whether the specified path segment is a resource type
     *
     * @param resourcePathSegment the path segment to analyze.
     * @return true or false
     */
    public static boolean IsResourceType(String resourcePathSegment) {
        if (StringUtils.isEmpty(resourcePathSegment)) {
            return false;
        }

        switch (resourcePathSegment.toLowerCase()) {
        case Paths.ATTACHMENTS_PATH_SEGMENT:
        case Paths.COLLECTIONS_PATH_SEGMENT:
        case Paths.DATABASES_PATH_SEGMENT:
        case Paths.PERMISSIONS_PATH_SEGMENT:
        case Paths.USERS_PATH_SEGMENT:
        case Paths.DOCUMENTS_PATH_SEGMENT:
        case Paths.STORED_PROCEDURES_PATH_SEGMENT:
        case Paths.TRIGGERS_PATH_SEGMENT:
        case Paths.USER_DEFINED_FUNCTIONS_PATH_SEGMENT:
        case Paths.CONFLICTS_PATH_SEGMENT:
        case Paths.PARTITION_KEY_RANGE_PATH_SEGMENT:
            return true;

        default:
            return false;
        }
    }

    /**
     * Joins the specified paths by appropriately padding them with '/'
     *
     * @param path1 the first path segment to join.
     * @param path2 the second path segment to join.
     * @return the concatenated path with '/'
     */
    public static String joinPath(String path1, String path2) {
        path1 = trimBeginingAndEndingSlashes(path1);
        String result = "/" + path1 + "/";

        if (!StringUtils.isEmpty(path2)) {
            path2 = trimBeginingAndEndingSlashes(path2);
            result += path2 + "/";
        }

        return result;
    }

    /**
     * Trims the beginning and ending '/' from the given path
     *
     * @param path the path to trim for beginning and ending slashes
     * @return the path without beginning and ending '/'
     */
    public static String trimBeginingAndEndingSlashes(String path) {
        if(path == null) {
            return null;
        }

        if (path.startsWith("/")) {
            path = path.substring(1);
        }

        if (path.endsWith("/")) {
            path = path.substring(0, path.length() - 1);
        }

        return path;
    }

    public static URL setQuery(String urlString, String query) {

        if (urlString == null)
            throw new IllegalStateException("urlString parameter can't be null.");
        query = Utils.removeLeadingQuestionMark(query);
        try {
            if (query != null && !query.isEmpty()) {
                return new URI(Utils.addTrailingSlash(urlString) + RuntimeConstants.Separators.Query[0] + query)
                        .toURL();
            } else {
                return new URI(Utils.addTrailingSlash(urlString)).toURL();
            }
        } catch (MalformedURLException e) {
            throw new IllegalStateException("Uri is invalid: ", e);
        } catch (URISyntaxException e) {
            throw new IllegalStateException("Uri is invalid: ", e);
        }
    }

    public static String createQuery(List<NameValuePair> queryParameters) {
        if (queryParameters == null)
            return "";

        StringBuilder queryString = new StringBuilder();

        for (NameValuePair nameValuePair : queryParameters) {
            String key = nameValuePair.getName();
            String value = nameValuePair.getValue();

            if (key != null && !key.isEmpty()) {
                if (queryString.length() > 0) {
                    queryString.append(RuntimeConstants.Separators.Query[1]);
                }

                queryString.append(key);

                if (value != null) {
                    queryString.append(RuntimeConstants.Separators.Query[2]);
                    queryString.append(value);
                }
            }
        }

        return queryString.toString();
    }

    /**
     * Given the full path to a resource, extract the collection path.
     *
     * @param resourceFullName the full path to the resource.
     * @return the path of the collection in which the resource is.
     */
    public static String getCollectionName(String resourceFullName) {
        if (resourceFullName != null) {
            resourceFullName = Utils.trimBeginingAndEndingSlashes(resourceFullName);

            int slashCount = 0;
            for (int i = 0; i < resourceFullName.length(); i++) {
                if (resourceFullName.charAt(i) == '/') {
                    slashCount++;
                    if (slashCount == 4) {
                        return resourceFullName.substring(0, i);
                    }
                }
            }
        }
        return resourceFullName;
    }

    public static Boolean isCollectionPartitioned(DocumentCollection collection) {
        if (collection == null) {
            throw new IllegalArgumentException("collection");
        }

        return collection.getPartitionKey() != null
                && collection.getPartitionKey().getPaths() != null
                && collection.getPartitionKey().getPaths().size() > 0;
    }

    public static boolean isCollectionChild(ResourceType type) {
        return type == ResourceType.Document ||
                type == ResourceType.Attachment ||
                type == ResourceType.Conflict ||
                type == ResourceType.StoredProcedure ||
                type == ResourceType.Trigger ||
                type == ResourceType.UserDefinedFunction;
    }

    public static boolean isWriteOperation(OperationType operationType) {
        return operationType == OperationType.Create ||
                operationType == OperationType.Upsert ||
                operationType == OperationType.Delete ||
                operationType == OperationType.Replace ||
                operationType == OperationType.ExecuteJavaScript;
    }

    public static boolean isReadOnlyOperation(OperationType operationType) {
        return operationType == OperationType.Read ||
                operationType == OperationType.ReadFeed ||
                operationType == OperationType.Head ||
                operationType == OperationType.HeadFeed ||
                operationType == OperationType.Query ||
                operationType == OperationType.SqlQuery;
    }

    public static boolean isFeedRequest(OperationType requestOperationType) {
        return requestOperationType == OperationType.Create ||
                requestOperationType == OperationType.Upsert ||
                requestOperationType == OperationType.ReadFeed ||
                requestOperationType == OperationType.Query ||
                requestOperationType == OperationType.SqlQuery ||
                requestOperationType == OperationType.HeadFeed;
    }

    private static String addTrailingSlash(String path) {
        if (path == null || path.isEmpty())
            path = new String(RuntimeConstants.Separators.Url);
        else if (path.charAt(path.length() - 1) != RuntimeConstants.Separators.Url[0])
            path = path + RuntimeConstants.Separators.Url[0];

        return path;
    }

    private static String removeLeadingQuestionMark(String path) {
        if (path == null || path.isEmpty())
            return path;

        if (path.charAt(0) == RuntimeConstants.Separators.Query[0])
            return path.substring(1);

        return path;
    }

    public static boolean isValidConsistency(ConsistencyLevel backendConsistency,
            ConsistencyLevel desiredConsistency) {
        switch (backendConsistency) {
        case Strong:
            return desiredConsistency == ConsistencyLevel.Strong ||
            desiredConsistency == ConsistencyLevel.BoundedStaleness ||
            desiredConsistency == ConsistencyLevel.Session ||
            desiredConsistency == ConsistencyLevel.Eventual ||
            desiredConsistency == ConsistencyLevel.ConsistentPrefix;

        case BoundedStaleness:
            return desiredConsistency == ConsistencyLevel.BoundedStaleness ||
            desiredConsistency == ConsistencyLevel.Session ||
            desiredConsistency == ConsistencyLevel.Eventual ||
            desiredConsistency == ConsistencyLevel.ConsistentPrefix;

        case Session:
        case Eventual:
        case ConsistentPrefix:
            return desiredConsistency == ConsistencyLevel.Session ||
            desiredConsistency == ConsistencyLevel.Eventual ||
            desiredConsistency == ConsistencyLevel.ConsistentPrefix;

        default:
            throw new IllegalArgumentException("backendConsistency");
        }
    }

    public static String getUserAgent(String sdkName, String sdkVersion) {
        String osName = System.getProperty("os.name");
        if (osName == null) {
            osName = "Unknown";
        }
        osName = osName.replaceAll("\\s", "");
        String userAgent = String.format("%s/%s JRE/%s %s/%s",
                osName,
                System.getProperty("os.version"),
                System.getProperty("java.version"),
                sdkName,
                sdkVersion);
        return userAgent;
    }

    public static ObjectMapper getSimpleObjectMapper() {
        return Utils.simpleObjectMapper;
    }

    public static String getCurrentTimeGMT() {
        return Utils.dateTimeFormatGMT.print(new DateTime(DateTimeZone.forID("GMT")));
    }

    public static int getConcurrencyFactor() {
        return Runtime.getRuntime().availableProcessors();
    }

    /**
     * Generates UUID using time-based approach (time+location based), using a random multi-cast Ethernet address
     * as the location part of UUID. This method has a noticeable performance advantage compared to getSecureRandomUUID.
     * Due to security reasons, it's only used to generate activity ID in the CosmosDB Java SDK.
     *
     * @return a random UUID generated using time-based approach
     */
    public static UUID getTimeBasedRandomUUID() {
        return TB_GENERATOR.generate();
    }

    /**
     * Generates UUID using default (shared) secure random number generator. This method is less performant compared to
     * getTimeBasedRandomUUID method. Due to security reasons, this method is used to auto generate Document ID by default.
     * As a side note, users can disable the auto-generation of Document ID in RequestOptions, and provide their own IDs,
     * which can be generated by a time-based generator if better performance is needed.
     *
     * @return a random UUID generated using secure random based approach
     */
    public static UUID getSecureRandomUUID() {
        return RD_GENERATOR.generate();
    }

    @SuppressWarnings("unchecked")
    public static <O, I> O as(I i, Class<O> klass) {
        if (i == null) {
            return null;
        }

        if (klass.isInstance(i)) {
            return (O) i;
        } else {
            return null;
        }
    }

    /**
     * Get the resource segment path based on provided resource type.
     *
     * @param resourceType the resource type
     * @return             a string representing the resource segment path
     */
    public static String getResourceSegement(ResourceType resourceType) {
        switch (resourceType) {
            case Attachment:
                return Paths.ATTACHMENTS_PATH_SEGMENT;
            case Database:
                return Paths.DATABASES_PATH_SEGMENT;
            case Conflict:
                return Paths.CONFLICTS_PATH_SEGMENT;
            case Document:
                return Paths.DOCUMENTS_PATH_SEGMENT;
            case DocumentCollection:
                return Paths.COLLECTIONS_PATH_SEGMENT;
            case Offer:
                return Paths.OFFERS_PATH_SEGMENT;
            case Permission:
                return Paths.PERMISSIONS_PATH_SEGMENT;
            case StoredProcedure:
                return Paths.STORED_PROCEDURES_PATH_SEGMENT;
            case Trigger:
                return Paths.TRIGGERS_PATH_SEGMENT;
            case UserDefinedFunction:
                return Paths.USER_DEFINED_FUNCTIONS_PATH_SEGMENT;
            case User:
                return Paths.USERS_PATH_SEGMENT;
            case PartitionKeyRange:
                return Paths.PARTITION_KEY_RANGE_PATH_SEGMENT;
            case Media:
                return Paths.MEDIA_PATH_SEGMENT;
            case DatabaseAccount:
                return "";
            default:
                return null;
        }
    }
    public static byte[] getUTF8Bytes(String str) throws UnsupportedEncodingException {
        return str.getBytes("UTF-8");
    }
}
