package com.microsoft.azure.documentdb.internal;

import java.net.URI;
import java.util.List;

import com.microsoft.azure.documentdb.DatabaseAccount;

/**
 * Defines an interface used to manage endpoint selection for geo-distributed database accounts in the Azure Cosmos DB
 * database service.
 */
public interface EndpointManager {

    /**
     * Returns the current write region endpoint.
     *
     * @return the write endpoint URI
     */
    public URI getWriteEndpoint();

    /**
     * Returns the current read region endpoint.
     *
     * @return the read endpoint URI
     */
    public URI getReadEndpoint();

    /**
     * Returns the target endpoint for a given request.
     *
     * @param request the request object
     * @return        the service endpoint URI
     */
    public URI resolveServiceEndpoint(DocumentServiceRequest request);

    /**
     * Refreshes the client side endpoint cache.
     */
    public void refreshEndpointList(DatabaseAccount databaseAccount, boolean forceRefresh);

    public void refreshEndpointList(DatabaseAccount databaseAccount);

    /**
     * Gets the Database Account resource
     *
     * @return the database account
     */
    public DatabaseAccount getDatabaseAccountFromAnyEndpoint();

    /**
     * Mark the current endpoint as unavailable for reads
     */
    public void markEndpointUnavailableForRead(URI endpoint);

    /**
     * Mark the current endpoint as unavailable for writes
     */
    public void markEndpointUnavailableForWrite(URI endpoint);

    /**
     * Returns the DatabaseAccount from the cache.
     *
     * @return the database account
     */
    public DatabaseAccount getDatabaseAccountFromCache();
    
    /**
     * Close the endpoint manager
     */
    public void close();

    /**
     * Returns all the available write endpoints, as returned from the service.
     *
     * @return the available write endpoints
     */
    public List<String> getOrderedWriteEndpoints();

    /**
     * Returns all the available read endpoints, as returned from the service.
     *
     * @return the available read endpoints
     */
    public List<String> getOrderedReadEndpoints();
    
    /**
     * Returns whether the given request can use multiple write locations.
     *
     * @param request the request object.
     * @return boolean indicating whether multiple write locations can be used.
     */
    public boolean canUseMultipleWriteLocations(DocumentServiceRequest request);
}
