/*
 * The MIT License (MIT)
 * Copyright (c) 2016 Microsoft Corporation
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.microsoft.azure.documentdb.internal;

import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Map;

import org.apache.http.HttpEntity;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.entity.StringEntity;

import com.microsoft.azure.documentdb.Resource;
import com.microsoft.azure.documentdb.SqlQuerySpec;


public class DocumentServiceRequest extends AbstractDocumentServiceRequest {
    private HttpEntity body;

    /**
     * Creates a with an HttpEntity.
     *
     * @param resourceId   the resource Id.
     * @param resourceType the resource type.
     * @param body         the HTTP entity.
     * @param headers      the request headers.
     */
    private DocumentServiceRequest(OperationType operationType, String resourceId, ResourceType resourceType,
            HttpEntity body, String path, Map<String, String> headers) {
        super(operationType, resourceId, resourceType, path, headers);

        this.body = body;
    }

    /**
     * Creates a with an HttpEntity.
     *
     * @param resourceType the resource type.
     * @param path         the relative URI path.
     * @param body         the HTTP entity.
     * @param headers      the request headers.
     */
    private DocumentServiceRequest(OperationType operationType,
            ResourceType resourceType,
            String path,
            HttpEntity body,
            Map<String, String> headers) {
        this(operationType, extractIdFromUri(path), resourceType, body, path, headers);
    }

    /**
     * Gets the HTTP entity body.
     *
     * @return the HTTP entity body.
     */
    public HttpEntity getBody() {
        return this.body;
    }

    /**
     * Creates a with a stream.
     *
     * @param operation    the operation type.
     * @param resourceType the resource type.
     * @param relativePath the relative URI path.
     * @param stream       the input stream of the request.
     * @param headers      the request headers.
     * @return the created document service request.
     */
    public static DocumentServiceRequest create(OperationType operation,
            ResourceType resourceType,
            String relativePath,
            InputStream stream,
            Map<String, String> headers) {
        HttpEntity body = new InputStreamEntity(stream, InternalConstants.StreamApi.STREAM_LENGTH_EOF);
        return new DocumentServiceRequest(operation, resourceType, relativePath, body, headers);
    }

    /**
     * Creates a with a resource.
     *
     * @param operation    the operation type.
     * @param resourceType the resource type.
     * @param relativePath the relative URI path.
     * @param resource     the resource of the request.
     * @param headers      the request headers.
     * @return the created document service request.
     */
    public static DocumentServiceRequest create(OperationType operation,
            ResourceType resourceType,
            String relativePath,
            Resource resource,
            Map<String, String> headers) {
        HttpEntity body = new StringEntity(resource.toJson(), StandardCharsets.UTF_8);
        return new DocumentServiceRequest(operation, resourceType, relativePath, body, headers);
    }

    /**
     * Creates a with a query.
     *
     * @param operation    the operation type.
     * @param resourceType the resource type.
     * @param relativePath the relative URI path.
     * @param query        the query.
     * @param headers      the request headers.
     * @return the created document service request.
     */
    public static DocumentServiceRequest create(OperationType operation,
            ResourceType resourceType,
            String relativePath,
            String query,
            Map<String, String> headers) {
        HttpEntity body = new StringEntity(query, StandardCharsets.UTF_8);
        return new DocumentServiceRequest(operation, resourceType, relativePath, body, headers);
    }

    /**
     * Creates a with a query.
     *
     * @param resourceType           the resource type.
     * @param relativePath           the relative URI path.
     * @param querySpec              the query.
     * @param queryCompatibilityMode the QueryCompatibilityMode mode.
     * @param headers                the request headers.
     * @return the created document service request.
     */
    public static DocumentServiceRequest create(ResourceType resourceType,
            String relativePath,
            SqlQuerySpec querySpec,
            QueryCompatibilityMode queryCompatibilityMode,
            Map<String, String> headers) {
        OperationType operation;
        String queryText;
        switch (queryCompatibilityMode) {
        case SqlQuery:
            if (querySpec.getParameters() != null && querySpec.getParameters().size() > 0) {
                throw new IllegalArgumentException(
                        String.format("Unsupported argument in query compatibility mode '{%s}'",
                                queryCompatibilityMode.name()));
            }

            operation = OperationType.SqlQuery;
            queryText = querySpec.getQueryText();
            break;

        case Default:
        case Query:
        default:
            operation = OperationType.Query;
            queryText = querySpec.toJson();
            break;
        }

        HttpEntity body = new StringEntity(queryText, StandardCharsets.UTF_8);
        return new DocumentServiceRequest(operation, resourceType, relativePath, body, headers);
    }

    /**
     * Creates a without body.
     *
     * @param operation    the operation type.
     * @param resourceType the resource type.
     * @param relativePath the relative URI path.
     * @param headers      the request headers.
     * @return the created document service request.
     */
    public static DocumentServiceRequest create(OperationType operation,
            ResourceType resourceType,
            String relativePath,
            Map<String, String> headers) {
        return new DocumentServiceRequest(operation, resourceType, relativePath, null, headers);
    }

    /**
     * Creates a with a resourceId.
     *
     * @param operation    the operation type.
     * @param resourceId   the resource id.
     * @param resourceType the resource type.
     * @param headers      the request headers.
     * @return the created document service request.
     */
    public static DocumentServiceRequest create(OperationType operation,
            String resourceId,
            ResourceType resourceType,
            Map<String, String> headers) {
        String path = PathsHelper.generatePath(resourceType, resourceId, Utils.isFeedRequest(operation));
        return new DocumentServiceRequest(operation, resourceId, resourceType, null, path, headers);
    }

}
