package com.microsoft.azure.documentdb.internal;

import com.microsoft.azure.documentdb.ConsistencyLevel;
import com.microsoft.azure.documentdb.DatabaseAccount;
import com.microsoft.azure.documentdb.DocumentClientException;
import com.microsoft.azure.documentdb.Error;
import com.microsoft.azure.documentdb.ReplicationPolicy;

import java.util.HashMap;
import java.util.Map;

/**
 * Used internally to provides functionality to work with database account configuration in the Azure Cosmos DB database service.
 */
public class BaseDatabaseAccountConfigurationProvider implements DatabaseAccountConfigurationProvider {
    private ConsistencyLevel desiredConsistencyLevel;
    private EndpointManager globalEndpointManager;

    public BaseDatabaseAccountConfigurationProvider(ConsistencyLevel desiredConsistencyLevel, EndpointManager globalEndpointManager) {
        this.desiredConsistencyLevel = desiredConsistencyLevel;
        this.globalEndpointManager = globalEndpointManager;
        //it will refresh the DatabaseAccount cache during initialization
        this.globalEndpointManager.getDatabaseAccountFromCache();
    }

    public ConsistencyLevel getStoreConsistencyPolicy() throws DocumentClientException {
        DatabaseAccount databaseAccount = this.globalEndpointManager.getDatabaseAccountFromCache();
        validateDatabaseAccount(databaseAccount);
        ConsistencyLevel databaseAccountConsistency = databaseAccount.getConsistencyPolicy().getDefaultConsistencyLevel();
        if (this.desiredConsistencyLevel == null) {
            return databaseAccountConsistency;
        } else if (!Utils.isValidConsistency(databaseAccountConsistency, this.desiredConsistencyLevel)) {
            throw new IllegalArgumentException(String.format(
            "ConsistencyLevel %1s specified in the request is invalid when service is configured with consistency level %2s. Ensure the request consistency level is not stronger than the service consistency level.",
            this.desiredConsistencyLevel.toString(),
            databaseAccountConsistency.toString()));
        } else {
            return this.desiredConsistencyLevel;
        }
    }

    public int getMaxReplicaSetSize() throws DocumentClientException {
        DatabaseAccount databaseAccount = this.globalEndpointManager.getDatabaseAccountFromCache();
        validateDatabaseAccount(databaseAccount);
        ReplicationPolicy replicationPolicy = databaseAccount.getReplicationPolicy();
        return replicationPolicy.getMaxReplicaSetSize();
    }

    @Override
    public String getQueryEngineConfiguration() {
        DatabaseAccount databaseAccount = this.globalEndpointManager.getDatabaseAccountFromCache();
        try {
            validateDatabaseAccount(databaseAccount);
        } catch (DocumentClientException e) {
            throw new IllegalStateException(e);
        }
        return this.globalEndpointManager.getDatabaseAccountFromCache().get("queryEngineConfiguration").toString();
    }

    private void validateDatabaseAccount(DatabaseAccount databaseAccount) throws DocumentClientException {
        if (databaseAccount == null) {
            int statusCode = HttpConstants.StatusCodes.SERVICE_UNAVAILABLE;
            Map<String, String> responseHeaders = new HashMap<String, String>();
            responseHeaders.put(HttpConstants.HttpHeaders.SUB_STATUS, Integer.toString(HttpConstants.SubStatusCodes.INVALID));
            throw new DocumentClientException(statusCode, new Error(Integer.toString(statusCode),
            "Database Account could not be retrieved. Please retry this operation."), responseHeaders);
        }
    }
}
