package com.microsoft.azure.documentdb;

import java.util.ArrayList;

import org.json.JSONArray;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.microsoft.azure.documentdb.internal.Utils;
import com.microsoft.azure.documentdb.internal.routing.PartitionKeyInternal;

/**
 * Represents a partition key value in the Azure Cosmos DB database service. A partition key identifies the partition
 * where the document is stored in.
 */
public class PartitionKey {

    private final Object[] key;
    private String keyString;
    private PartitionKeyInternal internalPartitionKey;

    /**
     * Constructor. Create a new instance of the PartitionKey object.
     *
     * @param key the value of the partition key.
     */
    public PartitionKey(final Object key) {
        this.key = new Object[] {key};
        this.internalPartitionKey = PartitionKeyInternal.fromObjectArray(new ArrayList<Object>() {{ add(key); }}, true);
    }

    /**
     * Create a new instance of the PartitionKey object from a serialized JSON string.
     *
     * @param jsonString the JSON string representation of this PartitionKey object.
     * @return the PartitionKey instance.
     */
    public static PartitionKey FromJsonString(String jsonString) {
        JSONArray array = new JSONArray(jsonString);
        PartitionKey key = new PartitionKey(array.get(0));

        return key;
    }

    /**
     * Gets the Key property.
     *
     * @return the value of the partition key.
     */
    Object[] getKey() {
        return this.key;
    }

    /**
     * Serialize the PartitionKey object to a JSON string.
     *
     * @return the string representation of this PartitionKey object.
     */
    public String toString() {
        if (this.keyString == null) {
            try {
                this.keyString = Utils.getSimpleObjectMapper().writeValueAsString(this.key);
            } catch (JsonProcessingException e) {
                throw new IllegalStateException("Failed to serialize PartitionKey", e);
            }
        }
        return this.keyString;
    }

    public PartitionKeyInternal getInternalPartitionKey() {
        return internalPartitionKey;
    }
}
