package com.microsoft.azure.documentdb;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.text.WordUtils;

import com.microsoft.azure.documentdb.internal.Constants;

/**
 * Represents the conflict resolution policy configuration for specifying how to resolve conflicts
 * in case writes from different regions result in conflicts on documents in the collection in the Azure Cosmos DB service.
 *
 * A collection with custom conflict resolution with no user-registered stored procedure.
 * <pre>{@code
 * DocumentCollection collectionSpec = new DocumentCollection();
 * collectionSpec.setId("Multi-master collection");
 *
 * ConflictResolutionPolicy policy = ConflictResolutionPolicy.createCustomPolicy();
 * collectionSpec.setConflictResolutionPolicy(policy);
 *
 * DocumentCollection collection = client.createCollection(databaseLink, collectionSpec, null)
 *         .toBlocking().single().getResource();
 *
 * }
 * </pre>
 *
 * A collection with custom conflict resolution with a user-registered stored procedure.
 * <pre>{@code
 * DocumentCollection collectionSpec = new DocumentCollection();
 * collectionSpec.setId("Multi-master collection");
 *
 * ConflictResolutionPolicy policy = ConflictResolutionPolicy.createCustomPolicy(conflictResolutionSprocName);
 * collectionSpec.setConflictResolutionPolicy(policy);
 *
 * DocumentCollection collection = client.createCollection(databaseLink, collectionSpec, null)
 *         .toBlocking().single().getResource();
 *
 * }
 * </pre>
 *
 * A collection with last writer wins conflict resolution, based on a path in the conflicting documents.
 * A collection with custom conflict resolution with a user-registered stored procedure.
 * <pre>{@code
 * DocumentCollection collectionSpec = new DocumentCollection();
 * collectionSpec.setId("Multi-master collection");
 *
 * ConflictResolutionPolicy policy = ConflictResolutionPolicy.createLastWriterWinsPolicy("/path/for/conflict/resolution");
 * collectionSpec.setConflictResolutionPolicy(policy);
 *
 * DocumentCollection collection = client.createCollection(databaseLink, collectionSpec, null)
 *         .toBlocking().single().getResource();
 *
 * }
 * </pre>
 */
public class ConflictResolutionPolicy extends JsonSerializable {
    
    /**
     * Initializes a new instance of the {@link ConflictResolutionPolicy} class for the Azure Cosmos DB service.
     */
    public ConflictResolutionPolicy() {
    }
    
    /**
     * Creates a LastWriterWins conflict resolution policy.
     * 
     * The Conflict resolution mode is set to {@link ConflictResolutionMode#LastWriterWins}.
     * 
     * The {@link Resource#getTimestamp()} path will be used for last writer wins conflict-resolution.
     * In case of a conflict occurring on a document, the document with the higher timestamp value wins.
     * 
     * @return policy the LastWriterWins conflict resolution policy
     */
    public static ConflictResolutionPolicy createLastWriterWinsPolicy() {
        ConflictResolutionPolicy policy = new ConflictResolutionPolicy();
        policy.setConflictResolutionMode(ConflictResolutionMode.LastWriterWins);
        return policy;
    }
    
    /**
     * Creates a LastWriterWins conflict resolution policy.
     * 
     * The Conflict resolution mode is set to {@link ConflictResolutionMode#LastWriterWins}.
     *      
     * The path which is present in each document in the Azure Cosmos DB service is used for last writer wins conflict-resolution.
     * This path must be present in each document and must be an integer value.
     * In case of a conflict occurring on a document, the document with the higher integer value in the specified path will be picked.
     *
     * @param path The path to check values for last-writer wins conflict resolution.
     *             That path is a rooted path of the property in the document, such as "/name/first".
     * 
     * @return policy the LastWriterWins conflict resolution policy
     */
    public static ConflictResolutionPolicy createLastWriterWinsPolicy(String path) {
        ConflictResolutionPolicy policy = new ConflictResolutionPolicy();
        policy.setConflictResolutionMode(ConflictResolutionMode.LastWriterWins);
        if (path != null) {
            policy.setConflictResolutionPath(path);
        }
        return policy;
    }
    
    /**
     * Creates a Custom conflict resolution policy.
     * 
     * The Conflict resolution mode is set to {@link ConflictResolutionMode#Custom}.
     *      
     * Sets the {@link StoredProcedure} which is used for conflict resolution in the Azure Cosmos DB service.
     * This stored procedure may be created after the {@link DocumentCollection} is created and can be changed as required.
     *
     * <ul>
     * <li>In case the stored procedure fails or throws an exception,
     * the conflict resolution will default to registering conflicts in the conflicts feed</li>
     * <li>The user can provide the stored procedure @see {@link Resource#getId()}</li>
     * </ul>
     *
     * @param sprocLink The link to the stored procedure to perform conflict resolution.
     * 
     * @return policy the Custom conflict resolution policy
     */
    public static ConflictResolutionPolicy createCustomPolicy(String sprocLink) {
        ConflictResolutionPolicy policy = new ConflictResolutionPolicy();
        policy.setConflictResolutionMode(ConflictResolutionMode.Custom);
        if (sprocLink != null) {
            policy.setConflictResolutionProcedure(sprocLink);
        }
        return policy;
    }

    public ConflictResolutionPolicy(String jsonString) {
        super(jsonString);
    }

    /**
     * Gets the {@link ConflictResolutionMode} in the Azure Cosmos DB service.
     * By default it is {@link ConflictResolutionMode#LastWriterWins}.
     *
     * @return ConflictResolutionMode.
     */
    public ConflictResolutionMode getConflictResolutionMode() {
        String strValue = super.getString(Constants.Properties.MODE);

        if (!StringUtils.isEmpty(strValue)) {
            try {
                return ConflictResolutionMode.valueOf(WordUtils.capitalize(super.getString(Constants.Properties.MODE)));
            } catch (IllegalArgumentException e) {
                this.getLogger().warn("Invalid ConflictResolutionMode value {}.", super.getString(Constants.Properties.MODE));
                return ConflictResolutionMode.Invalid;
            }
        }

        return ConflictResolutionMode.Invalid;
    }

    private void setConflictResolutionMode(ConflictResolutionMode mode) {
        super.set(Constants.Properties.MODE, mode.name());
    }

    /**
     * Gets the path which is present in each document in the Azure Cosmos DB service for last writer wins conflict-resolution.
     * This path must be present in each document and must be an integer value.
     * In case of a conflict occurring on a document, the document with the higher integer value in the specified path will be picked.
     * If the path is unspecified, by default the {@link Resource#getTimestamp()} path will be used.
     *
     * This value should only be set when using {@link ConflictResolutionMode#LastWriterWins}
     *
     * @return The path to check values for last-writer wins conflict resolution.
     * That path is a rooted path of the property in the document, such as "/name/first".
     */
    public String getConflictResolutionPath() {
        return super.getString(Constants.Properties.CONFLICT_RESOLUTION_PATH);
    }

    private void setConflictResolutionPath(String value) {
        super.set(Constants.Properties.CONFLICT_RESOLUTION_PATH, value);
    }

    /**
     * Gets the {@link StoredProcedure} which is used for conflict resolution in the Azure Cosmos DB service.
     * This stored procedure may be created after the {@link DocumentCollection} is created and can be changed as required.
     *
     * <ul>
     * <li>This value should only be set when using {@link ConflictResolutionMode#Custom}</li>
     * <li>In case the stored procedure fails or throws an exception,
     * the conflict resolution will default to registering conflicts in the conflicts feed</li>
     * <li>The user can provide the stored procedure @see {@link Resource#getId()}</li>
     * </ul>
     **
     * @return the stored procedure to perform conflict resolution.]
     */
    public String getConflictResolutionProcedure() {
        return super.getString(Constants.Properties.CONFLICT_RESOLUTION_PROCEDURE);
    }

    private void setConflictResolutionProcedure(String value) {
        super.set(Constants.Properties.CONFLICT_RESOLUTION_PROCEDURE, value);
    }
}