/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License.
 * See License.txt in the project root for license information.
 */

package com.microsoft.azure.datalake.store;


import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.microsoft.azure.datalake.store.acl.AclEntry;
import com.microsoft.azure.datalake.store.acl.AclStatus;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.regex.Pattern;

/**
 * protocol.Core class implements the calls for the RESP API. There is one method in Core for every
 * REST API supported by the server.
 * <P>
 * The methods in this class tend to be lower-level, exposing all the details of the underlying operation.
 * To call the methods, instantiate a {@link RequestOptions} object first. Assign any of the
 * member values as needed (e.g., the RetryPolicy). Then create a new {@link OperationResponse} object. The
 * {@link OperationResponse} is used for passing the call results and stats back from the call.
 * </P><P>
 * Failures originating in Core methods are communicated back through the {@link OperationResponse} parameter.
 * </P>
 * <P>
 * <B>Thread Safety: </B> all static methods in this class are thread-safe
 *
 * </P>
 */
public class Core {

    // no constructor - class has static methods only
    private Core() {}



    /**
     * create a file and write to it.
     *
     *
     * @param path the full path of the file to create
     * @param overwrite whether to overwrite the file if it already exists
     * @param octalPermission permissions for the file, as octal digits (For Example, {@code "755"}). Can be null.
     * @param contents byte array containing the contents to be written to the file. Can be {@code null}
     * @param offsetWithinContentsArray offset within the byte array passed in {@code contents}. Bytes starting
     *                                  at this offset will be written to server
     * @param length number of bytes from {@code contents} to be written
     * @param leaseId a String containing the lease ID (generated by client). Can be null.
     * @param sessionId a String containing the session ID (generated by client). Can be null.
     * @param createParent if true, then parent directories of the file are created if they are missing.
     * @param syncFlag Use {@link SyncFlag#DATA} when writing
     *                 more bytes to same file path. Most performant operation.
     *
     *                 Use {@link SyncFlag#METADATA} when metadata for the
     *                 file also needs to be updated especially file length
     *                 retrieved from
     *                 {@link #getFileStatus(String, ADLStoreClient, RequestOptions, OperationResponse)}
     *                 or {@link #listStatus(String, String, String, int, ADLStoreClient, RequestOptions, OperationResponse)} API call.
     *                 Has an overhead of updating metadata operation.
     *
     *                 Use {@link SyncFlag#CLOSE} when no more data is
     *                 expected to be written in this path. Adl backend would
     *                 update metadata, close the stream handle and
     *                 release the lease on the
     *                 path if valid leaseId is passed.
     *                 Expensive operation and should be used only when last
     *                 bytes are written.
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void create(String path,
                              boolean overwrite,
                              String octalPermission,
                              byte[] contents,
                              int offsetWithinContentsArray,
                              int length,
                              String leaseId,
                              String sessionId,
                              boolean createParent,
                              SyncFlag syncFlag,
                              ADLStoreClient client,
                              RequestOptions opts,
                              OperationResponse resp) {
        QueryParams qp = new QueryParams();
        qp.add("overwrite", Boolean.toString(overwrite));
        qp.add("syncFlag", syncFlag.name());
        qp.add("write", "true");  // This is to suppress the 307-redirect from server (standard WebHdfs behavior)
        if (leaseId != null && !leaseId.equals("")) {
            qp.add("leaseid", leaseId);
        }
        if (sessionId != null && !sessionId.equals("")) {
            qp.add("filesessionid", sessionId);
        }
        if (!createParent) {
            qp.add("CreateParent", "false");
        }
        if (octalPermission != null && !octalPermission.equals("")) {
            if (isValidOctal(octalPermission)) {
                qp.add("permission", octalPermission);
            } else {
                resp.successful = false;
                resp.message = "Invalid directory permissions specified: " + octalPermission;
                return;
            }
        }

        HttpTransport.makeCall(client, Operation.CREATE, path, qp, contents, offsetWithinContentsArray, length, opts, resp);
    }

    /**
     * append bytes to an existing file created with
     *
     * {@link #create(String, boolean, String, byte[], int, int, String, String, boolean, SyncFlag, ADLStoreClient, RequestOptions, OperationResponse)} (String, boolean, String, byte[], int, int, String, String, boolean, boolean, ADLStoreClient, RequestOptions, OperationResponse) create}.
     *
     * @param path the full path of the file to append to. The file must already exist.
     * @param offsetToAppendTo offset at which to append to to file. To let the server choose offset, pass {@code -1}.
     * @param contents byte array containing the contents to be written to the file. Can be {@code null}
     * @param offsetWithinContentsArray offset within the byte array passed in {@code contents}. Bytes starting
     *                                  at this offset will be written to server
     * @param length number of bytes from {@code contents} to be written
     * @param leaseId a String containing the lease ID (generated by client). Can be null.
     * @param sessionId a String containing the session ID (generated by client). Can be null.
     * @param syncFlag Use {@link SyncFlag#DATA} when writing
     *                 more bytes to same file path. Most performant operation.
     *
     *                 Use {@link SyncFlag#METADATA} when metadata for the
     *                 file also needs to be updated especially file length
     *                 retrieved from
     *                 {@link #getFileStatus(String, ADLStoreClient, RequestOptions, OperationResponse)}
     *                 or {@link #listStatus(String, String, String, int, ADLStoreClient, RequestOptions, OperationResponse)} API call.
     *                 Has an overhead of updating metadata operation.
     *
     *                 Use {@link SyncFlag#CLOSE} when no more data is
     *                 expected to be written in this path. Adl backend would
     *                 update metadata, close the stream handle and
     *                 release the lease on the
     *                 path if valid leaseId is passed.
     *                 Expensive operation and should be used only when last
     *                 bytes are written.
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void append(String path,
                              long offsetToAppendTo,
                              byte[] contents,
                              int offsetWithinContentsArray,
                              int length,
                              String leaseId,
                              String sessionId,
                              SyncFlag syncFlag,
                              ADLStoreClient client,
                              RequestOptions opts,
                              OperationResponse resp) {
        QueryParams qp = new QueryParams();
        qp.add("append", "true");  // This is to suppress the 307-redirect from server (standard WebHdfs behavior)
        qp.add("syncFlag", syncFlag.name());
        if (leaseId != null && !leaseId.equals("")) {
            qp.add("leaseid", leaseId);
        }
        if (sessionId != null && !sessionId.equals("")) {
            qp.add("filesessionid", sessionId);
        }
        if (offsetToAppendTo >= 0) {
            qp.add("offset", Long.toString(offsetToAppendTo));
        }

        HttpTransport.makeCall(client, Operation.APPEND, path, qp, contents, offsetWithinContentsArray, length, opts, resp);
    }

    /**
     * append bytes to a file. The offset is determined by the server. This enables multiple writers to
     * append concurrently to the same file. A file created with {@code concurrentAppend} can only be appended
     * with {@code concurrentAppend}.
     *
     * @param path the full path of the file to append to.
     * @param contents byte array containing the contents to be written to the file. Can be {@code null}
     * @param offsetWithinContentsArray offset within the byte array passed in {@code contents}. Bytes starting
     *                                  at this offset will be written to server
     * @param length number of bytes from {@code contents} to be written
     * @param autoCreate boolean specifying whether to create the file if it doesn't already exist
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void concurrentAppend(String path,
                                        byte[] contents,
                                        int offsetWithinContentsArray,
                                        int length,
                                        boolean autoCreate,
                                        ADLStoreClient client,
                                        RequestOptions opts,
                                        OperationResponse resp) {

        QueryParams qp = new QueryParams();
        if (autoCreate) qp.add("appendMode", "autocreate");

        HttpTransport.makeCall(client, Operation.CONCURRENTAPPEND, path, qp, contents, offsetWithinContentsArray, length, opts, resp);
    }

    /**
     * read from a file. This is the stateless read method, that reads bytes from an offset in a file.
     *
     *
     * @param path the full path of the file to read. The file must already exist.
     * @param offset the offset within the ADL file to read from
     * @param length the number of bytes to read from file
     * @param sessionId a String containing the session ID (generated by client). Can be null.
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     * @return returns an {@link com.microsoft.azure.datalake.store.ADLFileInputStream}
     */
    public static InputStream open(String path,
                                   long offset,
                                   long length,
                                   String sessionId,
                                   ADLStoreClient client,
                                   RequestOptions opts,
                                   OperationResponse resp) {
        return open(path, offset, length, sessionId, false, client, opts, resp);
    }




    /**
     * read from a file. This is the stateless read method, that reads bytes from an offset in a file.
     *
     *
     * @param path the full path of the file to read. The file must already exist.
     * @param offset the offset within the ADL file to read from
     * @param length the number of bytes to read from file
     * @param sessionId a String containing the session ID (generated by client). Can be null.
     * @param speculativeRead indicates if the read is speculative. Currently fails the read - so dont use.
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     * @return returns an {@link com.microsoft.azure.datalake.store.ADLFileInputStream}
     */
    public static InputStream open(String path,
                                   long offset,
                                   long length,
                                   String sessionId,
                                   boolean speculativeRead,
                                   ADLStoreClient client,
                                   RequestOptions opts,
                                   OperationResponse resp) {
        QueryParams qp = new QueryParams();
        qp.add("read", "true");
        if (offset < 0) {
            resp.successful = false;
            resp.message = "attempt to read from negative offset: " + offset;
            return null;
        }

        if (length < 0) {
            resp.successful = false;
            resp.message = "attempt to read negative length: " + length;
            return null;
        }

        if (offset > 0) qp.add("offset", Long.toString(offset));
        if (length > 0) qp.add("length", Long.toString(length));
        if (speculativeRead) qp.add("speculative", "true");
        if (sessionId != null && !sessionId.equals("")) {
            qp.add("filesessionid", sessionId);
        }

        HttpTransport.makeCall(client, Operation.OPEN, path, qp, null, 0, 0, opts, resp);

        if (resp.successful) {
            return resp.responseStream;
        } else {
            return null;
        }
    }

    /**
     * delete a file or directory from Azure Data Lake.
     *
     * @param path the full path of the file to delete. The file must already exist.
     * @param recursive if deleting a directory, then whether to delete all files an directories
     *                  in the directory hierarchy underneath
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     * @return returns {@code true} if the delete was successful. Also check {@code resp.successful}.
     */
    public static boolean delete(String path,
                                 boolean recursive,
                                 ADLStoreClient client,
                                 RequestOptions opts,
                                 OperationResponse resp) {
        QueryParams qp = new QueryParams();
        qp.add("recursive", (recursive? "true" : "false"));

        HttpTransport.makeCall(client, Operation.DELETE, path, qp, null, 0, 0, opts, resp);
        if (!resp.successful) return false;

        boolean returnValue = true;
        try {
            JsonFactory jf = new JsonFactory();
            JsonParser jp = jf.createParser(resp.responseStream);
            jp.nextToken();  // START_OBJECT - {
            jp.nextToken();  // FIELD_NAME - "boolean":
            jp.nextToken();  // boolean value
            returnValue = jp.getValueAsBoolean();
            jp.nextToken(); //  END_OBJECT - }  for boolean
            jp.close();
        } catch (IOException ex) {
            resp.successful = false;
            resp.message = "Unexpected error happened reading response stream or parsing JSon from delete()";
        } finally {
            try {
                resp.responseStream.close();
            } catch (IOException ex) {
                //swallow since it is only the closing of the stream
            }
        }
        return returnValue;
    }

    /**
     * rename a file.
     *
     * @param path the full path of the existing file to rename. (the old name)
     * @param overwrite overwrite the destination if it already exists and is a file or an
     *                  empty directory
     * @param destination the new name of the file
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     *
     * @return returns {@code true} if the rename was successful. Also check {@code resp.successful}.
     */
    public static boolean rename(String path,
                                 String destination,
                                 boolean overwrite,
                                 ADLStoreClient client,
                                 RequestOptions opts,
                                 OperationResponse resp) {

        if (destination == null || destination.equals(""))
            throw new IllegalArgumentException("destination cannot be null or empty");

        // prepend the prefix, if applicable, to the destination path
        String prefix = client.getFilePathPrefix();
        if (prefix!=null) {
            if (destination.charAt(0) == '/') {
                destination = prefix + destination;
            } else {
                destination = prefix + "/" + destination;
            }
        }

        QueryParams qp = new QueryParams();
        qp.add("destination", destination);

        if (overwrite) qp.add("renameoptions", "OVERWRITE");

        HttpTransport.makeCall(client, Operation.RENAME, path, qp, null, 0, 0, opts, resp);
        if (!resp.successful) return false;

        boolean returnValue = true;
        try {
            JsonFactory jf = new JsonFactory();
            JsonParser jp = jf.createParser(resp.responseStream);
            jp.nextToken();  // START_OBJECT - {
            jp.nextToken();  // FIELD_NAME - "boolean":
            jp.nextToken();  // boolean value
            returnValue = jp.getValueAsBoolean();
            jp.nextToken(); //  END_OBJECT - }  for boolean
            jp.close();
        } catch (IOException ex) {
            resp.successful = false;
            resp.message = "Unexpected error happened reading response stream or parsing JSon from rename()";
        } finally {
            try {
                resp.responseStream.close();
            } catch (IOException ex) {
                //swallow since it is only the closing of the stream
            }
        }
        return returnValue;
    }

    /**
     * creates a directory, and all it's parent directories if they dont exist.
     *
     * @param path the full path of the directory to create. Any missing parents in the path will also be created.
     * @param octalPermission permissions for the directory, as octal digits (For Example, {@code "755"}). Can be null.
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     *
     * @return returns {@code true} if the create was successful. Also check {@code resp.successful}.
     */
    public static boolean mkdirs(String path,
                                 String octalPermission,
                                 ADLStoreClient client,
                                 RequestOptions opts,
                                 OperationResponse resp) {

        QueryParams qp = new QueryParams();
        if (octalPermission != null && !octalPermission.equals("")) {
            if (isValidOctal(octalPermission)) {
                qp.add("permission", octalPermission);
            } else {
                resp.successful = false;
                resp.message = "Invalid directory permissions specified: " + octalPermission;
                return false;
            }
        }

        HttpTransport.makeCall(client, Operation.MKDIRS, path, qp, null, 0, 0, opts, resp);
        if (!resp.successful) return false;

        boolean returnValue = true;
        try {
            JsonFactory jf = new JsonFactory();
            JsonParser jp = jf.createParser(resp.responseStream);
            jp.nextToken();  // START_OBJECT - {
            jp.nextToken();  // FIELD_NAME - "boolean":
            jp.nextToken();  // boolean value
            returnValue = jp.getValueAsBoolean();
            jp.nextToken(); //  END_OBJECT - }  for boolean
            jp.close();
        } catch (IOException ex) {
            resp.successful = false;
            resp.message = "Unexpected error happened reading response stream or parsing JSon from mkdirs()";
        } finally {
            try {
                resp.responseStream.close();
            } catch (IOException ex) {
                //swallow since it is only the closing of the stream
            }
        }
        return returnValue;
    }


    /**
     * Sets the expiry time on a file.
     *
     * @param path path of the file to set expiry on
     * @param expiryOption {@link ExpiryOption} value specifying how to interpret the passed in time
     * @param milliseconds time duration in milliseconds
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void setExpiryTime(String path,
                                     ExpiryOption expiryOption,
                                     long milliseconds,
                                     ADLStoreClient client,
                                     RequestOptions opts,
                                     OperationResponse resp) {
        if (expiryOption == null) {
            resp.successful = false;
            resp.message = "null ExpiryOption passed to setExpiryTime";
            return;
        }

        if (milliseconds < 0) {
            resp.successful = false;
            resp.message = "Expiry time is negative " + Long.toString(milliseconds);
            return;
        }

        QueryParams qp = new QueryParams();
        qp.add("expiryOption", expiryOption.toString());
        qp.add("expireTime", Long.toString(milliseconds));

        HttpTransport.makeCall(client, Operation.SETEXPIRY, path, qp, null, 0, 0, opts, resp);
    }


    /**
     * Gets the content summary of a file or directory.
     * @param path path of the file or directory to query
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     * @return {@link ContentSummary} containing summary of information about the file or directory
     */
    public static ContentSummary getContentSummary(String path,
                                 ADLStoreClient client,
                                 RequestOptions opts,
                                 OperationResponse resp) {
        HttpTransport.makeCall(client, Operation.GETCONTENTSUMMARY, path, null, null, 0, 0, opts, resp);
        if (!resp.successful) return null;
        try {
            long length=0;
            long directoryCount=0;
            long fileCount=0;
            long spaceConsumed=0;

            JsonFactory jf = new JsonFactory();
            JsonParser jp = jf.createParser(resp.responseStream);
            String fieldName, fieldValue;
            jp.nextToken();
            while (jp.hasCurrentToken()) {
                if (jp.getCurrentToken() == JsonToken.FIELD_NAME) {
                    fieldName = jp.getCurrentName();
                    jp.nextToken();
                    fieldValue = jp.getText();

                    if (fieldName.equals("length")) length = Long.parseLong(fieldValue);
                    if (fieldName.equals("directoryCount")) directoryCount = Long.parseLong(fieldValue);
                    if (fieldName.equals("fileCount")) fileCount = Long.parseLong(fieldValue);
                    if (fieldName.equals("spaceConsumed")) spaceConsumed = Long.parseLong(fieldValue);
                }
                jp.nextToken();
            }
            jp.close();
            return new ContentSummary(length, directoryCount, fileCount, spaceConsumed);
        } catch (IOException ex) {
            resp.successful = false;
            resp.message = "Unexpected error happened reading response stream or parsing JSon from getContentSummary()";
            return null;
        } finally {
            try {
                resp.responseStream.close();
            } catch (IOException ex) {
                //swallow since it is only the closing of the stream
            }
        }
    }


    /**
     * Concatenate the specified list of files into the target filename. The target should not exist.
     * the source files will be deleted if the concatenate succeeds.
     *  @param path that full path of the target file to create
     * @param sources {@link List} of strings containing full pathnames of the files to concatenate.
     *                Cannot be null or empty.
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void concat(String path,
                              List<String> sources,
                              ADLStoreClient client,
                              RequestOptions opts,
                              OperationResponse resp) {
        concat(path, sources, client, false, opts, resp);
    }

    /**
     * Concatenate the specified list of files into the target filename. The target should not exist.
     * the source files will be deleted if the concatenate succeeds.
     *
     * @param path that full path of the target file to create
     * @param sources {@link List} of strings containing full pathnames of the files to concatenate.
     *                Cannot be null or empty.
     * @param deleteSourceDirectory specify whether the directory containing the source files should be deleted.
     *                              If source files specified in concat include <I>all</I> the files in the directory,
     *                              then specifying this parameter makes the concat delete the source directory along
     *                              with all the source files. Since a directory-level operation is faster than deleting
     *                              all the files one-by-one on the server, this might provide a performance boost for
     *                              some applications.
     *                              This is a specific performance optimizations for some apps like bulk upload apps,
     *                              where the needs match the functionality offered by this flag.
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void concat(String path,
                              List<String> sources,
                              ADLStoreClient client,
                              boolean deleteSourceDirectory,
                              RequestOptions opts,
                              OperationResponse resp) {
        if (sources == null || sources.size() == 0 ) {
            resp.successful = false;
            resp.message = "No source files specified to concatenate";
            return;
        }
        byte[] body = null;
        StringBuilder sb = new StringBuilder("sources=");
        boolean firstelem = true;
        HashSet<String> pathSet = new HashSet<String>(4096);  // 4096: reasonably large-enough number for "most" cases
        for (String item : sources) {
            if (item.equals(path)) {
                resp.successful = false;
                resp.message = "One of the source files to concatenate is the destination file";
                return;
            }

            // check that each source path occurs only once
            if (pathSet.contains(item)) {
                resp.successful = false;
                resp.message = "concat() source list contains a file more than once: " + item;
                return;
            } else {
                pathSet.add(item);
            }

            // prepend Filesystem prefix if needed to each of the paths
            String prefix = client.getFilePathPrefix();
            if (prefix!=null) {
                if (item.charAt(0) == '/') {
                    item = prefix + item;
                } else {
                    item = prefix + "/" + item;
                }
            }

            if (!firstelem) sb.append(',');
                       else firstelem = false;
            sb.append(item);
        }
        try {
            body = sb.toString().getBytes("UTF-8");
        } catch (UnsupportedEncodingException ex) {
            //This should't happen.
            assert false : "UTF-8 encoding is not supported";
        }

        QueryParams qp = null;
        if (deleteSourceDirectory) {
            qp = new QueryParams();
            qp.add("deleteSourceDirectory", "true");
        }

        HttpTransport.makeCall(client, Operation.MSCONCAT, path, qp, body, 0, body.length, opts, resp);
    }


    /**
     * Gets the directory metadata associated with a file or directory.
     *
     * @param path the file or directory to get metadata for
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     *
     * @return {@link DirectoryEntry} containing the metadata for the file
     */
    public static DirectoryEntry getFileStatus(String path,
                                               ADLStoreClient client,
                                               RequestOptions opts,
                                               OperationResponse resp) {
        // overload, to retain backwards compatibility
        return getFileStatus(path, null, client, opts, resp);
    }

    /**
     * Gets the directory metadata associated with a file or directory.
     *
     * @param path the file or directory to get metadata for
     * @param oidOrUpn {@link UserGroupRepresentation} enum specifying whether to return user and group information as
     *                                                OID or UPN
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     *
     * @return {@link DirectoryEntry} containing the metadata for the file
     */
    public static DirectoryEntry getFileStatus(String path,
                                               UserGroupRepresentation oidOrUpn,
                                               ADLStoreClient client,
                                               RequestOptions opts,
                                               OperationResponse resp) {
        QueryParams qp = new QueryParams();

        if (oidOrUpn == null) oidOrUpn = UserGroupRepresentation.OID;
        String tooid = (oidOrUpn == UserGroupRepresentation.OID)? "true" : "false";
        qp.add("tooid", tooid);

        HttpTransport.makeCall(client, Operation.GETFILESTATUS, path, qp, null, 0, 0, opts, resp);

        if (resp.successful) {
            try {
                String name = "";
                String fullName = "";
                long length = 0;
                String group = "";
                String user = "";
                Date lastAccessTime = null;
                Date lastModifiedTime = null;
                DirectoryEntryType type = null;
                String permission = "";
                int replicationFactor = 1;
                long blocksize = 256 * 1024 * 1024;
                boolean aclBit = true;
                Date expiryTime = null;

                JsonFactory jf = new JsonFactory();
                JsonParser jp = jf.createParser(resp.responseStream);
                String fieldName, fieldValue;
                jp.nextToken();
                while (jp.hasCurrentToken()) {
                    if (jp.getCurrentToken() == JsonToken.FIELD_NAME) {
                        fieldName = jp.getCurrentName();
                        jp.nextToken();
                        fieldValue = jp.getText();

                        if (fieldName.equals("length")) length = Long.parseLong(fieldValue);
                        if (fieldName.equals("type")) type = DirectoryEntryType.valueOf(fieldValue);
                        if (fieldName.equals("accessTime")) lastAccessTime = new Date(Long.parseLong(fieldValue));
                        if (fieldName.equals("modificationTime")) lastModifiedTime = new Date(Long.parseLong(fieldValue));
                        if (fieldName.equals("permission")) permission = fieldValue;
                        if (fieldName.equals("owner")) user = fieldValue;
                        if (fieldName.equals("group")) group = fieldValue;
                        if (fieldName.equals("blockSize")) blocksize = Long.parseLong(fieldValue);
                        if (fieldName.equals("replication")) replicationFactor = Integer.parseInt(fieldValue);
                        if (fieldName.equals("aclBit")) aclBit = Boolean.parseBoolean(fieldValue);
                        if (fieldName.equals("msExpirationTime")) {
                            long expiryms = Long.parseLong(fieldValue);
                            if (expiryms > 0) expiryTime = new Date(expiryms);
                        }
                    }
                    jp.nextToken();
                }
                jp.close();
                fullName = path;
                name = path.substring(path.lastIndexOf("/")+1);
                return new DirectoryEntry(name,
                                          fullName,
                                          length,
                                          group,
                                          user,
                                          lastAccessTime,
                                          lastModifiedTime,
                                          type,
                                          blocksize,
                                          replicationFactor,
                                          permission,
                                          aclBit,
                                          expiryTime
                        );
            } catch (IOException ex) {
                resp.successful = false;
                resp.message = "Unexpected error happened reading response stream or parsing JSon from getFileStatus()";
            } finally {
                try {
                    resp.responseStream.close();
                } catch (IOException ex) {
                    //swallow since it is only the closing of the stream
                }
            }
        }
        return null;
    }

    /**
     * enumerates the contents of a direcotry, returning a {@link List} of {@link DirectoryEntry} objects,
     * one per file or directory in the specified directory.
     * <P>
     * To avoid overwhelming the client or the server, the call may return a partial list, in which case
     * the caller should make the call again with the last entry from the returned list specified as the
     * {@code listAfter} parameter of the next call.
     * </P>
     * @param path the directory to enumerate
     * @param listAfter the filename after which to begin enumeration
     * @param listBefore the filename before which to end the enumeration
     * @param listSize the maximum number of entries in the returned list
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     *
     * @return {@link List}&lt;{@link DirectoryEntry}&gt; containing the contents of the directory
     */
    public static List<DirectoryEntry> listStatus(String path,
                                                  String listAfter,
                                                  String listBefore,
                                                  int listSize,
                                                  ADLStoreClient client,
                                                  RequestOptions opts,
                                                  OperationResponse resp) {
        return listStatus(path, listAfter, listBefore, listSize, null, client, opts, resp);
    }

    /**
     * enumerates the contents of a direcotry, returning a {@link List} of {@link DirectoryEntry} objects,
     * one per file or directory in the specified directory.
     * <P>
     * To avoid overwhelming the client or the server, the call may return a partial list, in which case
     * the caller should make the call again with the last entry from the returned list specified as the
     * {@code listAfter} parameter of the next call.
     * </P>
     * @param path the directory to enumerate
     * @param listAfter the filename after which to begin enumeration
     * @param listBefore the filename before which to end the enumeration
     * @param listSize the maximum number of entries in the returned list
     * @param oidOrUpn {@link UserGroupRepresentation} enum specifying whether to return user and group information as
     *                                                OID or UPN
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     *
     * @return {@link List}&lt;{@link DirectoryEntry}&gt; containing the contents of the directory
     */
    public static List<DirectoryEntry> listStatus(String path,
                                                  String listAfter,
                                                  String listBefore,
                                                  int listSize,
                                                  UserGroupRepresentation oidOrUpn,
                                                  ADLStoreClient client,
                                                  RequestOptions opts,
                                                  OperationResponse resp) {
        QueryParams qp = new QueryParams();

        if (listAfter!=null && !listAfter.equals("")) {
            qp.add("listAfter", listAfter);
        }
        if (listBefore!=null && !listBefore.equals("")) {
            qp.add("listBefore", listBefore);
        }
        if (listSize > 0) {
            qp.add("listSize", Integer.toString(listSize));
        }

        if (oidOrUpn == null) oidOrUpn = UserGroupRepresentation.OID;
        String tooid = (oidOrUpn == UserGroupRepresentation.OID)? "true" : "false";
        qp.add("tooid", tooid);

        HttpTransport.makeCall(client, Operation.LISTSTATUS, path, qp, null, 0, 0, opts, resp);

        if (resp.successful) {
            ArrayList<DirectoryEntry> list = new ArrayList<DirectoryEntry>();
            try {
                String name = "";
                String fullName = "";
                long length = 0;
                String group = "";
                String user = "";
                Date lastAccessTime = null;
                Date lastModifiedTime = null;
                DirectoryEntryType type = null;
                String permission = "";
                int replicationFactor = 1;
                long blocksize = 256 * 1024 * 1024;
                boolean aclBit = true;
                Date expiryTime = null;

                if (!path.endsWith("/")) { path = path + "/"; }

                JsonFactory jf = new JsonFactory();
                JsonParser jp = jf.createParser(resp.responseStream);
                String fieldName, fieldValue;

                jp.nextToken();  // START_OBJECT - {
                jp.nextToken();  // FIELD_NAME - "FileStatuses":
                jp.nextToken();  // START_OBJECT - {
                jp.nextToken();  // FIELD_NAME - "FileStatus":
                jp.nextToken();  // START_ARRAY - [
                jp.nextToken();
                while (jp.hasCurrentToken()) {
                    if (jp.getCurrentToken() == JsonToken.END_OBJECT) {
                        if ("".equals(name)) {   // called on a file instead of a directory
                            fullName = path;
                        } else {
                            fullName = path + name;
                        }
                        DirectoryEntry entry = new DirectoryEntry(name,
                                fullName,
                                length,
                                group,
                                user,
                                lastAccessTime,
                                lastModifiedTime,
                                type,
                                blocksize,
                                replicationFactor,
                                permission,
                                aclBit,
                                expiryTime);
                        list.add(entry);
                    }
                    if (jp.getCurrentToken() == JsonToken.FIELD_NAME) {
                        fieldName = jp.getCurrentName();
                        jp.nextToken();
                        fieldValue = jp.getText();

                        if (fieldName.equals("length")) length = Long.parseLong(fieldValue);
                        if (fieldName.equals("pathSuffix")) name = fieldValue;
                        if (fieldName.equals("type")) type = DirectoryEntryType.valueOf(fieldValue);
                        if (fieldName.equals("accessTime")) lastAccessTime = new Date(Long.parseLong(fieldValue));
                        if (fieldName.equals("modificationTime")) lastModifiedTime = new Date(Long.parseLong(fieldValue));
                        if (fieldName.equals("permission")) permission = fieldValue;
                        if (fieldName.equals("owner")) user = fieldValue;
                        if (fieldName.equals("group")) group = fieldValue;
                        if (fieldName.equals("blockSize")) blocksize = Long.parseLong(fieldValue);
                        if (fieldName.equals("replication")) replicationFactor = Integer.parseInt(fieldValue);
                        if (fieldName.equals("aclBit")) aclBit = Boolean.parseBoolean(fieldValue);
                        if (fieldName.equals("msExpirationTime")) {
                            long expiryms = Long.parseLong(fieldValue);
                            expiryTime = (expiryms > 0) ? new Date(expiryms) : null;
                        }
                    }
                    if (jp.getCurrentToken() == JsonToken.END_ARRAY) {
                        break;
                    }
                    jp.nextToken();
                }
                jp.nextToken(); //  END_OBJECT - }  // FileStatus
                jp.nextToken(); //  END_OBJECT - }  // FileStatuses
                jp.close();
                return list;
            } catch (IOException ex) {
                resp.successful = false;
                resp.message = "Unexpected error happened reading response stream or parsing JSon from listFiles()";
            } finally {
                try {
                    resp.responseStream.close();
                } catch (IOException ex) {
                    //swallow since it is only the closing of the stream
                }
            }
        }
        return null;
    }

    /**
     * sets one or both of the times (Modified and Access time) of the file or directory
     *
     * @param path the full path of the file or directory to {@code touch}
     * @param atime Access time as a long
     * @param mtime Modified time as a long
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void setTimes(String path,
                                long atime,
                                long mtime,
                                ADLStoreClient client,
                                RequestOptions opts,
                                OperationResponse resp) {
        if (atime < -1) {
            resp.message = "Invalid Access Time specified";
            resp.successful = false;
            return;
        }

        if (mtime < -1) {
            resp.message = "Invalid Modification Time specified";
            resp.successful = false;
            return;
        }

        if (atime == -1 && mtime == -1) {
            resp.message = "Access time and Modification time cannot both be unspecified";
        }

        QueryParams qp = new QueryParams();
        if (mtime != -1 ) qp.add("modificationtime", Long.toString(mtime));
        if (atime != -1 ) qp.add("accesstime",       Long.toString(atime));

        HttpTransport.makeCall(client, Operation.SETTIMES, path, qp, null, 0, 0, opts, resp);
    }

    /**
     * sets the owning user and group of the file. If the user or group are {@code null}, then they are not changed.
     * It is illegal to pass both user and owner as {@code null}.
     *
     * @param path the full path of the file
     * @param user the ID of the user, or {@code null}
     * @param group the ID of the group, or {@code null}
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void setOwner(String path,
                                String user,
                                String group,
                                ADLStoreClient client,
                                RequestOptions opts,
                                OperationResponse resp) {
        // at least one of owner or user must be set
        if (       (user == null  || user.equals(""))
                && (group == null || group.equals(""))
                ) {
            resp.successful = false;
            resp.message = "Both user and owner names cannot be blank";
            return;
        }

        QueryParams qp = new QueryParams();
        if (user!=null && !user.equals("")) {
            qp.add("owner", user);
        }
        if (group!=null && !group.equals("")) {
            qp.add("group", group);
        }

        HttpTransport.makeCall(client, Operation.SETOWNER, path, qp, null, 0, 0, opts, resp);
    }


    /**
     * Sets the permissions of the specified file ro directory. This sets the traditional unix read/write/execute
     * permissions for the file/directory. To set Acl's use the
     * {@link #setAcl(String, List, ADLStoreClient, RequestOptions, OperationResponse) setAcl} call.
     *
     *
     * @param path the full path of the file or directory ro set permissions for
     * @param octalPermissions the permissions to set, in unix octal form. For example, '644'.
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void setPermission(String path,
                                     String octalPermissions,
                                     ADLStoreClient client,
                                     RequestOptions opts,
                                     OperationResponse resp) {
        if (!isValidOctal(octalPermissions)) {
            resp.message = "Specified permissions are not valid Octal Permissions: " + octalPermissions;
            resp.successful = false;
            return;
        }

        QueryParams qp = new QueryParams();
        qp.add("permission", octalPermissions);

        HttpTransport.makeCall(client, Operation.SETPERMISSION, path, qp, null, 0, 0, opts, resp);
    }

    private static final Pattern octalPattern = Pattern.compile("[01]?[0-7]?[0-7]?[0-7]");
    public static boolean isValidOctal(String input) {
        return octalPattern.matcher(input).matches();
    }

    /**
     * checks whether the calling user has the required permissions for the file. the permissions to check
     * should be specified in the rwx parameter, as a unix permission string.
     *
     * @param path the full path of the file or directory to check
     * @param rwx the permission to check for, in rwx string form. The call returns true if the caller has
     *            all the requested permissions. For example, specifying {@code "r-x"} succeeds if the caller has
     *            read and execute permissions.
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void checkAccess(String path,
                                   String rwx,
                                   ADLStoreClient client,
                                   RequestOptions opts,
                                   OperationResponse resp) {
        if (rwx == null || rwx.trim().equals("")) {
            resp.message = "null or empty access specification passed in to check access for";
            resp.successful = false;
            return;
        }

        if (!isValidRwx(rwx)) {
            resp.message = "invalid access specification passed in to check access for: " + rwx;
            resp.successful = false;
            return;
        }

        QueryParams qp = new QueryParams();
        qp.add("fsaction", rwx);

        HttpTransport.makeCall(client, Operation.CHECKACCESS, path, qp, null, 0, 0, opts, resp);
    }

    private static final Pattern rwxPattern = Pattern.compile("[r-][w-][x-]");
    private static boolean isValidRwx(String input) {
        input = input.trim().toLowerCase();
        return rwxPattern.matcher(input).matches();
    }

    /**
     * Modify the acl entries for a file or directory. This call merges the supplied list with
     * existing ACLs. If an entry with the same scope, type and user already exists, then the permissions
     * are replaced. If not, than an new ACL entry if added.
     *
     *
     * @param path the path of the file or directory whose ACLs should be modified
     * @param aclSpec aclspec string containing the entries to add or modify
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void modifyAclEntries(String path,
                                        String aclSpec,
                                        ADLStoreClient client,
                                        RequestOptions opts,
                                        OperationResponse resp) {
        if (aclSpec == null || aclSpec.trim().equals("")) {
            resp.message = "null or empty AclSpec passed in to modifyAclEntries";
            resp.successful = false;
            return;
        }

        QueryParams qp = new QueryParams();
        qp.add("aclspec", aclSpec);

        HttpTransport.makeCall(client, Operation.MODIFYACLENTRIES, path, qp, null, 0, 0, opts, resp);
    }

    /**
     * Modify the acl entries for a file or directory. This call merges the supplied list with
     * existing ACLs. If an entry with the same scope, type and user already exists, then the permissions
     * are replaced. If not, than an new ACL entry if added.
     *
     *
     * @param path the path of the file or directory whose ACLs should be modified
     * @param aclSpec {@link List} of {@link AclEntry}s, containing the entries to add or modify
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void modifyAclEntries(String path,
                                        List<AclEntry> aclSpec,
                                        ADLStoreClient client,
                                        RequestOptions opts,
                                        OperationResponse resp) {
        if (aclSpec == null || aclSpec.size() == 0) {
            resp.message = "null or empty AclSpec passed in to modifyAclEntries";
            resp.successful = false;
            return;
        }

        QueryParams qp = new QueryParams();
        qp.add("aclspec", AclEntry.aclListToString(aclSpec));

        HttpTransport.makeCall(client, Operation.MODIFYACLENTRIES, path, qp, null, 0, 0, opts, resp);
    }

    /**
     * Removes the specified ACL entries from a file or directory.
     *
     * @param path the fll path of the file or directory to remove ACLs from
     * @param aclSpec aclspec string containing entries to remove
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void removeAclEntries(String path,
                                        String aclSpec,
                                        ADLStoreClient client,
                                        RequestOptions opts,
                                        OperationResponse resp) {
        if (aclSpec == null || aclSpec.trim().equals("")) {
            resp.message = "null or empty AclSpec passed in to removeAclEntries";
            resp.successful = false;
            return;
        }

        QueryParams qp = new QueryParams();
        qp.add("aclspec", aclSpec);

        HttpTransport.makeCall(client, Operation.REMOVEACLENTRIES, path, qp, null, 0, 0, opts, resp);
    }


    /**
     * Removes the specified ACL entries from a file or directory.
     *
     * @param path the fll path of the file or directory to remove ACLs from
     * @param aclSpec {@link List} of {@link AclEntry}s to remove
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void removeAclEntries(String path,
                                        List<AclEntry> aclSpec,
                                        ADLStoreClient client,
                                        RequestOptions opts,
                                        OperationResponse resp) {
        if (aclSpec == null || aclSpec.size() == 0) {
            resp.message = "null or empty AclSpec passed in to removeAclEntries";
            resp.successful = false;
            return;
        }

        QueryParams qp = new QueryParams();
        qp.add("aclspec", AclEntry.aclListToString(aclSpec, true));

        HttpTransport.makeCall(client, Operation.REMOVEACLENTRIES, path, qp, null, 0, 0, opts, resp);
    }

    /**
     * removes all default acl entries from a directory. The access ACLs for the directory itself are
     * not modified.
     *
     * @param path the full path of the directory from which to remove default ACLs
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void removeDefaultAcl(String path,
                                        ADLStoreClient client,
                                        RequestOptions opts,
                                        OperationResponse resp) {

        HttpTransport.makeCall(client, Operation.REMOVEDEFAULTACL, path, null, null, 0, 0, opts, resp);
    }

    /**
     * removes all acl entries from a file or directory.
     *
     * @param path the full path of the file or directory from which to remove ACLs
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void removeAcl(String path,
                                 ADLStoreClient client,
                                 RequestOptions opts,
                                 OperationResponse resp) {

        HttpTransport.makeCall(client, Operation.REMOVEACL, path, null, null, 0, 0, opts, resp);
    }

    /**
     * Sets the ACLs for a file or directory. If the file or directory already has any ACLs
     * associated with it, then all the existing ACLs are removed before adding the specified
     * ACLs.
     *
     * @param path the full path to the file or directory to set ACLs for.
     * @param aclSpec posix aclspec string containing the ACLs to set
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void setAcl(String path,
                              String aclSpec,
                              ADLStoreClient client,
                              RequestOptions opts,
                              OperationResponse resp) {
        if (aclSpec == null || aclSpec.trim().equals("")) {
            resp.message = "null or empty AclSpec passed in to setAcl";
            resp.successful = false;
            return;
        }

        QueryParams qp = new QueryParams();
        qp.add("aclspec", aclSpec);

        HttpTransport.makeCall(client, Operation.SETACL, path, qp, null, 0, 0, opts, resp);
    }

    /**
     * Sets the ACLs for a file or directory. If the file or directory already has any ACLs
     * associated with it, then all the existing ACLs are removed before adding the specified
     * ACLs.
     *
     * @param path the full path to the file or directory to set ACLs for.
     * @param aclSpec {@link List} of {@link AclEntry}s to set
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     */
    public static void setAcl(String path,
                              List<AclEntry> aclSpec,
                              ADLStoreClient client,
                              RequestOptions opts,
                              OperationResponse resp) {
        if (aclSpec == null || aclSpec.size() == 0) {
            resp.message = "null or empty AclSpec passed in to setAcl";
            resp.successful = false;
            return;
        }

        setAcl(path, AclEntry.aclListToString(aclSpec), client, opts, resp);
    }

    /**
     * Gets the current ACLs and permissions associated with a file or directory. Also returns the
     * current owning user and group for the file or directory.
     *
     * @param path the full path of the file or directory to get ACLs and permissions for
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     *
     * @return {@link AclStatus} object containing the ACLs, permissions and owners of the file or directory
     */
    public static AclStatus getAclStatus(String path,
                                         ADLStoreClient client,
                                         RequestOptions opts,
                                         OperationResponse resp) {
        return getAclStatus(path, null, client, opts, resp);
    }

    /**
     * Gets the current ACLs and permissions associated with a file or directory. Also returns the
     * current owning user and group for the file or directory.
     *
     * @param path the full path of the file or directory to get ACLs and permissions for
     * @param oidOrUpn {@link UserGroupRepresentation} enum specifying whether to return user and group information as
     *                                                OID or UPN
     * @param client the {@link ADLStoreClient}
     * @param opts options to change the behavior of the call
     * @param resp response from the call, and any error info generated by the call
     *
     * @return {@link AclStatus} object containing the ACLs, permissions and owners of the file or directory
     */
    public static AclStatus getAclStatus(String path,
                                         UserGroupRepresentation oidOrUpn,
                                         ADLStoreClient client,
                                         RequestOptions opts,
                                         OperationResponse resp) {
        QueryParams qp = new QueryParams();

        if (oidOrUpn == null) oidOrUpn = UserGroupRepresentation.OID;
        String tooid = (oidOrUpn == UserGroupRepresentation.OID)? "true" : "false";
        qp.add("tooid", tooid);

        HttpTransport.makeCall(client, Operation.GETACLSTATUS, path, qp, null, 0, 0, opts, resp);

        if (resp.successful) {
            AclStatus status = new AclStatus();
            ArrayList<AclEntry> list = new ArrayList<AclEntry>();
            status.aclSpec = list;
            try {
                JsonFactory jf = new JsonFactory();
                JsonParser jp = jf.createParser(resp.responseStream);
                String fieldName, fieldValue;

                jp.nextToken();  // START_OBJECT - {
                jp.nextToken();  // FIELD_NAME - "AclStatus":
                jp.nextToken();  // START_OBJECT - {
                jp.nextToken();
                while (jp.hasCurrentToken()) {
                    if (jp.getCurrentToken() == JsonToken.FIELD_NAME) {
                        fieldName = jp.getCurrentName();
                        if (fieldName.equals("entries")) {
                            jp.nextToken();  // START_ARRAY - [
                            jp.nextToken();
                            while (jp.hasCurrentToken() && jp.getCurrentToken() != JsonToken.END_ARRAY) {
                                String aclEntryString = jp.getText();
                                AclEntry aclEntry = AclEntry.parseAclEntry(aclEntryString);
                                list.add(aclEntry);
                                jp.nextToken();
                            }
                            jp.nextToken();  // current token is END_ARRAY, go to next token to continue with loop
                            continue;
                        }
                        jp.nextToken();  // get the value of the field
                        fieldValue = jp.getText();
                        if (fieldName.equals("group")) status.group = fieldValue;
                        if (fieldName.equals("owner")) status.owner = fieldValue;
                        if (fieldName.equals("permission")) status.octalPermissions = fieldValue;
                        if (fieldName.equals("stickyBit")) status.stickyBit = Boolean.valueOf(fieldValue);
                    }
                    jp.nextToken();
                }
                jp.close();
                return status;
            } catch (IOException ex) {
                resp.successful = false;
                resp.message = "Unexpected error happened reading response stream or parsing JSon from getAclStatus";
                return null;
            } finally {
                try {
                    resp.responseStream.close();
                } catch (IOException ex) {
                    //swallow since it is only the closing of the stream
                }
            }
        } else {
            return null;
        }
    }
}
