package com.mapr.security.client.examples;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import javax.net.ssl.HttpsURLConnection;

import java.net.HttpURLConnection;
import java.net.URL;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import java.security.cert.X509Certificate;
import java.util.List;
import java.util.Map;

import com.mapr.security.client.ClientSecurity;
import com.mapr.security.client.MapRClientSecurityException;

/*
 * The MapRClient class. This is a sample application used to illustrate how to invoke the MapR 
 * ticket-based authentication client APIs in the com.mapr.security.client.ClientSecurity class. 
 * Used to execute REST APIs in the command line.
 *
 * Sample invocation sequences:
 * 
 * export CLASSPATH=/opt/mapr/lib/maprfs-6.0.0-mapr.jar:/opt/mapr/lib/slf4j-api-1.7.12.jar:/opt/mapr/lib/protobuf-java-2.5.0.jar:/opt/mapr/lib/guava-14.0.1.jar:..:.
 * Invoking with no parameters will display a usage statement:
 * # java -cp $CLASSPATH com.mapr.security.client.MapRClient
 * List of commands: 
 *
 * gettoken   	Retrieves an authentication token that can be used in REST API authentication requests
 * exec   	Calls the REST API given in the URL, given the token
 * authtest   	test authentication as a generic client
 *
 * Testing if the ticket is valid:
 * # java -cp $CLASSPATH com.mapr.security.client.MapRClient authtest
 * Attempting to pick up credentials for cluster chyelin96.cluster.com
 * Using ticket file at /tmp/maprticket_0
 * 0 [main] DEBUG com.mapr.security.client.ClientSecurity  - Already have good ticket, done
 * Found a ticket that appears valid.
 *
 * To perform the challenge-response authentication handshake:
 * # java -cp $CLASSPATH com.mapr.security.client.examples.MapRClient challengeresponse -url https://10.10.30.96:11443/oozie/v1/admin/java-sys-properties 
 * Using cluster name chyelin96.cluster.com
 * 0 [main] DEBUG com.mapr.security.client.ClientSecurity  - Generating challenge for cluster chyelin96.cluster.com
 * 69 [main] DEBUG com.mapr.security.client.ClientSecurity  - Already have good ticket, done
 * 77 [main] DEBUG com.mapr.security.client.ClientSecurity  - Successfully obtained challenge
 * 77 [main] DEBUG com.mapr.security.client.examples.MapRClient  - Sending challenge to server
 * 168 [main] DEBUG com.mapr.security.client.ClientSecurity  - Response to challenge found
 * 168 [main] DEBUG com.mapr.security.client.ClientSecurity  - Successfully validated server response
 *
 * To perform the challenge-response handshake and retrieve an authentication token for future REST API calls (applicable
 * only to MapR services using the standard MapR authentication handler):
 * 
 * # java -cp $CLASSPATH com.mapr.security.client.examples.MapRClient gettoken -url https://10.10.30.96:11443/oozie/v1/admin/java-sys-properties
 * 0 [main] DEBUG com.mapr.security.client.ClientSecurity  - Evaluating challenge for cluster chyelin96.cluster.com
 * 68 [main] DEBUG com.mapr.security.client.ClientSecurity  - Already have good ticket, done
 * 77 [main] DEBUG com.mapr.security.client.ClientSecurity  - Successfully obtained challenge
 * 77 [main] DEBUG com.mapr.security.client.ClientSecurity  - Sending challenge to server
 * 169 [main] DEBUG com.mapr.security.client.ClientSecurity  - Obtained server response
 * 169 [main] DEBUG com.mapr.security.client.ClientSecurity  - Response to challenge found
 * 169 [main] DEBUG com.mapr.security.client.ClientSecurity  - Successfully validated server response
 * 169 [main] DEBUG com.mapr.security.client.ClientSecurity  - Extracting token from response message
 * 169 [main] DEBUG com.mapr.security.client.ClientSecurity  - Found token in response message
 * Obtained token: "u=root&p=root&t=multiauth&e=1514281723390&s=8CTnnUeMvVYzDqeP4p85DE2JPKo="
 *
 * Display the invocation sequence of the &quot;exec&quot; command which inserts the authentication token in the REST URL
 * and returns the result of executing the command:
 * # java -cp $CLASSPATH com.mapr.security.client.examples.MapRClient exec \
 *     -url https://10.10.30.96:11443/oozie/v1/admin/java-sys-properties
 * exec
 *   [ -cluster mapr cluster name ]
 *   [ -url rest-URL -token auth-token -request GET|POST|PUT|DELETE ]
 *
 * # java -cp $CLASSPATH com.mapr.security.client.MapRClient gettoken
 * gettoken
 *   [ -cluster mapr cluster name ]
 *   [ -url rest-URL ]
 *   
 * Example of using the &quot;exec&quot; command which inserts the authentication token in the REST URL
 * and returns the result of executing the command:
 * # java -cp $CLASSPATH com.mapr.security.client.MapRClient exec \
 *     -url https://10.10.30.96:11443/oozie/v1/admin/status \
 *     -token 'u=root&p=root&t=multiauth&e=1514281723390&s=8CTnnUeMvVYzDqeP4p85DE2JPKo='
 * {"systemMode":"NORMAL"}
 */
public class MapRClient {
  //private static final Logger LOG = LoggerFactory.getLogger(MapRClient.class);
  private static final String EXEC = "exec";
  private static final String EXEC_HELP = "Calls the REST API given in the URL, given the token";
    
  private static final String GET_TOKEN = "gettoken";
  private static final String GET_TOKEN_HELP = "Performs the MapR challenge-response handshake, and retrieves an \n\t\t\tauthentication token that can be used in REST API authentication requests";

  private static final String DO_CHALLENGE_RESPONSE = "challengeresponse";
  private static final String DO_CHALLENGE_RESPONSE_HELP = "Performs the MapR challenge-response handshake only";
  
  private static final String AUTHTEST = "authtest";
  private static final String AUTHTEST_HELP = "test authentication as a generic client";
    
  private static boolean verbose = false;
  private static String clusterName;
  
  /**
   * Name of the HTTP cookie used for the authentication token between the client 
   * and the server.  The value of the authentication cookie is the token returned 
   * from the MapR server after a successful challenge-response handshake. A sample
   * token looks like this:
   * <pre>
   * {@code
   * u=root&p=root&t=multiauth&e=1514281723390&s=8CTnnUeMvVYzDqeP4p85DE2JPKo=
   * }
   * </pre>
   * The authentication cookie header looks something like this:
   * <pre>
   * {@code
   * Cookie: hadoop.auth="u=root&p=root&t=multiauth&e=1514281723390&s=8CTnnUeMvVYzDqeP4p85DE2JPKo="
   * }
   * </pre>
   * The injectToken() method will inject the authentication token into the
   * REST API header. You may, however, choose to use any other method to add this
   * header to the HTTP REST API message yourself:
   * @see injectToken
   */
  public static final String AUTH_COOKIE = "hadoop.auth";

  private static final String AUTH_COOKIE_EQ = AUTH_COOKIE + "=";
  
  private HttpURLConnection conn;
  
  private static ClientSecurity cs;
  /**
   * The token class. This is the client side MapR authentication token
   * obtained from the server after a succesful challenge-response negotiation
   * protocol. It is outside the scope of this API to store the token for
   * subsequent use. It is the client application's responsibility to securely
   * store this token for use with subsequent REST API, and to obtain a new
   * token using the challenge-response protocol again when the current token 
   * has expired.
   *
   * A sample token looks like this:
   * <pre>
   * {@code
   * u=root&p=root&t=multiauth&e=1514281723390&s=8CTnnUeMvVYzDqeP4p85DE2JPKo=
   * }
   * </pre>
   */
  public static class Token {

    private String token;

    /**
     * Creates a token.
     */
    public Token() {
    }

    /**
     * Creates a token using an existing string representation of the token.
     *
     * @param tokenStr string representation of the tokenStr.
     */
    public Token(String tokenStr) {
      if (tokenStr == null) {
        throw new IllegalArgumentException("tokenStr cannot be null");
      }
      set(tokenStr);
    }

    /**
     * Determines if a token from the server has been set.
     *
     * @return  This method returns true if a token from the server has been set,
     *          false otherwise.
     */
    public boolean isSet() {
      return token != null;
    }

    /**
     * Sets a token.
     *
     * @param tokenStr string representation of the tokenStr.
     */
    void set(String tokenStr) {
      token = tokenStr;
    }

    /**
     * Returns the string representation of the token. This is the value that should be
     * included in the REST API calls to a MapR server in the authentication cookie.
     * A sample token looks like this:
     *
     * <pre>
     * {@code
     * u=root&p=root&t=multiauth&e=1514281723390&s=8CTnnUeMvVYzDqeP4p85DE2JPKo=
     * }
     * </pre>
     *
     * The authentication cookie header looks something like this:
     * <pre>
     * {@code
     * Cookie: hadoop.auth="&lt;token&gt;"
     * }
     * </pre>
     *
     * @return the string representation of the token.
     * @see injectToken
     */
    @Override
    public String toString() {
      return token;
    }
  }

    
  private MapRClient() {
    // private const
  }
    
  /**
   * Obtain the token from the HTTP response message. This is how MapR services using the standard
   * MapR authentication handler work.
   * 
   * If the server successfully authenticates the challenge, it will return a successful HTTP 200 OK 
   * response message, along with the response to the challenge (the second part of the challenge-
   * response handshake) in the Authorization header in the MAPR-Negotiate property. The client validates
   * the response to the challenge, and this is the scope of the client API for MapR ticket authentication.
   * 
   * For MapR services using the standard MapR authentication handler, the MapR authentication token is
   * also included in the server response message, in the "Set-Cookie" header, in the hadoop.auth property. 
   * A sample response message from the MapR server after a successful challenge-response handshake looks 
   * like this:
   * <pre>
   * {@code
   *  HTTP/1.1 200 OK
   *  Authorization: MAPR-Negotiate M0GtfTQNSVVzOOtTFI8cgA5MON5cb1beloL+C3k6tl9lY16VRrA6GEJCSag=
   *  Set-Cookie hadoop.auth="u=root&p=root&t=multiauth&e=1514275515051&s=0Zt1Zw5eAbVwvgoKqAncWfbZLiY="; 
   *             Path=/; Domain=.lab; Expires=Tue, 26-Dec-2017 08:05:15 GMT; Secure; HttpOnly
   * }
   * </pre>
   * 
   * It is possible that not all MapR services will implement the Set-Cookie property, so this code
   * is only included as an example. 
   * 
   * The method will first validate the server's response to the challenge contained in the 
   * Authorization header in the MAPR-Negotiate value, and if successful, return the token from
   * the Set-Cookie header in the hadoop.auth property to the caller.
   *
   * @param conn    The HTTP connection
   * @param url     The URL
   * @param token   Upon successful return, this will contain the token returned from the server
   * @throws MapRClientSecurityException  Error obtaining token
   * @throws IOException  I/O Exception
   */
  public static void getTokenFromResponseMessage (HttpURLConnection conn, URL url, Token token) 
    throws MapRClientSecurityException, IOException {
    if (conn == null) {
      conn = (HttpsURLConnection) url.openConnection();
    }
    /*
     * Get the server response
     */
    String base64EncodedAuthResponse = readResponse (conn);
    
    System.out.println ("Obtained Base-64 encoded response " + base64EncodedAuthResponse);
    cs.validateServerResponseToChallenge (base64EncodedAuthResponse);
    if (token != null) {
      /* 
       * Extract the authentication token from the connection
       * and stash it into the local variable - token. If token
       * is set we will not perform the handshake again
       */
      System.out.println("Extracting token from response message");
      extractToken(conn, token);
    }
  }

  /**
   * Function reads the response once client connects to the server.
   * Decodes the Base64 encoded string into bytes and constructs the
   * AuthenticationResp class from these bytes and returns this to caller
   * @return
   * @throws IOException
   * @throws AuthenticationException
   */
  private static String readResponse(HttpURLConnection conn) 
	throws IOException, MapRClientSecurityException {
	int status = conn.getResponseCode();

    /* Check to see if the error header is set
     * If its set, throw an exception and get out
     */
    String authorizationError = conn.getHeaderField(ClientSecurity.WWW_ERR_AUTHENTICATE);
    if (authorizationError != null)
    {
      String err = authorizationError.trim();
      throw new MapRClientSecurityException("Exception in server: " + err);
    }
    /* 
     * Verify that the server responded with 200. If it did it would mean
     * that the server has authenticated the client and generated the
     * authentication token. Client still has to make sure that it
     * verifies the server and whether to continue communication to the
     * server or not
     */
    if (status == HttpURLConnection.HTTP_OK)
    {
      String authHeader = null;
      Map<String, List<String>> headers = conn.getHeaderFields();
      List<String> wwwAuthHeaders = headers.get(ClientSecurity.AUTHORIZATION);
      if ( wwwAuthHeaders == null ) {
        throw new MapRClientSecurityException("No header : " +  ClientSecurity.AUTHORIZATION + " is present");
      }
      for ( String header : wwwAuthHeaders ) {
        if ( header != null && header.trim().startsWith(ClientSecurity.NEGOTIATE) ) {
          authHeader = header;
        }
      }
      if (authHeader == null ) {
        throw new MapRClientSecurityException("Invalid sequence, incorrect header" + wwwAuthHeaders);
      }

      String negotiation = authHeader.trim().substring((ClientSecurity.NEGOTIATE + " ").length()).trim();
      return negotiation;
    }     
    else
    {
      /* Invalid status */
      throw new MapRClientSecurityException("Incorrect status" + status);
    }
  }
  /*
   * Helper method that extracts an authentication token received from a connection.
   * The authentication token is found in the Set-Cookie header under the hadoop.auth 
   * property. An example of how the Set-Cookie header looks like is given below:
   * 
   * Set-Cookie: hadoop.auth="u=root&p=root&t=multiauth&e=1514276503614&s=z9TStHcWwX1yZVAoTKVtEyw1/QA="; 
   *    Path=/; 
   *    Domain=.lab; 
   *    Expires=Tue, 26-Dec-2017 08:21:43 GMT; 
   *    Secure; 
   *    HttpOnly
   *
   * In this example, we get the token to the following value:
   * u=root&p=root&t=multiauth&e=1514276503614&s=z9TStHcWwX1yZVAoTKVtEyw1/QA=
   * 
   * @param conn connection to extract the authentication token from.
   * @param token The authentication token.
   *
   * @throws IOException if an IO error occurred.
   * @throws MapRClientSecurityException if an authentication exception occurred.
   */
  private static void extractToken(HttpURLConnection conn, Token token) 
      throws IOException, MapRClientSecurityException {
    int respCode = conn.getResponseCode();
    if (respCode == HttpURLConnection.HTTP_OK
        || respCode == HttpURLConnection.HTTP_CREATED
        || respCode == HttpURLConnection.HTTP_ACCEPTED) {
      Map<String, List<String>> headers = conn.getHeaderFields();
      List<String> cookies = headers.get("Set-Cookie");
      if (cookies != null) {
        for (String cookie : cookies) {
        	  /*
        	   * Look for the authentication cookie AUTH_COOKIE "hadoop.auth"
        	   */
          if (cookie.startsWith(AUTH_COOKIE_EQ)) {
            String value = cookie.substring(AUTH_COOKIE_EQ.length());
            int separator = value.indexOf(";");
            if (separator > -1) {
              value = value.substring(0, separator);
            }
            if (value.length() > 0) {
            	  System.out.println("Found token in response message");
              token.set(value);
            }
          }
        }
      }
    } else {
      token.set(null);
      throw new MapRClientSecurityException("Authentication failed, status: " + conn.getResponseCode() +
                                        ", message: " + conn.getResponseMessage());
    }
  }
  /**
   * Injects an authentication token to send with a connection. This sets the cookie request property
   * with hadoop-auth set to the value of the token obtained during the negotiation phase. For example:
   * <pre>
   * {@code
   * Cookie: hadoop.auth="u=root&p=root&t=multiauth&e=1514276503614&s=z9TStHcWwX1yZVAoTKVtEyw1/QA="
   * }
   * </pre>
   * @param conn  The connection.
   * @param token The token to inject into the request header.
   */
  public static void injectToken(HttpURLConnection conn, Token token) {
    String t = token.token;
    if (t != null) {
      if (!t.startsWith("\"")) {
        t = "\"" + t + "\"";
      }
      conn.addRequestProperty("Cookie", AUTH_COOKIE_EQ + t);
    }
  }

  /**
   * Determines whether we are in the negotiation phase.
   * 
   * We are in the negotiation phase if we have an open HTTP connection and the response code
   * is 401 (Unauthorized). The response header fields should contain the WWW-Authenticate property
   * with the value starting with MAPR-Negotiate. For example: 
   * <pre>
   * {@code
   * HTTP/1.1 401 Unauthorized
   * WWW-Authenticate: MAPR-Negotiate
   * }
   * </pre>
   * @throws IOException I/O Exception
   * @return This method returns true if we are in the negotiation phase, false otherwise.
   */
  public boolean isNegotiate() throws IOException {
    boolean negotiate = false;
    if (conn.getResponseCode() == HttpURLConnection.HTTP_UNAUTHORIZED) {
      Map<String, List<String>> headers = conn.getHeaderFields();
      List<String> wwwAuthHeaders = headers.get(ClientSecurity.WWW_AUTHENTICATE);
      if ( wwwAuthHeaders != null ) {
        for ( String authHeader : wwwAuthHeaders ) {
          if ( authHeader != null && authHeader.trim().startsWith(ClientSecurity.NEGOTIATE) ) {
            return true;
          }
        }
      }
    }
    return negotiate;
  }
  /*
   * Function sets the HTTP header "Authorization" with the passed in string
   * and sends the challenge to the server
   *
   * @param conn  The HTTP URL connection object
   * @param url   The server URL.
   * @param authorizationToken - HTTP authorization header value to be set
   * @throws IOException
   * @throws MapRClientSecurityException
   */
  private static void sendChallenge(HttpURLConnection conn, URL url, String authorizationToken) 
    throws IOException, MapRClientSecurityException {
    System.out.println("Sending challenge to server");
    if (conn == null) {
      conn = (HttpsURLConnection) url.openConnection();
    }
    conn.setRequestMethod("OPTIONS");
    conn.setRequestProperty(ClientSecurity.AUTHORIZATION, ClientSecurity.NEGOTIATE + " " + authorizationToken);
    conn.connect();
  }
  
  /*
   * This function does the brunt of the work for HTTP authentication on the client side.
   * It completes the authentication in two passes.
   * 
   * In the first pass, we get the ticket information from the client machine, given
   * the cluster name. This is used to generate a challenge request which is sent to the server
   * at the given URL. The server should be able to decrypt this message and respond with the 
   * appropriate challenge response. At this point we are done with our first pass.
   *
   * During the next pass we verify the challenge response sent by the server and simply send an
   * acknowledgement to the server indicating the challenge response is validated and client has
   * completed the authentication sequence.
   *
   * @param conn         The HTTP connection
   * @param url          The REST API URL 
   * @param getToken     Set to true to both validate the response and get the authentication token 
   *                     from the response message that can be used for future REST API requests. If
   *                     set to false, this method will only do the MapR challenge-response handshake
   *                     and throw an exception if the handshake fails, but will not retrieve the token
   * @param token        Once authenticated this token can be used for future connections without
   *                     having to go through the negotiation protocol every time. Only set if 
   *                     getToken=true
   * @throws IOException I/O Exception
   * @throws MapRClientSecurityException Handshake failure
   */
  public static void doMaprHandshake (HttpURLConnection conn, URL url, boolean getToken, Token token) 
      throws IOException, MapRClientSecurityException {
    /*
     * Get the challenge string
     */
    String challengeString = cs.generateChallenge();
    System.out.println("Obtained challenge string " + challengeString);
    /*
     * Send the challenge string as an HTTP option using the "Authorization" property to the server
     */
    sendChallenge(conn, url, challengeString);
    /*
     * Extract the token from the response
     */
    if (getToken) {
      getTokenFromResponseMessage (conn, url, token);
    } else {
      getTokenFromResponseMessage (conn, url, null);
    }
  }
  
  private static void printGetTokenHelp() {
    StringBuilder builder = new StringBuilder();
    builder.append(GET_TOKEN);

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -cluster mapr cluster name ]");
      
    builder.append("\n");
    builder.append("\t");
    builder.append("[ -url rest-URL ]");
      
    System.out.println(builder.toString());
  }
  
  private static void printDoChallengeResponseHelp() {
    StringBuilder builder = new StringBuilder();
    builder.append(DO_CHALLENGE_RESPONSE);

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -cluster mapr cluster name ]");
      
    builder.append("\n");
    builder.append("\t");
    builder.append("[ -url rest-URL ]");
      
    System.out.println(builder.toString());
  }

  private static void printExecHelp() {
    StringBuilder builder = new StringBuilder();
    builder.append(EXEC);

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -cluster mapr cluster name ]");
        
    builder.append("\n");
    builder.append("\t");
    builder.append("[ -url rest-URL -token auth-token -request GET|POST|PUT|DELETE ]");
        
    System.out.println(builder.toString());
  }
    
  /*
   * Displays the help statement for the authtest command
   */
  private static void printAuthTestHelp() {
    StringBuilder builder = new StringBuilder();
    builder.append(AUTHTEST);

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -cluster mapr cluster name ]");

    System.out.println(builder.toString());
  }

  /*
   * Displays the command-specific help message for valid commands, or a general usage statement if
   * the command is invalid
   */
  private static void printHelp(String command) {
    if (command.equals(GET_TOKEN)) {
      printGetTokenHelp();
    } else if (command.equals(DO_CHALLENGE_RESPONSE)) {
    	  printDoChallengeResponseHelp();
    } else if (command.equals(EXEC)) {
      printExecHelp();
    } else if (command.equals(AUTHTEST)) {
      printAuthTestHelp();
    } else {
      printUsage();
    }
  }

  /*
   * Given the REST URL, perform the MapR challenge-response handshake using the MapR ticket, and 
   * optionally obtain the MapR authentication token that can be used in future REST API requests
   * to the MapR service for the validity of the token. 
   * 
   * Example of REST URL:
   * <pre>
   * {@code
   * https://10.10.20.61:8443/rest/node/list?columns=hostname
   * }
   * </pre>
   * 
   * Refer to the MapR online documentation for a full list of REST APIs.
   * 
   * @param restURL  The REST URL
   * @param getToken If set to true, perform the MapR challenge-response handshake and get the token. 
   *                 If set to false, perform the MapR challenge response handshake only, but do not
   *                 retrieve the authentication token. Note that although the MapR challenge-response
   *                 handshake is standard for all MapR services that support MapR ticket authentication,
   *                 it is possible that some services may have different ways of authenticating clients.
   */
  private static String doHandshakeWithToken(String restURL, boolean getToken, boolean verbose) {
    if (!cs.isClusterNameValid()) {
      System.out.println("ERROR: Invalid cluster name " + clusterName);
      return null;
    }
    TrustManager[] trustAllCerts = new TrustManager[] { 
      new X509TrustManager() {
        public java.security.cert.X509Certificate[] getAcceptedIssuers() { return null; }
        public void checkClientTrusted(X509Certificate[] certs, String authType) { }
        public void checkServerTrusted(X509Certificate[] certs, String authType) { }
      }
    };
    /*
     * Install the all-trusting trust manager
     */
    try {
      SSLContext sc = SSLContext.getInstance("SSL");
      sc.init(null, trustAllCerts, new java.security.SecureRandom());
      HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());
    } catch (Exception e) {
      System.out.println ("SSL exception: " + e.getMessage());
    }
    /*
     * Create all-trusting host name verifier
     */
    HostnameVerifier allHostsValid = new DefaultHostVerifier();
    /*
     * Install the all-trusting host verifier
     */
    HttpsURLConnection.setDefaultHostnameVerifier(allHostsValid);
    /*
     * Generate authentication header and validate response
     */
    try {
      URL url = new URL(restURL);
      HttpsURLConnection connection = (HttpsURLConnection)url.openConnection();
      if (getToken) { 
        Token token = new Token();
        doMaprHandshake (connection, url, true, token);
         System.out.println("Obtained token: " + token.toString());
        return token.toString();
      } else {
        doMaprHandshake (connection, url, false, null);
        return null;
      }
    } catch (Exception e) {
      System.out.println ("Exception occurred: " + e.getMessage());
    }
    return null;
  }
  /*
   * Execute the REST API specified in the URL restURL. 
   * 
   * If the -cluster argument is specified, then the cluster name specified in this argument will
   * be used. However, this will be overridden by the "cluster" argument in the REST URL query
   * string, if specified.
   * 
   * Example of REST URL:
   * <pre>
   * {@code
   * https://10.10.20.61:8443/rest/node/list?columns=hostname
   * }
   * </pre>
   * 
   * Refer to the MapR online documentation for a full list of REST APIs.
   */
  private static void execurl(String restURL, String restRequest, String authToken, boolean verbose) {
        
    if (!cs.isClusterNameValid()) {
      System.out.println("ERROR: Invalid cluster name " + clusterName);
      return;
    }
    TrustManager[] trustAllCerts = new TrustManager[] { 
      new X509TrustManager() {
        public java.security.cert.X509Certificate[] getAcceptedIssuers() { return null; }
        public void checkClientTrusted(X509Certificate[] certs, String authType) { }
        public void checkServerTrusted(X509Certificate[] certs, String authType) { }
      }
    };
    /*
     * Install the all-trusting trust manager
     */
    try {
      SSLContext sc = SSLContext.getInstance("SSL");
      sc.init(null, trustAllCerts, new java.security.SecureRandom());
      HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());
    } catch (Exception e) {
      System.out.println ("SSL exception");
      e.printStackTrace();
    }
    /*
     * Create all-trusting host name verifier
     */
    HostnameVerifier allHostsValid = new DefaultHostVerifier();
    /*
     * Install the all-trusting host verifier
     */
    HttpsURLConnection.setDefaultHostnameVerifier(allHostsValid);
    try {
      /*
       * If we get here, then we obtained the challenge
       * POST request to /login
       */
      Token token = new Token();
      token.set(authToken);
      URL url = new URL(restURL);
      HttpsURLConnection connection = (HttpsURLConnection)url.openConnection();
      /*
       * Next step is to embed the token in the header
       */
      connection.setRequestMethod(restRequest);
      injectToken(connection, token);
      connection.connect();
      
      BufferedReader br = new BufferedReader(new InputStreamReader(connection.getInputStream(), "UTF-8"));
      String inputLine;
      StringBuilder response = new StringBuilder();

      while ((inputLine = br.readLine()) != null)
        response.append(inputLine);
      System.out.println(response);
    } catch (Exception e) {
      System.out.println ("Exception occurred: " + e.getMessage());
    }
  }
  /*
   * Issues the REST API given in the parameter
   */
  private static void execute(String[] args) throws MapRClientSecurityException {
    String command = args[0];
    boolean clusterNameSpecified = false;
    boolean urlStringSet = false;
    boolean restRequestSet = false;
    String restRequestString = null;
    boolean authTokenSet = false;
    String authTokenString = null;
    String urlString = null;

    verbose = false;
    for (int i = 1; i < args.length; i++) {
      if (args[i].equals("-cluster") && i != args.length - 1) {
        clusterName = args[i+1];
        clusterNameSpecified = true;
        i++;
      } else if (args[i].equals("-url") && i != args.length - 1) {
        urlString = args[i+1];
        urlStringSet = true;
        i++;
      } else if (args[i].equals("-token") && i != args.length - 1) {
        authTokenString = args[i+1];
        authTokenSet = true;
        i++;
      } else if (args[i].equals("-request") && i != args.length - 1) {
        restRequestString = args[i+1];
        restRequestSet = true;
        i++;
      } else if (args[i].equals("-v") || (args[i].equals("--verbose"))) {
        verbose = true;
      } else {
        System.out.println("Invalid option: " + args[i]);
        printHelp(command);
        System.exit(1);
      }
    }
    /*
     * Use the default cluster if not specified
     */
    if (!clusterNameSpecified) {
      cs = new ClientSecurity();
    } else {
      cs = new ClientSecurity(clusterName);
    }

    clusterName = cs.getClusterName();
    System.out.println("Using cluster name " + clusterName);
    if (command.equals(AUTHTEST)) {
      System.out.println("Attempting to pick up credentials for cluster " + clusterName);
      System.out.println("Using ticket file at " + cs.getUserTicketAndKeyFileLocation());
 
      if (!cs.isClusterNameValid()) {
        System.out.println("Invalid cluster name " + clusterName);
        return;
      }
      if (cs.hasValidTicket()) {
        System.out.println("Found a ticket that appears valid.");
      }
      return;
    } else if (command.equals(DO_CHALLENGE_RESPONSE)) {
        if (verbose) {
          System.out.println("Using ticket file at " + cs.getUserTicketAndKeyFileLocation());
        }
        if (urlStringSet) {
          doHandshakeWithToken (urlString, false, verbose);
        } else {
          printGetTokenHelp ();
          System.exit(1);
        }
    } else if (command.equals(GET_TOKEN)) {
      if (verbose) {
        System.out.println("Using ticket file at " + cs.getUserTicketAndKeyFileLocation());
      }
      if (urlStringSet) {
        doHandshakeWithToken (urlString, true, verbose);
      } else {
        printGetTokenHelp ();
        System.exit(1);
      }
    } else if (command.equals(EXEC)) {
      if (verbose) {
        System.out.println("Using ticket file at " + cs.getUserTicketAndKeyFileLocation());
      }
      if (urlStringSet && authTokenSet) {
        /*
         * REST request defaults to GET if not specified
         */
        if (!restRequestSet) 
          restRequestString = "GET";
        if (verbose) 
          System.out.println("Using REST request " + restRequestString + " for URL " + urlString);
        execurl (urlString, restRequestString, authTokenString, verbose);
      } else {
        printExecHelp ();
        System.exit(1);
      }
    } else {
      printHelp(command);
      System.exit(1);
    }
  }
  
  private static void printUsage() {
    StringBuilder builder = new StringBuilder();
    builder.append("List of commands: \n");
      
    builder.append("\n");
    builder.append("  ");
    builder.append(DO_CHALLENGE_RESPONSE + "  ");
    builder.append("\t");
    builder.append(DO_CHALLENGE_RESPONSE_HELP);
      
    builder.append("\n");
    builder.append("  ");
    builder.append(GET_TOKEN + "  ");
    builder.append("\t\t");
    builder.append(GET_TOKEN_HELP);
      
    builder.append("\n");
    builder.append("  ");
    builder.append(EXEC + "  ");
    builder.append("\t\t");
    builder.append(EXEC_HELP);

    builder.append("\n");
    builder.append("  ");
    builder.append(AUTHTEST + "  ");
    builder.append("\t\t");
    builder.append(AUTHTEST_HELP);
    
    System.out.println(builder.toString());
  }

  public static void main(String args[]) {

    if (args == null || args.length < 1) {
      printUsage();
      return;
    }
    try {
      execute(args);
    } catch (MapRClientSecurityException e) {
      System.err.println(e.getMessage());
      System.exit(1);
    }
  }
}

