/*
The MIT License

Copyright (c) 2009 Infobright Inc.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.
*/

package com.infobright.io;

import java.io.IOException;



public class NamedPipeFactory {
 
  private final boolean isLinux;
  private final boolean isSolaris;
  private final boolean isWindows;
  private static final int defaultBufferSize = 102400;		// 100KB
  private static final int defaultTimeout = 10000;			// 10 sec
  
  //private final ClientProxy proxy;
  
  private final LinuxHelper linuxHelper;
  private final WindowsHelper windowsHelper;
  
  private NamedPipeFactory(String osName) {
    isLinux = osName.equalsIgnoreCase("linux");
    isSolaris = osName.equalsIgnoreCase("SunOS");
    isWindows = osName.toLowerCase().contains("win");
    linuxHelper = new LinuxHelper();
    windowsHelper = new WindowsHelper();
  }
  
  
  /**
   * Constructor to use for local connection.
   */
  public NamedPipeFactory() {
    this(System.getProperty("os.name"));
  }

  private class LinuxHelper {

    NamedPipe createPipe(String pipeName, int bufferSize)
        throws IOException {
        return new LinuxNamedPipe(pipeName, true, bufferSize);
    }
  }
  
  private class WindowsHelper {

    NamedPipe createServerPipe(String pipeName, boolean writeAllowed, boolean readAllowed, int bufsize, long timeout)
        throws IOException {
        return new WindowsNamedPipe(pipeName, writeAllowed, readAllowed, bufsize, timeout);
    }

    NamedPipe createClientPipe(String pipeName) throws IOException {
        return new WindowsNamedPipe(pipeName);
    }
  }

  public boolean isUnixMode() {
    return isLinux || isSolaris;
  }

  public boolean isWindowsMode() {
    return isWindows;
  }
  
  public NamedPipe createServer(String pipeName) throws IOException {
    return createServer(pipeName, true, true, defaultBufferSize, defaultTimeout);
  }
  
  public NamedPipe createServer(String pipeName, boolean writeAllowed, boolean readAllowed, int bufsize, long timeout) throws IOException {
    NamedPipe namedPipe;
    if (isUnixMode()) {
      namedPipe = linuxHelper.createPipe(pipeName, bufsize);
    } else if (isWindowsMode()) {
      namedPipe = windowsHelper.createServerPipe(pipeName, writeAllowed, readAllowed, bufsize, timeout);
    } else {
      throw new UnsupportedOperationException("Unsupported platform");
    }
    return namedPipe;
  }

  public NamedPipe createClient(String pipeName) throws IOException {
    NamedPipe namedPipe;
    if (isUnixMode()) {
      namedPipe = linuxHelper.createPipe(pipeName, defaultBufferSize);
    } else if (isWindowsMode()) {
      namedPipe = windowsHelper.createClientPipe(pipeName);
    } else {
      throw new UnsupportedOperationException("Unsupported platform");
    }
    return namedPipe;
  }
  
  public String getNativePipeName(String name) {
    if (isUnixMode()) {
      return LinuxNamedPipe.getNativeName(name);
    } else if (isWindowsMode()) {
      return WindowsNamedPipe.getNativeName(name);
    } else {
      throw new UnsupportedOperationException("Unsupported platform");
    }
  }
  
/*
  public PipeCallStrategy getStrategy(EtlLogger logger) {
    if (isUnixMode()) {
      return new UnixPipeCallStrategy(proxy, logger);
    } else if (isWindowsMode()) {
      return new WindowsPipeCallStrategy(proxy, logger);
    } else {
      throw new UnsupportedOperationException("Unsupported platform");
    }
  }*/
}
