/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.io.bigquery;

import com.google.api.services.bigquery.Bigquery;
import com.google.api.services.bigquery.model.ExternalDataConfiguration;
import com.google.api.services.bigquery.model.Job;
import com.google.api.services.bigquery.model.JobConfiguration;
import com.google.api.services.bigquery.model.JobConfigurationExtract;
import com.google.api.services.bigquery.model.JobConfigurationLoad;
import com.google.api.services.bigquery.model.JobReference;
import com.google.api.services.bigquery.model.Table;
import com.google.api.services.bigquery.model.TableReference;
import com.google.api.services.bigquery.model.TableSchema;
import com.google.cloud.hadoop.io.bigquery.BigQueryFileFormat;
import com.google.cloud.hadoop.io.bigquery.BigQueryStrings;
import com.google.cloud.hadoop.io.bigquery.BigQueryUtils;
import com.google.cloud.hadoop.util.ApiErrorExtractor;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.List;
import java.util.UUID;
import javax.annotation.Nullable;
import org.apache.hadoop.util.Progressable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BigQueryHelper {
    public static final String BIGQUERY_JOB_ID_PATTERN = "[a-zA-Z0-9_-]+";
    public static final int BIGQUERY_JOB_ID_MAX_LENGTH = 1024;
    protected static final Logger LOG = LoggerFactory.getLogger(BigQueryHelper.class);
    private ApiErrorExtractor errorExtractor = new ApiErrorExtractor();
    private Bigquery service;
    private static final Progressable NOP_PROGRESSABLE = new Progressable(){

        public void progress() {
        }
    };

    public BigQueryHelper(Bigquery service) {
        this.service = service;
    }

    public Bigquery getRawBigquery() {
        return this.service;
    }

    public void importFederatedFromGcs(String projectId, TableReference tableRef, @Nullable TableSchema schema, BigQueryFileFormat sourceFormat, List<String> gcsPaths) throws IOException {
        LOG.info("Importing into federated table '{}' from {} paths; path[0] is '{}'", new Object[]{BigQueryStrings.toString(tableRef), gcsPaths.size(), gcsPaths.isEmpty() ? "(empty)" : gcsPaths.get(0)});
        ExternalDataConfiguration externalConf = new ExternalDataConfiguration();
        externalConf.setSchema(schema);
        externalConf.setSourceUris(gcsPaths);
        externalConf.setSourceFormat(sourceFormat.getFormatIdentifier());
        if (schema == null) {
            LOG.info("No federated import schema provided, auto detecting schema.");
            externalConf.setAutodetect(Boolean.valueOf(true));
        } else {
            LOG.info("Using provided federated import schema '{}'.", (Object)schema.toString());
        }
        Table table = new Table();
        table.setTableReference(tableRef);
        table.setExternalDataConfiguration(externalConf);
        this.service.tables().insert(projectId, tableRef.getDatasetId(), table).execute();
    }

    public void importFromGcs(String projectId, TableReference tableRef, @Nullable TableSchema schema, BigQueryFileFormat sourceFormat, String writeDisposition, List<String> gcsPaths, boolean awaitCompletion) throws IOException, InterruptedException {
        LOG.info("Importing into table '{}' from {} paths; path[0] is '{}'; awaitCompletion: {}", new Object[]{BigQueryStrings.toString(tableRef), gcsPaths.size(), gcsPaths.isEmpty() ? "(empty)" : gcsPaths.get(0), awaitCompletion});
        JobConfigurationLoad loadConfig = new JobConfigurationLoad();
        loadConfig.setSchema(schema);
        loadConfig.setSourceFormat(sourceFormat.getFormatIdentifier());
        loadConfig.setSourceUris(gcsPaths);
        loadConfig.setDestinationTable(tableRef);
        loadConfig.setWriteDisposition(writeDisposition);
        if (schema == null) {
            LOG.info("No import schema provided, auto detecting schema.");
            loadConfig.setAutodetect(Boolean.valueOf(true));
        } else {
            LOG.info("Using provided import schema '{}'.", (Object)schema.toString());
        }
        JobConfiguration config = new JobConfiguration();
        config.setLoad(loadConfig);
        JobReference jobReference = this.createJobReference(projectId, "direct-bigqueryhelper-import");
        Job job = new Job();
        job.setConfiguration(config);
        job.setJobReference(jobReference);
        this.insertJobOrFetchDuplicate(projectId, job);
        if (awaitCompletion) {
            BigQueryUtils.waitForJobCompletion(this.getRawBigquery(), projectId, jobReference, NOP_PROGRESSABLE);
        }
    }

    public void exportBigQueryToGcs(String projectId, TableReference tableRef, List<String> gcsPaths, boolean awaitCompletion) throws IOException, InterruptedException {
        LOG.debug("exportBigQueryToGcs(bigquery, '{}', '{}', '{}', '{}')", new Object[]{projectId, BigQueryStrings.toString(tableRef), gcsPaths, awaitCompletion});
        LOG.info("Exporting table '{}' to {} paths; path[0] is '{}'; awaitCompletion: {}", new Object[]{BigQueryStrings.toString(tableRef), gcsPaths.size(), gcsPaths.isEmpty() ? "(empty)" : gcsPaths.get(0), awaitCompletion});
        JobConfigurationExtract extractConfig = new JobConfigurationExtract();
        extractConfig.setSourceTable(tableRef);
        extractConfig.setDestinationUris(gcsPaths);
        extractConfig.set("destinationFormat", (Object)"NEWLINE_DELIMITED_JSON");
        JobConfiguration config = new JobConfiguration();
        config.setExtract(extractConfig);
        JobReference jobReference = this.createJobReference(projectId, "direct-bigqueryhelper-export");
        Job job = new Job();
        job.setConfiguration(config);
        job.setJobReference(jobReference);
        this.insertJobOrFetchDuplicate(projectId, job);
        if (awaitCompletion) {
            BigQueryUtils.waitForJobCompletion(this.service, projectId, jobReference, NOP_PROGRESSABLE);
        }
    }

    public boolean tableExists(TableReference tableRef) throws IOException {
        try {
            Table fetchedTable = (Table)this.service.tables().get(tableRef.getProjectId(), tableRef.getDatasetId(), tableRef.getTableId()).execute();
            LOG.debug("Successfully fetched table '{}' for tableRef '{}'", (Object)fetchedTable, (Object)tableRef);
            return true;
        }
        catch (IOException ioe) {
            if (this.errorExtractor.itemNotFound(ioe)) {
                return false;
            }
            throw ioe;
        }
    }

    public Table getTable(TableReference tableRef) throws IOException {
        Bigquery.Tables.Get getTablesReply = this.service.tables().get(tableRef.getProjectId(), tableRef.getDatasetId(), tableRef.getTableId());
        return (Table)getTablesReply.execute();
    }

    public JobReference createJobReference(String projectId, String jobIdPrefix) {
        Preconditions.checkArgument((projectId != null ? 1 : 0) != 0, (Object)"projectId must not be null.");
        Preconditions.checkArgument((jobIdPrefix != null ? 1 : 0) != 0, (Object)"jobIdPrefix must not be null.");
        Preconditions.checkArgument((boolean)jobIdPrefix.matches(BIGQUERY_JOB_ID_PATTERN), (String)"jobIdPrefix '%s' must match pattern '%s'", (Object[])new Object[]{jobIdPrefix, BIGQUERY_JOB_ID_PATTERN});
        String fullJobId = String.format("%s-%s", jobIdPrefix, UUID.randomUUID().toString());
        Preconditions.checkArgument((fullJobId.length() <= 1024 ? 1 : 0) != 0, (String)"fullJobId '%s' has length '%s'; must be less than or equal to %s", (Object[])new Object[]{fullJobId, fullJobId.length(), 1024});
        return new JobReference().setProjectId(projectId).setJobId(fullJobId);
    }

    public void checkJobIdEquality(Job expected, Job actual) {
        Preconditions.checkState((actual.getJobReference() != null && actual.getJobReference().getJobId() != null && expected.getJobReference() != null && expected.getJobReference().getJobId() != null && actual.getJobReference().getJobId().equals(expected.getJobReference().getJobId()) ? 1 : 0) != 0, (String)"jobIds must match in '[expected|actual].getJobReference()' (got '%s' vs '%s')", (Object[])new Object[]{expected.getJobReference(), actual.getJobReference()});
    }

    public Job insertJobOrFetchDuplicate(String projectId, Job job) throws IOException {
        Preconditions.checkArgument((job.getJobReference() != null && job.getJobReference().getJobId() != null ? 1 : 0) != 0, (String)"Require non-null JobReference and JobId inside; getJobReference() == '%s'", (Object[])new Object[]{job.getJobReference()});
        Bigquery.Jobs.Insert insert = this.service.jobs().insert(projectId, job);
        Job response = null;
        try {
            response = (Job)insert.execute();
            LOG.debug("Successfully inserted job '{}'. Response: '{}'", (Object)job, (Object)response);
        }
        catch (IOException ioe) {
            if (this.errorExtractor.itemAlreadyExists(ioe)) {
                LOG.info(String.format("Fetching existing job after catching exception for duplicate jobId '%s'", job.getJobReference().getJobId()), (Throwable)ioe);
                response = (Job)this.service.jobs().get(projectId, job.getJobReference().getJobId()).execute();
            }
            LOG.info(String.format("Unhandled exception trying to insert job '%s'", job), (Throwable)ioe);
            throw ioe;
        }
        this.checkJobIdEquality(job, response);
        return response;
    }

    @VisibleForTesting
    void setErrorExtractor(ApiErrorExtractor errorExtractor) {
        this.errorExtractor = errorExtractor;
    }
}

