package com.carrotsearch.hppcrt.sets;

import com.carrotsearch.hppcrt.*;
import com.carrotsearch.hppcrt.cursors.*;
import com.carrotsearch.hppcrt.predicates.*;
import com.carrotsearch.hppcrt.procedures.*;
import com.carrotsearch.hppcrt.hash.*;

  
  
  
// If RH is defined, RobinHood Hashing is in effect
  

/**
 * A hash set of <code>Object</code>s, implemented using using open
 * addressing with linear probing for collision resolution.
 *
 * <p> In addition, the hashing strategy can be changed
 * by overriding ({@link #equalKeys(Object, Object)} and {@link #hashKey(Object)}) together,
 * which then replaces the usual ({@link #equals(Object)} and {@link #hashCode()}) from the keys themselves.
 * This is useful to define the equivalence of keys when the user has no control over the keys implementation.
 * </p>
 *
 * <p>
 * The internal buffers of this implementation ({@link #keys}, etc...)
 * are always allocated to the nearest size that is a power of two. When
 * the capacity exceeds the given load factor, the buffer size is doubled.
 * </p>
 * 
 *
 * <p>This implementation supports <code>null</code> keys.</p>
 * 
 * <p><b>Important note.</b> The implementation uses power-of-two tables and linear
 * probing, which may cause poor performance (many collisions) if hash values are
 * not properly distributed.
 *
 *
 *   <p> Robin-Hood hashing algorithm is also used to minimize variance
 *  in insertion and search-related operations, for an all-around smother operation at the cost
 *  of smaller peak performance:</p>
 *  <p> - Pedro Celis (1986) for the original Robin-Hood hashing paper, </p>
 *  <p> - <a href="cliff@leaninto.it">MoonPolySoft/Cliff Moon</a> for the initial Robin-hood on HPPC implementation,</p>
 *  <p> - <a href="vsonnier@gmail.com" >Vincent Sonnier</a> for the present implementation using cached hashes.</p>
 *
 */
 @javax.annotation.Generated(
    date = "2017-07-11T19:16:33+0200",
    value = "KTypeHashSet.java") 
public class ObjectHashSet<KType>
extends AbstractObjectCollection<KType>
implements ObjectLookupContainer<KType>, ObjectSet<KType>, Cloneable
{
    /**
     * Hash-indexed array holding all set entries.
     * <p>
     * Direct set iteration: iterate  {keys[i]} for i in [0; keys.length[ where keys[i] != 0/null, then also
     * {0/null} is in the set if {@link #allocatedDefaultKey} = true.
     * </p>
     */
    public  
    Object[]
              
            keys;

      
    /**

     *      * Caches the hash value = hash(keys[i]) & mask, if keys[i] != 0/null,
     * for every index i.
     *      * @see #assigned
     */
      
      
    protected int[] hash_cache;
      

    /**
     * True if key = 0/null is in the map.
     */
    public boolean allocatedDefaultKey = false;

    /**
     * Cached number of assigned slots in {@link #keys}.
     */
    protected int assigned;

    /**
     * The load factor for this map (fraction of allocated slots
     * before the buffers must be rehashed or reallocated).
     */
    protected final double loadFactor;

    /**
     * Resize buffers when {@link #keys} hits this value.
     */
    private int resizeAt;

    /**
     * Per-instance perturbation
     * introduced in rehashing to create a unique key distribution.
     */
    private final int perturbation = Containers.randomSeed32();

      

    /**
     * Override this method, together with {@link #equalKeys(Object, Object)}
     * to customize the hashing strategy. Note that this method is guaranteed
     * to be called with a non-null key argument.
     * By default, this method calls key.{@link #hashCode()}.
     * @param key Object to be hashed.
     * @return the hashed value of key, following the same semantic
     * as {@link #hashCode()};
     * @see #hashCode()
     * @see #equalKeys(Object, Object)
     */
    protected int hashKey(final KType key) {

        //default maps on Object.hashCode()
        return key.hashCode();
    }

    /**
     * Override this method together with {@link #hashKey(Object)}
     * to customize the hashing strategy. Note that this method is guaranteed
     * to be called with both non-null arguments.
     * By default, this method calls a.{@link #equals(b)}.
     * @param a not-null Object to be compared
     * @param b not-null Object to be compared
     * @return true if a and b are considered equal, following the same
     * semantic as {@link #equals(Object)}.
     * @see #equals(Object)
     * @see #hashKey(Object)
     */
    protected boolean equalKeys(final KType a, final KType b) {

        //default maps on Object.equals()
        return ((a).equals((b)));
    }

      

    /**
     * Default constructor: Creates a hash set with the default capacity of {@link Containers#DEFAULT_EXPECTED_ELEMENTS},
     * load factor of {@link HashContainers#DEFAULT_LOAD_FACTOR}.
     */
    public ObjectHashSet() {
        this(Containers.DEFAULT_EXPECTED_ELEMENTS, HashContainers.DEFAULT_LOAD_FACTOR);
    }

    /**
     * Creates a hash set with the given capacity,
     * load factor of {@link HashContainers#DEFAULT_LOAD_FACTOR}.
     */
    public ObjectHashSet(final int initialCapacity) {
        this(initialCapacity, HashContainers.DEFAULT_LOAD_FACTOR);
    }

    /**
     * Creates a hash set with the given capacity and load factor.
     */
    public ObjectHashSet(final int initialCapacity, final double loadFactor) {
        this.loadFactor = loadFactor;
        //take into account of the load factor to guarantee no reallocations before reaching  initialCapacity.
        allocateBuffers(HashContainers.minBufferSize(initialCapacity, loadFactor));
    }

    /**
     * Creates a hash set from elements of another container. Default load factor is used.
     */
    public ObjectHashSet(final ObjectContainer<KType> container) {
        this(container.size());
        addAll(container);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean add(KType key) {

        if (((key) == null)) {

            if (this.allocatedDefaultKey) {

                return false;
            }

            this.allocatedDefaultKey = true;

            return true;
        }

        final int mask = this.keys.length - 1;

        final KType[] keys = ((KType[])(this.keys));

        int slot = (BitMixer.mix(hashKey((key)) , this.perturbation)) & mask;
        KType existing;

          
        final int[] cached = this.hash_cache;
        KType tmpKey;
        int tmpAllocated;
        int initial_slot = slot;
        int dist = 0;
        int existing_distance = 0;

          

          

        while (!((existing = keys[slot]) == null)) {

              
            existing_distance = probe_distance(slot, cached);

            //When first entering the while loop, then key == original key to search.
            //So either:
            //1) key is immediately found and the routine bail out,
            //or
            //2) If the Robin-hood criteria of distance is not met, we search the next slot, (usual linear probing)
            //or
            //3) else the criteria of distance is met, then (key) is swapped with the ones in
            //slot position which becomes the new (key) to consider. This is OK because keys are swapped only if dist > existing_distance,
            //i.e only if the key to add is NOT in the set, see contains(). So we steal the rich (a previously entered key, favored because having being inserted
            //in a less crowed array) to give to the poor, the now inserted key. Then, we start searching again in the next slot.

              
              

              
            // Robin-hood shortcut: if key exists, it can only be found in dist <= existing_distance range.
            //indeed we should expect to never see an existing element with a shorter probe count (existing_distance)
            //than our current count (dist): if that had happened, there would’ve been a swap during insertion, see below.
            //also see contains() and remove() for the same trick.
              
            if (   dist <= existing_distance &&    (equalKeys((key), (existing)))) {
                return false;
            }

              
            //re-shuffle keys to minimize variance
            if (dist > existing_distance) {

                //we actually enter here only if the key to add is NOT in the set.

                //swap current (key, value, initial_slot) with slot places
                tmpKey = keys[slot];
                keys[slot] = key;
                key = tmpKey;

                tmpAllocated = cached[slot];
                cached[slot] = initial_slot;
                initial_slot = tmpAllocated;

                  

                dist = existing_distance;
            }
              

            slot = (slot + 1) & mask;
              
            dist++;
              
        }

        // Check if we need to grow. If so, reallocate new data,
        // fill in the last element and rehash.
        if (this.assigned == this.resizeAt) {

            expandAndAdd(key, slot);
        } else {
            this.assigned++;
              
            cached[slot] = initial_slot;
               

            keys[slot] = key;

              
              
              
        }
        return true;
    }

    /**
     * Adds two elements to the set.
     */
    public int add(final KType e1, final KType e2) {
        int count = 0;
        if (add(e1)) {
            count++;
        }
        if (add(e2)) {
            count++;
        }
        return count;
    }

    /**
     * Vararg-signature method for adding elements to this set.
     * <p><b>This method is handy, but costly if used in tight loops (anonymous
     * array passing)</b></p>
     *
     * @return Returns the number of elements that were added to the set
     * (were not present in the set).
     */
    public int add(final KType... elements) {
        int count = 0;
        for (final KType e : elements) {
            if (add(e)) {
                count++;
            }
        }
        return count;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int addAll(final ObjectContainer<? extends KType> container) {
        return addAll((Iterable<? extends ObjectCursor<? extends KType>>) container);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int addAll(final Iterable<? extends ObjectCursor<? extends KType>> iterable) {
        int count = 0;
        for (final ObjectCursor<? extends KType> cursor : iterable) {
            if (add(cursor.value)) {
                count++;
            }
        }
        return count;
    }

    /**
     * Expand the internal storage buffers (capacity) or rehash current
     * keys and values if there are a lot of deleted slots.
     */
    private void expandAndAdd(final KType pendingKey, final int freeSlot) {
        assert this.assigned == this.resizeAt;

        //default sentinel value is never in the keys[] array, so never trigger reallocs
        assert (!((pendingKey) == null));

        // Try to allocate new buffers first. If we OOM, it'll be now without
        // leaving the data structure in an inconsistent state.
        final KType[] oldKeys = ((KType[])(this.keys));

        allocateBuffers(HashContainers.nextBufferSize(this.keys.length, this.assigned, this.loadFactor));

        // We have succeeded at allocating new data so insert the pending key/value at
        // the free slot in the old arrays before rehashing.

        this.assigned++;

        oldKeys[freeSlot] = pendingKey;

        //Variables for adding
        final int mask = this.keys.length - 1;

        KType key = (null);
        //adding phase
        int slot = -1;

        final KType[] keys = ((KType[])(this.keys));

          
        final int[] cached = this.hash_cache;
          

          
        KType tmpKey = (null);
        int tmpAllocated = -1;
        int initial_slot = -1;
        int dist = -1;
        int existing_distance = -1;
          

        //iterate all the old arrays to add in the newly allocated buffers
        //It is important to iterate backwards to minimize the conflict chain length !
        final int perturb = this.perturbation;

        for (int i = oldKeys.length; --i >= 0;) {

            //only consider non-empty slots, of course
            if (!((key = oldKeys[i]) == null)) {

                slot = (BitMixer.mix(hashKey((key)) , (perturb))) & mask;

                  
                initial_slot = slot;
                dist = 0;
                  

                //similar to add(), except all inserted keys are known to be unique.
                while ((!(((keys)[(slot)]) == null))) {
                      
                    //re-shuffle keys to minimize variance
                    existing_distance = probe_distance(slot, cached);

                    if (dist > existing_distance) {
                        //swap current (key, value, initial_slot) with slot places
                        tmpKey = keys[slot];
                        keys[slot] = key;
                        key = tmpKey;

                        tmpAllocated = cached[slot];
                        cached[slot] = initial_slot;
                        initial_slot = tmpAllocated;

                          

                        dist = existing_distance;
                    } //endif
                      

                    slot = (slot + 1) & mask;

                      
                    dist++;
                      
                } //end while

                //place it at that position
                  
                cached[slot] = initial_slot;
                  

                keys[slot] = key;

                  
                  
                  
            }
        }
    }

    /**
     * Allocate internal buffers for a given capacity.
     *
     * @param capacity New capacity (must be a power of two).
     */
    @SuppressWarnings("boxing")
    private void allocateBuffers(final int capacity) {
        try {

            final KType[] keys = ((KType[])new Object[(capacity)]);

              
            final int[] allocated = new int[capacity];
              

            this.keys = keys;

              
            this.hash_cache = allocated;
              

            //allocate so that there is at least one slot that remains allocated = false
            //this is compulsory to guarantee proper stop in searching loops
            this.resizeAt = HashContainers.expandAtCount(capacity, this.loadFactor);
        } catch (final OutOfMemoryError e) {

            throw new BufferAllocationException(
                    "Not enough memory to allocate buffers to grow from %d -> %d elements",
                    e,
                    (this.keys == null) ? 0 : this.keys.length,
                            capacity);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int removeAll(final KType key) {
        return remove(key) ? 1 : 0;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean remove(final KType key) {

        if (((key) == null)) {

            if (this.allocatedDefaultKey) {

                this.allocatedDefaultKey = false;
                return true;
            }

            return false;
        }

        final int mask = this.keys.length - 1;

        final KType[] keys = ((KType[])(this.keys));

        int slot = (BitMixer.mix(hashKey((key)) , this.perturbation)) & mask;
        KType existing;

          
        int dist = 0;
        final int[] cached = this.hash_cache;
          

        while (!((existing = keys[slot]) == null)
                  && dist <= probe_distance(slot, cached)   ) {
            if ((equalKeys((key), (existing)))) {

                shiftConflictingKeys(slot);
                return true;
            }
            slot = (slot + 1) & mask;

              
            dist++;
              
        } //end while true

        return false;
    }

    /**
     * Shift all the slot-conflicting keys allocated to (and including) <code>slot</code>.
     */
    private void shiftConflictingKeys(int gapSlot) {

        final int mask = this.keys.length - 1;

        final KType[] keys = ((KType[])(this.keys));

          
        final int[] cached = this.hash_cache;
           

        // Perform shifts of conflicting keys to fill in the gap.
        int distance = 0;

        while (true) {

            final int slot = (gapSlot + (++distance)) & mask;

            final KType existing = keys[slot];

            if (((existing) == null)) {
                break;
            }

              
            //use the cached value, no need to recompute
            final int idealSlotModMask = cached[slot];
              
              

            //original HPPC code: shift = (slot - idealSlot) & mask;
            //equivalent to shift = (slot & mask - idealSlot & mask) & mask;
            //since slot and idealSlotModMask are already folded, we have :
            final int shift = (slot - idealSlotModMask) & mask;

            if (shift >= distance) {
                // Entry at this position was originally at or before the gap slot.
                // Move the conflict-shifted entry to the gap's position and repeat the procedure
                // for any entries to the right of the current position, treating it
                // as the new gap.
                keys[gapSlot] = existing;

                  
                cached[gapSlot] = idealSlotModMask;
                  
                  

                gapSlot = slot;
                distance = 0;
            }
        } //end while

        // Mark the last found gap slot without a conflict as empty.
        keys[gapSlot] = (null);

        this.assigned--;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean contains(final KType key) {

        if (((key) == null)) {

            return this.allocatedDefaultKey;
        }

        final int mask = this.keys.length - 1;

        final KType[] keys = ((KType[])(this.keys));

        int slot = (BitMixer.mix(hashKey((key)) , this.perturbation)) & mask;
        KType existing;

          
        final int[] cached = this.hash_cache;
        int dist = 0;
          

        while (!((existing = keys[slot]) == null)
                  && dist <= probe_distance(slot, cached)   ) {
            if ((equalKeys((key), (existing)))) {
                return true;
            }
            slot = (slot + 1) & mask;

              
            dist++;
              
        } //end while true

        return false;
    }

    /**
     * {@inheritDoc}
     *
     * <p>Does not release internal buffers.</p>
     */
    @Override
    public void clear() {
        this.assigned = 0;

        // States are always cleared.
        this.allocatedDefaultKey = false;

        //Faster than Arrays.fill(keys, null); // Help the GC.
        ObjectArrays.blankArray(this.keys, 0, this.keys.length);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int size() {
        return this.assigned + (this.allocatedDefaultKey ? 1 : 0);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int capacity() {

        return this.resizeAt;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int hashCode() {
        int h = 0;

        //allocated default key has hash = 0

        final KType[] keys = ((KType[])(this.keys));

        for (int i = keys.length; --i >= 0;) {
            KType existing;
            if (!((existing = keys[i]) == null)) {
                h += BitMixer.mix(existing);
            }
        }

        return h;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean equals(final Object obj) {
        if (obj != null) {
            if (obj == this) {
                return true;
            }

            //must be of the same class, subclasses are not comparable
            if (obj.getClass() != this.getClass()) {

                return false;
            }

            @SuppressWarnings("unchecked")
            final ObjectSet<KType> other = (ObjectSet<KType>) obj;

            //must be of the same size
            if (other.size() != this.size()) {
                return false;
            }

            final EntryIterator it = this.iterator();

            while (it.hasNext()) {
                if (!other.contains(it.next().value)) {
                    //recycle
                    it.release();
                    return false;
                }
            }
            return true;
        }
        return false;
    }

    /**
     * An iterator implementation for {@link #iterator}.
     * Holds a ObjectCursor returning (value, index) = (Object value, index the position in {@link ObjectHashSet#keys}, or keys.length for key = 0/null.)
     */
    public final class EntryIterator extends AbstractIterator<ObjectCursor<KType>>
    {
        public final ObjectCursor<KType> cursor;

        public EntryIterator() {
            this.cursor = new ObjectCursor<KType>();
            this.cursor.index = -2;
        }

        /**
         * Iterate backwards w.r.t the buffer, to
         * minimize collision chains when filling another hash container (ex. with putAll())
         */
        @Override
        protected ObjectCursor<KType> fetch() {
            if (this.cursor.index == ObjectHashSet.this.keys.length + 1) {

                if (ObjectHashSet.this.allocatedDefaultKey) {

                    this.cursor.index = ObjectHashSet.this.keys.length;
                    this.cursor.value = (null);

                    return this.cursor;

                }
                //no value associated with the default key, continue iteration...
                this.cursor.index = ObjectHashSet.this.keys.length;

            }

            int i = this.cursor.index - 1;

            while (i >= 0 && !(!(((((KType[])(ObjectHashSet.this.keys)))[(i)]) == null))) {
                i--;
            }

            if (i == -1) {
                return done();
            }

            this.cursor.index = i;
            this.cursor.value = ((KType)(ObjectHashSet.this.keys[i]));
            return this.cursor;
        }
    }

    /**
     * internal pool of EntryIterator
     */
    protected final IteratorPool<ObjectCursor<KType>, EntryIterator> entryIteratorPool = new IteratorPool<ObjectCursor<KType>, EntryIterator>(
            new ObjectFactory<EntryIterator>() {

                @Override
                public EntryIterator create() {

                    return new EntryIterator();
                }

                @Override
                public void initialize(final EntryIterator obj) {
                    obj.cursor.index = ObjectHashSet.this.keys.length + 1;
                }

                @Override
                public void reset(final EntryIterator obj) {
                      
                    obj.cursor.value = null;
                      

                }
            });

    /**
     * {@inheritDoc}
     *
     */
    @Override
    public EntryIterator iterator() {
        //return new EntryIterator();
        return this.entryIteratorPool.borrow();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public <T extends ObjectProcedure<? super KType>> T forEach(final T procedure) {
        if (this.allocatedDefaultKey) {

            procedure.apply((null));
        }

        final KType[] keys = ((KType[])(this.keys));

        //Iterate in reverse for side-stepping the longest conflict chain
        //in another hash, in case apply() is actually used to fill another hash container.
        for (int i = keys.length - 1; i >= 0; i--) {
            KType existing;
            if (!((existing = keys[i]) == null)) {
                procedure.apply(existing);
            }
        }

        return procedure;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public KType[] toArray(final KType[] target) {
        int count = 0;

        if (this.allocatedDefaultKey) {

            target[count++] = (null);
        }

        final KType[] keys = ((KType[])(this.keys));

        for (int i = 0; i < keys.length; i++) {
            KType existing;
            if (!((existing = keys[i]) == null)) {
                target[count++] = existing;
            }
        }

        assert count == this.size();

        return target;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ObjectHashSet<KType> clone() {
        //clone to size() to prevent eventual exponential growth
        final ObjectHashSet<KType> cloned = new ObjectHashSet<KType>(this.size(), this.loadFactor);

        //We must NOT clone, because of the independent perturbation seeds
        cloned.addAll(this);

        return cloned;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public <T extends ObjectPredicate<? super KType>> T forEach(final T predicate) {
        if (this.allocatedDefaultKey) {

            if (!predicate.apply((null))) {

                return predicate;
            }
        }

        final KType[] keys = ((KType[])(this.keys));

        //Iterate in reverse for side-stepping the longest conflict chain
        //in another hash, in case apply() is actually used to fill another hash container.
        for (int i = keys.length - 1; i >= 0; i--) {
            KType existing;
            if (!((existing = keys[i]) == null)) {
                if (!predicate.apply(existing)) {
                    break;
                }
            }
        }

        return predicate;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int removeAll(final ObjectPredicate<? super KType> predicate) {
        final int before = this.size();

        if (this.allocatedDefaultKey) {

            if (predicate.apply((null))) {
                this.allocatedDefaultKey = false;
            }
        }

        final KType[] keys = ((KType[])(this.keys));

        for (int i = 0; i < keys.length;) {
            KType existing;
            if (!((existing = keys[i]) == null) && predicate.apply(existing)) {

                shiftConflictingKeys(i);
                // Shift, do not increment slot.
            } else {
                i++;
            }
        }

        return before - this.size();
    }

    /**
     * Create a set from a variable number of arguments or an array of <code>Object</code>.
     */
    public static <KType> ObjectHashSet<KType> from(final KType... elements) {
        final ObjectHashSet<KType> set = new ObjectHashSet<KType>(elements.length);
        set.add(elements);
        return set;
    }

    /**
     * Create a set from elements of another container.
     */
    public static <KType> ObjectHashSet<KType> from(final ObjectContainer<KType> container) {
        return new ObjectHashSet<KType>(container);
    }

    /**
     * Create a new hash set with default parameters (shortcut
     * instead of using a constructor).
     */
    public static <KType> ObjectHashSet<KType> newInstance() {
        return new ObjectHashSet<KType>();
    }

    /**
     * Returns a new object of this class with no need to declare generic type (shortcut
     * instead of using a constructor).
     */
    public static <KType> ObjectHashSet<KType> newInstance(final int initialCapacity, final double loadFactor) {
        return new ObjectHashSet<KType>(initialCapacity, loadFactor);
    }

    //Test for existence in template
      

      
    private int probe_distance(final int slot, final int[] cached) {

        final int rh = cached[slot];

          

        if (slot < rh) {
            //wrap around
            return slot - rh + cached.length;
        }

        return slot - rh;
    }
      

      

      

      
}
