package com.carrotsearch.hppcrt.heaps;

import java.util.*;

import com.carrotsearch.hppcrt.*;
import com.carrotsearch.hppcrt.cursors.*;
import com.carrotsearch.hppcrt.hash.BitMixer;
import com.carrotsearch.hppcrt.predicates.*;
import com.carrotsearch.hppcrt.procedures.*;
import com.carrotsearch.hppcrt.strategies.*;

  
/**
 * A Heap-based, min-priority queue of <code>Object</code>s.
 * i.e. top() is the smallest element,
 * as defined by Sedgewick: Algorithms 4th Edition (2011).
 * It assure O(log(N)) complexity for insertion,  deletion and update priority of the min element,
 * and constant time to examine the min element by {@link #top()}.
 * <p><b>Important: </b>
 * Ordering of elements must be defined either
 *  * by {@link Comparable}
 *  *  or by a custom comparator provided in constructors,
 * see {@link #comparator()} .
 */
  
 @javax.annotation.Generated(
    date = "2017-07-11T19:16:22+0200",
    value = "KTypeHeapPriorityQueue.java") 
public class ObjectHeapPriorityQueue<KType> extends AbstractObjectCollection<KType>
implements ObjectPriorityQueue<KType>, Cloneable
{
    /**
     * Internal array for storing the priority queue.
     * <p>
     * Direct priority queue iteration: iterate buffer[i] for i in [1; size()] (included) but is out-of-order w.r.t {@link #popTop()}
     * </p>
     */
    public  
    Object[]
              
            buffer;

    /**
     * Number of elements in the queue.
     */
    protected int elementsCount;

    /**
     * Defines the Comparator ordering of the queue,
     * If null, natural ordering is used.
     */
      
    protected Comparator<? super KType> comparator;
      

    /**
     * Buffer resizing strategy.
     */
    protected final ArraySizingStrategy resizer;

    /**
     * internal pool of ValueIterator (must be created in constructor)
     */
    protected final IteratorPool<ObjectCursor<KType>, ValueIterator> valueIteratorPool;

    /**
     * The current value set for removeAll
     */
    protected KType currentOccurenceToBeRemoved;

    /**
     * Internal predicate for removeAll
     */
    protected ObjectPredicate<? super KType> removeAllOccurencesPredicate = new ObjectPredicate<KType>() {

        @Override
        public final boolean apply(final KType value) {

            if (((value) == null ? (ObjectHeapPriorityQueue.this.currentOccurenceToBeRemoved) == null : (value).equals((ObjectHeapPriorityQueue.this.currentOccurenceToBeRemoved)))) {

                return true;
            }

            return false;
        }
    };

    /**
     * Default value returned when specified
     * in methods.
     * @see #top()
     */
    protected KType defaultValue;

    /**
     * Create with a Comparator, an initial capacity, and a custom buffer resizing strategy.
     */
    public ObjectHeapPriorityQueue(  final Comparator<? super KType> comp,
              final int initialCapacity, final ArraySizingStrategy resizer) {
        this.comparator = comp;

        assert resizer != null;

        this.resizer = resizer;

        //1-based index buffer, assure allocation
        ensureBufferSpace(Math.max(Containers.DEFAULT_EXPECTED_ELEMENTS, initialCapacity));

        this.valueIteratorPool = new IteratorPool<ObjectCursor<KType>, ValueIterator>(new ObjectFactory<ValueIterator>() {

            @Override
            public ValueIterator create() {
                return new ValueIterator();
            }

            @Override
            public void initialize(final ValueIterator obj) {
                obj.cursor.index = 0;
                obj.size = ObjectHeapPriorityQueue.this.size();
                obj.buffer = ((KType[])(ObjectHeapPriorityQueue.this.buffer));
            }

            @Override
            public void reset(final ValueIterator obj) {
                // for GC sake
                obj.buffer = null;

                  
                obj.cursor.value = null;
                  

            }
        });
    }

    /**
     * Create with default sizing strategy and initial capacity
     * ({@link Containers#DEFAULT_EXPECTED_ELEMENTS})
     * using a specific Comparator.
     * 
     * @see BoundedProportionalArraySizingStrategy
     */
    public ObjectHeapPriorityQueue(  final Comparator<? super KType> comp
              )
    {
        this(comp, Containers.DEFAULT_EXPECTED_ELEMENTS);
    }

      
    /**
     * Default constructor: create with a default
     * numbers of elements ({@link Containers#DEFAULT_EXPECTED_ELEMENTS}),
     * using the Comparable natural ordering.
     */
      
    public ObjectHeapPriorityQueue() {
        this(null, Containers.DEFAULT_EXPECTED_ELEMENTS);
    }

      
    /**
     * Create with an initial capacity,
     * using the Comparable natural ordering
     */
      
    public ObjectHeapPriorityQueue(final int initialCapacity) {
        this(null, initialCapacity, new BoundedProportionalArraySizingStrategy());
    }

    /**
     * Create with a given initial capacity, using a
     * Comparator for ordering.
     * 
     * @see BoundedProportionalArraySizingStrategy
     */
    public ObjectHeapPriorityQueue(  final Comparator<? super KType> comp,
              final int initialCapacity)
    {
        this(comp, initialCapacity, new BoundedProportionalArraySizingStrategy());
    }

    /**
     * Creates a new heap from elements of another container.
     */
    public ObjectHeapPriorityQueue(final ObjectContainer<? extends KType> container) {
        this(container.size());
        addAll(container);
    }

    /**
     * Create a heap from elements of another container (constructor shortcut)
     */
    public static/*  */<KType> /*  */
    ObjectHeapPriorityQueue<KType> from(final ObjectContainer<KType> container) {
        return new ObjectHeapPriorityQueue<KType>(container);
    }

    /**
     * Create a heap from a variable number of arguments or an array of
     * <code>Object</code>.
     */
    public static/*  */<KType> /*  */
    ObjectHeapPriorityQueue<KType> from(final KType... elements) {
        final ObjectHeapPriorityQueue<KType> heap = new ObjectHeapPriorityQueue<KType>(elements.length);

        for (final KType elem : elements) {

            heap.add(elem);
        }

        return heap;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int removeAll(final KType e1) {
        this.currentOccurenceToBeRemoved = e1;
        return removeAll(this.removeAllOccurencesPredicate);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int removeAll(final ObjectPredicate<? super KType> predicate) {
        //remove by position
        int deleted = 0;
        final KType[] buffer = ((KType[])(this.buffer));
        int elementsCount = this.elementsCount;

        //1-based index
        int pos = 1;

        try {
            while (pos <= elementsCount) {
                //delete it
                if (predicate.apply(buffer[pos])) {
                    //put the last element at position pos, like in deleteIndex()
                    buffer[pos] = buffer[elementsCount];

                    //for GC
                      
                    buffer[elementsCount] = (null);
                      

                    //Diminish size
                    elementsCount--;
                    deleted++;
                } //end if to delete
                else {
                    pos++;
                }
            } //end while
        } finally {
            this.elementsCount = elementsCount;
            //reestablish heap
            updatePriorities();
        }

          

        return deleted;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void clear() {
          
        //1-based indexing
        ObjectArrays.blankArray(this.buffer, 1, this.elementsCount + 1);
          
        this.elementsCount = 0;
    }

    /**
     * An iterator implementation for {@link ObjectHeapPriorityQueue#iterator}.
     * Holds a ObjectCursor returning (value, index) = (Object value, index the position in heap {@link ObjectHeapPriorityQueue#buffer}.)
     */
    public final class ValueIterator extends AbstractIterator<ObjectCursor<KType>>
    {
        public final ObjectCursor<KType> cursor;

        private KType[] buffer;
        private int size;

        public ValueIterator() {
            this.cursor = new ObjectCursor<KType>();
            //index 0 is not used in Priority queue
            this.cursor.index = 0;
            this.size = ObjectHeapPriorityQueue.this.size();
            this.buffer = ((KType[])(ObjectHeapPriorityQueue.this.buffer));
        }

        @Override
        protected ObjectCursor<KType> fetch() {
            //priority is 1-based index
            if (this.cursor.index == this.size) {
                return done();
            }

            this.cursor.value = this.buffer[++this.cursor.index];
            return this.cursor;
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ValueIterator iterator() {
        //return new ValueIterator(buffer, size());
        return this.valueIteratorPool.borrow();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean contains(final KType element) {
        //1-based index
        final int size = this.elementsCount;
        final KType[] buff = ((KType[])(this.buffer));

        for (int i = 1; i <= size; i++) {
            if (((element) == null ? (buff[i]) == null : (element).equals((buff[i])))) {
                return true;
            }
        } //end for

        return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int size() {
        return this.elementsCount;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int capacity() {

        return this.buffer.length - 1;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public <T extends ObjectProcedure<? super KType>> T forEach(final T procedure) {
        final KType[] buff = ((KType[])(this.buffer));
        final int size = this.elementsCount;

        for (int i = 1; i <= size; i++) {
            procedure.apply(buff[i]);
        }

        return procedure;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public <T extends ObjectPredicate<? super KType>> T forEach(final T predicate) {
        final KType[] buff = ((KType[])(this.buffer));
        final int size = this.elementsCount;

        for (int i = 1; i <= size; i++) {
            if (!predicate.apply(buff[i])) {
                break;
            }
        }

        return predicate;
    }

    /**
     * Insert a Object into the queue.
     * cost: O(log(N)) for a N sized queue
     */
    @Override
    public void add(final KType element) {
        ensureBufferSpace(1);

        //add at the end
        this.elementsCount++;
        this.buffer[this.elementsCount] = element;

        //swim last element
        swim(this.elementsCount);
    }

    /**
     * {@inheritDoc}
     * cost: O(1)
     */
    @Override
    public KType top() {
        KType elem = this.defaultValue;

        if (this.elementsCount > 0) {
            elem = ((KType)(this.buffer[1]));
        }

        return elem;
    }

    /**
     * {@inheritDoc}
     * cost: O(log(N)) for a N sized queue
     */
    @Override
    public KType popTop() {
        KType elem = this.defaultValue;

        if (this.elementsCount > 0) {
            elem = ((KType)(this.buffer[1]));

            if (this.elementsCount == 1) {
                //for GC
                  
                this.buffer[1] = (null);
                  
                //diminish size
                this.elementsCount = 0;
            } else {
                //at least 2 elements
                //put the last element in first position

                this.buffer[1] = this.buffer[this.elementsCount];

                //for GC
                  
                this.buffer[this.elementsCount] = (null);
                  

                //diminish size
                this.elementsCount--;

                //percolate down the first element
                sink(1);
            }
        }

        return elem;
    }

    /**
     * Adds all elements from another container.
     * cost: O(N*log(N)) for N elements
     */
    public int addAll(final ObjectContainer<? extends KType> container) {
        return addAll((Iterable<? extends ObjectCursor<? extends KType>>) container);
    }

    /**
     * Adds all elements from another iterable.
     * cost: O(N*log(N)) for N elements
     */
    public int addAll(final Iterable<? extends ObjectCursor<? extends KType>> iterable) {
        int size = 0;
        final KType[] buff = ((KType[])(this.buffer));
        int count = this.elementsCount;

        for (final ObjectCursor<? extends KType> cursor : iterable) {
            ensureBufferSpace(1);
            count++;
            buff[count] = cursor.value;
            size++;
        }

        this.elementsCount = count;

        //restore heap
        updatePriorities();
          

        return size;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int hashCode() {
        int h = 1;
        final int max = this.elementsCount;
        final KType[] buff = ((KType[])(this.buffer));

        //1-based index
        for (int i = 1; i <= max; i++) {
            h = 31 * h + BitMixer.mix(buff[i]);
        }
        return h;
    }

    /**
     * {@inheritDoc}
     * cost: O(n*log(N))
     */
    @Override
    public void updatePriorities() {
        if (this.comparator == null) {
            for (int k = this.elementsCount >> 1; k >= 1; k--) {
                sinkComparable(k);
            }
        } else {
            for (int k = this.elementsCount >> 1; k >= 1; k--) {
                sinkComparator(k);
            }
        }
    }

    /**
     * {@inheritDoc}
     * cost: O(log(N))
     */
    @Override
    public void updateTopPriority() {
        //only attempt to sink if there is at least 2 elements....
        if (this.elementsCount > 1) {

            sink(1);
        }
    }

    /**
     * Clone this object. The returned clone will use the same resizing strategy and comparator.
     */
    @Override
    public ObjectHeapPriorityQueue<KType> clone() {
        //real constructor call, of a place holder
        final ObjectHeapPriorityQueue<KType> cloned = new ObjectHeapPriorityQueue<KType>(this.comparator,
                Containers.DEFAULT_EXPECTED_ELEMENTS, this.resizer);

        //clone raw buffers
        cloned.buffer = this.buffer.clone();

        cloned.defaultValue = this.defaultValue;
        cloned.elementsCount = this.elementsCount;

        return cloned;
    }

    /**
     * this instance and obj can only be equal to this if either: <pre>
     * (both don't have set comparators)
     * or
     * (both have equal comparators defined by {@link #comparator()}.equals(obj.comparator))</pre>
     * then, both heap elements are compared with {@link #equals(Object)} iterating their {@link #buffer}.
     */
    @Override
    /*  */
    @SuppressWarnings("unchecked")
    /*  */
    public boolean equals(final Object obj) {
        if (obj != null) {
            if (obj == this) {
                return true;
            }

            //we can only compare both ObjectHeapPriorityQueue and not subclasses between themselves
            //that has the same comparison function reference
            if (obj.getClass() != this.getClass()) {
                return false;
            }

            final ObjectHeapPriorityQueue<KType> other = (ObjectHeapPriorityQueue<KType>) obj;

            if (other.size() != this.size()) {

                return false;
            }

            //If one comparator is null, and the other not, we cannot compare them, same if
            //both comparators are different because the heap behavior will be different, even elements are equal.
            if (!((this.comparator == null && other.comparator == null) || (this.comparator != null && this.comparator.equals(other.comparator)))) {

                return false;
            }

            final ValueIterator it = this.iterator();
            final ValueIterator itOther = other.iterator();

            while (it.hasNext()) {

                final KType myVal = it.next().value;
                final KType otherVal = itOther.next().value;

                if (!((myVal) == null ? (otherVal) == null : (myVal).equals((otherVal)))) {
                    //recycle
                    it.release();
                    itOther.release();
                    return false;
                }
            } //end while
            itOther.release();

            return true;
        }

        return false;
    }

    /**
     * Ensures the internal buffer has enough free slots to store
     * <code>expectedAdditions</code>. Increases internal buffer size if needed.
     */
    @SuppressWarnings("boxing")
    protected void ensureBufferSpace(final int expectedAdditions) {
        final int bufferLen = this.buffer == null ? 0 : this.buffer.length;

        //element of index 0 is not used
        if (this.elementsCount + 1 > bufferLen - expectedAdditions) {
            int newSize = this.resizer.grow(bufferLen, this.elementsCount, expectedAdditions);

            //first allocation, reserve an additional slot because index 0  is not used
            if (this.buffer == null) {
                newSize++;
            }

            try {
                final KType[] newBuffer = ((KType[])new Object[(newSize)]);

                if (bufferLen > 0) {
                    System.arraycopy(this.buffer, 0, newBuffer, 0, this.buffer.length);
                }

                this.buffer = newBuffer;
            } catch (final OutOfMemoryError e) {

                throw new BufferAllocationException(
                        "Not enough memory to allocate buffers to grow from %d -> %d elements",
                        e,
                        bufferLen,
                        newSize);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public KType[] toArray(final KType[] target) {
        //copy from index 1
        System.arraycopy(this.buffer, 1, target, 0, this.elementsCount);
        return target;
    }

    /**
     * Get the custom comparator used for comparing elements
     * @return null if no custom comparator was set, i.e natural ordering
     * of <code>Object</code>s is used instead
     *  , which means objects in this case must be {@link Comparable}.
     *      */
      
    public Comparator<? super KType>
      
    comparator() {

        return this.comparator;
    }

    /**
     * Returns the "default value" value used
     * in methods returning "default value"
     */
    @Override
    public KType getDefaultValue()
    {
        return this.defaultValue;
    }

    /**
     * Set the "default value" value to be used
     * in methods returning "default value"
     */
    @Override
    public void setDefaultValue(final KType defaultValue)
    {
        this.defaultValue = defaultValue;
    }

    /**
     * Sink function for Comparable elements
     * 
     * @param k
     */
    private void sinkComparable(int k) {
        final int N = this.elementsCount;
        final KType[] buffer = ((KType[])(this.buffer));

        KType tmp;
        int child;

        while ((k << 1) <= N) {
            //get the child of k
            child = k << 1;

            if (child < N && (((Comparable<? super KType>) (buffer[child])).compareTo((buffer[child + 1])) > 0)) {
                child++;
            }

            if (!(((Comparable<? super KType>) (buffer[k])).compareTo((buffer[child])) > 0)) {
                break;
            }

            //swap k and child
            tmp = buffer[k];
            buffer[k] = buffer[child];
            buffer[child] = tmp;

            k = child;
        } //end while
    }

    /**
     * Sink function for ObjectComparator elements
     * 
     * @param k
     */
    private void sinkComparator(int k) {
        final int N = this.elementsCount;
        final KType[] buffer = ((KType[])(this.buffer));

        KType tmp;
        int child;
          
        final Comparator<? super KType> comp = this.comparator;
          

        while ((k << 1) <= N) {
            //get the child of k
            child = k << 1;

            if (child < N && comp.compare(buffer[child], buffer[child + 1]) > 0) {
                child++;
            }

            if (comp.compare(buffer[k], buffer[child]) <= 0) {
                break;
            }

            //swap k and child
            tmp = buffer[k];
            buffer[k] = buffer[child];
            buffer[child] = tmp;

            k = child;
        } //end while
    }

    /**
     * Swim function for Comparable elements
     * 
     * @param k
     */
    private void swimComparable(int k) {
        KType tmp;
        int parent;
        final KType[] buffer = ((KType[])(this.buffer));

        while (k > 1 && (((Comparable<? super KType>) (buffer[k >> 1])).compareTo((buffer[k])) > 0)) {
            //swap k and its parent
            parent = k >> 1;

        tmp = buffer[k];
        buffer[k] = buffer[parent];
        buffer[parent] = tmp;

        k = parent;
        }
    }

    /**
     * Swim function for Comparator elements
     * 
     * @param k
     */
    private void swimComparator(int k) {
        KType tmp;
        int parent;
        final KType[] buffer = ((KType[])(this.buffer));

          
        final Comparator<? super KType> comp = this.comparator;
          

        while (k > 1 && comp.compare(buffer[k >> 1], buffer[k]) > 0) {
            //swap k and its parent
            parent = k >> 1;
        tmp = buffer[k];
        buffer[k] = buffer[parent];
        buffer[parent] = tmp;

        k = parent;
        }
    }

    private void swim(final int k) {
        if (this.comparator == null) {
            swimComparable(k);
        } else {
            swimComparator(k);
        }
    }

    private void sink(final int k) {
        if (this.comparator == null) {
            sinkComparable(k);
        } else {
            sinkComparator(k);
        }
    }

      
}
