package com.carrotsearch.hppcrt.heaps;

import java.util.*;

import com.carrotsearch.hppcrt.*;
import com.carrotsearch.hppcrt.cursors.*;
import com.carrotsearch.hppcrt.hash.BitMixer;
import com.carrotsearch.hppcrt.predicates.*;
import com.carrotsearch.hppcrt.procedures.*;
import com.carrotsearch.hppcrt.strategies.*;

  
/**
 * A Heap-based, min-priority queue of <code>double</code>s.
 * i.e. top() is the smallest element,
 * as defined by Sedgewick: Algorithms 4th Edition (2011).
 * It assure O(log(N)) complexity for insertion,  deletion and update priority of the min element,
 * and constant time to examine the min element by {@link #top()}.
 * <p><b>Important: </b>
 * Ordering of elements must be defined either
 *  * by natural ordering
 *  *  or by a custom comparator provided in constructors,
 * see {@link #comparator()} .
 */
  
 @javax.annotation.Generated(
    date = "2017-07-11T19:16:22+0200",
    value = "KTypeHeapPriorityQueue.java") 
public class DoubleHeapPriorityQueue extends AbstractDoubleCollection
implements DoublePriorityQueue, Cloneable
{
    /**
     * Internal array for storing the priority queue.
     * <p>
     * Direct priority queue iteration: iterate buffer[i] for i in [1; size()] (included) but is out-of-order w.r.t {@link #popTop()}
     * </p>
     */
    public               double []
               
            buffer;

    /**
     * Number of elements in the queue.
     */
    protected int elementsCount;

    /**
     * Defines the Comparator ordering of the queue,
     * If null, natural ordering is used.
     */
         protected DoubleComparator comparator;
     

    /**
     * Buffer resizing strategy.
     */
    protected final ArraySizingStrategy resizer;

    /**
     * internal pool of ValueIterator (must be created in constructor)
     */
    protected final IteratorPool<DoubleCursor, ValueIterator> valueIteratorPool;

    /**
     * The current value set for removeAll
     */
    protected double currentOccurenceToBeRemoved;

    /**
     * Internal predicate for removeAll
     */
    protected DoublePredicate removeAllOccurencesPredicate = new DoublePredicate() {

        @Override
        public final boolean apply(final double value) {

            if ((Double.doubleToLongBits((value)) == Double.doubleToLongBits((DoubleHeapPriorityQueue.this.currentOccurenceToBeRemoved)))) {

                return true;
            }

            return false;
        }
    };

    /**
     * Default value returned when specified
     * in methods.
     * @see #top()
     */
    protected double defaultValue;

    /**
     * Create with a Comparator, an initial capacity, and a custom buffer resizing strategy.
     */
    public DoubleHeapPriorityQueue(             DoubleComparator comp,
             final int initialCapacity, final ArraySizingStrategy resizer) {
        this.comparator = comp;

        assert resizer != null;

        this.resizer = resizer;

        //1-based index buffer, assure allocation
        ensureBufferSpace(Math.max(Containers.DEFAULT_EXPECTED_ELEMENTS, initialCapacity));

        this.valueIteratorPool = new IteratorPool<DoubleCursor, ValueIterator>(new ObjectFactory<ValueIterator>() {

            @Override
            public ValueIterator create() {
                return new ValueIterator();
            }

            @Override
            public void initialize(final ValueIterator obj) {
                obj.cursor.index = 0;
                obj.size = DoubleHeapPriorityQueue.this.size();
                obj.buffer = ((DoubleHeapPriorityQueue.this.buffer));
            }

            @Override
            public void reset(final ValueIterator obj) {
                // for GC sake
                obj.buffer = null;

                  

            }
        });
    }

    /**
     * Create with default sizing strategy and initial capacity
     * ({@link Containers#DEFAULT_EXPECTED_ELEMENTS})
     * using a specific Comparator.
     * 
     * @see BoundedProportionalArraySizingStrategy
     */
    public DoubleHeapPriorityQueue(     DoubleComparator comp
     )
    {
        this(comp, Containers.DEFAULT_EXPECTED_ELEMENTS);
    }

      
    /**
     * Default constructor: create with a default
     * numbers of elements ({@link Containers#DEFAULT_EXPECTED_ELEMENTS}),
     * using the natural ordering of <code>double</code>s.
     */
      
    public DoubleHeapPriorityQueue() {
        this(null, Containers.DEFAULT_EXPECTED_ELEMENTS);
    }

      
    /**
     * Create with an initial capacity,
     * using the natural ordering of <code>double</code>s
     */
      
    public DoubleHeapPriorityQueue(final int initialCapacity) {
        this(null, initialCapacity, new BoundedProportionalArraySizingStrategy());
    }

    /**
     * Create with a given initial capacity, using a
     * Comparator for ordering.
     * 
     * @see BoundedProportionalArraySizingStrategy
     */
    public DoubleHeapPriorityQueue(             DoubleComparator comp,
                       final int initialCapacity)
    {
        this(comp, initialCapacity, new BoundedProportionalArraySizingStrategy());
    }

    /**
     * Creates a new heap from elements of another container.
     */
    public DoubleHeapPriorityQueue(final DoubleContainer container) {
        this(container.size());
        addAll(container);
    }

    /**
     * Create a heap from elements of another container (constructor shortcut)
     */
    public static/*  */
    DoubleHeapPriorityQueue from(final DoubleContainer container) {
        return new DoubleHeapPriorityQueue(container);
    }

    /**
     * Create a heap from a variable number of arguments or an array of
     * <code>double</code>.
     */
    public static/*  */
    DoubleHeapPriorityQueue from(final double... elements) {
        final DoubleHeapPriorityQueue heap = new DoubleHeapPriorityQueue(elements.length);

        for (final double elem : elements) {

            heap.add(elem);
        }

        return heap;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int removeAll(final double e1) {
        this.currentOccurenceToBeRemoved = e1;
        return removeAll(this.removeAllOccurencesPredicate);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int removeAll(final DoublePredicate predicate) {
        //remove by position
        int deleted = 0;
        final double[] buffer = ((this.buffer));
        int elementsCount = this.elementsCount;

        //1-based index
        int pos = 1;

        try {
            while (pos <= elementsCount) {
                //delete it
                if (predicate.apply(buffer[pos])) {
                    //put the last element at position pos, like in deleteIndex()
                    buffer[pos] = buffer[elementsCount];

                    //for GC
                      

                    //Diminish size
                    elementsCount--;
                    deleted++;
                } //end if to delete
                else {
                    pos++;
                }
            } //end while
        } finally {
            this.elementsCount = elementsCount;
            //reestablish heap
            updatePriorities();
        }

          

        return deleted;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void clear() {
          
        this.elementsCount = 0;
    }

    /**
     * An iterator implementation for {@link DoubleHeapPriorityQueue#iterator}.
     * Holds a DoubleCursor returning (value, index) = (double value, index the position in heap {@link DoubleHeapPriorityQueue#buffer}.)
     */
    public final class ValueIterator extends AbstractIterator<DoubleCursor>
    {
        public final DoubleCursor cursor;

        private double[] buffer;
        private int size;

        public ValueIterator() {
            this.cursor = new DoubleCursor();
            //index 0 is not used in Priority queue
            this.cursor.index = 0;
            this.size = DoubleHeapPriorityQueue.this.size();
            this.buffer = ((DoubleHeapPriorityQueue.this.buffer));
        }

        @Override
        protected DoubleCursor fetch() {
            //priority is 1-based index
            if (this.cursor.index == this.size) {
                return done();
            }

            this.cursor.value = this.buffer[++this.cursor.index];
            return this.cursor;
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ValueIterator iterator() {
        //return new ValueIterator(buffer, size());
        return this.valueIteratorPool.borrow();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean contains(final double element) {
        //1-based index
        final int size = this.elementsCount;
        final double[] buff = ((this.buffer));

        for (int i = 1; i <= size; i++) {
            if ((Double.doubleToLongBits((element)) == Double.doubleToLongBits((buff[i])))) {
                return true;
            }
        } //end for

        return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int size() {
        return this.elementsCount;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int capacity() {

        return this.buffer.length - 1;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public <T extends DoubleProcedure> T forEach(final T procedure) {
        final double[] buff = ((this.buffer));
        final int size = this.elementsCount;

        for (int i = 1; i <= size; i++) {
            procedure.apply(buff[i]);
        }

        return procedure;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public <T extends DoublePredicate> T forEach(final T predicate) {
        final double[] buff = ((this.buffer));
        final int size = this.elementsCount;

        for (int i = 1; i <= size; i++) {
            if (!predicate.apply(buff[i])) {
                break;
            }
        }

        return predicate;
    }

    /**
     * Insert a double into the queue.
     * cost: O(log(N)) for a N sized queue
     */
    @Override
    public void add(final double element) {
        ensureBufferSpace(1);

        //add at the end
        this.elementsCount++;
        this.buffer[this.elementsCount] = element;

        //swim last element
        swim(this.elementsCount);
    }

    /**
     * {@inheritDoc}
     * cost: O(1)
     */
    @Override
    public double top() {
        double elem = this.defaultValue;

        if (this.elementsCount > 0) {
            elem = ((this.buffer[1]));
        }

        return elem;
    }

    /**
     * {@inheritDoc}
     * cost: O(log(N)) for a N sized queue
     */
    @Override
    public double popTop() {
        double elem = this.defaultValue;

        if (this.elementsCount > 0) {
            elem = ((this.buffer[1]));

            if (this.elementsCount == 1) {
                //for GC
                  
                //diminish size
                this.elementsCount = 0;
            } else {
                //at least 2 elements
                //put the last element in first position

                this.buffer[1] = this.buffer[this.elementsCount];

                //for GC
                  

                //diminish size
                this.elementsCount--;

                //percolate down the first element
                sink(1);
            }
        }

        return elem;
    }

    /**
     * Adds all elements from another container.
     * cost: O(N*log(N)) for N elements
     */
    public int addAll(final DoubleContainer container) {
        return addAll((Iterable<? extends DoubleCursor>) container);
    }

    /**
     * Adds all elements from another iterable.
     * cost: O(N*log(N)) for N elements
     */
    public int addAll(final Iterable<? extends DoubleCursor> iterable) {
        int size = 0;
        final double[] buff = ((this.buffer));
        int count = this.elementsCount;

        for (final DoubleCursor cursor : iterable) {
            ensureBufferSpace(1);
            count++;
            buff[count] = cursor.value;
            size++;
        }

        this.elementsCount = count;

        //restore heap
        updatePriorities();
          

        return size;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int hashCode() {
        int h = 1;
        final int max = this.elementsCount;
        final double[] buff = ((this.buffer));

        //1-based index
        for (int i = 1; i <= max; i++) {
            h = 31 * h + BitMixer.mix(buff[i]);
        }
        return h;
    }

    /**
     * {@inheritDoc}
     * cost: O(n*log(N))
     */
    @Override
    public void updatePriorities() {
        if (this.comparator == null) {
            for (int k = this.elementsCount >> 1; k >= 1; k--) {
                sinkComparable(k);
            }
        } else {
            for (int k = this.elementsCount >> 1; k >= 1; k--) {
                sinkComparator(k);
            }
        }
    }

    /**
     * {@inheritDoc}
     * cost: O(log(N))
     */
    @Override
    public void updateTopPriority() {
        //only attempt to sink if there is at least 2 elements....
        if (this.elementsCount > 1) {

            sink(1);
        }
    }

    /**
     * Clone this object. The returned clone will use the same resizing strategy and comparator.
     */
    @Override
    public DoubleHeapPriorityQueue clone() {
        //real constructor call, of a place holder
        final DoubleHeapPriorityQueue cloned = new DoubleHeapPriorityQueue(this.comparator,
                Containers.DEFAULT_EXPECTED_ELEMENTS, this.resizer);

        //clone raw buffers
        cloned.buffer = this.buffer.clone();

        cloned.defaultValue = this.defaultValue;
        cloned.elementsCount = this.elementsCount;

        return cloned;
    }

    /**
     * this instance and obj can only be equal to this if either: <pre>
     * (both don't have set comparators)
     * or
     * (both have equal comparators defined by {@link #comparator()}.equals(obj.comparator))</pre>
     * then, both heap elements are compared with {@link #equals(Object)} iterating their {@link #buffer}.
     */
    @Override
    /*  */
    public boolean equals(final Object obj) {
        if (obj != null) {
            if (obj == this) {
                return true;
            }

            //we can only compare both DoubleHeapPriorityQueue and not subclasses between themselves
            //that has the same comparison function reference
            if (obj.getClass() != this.getClass()) {
                return false;
            }

            final DoubleHeapPriorityQueue other = (DoubleHeapPriorityQueue) obj;

            if (other.size() != this.size()) {

                return false;
            }

            //If one comparator is null, and the other not, we cannot compare them, same if
            //both comparators are different because the heap behavior will be different, even elements are equal.
            if (!((this.comparator == null && other.comparator == null) || (this.comparator != null && this.comparator.equals(other.comparator)))) {

                return false;
            }

            final ValueIterator it = this.iterator();
            final ValueIterator itOther = other.iterator();

            while (it.hasNext()) {

                final double myVal = it.next().value;
                final double otherVal = itOther.next().value;

                if (!(Double.doubleToLongBits((myVal)) == Double.doubleToLongBits((otherVal)))) {
                    //recycle
                    it.release();
                    itOther.release();
                    return false;
                }
            } //end while
            itOther.release();

            return true;
        }

        return false;
    }

    /**
     * Ensures the internal buffer has enough free slots to store
     * <code>expectedAdditions</code>. Increases internal buffer size if needed.
     */
    @SuppressWarnings("boxing")
    protected void ensureBufferSpace(final int expectedAdditions) {
        final int bufferLen = this.buffer == null ? 0 : this.buffer.length;

        //element of index 0 is not used
        if (this.elementsCount + 1 > bufferLen - expectedAdditions) {
            int newSize = this.resizer.grow(bufferLen, this.elementsCount, expectedAdditions);

            //first allocation, reserve an additional slot because index 0  is not used
            if (this.buffer == null) {
                newSize++;
            }

            try {
                final double[] newBuffer = (new double[(newSize)]);

                if (bufferLen > 0) {
                    System.arraycopy(this.buffer, 0, newBuffer, 0, this.buffer.length);
                }

                this.buffer = newBuffer;
            } catch (final OutOfMemoryError e) {

                throw new BufferAllocationException(
                        "Not enough memory to allocate buffers to grow from %d -> %d elements",
                        e,
                        bufferLen,
                        newSize);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public double[] toArray(final double[] target) {
        //copy from index 1
        System.arraycopy(this.buffer, 1, target, 0, this.elementsCount);
        return target;
    }

    /**
     * Get the custom comparator used for comparing elements
     * @return null if no custom comparator was set, i.e natural ordering
     * of <code>double</code>s is used instead
     *      */
                                                                                         public DoubleComparator
                                                                                     
    comparator() {

        return this.comparator;
    }

    /**
     * Returns the "default value" value used
     * in methods returning "default value"
     */
    @Override
    public double getDefaultValue()
    {
        return this.defaultValue;
    }

    /**
     * Set the "default value" value to be used
     * in methods returning "default value"
     */
    @Override
    public void setDefaultValue(final double defaultValue)
    {
        this.defaultValue = defaultValue;
    }

    /**
     * Sink function for Comparable elements
     * 
     * @param k
     */
    private void sinkComparable(int k) {
        final int N = this.elementsCount;
        final double[] buffer = ((this.buffer));

        double tmp;
        int child;

        while ((k << 1) <= N) {
            //get the child of k
            child = k << 1;

            if (child < N && (Double.compare((buffer[child]) , (buffer[child + 1])) > 0)) {
                child++;
            }

            if (!(Double.compare((buffer[k]) , (buffer[child])) > 0)) {
                break;
            }

            //swap k and child
            tmp = buffer[k];
            buffer[k] = buffer[child];
            buffer[child] = tmp;

            k = child;
        } //end while
    }

    /**
     * Sink function for DoubleComparator elements
     * 
     * @param k
     */
    private void sinkComparator(int k) {
        final int N = this.elementsCount;
        final double[] buffer = ((this.buffer));

        double tmp;
        int child;
                     DoubleComparator comp = this.comparator;
             

        while ((k << 1) <= N) {
            //get the child of k
            child = k << 1;

            if (child < N && comp.compare(buffer[child], buffer[child + 1]) > 0) {
                child++;
            }

            if (comp.compare(buffer[k], buffer[child]) <= 0) {
                break;
            }

            //swap k and child
            tmp = buffer[k];
            buffer[k] = buffer[child];
            buffer[child] = tmp;

            k = child;
        } //end while
    }

    /**
     * Swim function for Comparable elements
     * 
     * @param k
     */
    private void swimComparable(int k) {
        double tmp;
        int parent;
        final double[] buffer = ((this.buffer));

        while (k > 1 && (Double.compare((buffer[k >> 1]) , (buffer[k])) > 0)) {
            //swap k and its parent
            parent = k >> 1;

        tmp = buffer[k];
        buffer[k] = buffer[parent];
        buffer[parent] = tmp;

        k = parent;
        }
    }

    /**
     * Swim function for Comparator elements
     * 
     * @param k
     */
    private void swimComparator(int k) {
        double tmp;
        int parent;
        final double[] buffer = ((this.buffer));

                     DoubleComparator comp = this.comparator;
             

        while (k > 1 && comp.compare(buffer[k >> 1], buffer[k]) > 0) {
            //swap k and its parent
            parent = k >> 1;
        tmp = buffer[k];
        buffer[k] = buffer[parent];
        buffer[parent] = tmp;

        k = parent;
        }
    }

    private void swim(final int k) {
        if (this.comparator == null) {
            swimComparable(k);
        } else {
            swimComparator(k);
        }
    }

    private void sink(final int k) {
        if (this.comparator == null) {
            sinkComparable(k);
        } else {
            sinkComparator(k);
        }
    }

      
}
