/*
 * Copyright (C) 2011 Thomas Akehurst
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.github.tomakehurst.wiremock.http;

import com.github.tomakehurst.wiremock.extension.requestfilter.ContinueAction;
import com.github.tomakehurst.wiremock.extension.requestfilter.RequestFilter;
import com.github.tomakehurst.wiremock.extension.requestfilter.RequestFilterAction;
import com.github.tomakehurst.wiremock.extension.requestfilter.StopAction;
import com.github.tomakehurst.wiremock.stubbing.ServeEvent;
import com.github.tomakehurst.wiremock.verification.LoggedRequest;
import com.google.common.base.Stopwatch;

import java.util.List;

import static com.github.tomakehurst.wiremock.common.LocalNotifier.notifier;
import static com.google.common.collect.Lists.newArrayList;
import static java.util.concurrent.TimeUnit.MILLISECONDS;

public abstract class AbstractRequestHandler implements RequestHandler, RequestEventSource {

	protected List<RequestListener> listeners = newArrayList();
	protected final ResponseRenderer responseRenderer;
	protected final List<RequestFilter> requestFilters;

	public AbstractRequestHandler(ResponseRenderer responseRenderer, List<RequestFilter> requestFilters) {
		this.responseRenderer = responseRenderer;
		this.requestFilters = requestFilters;
	}

	@Override
	public void addRequestListener(RequestListener requestListener) {
		listeners.add(requestListener);
	}

	protected void beforeResponseSent(ServeEvent serveEvent, Response response) {}
    protected void afterResponseSent(ServeEvent serveEvent, Response response) {}

	@Override
	public void handle(Request request, HttpResponder httpResponder) {
        Stopwatch stopwatch = Stopwatch.createStarted();

		ServeEvent serveEvent;
		Request originalRequest = request;
		if (!requestFilters.isEmpty()) {
			RequestFilterAction requestFilterAction = processFilters(request, requestFilters, RequestFilterAction.continueWith(request));
			if (requestFilterAction instanceof ContinueAction) {
				Request wrappedRequest = ((ContinueAction) requestFilterAction).getRequest();
				originalRequest = wrappedRequest;
				serveEvent = handleRequest(wrappedRequest);
			} else {
				serveEvent = ServeEvent.of(LoggedRequest.createFrom(request), ((StopAction) requestFilterAction).getResponseDefinition());
			}
		} else {
			serveEvent = handleRequest(request);
		}

		ResponseDefinition responseDefinition = serveEvent.getResponseDefinition();
		responseDefinition.setOriginalRequest(originalRequest);
		Response response = responseRenderer.render(serveEvent);
		ServeEvent completedServeEvent = serveEvent.complete(response, (int) stopwatch.elapsed(MILLISECONDS));

		if (logRequests()) {
			notifier().info("Request received:\n" +
					formatRequest(request) +
					"\n\nMatched response definition:\n" + responseDefinition +
					"\n\nResponse:\n" + response);
		}

		for (RequestListener listener: listeners) {
			listener.requestReceived(request, response);
		}

        beforeResponseSent(completedServeEvent, response);

		stopwatch.reset();
		stopwatch.start();
		httpResponder.respond(request, response);

        completedServeEvent.afterSend((int) stopwatch.elapsed(MILLISECONDS));
        afterResponseSent(completedServeEvent, response);
        stopwatch.stop();
	}

	private static RequestFilterAction processFilters(Request request, List<RequestFilter> requestFilters, RequestFilterAction lastAction) {
		if (requestFilters.isEmpty()) {
			return lastAction;
		}

		RequestFilterAction action = requestFilters.get(0).filter(request);

		if (action instanceof ContinueAction) {
			Request newRequest = ((ContinueAction) action).getRequest();
			return processFilters(newRequest, requestFilters.subList(1, requestFilters.size()), action);
		}

		return action;
	}

	protected String formatRequest(Request request) {
		StringBuilder sb = new StringBuilder();
		sb.append(request.getClientIp())
				.append(" - ")
				.append(request.getMethod())
				.append(" ")
				.append(request.getUrl());

		if (request.isBrowserProxyRequest()) {
			sb.append(" (via browser proxy request)");
		}

		sb.append("\n\n");
		sb.append(request.getHeaders());

		if (request.getBody() != null) {
			sb.append(request.getBodyAsString()).append("\n");
		}

		return sb.toString();
	}

	protected boolean logRequests() { return false; }

	protected abstract ServeEvent handleRequest(Request request);
}
