/*
 * Decompiled with CFR 0.152.
 */
package com.epam.reportportal.service.launch.lock;

import com.epam.reportportal.exception.InternalReportPortalClientException;
import com.epam.reportportal.listeners.ListenerParameters;
import com.epam.reportportal.service.launch.lock.AbstractLaunchIdLock;
import com.epam.reportportal.utils.Waiter;
import jakarta.annotation.Nonnull;
import jakarta.annotation.Nullable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LaunchIdLockFile
extends AbstractLaunchIdLock {
    private static final Logger LOGGER = LoggerFactory.getLogger(LaunchIdLockFile.class);
    public static final Charset LOCK_FILE_CHARSET = StandardCharsets.ISO_8859_1;
    public static final String TIME_SEPARATOR = ":";
    private static final String LINE_SEPARATOR = System.lineSeparator();
    private final File lockFile;
    private final File syncFile;
    private final long fileWaitTimeout;
    private static volatile String lockUuid;
    private static volatile Pair<RandomAccessFile, FileLock> mainLock;

    public LaunchIdLockFile(@Nonnull ListenerParameters listenerParameters) {
        super(listenerParameters);
        this.lockFile = new File(this.parameters.getLockFileName());
        this.syncFile = new File(this.parameters.getSyncFileName());
        this.fileWaitTimeout = this.parameters.getLockWaitTimeout();
    }

    @Nullable
    private Pair<RandomAccessFile, FileLock> obtainLock(@Nonnull File file) {
        RandomAccessFile lockAccess;
        String filePath = file.getPath();
        try {
            lockAccess = new RandomAccessFile(file, "rwd");
        }
        catch (FileNotFoundException e) {
            LOGGER.debug("Unable to open '{}' file: {}", new Object[]{filePath, e.getLocalizedMessage(), e});
            return null;
        }
        try {
            FileLock lock = lockAccess.getChannel().tryLock();
            if (lock == null) {
                LaunchIdLockFile.closeAccess(lockAccess);
                return null;
            }
            return Pair.of((Object)lockAccess, (Object)lock);
        }
        catch (OverlappingFileLockException e) {
            LOGGER.debug("Lock already acquired on '{}' file: {}", new Object[]{filePath, e.getLocalizedMessage(), e});
        }
        catch (ClosedChannelException e) {
            LOGGER.warn("Channel was already closed on '{}' file: {}", new Object[]{filePath, e.getLocalizedMessage(), e});
        }
        catch (IOException e) {
            LOGGER.warn("Unexpected I/O exception while obtaining mainLock on '{}' file: {}", new Object[]{filePath, e.getLocalizedMessage(), e});
        }
        LaunchIdLockFile.closeAccess(lockAccess);
        return null;
    }

    private static void releaseLock(@Nonnull FileLock lock) {
        try {
            lock.release();
        }
        catch (ClosedChannelException e) {
            LOGGER.warn("Channel was already closed for file mainLock: {}", (Object)e.getLocalizedMessage(), (Object)e);
        }
        catch (IOException e) {
            LOGGER.warn("Unexpected I/O exception while releasing file mainLock: {}", (Object)e.getLocalizedMessage(), (Object)e);
        }
    }

    private static void closeAccess(@Nonnull RandomAccessFile access) {
        try {
            access.close();
        }
        catch (IOException e) {
            LOGGER.warn("Unexpected I/O exception while closing file: {}", (Object)e.getLocalizedMessage(), (Object)e);
        }
    }

    @Nullable
    private static String readLaunchUuid(@Nonnull RandomAccessFile access) throws IOException {
        String launchRecord = access.readLine();
        if (launchRecord == null) {
            return null;
        }
        return launchRecord.substring(launchRecord.indexOf(TIME_SEPARATOR) + 1);
    }

    private static void writeString(@Nonnull RandomAccessFile access, @Nonnull String text) throws IOException {
        access.write(text.getBytes(LOCK_FILE_CHARSET));
    }

    private static void writeLine(@Nonnull RandomAccessFile access, @Nonnull String text) throws IOException {
        LaunchIdLockFile.writeString(access, text + LINE_SEPARATOR);
    }

    private static void closeIo(@Nonnull Pair<RandomAccessFile, FileLock> io) {
        LaunchIdLockFile.releaseLock((FileLock)io.getRight());
        LaunchIdLockFile.closeAccess((RandomAccessFile)io.getLeft());
    }

    @Nullable
    private <T> T executeOperation(@Nonnull IoOperation<T> operation, @Nonnull Pair<RandomAccessFile, FileLock> fileIo) {
        try {
            return operation.execute(fileIo);
        }
        catch (IOException e) {
            LOGGER.error("Unable to read/write a file after obtaining mainLock: {}", (Object)e.getMessage(), (Object)e);
            return null;
        }
    }

    @Nullable
    private <T> T executeBlockingOperation(@Nonnull IoOperation<T> operation, @Nonnull File file) {
        return (T)new Waiter("Wait for a blocking operation on file '" + file.getPath() + "'").duration(this.fileWaitTimeout, TimeUnit.MILLISECONDS).applyRandomDiscrepancy(0.1f).till(() -> {
            Pair<RandomAccessFile, FileLock> fileIo = this.obtainLock(file);
            if (fileIo != null) {
                Object result = this.executeOperation(operation, fileIo);
                LaunchIdLockFile.closeIo(fileIo);
                return result;
            }
            return null;
        });
    }

    private void rewriteWith(@Nonnull RandomAccessFile access, @Nonnull String content) throws IOException {
        access.setLength(content.length());
        LaunchIdLockFile.writeLine(access, content);
    }

    void reset() {
        if (mainLock != null) {
            LaunchIdLockFile.closeIo(mainLock);
            mainLock = null;
        }
        lockUuid = null;
    }

    @Nonnull
    private static String getRecord(@Nonnull String instanceUuid) {
        return System.currentTimeMillis() + TIME_SEPARATOR + instanceUuid;
    }

    private void writeLaunchUuid(@Nonnull Pair<RandomAccessFile, FileLock> syncIo) {
        String launchRecord = LaunchIdLockFile.getRecord(lockUuid);
        try {
            this.rewriteWith((RandomAccessFile)syncIo.getLeft(), launchRecord);
            this.rewriteWith((RandomAccessFile)mainLock.getLeft(), launchRecord);
        }
        catch (IOException e) {
            String error = "Unable to read/write a file after obtaining lock: " + e.getMessage();
            LOGGER.warn(error, (Throwable)e);
            this.reset();
            throw new InternalReportPortalClientException(error, e);
        }
    }

    private static void appendUuid(@Nonnull RandomAccessFile access, @Nonnull String uuid) throws IOException {
        access.seek(access.length());
        LaunchIdLockFile.writeLine(access, LaunchIdLockFile.getRecord(uuid));
    }

    private void writeInstanceUuid(@Nonnull String instanceUuid) {
        this.executeBlockingOperation(new UuidAppend(instanceUuid), this.syncFile);
    }

    @Nullable
    private String obtainLaunch(@Nonnull String instanceUuid) {
        return this.executeBlockingOperation(new LaunchRead(instanceUuid), this.syncFile);
    }

    @Override
    @Nullable
    public String obtainLaunchUuid(@Nonnull String instanceUuid) {
        Objects.requireNonNull(instanceUuid);
        if (mainLock != null) {
            if (!instanceUuid.equals(lockUuid)) {
                this.writeInstanceUuid(instanceUuid);
            }
            return lockUuid;
        }
        Pair<RandomAccessFile, FileLock> syncLock = this.obtainLock(this.syncFile);
        if (syncLock != null) {
            if (mainLock == null) {
                Pair<RandomAccessFile, FileLock> lock = this.obtainLock(this.lockFile);
                if (lock != null) {
                    lockUuid = instanceUuid;
                    mainLock = lock;
                    this.writeLaunchUuid(syncLock);
                    LaunchIdLockFile.closeIo(syncLock);
                    return instanceUuid;
                }
                this.executeOperation(new LaunchRead(instanceUuid), syncLock);
                LaunchIdLockFile.closeIo(syncLock);
            } else {
                this.executeOperation(new UuidAppend(instanceUuid), syncLock);
                LaunchIdLockFile.closeIo(syncLock);
                return lockUuid;
            }
        }
        return this.obtainLaunch(instanceUuid);
    }

    @Override
    public void updateInstanceUuid(@Nonnull String instanceUuid) {
        IoOperation<Boolean> uuidUpdate = fileIo -> {
            String line;
            ArrayList<String> recordList = new ArrayList<String>();
            RandomAccessFile fileAccess = (RandomAccessFile)fileIo.getKey();
            boolean needUpdate = false;
            while ((line = fileAccess.readLine()) != null) {
                String record = line.trim();
                String uuid = record.substring(record.indexOf(TIME_SEPARATOR) + 1);
                if (instanceUuid.equals(uuid)) {
                    String newRecord = LaunchIdLockFile.getRecord(instanceUuid);
                    if (!newRecord.equals(record)) {
                        needUpdate = true;
                        recordList.add(newRecord);
                        continue;
                    }
                    recordList.add(record);
                    continue;
                }
                recordList.add(record);
            }
            if (needUpdate) {
                fileAccess.seek(0L);
                for (String record : recordList) {
                    LaunchIdLockFile.writeLine(fileAccess, record);
                }
            }
            return needUpdate;
        };
        this.executeBlockingOperation(uuidUpdate, this.syncFile);
    }

    @Override
    public void finishInstanceUuid(@Nonnull String instanceUuid) {
        IoOperation<Boolean> uuidRemove = fileIo -> {
            String line;
            ArrayList<String> recordList = new ArrayList<String>();
            RandomAccessFile fileAccess = (RandomAccessFile)fileIo.getKey();
            boolean needUpdate = false;
            while ((line = fileAccess.readLine()) != null) {
                String record = line.trim();
                String launchUuid = record.substring(record.indexOf(TIME_SEPARATOR) + 1);
                if (instanceUuid.equals(launchUuid)) {
                    needUpdate = true;
                    continue;
                }
                recordList.add(record);
            }
            if (!needUpdate) {
                return false;
            }
            String recordNl = System.currentTimeMillis() + TIME_SEPARATOR + instanceUuid + LINE_SEPARATOR;
            long newLength = fileAccess.length() - (long)recordNl.length();
            if (newLength > 0L) {
                fileAccess.setLength(newLength);
                fileAccess.seek(0L);
                for (String record : recordList) {
                    LaunchIdLockFile.writeLine(fileAccess, record);
                }
                return false;
            }
            ((RandomAccessFile)fileIo.getKey()).setLength(0L);
            return true;
        };
        Boolean isLast = this.executeBlockingOperation(uuidRemove, this.syncFile);
        if (isLast != null && isLast.booleanValue() && !this.syncFile.delete()) {
            LOGGER.warn("Unable to delete synchronization file: {}", (Object)this.syncFile.getPath());
        }
        if (mainLock != null && lockUuid.equals(instanceUuid)) {
            this.reset();
            if (!this.lockFile.delete()) {
                LOGGER.warn("Unable to delete locking file: {}", (Object)this.lockFile.getPath());
            }
        }
    }

    @Override
    @Nonnull
    public Collection<String> getLiveInstanceUuids() {
        IoOperation<List> uuidListRead = fileIo -> {
            String line;
            ArrayList<String> recordList = new ArrayList<String>();
            RandomAccessFile fileAccess = (RandomAccessFile)fileIo.getKey();
            while ((line = fileAccess.readLine()) != null) {
                String record = line.trim();
                recordList.add(record);
            }
            return recordList;
        };
        long timeoutTime = System.currentTimeMillis() - this.fileWaitTimeout;
        return Optional.ofNullable(this.executeBlockingOperation(uuidListRead, this.syncFile)).orElse(Collections.emptyList()).stream().map(r -> Pair.of((Object)Long.parseLong(r.substring(0, r.indexOf(TIME_SEPARATOR))), (Object)r.substring(r.indexOf(TIME_SEPARATOR) + 1))).filter(r -> (Long)r.getKey() > timeoutTime).map(Pair::getValue).collect(Collectors.toSet());
    }

    private static class LaunchRead
    extends UuidAppend {
        public LaunchRead(@Nonnull String instanceUuid) {
            super(instanceUuid);
        }

        @Override
        public String execute(@Nonnull Pair<RandomAccessFile, FileLock> lock) throws IOException {
            RandomAccessFile access = (RandomAccessFile)lock.getKey();
            String launchUuid = LaunchIdLockFile.readLaunchUuid(access);
            super.execute((Pair)lock);
            return Optional.ofNullable(launchUuid).orElse(this.uuid);
        }
    }

    private static class UuidAppend
    implements IoOperation<String> {
        final String uuid;

        public UuidAppend(@Nonnull String instanceUuid) {
            this.uuid = instanceUuid;
        }

        @Override
        public String execute(@Nonnull Pair<RandomAccessFile, FileLock> lock) throws IOException {
            RandomAccessFile access = (RandomAccessFile)lock.getKey();
            LaunchIdLockFile.appendUuid(access, this.uuid);
            return this.uuid;
        }
    }

    private static interface IoOperation<T> {
        public T execute(@Nonnull Pair<RandomAccessFile, FileLock> var1) throws IOException;
    }
}

