/*
 * Decompiled with CFR 0.152.
 */
package com.digitalsanctuary.cf.turnstile.service;

import com.digitalsanctuary.cf.turnstile.config.TurnstileConfigProperties;
import com.digitalsanctuary.cf.turnstile.dto.TurnstileResponse;
import com.digitalsanctuary.cf.turnstile.dto.ValidationResult;
import com.digitalsanctuary.cf.turnstile.exception.TurnstileConfigurationException;
import com.digitalsanctuary.cf.turnstile.exception.TurnstileNetworkException;
import com.digitalsanctuary.cf.turnstile.exception.TurnstileValidationException;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Timer;
import jakarta.annotation.PostConstruct;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.http.HttpServletRequest;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.LongAdder;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.HttpServerErrorException;
import org.springframework.web.client.ResourceAccessException;
import org.springframework.web.client.RestClient;

@Service
public class TurnstileValidationService {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(TurnstileValidationService.class);
    private static final String UNKNOWN = "unknown";
    private final RestClient turnstileRestClient;
    private final TurnstileConfigProperties properties;
    private final Optional<MeterRegistry> meterRegistry;
    private final LongAdder validationCount = new LongAdder();
    private final LongAdder successCount = new LongAdder();
    private final LongAdder errorCount = new LongAdder();
    private final LongAdder networkErrorCount = new LongAdder();
    private final LongAdder configErrorCount = new LongAdder();
    private final LongAdder validationErrorCount = new LongAdder();
    private final LongAdder inputErrorCount = new LongAdder();
    private final AtomicLong lastResponseTime = new AtomicLong();
    private final AtomicLong totalResponseTime = new AtomicLong();
    private final AtomicLong responseCount = new AtomicLong();
    private Counter validationCounter;
    private Counter successCounter;
    private Counter errorCounter;
    private Counter networkErrorCounter;
    private Counter configErrorCounter;
    private Counter validationErrorCounter;
    private Counter inputErrorCounter;
    private Timer responseTimer;

    public TurnstileValidationService(@Qualifier(value="turnstileRestClient") RestClient turnstileRestClient, TurnstileConfigProperties properties, Optional<MeterRegistry> meterRegistry) {
        this.turnstileRestClient = turnstileRestClient;
        this.properties = properties;
        this.meterRegistry = meterRegistry;
        this.initMetrics();
    }

    public TurnstileValidationService(@Qualifier(value="turnstileRestClient") RestClient turnstileRestClient, TurnstileConfigProperties properties) {
        this(turnstileRestClient, properties, Optional.empty());
    }

    private void initMetrics() {
        this.meterRegistry.ifPresent(registry -> {
            log.info("Initializing Turnstile metrics with MeterRegistry");
            this.validationCounter = Counter.builder((String)"turnstile.validation.requests").description("Total number of Turnstile validation requests").register(registry);
            this.successCounter = Counter.builder((String)"turnstile.validation.success").description("Number of successful Turnstile validations").register(registry);
            this.errorCounter = Counter.builder((String)"turnstile.validation.errors").description("Number of failed Turnstile validations").register(registry);
            this.networkErrorCounter = Counter.builder((String)"turnstile.validation.errors.network").description("Number of Turnstile validation network errors").register(registry);
            this.configErrorCounter = Counter.builder((String)"turnstile.validation.errors.config").description("Number of Turnstile validation configuration errors").register(registry);
            this.validationErrorCounter = Counter.builder((String)"turnstile.validation.errors.token").description("Number of Turnstile validation token errors").register(registry);
            this.inputErrorCounter = Counter.builder((String)"turnstile.validation.errors.input").description("Number of Turnstile validation input errors").register(registry);
            this.responseTimer = Timer.builder((String)"turnstile.validation.response.time").description("Response time for Turnstile validation requests").register(registry);
        });
    }

    @PostConstruct
    public void onStartup() {
        log.info("TurnstileValidationService started");
        log.info("Turnstile URL: {}", (Object)this.properties.getUrl());
        log.info("Turnstile Sitekey: {}", (Object)this.properties.getSitekey());
        log.info("Turnstile Metrics enabled: {}", (Object)this.properties.getMetrics().isEnabled());
        log.info("Turnstile Health Check enabled: {}", (Object)this.properties.getMetrics().isHealthCheckEnabled());
        if (this.properties.getSecret() == null || this.properties.getSecret().isBlank()) {
            log.error("Turnstile secret key is not configured. Validation will fail.");
        }
        if (this.properties.getUrl() == null || this.properties.getUrl().isBlank()) {
            log.error("Turnstile URL is not configured. Validation will fail.");
        }
    }

    public boolean validateTurnstileResponse(String token) {
        return this.validateTurnstileResponse(token, null);
    }

    public boolean validateTurnstileResponse(String token, String remoteIp) {
        try {
            ValidationResult result = this.validateTurnstileResponseDetailed(token, remoteIp);
            return result.isSuccess();
        }
        catch (Exception e) {
            log.error("Unexpected error during Turnstile validation: {}", (Object)e.getMessage(), (Object)e);
            return false;
        }
    }

    public ValidationResult validateTurnstileResponseDetailed(String token) {
        return this.validateTurnstileResponseDetailed(token, null);
    }

    public ValidationResult validateTurnstileResponseDetailed(String token, String remoteIp) {
        long startTime = System.currentTimeMillis();
        this.validationCount.increment();
        if (this.validationCounter != null) {
            this.validationCounter.increment();
        }
        log.trace("Starting validation for token: {} with remoteIp: {}", (Object)token, (Object)remoteIp);
        if (token == null) {
            log.warn("Turnstile validation failed: token cannot be null");
            this.recordError(ValidationResult.ValidationResultType.INPUT_ERROR);
            return ValidationResult.inputError("Token cannot be null");
        }
        if (token.isEmpty() || token.isBlank()) {
            log.warn("Turnstile validation failed: token cannot be empty or blank");
            this.recordError(ValidationResult.ValidationResultType.INPUT_ERROR);
            return ValidationResult.inputError("Token cannot be empty or blank");
        }
        if (token.length() < 20) {
            log.warn("Turnstile validation failed: token appears to be too short to be valid (length: {})", (Object)token.length());
            this.recordError(ValidationResult.ValidationResultType.INPUT_ERROR);
            return ValidationResult.inputError("Token is too short to be valid (length: " + token.length() + ")");
        }
        if (remoteIp != null && (remoteIp.isEmpty() || remoteIp.isBlank())) {
            log.warn("Turnstile validation: ignoring empty or blank remoteIp");
            remoteIp = null;
        }
        if (this.properties.getSecret() == null || this.properties.getSecret().isBlank()) {
            String msg = "Turnstile secret key is not configured";
            log.error(msg);
            this.recordError(ValidationResult.ValidationResultType.CONFIGURATION_ERROR);
            throw new TurnstileConfigurationException(msg);
        }
        if (this.properties.getUrl() == null || this.properties.getUrl().isBlank()) {
            String msg = "Turnstile URL is not configured";
            log.error(msg);
            this.recordError(ValidationResult.ValidationResultType.CONFIGURATION_ERROR);
            throw new TurnstileConfigurationException(msg);
        }
        HashMap<String, String> requestBody = new HashMap<String, String>();
        requestBody.put("secret", this.properties.getSecret());
        requestBody.put("response", token);
        Optional.ofNullable(remoteIp).ifPresent(ip -> requestBody.put("remoteip", (String)ip));
        log.trace("Making request to Cloudflare Turnstile API at: {}", (Object)this.properties.getUrl());
        try {
            if (this.responseTimer != null) {
                return (ValidationResult)this.responseTimer.record(() -> this.executeValidationRequest(requestBody, startTime));
            }
            return this.executeValidationRequest(requestBody, startTime);
        }
        catch (HttpClientErrorException e) {
            log.error("Client error during Turnstile validation: {}", (Object)e.getMessage(), (Object)e);
            this.recordError(ValidationResult.ValidationResultType.NETWORK_ERROR);
            throw new TurnstileNetworkException("Client error: " + e.getMessage(), e);
        }
        catch (HttpServerErrorException e) {
            log.error("Server error during Turnstile validation: {}", (Object)e.getMessage(), (Object)e);
            this.recordError(ValidationResult.ValidationResultType.NETWORK_ERROR);
            throw new TurnstileNetworkException("Server error: " + e.getMessage(), e);
        }
        catch (ResourceAccessException e) {
            log.error("Network error during Turnstile validation: {}", (Object)e.getMessage(), (Object)e);
            this.recordError(ValidationResult.ValidationResultType.NETWORK_ERROR);
            throw new TurnstileNetworkException("Network error: " + e.getMessage(), e);
        }
        catch (TurnstileValidationException e) {
            this.recordError(ValidationResult.ValidationResultType.INVALID_TOKEN);
            throw e;
        }
        catch (Exception e) {
            log.error("Unexpected error during Turnstile validation: {}", (Object)e.getMessage(), (Object)e);
            this.recordError(ValidationResult.ValidationResultType.NETWORK_ERROR);
            throw new TurnstileNetworkException("Unexpected error: " + e.getMessage(), e);
        }
    }

    private ValidationResult executeValidationRequest(Map<String, String> requestBody, long startTime) {
        TurnstileResponse response = (TurnstileResponse)((RestClient.RequestBodySpec)((RestClient.RequestBodySpec)this.turnstileRestClient.post().uri(this.properties.getUrl(), new Object[0])).header("Content-Type", new String[]{"application/json"})).body(requestBody).retrieve().body(TurnstileResponse.class);
        long responseTime = System.currentTimeMillis() - startTime;
        this.lastResponseTime.set(responseTime);
        this.totalResponseTime.addAndGet(responseTime);
        this.responseCount.incrementAndGet();
        log.debug("Turnstile response: {} (took {}ms)", (Object)response, (Object)responseTime);
        if (response == null) {
            log.warn("Turnstile API returned null response");
            this.recordError(ValidationResult.ValidationResultType.NETWORK_ERROR);
            return ValidationResult.networkError("Cloudflare returned an empty response");
        }
        if (response.isSuccess()) {
            log.debug("Turnstile validation successful");
            this.successCount.increment();
            if (this.successCounter != null) {
                this.successCounter.increment();
            }
            return ValidationResult.success();
        }
        log.warn("Turnstile validation failed with error codes: {}", response.getErrorCodes());
        this.recordError(ValidationResult.ValidationResultType.INVALID_TOKEN);
        throw new TurnstileValidationException("Token validation failed", response.getErrorCodes());
    }

    private void recordError(ValidationResult.ValidationResultType resultType) {
        this.errorCount.increment();
        if (this.errorCounter != null) {
            this.errorCounter.increment();
        }
        switch (resultType) {
            case NETWORK_ERROR: {
                this.networkErrorCount.increment();
                if (this.networkErrorCounter == null) break;
                this.networkErrorCounter.increment();
                break;
            }
            case CONFIGURATION_ERROR: {
                this.configErrorCount.increment();
                if (this.configErrorCounter == null) break;
                this.configErrorCounter.increment();
                break;
            }
            case INVALID_TOKEN: {
                this.validationErrorCount.increment();
                if (this.validationErrorCounter == null) break;
                this.validationErrorCounter.increment();
                break;
            }
            case INPUT_ERROR: {
                this.inputErrorCount.increment();
                if (this.inputErrorCounter == null) break;
                this.inputErrorCounter.increment();
                break;
            }
        }
    }

    public long getValidationCount() {
        return this.validationCount.sum();
    }

    public long getSuccessCount() {
        return this.successCount.sum();
    }

    public long getErrorCount() {
        return this.errorCount.sum();
    }

    public long getNetworkErrorCount() {
        return this.networkErrorCount.sum();
    }

    public long getConfigErrorCount() {
        return this.configErrorCount.sum();
    }

    public long getValidationErrorCount() {
        return this.validationErrorCount.sum();
    }

    public long getInputErrorCount() {
        return this.inputErrorCount.sum();
    }

    public long getLastResponseTime() {
        return this.lastResponseTime.get();
    }

    public double getAverageResponseTime() {
        long count = this.responseCount.get();
        return count > 0L ? (double)this.totalResponseTime.get() / (double)count : 0.0;
    }

    public double getErrorRate() {
        long total = this.validationCount.sum();
        return total > 0L ? (double)this.errorCount.sum() * 100.0 / (double)total : 0.0;
    }

    @Deprecated
    public String getTurnsiteSitekey() {
        return this.getTurnstileSitekey();
    }

    public String getTurnstileSitekey() {
        return this.properties.getSitekey();
    }

    public String getClientIpAddress(ServletRequest request) {
        if (request instanceof HttpServletRequest) {
            String[] headers;
            HttpServletRequest httpRequest = (HttpServletRequest)request;
            for (String header : headers = new String[]{"X-Forwarded-For", "Proxy-Client-IP", "WL-Proxy-Client-IP", "HTTP_CLIENT_IP", "HTTP_X_FORWARDED_FOR"}) {
                String ip = httpRequest.getHeader(header);
                if (ip == null || ip.isEmpty() || UNKNOWN.equalsIgnoreCase(ip)) continue;
                return ip;
            }
        }
        return request.getRemoteAddr();
    }
}

