/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.oss.dsbulk.workflow.commons.log;

import com.datastax.oss.driver.api.core.AllNodesFailedException;
import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.DriverTimeoutException;
import com.datastax.oss.driver.api.core.ProtocolVersion;
import com.datastax.oss.driver.api.core.RequestThrottlingException;
import com.datastax.oss.driver.api.core.connection.BusyConnectionException;
import com.datastax.oss.driver.api.core.connection.FrameTooLongException;
import com.datastax.oss.driver.api.core.cql.BatchStatement;
import com.datastax.oss.driver.api.core.cql.BatchableStatement;
import com.datastax.oss.driver.api.core.cql.ExecutionInfo;
import com.datastax.oss.driver.api.core.cql.Statement;
import com.datastax.oss.driver.api.core.servererrors.InvalidQueryException;
import com.datastax.oss.driver.api.core.servererrors.QueryExecutionException;
import com.datastax.oss.driver.api.core.servererrors.ServerError;
import com.datastax.oss.driver.api.core.type.codec.registry.CodecRegistry;
import com.datastax.oss.driver.shaded.guava.common.base.Joiner;
import com.datastax.oss.dsbulk.connectors.api.ErrorRecord;
import com.datastax.oss.dsbulk.connectors.api.Record;
import com.datastax.oss.dsbulk.executor.api.exception.BulkExecutionException;
import com.datastax.oss.dsbulk.executor.api.result.ReadResult;
import com.datastax.oss.dsbulk.executor.api.result.Result;
import com.datastax.oss.dsbulk.executor.api.result.WriteResult;
import com.datastax.oss.dsbulk.format.row.RowFormatter;
import com.datastax.oss.dsbulk.format.statement.StatementFormatVerbosity;
import com.datastax.oss.dsbulk.format.statement.StatementFormatter;
import com.datastax.oss.dsbulk.workflow.api.error.ErrorThreshold;
import com.datastax.oss.dsbulk.workflow.api.error.TooManyErrorsException;
import com.datastax.oss.dsbulk.workflow.commons.log.LogManagerUtils;
import com.datastax.oss.dsbulk.workflow.commons.log.PositionsTracker;
import com.datastax.oss.dsbulk.workflow.commons.log.Range;
import com.datastax.oss.dsbulk.workflow.commons.schema.InvalidMappingException;
import com.datastax.oss.dsbulk.workflow.commons.statement.MappedStatement;
import com.datastax.oss.dsbulk.workflow.commons.statement.UnmappableStatement;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.IOException;
import java.io.PrintWriter;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.StandardOpenOption;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.FluxSink;
import reactor.core.publisher.Hooks;
import reactor.core.publisher.Mono;
import reactor.core.publisher.UnicastProcessor;

public class LogManager
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(LogManager.class);
    private static final String MAPPING_ERRORS_FILE = "mapping-errors.log";
    private static final String CONNECTOR_ERRORS_FILE = "connector-errors.log";
    private static final String UNLOAD_ERRORS_FILE = "unload-errors.log";
    private static final String LOAD_ERRORS_FILE = "load-errors.log";
    private static final String CAS_ERRORS_FILE = "paxos-errors.log";
    private static final String CONNECTOR_BAD_FILE = "connector.bad";
    private static final String MAPPING_BAD_FILE = "mapping.bad";
    private static final String LOAD_BAD_FILE = "load.bad";
    private static final String CAS_BAD_FILE = "paxos.bad";
    private static final String POSITIONS_FILE = "positions.txt";
    private final CqlSession session;
    private final Path operationDirectory;
    private final ErrorThreshold errorThreshold;
    private final ErrorThreshold queryWarningsThreshold;
    private final boolean trackPositions;
    private final StatementFormatter statementFormatter;
    private final StatementFormatVerbosity statementFormatVerbosity;
    private final RowFormatter rowFormatter;
    private final AtomicInteger errors = new AtomicInteger(0);
    private final LongAdder totalItems = new LongAdder();
    private final AtomicInteger queryWarnings = new AtomicInteger(0);
    private final AtomicBoolean queryWarningsEnabled = new AtomicBoolean(true);
    private final LoadingCache<Path, PrintWriter> openFiles = Caffeine.newBuilder().build(path -> new PrintWriter(Files.newBufferedWriter(path, StandardCharsets.UTF_8, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE)));
    private CodecRegistry codecRegistry;
    private ProtocolVersion protocolVersion;
    private PositionsTracker positionsTracker;
    private PrintWriter positionsPrinter;
    private FluxSink<ErrorRecord> failedRecordSink;
    private FluxSink<ErrorRecord> unmappableRecordSink;
    private FluxSink<UnmappableStatement> unmappableStatementSink;
    private FluxSink<WriteResult> failedWriteSink;
    private FluxSink<WriteResult> failedCASWriteSink;
    private FluxSink<ReadResult> failedReadSink;
    private FluxSink<Record> positionsSink;
    private UnicastProcessor<Void> uncaughtExceptionProcessor;
    private FluxSink<Void> uncaughtExceptionSink;
    private AtomicBoolean invalidMappingWarningDone;

    public LogManager(CqlSession session, Path operationDirectory, ErrorThreshold errorThreshold, ErrorThreshold queryWarningsThreshold, boolean trackPositions, StatementFormatter statementFormatter, StatementFormatVerbosity statementFormatVerbosity, RowFormatter rowFormatter) {
        this.session = session;
        this.operationDirectory = operationDirectory;
        this.errorThreshold = errorThreshold;
        this.queryWarningsThreshold = queryWarningsThreshold;
        this.trackPositions = trackPositions;
        this.statementFormatter = statementFormatter;
        this.statementFormatVerbosity = statementFormatVerbosity;
        this.rowFormatter = rowFormatter;
    }

    public void init() {
        this.codecRegistry = this.session.getContext().getCodecRegistry();
        this.protocolVersion = this.session.getContext().getProtocolVersion();
        this.positionsTracker = new PositionsTracker();
        this.failedRecordSink = this.newFailedRecordSink();
        this.unmappableRecordSink = this.newUnmappableRecordSink();
        this.unmappableStatementSink = this.newUnmappableStatementSink();
        this.failedWriteSink = this.newFailedWriteResultSink();
        this.failedCASWriteSink = this.newFailedCASWriteSink();
        this.failedReadSink = this.newFailedReadResultSink();
        this.positionsSink = this.newPositionsSink();
        this.uncaughtExceptionProcessor = UnicastProcessor.create();
        this.uncaughtExceptionSink = this.uncaughtExceptionProcessor.sink();
        this.invalidMappingWarningDone = new AtomicBoolean(false);
        Hooks.onErrorDropped(t -> this.uncaughtExceptionSink.error(t));
        Thread.setDefaultUncaughtExceptionHandler((thread, t) -> this.uncaughtExceptionSink.error(t));
    }

    public Path getOperationDirectory() {
        return this.operationDirectory;
    }

    public int getTotalErrors() {
        return this.errors.get();
    }

    @Override
    public void close() throws IOException {
        this.failedRecordSink.complete();
        this.unmappableRecordSink.complete();
        this.unmappableStatementSink.complete();
        this.failedWriteSink.complete();
        this.failedCASWriteSink.complete();
        this.failedReadSink.complete();
        this.uncaughtExceptionSink.complete();
        this.openFiles.asMap().values().forEach(pw -> {
            pw.flush();
            pw.close();
        });
        this.positionsSink.complete();
        if (this.trackPositions && !this.positionsTracker.isEmpty()) {
            this.positionsPrinter = new PrintWriter(Files.newBufferedWriter(this.operationDirectory.resolve(POSITIONS_FILE), StandardCharsets.UTF_8, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE));
            new TreeMap<URI, List<Range>>(this.positionsTracker.getPositions()).forEach((resource, ranges) -> LogManager.appendToPositionsFile(resource, ranges, this.positionsPrinter));
            this.positionsPrinter.flush();
            this.positionsPrinter.close();
        }
    }

    public void reportLastLocations() {
        List debugFiles;
        PathMatcher badFileMatcher = FileSystems.getDefault().getPathMatcher("glob:*.bad");
        Set files = this.openFiles.asMap().keySet();
        List badFiles = files.stream().map(Path::getFileName).filter(badFileMatcher::matches).collect(Collectors.toList());
        if (!badFiles.isEmpty()) {
            LOGGER.info("Rejected records can be found in the following file(s): {}", (Object)Joiner.on((String)", ").join(badFiles));
        }
        if (!(debugFiles = files.stream().map(Path::getFileName).filter(path -> !badFileMatcher.matches((Path)path)).collect(Collectors.toList())).isEmpty()) {
            LOGGER.info("Errors are detailed in the following file(s): {}", (Object)Joiner.on((String)", ").join(debugFiles));
        }
        if (this.positionsTracker != null) {
            LOGGER.info("Last processed positions can be found in {}", (Object)POSITIONS_FILE);
        }
    }

    @NonNull
    public Function<Flux<Void>, Flux<Void>> newTerminationHandler() {
        return upstream -> upstream.doOnTerminate(() -> this.failedRecordSink.complete()).doOnTerminate(() -> this.unmappableRecordSink.complete()).doOnTerminate(() -> this.unmappableStatementSink.complete()).doOnTerminate(() -> this.failedWriteSink.complete()).doOnTerminate(() -> this.failedReadSink.complete()).doOnTerminate(() -> this.failedCASWriteSink.complete()).doOnTerminate(() -> this.uncaughtExceptionSink.complete()).mergeWith(this.uncaughtExceptionProcessor);
    }

    @NonNull
    public Function<Flux<BatchableStatement<?>>, Flux<BatchableStatement<?>>> newUnmappableStatementsHandler() {
        return upstream -> upstream.flatMap(stmt -> {
            if (stmt instanceof UnmappableStatement) {
                try {
                    this.unmappableStatementSink.next((Object)((UnmappableStatement)stmt));
                    return this.maybeTriggerOnError(null, this.errors.incrementAndGet());
                }
                catch (Exception e) {
                    return Flux.error((Throwable)e);
                }
            }
            return Flux.just((Object)stmt);
        }, 1, 1);
    }

    @NonNull
    public Function<Flux<Record>, Flux<Record>> newFailedRecordsHandler() {
        return upstream -> upstream.flatMap(r -> {
            if (r instanceof ErrorRecord) {
                try {
                    this.failedRecordSink.next((Object)((ErrorRecord)r));
                    return this.maybeTriggerOnError(null, this.errors.incrementAndGet());
                }
                catch (Exception e) {
                    return Flux.error((Throwable)e);
                }
            }
            return Flux.just((Object)r);
        }, 1, 1);
    }

    @NonNull
    public Function<Flux<Record>, Flux<Record>> newUnmappableRecordsHandler() {
        return upstream -> upstream.flatMap(r -> {
            if (r instanceof ErrorRecord) {
                try {
                    this.unmappableRecordSink.next((Object)((ErrorRecord)r));
                    return this.maybeTriggerOnError(null, this.errors.incrementAndGet());
                }
                catch (Exception e) {
                    return Flux.error((Throwable)e);
                }
            }
            return Flux.just((Object)r);
        }, 1, 1);
    }

    @NonNull
    public Function<Flux<WriteResult>, Flux<WriteResult>> newFailedWritesHandler() {
        return upstream -> upstream.flatMap(r -> {
            try {
                if (!r.isSuccess()) {
                    this.failedWriteSink.next(r);
                    assert (r.getError().isPresent());
                    Throwable cause = ((BulkExecutionException)r.getError().get()).getCause();
                    return this.maybeTriggerOnError(cause, this.errors.addAndGet(r.getBatchSize()));
                }
                if (!r.wasApplied()) {
                    this.failedCASWriteSink.next(r);
                    return this.maybeTriggerOnError(null, this.errors.addAndGet(r.getBatchSize()));
                }
                return Flux.just((Object)r);
            }
            catch (Exception e) {
                return Flux.error((Throwable)e);
            }
        }, 1, 1);
    }

    @NonNull
    public Function<Flux<ReadResult>, Flux<ReadResult>> newFailedReadsHandler() {
        return upstream -> upstream.flatMap(r -> {
            if (r.isSuccess()) {
                return Flux.just((Object)r);
            }
            try {
                this.failedReadSink.next(r);
                assert (r.getError().isPresent());
                Throwable cause = ((BulkExecutionException)r.getError().get()).getCause();
                return this.maybeTriggerOnError(cause, this.errors.incrementAndGet());
            }
            catch (Exception e) {
                return Flux.error((Throwable)e);
            }
        }, 1, 1);
    }

    public <T extends Result> Function<Flux<T>, Flux<T>> newQueryWarningsHandler() {
        return upstream -> upstream.doOnNext(result -> {
            if (this.queryWarningsEnabled.get()) {
                result.getExecutionInfo().ifPresent(this::maybeLogQueryWarnings);
            }
        });
    }

    public Function<Flux<WriteResult>, Flux<Void>> newResultPositionsHandler() {
        return upstream -> upstream.map(Result::getStatement).transform(this.newStatementToRecordMapper()).flatMap(this::sendToPositionsSink).then().flux();
    }

    public <T> Function<Flux<T>, Flux<T>> newTotalItemsCounter() {
        return upstream -> upstream.doOnNext(r -> this.totalItems.increment());
    }

    @NonNull
    private Function<Flux<? extends Statement<?>>, Flux<Record>> newStatementToRecordMapper() {
        return upstream -> upstream.flatMap(statement -> {
            if (statement instanceof BatchStatement) {
                return Flux.fromIterable((Iterable)((BatchStatement)statement));
            }
            return Flux.just((Object)statement);
        }).cast(MappedStatement.class).map(MappedStatement::getRecord);
    }

    @NonNull
    private FluxSink<ErrorRecord> newFailedRecordSink() {
        UnicastProcessor processor = UnicastProcessor.create();
        Flux flux = processor.flatMap(this::appendFailedRecordToDebugFile);
        if (this.trackPositions) {
            flux = flux.flatMap(record -> this.appendToBadFile((Record)record, CONNECTOR_BAD_FILE)).flatMap(this::sendToPositionsSink);
        }
        flux.subscribe(v -> {}, this::onSinkError);
        return processor.sink(FluxSink.OverflowStrategy.BUFFER);
    }

    @NonNull
    private FluxSink<ErrorRecord> newUnmappableRecordSink() {
        UnicastProcessor processor = UnicastProcessor.create();
        processor.flatMap(this::appendUnmappableReadResultToDebugFile).subscribe(v -> {}, this::onSinkError);
        return processor.sink(FluxSink.OverflowStrategy.BUFFER);
    }

    @NonNull
    private FluxSink<UnmappableStatement> newUnmappableStatementSink() {
        UnicastProcessor processor = UnicastProcessor.create();
        processor.doOnNext(this::maybeWarnInvalidMapping).flatMap(this::appendUnmappableStatementToDebugFile).transform(this.newStatementToRecordMapper()).flatMap(record -> this.appendToBadFile((Record)record, MAPPING_BAD_FILE)).flatMap(this::sendToPositionsSink).subscribe(v -> {}, this::onSinkError);
        return processor.sink(FluxSink.OverflowStrategy.BUFFER);
    }

    @NonNull
    private FluxSink<WriteResult> newFailedWriteResultSink() {
        UnicastProcessor processor = UnicastProcessor.create();
        processor.flatMap(this::appendFailedWriteResultToDebugFile).map(Result::getStatement).transform(this.newStatementToRecordMapper()).flatMap(record -> this.appendToBadFile((Record)record, LOAD_BAD_FILE)).flatMap(this::sendToPositionsSink).subscribe(v -> {}, this::onSinkError);
        return processor.sink(FluxSink.OverflowStrategy.BUFFER);
    }

    @NonNull
    private FluxSink<WriteResult> newFailedCASWriteSink() {
        UnicastProcessor processor = UnicastProcessor.create();
        processor.flatMap(this::appendFailedCASWriteResultToDebugFile).map(Result::getStatement).transform(this.newStatementToRecordMapper()).flatMap(record -> this.appendToBadFile((Record)record, CAS_BAD_FILE)).flatMap(this::sendToPositionsSink).subscribe(v -> {}, this::onSinkError);
        return processor.sink(FluxSink.OverflowStrategy.BUFFER);
    }

    @NonNull
    private FluxSink<ReadResult> newFailedReadResultSink() {
        UnicastProcessor processor = UnicastProcessor.create();
        processor.flatMap(this::appendFailedReadResultToDebugFile).subscribe(v -> {}, this::onSinkError);
        return processor.sink(FluxSink.OverflowStrategy.BUFFER);
    }

    @NonNull
    private FluxSink<Record> newPositionsSink() {
        UnicastProcessor processor = UnicastProcessor.create();
        processor.doOnNext(record -> this.positionsTracker.update(record.getResource(), record.getPosition())).subscribe(v -> {}, this::onSinkError);
        return processor.sink(FluxSink.OverflowStrategy.BUFFER);
    }

    private Mono<Record> appendToBadFile(Record record, String file) {
        try {
            Object source = record.getSource();
            if (source != null) {
                Path logFile = this.operationDirectory.resolve(file);
                PrintWriter writer = (PrintWriter)this.openFiles.get((Object)logFile);
                assert (writer != null);
                LogManagerUtils.printAndMaybeAddNewLine(source.toString(), writer);
                writer.flush();
            }
            return Mono.just((Object)record);
        }
        catch (Exception e) {
            return Mono.error((Throwable)e);
        }
    }

    private Mono<WriteResult> appendFailedWriteResultToDebugFile(WriteResult result) {
        return this.appendStatement(result, LOAD_ERRORS_FILE, true);
    }

    private Mono<WriteResult> appendFailedCASWriteResultToDebugFile(WriteResult result) {
        return this.appendStatement(result, CAS_ERRORS_FILE, true);
    }

    private Mono<ReadResult> appendFailedReadResultToDebugFile(ReadResult result) {
        return this.appendStatement(result, UNLOAD_ERRORS_FILE, true);
    }

    private <R extends Result> Mono<R> appendStatement(R result, String logFileName, boolean appendNewLine) {
        try {
            WriteResult writeResult;
            Path logFile = this.operationDirectory.resolve(logFileName);
            PrintWriter writer = (PrintWriter)this.openFiles.get((Object)logFile);
            assert (writer != null);
            writer.print("Statement: ");
            String format = this.statementFormatter.format(result.getStatement(), this.statementFormatVerbosity, this.protocolVersion, this.codecRegistry);
            LogManagerUtils.printAndMaybeAddNewLine(format, writer);
            if (result instanceof WriteResult && (writeResult = (WriteResult)result).isSuccess() && !writeResult.wasApplied()) {
                writer.println("Failed conditional updates: ");
                writeResult.getFailedWrites().forEach(row -> {
                    String failed = this.rowFormatter.format(row, this.protocolVersion, this.codecRegistry);
                    LogManagerUtils.printAndMaybeAddNewLine(failed, writer);
                });
            }
            if (result.getError().isPresent()) {
                // empty if block
            }
            if (appendNewLine) {
                writer.println();
            }
            writer.flush();
            return Mono.just(result);
        }
        catch (Exception e) {
            return Mono.error((Throwable)e);
        }
    }

    private Mono<UnmappableStatement> appendUnmappableStatementToDebugFile(UnmappableStatement statement) {
        try {
            Path logFile = this.operationDirectory.resolve(MAPPING_ERRORS_FILE);
            PrintWriter writer = (PrintWriter)this.openFiles.get((Object)logFile);
            assert (writer != null);
            Record record = statement.getRecord();
            writer.println("Resource: " + record.getResource());
            writer.println("Position: " + record.getPosition());
            if (record.getSource() != null) {
                writer.println("Source: " + LogManagerUtils.formatSource(record));
            }
            writer.println();
            writer.flush();
            return Mono.just((Object)statement);
        }
        catch (Exception e) {
            return Mono.error((Throwable)e);
        }
    }

    private Mono<ErrorRecord> appendUnmappableReadResultToDebugFile(ErrorRecord record) {
        try {
            Path logFile = this.operationDirectory.resolve(MAPPING_ERRORS_FILE);
            PrintWriter writer = (PrintWriter)this.openFiles.get((Object)logFile);
            assert (writer != null);
            if (record.getSource() instanceof ReadResult) {
                ReadResult source = (ReadResult)record.getSource();
                this.appendStatement(source, MAPPING_ERRORS_FILE, false);
                source.getRow().ifPresent(row -> {
                    writer.print("Row: ");
                    String format = this.rowFormatter.format(row, this.protocolVersion, this.codecRegistry);
                    LogManagerUtils.printAndMaybeAddNewLine(format, writer);
                });
            }
            writer.println();
            writer.flush();
            return Mono.just((Object)record);
        }
        catch (Exception e) {
            return Mono.error((Throwable)e);
        }
    }

    private Mono<ErrorRecord> appendFailedRecordToDebugFile(ErrorRecord record) {
        try {
            Path logFile = this.operationDirectory.resolve(CONNECTOR_ERRORS_FILE);
            PrintWriter writer = (PrintWriter)this.openFiles.get((Object)logFile);
            assert (writer != null);
            writer.println("Resource: " + record.getResource());
            writer.println("Position: " + record.getPosition());
            if (record.getSource() != null) {
                writer.println("Source: " + LogManagerUtils.formatSource((Record)record));
            }
            writer.println();
            writer.flush();
            return Mono.just((Object)record);
        }
        catch (Exception e) {
            return Mono.error((Throwable)e);
        }
    }

    @NonNull
    private Mono<Record> sendToPositionsSink(Record record) {
        try {
            this.positionsSink.next((Object)record);
            return Mono.just((Object)record);
        }
        catch (Exception e) {
            return Mono.error((Throwable)e);
        }
    }

    private static void appendToPositionsFile(URI resource, List<Range> positions, PrintWriter positionsPrinter) {
        positionsPrinter.print(resource);
        positionsPrinter.print(':');
        positions.stream().findFirst().ifPresent(pos -> positionsPrinter.print(pos.getUpper()));
        positionsPrinter.println();
    }

    private <T> Flux<T> maybeTriggerOnError(@Nullable Throwable error, int currentErrorCount) {
        if (error != null && LogManager.isUnrecoverable(error)) {
            return Flux.error((Throwable)error);
        }
        if (this.errorThreshold.checkThresholdExceeded((long)currentErrorCount, (Number)this.totalItems)) {
            return Flux.error((Throwable)new TooManyErrorsException(this.errorThreshold));
        }
        return Flux.empty();
    }

    private void maybeWarnInvalidMapping(UnmappableStatement stmt) {
        if (stmt.getError() instanceof InvalidMappingException && this.invalidMappingWarningDone.compareAndSet(false, true)) {
            LOGGER.warn("At least 1 record does not match the provided schema.mapping or schema.query. Please check that the connector configuration and the schema configuration are correct.");
        }
    }

    private void maybeLogQueryWarnings(ExecutionInfo info) {
        for (String warning : info.getWarnings()) {
            if (this.queryWarningsThreshold.checkThresholdExceeded((long)this.queryWarnings.incrementAndGet(), (Number)this.totalItems)) {
                this.queryWarningsEnabled.set(false);
                LOGGER.warn("The maximum number of logged query warnings has been exceeded ({}); subsequent warnings will not be logged.", (Object)this.queryWarningsThreshold.thresholdAsString());
                break;
            }
            LOGGER.warn("Query generated server-side warning: " + warning);
        }
    }

    private void onSinkError(Throwable error) {
        LOGGER.error("Error while writing to log files, aborting", error);
        this.uncaughtExceptionSink.error(error);
    }

    private static boolean isUnrecoverable(Throwable error) {
        if (error instanceof AllNodesFailedException) {
            for (Throwable child : ((AllNodesFailedException)error).getAllErrors().values().stream().flatMap(Collection::stream).collect(Collectors.toList())) {
                if (!LogManager.isUnrecoverable(child)) continue;
                return true;
            }
            return false;
        }
        return !(error instanceof ServerError) && !(error instanceof QueryExecutionException) && !(error instanceof InvalidQueryException) && !(error instanceof DriverTimeoutException) && !(error instanceof RequestThrottlingException) && !(error instanceof FrameTooLongException) && !(error instanceof BusyConnectionException);
    }
}

