/*
 * Decompiled with CFR 0.152.
 */
package com.cedarsoftware.util;

import java.lang.ref.WeakReference;
import java.util.AbstractMap;
import java.util.AbstractSet;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;

public class TTLCache<K, V>
implements Map<K, V> {
    private final long ttlMillis;
    private final int maxSize;
    private final ConcurrentHashMap<Object, CacheEntry> cacheMap;
    private final ReentrantLock lock = new ReentrantLock();
    private final Node head;
    private final Node tail;
    private static final Object NULL_ITEM = new Object();
    private static final ScheduledExecutorService scheduler = Executors.newSingleThreadScheduledExecutor();

    public TTLCache(long ttlMillis) {
        this(ttlMillis, -1, 60000L);
    }

    public TTLCache(long ttlMillis, int maxSize) {
        this(ttlMillis, maxSize, 60000L);
    }

    public TTLCache(long ttlMillis, int maxSize, long cleanupIntervalMillis) {
        if (ttlMillis < 1L) {
            throw new IllegalArgumentException("TTL must be at least 1 millisecond.");
        }
        if (cleanupIntervalMillis < 10L) {
            throw new IllegalArgumentException("cleanupIntervalMillis must be at least 10 milliseconds.");
        }
        this.ttlMillis = ttlMillis;
        this.maxSize = maxSize;
        this.cacheMap = new ConcurrentHashMap();
        this.head = new Node(null, null);
        this.head.next = this.tail = new Node(null, null);
        this.tail.prev = this.head;
        this.schedulePurgeTask(cleanupIntervalMillis);
    }

    private void schedulePurgeTask(long cleanupIntervalMillis) {
        WeakReference cacheRef = new WeakReference(this);
        PurgeTask purgeTask = new PurgeTask(cacheRef);
        scheduler.scheduleAtFixedRate(purgeTask, cleanupIntervalMillis, cleanupIntervalMillis, TimeUnit.MILLISECONDS);
    }

    private Object toCacheItem(Object item) {
        return item == null ? NULL_ITEM : item;
    }

    private <T> T fromCacheItem(Object cacheItem) {
        return (T)(cacheItem == NULL_ITEM ? null : cacheItem);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void purgeExpiredEntries() {
        long currentTime = System.currentTimeMillis();
        Iterator<Map.Entry<Object, CacheEntry>> it = this.cacheMap.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<Object, CacheEntry> entry = it.next();
            if (entry.getValue().expiryTime >= currentTime) continue;
            it.remove();
            this.lock.lock();
            try {
                this.unlink(entry.getValue().node);
            }
            finally {
                this.lock.unlock();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void removeEntry(Object cacheKey) {
        CacheEntry entry = this.cacheMap.remove(cacheKey);
        if (entry != null) {
            Node node = entry.node;
            this.lock.lock();
            try {
                this.unlink(node);
            }
            finally {
                this.lock.unlock();
            }
        }
    }

    private void unlink(Node node) {
        node.prev.next = node.next;
        node.next.prev = node.prev;
        node.prev = null;
        node.next = null;
        node.value = null;
    }

    private void moveToTail(Node node) {
        node.prev.next = node.next;
        node.next.prev = node.prev;
        node.prev = this.tail.prev;
        node.next = this.tail;
        this.tail.prev.next = node;
        this.tail.prev = node;
    }

    private void insertAtTail(Node node) {
        node.prev = this.tail.prev;
        node.next = this.tail;
        this.tail.prev.next = node;
        this.tail.prev = node;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public V put(K key, V value) {
        Object cacheKey = this.toCacheItem(key);
        Object cacheValue = this.toCacheItem(value);
        long expiryTime = System.currentTimeMillis() + this.ttlMillis;
        Node node = new Node(cacheKey, cacheValue);
        CacheEntry newEntry = new CacheEntry(node, expiryTime);
        CacheEntry oldEntry = this.cacheMap.put(cacheKey, newEntry);
        boolean acquired = this.lock.tryLock();
        try {
            if (acquired) {
                Node lruNode;
                this.insertAtTail(node);
                if (this.maxSize > -1 && this.cacheMap.size() > this.maxSize && (lruNode = this.head.next) != this.tail) {
                    this.removeEntry(lruNode.key);
                }
            }
        }
        finally {
            if (acquired) {
                this.lock.unlock();
            }
        }
        return oldEntry != null ? (V)this.fromCacheItem(oldEntry.node.value) : null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public V get(Object key) {
        Object cacheKey = this.toCacheItem(key);
        CacheEntry entry = this.cacheMap.get(cacheKey);
        if (entry == null) {
            return null;
        }
        long currentTime = System.currentTimeMillis();
        if (entry.expiryTime < currentTime) {
            this.removeEntry(cacheKey);
            return null;
        }
        Object value = this.fromCacheItem(entry.node.value);
        boolean acquired = this.lock.tryLock();
        try {
            if (acquired) {
                this.moveToTail(entry.node);
            }
        }
        finally {
            if (acquired) {
                this.lock.unlock();
            }
        }
        return (V)value;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public V remove(Object key) {
        Object cacheKey = this.toCacheItem(key);
        CacheEntry entry = this.cacheMap.remove(cacheKey);
        if (entry != null) {
            Object value = this.fromCacheItem(entry.node.value);
            this.lock.lock();
            try {
                this.unlink(entry.node);
            }
            finally {
                this.lock.unlock();
            }
            return (V)value;
        }
        return null;
    }

    @Override
    public void clear() {
        this.cacheMap.clear();
        this.lock.lock();
        try {
            this.head.next = this.tail;
            this.tail.prev = this.head;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public int size() {
        return this.cacheMap.size();
    }

    @Override
    public boolean isEmpty() {
        return this.cacheMap.isEmpty();
    }

    @Override
    public boolean containsKey(Object key) {
        Object cacheKey = this.toCacheItem(key);
        CacheEntry entry = this.cacheMap.get(cacheKey);
        if (entry == null) {
            return false;
        }
        if (entry.expiryTime < System.currentTimeMillis()) {
            this.removeEntry(cacheKey);
            return false;
        }
        return true;
    }

    @Override
    public boolean containsValue(Object value) {
        Object cacheValue = this.toCacheItem(value);
        for (CacheEntry entry : this.cacheMap.values()) {
            Object entryValue = entry.node.value;
            if (!Objects.equals(entryValue, cacheValue)) continue;
            return true;
        }
        return false;
    }

    @Override
    public void putAll(Map<? extends K, ? extends V> m) {
        for (Map.Entry<K, V> e : m.entrySet()) {
            this.put(e.getKey(), e.getValue());
        }
    }

    @Override
    public Set<K> keySet() {
        HashSet keys = new HashSet();
        for (CacheEntry entry : this.cacheMap.values()) {
            Object key = this.fromCacheItem(entry.node.key);
            keys.add(key);
        }
        return keys;
    }

    @Override
    public Collection<V> values() {
        ArrayList values = new ArrayList();
        for (CacheEntry entry : this.cacheMap.values()) {
            Object value = this.fromCacheItem(entry.node.value);
            values.add(value);
        }
        return values;
    }

    @Override
    public Set<Map.Entry<K, V>> entrySet() {
        return new EntrySet();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Map)) {
            return false;
        }
        Map other = (Map)o;
        this.lock.lock();
        try {
            boolean bl = this.entrySet().equals(other.entrySet());
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public int hashCode() {
        this.lock.lock();
        try {
            int hashCode = 1;
            Node node = this.head.next;
            while (node != this.tail) {
                Object key = this.fromCacheItem(node.key);
                Object value = this.fromCacheItem(node.value);
                hashCode = 31 * hashCode + (key == null ? 0 : key.hashCode());
                hashCode = 31 * hashCode + (value == null ? 0 : value.hashCode());
                node = node.next;
            }
            int n = hashCode;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String toString() {
        this.lock.lock();
        try {
            StringBuilder sb = new StringBuilder();
            sb.append('{');
            Iterator<Map.Entry<K, V>> it = this.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry<K, V> entry = it.next();
                sb.append(entry.getKey()).append('=').append(entry.getValue());
                if (!it.hasNext()) continue;
                sb.append(", ");
            }
            sb.append('}');
            String string = sb.toString();
            return string;
        }
        finally {
            this.lock.unlock();
        }
    }

    public static void shutdown() {
        scheduler.shutdown();
    }

    private static class Node {
        final Object key;
        Object value;
        Node prev;
        Node next;

        Node(Object key, Object value) {
            this.key = key;
            this.value = value;
        }
    }

    private static class PurgeTask
    implements Runnable {
        private final WeakReference<TTLCache<?, ?>> cacheRef;
        private volatile boolean canceled = false;

        PurgeTask(WeakReference<TTLCache<?, ?>> cacheRef) {
            this.cacheRef = cacheRef;
        }

        @Override
        public void run() {
            TTLCache cache = (TTLCache)this.cacheRef.get();
            if (cache == null) {
                this.cancel();
            } else {
                cache.purgeExpiredEntries();
            }
        }

        private void cancel() {
            if (!this.canceled) {
                this.canceled = true;
            }
        }
    }

    private static class CacheEntry {
        final Node node;
        final long expiryTime;

        CacheEntry(Node node, long expiryTime) {
            this.node = node;
            this.expiryTime = expiryTime;
        }
    }

    private class EntrySet
    extends AbstractSet<Map.Entry<K, V>> {
        private EntrySet() {
        }

        @Override
        public Iterator<Map.Entry<K, V>> iterator() {
            return new EntryIterator();
        }

        @Override
        public int size() {
            return TTLCache.this.size();
        }

        @Override
        public void clear() {
            TTLCache.this.clear();
        }
    }

    private class EntryIterator
    implements Iterator<Map.Entry<K, V>> {
        private final Iterator<Map.Entry<Object, CacheEntry>> iterator;
        private Map.Entry<Object, CacheEntry> current;

        EntryIterator() {
            this.iterator = TTLCache.this.cacheMap.entrySet().iterator();
        }

        @Override
        public boolean hasNext() {
            return this.iterator.hasNext();
        }

        @Override
        public Map.Entry<K, V> next() {
            this.current = this.iterator.next();
            Object key = TTLCache.this.fromCacheItem(this.current.getValue().node.key);
            Object value = TTLCache.this.fromCacheItem(this.current.getValue().node.value);
            return new AbstractMap.SimpleEntry<Object, Object>(key, value);
        }

        @Override
        public void remove() {
            if (this.current == null) {
                throw new IllegalStateException();
            }
            Object cacheKey = this.current.getKey();
            TTLCache.this.removeEntry(cacheKey);
            this.current = null;
        }
    }
}

