/**
 * Hub Common
 *
 * Copyright (C) 2017 Black Duck Software, Inc.
 * http://www.blackducksoftware.com/
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.blackducksoftware.integration.hub.dataservice.notification.transformer;

import java.util.ArrayList;
import java.util.List;

import com.blackducksoftware.integration.hub.api.item.MetaService;
import com.blackducksoftware.integration.hub.api.notification.NotificationRequestService;
import com.blackducksoftware.integration.hub.api.policy.PolicyRequestService;
import com.blackducksoftware.integration.hub.api.project.version.ProjectVersionRequestService;
import com.blackducksoftware.integration.hub.dataservice.model.ProjectVersionModel;
import com.blackducksoftware.integration.hub.dataservice.notification.model.NotificationContentItem;
import com.blackducksoftware.integration.hub.dataservice.notification.model.VulnerabilityContentItem;
import com.blackducksoftware.integration.hub.exception.HubIntegrationException;
import com.blackducksoftware.integration.hub.exception.HubItemTransformException;
import com.blackducksoftware.integration.hub.model.view.ComponentVersionView;
import com.blackducksoftware.integration.hub.model.view.NotificationView;
import com.blackducksoftware.integration.hub.model.view.VulnerabilityNotificationView;
import com.blackducksoftware.integration.hub.model.view.components.AffectedProjectVersion;
import com.blackducksoftware.integration.hub.model.view.components.VulnerabilityNotificationContent;
import com.blackducksoftware.integration.hub.model.view.components.VulnerabilitySourceQualifiedId;
import com.blackducksoftware.integration.hub.service.HubResponseService;
import com.blackducksoftware.integration.log.IntLogger;

public class VulnerabilityTransformer extends AbstractNotificationTransformer {
    private final IntLogger logger;

    public VulnerabilityTransformer(final HubResponseService hubResponseService, final NotificationRequestService notificationService,
            final ProjectVersionRequestService projectVersionService, final PolicyRequestService policyService,
            final MetaService metaService,
            final IntLogger logger) {
        super(hubResponseService, notificationService, projectVersionService, policyService,
                metaService);
        this.logger = logger;
    }

    @Override
    public List<NotificationContentItem> transform(final NotificationView item) throws HubItemTransformException {
        final List<NotificationContentItem> notificationContentItems = new ArrayList<>();
        try {
            final VulnerabilityNotificationView vulnerabilityNotificationItem = (VulnerabilityNotificationView) item;
            final VulnerabilityNotificationContent vulnerabilityNotificationContent = vulnerabilityNotificationItem.content;

            final String componentName = vulnerabilityNotificationContent.componentName;
            final String componentVersionUrl = vulnerabilityNotificationContent.componentVersionLink;
            final ComponentVersionView fullComponentVersion = getComponentVersion(componentVersionUrl);

            final List<VulnerabilitySourceQualifiedId> addedVulnList = extractIdList(
                    vulnerabilityNotificationContent.newVulnerabilityCount,
                    vulnerabilityNotificationContent.newVulnerabilityIds);
            final List<VulnerabilitySourceQualifiedId> updatedVulnList = extractIdList(
                    vulnerabilityNotificationContent.updatedVulnerabilityCount,
                    vulnerabilityNotificationContent.updatedVulnerabilityIds);
            final List<VulnerabilitySourceQualifiedId> deletedVulnList = extractIdList(
                    vulnerabilityNotificationContent.deletedVulnerabilityCount,
                    vulnerabilityNotificationContent.deletedVulnerabilityIds);

            final List<AffectedProjectVersion> affectedProjectVersionList = vulnerabilityNotificationItem.content.affectedProjectVersions;
            if (affectedProjectVersionList == null || affectedProjectVersionList.isEmpty()) {
                notificationContentItems
                        .add(new VulnerabilityContentItem(item.createdAt, new ProjectVersionModel(),
                                componentName, fullComponentVersion,
                                componentVersionUrl, addedVulnList,
                                updatedVulnList,
                                deletedVulnList, null));
            } else {
                for (final AffectedProjectVersion affectedProjectVersion : affectedProjectVersionList) {
                    final ProjectVersionModel fullProjectVersion;
                    try {
                        fullProjectVersion = createFullProjectVersion(affectedProjectVersion.projectVersion,
                                affectedProjectVersion.projectName, affectedProjectVersion.projectVersionName);
                    } catch (final HubIntegrationException e) {
                        final Throwable originalException = getOriginalException(e);
                        // Testing the message is awful, but unfortunately the exception type (IOException) does not
                        // distinguish Forbidden from other errors.
                        if (originalException.getMessage().contains("Forbidden")) {
                            logger.warn(String.format("Skipping an affected projectVersion for this vulnerability. ProjectVersion: %s / %s; Reason: %s",
                                    affectedProjectVersion.projectName, affectedProjectVersion.projectVersionName,
                                    e.getMessage()));
                            continue;
                        } else {
                            throw e;
                        }
                    }
                    notificationContentItems
                            .add(new VulnerabilityContentItem(item.createdAt, fullProjectVersion,
                                    componentName, fullComponentVersion,
                                    componentVersionUrl, addedVulnList,
                                    updatedVulnList,
                                    deletedVulnList, affectedProjectVersion.componentIssueUrl));
                }
            }
        } catch (final Exception e) {
            throw new HubItemTransformException(e);
        }
        return notificationContentItems;
    }

    private Throwable getOriginalException(final HubIntegrationException e) {
        Throwable originalException = e;
        while (originalException.getCause() != null) {
            originalException = originalException.getCause();
        }
        return originalException;
    }

    private List<VulnerabilitySourceQualifiedId> extractIdList(final int count,
            final List<VulnerabilitySourceQualifiedId> idList) {
        if (count > 0 && idList != null && idList.size() > 0) {
            return idList;
        }
        return new ArrayList<>();
    }

}
