// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.azure.storage.file.share.models;

/**
 * A set of options for selecting shares from Storage File service.
 *
 * <ul>
 *     <li>
 *         Providing {@link ListSharesOptions#setPrefix(String) prefix} will filter selections to
 *         {@link ShareItem shares} that that begin with the prefix.
 *     </li>
 *     <li>
 *         Providing {@link ListSharesOptions#setMaxResultsPerPage(Integer) maxResultsPerPage} will limit the number of
 *         {@link ShareItem shares} returned in a single page.
 *     </li>
 *     <li>
 *         Setting {@link ListSharesOptions#setIncludeMetadata(boolean) includeMetadata} to true will include the
 *         metadata of each {@link ShareItem share}, if false {@link ShareItem#getMetadata()}  metadata} for each share
 *         will be {@code null}.
 *     </li>
 *     <li>
 *         Setting {@link ListSharesOptions#setIncludeSnapshots(boolean) includeSnapshots} to true will include
 *         snapshots of each {@link ShareItem share}, the snapshot will be included as separate items in the response
 *         and will be identifiable by {@link ShareItem#getSnapshot()}  snapshot} having a value. The base share will
 *         contain {@code null} for the snapshot.
 *     </li>
 *     <li>
 *         Setting {@link ListSharesOptions#setIncludeDeleted(boolean) includeDeleted} to true will include
 *         deleted {@link ShareItem shares}, the deleted shares will be included as separate items in the response
 *         and will be identifiable by {@link ShareItem#isDeleted()} having a <code>true</code>value.
 *     </li>
 * </ul>
 */
public final class ListSharesOptions {
    private String prefix;
    private Integer maxResultsPerPage;
    private boolean includeMetadata;
    private boolean includeSnapshots;
    private boolean includeDeleted;

    /**
     * Creates a new instance of {@link ListSharesOptions}.
     */
    public ListSharesOptions() {
    }

    /**
     * Sets the prefix that a share must match to be included in the listing.
     *
     * @param prefix The prefix that shares must start with to pass the filter
     * @return An updated ListSharesOptions object
     */
    public ListSharesOptions setPrefix(String prefix) {
        this.prefix = prefix;
        return this;
    }

    /**
     * Gets the prefix that a share must match to be included in the listing
     * @return the prefix that a share must match to be included in the listing
     */
    public String getPrefix() {
        return prefix;
    }

    /**
     * Sets the maximum number of shares to include in a single response.
     *
     * @param maxResultsPerPage Maximum number of shares to include in a single response. This value must be between 1
     * and 5000.
     * @return An updated ListSharesOptions object
     */
    public ListSharesOptions setMaxResultsPerPage(Integer maxResultsPerPage) {
        this.maxResultsPerPage = maxResultsPerPage;
        return this;
    }

    /**
     * Gets the maximum number of shares to include in a single response
     * @return the maximum number of shares to include in a single response
     */
    public Integer getMaxResultsPerPage() {
        return maxResultsPerPage;
    }

    /**
     * Sets the status of including share metadata when listing shares
     * <p>
     * If listing snapshots as well this will also determine if the snapshots have their metadata included as well.
     *
     * @param includeMetadata Flag indicating if metadata should be including in the listing
     * @return An updated ListSharesOptions object
     */
    public ListSharesOptions setIncludeMetadata(boolean includeMetadata) {
        this.includeMetadata = includeMetadata;
        return this;
    }

    /**
     * Gets the status of include share metadata when listing shares
     *
     * @return the status of include share metadata when listing shares
     */
    public boolean isIncludeMetadata() {
        return includeMetadata;
    }

    /**
     * Sets the status of including share snapshots when listing shares
     *
     * @param includeSnapshots Flag indicating if snapshots should be included in the listing
     * @return An updated ListSharesOptions object
     */
    public ListSharesOptions setIncludeSnapshots(boolean includeSnapshots) {
        this.includeSnapshots = includeSnapshots;
        return this;
    }

    /**
     * Gets the status of including share snapshots when listing shares
     *
     * @return the status of including share snapshots when listing shares
     */
    public boolean isIncludeSnapshots() {
        return includeSnapshots;
    }

    /**
     * Sets the status of including deleted shares when listing shares
     *
     * @param includeDeleted Flag indicating if deleted shares should be included in the listing
     * @return An updated ListSharesOptions object
     */
    public ListSharesOptions setIncludeDeleted(boolean includeDeleted) {
        this.includeDeleted = includeDeleted;
        return this;
    }

    /**
     * Gets the status of including deleted shares when listing shares
     *
     * @return the status of including deleted shares when listing shares
     */
    public boolean isIncludeDeleted() {
        return includeDeleted;
    }
}
